/*
 * Copyright (c) 2002 - 2004 Dmitry Dicky diwil@mail.ru
 * All rights reserved.
 *
 * Modifications Copyright (c) 2010 Diane Gagne diane@hartmantech.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS `AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Modified 2010 by Diane Gagne at Hartman Technica to include overflow, 
 * underflow, NaN and denormalized number handling as well as improve 
 * the speed performance.
 *
 * $Id: mulsf.c,v 1.10 2004/05/28 10:11:48 diwil Exp $
 */

#include "mathsf.h"

unsigned long
__mulsf3(long a1, long a2)
{
  register unsigned long result;
  register int sign;
  register int exp; 
  register int tmp;  

  //calculate the sign
  sign = ((a1>>SHORT_SHIFT^a2>>SHORT_SHIFT))&SHORT_SIGNBIT_MASK;
  
  //take the exponent for each operand
  exp = SHORT_EXPONENT_MASK & (a1>>SHORT_SHIFT);
  tmp = SHORT_EXPONENT_MASK & (a2>>SHORT_SHIFT);
  
#if __MSP430LIBC_IEEE754_ERRORS__
  //check if eather operand is an infinity or NaN
  //if so the exponent will be all 1
  if(exp == SHORT_EXPONENT_MASK)
    {
      //if the first number is NaN return a Nan taken from the operand
      if((a1 & MANTISSA_MASK) != 0)
	return (a1);
      //otherwise the number is infinity
      else
	{
	  //if operand 2 is infinity or NaN return a NaN
	  if(tmp == SHORT_EXPONENT_MASK)
	    return NAN;
	  //if operand 2 is 0 return a NaN
	  else
	    if((a2 & NO_SIGN) == 0)
	      return NAN;
	  //Otherwise return a signed infinity
	  return (a1&NO_SIGN)|(long)sign<<SHORT_SHIFT;
	}
    }
  
  //Check if operand 2 is infinity or NaN
  if(tmp == SHORT_EXPONENT_MASK)
    {
      //if the operand is NaN return a NaN
      if((a2 & MANTISSA_MASK)!=0)
	return a2;
      //if the first operand is 0 reutun a NaN
      else
	{
	  if((a1 & NO_SIGN) == 0)
	    return NAN;
	  //otherwise return a signed infinity
	  return (a2&NO_SIGN)|(long)sign<<SHORT_SHIFT;
	}
    }

  //normalize any denormalized numbers and look for multiply by 0
  if(exp == 0)
    {
      //get only the mantissa part of the first number
      a1 = a1 & MANTISSA_MASK;
      //if the number is 0 return a signed 0
      if(a1 == POSITIVE_ZERO)
	return (long)sign<<SHORT_SHIFT;
      //otherwise denormalise the number
      else
	{
	  //do one shift to start
	  a1 = a1<<1;
	  //while the 24th bit is not filled repeatedly shift the 
	  //mantissa right by 1 and subtract 1 from the exponent
	  while((a1 & MANTISSA_BIT) != MANTISSA_BIT)
	    {
	      a1 = a1<<1;
	      exp = exp-(1<<EXPONENT_SHIFT);
	    }
	}
    }
  //if it is already normalized get the mantissa with the hidden bit
  else
    a1 = MANT(a1);

  //do the same thing for the second operand
  if(tmp == 0)
    {
      a2 = a2 & MANTISSA_MASK;
      if((a2 & NO_SIGN) == 0)
	return (long)sign<<SHORT_SHIFT;
      else
	{
	      a2 = a2<<1;
	  while((a2 & MANTISSA_BIT) != MANTISSA_BIT)
	    {
	      a2 = a2<<1;
	      tmp = tmp-(1<<EXPONENT_SHIFT);
	    }
	}
    }
  else
    a2 = MANT(a2);
#else
  if(!a1 || !a2)
    return 0;
  a1 = MANT(a1);
  a2 = MANT(a2); 
#endif

  exp -= EXCESS<<EXPONENT_SHIFT;
  exp += tmp;

  result = __fpmulparts(a1, a2);
 
  //normalize and round the result
  if (result & LONG_SIGNBIT_MASK)
    {
      /* round */
      result += 0x80ul;
      result >>= (EXPONENT_SHIFT+1);
    }
  
  else
    {
      result += 0x40ul;
      result >>= EXPONENT_SHIFT;
      exp -= 1<<EXPONENT_SHIFT;
    }
  
 
  if (result & (HIDDEN<<1))
    {
      result >>= 1;
      exp += 1<<EXPONENT_SHIFT;
    }

  result &= ~HIDDEN;

#if __MSP430LIBC_IEEE754_ERRORS__
 //look for overflow or underflow
  if(exp <= 0)
    {
      //if the second exponent is under 1 it is underflow
      if(tmp <= MAX_EXPONENT<<EXPONENT_SHIFT)
	{
	  //if the total exponent is less that -143 return 0
	  if(exp < MIN_EXPONENT<<EXPONENT_SHIFT)
	    return(POSITIVE_ZERO|(long)sign<<SHORT_SHIFT);
	  //otherwise denormalize the result
	  else
	    {
	      result = result|MANTISSA_BIT;
	      result = result>>((-exp)>>EXPONENT_SHIFT);
	      result += 1;
	      result = result>>1;
	      exp = 0x0000;
	    }
	}
      //if exponent 2 was posative return a signed infinity
      else
	return (POSITIVE_INFINITY|(long)sign<<SHORT_SHIFT);
    }
  //the odd overflow will not appear negative so return infinity for them too
  else if(exp > EXPONENT_OVERFLOW<<EXPONENT_SHIFT)
    return (long)sign<<SHORT_SHIFT | POSITIVE_INFINITY;
#endif

  //put all the parts together again
  result |= ((long)(sign | exp)<<SHORT_SHIFT);

  return result;                                                                                                            
}

#if !defined MSP430_HAS_HWMUL
//bit bashing multiply with no hardware multiplier
unsigned long __fpmulparts(unsigned long a1, unsigned long a2)
{
  unsigned long long mult = (unsigned long long)a2;
  unsigned long long total = 0;
  unsigned long ret;

  int bit;
  for (bit = 0; bit < MANTISSA_BITS; bit++)
    {  
      if (a1 & 0x1)
	{
	  total+=mult;
	}
      mult<<=1;
      a1 >>= 1;
    }
  ret = total>>SHORT_SHIFT;
  return ret;
}

#endif

