/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.nativec.jna;

import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import sun.misc.Unsafe;
import sun.nio.ch.DirectBuffer;

import com.sun.jna.Native;
import com.sun.jna.Pointer;

import de.intarsys.nativec.api.INativeHandle;

public class JnaNativeHandle implements INativeHandle {

	private static final Pointer NULL = Pointer.createConstant(0);

	private static Unsafe unsafe;

	static {
		// todo get rid of unsafe Unsafe
		try {
			Field field = Unsafe.class.getDeclaredField("theUnsafe");
			field.setAccessible(true);
			unsafe = (Unsafe) field.get(null);
		} catch (Exception e) {
			throw new RuntimeException("can't load sun.misc.Unsafe");
		}
	}

	final private long address;

	private ByteBuffer buffer;

	private int offset;

	final private Pointer pointer;

	private int size;

	public JnaNativeHandle(ByteBuffer pBuffer) {
		if (pBuffer.isDirect()) {
			this.size = pBuffer.limit() - pBuffer.position();
			this.buffer = pBuffer;
			this.offset = this.buffer.position();
			this.address = ((DirectBuffer) pBuffer).address() + offset;
			this.pointer = NULL.share(address);
		} else {
			this.size = pBuffer.limit() - pBuffer.position();
			this.buffer = ByteBuffer.allocateDirect(size).order(
					ByteOrder.nativeOrder());
			this.offset = this.buffer.position();
			this.address = ((DirectBuffer) this.buffer).address() + offset;
			this.pointer = NULL.share(address);
			pointer.write(0, pBuffer.array(), pBuffer.position(), this.size);
		}
	}

	public JnaNativeHandle(JnaNativeHandle handle, int offset) {
		this.address = handle.address + offset;
		this.pointer = NULL.share(address);
		this.buffer = handle.buffer;
		this.offset = handle.offset + offset;
		this.size = handle.size - offset;
	}

	public JnaNativeHandle(long address) {
		this.address = address;
		this.pointer = NULL.share(address);
	}

	public JnaNativeHandle(Pointer pointer) {
		this.pointer = pointer;
		// yes, I know ...
		this.address = Long.parseLong(pointer.toString().substring(9), 16);
	}

	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof INativeHandle)) {
			return false;
		}
		INativeHandle otherHandle = (INativeHandle) obj;
		return address == otherHandle.getAddress()
				&& size == otherHandle.getSize();
	}

	public long getAddress() {
		return address;
	}

	public byte getByte(int index) {
		return buffer.get(offset + index);
	}

	public byte[] getByteArray(int index, int count) {
		byte[] result = new byte[count];
		buffer.position(offset + index);
		buffer.get(result);
		return result;
	}

	public long getCLong(int index) {
		if (Native.LONG_SIZE == 4) {
			return buffer.getInt(offset + index);
		}
		return buffer.getLong(offset + index);
	}

	public int getInt(int index) {
		return buffer.getInt(offset + index);
	}

	public long getLong(int index) {
		return buffer.getLong(offset + index);
	}

	public INativeHandle getNativeHandle(int index) {
		long indirectAddress = unsafe.getAddress(address + index);
		return new JnaNativeHandle(indirectAddress);
	}

	public Pointer getPointer() {
		return pointer;
	}

	public short getShort(int index) {
		return buffer.getShort(offset + index);
	}

	public int getSize() {
		return size;
	}

	public String getString(int index) {
		// if (buffer == null) {
		// return null;
		// }
		// // todo encoding
		// buffer.position(offset + index);
		// int i = 0;
		// while (buffer.get() != 0) {
		// i++;
		// }
		// buffer.position(offset + index);
		// byte[] valueBytes = new byte[i];
		// buffer.get(valueBytes);
		// return new String(valueBytes);
		return getPointer().getString(index);
	}

	public String getWideString(int index) {
		return getPointer().getString(index, true);
	}

	@Override
	public int hashCode() {
		// not quite sophisticated - but we don't want to hash anyway
		return (int) address + size;
	}

	public INativeHandle offset(int offset) {
		return new JnaNativeHandle(this, offset);
	}

	public void setByte(int index, byte value) {
		buffer.put(offset + index, value);
	}

	public void setByteArray(int index, byte[] value, int valueOffset,
			int valueCount) {
		buffer.position(offset + index);
		buffer.put(value, valueOffset, valueCount);
	}

	public void setCLong(int index, long value) {
		if (Native.LONG_SIZE == 4) {
			buffer.putInt(offset + index, (int) value);
			return;
		}
		buffer.putLong(offset + index, value);
	}

	public void setInt(int index, int value) {
		buffer.putInt(offset + index, value);
	}

	public void setLong(int index, long value) {
		buffer.putLong(offset + index, value);
	}

	public void setNativeHandle(int index, INativeHandle handle) {
		unsafe.putAddress(address + index, handle.getAddress());
	}

	public void setShort(int index, short value) {
		buffer.putShort(offset + index, value);
	}

	public void setSize(int pSize) {
		if (buffer == null) {
			this.buffer = pointer.getByteBuffer(0, pSize);
			this.offset = buffer.position();
		} else {
			if (this.size < pSize) {
				this.buffer = pointer.getByteBuffer(0, pSize);
				this.offset = buffer.position();
			}
		}
		this.size = pSize;
	}

	public void setString(int index, String value) {
		// // todo encoding
		getPointer().setString(index, value);
	}

	public void setWideString(int index, String value) {
		getPointer().setString(index, value, true);
	}

}
