/***************************************************************************

    Midway MCR-68k system

    driver by Bryan McPhail, Aaron Giles

    Games supported:
        * Zwackery (Chip Squeak Deluxe)
        * Xenopohobe (Sounds Good)
        * Spy Hunter 2 (Sounds Good/Turbo Chip Squeak)
        * Blasted (Sounds Good)
        * Arch Rivals
        * Tri-Sports
        * Pigskin 621AD

    Known bugs:
        * none at this time

    Clock Crystals:
        * Xenophobe
            - CPU/Video Board: XTAL 16.000MHz at 1D, XTAL 20.000MHz at 1F
            - Sound Board:     XTAL 16.000MHz

****************************************************************************

    Memory map

****************************************************************************

    ========================================================================
    CPU #1
    ========================================================================
    000000-03FFFF   R     xxxxxxxx xxxxxxxx    Program ROM
    060000-063FFF   R/W   xxxxxxxx xxxxxxxx    Program RAM
    070000-070FFF   R/W   xxxxxxxx xxxxxxxx    Background video RAM
    071000-071FFF   R/W   xxxxxxxx xxxxxxxx    Additional RAM
    080000-080FFF   R/W   xxxxxxxx xxxxxxxx    Sprite RAM
    090000-0900FF     W   xxxxxxxx xxxxxxxx    Palette RAM
    0A0000-0A000F   R/W   xxxxxxxx --------    M6840 I/O
    0B0000-0B0001     W   -------- --------    Watchdog
    0C0000-0C0001     W   -------- xxxxxxxx    Sound I/O
    0D0000-0D0001   R     xxxxxxxx xxxxxxxx    External inputs
    0E0000-0E0001   R     xxxxxxxx xxxxxxxx    External inputs
    0F0000-0F0001   R     xxxxxxxx xxxxxxxx    External inputs
    ========================================================================
    Interrupts:
        NMI ???
        INT generated by CTC
    ========================================================================
    NOTE: Pigskin and Tri-Sports have similar memory maps; the addresses
            are simply shuffled around
    ========================================================================

***************************************************************************/

#include "emu.h"
#include "cpu/m68000/m68000.h"
#include "audio/mcr.h"
#include "audio/williams.h"
#include "machine/nvram.h"
#include "includes/mcr68.h"



/*************************************
 *
 *  Zwackery-specific handlers
 *
 *************************************/

READ8_DEVICE_HANDLER( zwackery_port_2_r )
{
	int result = input_port_read(device->machine(), "IN2");
	int wheel = input_port_read(device->machine(), "IN5");

	return result | ((wheel >> 2) & 0x3e);
}


static READ16_HANDLER( zwackery_6840_r )
{
	/* Zwackery does a timer test:                          */
	/* It loads $1388 into one of the timers clocked by E   */
	/* Then it sits in a tight loop counting down from $4E4 */
	/*       BTST #$1,($2,A0)                               */
	/*       DBNE D1,*-6                                    */
	/* It expects D1 to end up between 0 and 5; in order to */
	/* make this happen, we must assume that reads from the */
	/* 6840 take 14 additional cycles                       */
	device_adjust_icount(&space->device(), -14);
	return mcr68_6840_upper_r(space,offset,0xffff);
}



/*************************************
 *
 *  Xenophobe-specific handlers
 *
 *************************************/

static WRITE16_HANDLER( xenophobe_control_w )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	COMBINE_DATA(&state->m_control_word);
/*  soundsgood_reset_w(~state->m_control_word & 0x0020);*/
	soundsgood_data_w(space, offset, ((state->m_control_word & 0x000f) << 1) | ((state->m_control_word & 0x0010) >> 4));
}



/*************************************
 *
 *  Blasted-specific handlers
 *
 *************************************/

static WRITE16_HANDLER( blasted_control_w )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	COMBINE_DATA(&state->m_control_word);
/*  soundsgood_reset_w(~state->m_control_word & 0x0020);*/
	soundsgood_data_w(space, offset, (state->m_control_word >> 8) & 0x1f);
}



/*************************************
 *
 *  Spy Hunter 2-specific handlers
 *
 *************************************/

static READ16_HANDLER( spyhunt2_port_0_r )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	static const char *const portnames[] = { "AN1", "AN2", "AN3", "AN4" };
	int result = input_port_read(space->machine(), "IN0");
	int which = (state->m_control_word >> 3) & 3;
	int analog = input_port_read(space->machine(), portnames[which]);

	return result | ((soundsgood_status_r(space, 0) & 1) << 5) | (analog << 8);
}


static READ16_HANDLER( spyhunt2_port_1_r )
{
	int result = input_port_read(space->machine(), "IN1");
	return result | ((turbocs_status_r(space, 0) & 1) << 7);
}


static WRITE16_HANDLER( spyhunt2_control_w )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	COMBINE_DATA(&state->m_control_word);

/*  turbocs_reset_w(~state->m_control_word & 0x0080);*/
	turbocs_data_w(space, offset, (state->m_control_word >> 8) & 0x001f);

	soundsgood_reset_w(space->machine(), ~state->m_control_word & 0x2000);
	soundsgood_data_w(space, offset, (state->m_control_word >> 8) & 0x001f);
}



/*************************************
 *
 *  Arch Rivals-specific handlers
 *
 *************************************/

/*
 *  Williams 49-way joystick
 *
 *  The joystick works on a 7x7 grid system:
 *
 *      + + + | + + +
 *      + + + | + + +
 *      + + + | + + +
 *      ------+------
 *      + + + | + + +
 *      + + + | + + +
 *      + + + | + + +
 *
 *  Each axis has 7 positions, reported as follows
 *  in 4 bits/axis:
 *
 *      0111 = left/up full
 *      0011 = left/up 2/3
 *      0001 = left/up 1/3
 *      0000 = center
 *      1100 = right/down 1/3
 *      1110 = right/down 2/3
 *      1111 = right/down full
 */

static const UINT8 translate49[7] = { 0x7, 0x3, 0x1, 0x0, 0xc, 0xe, 0xf };

static READ16_HANDLER( archrivl_port_1_r )
{
	return (translate49[input_port_read(space->machine(), "49WAYY2") >> 4] << 12) |
			(translate49[input_port_read(space->machine(), "49WAYX2") >> 4] << 8) |
			(translate49[input_port_read(space->machine(), "49WAYY1") >> 4] << 4) |
			(translate49[input_port_read(space->machine(), "49WAYX1") >> 4] << 0);
}


static WRITE16_HANDLER( archrivl_control_w )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	COMBINE_DATA(&state->m_control_word);
	williams_cvsd_reset_w(space->machine(), ~state->m_control_word & 0x0400);
	williams_cvsd_data_w(space->machine(), state->m_control_word & 0x3ff);
}



/*************************************
 *
 *  Pigskin-specific handlers
 *
 *************************************/

static WRITE16_HANDLER( pigskin_protection_w )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	/* ignore upper-byte only */
	if (ACCESSING_BITS_0_7)
	{
		/* track the last 5 bytes */
		state->m_protection_data[0] = state->m_protection_data[1];
		state->m_protection_data[1] = state->m_protection_data[2];
		state->m_protection_data[2] = state->m_protection_data[3];
		state->m_protection_data[3] = state->m_protection_data[4];
		state->m_protection_data[4] = data & 0xff;

		logerror("%06X:protection_w=%02X\n", cpu_get_previouspc(&space->device()), data & 0xff);
	}
}


static READ16_HANDLER( pigskin_protection_r )
{
	mcr68_state *state = space->machine().driver_data<mcr68_state>();
	/* based on the last 5 bytes return a value */
	if (state->m_protection_data[4] == 0xe3 && state->m_protection_data[3] == 0x94)
		return 0x00;	/* must be <= 1 */
	if (state->m_protection_data[4] == 0xc7 && state->m_protection_data[3] == 0x7b && state->m_protection_data[2] == 0x36)
		return 0x00;	/* must be <= 1 */
	if (state->m_protection_data[4] == 0xc7 && state->m_protection_data[3] == 0x7b)
		return 0x07;	/* must be > 5 */
	if (state->m_protection_data[4] == 0xc7 && state->m_protection_data[3] == 0x1f && state->m_protection_data[2] == 0x03 &&
		state->m_protection_data[1] == 0x25 && state->m_protection_data[0] == 0x36)
		return 0x00;	/* must be < 3 */

	logerror("Protection read after unrecognized sequence: %02X %02X %02X %02X %02X\n",
			state->m_protection_data[0], state->m_protection_data[1], state->m_protection_data[2], state->m_protection_data[3], state->m_protection_data[4]);

	return 0x00;
}


static READ16_HANDLER( pigskin_port_1_r )
{
	/* see archrivl_port_1_r for 49-way joystick description */
	return input_port_read(space->machine(), "IN1") |
			(translate49[input_port_read(space->machine(), "49WAYX1") >> 4] << 12) |
			(translate49[input_port_read(space->machine(), "49WAYY1") >> 4] << 8);
}


static READ16_HANDLER( pigskin_port_2_r )
{
	/* see archrivl_port_1_r for 49-way joystick description */
	return input_port_read(space->machine(), "DSW") |
			(translate49[input_port_read(space->machine(), "49WAYX2") >> 4] << 12) |
			(translate49[input_port_read(space->machine(), "49WAYY2") >> 4] << 8);
}



/*************************************
 *
 *  Tri-Sports-specific handlers
 *
 *************************************/

static READ16_HANDLER( trisport_port_1_r )
{
	int xaxis = (INT8)input_port_read(space->machine(), "AN1");
	int yaxis = (INT8)input_port_read(space->machine(), "AN2");
	int result = input_port_read(space->machine(), "IN1");

	result |= (xaxis & 0x3c) << 6;
	result |= (yaxis & 0x3c) << 10;

	return result;
}



/*************************************
 *
 *  Main CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( mcr68_map, AS_PROGRAM, 16 )
	ADDRESS_MAP_UNMAP_HIGH
	ADDRESS_MAP_GLOBAL_MASK(0x1fffff)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x060000, 0x063fff) AM_RAM
	AM_RANGE(0x070000, 0x070fff) AM_RAM_WRITE(mcr68_videoram_w) AM_BASE_MEMBER(mcr68_state, m_videoram)
	AM_RANGE(0x071000, 0x071fff) AM_RAM
	AM_RANGE(0x080000, 0x080fff) AM_RAM AM_BASE_SIZE_MEMBER(mcr68_state, m_spriteram, m_spriteram_size)
	AM_RANGE(0x090000, 0x09007f) AM_WRITE(mcr68_paletteram_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0x0a0000, 0x0a000f) AM_READWRITE(mcr68_6840_upper_r, mcr68_6840_upper_w)
	AM_RANGE(0x0b0000, 0x0bffff) AM_WRITE(watchdog_reset16_w)
	AM_RANGE(0x0d0000, 0x0dffff) AM_READ_PORT("IN0")
	AM_RANGE(0x0e0000, 0x0effff) AM_READ_PORT("IN1")
	AM_RANGE(0x0f0000, 0x0fffff) AM_READ_PORT("DSW")
ADDRESS_MAP_END



/*************************************
 *
 *  Zwackery main CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( zwackery_map, AS_PROGRAM, 16 )
	ADDRESS_MAP_UNMAP_HIGH
	AM_RANGE(0x000000, 0x037fff) AM_ROM
	AM_RANGE(0x080000, 0x080fff) AM_RAM
	AM_RANGE(0x084000, 0x084fff) AM_RAM
	AM_RANGE(0x100000, 0x10000f) AM_READWRITE(zwackery_6840_r, mcr68_6840_upper_w)
	AM_RANGE(0x104000, 0x104007) AM_DEVREADWRITE8_MODERN("pia0", pia6821_device, read, write, 0xff00)
	AM_RANGE(0x108000, 0x108007) AM_DEVREADWRITE8_MODERN("pia1", pia6821_device, read, write, 0x00ff)
	AM_RANGE(0x10c000, 0x10c007) AM_DEVREADWRITE8_MODERN("pia2", pia6821_device, read, write, 0x00ff)
	AM_RANGE(0x800000, 0x800fff) AM_RAM_WRITE(zwackery_videoram_w) AM_BASE_MEMBER(mcr68_state, m_videoram)
	AM_RANGE(0x802000, 0x803fff) AM_RAM_WRITE(zwackery_paletteram_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0xc00000, 0xc00fff) AM_RAM_WRITE(zwackery_spriteram_w) AM_BASE_SIZE_MEMBER(mcr68_state, m_spriteram, m_spriteram_size)
ADDRESS_MAP_END



/*************************************
 *
 *  Pigskin CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( pigskin_map, AS_PROGRAM, 16 )
	ADDRESS_MAP_UNMAP_HIGH
	ADDRESS_MAP_GLOBAL_MASK(0x1fffff)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x080000, 0x08ffff) AM_READ(pigskin_port_1_r)
	AM_RANGE(0x0a0000, 0x0affff) AM_READ(pigskin_port_2_r)
	AM_RANGE(0x0c0000, 0x0c007f) AM_WRITE(mcr68_paletteram_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0x0e0000, 0x0effff) AM_WRITE(watchdog_reset16_w)
	AM_RANGE(0x100000, 0x100fff) AM_RAM_WRITE(mcr68_videoram_w) AM_BASE_MEMBER(mcr68_state, m_videoram)
	AM_RANGE(0x120000, 0x120001) AM_READWRITE(pigskin_protection_r, pigskin_protection_w)
	AM_RANGE(0x140000, 0x143fff) AM_RAM
	AM_RANGE(0x160000, 0x1607ff) AM_RAM AM_BASE_SIZE_MEMBER(mcr68_state, m_spriteram, m_spriteram_size)
	AM_RANGE(0x180000, 0x18000f) AM_READWRITE(mcr68_6840_upper_r, mcr68_6840_upper_w)
	AM_RANGE(0x1a0000, 0x1affff) AM_WRITE(archrivl_control_w)
	AM_RANGE(0x1e0000, 0x1effff) AM_READ_PORT("IN0")
ADDRESS_MAP_END



/*************************************
 *
 *  Tri-Sports CPU memory handlers
 *
 *************************************/

static ADDRESS_MAP_START( trisport_map, AS_PROGRAM, 16 )
	ADDRESS_MAP_UNMAP_HIGH
	ADDRESS_MAP_GLOBAL_MASK(0x1fffff)
	AM_RANGE(0x000000, 0x03ffff) AM_ROM
	AM_RANGE(0x080000, 0x08ffff) AM_READ(trisport_port_1_r)
	AM_RANGE(0x0a0000, 0x0affff) AM_READ_PORT("DSW")
	AM_RANGE(0x100000, 0x103fff) AM_RAM AM_SHARE("nvram")
	AM_RANGE(0x120000, 0x12007f) AM_WRITE(mcr68_paletteram_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0x140000, 0x1407ff) AM_RAM AM_BASE_SIZE_MEMBER(mcr68_state, m_spriteram, m_spriteram_size)
	AM_RANGE(0x160000, 0x160fff) AM_RAM_WRITE(mcr68_videoram_w) AM_BASE_MEMBER(mcr68_state, m_videoram)
	AM_RANGE(0x180000, 0x18000f) AM_READWRITE(mcr68_6840_upper_r, mcr68_6840_upper_w)
	AM_RANGE(0x1a0000, 0x1affff) AM_WRITE(archrivl_control_w)
	AM_RANGE(0x1c0000, 0x1cffff) AM_WRITE(watchdog_reset16_w)
	AM_RANGE(0x1e0000, 0x1effff) AM_READ_PORT("IN0")
ADDRESS_MAP_END



/*************************************
 *
 *  Port definitions
 *
 *************************************/

static INPUT_PORTS_START( zwackery )
	PORT_START("IN0")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE( 0x0010, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* sword */

	PORT_START("IN1")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_SPECIAL )	/* sound communications */

	PORT_START("IN2")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* spell up */
	PORT_BIT( 0x3e, IP_ACTIVE_HIGH, IPT_UNUSED )	/* encoder wheel */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* shield */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 )	/* spell down */

	PORT_START("IN3")
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW")
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 6C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x38, 0x00, "Buy-in" )
	PORT_DIPSETTING(    0x00, "1 coin" )
	PORT_DIPSETTING(    0x08, "2 coins" )
	PORT_DIPSETTING(    0x10, "3 coins" )
	PORT_DIPSETTING(    0x18, "4 coins" )
	PORT_DIPSETTING(    0x20, "5 coins" )
	PORT_DIPSETTING(    0x28, "6 coins" )
	PORT_DIPSETTING(    0x30, "7 coins" )
	PORT_DIPSETTING(    0x38, DEF_STR( None ) )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( Easier ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Harder ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Hardest ) )

	PORT_START("IN5")
	PORT_BIT( 0xff, 0x00, IPT_DIAL ) PORT_SENSITIVITY(50) PORT_KEYDELTA(10) PORT_CODE_DEC(KEYCODE_Z) PORT_CODE_INC(KEYCODE_X) PORT_REVERSE
INPUT_PORTS_END


static INPUT_PORTS_START( xenophob )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0018, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW")
	PORT_BIT( 0x0003, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Coins per Life Unit" )
	PORT_DIPSETTING(      0x0008, "1" )
	PORT_DIPSETTING(      0x0000, "2" )
	PORT_DIPNAME( 0x0010, 0x0010, "Life Unit" )
	PORT_DIPSETTING(      0x0010, "1000" )
	PORT_DIPSETTING(      0x0000, "2000" )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x00c0, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Easy ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Medium ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Hard ) )
	PORT_DIPSETTING(      0x00c0, "Medium (duplicate)" )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


static INPUT_PORTS_START( spyhunt2 )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_SPECIAL ) /* SG status */
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNKNOWN ) /* Oddly enough, if you assign this control to a key, it makes both player wheels go left to fifteen */

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(2) PORT_NAME ("P2 1st Gear")/* 1st gear */
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(2) PORT_NAME ("P2 2nd Gear")/* 2nd gear */
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_PLAYER(2) PORT_NAME ("P2 3rd Gear")/* 3rd gear */
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(1) PORT_NAME ("P1 1st Gear")/* 1st gear */
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(1) PORT_NAME ("P1 2nd Gear")/* 2nd gear */
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_PLAYER(1) PORT_NAME ("P1 3rd Gear")/* 3rd gear */
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_SPECIAL )               /* TCS status */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2) PORT_NAME ("P2 L Trigger")/* Left Trigger */
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2) PORT_NAME ("P2 L Button")/* Left Button */
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2) PORT_NAME ("P2 R Trigger")/* Right Trigger */
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(2) PORT_NAME ("P2 R Button")/* Right Button */
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1) PORT_NAME ("P1 L Trigger")/* Left Trigger */
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1) PORT_NAME ("P1 L Button")/* Left Button */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1) PORT_NAME ("P1 R Trigger")/* Right Trigger */
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(1) PORT_NAME ("P1 R Button")/* Right Button */

	PORT_START("DSW")	/* dipswitches */
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0000, "1C/2C (duplicate)" )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0018, 0x0008, "Point Threshholds" )
	PORT_DIPSETTING(      0x0008, DEF_STR( Easy ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( Medium ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( Hard ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x0060, 0x0060, "Free Timer After" )
	PORT_DIPSETTING(      0x0000, "30 sec" )
	PORT_DIPSETTING(      0x0040, "45 sec" )
	PORT_DIPSETTING(      0x0060, "60 sec" )
	PORT_DIPSETTING(      0x0020, "90 sec" )
	PORT_DIPNAME( 0x0080, 0x0080, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN1")	/* analog ports for steering and pedals */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("AN2")
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(0x30,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)

	PORT_START("AN3")
	PORT_BIT( 0xff, 0x80, IPT_PADDLE ) PORT_MINMAX(0x10,0xf0) PORT_SENSITIVITY(80) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("AN4")
	PORT_BIT( 0xff, 0x80, IPT_PADDLE ) PORT_MINMAX(0x10,0xf0) PORT_SENSITIVITY(80) PORT_KEYDELTA(10) PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END


static INPUT_PORTS_START( blasted )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED/* credit w/bill */ )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED/* credit 1 w/bill */ )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED/* credit 2 w/bill */ )
	PORT_BIT( 0xf000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("DSW")
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0000, "1C/2C (duplicate)" )
	PORT_DIPNAME( 0x000c, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Easy ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Medium ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Hard ) )
	PORT_DIPSETTING(      0x000c, "Medium (duplicate)" )
	PORT_DIPNAME( 0x0020, 0x0020, "Dollar Receptor" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Rack Advance (Cheat)" ) PORT_CODE(KEYCODE_F1)
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

static INPUT_PORTS_START( intlaser )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(4)
	PORT_BIT( 0xf000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(4)

	PORT_START("DSW")
    PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
    PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unknown ) )
    PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
    PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


static INPUT_PORTS_START( archrivl )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)

	PORT_START("IN1")
	PORT_BIT( 0xffff, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1/2 joysticks go here */

	PORT_START("DSW")
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Game_Time ) )
	PORT_DIPSETTING(      0x0003, "Preset Time" )
	PORT_DIPSETTING(      0x0002, "Preset + 10sec" )
	PORT_DIPSETTING(      0x0001, "Preset + 20sec" )
	PORT_DIPSETTING(      0x0000, "Preset + 30sec" )
	PORT_DIPNAME( 0x001c, 0x001c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0014, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x001c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Team Names" )
	PORT_DIPSETTING(      0x0020, "Default" )
	PORT_DIPSETTING(      0x0000, "Hometown Heroes" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("49WAYX1")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_X ) PORT_PLAYER(1) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("49WAYY1")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_Y ) PORT_PLAYER(1) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("49WAYX2")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_X ) PORT_PLAYER(2) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("49WAYY2")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_Y ) PORT_PLAYER(2) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)
INPUT_PORTS_END


static INPUT_PORTS_START( pigskin )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN1")
	PORT_BIT( 0x000f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1 joystick goes here */

	PORT_START("DSW")
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Game_Time ) )
	PORT_DIPSETTING(      0x0000, "Shortest" )
	PORT_DIPSETTING(      0x0002, "Short" )
	PORT_DIPSETTING(      0x0003, DEF_STR( Medium ) )
	PORT_DIPSETTING(      0x0001, "Long" )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, "Set Your Own" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Test Switch" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Coin Chutes" )
	PORT_DIPSETTING(      0x0000, "Individual" )
	PORT_DIPSETTING(      0x0040, "Common" )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Joystick ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Standard ) )
	PORT_DIPSETTING(      0x0000, "Rotated" )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 2 joystick goes here */

	PORT_START("49WAYX1")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_X ) PORT_PLAYER(1) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE

	PORT_START("49WAYY1")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_Y ) PORT_PLAYER(1) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)

	PORT_START("49WAYX2")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_X ) PORT_PLAYER(2) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10) PORT_REVERSE

	PORT_START("49WAYY2")	/* converted to standard 49-way inputs */
	PORT_BIT( 0xff, 0x38, IPT_AD_STICK_Y ) PORT_PLAYER(2) PORT_MINMAX(0x00,0x6f) PORT_SENSITIVITY(100) PORT_KEYDELTA(10)
INPUT_PORTS_END


static INPUT_PORTS_START( trisport )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0f00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("IN1")
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* analog controls go here */

	PORT_START("DSW")
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0000, "Battery Options" )
	PORT_DIPNAME( 0x0018, 0x0018, "Pool Turns" )
	PORT_DIPSETTING(      0x0010, "5" )
	PORT_DIPSETTING(      0x0008, "6" )
	PORT_DIPSETTING(      0x0018, "7" )
	PORT_DIPSETTING(      0x0000, "8" )
	PORT_DIPNAME( 0x0020, 0x0020, "Bowling Difficulty" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Standard ) )
	PORT_DIPSETTING(      0x0000, "Advanced" )
	PORT_DIPNAME( 0x0040, 0x0040, "Shot Timer" )
	PORT_DIPSETTING(      0x0000, "Slower" )
	PORT_DIPSETTING(      0x0040, DEF_STR( Standard ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Golf Holes" )
	PORT_DIPSETTING(      0x0080, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("AN1")
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_X ) PORT_SENSITIVITY(100) PORT_KEYDELTA(20) PORT_PLAYER(1)

	PORT_START("AN2")
    PORT_BIT( 0xff, 0, IPT_TRACKBALL_Y ) PORT_SENSITIVITY(100) PORT_KEYDELTA(20) PORT_REVERSE PORT_PLAYER(1)
INPUT_PORTS_END



/*************************************
 *
 *  Graphics definitions
 *
 *************************************/

static const gfx_layout mcr68_bg_layout =
{
	8,8,
	RGN_FRAC(1,2),
	4,
	{ STEP2(RGN_FRAC(1,2),1), STEP2(RGN_FRAC(0,2),1) },
	{ STEP8(0,2) },
	{ STEP8(0,16) },
	16*8
};


static const gfx_layout mcr68_sprite_layout =
{
	32,32,
	RGN_FRAC(1,4),
	4,
	{ STEP4(0,1) },
	{ STEP2(RGN_FRAC(0,4)+0,4), STEP2(RGN_FRAC(1,4)+0,4), STEP2(RGN_FRAC(2,4)+0,4), STEP2(RGN_FRAC(3,4)+0,4),
	  STEP2(RGN_FRAC(0,4)+8,4), STEP2(RGN_FRAC(1,4)+8,4), STEP2(RGN_FRAC(2,4)+8,4), STEP2(RGN_FRAC(3,4)+8,4),
	  STEP2(RGN_FRAC(0,4)+16,4), STEP2(RGN_FRAC(1,4)+16,4), STEP2(RGN_FRAC(2,4)+16,4), STEP2(RGN_FRAC(3,4)+16,4),
	  STEP2(RGN_FRAC(0,4)+24,4), STEP2(RGN_FRAC(1,4)+24,4), STEP2(RGN_FRAC(2,4)+24,4), STEP2(RGN_FRAC(3,4)+24,4) },
	{ STEP32(0,32) },
	32*32
};

static const gfx_layout zwackery_layout =
{
	16,16,
	RGN_FRAC(1,2),
	8,
	{ 0, 0, 0, 0, 0, 0, 0, 0 },
	{ STEP4(3,-1), STEP4(11,-1), STEP4(19,-1), STEP4(27,-1) },
	{ 4, RGN_FRAC(1,2)+4, 0, RGN_FRAC(1,2)+0, 36, RGN_FRAC(1,2)+36, 32, RGN_FRAC(1,2)+32,
	  68, RGN_FRAC(1,2)+68, 64, RGN_FRAC(1,2)+64, 100, RGN_FRAC(1,2)+100, 96, RGN_FRAC(1,2)+96 },
	128
};

static GFXDECODE_START( mcr68 )
	GFXDECODE_SCALE( "gfx1", 0, mcr68_bg_layout,     0, 4, 2, 2 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr68_sprite_layout, 0, 4 )
GFXDECODE_END

static GFXDECODE_START( zwackery )
	GFXDECODE_ENTRY( "gfx1", 0, zwackery_layout,       0, 16 )
	GFXDECODE_ENTRY( "gfx2", 0, mcr68_sprite_layout, 0x800, 32 )
	GFXDECODE_ENTRY( "gfx1", 0, zwackery_layout,       0, 16 )	/* yes, an extra copy */
GFXDECODE_END



/*************************************
 *
 *  Machine drivers
 *
 *************************************/

/*=================================================================

    Timing for these games is crucial. They all use the 6840 timer
    to preemptively multitask during each frame. The clock on the
    6840 timer is taken from the 68000's E clock, which runs at
    1/10th the speed of the 68000 itself.

    All the games run in a sequence of six steps per frame, using
    counter 1 on the 6840 to time each step. The sum total of the
    6 programmed steps for each game determines how many E clocks
    should be generated per frame, which in turn determines the
    clock we should expect the CPU to have.

    Ideal CPU timings, based on counter usage:

        Zwackery:     7652400
        Xenophobe:    7723800
        Spy Hunter 2: 7723800
        Blasted:      7798800
        Arch Rivals:  7799100
        Pigskin:      9211200
        Tri-Sports:   9211200

    Currently, we are using the Xenophobe CPU for the first four
    until we spot problems.

=================================================================*/

static MACHINE_CONFIG_START( zwackery, mcr68_state )

	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M68000, 7652400)	/* should be XTAL_16MHz/2 */
	MCFG_CPU_PROGRAM_MAP(zwackery_map)
	MCFG_CPU_VBLANK_INT("screen", mcr68_interrupt)

//  MCFG_WATCHDOG_VBLANK_INIT(8)
	MCFG_MACHINE_START(zwackery)
	MCFG_MACHINE_RESET(zwackery)

	MCFG_PIA6821_ADD("pia0", zwackery_pia0_intf)
	MCFG_PIA6821_ADD("pia1", zwackery_pia1_intf)
	MCFG_PIA6821_ADD("pia2", zwackery_pia2_intf)

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(30)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(2500) /* not accurate */)
	MCFG_SCREEN_SIZE(32*16, 30*16)
	MCFG_SCREEN_VISIBLE_AREA(0, 32*16-1, 0, 30*16-1)
	MCFG_SCREEN_UPDATE_STATIC(zwackery)

	MCFG_GFXDECODE(zwackery)
	MCFG_PALETTE_LENGTH(4096)

	MCFG_VIDEO_START(zwackery)

	/* sound hardware */
	MCFG_FRAGMENT_ADD(chip_squeak_deluxe)
MACHINE_CONFIG_END


static MACHINE_CONFIG_START( mcr68, mcr68_state )

	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M68000, 7723800)
	MCFG_CPU_PROGRAM_MAP(mcr68_map)
	MCFG_CPU_VBLANK_INT("screen", mcr68_interrupt)

	MCFG_WATCHDOG_VBLANK_INIT(8)
	MCFG_MACHINE_START(mcr68)
	MCFG_MACHINE_RESET(mcr68)

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(30)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(2500) /* not accurate */)
	MCFG_SCREEN_SIZE(32*16, 30*16)
	MCFG_SCREEN_VISIBLE_AREA(0, 32*16-1, 0, 30*16-1)
	MCFG_SCREEN_UPDATE_STATIC(mcr68)

	MCFG_GFXDECODE(mcr68)
	MCFG_PALETTE_LENGTH(64)

	MCFG_VIDEO_START(mcr68)

	/* sound hardware -- determined by specific machine */
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( xenophob, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(sounds_good)
MACHINE_CONFIG_END

static MACHINE_CONFIG_DERIVED( intlaser, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(sounds_good)

	MCFG_WATCHDOG_VBLANK_INIT(800)
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( spyhunt2, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(sounds_good)
	MCFG_DEVICE_REMOVE("mono")
	MCFG_FRAGMENT_ADD(turbo_chip_squeak)
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( archrivl, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(williams_cvsd_sound)
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( pigskin, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(williams_cvsd_sound)

	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_PROGRAM_MAP(pigskin_map)
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( trisport, mcr68 )

	/* basic machine hardware */
	MCFG_FRAGMENT_ADD(williams_cvsd_sound)

	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_PROGRAM_MAP(trisport_map)

	MCFG_NVRAM_ADD_0FILL("nvram")
MACHINE_CONFIG_END



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( zwackery )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pro0.bin",   0x00000, 0x4000, CRC(6fb9731c) SHA1(ee5b297ef2b4cf20df5e776f1c585b51f174bfa7) )
	ROM_LOAD16_BYTE( "pro1.bin",   0x00001, 0x4000, CRC(84b92555) SHA1(9b4af81374828c1742c1e13fc425eea2973b0867) )
	ROM_LOAD16_BYTE( "pro2.bin",   0x08000, 0x4000, CRC(e6977a2a) SHA1(602bf3f7e0f4080cb5b72d8fd3ee9fd11f27c558) )
	ROM_LOAD16_BYTE( "pro3.bin",   0x08001, 0x4000, CRC(f5d0a60e) SHA1(7e0e4936cb37ac16d6db5533ae4aecdfb07ead93) )
	ROM_LOAD16_BYTE( "pro4.bin",   0x10000, 0x4000, CRC(ec5841d9) SHA1(4bafe614e8993994b0ea9aedc8dc2474361e4594) )
	ROM_LOAD16_BYTE( "pro5.bin",   0x10001, 0x4000, CRC(d7d99ce0) SHA1(fdf428ab9c96dae555d49bac47495613ba265452) )
	ROM_LOAD16_BYTE( "pro6.bin",   0x18000, 0x4000, CRC(b9fe7bf5) SHA1(a94f80f49b4520a2c1098eee8983560b4ecdf3d5) )
	ROM_LOAD16_BYTE( "pro7.bin",   0x18001, 0x4000, CRC(5e261b3b) SHA1(dcf99f528c9e3b4f8b52d413c088559bfb37d733) )
	ROM_LOAD16_BYTE( "pro8.bin",   0x20000, 0x4000, CRC(55e380a5) SHA1(e3fef8486858cd714086449327a93b4a70ae73ff) )
	ROM_LOAD16_BYTE( "pro9.bin",   0x20001, 0x4000, CRC(12249dca) SHA1(154170286047ea78645d45dfdd895a597dad17da) )
	ROM_LOAD16_BYTE( "pro10.bin",  0x28000, 0x4000, CRC(6a39a8ca) SHA1(8ac9c3e60dc6f1918bfb95acf3ee170cedfb20ea) )
	ROM_LOAD16_BYTE( "pro11.bin",  0x28001, 0x4000, CRC(ad6b45bc) SHA1(118496e898654b028f008a3d493e693ba000ef38) )
	ROM_LOAD16_BYTE( "pro12.bin",  0x30000, 0x4000, CRC(e2d25e1f) SHA1(5d8ff303441eccf431422b453a173983a4513630) )
	ROM_LOAD16_BYTE( "pro13.bin",  0x30001, 0x4000, CRC(e131f9b8) SHA1(08b131f2acc84d4c2c931bfd24e7de3d92a8a817) )

	ROM_REGION( 0x20000, "csdcpu", 0 )
	ROM_LOAD16_BYTE( "csd7.bin",  0x00000, 0x2000, CRC(5501f54b) SHA1(84c0851fb868e81400cfe3ebfd7b91fe98a47bac) )
	ROM_LOAD16_BYTE( "csd17.bin", 0x00001, 0x2000, CRC(2e482580) SHA1(92bd3e64ff580800ee16579d97bcb8b3bd9f755c) )
	ROM_LOAD16_BYTE( "csd8.bin",  0x04000, 0x2000, CRC(13366575) SHA1(bcf25a7d4c6b2ccd7cd9978edafc66ef0cadfe72) )
	ROM_LOAD16_BYTE( "csd18.bin", 0x04001, 0x2000, CRC(bcfe5820) SHA1(ca32daa645851a2373b3cdb8a5e63ebda84aa762) )

	ROM_REGION( 0x8000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "tileh.bin",    0x00000, 0x4000, CRC(a7237eb1) SHA1(197e5838ac2bc732ae9eb33a9257b9391d50abf8) )
	ROM_LOAD( "tileg.bin",    0x04000, 0x4000, CRC(626cc69b) SHA1(86142bafa78f45d1a0bed0b83f3558b21384fa1a) )

	ROM_REGION( 0x20000, "gfx2", 0 )
	ROM_LOAD( "spr6h.bin",    0x00000, 0x4000, CRC(a51158dc) SHA1(8d3b0054950443fdf57f83dcb973d05f6c7ad9c8) )
	ROM_LOAD( "spr7h.bin",    0x04000, 0x4000, CRC(941feecf) SHA1(8e88c956332e78dc7e55139879f2272116415714) )
	ROM_LOAD( "spr6j.bin",    0x08000, 0x4000, CRC(f3eef316) SHA1(026e18bdfdda8cc9d0774e6d9d758686bf16992c) )
	ROM_LOAD( "spr7j.bin",    0x0c000, 0x4000, CRC(a8a34033) SHA1(abd9fde84bb079c84126ad04d584ec03b44b60cd) )
	ROM_LOAD( "spr10h.bin",   0x10000, 0x4000, CRC(a99daea6) SHA1(c323e05f398b7e9e04b75fd8ac5e8ab675236d66) )
	ROM_LOAD( "spr11h.bin",   0x14000, 0x4000, CRC(c1a767fb) SHA1(c16e09b39b09d409b534ce4c53366e43237a3759) )
	ROM_LOAD( "spr10j.bin",   0x18000, 0x4000, CRC(4dd04376) SHA1(069b64397e7a961c1fc246671472f759bd9f6c03) )
	ROM_LOAD( "spr11j.bin",   0x1c000, 0x4000, CRC(e8c6a880) SHA1(dd3d52ddbc36e244b96cfb87e6a80adb94626407) )

	ROM_REGION( 0x8000, "gfx3", 0 )	/* bg color maps */
	ROM_LOAD16_BYTE( "tilef.bin",  0x0000, 0x4000, CRC(a0dfcd7e) SHA1(0fc6723eddef2a96de9bf1f48006dd067c148540) )
	ROM_LOAD16_BYTE( "tilee.bin",  0x0001, 0x4000, CRC(ab504dc8) SHA1(4ebdcd42624e94c29ccdb8247bfff2d8e936ddd7) )

	ROM_REGION( 0x000D, "plds", 0 )
	/* According to the manual these pal's are located on the "Venus CPU" board */
	ROM_LOAD( "pal.d5",    0x0000, 0x00001, NO_DUMP ) /* marked H-T in manual */
	ROM_LOAD( "pal.d2",    0x0001, 0x00001, NO_DUMP ) /* marked V-T in manual */
	ROM_LOAD( "pal.d4",    0x0002, 0x00001, NO_DUMP ) /* marked MISC V&H PAL in manual */
	ROM_LOAD( "pal.d3",    0x0003, 0x00001, NO_DUMP ) /* marked MISC CUSTOM PAL in manual */
	ROM_LOAD( "pal.e6",    0x0004, 0x00001, NO_DUMP ) /* marked CPU WTS PAL in manual*/
	ROM_LOAD( "pal.f8",    0x0005, 0x00001, NO_DUMP ) /* marked CPU IOC PAL in manual*/
	ROM_LOAD( "pal.a5",    0x0006, 0x00001, NO_DUMP ) /* marked CPU RMD PAL in manual*/
	/* According to the manual these pal's are located on the "Venus VIDEO" board */
	ROM_LOAD( "pal.1f",    0x0007, 0x00001, NO_DUMP ) /* marked PAL FGBDCD in manual*/
	ROM_LOAD( "pal.1d",    0x0008, 0x00001, NO_DUMP ) /* marked PAL HCT in manual*/
	/* According to the manual these pal's are located on the "Venus BACKGROUND" board */
	ROM_LOAD( "pal.1c",    0x0009, 0x00001, NO_DUMP ) /* marked BGBPE PAL in manual*/
	ROM_LOAD( "pal.5c",    0x000a, 0x00001, NO_DUMP ) /* marked HCT PAL in manual*/
	ROM_LOAD( "pal.5j",    0x000b, 0x00001, NO_DUMP ) /* marked BGBDCD PAL in manual*/
	/* According to the manual this pal is located on the "Artificial Artist" board */
	ROM_LOAD( "pal20.u15", 0x000c, 0x00001, NO_DUMP ) /* marked CSD002R0 in manual, pal type not specified */
ROM_END

/*
    Xenophobe

    CPU/Video Board: A080-91871-E000
    Sound Board:     A080-91863-B000
*/

ROM_START( xenophob ) /* Service mode shows "VERSION CO" */
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "xeno_pro.3c",  0x00000, 0x10000, CRC(f44c2e60) SHA1(9130b26eb1e0e0a75f9fdec898e1f0976de8a766) )
	ROM_LOAD16_BYTE( "xeno_pro.3b",  0x00001, 0x10000, CRC(01609a3b) SHA1(9e065bc72f56439a885bfdfc8eb60df666df7c37) )
	ROM_LOAD16_BYTE( "xeno_pro.2c",  0x20000, 0x10000, CRC(e45bf669) SHA1(52b0ffd2311e4d300410de57fbddacab4b9857a1) )
	ROM_LOAD16_BYTE( "xeno_pro.2b",  0x20001, 0x10000, CRC(da5d39d5) SHA1(f61b239eb3108faec2f3dbb8139c8d01b0e29873) )

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "xeno_snd.u7",  0x00000, 0x10000, CRC(77561d15) SHA1(8c23a9270d54be6380f2d23939b6c6d8c31e334b) )
	ROM_LOAD16_BYTE( "xeno_snd.u17", 0x00001, 0x10000, CRC(837a1a71) SHA1(d7d60ef1fd11e5e84dd1ffb9a077686bd2fb452e) )
	ROM_LOAD16_BYTE( "xeno_snd.u8",  0x20000, 0x10000, CRC(6e2915c7) SHA1(df1f35f6b743afbab0a3a29adce3639a8c9dc66f) )
	ROM_LOAD16_BYTE( "xeno_snd.u18", 0x20001, 0x10000, CRC(12492145) SHA1(c891e86750e51d3b7b93fc96a9f5173fd7869756) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "xeno_bg.11d",  0x00000, 0x08000, CRC(3d2cf284) SHA1(39a666197584f24d19d43634b8e2f3923081a336) )
	ROM_LOAD( "xeno_bg.12d",  0x08000, 0x08000, CRC(c32288b1) SHA1(a9f9f61c4287cb1e5490b4b7ad4fd94bba1c8fac) )

	ROM_REGION( 0x40000, "gfx2", 0 )
	ROM_LOAD( "xeno_fg.7j",   0x00000, 0x10000, CRC(b12eddb2) SHA1(a10df077d60cd821acbdbd32abe6dafd1217737c) )
	ROM_LOAD( "xeno_fg.8j",   0x10000, 0x10000, CRC(20e682f5) SHA1(1009f7ec56998df8a1d5ecd724d0523c435c9ee0) )
	ROM_LOAD( "xeno_fg.9j",   0x20000, 0x10000, CRC(82fb3e09) SHA1(f06e9df20044244a6c174f4876e615ccc18e1cba) )
	ROM_LOAD( "xeno_fg.10j",  0x30000, 0x10000, CRC(6a7a3516) SHA1(1def9c134220eac9ba5e46d38282ff18f51b6398) )

	/* PLD's located on the cpu/video board */
	ROM_REGION( 0x000C, "cpu_plds", 0 )
	ROM_LOAD( "b61a-49aaj-axad.bin", 0x00000, 0x00001, NO_DUMP ) /* PAL20L8 at 9B */
	ROM_LOAD( "b75a-50aaj-bxad.bin", 0x00001, 0x00001, NO_DUMP ) /* PAL16L8 at 1J */
	ROM_LOAD( "b75a-50aaj-axad.bin", 0x00002, 0x00001, NO_DUMP ) /* PAL16L8 at 2J */
	ROM_LOAD( "b75a-41aaj-axad.bin", 0x00003, 0x00001, NO_DUMP ) /* PAL16R4 at 2K */
	ROM_LOAD( "b75a-41aaj-bxab.bin", 0x00004, 0x00001, NO_DUMP ) /* PAL16R4 at 14K */
	ROM_LOAD( "a59a26axlaxhd.bin",   0x00005, 0x00001, NO_DUMP ) /* PLS153 at 11J */
	ROM_LOAD( "a59a26axlbxhd.bin",   0x00006, 0x00001, NO_DUMP ) /* PLS153 at 12J */
	ROM_LOAD( "a59a26axlcxhd.bin",   0x00007, 0x00001, NO_DUMP ) /* PLS153 at 14H */
	ROM_LOAD( "0066-316bx-xxqx.bin", 0x00008, 0x00001, NO_DUMP ) /* 20 Pin PLD? at 14E */
	ROM_LOAD( "0066-314bx-xxqx.bin", 0x00009, 0x00001, NO_DUMP ) /* 24 Pin PLD? at 14F */
	ROM_LOAD( "0066-315bx-xxqx.bin", 0x0000A, 0x00001, NO_DUMP ) /* 20 Pin PLD? at 15E */
	ROM_LOAD( "0066-313bx-xxqx.bin", 0x0000B, 0x00001, NO_DUMP ) /* 24 Pin PLD? at 15F */

	/* PLD located on the "Sounds Good" board */
	ROM_REGION( 0x0001, "snd_pld", 0 )
	ROM_LOAD( "e36a31axnax00.bin",   0x00000, 0x00001, NO_DUMP ) /* PAL20L10 at U15 */
ROM_END


ROM_START( spyhunt2 )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "sh23c.bin",  0x00000, 0x10000, CRC(30b91c90) SHA1(5b76f4e512b17ee80de2694807aa4e2499c2ef8b) )
	ROM_LOAD16_BYTE( "sh23b.bin",  0x00001, 0x10000, CRC(f64513c6) SHA1(e42cab599e489a0ba422b28c5cfda0f9c3a60601) )
	ROM_LOAD16_BYTE( "sh22c.bin",  0x20000, 0x10000, CRC(8ee65009) SHA1(6adb00888f739b59e3ace1a6eaf1c58c4583d7fd) )
	ROM_LOAD16_BYTE( "sh22b.bin",  0x20001, 0x10000, CRC(850c21ad) SHA1(3b944545cb469e2c53166a91eb2834c5f3891ddf) )

	ROM_REGION( 0x10000, "tcscpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbo-cs.u5", 0x08000, 0x4000, CRC(4b1d8a66) SHA1(a1a2f9fe3fc42b668ec97ad6c6ea6032f1dc0695) )
	ROM_LOAD( "turbo-cs.u4", 0x0c000, 0x4000, CRC(3722ce48) SHA1(ae064be590c067bda66ca7a72c212ad47f3eb1c5) )

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "sh2u7.bin",  0x00000, 0x10000, CRC(02362ea4) SHA1(2d37f06c9156554b8140ed565f6fdd1ef67bb54f) )
	ROM_LOAD16_BYTE( "sh2u17.bin", 0x00001, 0x10000, CRC(e29a2c37) SHA1(e0d4df90b533d3325c905d42ddc6876667f32c82) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "sh2bg0.bin",  0x00000, 0x08000, CRC(cb3c3d8e) SHA1(5135eefd311e21b62628a6b6dc483ff80e5594fd) )
	ROM_LOAD( "sh2bg1.bin",  0x08000, 0x08000, CRC(029d4af1) SHA1(9c2afa7b48f27d09d0e5487ec2a54f903e83c91c) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "fg0.7j",   0x00000, 0x20000, CRC(55ce12ea) SHA1(db8677e05009db6c79a727f42f35d60f4a716fc2) )
	ROM_LOAD( "fg1.8j",   0x20000, 0x20000, CRC(692afb67) SHA1(5669298a646deb2f82e438ae52de03f81a9e11a7) )
	ROM_LOAD( "fg2.9j",   0x40000, 0x20000, CRC(f1aba383) SHA1(56d1f7e9eb430671076415dd87fe77a38fadad84) )
	ROM_LOAD( "fg3.10j",  0x60000, 0x20000, CRC(d3475ff8) SHA1(aa7a283a190a6c43e365fcd9242c5d0b920dbf32) )

	ROM_REGION( 0x0006, "plds", 0 )
	/* According to the manual these pal's are located on the Video Game board */
	ROM_LOAD( "pal20l8.9b",   0x00000, 0x00001, NO_DUMP ) /* marked COLARB in manual */
	ROM_LOAD( "pal16l8.1j",   0x00001, 0x00001, NO_DUMP ) /* marked IODCD in manual */
	ROM_LOAD( "pal16l8.2j",   0x00002, 0x00001, NO_DUMP ) /* marked MEMDCD in manual */
	ROM_LOAD( "pal16r4.2k",   0x00003, 0x00001, NO_DUMP ) /* marked DTACK in manual */
	ROM_LOAD( "pal16r4.14k",  0x00004, 0x00001, NO_DUMP ) /* marked HSYNC in manual*/
	/* According to the manual this pal is located on the "Sounds Good" board */
	ROM_LOAD( "pal20.u15",    0x00005, 0x00001, NO_DUMP ) /* marked SG01R0 in manual, pal type not specified */
ROM_END


ROM_START( spyhunt2a )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "3c",  0x00000, 0x10000, CRC(5b92aadf) SHA1(0d0e2606fc5346e396f0a7b8ceb190ee42e485c3) )
	ROM_LOAD16_BYTE( "3b",  0x00001, 0x10000, CRC(6ed0a25f) SHA1(542f77889b0cfdfeeff47e5beaef97c7516b77e1) )
	ROM_LOAD16_BYTE( "2c",  0x20000, 0x10000, CRC(bc834f3f) SHA1(05f6ab508ce2ebe55665e97114070e9d81db48c8) )
	ROM_LOAD16_BYTE( "2b",  0x20001, 0x10000, CRC(8a9f7ef3) SHA1(353ebb0a3782c183cc9be800584903e23ca507d9) )

	ROM_REGION( 0x10000, "tcscpu", 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbo-cs.u5", 0x08000, 0x4000, CRC(4b1d8a66) SHA1(a1a2f9fe3fc42b668ec97ad6c6ea6032f1dc0695) )
	ROM_LOAD( "turbo-cs.u4", 0x0c000, 0x4000, CRC(3722ce48) SHA1(ae064be590c067bda66ca7a72c212ad47f3eb1c5) )

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "sh2u7.bin",  0x00000, 0x10000, CRC(02362ea4) SHA1(2d37f06c9156554b8140ed565f6fdd1ef67bb54f) )
	ROM_LOAD16_BYTE( "sh2u17.bin", 0x00001, 0x10000, CRC(e29a2c37) SHA1(e0d4df90b533d3325c905d42ddc6876667f32c82) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "bg0.11d",  0x00000, 0x08000, CRC(81efef7a) SHA1(74a6757b374b9f1e0c5a33a13fa492ae2e4347a0) )
	ROM_LOAD( "bg1.12d",  0x08000, 0x08000, CRC(6a902e4d) SHA1(0c7f1bcd63f48cf1ad58f8abdcfe9fe9bc25acbc) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "fg0.7j",   0x00000, 0x20000, CRC(55ce12ea) SHA1(db8677e05009db6c79a727f42f35d60f4a716fc2) )
	ROM_LOAD( "fg1.8j",   0x20000, 0x20000, CRC(692afb67) SHA1(5669298a646deb2f82e438ae52de03f81a9e11a7) )
	ROM_LOAD( "fg2.9j",   0x40000, 0x20000, CRC(f1aba383) SHA1(56d1f7e9eb430671076415dd87fe77a38fadad84) )
	ROM_LOAD( "fg3.10j",  0x60000, 0x20000, CRC(d3475ff8) SHA1(aa7a283a190a6c43e365fcd9242c5d0b920dbf32) )

	ROM_REGION( 0x0006, "plds", 0 )
	/* According to the manual these pal's are located on the Video Game board */
	ROM_LOAD( "pal20l8.9b",   0x00000, 0x00001, NO_DUMP ) /* marked COLARB in manual */
	ROM_LOAD( "pal16l8.1j",   0x00001, 0x00001, NO_DUMP ) /* marked IODCD in manual */
	ROM_LOAD( "pal16l8.2j",   0x00002, 0x00001, NO_DUMP ) /* marked MEMDCD in manual */
	ROM_LOAD( "pal16r4.2k",   0x00003, 0x00001, NO_DUMP ) /* marked DTACK in manual */
	ROM_LOAD( "pal16r4.14k",  0x00004, 0x00001, NO_DUMP ) /* marked HSYNC in manual*/
	/* According to the manual this pal is located on the "Sounds Good" board */
	ROM_LOAD( "pal20.u15",    0x00005, 0x00001, NO_DUMP ) /* marked SG01R0 in manual, pal type not specified */
ROM_END


ROM_START( blasted ) /* Service mode shows "prod. code v.1" and the date 4/27/88 */
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "3c",  0x00000, 0x10000, CRC(b243b7df) SHA1(b44179c30e5286362b0be4e2e9b0742e7e27f7c9) )
	ROM_LOAD16_BYTE( "3b",  0x00001, 0x10000, CRC(627e30d3) SHA1(c430191dd539a22603e49df4c4cb697747a0cd02) )
	ROM_LOAD16_BYTE( "2c",  0x20000, 0x10000, CRC(026f30bf) SHA1(de327ab5bd4dc9456fa5a91f3ccd293b3ab8c5c2) )
	ROM_LOAD16_BYTE( "2b",  0x20001, 0x10000, CRC(8e0e91a9) SHA1(2dc2927a1fd552ead446606a902a2ba0c4595798) )

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "blasted.u7",  0x00000, 0x10000, CRC(8d7c8ef6) SHA1(a414e91c20202f800f3e01e4c430e3f99e3df5bb) )
	ROM_LOAD16_BYTE( "blasted.u17", 0x00001, 0x10000, CRC(c79040b9) SHA1(e6fa173ff5fb681ddfef831f1ef237a7c4303f32) )
	ROM_LOAD16_BYTE( "blasted.u8",  0x20000, 0x10000, CRC(c53094c0) SHA1(8c54cefe8030bf18b9585008a4a6cf8a7dc23f71) )
	ROM_LOAD16_BYTE( "blasted.u18", 0x20001, 0x10000, CRC(85688160) SHA1(de0494f3d457ba81d03700ca569fe3fe4f10a8a1) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "11d",  0x00000, 0x08000, CRC(d8ed5cbc) SHA1(0ad8b894f40892c734fdda056fefe0aac44f9a82) )
	ROM_LOAD( "12d",  0x08000, 0x08000, CRC(60d00c69) SHA1(900d5915a1369c2fc8305c2baeafede5ca627a6b) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "fg0",  0x00000, 0x20000, CRC(5034ae8a) SHA1(45bee41af64fd52311cb470332de4b1caa0f9c9d) )
	ROM_LOAD( "fg1",  0x20000, 0x20000, CRC(4fbdba58) SHA1(5dfaca5447e96d904028a14ef01ab6bd972011e6) )
	ROM_LOAD( "fg2",  0x40000, 0x20000, CRC(8891f6f8) SHA1(af07aa290eff3b9632b238d8b5a37280961f63f7) )
	ROM_LOAD( "fg3",  0x60000, 0x20000, CRC(18e4a130) SHA1(2412b45ca58b36515c80b0888a5d35303a5ce5a2) )

	ROM_REGION( 0x0006, "plds", 0 )
	/* According to the manual these pal's are located on the Video Game board */
	ROM_LOAD( "pal20l8.9b",   0x00000, 0x00001, NO_DUMP ) /* marked COLARB in manual */
	ROM_LOAD( "pal16l8.1j",   0x00001, 0x00001, NO_DUMP ) /* marked IODCD in manual */
	ROM_LOAD( "pal16l8.2j",   0x00002, 0x00001, NO_DUMP ) /* marked MEMDCD in manual */
	ROM_LOAD( "pal16r4.2k",   0x00003, 0x00001, NO_DUMP ) /* marked DTACK in manual */
	ROM_LOAD( "pal16r4.14k",  0x00004, 0x00001, NO_DUMP ) /* marked HSYNC in manual*/
	/* According to the manual this pal is located on the "Sounds Good" board */
	ROM_LOAD( "pal20.u15",    0x00005, 0x00001, NO_DUMP ) /* marked SG01R0 in manual, pal type not specified */
ROM_END

ROM_START( intlaser ) /* Service mode shows "TOP SECRET PROJ. #F01" and the date 10/01/87 */
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "3c.bin",  0x00000, 0x10000, CRC(ddab582a) SHA1(db124e35b7b33d282f71104412a8dab71ce71cb4) )
	ROM_LOAD16_BYTE( "3b.bin",  0x00001, 0x10000, CRC(e4498eca) SHA1(69cbc80ab9a801e957a74278475869d3b54e9a2a) )
	ROM_LOAD16_BYTE( "2c.bin",  0x20000, 0x10000, CRC(d2cca853) SHA1(69e4ee8203c6dda7b4ec97c247fbcdc9fdc9ff8d) )
	ROM_LOAD16_BYTE( "2b.bin",  0x20001, 0x10000, CRC(3802cfe2) SHA1(d10c802500bae14acc3230ca34c2d1806b68ce4a) )

	ROM_REGION( 0x40000, "sgcpu", 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "u7.bin",  0x00000, 0x10000, CRC(19ad1e45) SHA1(838ad7304248690d3fdf9e4edf3856936bf36d42) )
	ROM_LOAD16_BYTE( "u17.bin", 0x00001, 0x10000, CRC(d6118949) SHA1(9e059f28d9eb8dee10301662a65588cffaf6fd16) )
	ROM_LOAD16_BYTE( "u8.bin",  0x20000, 0x10000, CRC(d6cc99aa) SHA1(b970d6e87778959cf7322158b8df26c5028e3f45) )
	ROM_LOAD16_BYTE( "u18.bin", 0x20001, 0x10000, CRC(3488a5cd) SHA1(9b8d37ee7de01982eee074244e14d33eef3644c8) )

	ROM_REGION( 0x10000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "11d.bin",  0x00000, 0x08000, CRC(ce508d5f) SHA1(2118f3654b4306739cdeed9204dff3eda9de3cb7) )
	ROM_LOAD( "12d.bin",  0x08000, 0x08000, CRC(fbcb3391) SHA1(7fbb59e86ac7c01581786f2e46fa9f400d75647b) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "7j.bin",  0x00000, 0x20000, CRC(ac050bd7) SHA1(1a5e67661039ec4d1fa078315c09941dfb2b98d8) )
	ROM_LOAD( "8j.bin",  0x20000, 0x20000, CRC(f10b12b3) SHA1(30770ed5c803a9937d90f9f27204e51f665652d7) )
	ROM_LOAD( "9j.bin",  0x40000, 0x20000, CRC(a18f6911) SHA1(6b50d0c8ca4886f04493756995a0ea27a7ca5433) )
	ROM_LOAD( "10j.bin", 0x60000, 0x20000, CRC(203b55b8) SHA1(72311af32039d09f3b0f4641b71eaf836302fc9a) )

	ROM_REGION( 0x0006, "plds", 0 )
	/* According to the manual these pal's are located on the Video Game board */
	ROM_LOAD( "pal20l8.9b",   0x00000, 0x00001, NO_DUMP ) /* marked COLARB in manual */
	ROM_LOAD( "pal16l8.1j",   0x00001, 0x00001, NO_DUMP ) /* marked IODCD in manual */
	ROM_LOAD( "pal16l8.2j",   0x00002, 0x00001, NO_DUMP ) /* marked MEMDCD in manual */
	ROM_LOAD( "pal16r4.2k",   0x00003, 0x00001, NO_DUMP ) /* marked DTACK in manual */
	ROM_LOAD( "pal16r4.14k",  0x00004, 0x00001, NO_DUMP ) /* marked HSYNC in manual*/
	/* According to the manual this pal is located on the "Sounds Good" board */
	ROM_LOAD( "pal20.u15",    0x00005, 0x00001, NO_DUMP ) /* marked SG01R0 in manual, pal type not specified */
ROM_END


ROM_START( archrivl )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "3c-rev2",  0x00000, 0x10000, CRC(60d4b760) SHA1(9c24c72f62310475b0dade85299cb661904f8f41) ) /* Reports as rev 4.0 6/29/89 */
	ROM_LOAD16_BYTE( "3b-rev2",  0x00001, 0x10000, CRC(e0c07a8d) SHA1(ace5b480d4c2cd3d78dff0e284cf13a8d28c40b7) )
	ROM_LOAD16_BYTE( "2c-rev2",  0x20000, 0x10000, CRC(cc2893f7) SHA1(44931299cb98e27ac2f11b3922da76895fbfe0a7) )
	ROM_LOAD16_BYTE( "2b-rev2",  0x20001, 0x10000, CRC(fa977050) SHA1(67c66995da755401162f7e668b97eb42ac769ec0) )

	ROM_REGION( 0x90000, "cvsdcpu", 0 )  /* Audio System board */
	ROM_LOAD( "u4.snd",  0x10000, 0x08000, CRC(96b3c652) SHA1(1bb576d0bf6b6b8df24e7b9352a33e97dd8ebdcb) )
	ROM_RELOAD(          0x18000, 0x08000 )
	ROM_RELOAD(          0x20000, 0x08000 )
	ROM_RELOAD(          0x28000, 0x08000 )
	ROM_LOAD( "u19.snd", 0x30000, 0x08000, CRC(c4b3dc23) SHA1(87e6eaec82d749ad28e0fa3d0efecd8a4aaf5cd7) )
	ROM_RELOAD(          0x38000, 0x08000 )
	ROM_RELOAD(          0x40000, 0x08000 )
	ROM_RELOAD(          0x48000, 0x08000 )
	ROM_LOAD( "u20.snd", 0x50000, 0x08000, CRC(f7907a02) SHA1(3fabb2b7fd82e773d7b6db53c5328b5866d70617) )
	ROM_RELOAD(          0x58000, 0x08000 )
	ROM_RELOAD(          0x60000, 0x08000 )
	ROM_RELOAD(          0x68000, 0x08000 )

	ROM_REGION( 0x20000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "11d-rev1",  0x00000, 0x10000, CRC(7eb3d7c6) SHA1(8544d04929cdb36fa7f0dcb67e0b7fd8c7b0fc2b) )
	ROM_LOAD( "12d-rev1",  0x10000, 0x10000, CRC(31e68050) SHA1(e25871beb08a8706af70d277fa7305a1f4d7d3e2) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "7j-rev1",   0x00000, 0x20000, CRC(148ce28c) SHA1(d7dc59d9ae8dc61ecc58a6172fd40aa3926b1f6f) )
	ROM_LOAD( "8j-rev1",   0x20000, 0x20000, CRC(58187ac2) SHA1(0bd58598720c41b3c393d47b3b1d6b30696b3a6f) )
	ROM_LOAD( "9j-rev1",   0x40000, 0x20000, CRC(0dd1204e) SHA1(bdc9b74e7ae8f071d2eb3ce957eec484f02ef876) )
	ROM_LOAD( "10j-rev1",  0x60000, 0x20000, CRC(eb3d0344) SHA1(9db7c7bca45f56550c9a9623f96565901968d0c3) )

	ROM_REGION( 0x1200, "plds", 0 )
	ROM_LOAD( "pls153.11j",   0x0000, 0x00eb, CRC(761c3b56) SHA1(06c1717face55cc5b05ec45be9525a3d25419b85) )
	ROM_LOAD( "pls153.12j",   0x0100, 0x00eb, CRC(48eed036) SHA1(146b47ecb341b074acad0e4da2d81ff921bbaf7a) )
	ROM_LOAD( "pls153.14h",   0x0200, 0x00eb, CRC(d4203273) SHA1(59fde5850ad55e257f10db857dfb9a1e929fc1ec) )
	ROM_LOAD( "pal12h6.14e",  0x0300, 0x0034, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4a.14k", 0x0400, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4a.2k",  0x0600, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r6a.15e", 0x0800, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8a.1j",  0x0a00, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8a.2j",  0x0c00, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal20l8a.9b",  0x0e00, 0x0144, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pl20x10a.14f", 0x1000, 0x00cc, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pl20x10a.15f", 0x1100, 0x00cc, NO_DUMP ) /* PAL is read protected */
ROM_END


ROM_START( archrivl2 )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "archrivl.4",  0x00000, 0x10000, CRC(3c545740) SHA1(84a467756c959385a3ec3b97026823470bbab7ab) ) /* Reports as rev 2.0 5/03/89 */
	ROM_LOAD16_BYTE( "archrivl.2",  0x00001, 0x10000, CRC(bc4df2b9) SHA1(7314d03d4cf7e8a83135fa67969dda3088e212fb) )
	ROM_LOAD16_BYTE( "archrivl.3",  0x20000, 0x10000, CRC(d6d08ff7) SHA1(bbbd4b5c3218c9bb461b17e536191d40ab39f67c) )
	ROM_LOAD16_BYTE( "archrivl.1",  0x20001, 0x10000, CRC(92f3a43d) SHA1(45fdcbacd65f5898d54cc2ac95639b7ee2c097e6) )

	ROM_REGION( 0x90000, "cvsdcpu", 0 )  /* Audio System board */
	ROM_LOAD( "u4.snd",  0x10000, 0x08000, CRC(96b3c652) SHA1(1bb576d0bf6b6b8df24e7b9352a33e97dd8ebdcb) )
	ROM_RELOAD(          0x18000, 0x08000 )
	ROM_RELOAD(          0x20000, 0x08000 )
	ROM_RELOAD(          0x28000, 0x08000 )
	ROM_LOAD( "u19.snd", 0x30000, 0x08000, CRC(c4b3dc23) SHA1(87e6eaec82d749ad28e0fa3d0efecd8a4aaf5cd7) )
	ROM_RELOAD(          0x38000, 0x08000 )
	ROM_RELOAD(          0x40000, 0x08000 )
	ROM_RELOAD(          0x48000, 0x08000 )
	ROM_LOAD( "u20.snd", 0x50000, 0x08000, CRC(f7907a02) SHA1(3fabb2b7fd82e773d7b6db53c5328b5866d70617) )
	ROM_RELOAD(          0x58000, 0x08000 )
	ROM_RELOAD(          0x60000, 0x08000 )
	ROM_RELOAD(          0x68000, 0x08000 )

	ROM_REGION( 0x20000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "11d-rev1",  0x00000, 0x10000, CRC(7eb3d7c6) SHA1(8544d04929cdb36fa7f0dcb67e0b7fd8c7b0fc2b) )
	ROM_LOAD( "12d-rev1",  0x10000, 0x10000, CRC(31e68050) SHA1(e25871beb08a8706af70d277fa7305a1f4d7d3e2) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "7j-rev1",   0x00000, 0x20000, CRC(148ce28c) SHA1(d7dc59d9ae8dc61ecc58a6172fd40aa3926b1f6f) )
	ROM_LOAD( "8j-rev1",   0x20000, 0x20000, CRC(58187ac2) SHA1(0bd58598720c41b3c393d47b3b1d6b30696b3a6f) )
	ROM_LOAD( "9j-rev1",   0x40000, 0x20000, CRC(0dd1204e) SHA1(bdc9b74e7ae8f071d2eb3ce957eec484f02ef876) )
	ROM_LOAD( "10j-rev1",  0x60000, 0x20000, CRC(eb3d0344) SHA1(9db7c7bca45f56550c9a9623f96565901968d0c3) )

	ROM_REGION( 0x1200, "plds", 0 )
	ROM_LOAD( "pls153.11j",   0x0000, 0x00eb, CRC(761c3b56) SHA1(06c1717face55cc5b05ec45be9525a3d25419b85) )
	ROM_LOAD( "pls153.12j",   0x0100, 0x00eb, CRC(48eed036) SHA1(146b47ecb341b074acad0e4da2d81ff921bbaf7a) )
	ROM_LOAD( "pls153.14h",   0x0200, 0x00eb, CRC(d4203273) SHA1(59fde5850ad55e257f10db857dfb9a1e929fc1ec) )
	ROM_LOAD( "pal12h6.14e",  0x0300, 0x0034, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4a.14k", 0x0400, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4a.2k",  0x0600, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r6a.15e", 0x0800, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8a.1j",  0x0a00, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8a.2j",  0x0c00, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal20l8a.9b",  0x0e00, 0x0144, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pl20x10a.14f", 0x1000, 0x00cc, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pl20x10a.15f", 0x1100, 0x00cc, NO_DUMP ) /* PAL is read protected */
ROM_END


ROM_START( pigskin ) /* Initial boot screen reports KIT CODE REV 1.1K 8/01/90 */
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pigskin-k_la1.a5",  0x00000, 0x10000, CRC(ab61c29b) SHA1(01cf2d9b3f41442280e614541d5651c6e46b4a4b) )
	ROM_LOAD16_BYTE( "pigskin-k_la1.b5",  0x00001, 0x10000, CRC(55a802aa) SHA1(a75f54bce5aad3f7375ab15ad204744e2f6fdc92) )
	ROM_LOAD16_BYTE( "pigskin-k_la1.a6",  0x20000, 0x10000, CRC(4d8b7e50) SHA1(9e5d0edf1603e11f22d3129a2b8865ebcb5e27f9) )
	ROM_LOAD16_BYTE( "pigskin-k_la1.b6",  0x20001, 0x10000, CRC(1194f187) SHA1(e7cebe5322a5c8e382b6773939be5bc88492f289) )

	ROM_REGION( 0x90000, "cvsdcpu", 0 )  /* Audio System board */
	ROM_LOAD( "pigskin_sl1.u4",  0x10000, 0x10000, CRC(6daf2d37) SHA1(4c8098520fe44e36b01389bcfcfe3ad1d027cbde) )
	ROM_RELOAD(                  0x20000, 0x10000 )
	ROM_LOAD( "pigskin_sl1.u19", 0x30000, 0x10000, CRC(56fd16a3) SHA1(b91aabdbd3185355f2b7177fc4d3a86fa110f51d) )
	ROM_RELOAD(                  0x40000, 0x10000 )
	ROM_LOAD( "pigskin_sl1.u20", 0x50000, 0x10000, CRC(5d032fb8) SHA1(a236cdc64856637e560bec7119b051fac13efbe0) )
	ROM_RELOAD(                  0x60000, 0x10000 )

	ROM_REGION( 0x20000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "pigskin_la1.e2",  0x00000, 0x10000, CRC(12d5737b) SHA1(73040233bb86eaa42257112e2f0540de1206e310) )
	ROM_LOAD( "pigskin_la1.e1",  0x10000, 0x10000, CRC(460202a9) SHA1(8c2f7ae3615519e13e750c99b89ccb28e9946bb8) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "pigskin_la3.h15", 0x00000, 0x20000, CRC(2655d03f) SHA1(de2e2a7fb40844d921b6afaa724a7089a0397270) )
	ROM_LOAD( "pigskin_la3.h17", 0x20000, 0x20000, CRC(31c52ea7) SHA1(1fcb99a70494eecb970bc47bd47ef04a170fab5d) )
	ROM_LOAD( "pigskin_la3.h18", 0x40000, 0x20000, CRC(b36c4109) SHA1(4eeae5dfbbe061ea55d7429a7386ec75c734befb) )
	ROM_LOAD( "pigskin_la3.h14", 0x60000, 0x20000, CRC(09c87104) SHA1(34ac22bdcd8218b9d6a0b4219c67b9b0cc000375) )
ROM_END


ROM_START( pigskina ) /* Initial boot screen reports REV 2.0 7/06/90 */
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "pigskin_la2.a5",  0x00000, 0x10000, CRC(f75d36dd) SHA1(6afc8fbc900e17f9281ee214097d8ebd651d9291) )
	ROM_LOAD16_BYTE( "pigskin_la2.b5",  0x00001, 0x10000, CRC(c5ffdfad) SHA1(3b234f3629c8f21199f4845df7f44c43fd775c9b) )
	ROM_LOAD16_BYTE( "pigskin_la2.a6",  0x20000, 0x10000, CRC(2fc91002) SHA1(64d270b78c69d3f4fb36d1233a1632d6ba3d87a5) )
	ROM_LOAD16_BYTE( "pigskin_la2.b6",  0x20001, 0x10000, CRC(0b93dc66) SHA1(f3b516a1d1e4abd7b0d56243949e9cd7ac79178b) )

	ROM_REGION( 0x90000, "cvsdcpu", 0 )  /* Audio System board */
	ROM_LOAD( "pigskin_sl1.u4",  0x10000, 0x10000, CRC(6daf2d37) SHA1(4c8098520fe44e36b01389bcfcfe3ad1d027cbde) )
	ROM_RELOAD(                  0x20000, 0x10000 )
	ROM_LOAD( "pigskin_sl1.u19", 0x30000, 0x10000, CRC(56fd16a3) SHA1(b91aabdbd3185355f2b7177fc4d3a86fa110f51d) )
	ROM_RELOAD(                  0x40000, 0x10000 )
	ROM_LOAD( "pigskin_sl1.u20", 0x50000, 0x10000, CRC(5d032fb8) SHA1(a236cdc64856637e560bec7119b051fac13efbe0) )
	ROM_RELOAD(                  0x60000, 0x10000 )

	ROM_REGION( 0x20000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "pigskin_la1.e2",  0x00000, 0x10000, CRC(12d5737b) SHA1(73040233bb86eaa42257112e2f0540de1206e310) )
	ROM_LOAD( "pigskin_la1.e1",  0x10000, 0x10000, CRC(460202a9) SHA1(8c2f7ae3615519e13e750c99b89ccb28e9946bb8) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "pigskin_la1.h15", 0x00000, 0x20000, CRC(e43d5d93) SHA1(e7592ba11601f2d20d54d52f436a239671c4d3ac) )
	ROM_LOAD( "pigskin_la1.h17", 0x20000, 0x20000, CRC(6b780f1e) SHA1(a0689feb38ad31eff5604d80562d9a936b30a011) )
	ROM_LOAD( "pigskin_la1.h18", 0x40000, 0x20000, CRC(5e50f940) SHA1(c9593b11934fd6da2b6c971859c0581fd92a915f) )
	ROM_LOAD( "pigskin_la1.h14", 0x60000, 0x20000, CRC(f26279f4) SHA1(9a8cd5aa359f408c93aa7f322b6eac17be52f3d3) )
ROM_END


ROM_START( trisport )
	ROM_REGION( 0x40000, "maincpu", 0 )
	ROM_LOAD16_BYTE( "la3.a5",  0x00000, 0x10000, CRC(fe1e9e37) SHA1(583f18531583e038ca57a592b6a6c305896bf2c5) )
	ROM_LOAD16_BYTE( "la3.b5",  0x00001, 0x10000, CRC(f352ec81) SHA1(446a68f231ca57540a295742d67ce9f1a8364b15) )
	ROM_LOAD16_BYTE( "la3.a6",  0x20000, 0x10000, CRC(9c6a1398) SHA1(ee115d9207f3a9034b7c9eccd2ff151d9c923c9a) )
	ROM_LOAD16_BYTE( "la3.b6",  0x20001, 0x10000, CRC(597b564c) SHA1(090da3ec0c86035cc41a9caea182b8a5419c3be9) )

	ROM_REGION( 0x90000, "cvsdcpu", 0 )  /* Audio System board */
	ROM_LOAD( "sl1-snd.u4",  0x10000, 0x10000, CRC(0ed8c904) SHA1(21292a001c4c44f87b8782c706e5c346b767cd6b) )
	ROM_RELOAD(              0x20000, 0x10000 )
	ROM_LOAD( "sl1-snd.u19", 0x30000, 0x10000, CRC(b57d7d7e) SHA1(483f718f1cc4549baf5696935532d30803254a19) )
	ROM_RELOAD(              0x40000, 0x10000 )
	ROM_LOAD( "sl1-snd.u20", 0x50000, 0x08000, CRC(3ae15c08) SHA1(6b0fd09c39da08d1f67b6dd4287e8d2894522e1d) )
	ROM_RELOAD(              0x58000, 0x08000 )
	ROM_RELOAD(              0x60000, 0x08000 )
	ROM_RELOAD(              0x68000, 0x08000 )

	ROM_REGION( 0x20000, "gfx1", ROMREGION_INVERT )
	ROM_LOAD( "la2.e2",  0x00000, 0x10000, CRC(f61149a0) SHA1(a43d184db23c7f194042709550e7bf36b838ee5c) )
	ROM_LOAD( "la2.e1",  0x10000, 0x10000, CRC(cf753497) SHA1(e076093134f346e4cdb8d4a71fefef70f3fc1328) )

	ROM_REGION( 0x80000, "gfx2", 0 )
	ROM_LOAD( "la2.h15", 0x00000, 0x20000, CRC(18a44d43) SHA1(13ac27860b221b4ee950bc239ca941dadcb06832) )
	ROM_LOAD( "la2.h17", 0x20000, 0x20000, CRC(874cd237) SHA1(32983b7fcac3b1a53da3529c5239a1131bb64d4f) )
	ROM_LOAD( "la2.h18", 0x40000, 0x20000, CRC(f7637a18) SHA1(21dbe2bed406908f17f5860cd2ca502b8f481f51) )
	ROM_LOAD( "la2.h14", 0x60000, 0x20000, CRC(403f9401) SHA1(6ff027943016d894b758dc8f189850bec1ee9360) )

	ROM_REGION( 0x1400, "plds", 0 )
	ROM_LOAD( "pal20l8.g5",   0x0000, 0x0144, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal20x10.f7",  0x0200, 0x00cc, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal20x10.e9",  0x0300, 0x00cc, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal20l8.d4",   0x0400, 0x0144, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8.d6",   0x0600, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4a.c11", 0x0800, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r4.e10",  0x0a00, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pls153a.f14",  0x0c00, 0x00eb, CRC(48eed036) SHA1(146b47ecb341b074acad0e4da2d81ff921bbaf7a) )
	ROM_LOAD( "pls153a.f15",  0x0d00, 0x00eb, CRC(761c3b56) SHA1(06c1717face55cc5b05ec45be9525a3d25419b85) )
	ROM_LOAD( "pls153a.e19",  0x0e00, 0x00eb, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16l8.f8",   0x1000, 0x0104, NO_DUMP ) /* PAL is read protected */
	ROM_LOAD( "pal16r6a.e11", 0x1200, 0x0104, NO_DUMP ) /* PAL is read protected */
ROM_END



/*************************************
 *
 *  Driver initialization
 *
 *************************************/

static void mcr68_common_init(running_machine &machine, int sound_board, int clip, int xoffset)
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr_sound_init(machine, sound_board);

	state->m_sprite_clip = clip;
	state->m_sprite_xoffset = xoffset;

	state_save_register_global(machine, state->m_control_word);
}


static DRIVER_INIT( zwackery )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_CHIP_SQUEAK_DELUXE, 0, 0);

	/* Zwackery doesn't care too much about this value; currently taken from Blasted */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);
}


static DRIVER_INIT( xenophob )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_SOUNDS_GOOD, 0, -4);

	/* Xenophobe doesn't care too much about this value; currently taken from Blasted */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);

	/* install control port handler */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_write_handler(0x0c0000, 0x0cffff, FUNC(xenophobe_control_w));
}


static DRIVER_INIT( spyhunt2 )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_TURBO_CHIP_SQUEAK | MCR_SOUNDS_GOOD, 0, -6);

	/* Spy Hunter 2 doesn't care too much about this value; currently taken from Blasted */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);

	/* analog port handling is a bit tricky */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_write_handler(0x0c0000, 0x0cffff, FUNC(spyhunt2_control_w));
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_read_handler(0x0d0000, 0x0dffff, FUNC(spyhunt2_port_0_r));
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_read_handler(0x0e0000, 0x0effff, FUNC(spyhunt2_port_1_r));
}


static DRIVER_INIT( blasted )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_SOUNDS_GOOD, 0, 0);

	/* Blasted checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 220-256 E clocks (i.e., 2200-2560 CPU clocks) */
	/* after the 493; we also allow 16 E clocks for latency  */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);

	/* handle control writes */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_write_handler(0x0c0000, 0x0cffff, FUNC(blasted_control_w));

	/* 6840 is mapped to the lower 8 bits */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_readwrite_handler(0x0a0000, 0x0a000f, FUNC(mcr68_6840_lower_r), FUNC(mcr68_6840_lower_w));
}

static DRIVER_INIT( intlaser )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_SOUNDS_GOOD, 0, 0);

	/* Copied from Blasted */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);

	/* handle control writes */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_write_handler(0x0c0000, 0x0cffff, FUNC(blasted_control_w));

}



static DRIVER_INIT( archrivl )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_WILLIAMS_SOUND, 16, 0);

	/* Arch Rivals doesn't care too much about this value; currently taken from Blasted */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * (256 + 16);

	/* handle control writes */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_write_handler(0x0c0000, 0x0cffff, FUNC(archrivl_control_w));

	/* 49-way joystick handling is a bit tricky */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_read_handler(0x0e0000, 0x0effff, FUNC(archrivl_port_1_r));

	/* 6840 is mapped to the lower 8 bits */
	machine.device("maincpu")->memory().space(AS_PROGRAM)->install_legacy_readwrite_handler(0x0a0000, 0x0a000f, FUNC(mcr68_6840_lower_r), FUNC(mcr68_6840_lower_w));
}


static DRIVER_INIT( pigskin )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_WILLIAMS_SOUND, 16, 0);

	/* Pigskin doesn't care too much about this value; currently taken from Tri-Sports */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * 115;

	state_save_register_global_array(machine, state->m_protection_data);
}


static DRIVER_INIT( trisport )
{
	mcr68_state *state = machine.driver_data<mcr68_state>();
	mcr68_common_init(machine, MCR_WILLIAMS_SOUND, 0, 0);

	/* Tri-Sports checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 87-119 E clocks (i.e., 870-1190 CPU clocks) */
	/* after the 493 */
	state->m_timing_factor = attotime::from_hz(machine.device("maincpu")->unscaled_clock() / 10) * 115;
}



/*************************************
 *
 *  Game drivers
 *
 *************************************/

GAME( 1984, zwackery, 0,        zwackery, zwackery, zwackery, ROT0,   "Bally Midway", "Zwackery", GAME_SUPPORTS_SAVE )
GAME( 1987, xenophob, 0,        xenophob, xenophob, xenophob, ROT0,   "Bally Midway", "Xenophobe", GAME_SUPPORTS_SAVE )
GAME( 1987, spyhunt2, 0,        spyhunt2, spyhunt2, spyhunt2, ROT0,   "Bally Midway", "Spy Hunter 2 (rev 2)", GAME_SUPPORTS_SAVE )
GAME( 1987, spyhunt2a,spyhunt2, spyhunt2, spyhunt2, spyhunt2, ROT0,   "Bally Midway", "Spy Hunter 2 (rev 1)", GAME_SUPPORTS_SAVE )
GAME( 1988, blasted,  0,        xenophob, blasted,  blasted,  ROT0,   "Bally Midway", "Blasted", GAME_SUPPORTS_SAVE )
GAME( 1987, intlaser, blasted,  intlaser, intlaser, intlaser, ROT0,   "Bally Midway", "International Team Laser (prototype)", GAME_SUPPORTS_SAVE )
GAME( 1989, archrivl, 0,        archrivl, archrivl, archrivl, ROT0,   "Bally Midway", "Arch Rivals (rev 4.0 6/29/89)", GAME_SUPPORTS_SAVE )
GAME( 1989, archrivl2,archrivl, archrivl, archrivl, archrivl, ROT0,   "Bally Midway", "Arch Rivals (rev 2.0 5/03/89)", GAME_SUPPORTS_SAVE )
GAME( 1989, trisport, 0,        trisport, trisport, trisport, ROT270, "Bally Midway", "Tri-Sports", GAME_SUPPORTS_SAVE )
GAME( 1990, pigskin,  0,        pigskin,  pigskin,  pigskin,  ROT0,   "Midway", "Pigskin 621AD (rev 1.1K 8/01/90)", GAME_SUPPORTS_SAVE )
GAME( 1990, pigskina, pigskin,  pigskin,  pigskin,  pigskin,  ROT0,   "Midway", "Pigskin 621AD (rev 2.0 7/06/90)", GAME_SUPPORTS_SAVE )
