/* InvariantListLoader.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.list;

import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.list.invariants.InvariantList;

import org.jdom.Document;
import org.jdom.Element;

/**
 *
 */
public class InvariantListLoader {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//InvariantList 1.0//EN";
    
    //
    private InvariantListLoader() { }
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    //
    private static void loadInvariants(InvariantList list, Element element, String invariantIdAttribute) {
        for (Object obj: element.getChildren("invariant")) {
            try {
                Element child = (Element)obj;
                Invariant inv = InvariantManager.getInstance().getInvariant(child.getAttributeValue(invariantIdAttribute));
                list.add(inv);
            } catch (UnknownInvariantException ex) {
                LOGGER.log(Level.WARNING, "Ignored unknown invariant {0}", ex.getId());
                // ignore unknown invariants
            }
        }
    }
    
    /**
     * Load a invariant list from an XML file.
     * @param list List to which the invariants are added. It may be necessary to clear the list before calling
     * this method.
     * @param file XML file from which the list will be loaded.
     * @see InvariantListSaver#save(InvariantList,File)
     */
    public static void load(InvariantList list, InputStream input) throws IOException {
        Document document = LoaderSaverHelper.loadDocument(input);
        if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
            throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
        loadInvariants(list, document.getRootElement(), "id");
    }
    
    /**
     * Load an invariant list from an XML file.
     * This is a compatibility method for loading grinvin-1.0 invariants from graph lists.
     */
    public static void load_1_0(InvariantList list, InputStream input) throws IOException {
        loadInvariants(list, LoaderSaverHelper.loadElementNonValidating(input, "graphlistelements"), "invariantId");
    }
    
}
