/* WorksheetListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;

import org.grinvin.graphs.GraphURIType;
import org.grinvin.worksheet.WorksheetWindow;
import org.grinvin.worksheet.WorksheetWindowListModel;

/**
 *
 */
public class WorksheetListComponent extends JList {
    
    //
    private WorksheetWindowListModel model;
    
    // shared instance
    private static final ListCellRenderer RENDERER = new Renderer();
    
    public WorksheetListComponent(WorksheetWindowListModel model) {
        super(model);
        this.model = model;
        setCellRenderer(RENDERER);
        
        // double click shows corresponding window and brings it to front
        this.addMouseListener(new MouseAdapter() {
            public void mouseClicked(MouseEvent evt) {
                if (isEnabled() && evt.getClickCount() > 1) {
                    int index = locationToIndex(evt.getPoint());
                    WorksheetWindow window =
                            WorksheetListComponent.this.model.get(index);
                    window.setVisible(true);
                    window.toFront();
                }
            }
        });
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                deleteSelectedElements();
            }
        });
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("ENTER"), "openSelectedElements");
        getActionMap().put("openSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                openSelectedElements();
            }
        });
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        //TODO: add warning or undo redo
        
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // remove all indicate elements and dispose the corresponding windows
        for (int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                model.remove(i).dispose();
    }
    
    /**
     * open selected elements.
     */
    public void openSelectedElements() {
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // open all indicated elements
        for (int i = iMax; i >= iMin; i--) {
            if (selectionModel.isSelectedIndex(i)) {
                WorksheetWindow window = model.get(i);
                window.setVisible(true);
                window.toFront();
            }
        }
        
    }
    
    //
    private static class Renderer extends DefaultListCellRenderer {
        
        //
        Renderer() {
            // avoid creation of access type
        }
        
        public Component getListCellRendererComponent (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            WorksheetWindow window  = (WorksheetWindow)value;
            setText(window.getModel().getName());
            setIcon(GraphURIType.LIST.getSmallIcon());
            return this;
        }
    }
    
}
