/* GraphListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;

import org.grinvin.gui.dnd.CommonGraphListTransferHandler;
import org.grinvin.gui.dnd.HasAccessControl;
import org.grinvin.worksheet.HasSelectableGraphList;
import org.grinvin.worksheet.actions.GraphListElementPopupMenu;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.gui.components.render.GraphListElementRenderer;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.gui.windows.GraphPropertiesWindow;
import org.grinvin.list.actions.RenameAction;
import org.grinvin.list.actions.GraphRenamePanel;

/**
 *
 * @author adpeeter
 */
public class GraphListComponent extends JList implements HasSelectableGraphList, HasAccessControl, MouseListener {
    
    //
    protected GraphListModel model;
    
    //
    private int dragOperations;
    
    //
    private int dropOperations;
    
    //
    private final GraphListElementPopupMenu popupMenu;
    
    public GraphListComponent(GraphListModel model) {
        this(model, new DefaultListSelectionModel());
    }
    
    public GraphListComponent(GraphListModel model, ListSelectionModel selectionModel) {
        this(model, selectionModel, LocalTransferHandler.COPY_OR_MOVE, LocalTransferHandler.COPY_OR_MOVE);
    }
    
    public GraphListComponent(GraphListModel model, int dragOperations, int dropOperations) {
        this(model, new DefaultListSelectionModel(), dragOperations, dropOperations);
    }
    
    //
    public GraphListComponent(GraphListModel model, ListSelectionModel selectionModel, int dragOperations, int dropOperations) {
        super(model);
        setSelectionModel(selectionModel);
        
        this.model = model;
        this.dragOperations = dragOperations;
        this.dropOperations = dropOperations;
        
        setDragEnabled(true);
        setTransferHandler(CommonGraphListTransferHandler.getInstance());
        setCellRenderer(GraphListElementRenderer.getInstance());
        setOpaque(false); // we paint our own background
        
        popupMenu = new GraphListElementPopupMenu(model, selectionModel);
        
        // Keyboard interaction
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                if(!isReadOnly())
                    deleteSelectedElements();
            }
        });
                
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("F2"), "startEditing");
        getActionMap().put("startEditing", new RenameAction<GraphListElement>(model, selectionModel, new GraphRenamePanel(), ResourceBundle.getBundle("org.grinvin.worksheet.actions.resources"), "renameGraphListElement.description"));
        
        // mouse interaction
        addMouseListener(this);
        
        // paint the watermark?
        this.paintWatermark = true;
    }
    
    // paint the watermark?
    private boolean paintWatermark;
    
    /**
     * Configure whether to paint the watermark or not. If no watermark
     * is painted, the list background is completely transparent.
     * Note that this behavious is different from a {@link GraphCellListComponent}
     * which has no watermark.
     */
    public void setPaintWatermark(boolean paintWatermark) {
        this.paintWatermark = paintWatermark;
    }
    
    /**
     * Adds a watermark to the list.
     */
    @Override
    protected void paintComponent(Graphics g) {
        if (paintWatermark) {
            g.setColor(getBackground());
            g.fillRect(0, 0, getWidth(), getHeight());
            GraphCellListComponent.WATERMARK_PAINTER.paint(this, g);
        }
        super.paintComponent(g);
    }
    
    //
    protected ListSelectionModel createListSelectionModel() {
        return new DefaultListSelectionModel();
    }
    
    //
    public GraphList getGraphList() {
        return model;
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        for(int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                model.remove(i);
        
        // TODO: this used to be done in another way: first gather
        // all elements to be removed into a list and then delete
        // all elements of that list in one go. Why?
    }
    
    /**
     * Rename currently selected element.
     */
    public void renameSelectedElement() {
        ListSelectionModel selectionModel = getSelectionModel();
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax !=iMin)
            return ; // no selection or more than a single selection
        
        // TODO: remove common code with RenameGraphListeElementAction
        GraphListElement gle = getGraphList().get(iMin);
        if (gle.isNameEditable()) {
            new GraphRenamePanel().showDialog(gle, this);
        }
    }
    
    
    //
    private GraphListElement getElementAtPoint(Point point) {
        
        int row = locationToIndex(point);
        if (row < 0 || !getCellBounds(row,row).contains(point))
            return null;
        else
            return model.get(row);
    }
    
    //
    private GraphListElement[] getSelectedElements() {
        Object[] elements = getSelectedValues();
        GraphListElement[] newElements = new GraphListElement[elements.length];
        for(int i=0;i<elements.length;i++)
            newElements[i] = (GraphListElement)elements[i];
        return newElements;
    }
    
    public void mouseClicked(MouseEvent e) {
        if (isEnabled() && e.getClickCount() == 2 && !e.isPopupTrigger()) {
            GraphListElement gle = getElementAtPoint(e.getPoint());
            if (gle != null)
                new GraphPropertiesWindow(gle).setVisible(true);
        }
    }
    
    public void mousePressed(MouseEvent e) {
        if (isEnabled() && e.isPopupTrigger()) {
            int index = locationToIndex(e.getPoint());
            if(!isSelectedIndex(index))
                setSelectedIndex(index);
            popupMenu.show(getSelectedElements(), this,e.getX(),e.getY());
        }
    }
    
    public void mouseReleased(MouseEvent e) {
        if (isEnabled() && e.isPopupTrigger()) {
            int index = locationToIndex(e.getPoint());
            if(!isSelectedIndex(index))
                setSelectedIndex(index);
            popupMenu.show(getSelectedElements(), this,e.getX(),e.getY());
        }
    }
    
    public void mouseEntered(MouseEvent e) {
        // intentionally empty
    }
    
    public void mouseExited(MouseEvent e) {
        // intentionally empty
    }

    public int getDragOperations() {
        return dragOperations;
    }

    public int getDropOperations() {
        return dropOperations;
    }
    
    //
    private boolean isReadOnly = false;
    
    //
    public void setIsReadOnly(boolean isReadOnly) {
        this.isReadOnly = isReadOnly;
    }
    
    //
    public boolean isReadOnly() {
        return isReadOnly;
    }
    
}
