/* GeneratorInstanceListComponent.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.components;

import be.ugent.caagt.swirl.dnd.LocalTransferHandler;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.util.ResourceBundle;
import javax.swing.AbstractAction;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;

import org.grinvin.gui.dnd.HasAccessControl;
import org.grinvin.graphs.GraphURIType;
import org.grinvin.generators.graphs.GraphGeneratorInstance;
import org.grinvin.gui.WatermarkPainter;
import org.grinvin.gui.dnd.GeneratorDropHandler;
import org.grinvin.list.HasSelectableValues;
import org.grinvin.gui.dnd.SelectionDragHandler;
import org.grinvin.list.generators.GraphGeneratorInstanceListModel;

/**
 * List of generator instances.
 */
public class GeneratorInstanceListComponent extends JList implements HasSelectableValues, HasAccessControl {
    
    // shared renderer instance
    private static final ListCellRenderer RENDERER = new Renderer ();
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.worksheet.resources";
    
    // shared watermark painter
    private static final WatermarkPainter WATERMARK_PAINTER = new WatermarkPainter(
            20,
            ResourceBundle.getBundle(BUNDLE_NAME).getString("GeneratorInstanceList.emptytext"),
            new Color(245, 228, 211)
            );

    // shared transfer handler
    protected static final LocalTransferHandler TRANSFER_HANDLER;
    
    static {
        TRANSFER_HANDLER = new LocalTransferHandler();
        TRANSFER_HANDLER.addDropHandler(GeneratorDropHandler.getInstance());
        TRANSFER_HANDLER.setDragHandler(new SelectionDragHandler(GraphGeneratorInstance.class));
    }
    
    /**
     * Return the data model for this list.
     */
    @SuppressWarnings("unchecked") 
    @Override
    public GraphGeneratorInstanceListModel getModel () {
        return (GraphGeneratorInstanceListModel)super.getModel();
    }
 
    public GeneratorInstanceListComponent(GraphGeneratorInstanceListModel model) {
        this(model, LocalTransferHandler.COPY_OR_MOVE, LocalTransferHandler.COPY_OR_MOVE);
    }
    
    public GeneratorInstanceListComponent(GraphGeneratorInstanceListModel model, int dragOperations, int dropOperations) {
        super(model);
        
        this.dragOperations = dragOperations;
        this.dropOperations = dropOperations;
        
        setTransferHandler(TRANSFER_HANDLER);
        setDragEnabled(true);
        setCellRenderer(RENDERER);
        setOpaque(false); // we paint our own background
        setPaintWatermark(true);
        
        // keyboard interaction
        // TODO: factor out common code with other lists
        getInputMap(WHEN_FOCUSED).put(KeyStroke.getKeyStroke("DELETE"), "deleteSelectedElements");
        getActionMap().put("deleteSelectedElements", new AbstractAction() {
            public void actionPerformed(ActionEvent ev) {
                if(!isReadOnly())
                    deleteSelectedElements();
            }
        });
    }
    
    //
    private int dragOperations;

    //
    private int dropOperations;
     
    // paint the watermark?
    private boolean paintWatermark;
    
    /**
     * Configure whether to paint the watermark or not. If no watermark
     * is painted, the list background is completely transparent.
     * Note that this behavious is different from a {@link GraphCellListComponent}
     * which has no watermark.
     */
    public void setPaintWatermark(boolean paintWatermark) {
        this.paintWatermark = paintWatermark;
    }
    
    /**
     * Adds a watermark to the list.
     */
    @Override
    protected void paintComponent(Graphics g) {
        if(paintWatermark){
            g.setColor(getBackground());
            g.fillRect(0, 0, getWidth(), getHeight());
            WATERMARK_PAINTER.paint(this, g);
        }
        super.paintComponent(g);
    }
    
    /**
     * Delete currently selected elements.
     */
    public void deleteSelectedElements() {
        //TODO: code copied almost verbatim from WorksheetListComponent (except last line)
        
        // iterate over selected indices
        ListSelectionModel selectionModel = getSelectionModel();
        
        int iMin = selectionModel.getMinSelectionIndex();
        int iMax = selectionModel.getMaxSelectionIndex();
        
        if (iMin < 0 || iMax < iMin)
            return ; // no selection
        
        // remove all indicate elements
        for (int i = iMax; i >= iMin; i--)
            if (selectionModel.isSelectedIndex(i))
                getModel().remove(i);
    }

    public int getDragOperations() {
        return dragOperations;
    }
    
    public int getDropOperations() {
        return dropOperations;
    }
    
    //
    private boolean isReadOnly = false;
    
    //
    public void setIsReadOnly(boolean isReadOnly) {
        this.isReadOnly = isReadOnly;
    }
    
    //
    public boolean isReadOnly() {
        return isReadOnly;
    }
    
    //
    private static class Renderer extends DefaultListCellRenderer {
        // TODO: refactor common code with other renderers?
        
        Renderer () {
            // avoid creation of access type
        }
        
        @Override
        public Component getListCellRendererComponent
                (JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            GraphGeneratorInstance instance = (GraphGeneratorInstance)value;
            setText(instance.getName());
            setIcon(GraphURIType.GENERATOR.getSmallIcon());
            setOpaque(isSelected);
            Color bg = getBackground();
            setBackground(new Color(bg.getRed(), bg.getGreen(), bg.getBlue(), 128));
            return this;
        }
    }
    
}
