/* SpringEmbedder.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.actions;

import be.ugent.caagt.swirl.ActionRepeater;
import java.awt.event.ActionEvent;
import javax.swing.JButton;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.Graphs;
import org.grinvin.graphs.Vertex;
import org.grinvin.gred.MutableGraphPanel;

/**
 * Action that changes the current embedding by means of a
 * spring embedder algorithm. For this action to work the following steps
 * should be followed:
 * <ul>
 * <li>Create a <code>JButton</code> form this action.</li>
 * <li>Attach the 'repeater' to this button.</li>
 * </ul>
 * For example:
 * <pre>
 *    SpringEmbedder se = new SpringEmbedder (...);
 *    ...
 *    JButton button = new JButton (se);
 *    ...
 *    se.attachRepeaterTo(button);
 * </pre>
 */
public class SpringEmbedder extends MutableGraphPanelAction {

    //
    private static final String RESOURCE_KEY = "SpringEmbedder";

    //
    public SpringEmbedder(MutableGraphPanel panel) {
        super(panel);
    }

    private class Repeater extends ActionRepeater {
        //
        private double adjust;
        //
        private Embedding embedding;
        //
        private GraphView graph;
        //
        private boolean[][] adj;
        //
        private double[][] velocities;

        Repeater() {
            super();
            setInterval(150);
            this.adjust = 1.0;
        }

        /**
         * Copies original embedding and initializes internal datastructures
         * when button is first pressed.
         */
        @Override
        public void buttonFirstPressed() {
            this.embedding = panel.getEmbedding();
            this.graph = panel.getGraph();
            this.adj = Graphs.booleanAdjacencyMatrix(graph);
            this.velocities = new double[adj.length][2];
            panel.initiateEmbeddingChange();
        }
        //
        private static final double EDGE_LENGTH = 0.6;
        //
        private static final double NONEDGE_LENGTH = 3 * EDGE_LENGTH;
        //
        private static final double FORCE = 0.03;
        //
        private static final double FRICTION = 0.85;

        /**
         * Performs a single adjustment of the coordinates every time the
         * timer ticks.
         */
        public void doAction() {
            int n = graph.getNumberOfVertices();
            if (n == 0) {
                return;
            }

            // adjust velocities according to forces
            for (int i = 0; i < n; i++) {
                double[] coord1 = embedding.getCoordinates(graph.getVertex(i));
                double x = coord1[0];
                double y = coord1[1];
                for (int j = i + 1; j < n; j++) {
                    double[] coord2 = embedding.getCoordinates(graph.getVertex(j));
                    double dx = coord2[0] - x;
                    double dy = coord2[1] - y;
                    double dist = Math.hypot(dx, dy);

                    dist /= adjust;

                    if (dist == 0) {
                        dx = 1.0;
                        dy = 0.0;
                    } else {
                        dx /= dist;
                        dy /= dist;
                    }
                    if (adj[i][j]) {
                        dist = (dist - EDGE_LENGTH) / EDGE_LENGTH;
                    } else {
                        dist = (dist - NONEDGE_LENGTH) / EDGE_LENGTH;
                    }

                    // force proportional to distance from equilibrium
                    dx = dx * dist * adjust * FORCE;
                    dy = dy * dist * adjust * FORCE;
                    velocities[j][0] -= dx;
                    velocities[j][1] -= dy;
                    velocities[i][0] += dx;
                    velocities[i][1] += dy;
                }
            }

            // dampen velocities
            for (int i = 0; i < n; i++) {
                for (int j = 0; j < 2; j++) {
                    velocities[i][j] *= FRICTION;
                }
            }

            // determine vertex bounds
            double[] min = embedding.getCoordinates(graph.getVertex(0));
            double[] max = embedding.getCoordinates(graph.getVertex(0)); // copy

            for (int i = 1; i < n; i++) {
                double[] coords = embedding.getCoordinates(graph.getVertex(i));
                for (int j = 0; j < 2; j++) {
                    if (coords[j] < min[j]) {
                        min[j] = coords[j];
                    } else if (coords[j] > max[j]) {
                        max[j] = coords[j];
                    }
                }
            }

            // compute shift towards the center
            double[] offset = new double[2];
            for (int j = 0; j < 2; j++) {
                double offs = (min[j] + max[j]) / 2;
                if (offs < -0.02 || offs > 0.02) {
                    offset[j] = 0.25 * offs;
                } else {
                    offset[j] = offs;
                }
            }

            // adjust potentials
            double size = Math.max(max[1] - min[1], max[0] - min[0]);
            if (size > 0.0) {
                size = 2.0 / size;
                if (size < 0.95 || size > 1.15) {
                    size = Math.pow(size, 0.25);
                }
                this.adjust = this.adjust * size; // PMD must see that this is an assignment
            } else {
                size = 1.0;
            }

            // apply velocities, perform shift and scale
            for (Vertex v : graph.vertices()) {
                double[] coords = embedding.getCoordinates(v);
                int index = v.getIndex();
                for (int j = 0; j < 2; j++) {
                    coords[j] = size * (coords[j] - offset[j] + velocities[index][j]);
                }
                embedding.setCoordinates(v, coords);
            }

        }

        /**
         * Reverts to saved embedding when the button press is canceled.
         */
        @Override
        public void buttonPressCanceled() {
            panel.undoEmbeddingChange();
        }
    }

    /**
     * Consolidates the changes.
     */
    public void actionPerformed(ActionEvent e) {
        panel.finalizeEmbeddingChange(RESOURCE_KEY);
    }

    /**
     * Attach a repeater to the button associated with this action.
     * Must be called after the button is created from the action for
     * the spring embedder to work properly.
     */
    public void attachRepeaterTo(JButton button) {
        new Repeater().registerWith(button);
    }
}
