/* GraphPanel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred;

import be.ugent.caagt.swirl.dnd.DropHandler;
import be.ugent.caagt.swirl.dnd.LocalTransferHandler;
import be.ugent.caagt.swirl.undoredo.UndoManager;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetAdapter;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;
import java.util.ResourceBundle;
import java.util.TooManyListenersException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import org.grinvin.graphs.DefaultEmbeddingModel;
import org.grinvin.graphs.DefaultGraphBundle;
import org.grinvin.graphs.DefaultGraphModel;
import org.grinvin.graphs.Edge;
import org.grinvin.graphs.Element;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.EmbeddingListener;
import org.grinvin.graphs.EmbeddingModel;
import org.grinvin.graphs.EmbeddingView;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.GraphListener;
import org.grinvin.graphs.GraphModel;
import org.grinvin.graphs.GraphURIType;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.MutableEmbeddingModel;
import org.grinvin.graphs.MutableGraphModel;
import org.grinvin.graphs.Vertex;
import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryManager;
import org.grinvin.factories.graphs.GraphFactory;
import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.AnnotationListener;
import org.grinvin.graphs.AnnotationModel;
import org.grinvin.graphs.AnnotationView;
import org.grinvin.graphs.DefaultAnnotationModel;
import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.MutableAnnotationModel;
import org.grinvin.gred.undoable.UndoableGraph;
import org.grinvin.gred.guides.GuidesListener;
import org.grinvin.gred.undoable.AddElements;
import org.grinvin.gred.guides.NullGuides;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.graphs.render.Renderer;

/**
 * Panel which displays a given graph embedding and reacts to changes in
 * both the embedding and the graph it represents.
 * The graph embedding which is displayed on this panel must be of type
 * {@link EmbeddingModel} and its peer graph must be of type {@link GraphModel}.
 * Some methods of this class however
 * require the peer graph to be of type {@link MutableGraphModel}.
 *
 * The panel displays a two-dimensional embedding which is assumed to use coordinates
 * in the range (-1.0,-1.0)-(1.0,1.0). These coordinates are scaled by a fixed scale factor
 * which is determined at construction time. The center of the panel is used as the center
 * of the coordinate system.<p>
 *
 * The actual rendering of the graph elements is delegated to a {@link Renderer}
 * which is assigned to this panel. Extra information on how to render specific
 * elements of the graph is retrieved from a {@link GraphContext} object
 * that is associated with this panel.<p>
 */
public class GraphPanel extends JPanel
        implements GraphListener, EmbeddingListener, AnnotationListener, Observer, ComponentListener {
    
    /**
     * Panel background color.
     */
    protected Color backgroundColor = GraphURIType.GRAPH_SESSION.getIconBackgroundColor();
    
    //
    protected EmbeddingModel embedding;
    
    //
    protected AnnotationModel annotation;
    
    //
    protected GraphView graph;
    
    //
    protected Renderer renderer;
    
    //
    protected GraphContext context;
    
    //
    protected UndoManager undoManager;
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.gred.editor";
    
    //
    public boolean isDirty() {
        return false;
    }
    
    //
    public void setDirty(boolean dirty) {
        //only needed for MutableGraphPanel
    }
    
    /**
     * Return the undo manager used by this panel.
     */
    public UndoManager getUndoManager() {
        return undoManager;
    }
    
    /**
     * Return the graph context associated with this panel.
     */
    public GraphContext getContext() {
        return context;
    }
    
    /**
     * Return the graph displayed by this panel.
     */
    public GraphView getGraph() {
        return graph;
    }
    
    /**
     * Return the embedding used by this panel.
     */
    public EmbeddingView getEmbedding() {
        return embedding;
    }
    
    /**
     * Return the annotation used by this panel.
     */
    public AnnotationView getAnnotation() {
        return annotation;
    }
    
    //
    protected double scale;
    
    /**
     * Create a new graph panel with given peer. The panel is given a preferred size
     * large enough to display the coordinate range (-1.1,-1.1)-(1.1,1.1).
     * @param embedding Embedding displayed in this panel. The peer of this embedding
     * should be of type GraphModel.
     * @param renderer Renderer for this panel.
     * @param context Graph context for this panel.
     * @param scale Number of pixels corresponding to a unit length in the embedding.
     */
    public GraphPanel(EmbeddingModel embedding, AnnotationModel annotation, Renderer renderer, GraphContext context, double scale) {
        super();
        
        if (embedding.getPeer() != annotation.getPeer())
            throw new RuntimeException("Embedding and annotation peer graph differ.");
        
        initEmbedding(embedding);
        initAnnotation(annotation);
        
        this.renderer = renderer;
        
        this.context = context;
        context.addObserver(this);
        
        this.scale = scale;
        
        this.guides = DEFAULT_GUIDES; // null guides
        
        setBackground(backgroundColor);
        addComponentListener(this);
        
        int prefSize = (int)(scale * 13 / 5);
        setPreferredSize(new Dimension(prefSize, prefSize));
        
        this.undoManager = new UndoManager();
        
        setTransferHandler(TRANSFER_HANDLER);
        
    }
    
    //
    private void initEmbedding(EmbeddingModel embedding) {
        this.embedding = embedding;
        if (embedding.getDimension() != 2) {
            throw new IllegalArgumentException("Cannot display embeddings of dimension other than 2");
        }
        embedding.addEmbeddingListener(this);
        
        graph = embedding.getPeer();
        if (graph instanceof GraphModel) {
            ((GraphModel)graph).addGraphListener(this);
        } else {
            throw new IllegalArgumentException("Embedding should have peer of type GraphModel");
        }
    }
    
    private void initAnnotation(AnnotationModel annotation) {
        this.annotation = annotation;
        annotation.addAnnotationListener(this);
    }
    
    /**
     * Decouples all listeners. Should be called immediately before the panel
     * is removed so it can be garbage collected.
     */
    public void close() {
        if (embedding != null) {
            embedding.removeEmbeddingListener(this);
            if (graph instanceof GraphModel)
                ((GraphModel)graph).removeGraphListener(this);
        }
    }
    
    /**
     * Copies the current Graph to the given GraphBundle.
     * @param bundle the GraphBundle to be filled
     */
    public void fillGraphBundle(GraphBundle bundle) {
        // TODO: should not be here ??
        Map<Vertex, Vertex> vertices = new HashMap<Vertex, Vertex>();
        Graph new_graph = bundle.createGraph();
        Embedding newembedding = bundle.createEmbedding();
        newembedding.setDimension(embedding.getDimension());
        Annotation newannotation = bundle.createAnnotation();
        
        for (Vertex oldv : graph.vertices()) {
            Vertex newv = new_graph.addNewVertex();
            newembedding.setCoordinates(newv,  embedding.getCoordinates(oldv));
            newannotation.setAnnotation(newv, annotation.getAnnotation(oldv));
            vertices.put(oldv, newv);
        }
        
        for (Edge olde : graph.edges()) {
            Edge newe = new_graph.addNewEdge(vertices.get(olde.getFirstEndpoint()), vertices.get(olde.getSecondEndpoint()));
            newannotation.setAnnotation(newe, annotation.getAnnotation(olde));
        }
    }
    
    /* ============================================================
     * DROP FUNCIONALITY
     * ============================================================ */
    
    
    /**
     * Change the embedding to be used by this panel. Clears the selection.
     */
    public void setGraph(EmbeddingModel embedding, AnnotationModel annotation) {
        context.clearSelection();
        context.setRollOver(null);
        close();
        initEmbedding(embedding);
        undoManager.clear();
        this.annotation = annotation;
        repaint();
    }
    
    /**
     * Change the embedding and graph to be used by this panel to a copy of the
     * given embedding.
     */
    public void loadGraph(EmbeddingView newEmbedding, AnnotationView newAnnotation) {
        MutableGraphModel newGraph = new DefaultGraphModel();
        newGraph.copy(newEmbedding.getPeer());
        MutableEmbeddingModel newModel = new DefaultEmbeddingModel(newGraph, 2);
        newModel.copy(newEmbedding, newGraph);
        MutableAnnotationModel newAnnotationModel = new DefaultAnnotationModel(newGraph);
        if (newAnnotation != null)
            newAnnotationModel.copy(newAnnotation, newGraph);
        setGraph(newModel, newAnnotationModel);
    }
    
    /* ============================================================
     * TRACK DROP LOCATION
     * ============================================================ */
    
    // TODO: copy of CellList from SWIRL. Please refactor
    
    //
    private boolean dropInProgress;
    
    //
    private Point dropLocation;
    
    /**
     * Is a drag-and-drop operation in progress where
     * this component is a target?
     */
    public boolean isDropInProgress() {
        return dropInProgress;
    }
    
    /**
     * Return the last drag-and-drop location where this component is a target.
     */
    public Point getDropLocation() {
        return dropLocation;
    }
    
    //
    @SuppressWarnings("PMD.SingularField")
    private DropListener dropListener;
    
    //
    public void setDropTarget(DropTarget dt) {
        DropTarget oldDropTarget = getDropTarget();
        if (oldDropTarget != null) {
            oldDropTarget.removeDropTargetListener(dropListener);
        }
        if (dt != null) {
            if (dropListener == null)
                dropListener = new DropListener(); // lazy init
            try {
                dt.addDropTargetListener(dropListener);
            } catch (TooManyListenersException ex) {
                assert false : "Unexpected exception: " + ex;
            }
        }
        super.setDropTarget(dt);
    }
    
    /**
     * Follows movement of the drop location during drag-and-drop.
     */
    private class DropListener extends DropTargetAdapter {
        
        DropListener() {
            // avoids creation of access type
        }
        
        public void dragOver(DropTargetDragEvent dtde) {
            dropInProgress = true;
            dropLocation = dtde.getLocation();
        }
        
        public void drop(DropTargetDropEvent dtde) {
            dropInProgress = false;
        }
        
        public void dragEnter(DropTargetDragEvent dtde) {
            dropInProgress = true;
            dropLocation = dtde.getLocation();
        }
        
        public void dragExit(DropTargetEvent dte) {
            dropInProgress = false;
        }
    }
    
    /* ============================================================
     * DRAG AND DROP
     * ============================================================ */
    
    /**
     * Handles the dropping of a single graph list element.
     */
    private static class GraphPanelDropHandler implements DropHandler {
        
        // TODO: this is very similar to GraphListElementDropHandler,
        // factor out common code
        
        GraphPanelDropHandler() {
            // avoids creation of access type
        }
        
        public boolean allowsMultipleDrops(JComponent target) {
            return false;
        }
        
        //
        private static final Class GRAPH_LIST_ELEMENT = GraphListElement.class;
        
        public Class<?> getDropClass(JComponent target) {
            return GRAPH_LIST_ELEMENT;
        }
        
        public boolean acceptDrop(JComponent target, Object object, int seqNr) {
            if (target instanceof GraphPanel) {
                GraphBundleView bundle = ((GraphListElement)object).getBundle();
            if (bundle.getEmbedding().getDimension() != 2)
                    return false;
                if (((GraphPanel)target).isDirty()) {
                    ResourceBundle rbundle = ResourceBundle.getBundle(BUNDLE_NAME);
                    int response = JOptionPane.showConfirmDialog(null,
                            rbundle.getString("overwrite.text"),
                            rbundle.getString("overwrite.title"),
                            JOptionPane.YES_NO_OPTION);
                    if (response != JOptionPane.YES_OPTION)
                        return false;
                }
                ((GraphPanel)target).loadGraph(bundle.getEmbedding(), bundle.getAnnotation());
                return false;
                // return true; // #sr110 easy solution
            } else
                return false;
        }
    }
    
    /**
     * The shared drop handler for graph list elements.
     */
    private static final GraphPanelDropHandler GRAPH_DROP_HANDLER = new GraphPanelDropHandler();
    
    
    /**
     * Drop handler for factories.
     */
    private static class FactoryDropper implements DropHandler {
        // TODO: this is very similar to FactoryDropHandler,
        // factor out common code
        
        //
        FactoryDropper() {
            // avoid construction of access type
        }
        
        //
        private static final Class GRAPH_FACTORY = GraphFactory.class;
        
        
        public boolean allowsMultipleDrops(JComponent target) {
            return false;
        }
        
        public Class<?> getDropClass(JComponent target) {
            return GRAPH_FACTORY;
        }
        
        public boolean acceptDrop(JComponent target, Object object, int seqNr) {
            if (seqNr > 0)
                return false;
            if (target instanceof GraphPanel) {
                try {
                    GraphFactory factory = (GraphFactory)object;
                    if (FactoryManager.configureFactory(factory)) {
                        GraphBundle bundle = new DefaultGraphBundle();
                        factory.createGraph(bundle);
                        if (((GraphPanel)target).isDirty()) {
                            ResourceBundle rbundle = ResourceBundle.getBundle(BUNDLE_NAME);
                            int response = JOptionPane.showConfirmDialog(null,
                                    rbundle.getString("overwrite.text"),
                                    rbundle.getString("overwrite.title"),
                                    JOptionPane.YES_NO_OPTION);
                            if (response != JOptionPane.YES_OPTION)
                                return false;
                            
                        }
                        ((GraphPanel)target).loadGraph(bundle.getEmbedding(), bundle.getAnnotation());
                        return false; // cf. #sr 110
                    } else {
                        return false;
                    }
                } catch (FactoryException ex) {
                    Logger.getLogger("org.grinvin.factories").log(Level.WARNING, "Failed to create graph from factory", ex);
                    return false;
                }
            } else
                return false;
        }
        
    }
    
    /**
     * The shared drop handler for factory drops
     */
    private static final FactoryDropper FACTORY_DROP_HANDLER = new FactoryDropper();
    
    /**
     * The unique shared local transfer handler.
     */
    private static final LocalTransferHandler TRANSFER_HANDLER = new LocalTransferHandler();
    
    static {
        TRANSFER_HANDLER.addDropHandler(GRAPH_DROP_HANDLER);
        TRANSFER_HANDLER.addDropHandler(FACTORY_DROP_HANDLER);
    }
    
    
    /* ============================================================
     * GUIDES
     * ============================================================ */
    
    //
    private static final Guides DEFAULT_GUIDES = new NullGuides();
    
    /**
     * Current {@link Guides} object for this panel.
     */
    protected Guides guides;
    
    /**
     * Return the current guides for this panel.
     */
    public Guides getGuides() {
        return guides;
    }
    
    /**
     * Install guides for this panel.
     * @param guides New guide object or null to remove the guides.
     */
    public void setGuides(Guides guides) {
        if (guides == null)
            this.guides = DEFAULT_GUIDES;
        else
            this.guides = guides;
        fireGuidesChanged();
        repaint();
    }
    
    List<GuidesListener> guidesListeners = new ArrayList<GuidesListener>();
    
    private void fireGuidesChanged(){
        for(GuidesListener l : guidesListeners)
            l.guidesChanged();
    }
    
    public void addGuidesListener(GuidesListener l){
        guidesListeners.add(l);
    }
    
    public void removeGuidesListener(GuidesListener l){
        guidesListeners.remove(l);
    }
    
    /* ============================================================
     * PAINT
     * ============================================================ */
    
    /**
     * Paints the graph on the panel using the current renderer.
     * Paints background, edges and then vertices.
     */
    @Override protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D)g.create(); // TODO: regression test for forgetting
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                RenderingHints.VALUE_ANTIALIAS_ON);
        g2.translate(getWidth()/2,getHeight()/2);
        paintBackground(g2);
        paintEdges(g2);
        paintVertices(g2);
    }
    
    /**
     * Paint a background overlay.
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     * <p>This implementation is delegates to the current guides object.
     */
    protected void paintBackground(Graphics2D g2) {
        guides.paint(g2, scale, getWidth()/scale/2, getHeight()/scale/2);
    }
    
    /**
     * Paint all edges of the graph using the current renderer.<p>
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     */
    protected void paintEdges(Graphics2D g2) {
        for(Edge e : graph.edges()) {
            double[] firstCoordinates = embedding.getCoordinates(e.getFirstEndpoint());
            double[] secondCoordinates = embedding.getCoordinates(e.getSecondEndpoint());
            /* Zoom */
            firstCoordinates[0] *= scale;
            firstCoordinates[1] *= -scale;
            secondCoordinates[0] *= scale;
            secondCoordinates[1] *= -scale;
            renderer.paint(e, firstCoordinates, secondCoordinates,
                    context.isRollOver(e), context.isSelected(e), annotation.getAnnotation(e), g2);
        }
    }
    
    /**
     * Paint all vertices of the graph using the current vertex renderer.<p>
     * See {@link #paintComponent} for the order in which the various
     * paint methods are executed.
     */
    protected void paintVertices(Graphics2D g2) {
        for (Vertex v : graph.vertices()) {
            double[] coordinates = embedding.getCoordinates(v);
            /* Zoom */
            coordinates[0] *= scale;
            coordinates[1] *= -scale;
            renderer.paint(v, coordinates,
                    context.isRollOver(v), context.isSelected(v), annotation.getAnnotation(v), g2);
        }
    }
    
    /* ============================================================
     * GRAPH LISTENER
     * ============================================================ */
    
    // implements GraphListener
    public void vertexChanged(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements GraphListener
    public void vertexAdded(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements GraphListener
    public void vertexRemoved(Vertex vertex) {
        repaint(); // TODO: only vertex and adjacent edges
    }
    
    // implements GraphListener
    public void vertexRestored(Vertex vertex) {
        repaint(); // TODO: only vertex
    }
    
    // implements GraphListener
    public void edgeRemoved(Edge edge) {
        repaint(); // TODO: only edge and adjacent vertices
    }
    
    // implements GraphListener
    public void edgeRestored(Edge edge) {
        repaint(); // TODO: only edge and adjacent vertices
    }
    
    // implements GraphListener
    public void edgeChanged(Edge edge) {
        repaint(); // TODO: only edge area
    }
    
    // implements GraphListener
    public void edgeAdded(Edge edge) {
        repaint(); // TODO: only edge area
    }
    
    // implements GraphListener
    public void graphChanged() {
        repaint();
    }
    
    /* ============================================================
     * EMBEDDING LISTENER
     * ============================================================ */
    
    // implements EmbeddingListener
    public void vertexCoordinatesChanged(Vertex vertex) {
        repaint(); // TODO: only vertex area
    }
    
    // implements EmbeddingListener
    public void dimensionChanged(int oldDimension) {
        if (embedding.getDimension() != 2) {
            throw new IllegalStateException
                    ("Cannot display embeddings of dimension other than 2");
        }
    }
    
    // implements EmbeddingListener
    public void embeddingChanged() {
        graph = embedding.getPeer();
        repaint();
    }
    
    /* ============================================================
     * ANNOTATION LISTENER
     * ============================================================ */
    
    public void elementAnnotationChanged(Element element) {
        repaint();
    }
    
    public void annotationChanged() {
        repaint();
    }
    
    /* ============================================================
     * CONTEXT LISTENER
     * ============================================================ */
    
    // implements Observer
    public void update(Observable o, Object arg) {
        repaint();
    }
    
    /* ============================================================
     * COMPONENT LISTENER
     * ============================================================ */
    
    // implements ComponentListener
    public void componentShown(ComponentEvent e) {
        // does nothing
    }
    
    /**
     * Recenters the drawing when the component changes size.
     */
    public void componentResized(ComponentEvent e) {
        repaint();
    }
    
    // implements ComponentListener
    public void componentMoved(ComponentEvent e) {
        // does nothing
    }
    
    // implements ComponentListener
    public void componentHidden(ComponentEvent e) {
        // does nothing
    }
    
    /*============================================================
     * MANIPULATE THE SELECTION
     *============================================================*/
    
    /**
     * Clear the selection.
     */
    public void clearSelection() {
        context.clearSelection();
        context.setRollOver(null);
    }
    
    /**
     * Select all graph elements.
     */
    public void selectAll() {
        Collection<Element> elements = new ArrayList<Element> ();
        for (Vertex v: graph.vertices())
            elements.add(v);
        for (Edge e: graph.edges())
            elements.add(e);
        context.setSelection(elements);
    }
    
    /* ============================================================
     * UTILITIES FOR MOUSE HANDLERS
     * ============================================================ */
    
    // Tolerance used for 'roll over'-checking, in pixels
    private static final double ROLL_OVER_TOLERANCE = 25.0;
    
    /**
     * Check whether the given vertex can count as a rollover element
     * for the given coordinates. Coordinates are in pixels, translated
     * towards the center of the panel, Y-axis inverted, but not scaled.
     */
    private boolean checkRollOver(Vertex v, int panelX, int panelY) {
        // TODO: see RenderedVertex
        double[] coord = embedding.getCoordinates(v);
        double dx = scale*coord[0] - panelX;
        double dy = scale*coord[1] - panelY;
        return dx*dx + dy*dy <= ROLL_OVER_TOLERANCE;
    }
    
    // Bounding box tolerance used for 'roll over'-checking, in pixels
    private static final double ROLL_OVER_TOLERANCE_BB = 3.0;
    
    /**
     * Check whether the given edge can count as a rollover element
     * for the given coordinates. Coordinates are in pixels, translated
     * towards the center of the panel, Y-axis inverted, but not scaled.
     */
    private boolean checkRollOver(Edge e, int panelX, int panelY) {
        double[] coord0 = embedding.getCoordinates(e.getFirstEndpoint());
        double x0 = scale*coord0[0];
        double y0 = scale*coord0[1];
        double[] coord1 = embedding.getCoordinates(e.getSecondEndpoint());
        double x1 = scale*coord1[0];
        double y1 = scale*coord1[1];
        // bounding rectangle
        double height = y1 - y0;
        double width = x1 - x0;
        
        // should be (almost) inside bounding rectangle
        if (width > 0) {
            if (panelX + ROLL_OVER_TOLERANCE_BB < x0)
                return false;
            else if (panelX > x1 + ROLL_OVER_TOLERANCE_BB)
                return false;
        } else {
            
            if (panelX + ROLL_OVER_TOLERANCE_BB < x1)
                return false;
            else if (panelX > x0 + ROLL_OVER_TOLERANCE_BB)
                return false;
        }
        if (height > 0) {
            if (panelY + ROLL_OVER_TOLERANCE_BB < y0)
                return false;
            else if (panelY > y1 + ROLL_OVER_TOLERANCE_BB)
                return false;
        } else {
            
            if (panelY + ROLL_OVER_TOLERANCE_BB < y1)
                return false;
            else if (panelY > y0 + ROLL_OVER_TOLERANCE_BB)
                return false;
        }
        
        // bounding rectangle very small
        if (width > -1.0 && width < 1.0 && height >-1.0 && height < 1.0)
            return true;
        
        // should be close enough to the line
        if (Math.abs(width) > Math.abs(height)) {
            double dy = y0 + height*(panelX-x0) / width - panelY;
            if (dy*dy > ROLL_OVER_TOLERANCE)
                return false;
        } else {
            double dx = x0 + width*(panelY - y0) / height - panelX;
            if (dx*dx > ROLL_OVER_TOLERANCE)
                return false;
        }
        return true;
        
    }
    
    /**
     * Checks whether the given vertex is still attached to its parent.
     */
    public boolean isAttached (Vertex vertex) {
        return graph.contains(vertex);
    }
    
    /**
     * Checks whether the given edge is still attached to its parent.
     */
    public boolean isAttached (Edge edge) {
        return graph.contains(edge);
    }
    
    /**
     * Set the roll over element in the corresponding graph context
     * to an element which lies near to the given mouse coordinates.
     */
    public void setRollOver(int mouseX, int mouseY) {
        
        //
        int panelX = mouseX - getWidth()/2;
        int panelY = getHeight()/2 - mouseY;
        
        Element rollOver = context.getRollOver();
        
        // check current rollOver first if it is a vertex
        if (rollOver instanceof Vertex && 
                isAttached ((Vertex)rollOver) && // current rollover might have disappeared
                checkRollOver((Vertex)rollOver, panelX, panelY))
                return; // roll over did not change
        
        // check list of vertices
        for (Vertex v: graph.vertices()) {
            if (checkRollOver(v, panelX, panelY)) {
                context.setRollOver(v);
                return ;
            }
        }
        
        // check current rollOver if it is an edge
        if (rollOver instanceof Edge &&
                isAttached ((Edge)rollOver) && // current rollover might have disappeared
                checkRollOver((Edge)rollOver, panelX, panelY))
                return; // roll over did not change
        
        // check list of edges
        for (Edge e: graph.edges()) {
            if (checkRollOver(e, panelX, panelY)) {
                context.setRollOver(e);
                return ;
            }
        }
        
        // none found
        context.setRollOver(null);
    }
    
    /**
     * Convert the given rectangle in mouse coordinates to a rectangle
     * in embedding coordinates.
     */
    public Rectangle2D embeddedRectangle(Rectangle mouseRect) {
        return new Rectangle2D.Double
                ((mouseRect.x - getWidth()/2)/scale,
                (getHeight()/2 - mouseRect.y - mouseRect.height)/scale,
                mouseRect.width / scale,
                mouseRect.height / scale);
    }
    
    /**
     * Check whether the given vertex has coordinates within the given
     * rectangle.
     */
    private boolean isContainedIn(Vertex v, Rectangle2D rect) {
        double[] coord = embedding.getCoordinates(v);
        return rect.contains(coord[0], coord[1]);
    }
    
    /**
     * Check whether the given edge lies within the given rectangle.
     */
    private boolean isContainedIn(Edge e, Rectangle2D rect) {
        return isContainedIn(e.getFirstEndpoint(), rect) &&
                isContainedIn(e.getSecondEndpoint(), rect);
    }
    
    /**
     * Add all elements to the given set whose embedding coordinates
     * are within the given bounds.
     * @param set Set to which the elements should be added
     * @return The value of the set argument
     */
    public Collection<Element> addInsideOfRectangle(Rectangle2D bounds, Collection<Element> set) {
        for (Vertex v: graph.vertices())
            if (isContainedIn(v, bounds))
                set.add(v);
        for (Edge e: graph.edges())
            if (isContainedIn(e, bounds))
                set.add(e);
        return set;
    }
    
    
    //
    protected Element anchor;
    
    /**
     * Return the 'anchor' element. The anchor element
     * is a temporary element used by some mouse handlers.
     */
    public Element getAnchorElement() {
        return anchor;
    }
    
    /**
     * Set the 'anchor' element. The anchor element
     * is a temporary element used by some mouse handlers.
     */
    public void setAnchorElement(Element anchor) {
        this.anchor = anchor;
    }
    
    /**
     * Store the mouse coordinates of the given vertex into the given array.
     * The array must have dimension at least 2.
     */
    public void mouseCoordinates(Vertex v, int[] coord) {
        double[] c = embedding.getCoordinates(v);
        coord[0] = (int)(getWidth()/2 + scale*c[0]);
        coord[1] = (int)(getHeight()/2 - scale*c[1]);
    }
    
    /**
     * Join the anchor to the given vertex.
     * @param vertex Vertex to which the anchor should be joined
     * @param leader Is this command the first of an undo/redo group?
     * @throws ClassCastException when the underlying model is not a (mutable) graph.
     */
    public void joinAnchorToVertex(Vertex vertex, boolean leader) {
        
        if (graph instanceof UndoableGraph && anchor instanceof Vertex && anchor != vertex && !graph.areAdjacent(vertex, (Vertex) anchor)) {
            
            // execute action
            Edge edge = ((UndoableGraph)graph).addNewEdge((Vertex)anchor, vertex);
            
            // register command with undo manager
            undoManager.add(new AddElements((UndoableGraph)graph, edge), leader);
            
            // set new rollover
            context.setRollOver(vertex);
        }
    }
    
}
