/* DefaultAnnotationModel.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.graphs;

import javax.swing.event.EventListenerList;

/**
 *
 */
public class DefaultAnnotationModel extends DefaultAnnotation implements MutableAnnotationModel {
    
    //
    private EventListenerList listenerList = new EventListenerList();
    
    //
    private static final Class<AnnotationListener>
            ANNOTATION_LISTENER_CLASS = AnnotationListener.class;
    
    /**
     * Create an annotation for the given peer graph.
     */
    public DefaultAnnotationModel(GraphView peer) {
        super(peer);
    }

    // implements AnnotationModel
    public void addAnnotationListener(AnnotationListener l) {
        listenerList.add(ANNOTATION_LISTENER_CLASS, l);
    }

    // implements AnnotationModel
    public void removeAnnotationListener(AnnotationListener l) {
        listenerList.remove(ANNOTATION_LISTENER_CLASS, l);
    }
    
    /**
     * Notify all listeners of the fact that the annotation of an element
     * was changed.
     */
    public void fireElementAnnotationChanged(Element element) {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==ANNOTATION_LISTENER_CLASS)
                ((AnnotationListener)listeners[i+1]).elementAnnotationChanged(element);
    }

    /**
     * Notify all listeners of the fact that the entire annotation
     * was changed.
     */
    public void fireAnnotationChanged() {
        Object[] listeners = listenerList.getListenerList ();
        for (int i = listeners.length-2; i>=0; i-=2)
            if (listeners[i]==ANNOTATION_LISTENER_CLASS)
                ((AnnotationListener)listeners[i+1]).annotationChanged();
    }
    
    @Override
    public void setAnnotation(Vertex vertex, Object annotation) {
        super.setAnnotation(vertex, annotation);
        fireElementAnnotationChanged(vertex);
    }

    @Override
    public void setAnnotation(Edge edge, Object annotation) {
        super.setAnnotation(edge, annotation);
        fireElementAnnotationChanged(edge);
    }
    
    @Override
    public void copy(AnnotationView original) {
        super.copy(original);
        fireAnnotationChanged();
    }
    
    @Override
    public void copy(AnnotationView original, GraphView newPeer) {
        super.copy(original, newPeer);
        fireAnnotationChanged();
    }
   
}
