/* GraphGeneratorInstanceManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.generators.graphs;

import java.util.HashMap;
import java.util.Map;

import org.grinvin.factories.FactoryException;
import org.grinvin.factories.FactoryManager;
import org.grinvin.params.ParameterList;

/**
 * Manages the mapping between Graph generator instances and their identifiers.
 */
public class GraphGeneratorInstanceManager {
    
    // TODO: factor out code in common with other managers of this kind
    
    //
    private static final GraphGeneratorInstanceManager SINGLETON = new GraphGeneratorInstanceManager();
    
    /**
     * Maps a Graph uri to the corresponding object of type GraphGeneratorInstance.
     */
    private final Map<String, GraphGeneratorInstance> instances;
    
    //
    private GraphGeneratorInstanceManager() {
        instances = new HashMap<String, GraphGeneratorInstance>();
    }
    
    /**
     * Get the singleton instance of this class
     * @return the single shared instance of this class
     */
    public static GraphGeneratorInstanceManager getInstance() {
        return SINGLETON;
    }
    
    //
    private void registerGraphGeneratorInstance(GraphGeneratorInstance element) {
        instances.put(element.getId(), element);
    }
    
    /**
     * Get the {@link GraphGeneratorInstance} with the given id. If necessary,
     * the instance is created anew.
     */
    public GraphGeneratorInstance getGraphGeneratorInstance(String id) throws UnknownGraphGeneratorInstanceException {
        GraphGeneratorInstance element = instances.get(id);
        if (element == null) {
            int pos = id.indexOf('?');
            try {
                if (pos >= 0)
                    element = createGraphGeneratorInstance(id.substring(0,pos), id.substring(pos+1));
                else
                    element = createGraphGeneratorInstance(id, "");
            } catch (FactoryException ex) {
                    throw new UnknownGraphGeneratorInstanceException("Invalid graph generator instance", id, ex);
            }
            registerGraphGeneratorInstance(element);
        }
        return element;
    }
    
    /**
     * Create and register an instance from the given generator with its current parameters.
     * @return the generator instance, unless a generator instance with the same id was
     * already available in cache
     */
    public GraphGeneratorInstance createInstance (GraphGenerator generator) throws FactoryException{
        // TODO: use the same principle for GraphListElementManager?
        
        String id = generator.getInstanceId();
        GraphGeneratorInstance element = instances.get(id);
        if (element == null) {
            element = generator.createInstance();
            registerGraphGeneratorInstance(element);
        }
        return element;
    }
   
    
    //
    private GraphGeneratorInstance createGraphGeneratorInstance(String parentId, String queryString)
    throws FactoryException{
        GraphGenerator generator = FactoryManager.getGeneratorBroker().get(parentId);
        if (generator == null)
            throw new FactoryException("Could not create generator " + parentId);
        ParameterList list = generator.getParameters();
        generator.setParameterValues(list.parseQueryString(queryString));
        return generator.createInstance();
    }
    
}
