/*

  sshd-confprogs.c

  Author: Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Special OID configuration handlers for sshd.

*/

#include "sshincludes.h"
#include "sshgetopt.h"
#include "sshdsprintf.h"
#include "sshmatch.h"

#include "ssh-f-configd.h"

#define SSH_DEBUG_MODULE "Ssh-F-ConfigD-SshD-Conf"

/* SubSystems cofiguration parameter */
SshFConfigDPolicyTransferStatus sshd_conf_handle_1_3_6_1_4_1_2213_36_1_4(
                                                 SshFConfigDConf conf,
                                                 SshFConfigDProg prog,
                                                 SshFConfigDOidKeyPair pair,
                                                 DfPolicyVar var)
{
  enum DfPolicyType var_type;
  unsigned int rows, cols, x;
  SshBufferStruct buffer[1];
  SSH_F_CONFIGD_CHAR_TYPE *val_buf;
  char *sub_name, *sub_prog, *new_val, *t, c;
  Boolean something_retrieved;
  Boolean something_changed;
  
  if (DfpGetType(var, &var_type) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Can't retrieve DFP_STRING OID %s (%s).",
                    conf->av0, pair->oid, pair->key);
      return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
    }
  if (var_type != DFP_TABLE)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Invalid value type %d for OID %s (%s).",
                    conf->av0, (int)var_type, pair->oid, pair->key);
      return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
    }
  if (DfpGetTableDimensions(var, &rows, &cols) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Can't get table dimensions for OID %s (%s).",
                    conf->av0, pair->oid, pair->key);
      return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
    }
  if (cols != 2)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: OID %s (%s) table has %u columns.  Expected 2.",
                    conf->av0, pair->oid, pair->key, cols);
      return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
    }
  val_buf = ssh_xcalloc(SSH_F_CONFIGD_POLICY_VAR_MAX_LEN, sizeof(val_buf[0]));
  ssh_buffer_init(buffer);
  something_retrieved = FALSE;
  for (x = 0; x < rows; x++)
    {
      if (DfpGetCellString(var,
                           x, 0,
                           val_buf,
                           SSH_F_CONFIGD_POLICY_VAR_MAX_LEN) == DFP_SUCCESS)
        {
          ssh_dsprintf(&sub_name, "subsystem-%s", val_buf);
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                        "%s: Can't get subsystem name %u from OID %s (%s).",
                        conf->av0, x, pair->oid, pair->key, cols);
          continue;
        }
      if (DfpGetCellString(var,
                           x, 1,
                           val_buf,
                           SSH_F_CONFIGD_POLICY_VAR_MAX_LEN) == DFP_SUCCESS)
        {
          ssh_dsprintf(&sub_prog, "%s", val_buf);
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                        "%s: Can't get subsystem program %u from OID %s (%s).",
                        conf->av0, x, pair->oid, pair->key, cols);
          ssh_xfree(sub_name);
          continue;
        }
      something_retrieved = TRUE;
      ssh_xbuffer_append(buffer, sub_name, strlen(sub_name));
      c = ' ';
      ssh_xbuffer_append(buffer, &c, 1);
      c = '"';
      ssh_xbuffer_append(buffer, &c, 1);
      ssh_xbuffer_append(buffer, sub_prog, strlen(sub_prog));
      c = '"';
      ssh_xbuffer_append(buffer, &c, 1);
      c = '\n';
      ssh_xbuffer_append(buffer, &c, 1);
      ssh_xfree(sub_name);
      ssh_xfree(sub_prog);
    }
  ssh_xfree(val_buf);
  DfpClose(var);
  if (! something_retrieved)
    {
      something_changed = FALSE;
      new_val = NULL;
    }
  else
    {
      new_val = ssh_xmemdup(ssh_buffer_ptr(buffer), ssh_buffer_len(buffer));
    }
  if ((new_val == NULL) && (pair->val == NULL))
    {
      something_changed = FALSE;
      pair->updated = FALSE;
    }
  else if (new_val == NULL)
    {
      something_changed = TRUE;
      pair->timestamp = ssh_time();
      ssh_xfree(pair->val);
      pair->val = NULL;
      pair->updated = FALSE;
    }
  else if (pair->val == NULL)
    {
      something_changed = TRUE;
      pair->val = new_val;
      pair->timestamp = ssh_time();
      pair->updated = FALSE;
    }
  else if (strcmp(pair->val, new_val) == 0)
    {
      something_changed = FALSE;
      ssh_xfree(new_val);
    }
  else
    {
      something_changed = TRUE;
      pair->timestamp = ssh_time();
      ssh_xfree(pair->val);
      pair->val = new_val;
      pair->updated = TRUE;
    }
  
  ssh_buffer_clear(buffer);
  ssh_xbuffer_append_cstrs(buffer, "\n# OID: ", NULL);
  ssh_xbuffer_append_cstrs(buffer, pair->oid, NULL);
  if (pair->timestamp > 0)
    {
      ssh_xbuffer_append_cstrs(buffer, "\n# ", NULL);
      if (pair->updated)
        {
          ssh_xbuffer_append_cstrs(buffer, "Updated", NULL);
        }
      else
        {
          ssh_xbuffer_append_cstrs(buffer, "Initialized", NULL);
        }
      ssh_xbuffer_append_cstrs(buffer, ": ", NULL);
      t = ssh_readable_time_string(pair->timestamp, FALSE);
      ssh_xbuffer_append_cstrs(buffer, t, NULL);
      ssh_xfree(t);
      ssh_xbuffer_append_cstrs(buffer, " UTC", NULL);
    }
  if (pair->val != NULL)
    {
      ssh_xbuffer_append_cstrs(buffer, "\n", NULL);
      ssh_xbuffer_append_cstrs(buffer, pair->val, NULL);
    }
  else
    {
      ssh_xbuffer_append_cstrs(buffer, "\n# `", NULL);
      ssh_xbuffer_append_cstrs(buffer, pair->key, NULL);
      ssh_xbuffer_append_cstrs(buffer, "' is not set by FSMA\n", NULL);
    }
  ssh_xfree(pair->val_raw);
  pair->val_raw = ssh_xmemdup(ssh_buffer_ptr(buffer), ssh_buffer_len(buffer));
  pair->val_is_raw = TRUE;
  ssh_buffer_uninit(buffer);
  if (something_changed)
    return SSH_F_CONFIGD_POLICY_CHANGED;
  else if (something_retrieved)
    return SSH_F_CONFIGD_POLICY_NOT_CHANGED;
  else
    return SSH_F_CONFIGD_POLICY_TRANSFER_FAILED;
}
