/*

  ssh-f-configd.c

  Author: Timo J. Rinne <tri@ssh.com>

  Copyright (C) 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Main module of Secure Shell Configuration Daemon for F-Secure FSMA.

*/

#include "sshincludes.h"
#include "sshgetopt.h"
#include "sshdsprintf.h"
#include "sshmatch.h"

#include "ssh-f-configd.h"

#define SSH_DEBUG_MODULE "Ssh-F-ConfigD"

void ssh_f_configd_conf_finalize(SshFConfigDConf conf);
void ssh_f_configd_clean_oid_pattern(char *oid);
Boolean ssh_f_configd_parse_oid_keyword_pair(char *s,
                                             char **oid,
                                             char **key);
Boolean ssh_f_configd_oid_match(char *pattern, char *oid);
FSMAERROR ssh_f_config_fsma_message(FSMASESSION session, 
                                    FSMAMESSAGE message,
                                    void *context);
void ssh_f_configd_write_pid_file(SshFConfigDConf conf);

RETSIGTYPE ssh_f_configd_kill_signal(int signum);

static SshFConfigDConf conf_for_signal_handler = NULL;
RETSIGTYPE ssh_f_configd_kill_signal(int signum)
{
  if (conf_for_signal_handler != NULL)
    conf_for_signal_handler->exit_signal = signum;
  ssh_f_configd_terminate(conf_for_signal_handler, 1);
}

void ssh_f_configd_terminate(SshFConfigDConf conf, int exit_value)
{
  if ((conf == NULL) || conf->log_entry_on_termination)
    {
      if ((conf != NULL) && (conf->exit_signal != 0))
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                        "%s: Receiving signal %d.",
                        conf->av0, conf->exit_signal);
        }
      if (exit_value == 0)
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_NOTICE, 
                    "%s%sTerminating.", 
                    (((conf != NULL) && (conf->av0 != NULL)) ? conf->av0 : ""),
                    (((conf != NULL) && (conf->av0 != NULL)) ? ": " : ""));
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s%sTerminating with abnormal error code %d.",
                    (((conf != NULL) && (conf->av0 != NULL)) ? conf->av0 : ""),
                    (((conf != NULL) && (conf->av0 != NULL)) ? ": " : ""),
                    (int)exit_value);
        }
    }
  if (conf != NULL)
    {
      if (conf->fsma_session_active)
        {
          DfpUnregister(conf->fsma_session);
          FSMAUnregisterModule(conf->fsma_session);
          conf->fsma_session_active = FALSE;
        }
      if (conf->pid_file_written)
        remove(conf->pid_file);
    }
  exit(exit_value);
  /*NOTREACHED*/
}

Boolean ssh_f_configd_oid_match(char *pattern, char *oid)
{
  return ssh_match_pattern(oid, pattern);
}

Boolean ssh_f_configd_parse_oid_keyword_pair(char *s,
                                             char **oid,
                                             char **key)
{
  char *p1, *p2, *p3;

  p1 = s;
  while ((! ((isdigit(*p1)) || (*p1 == '.'))) && (*p1 != '\0'))
    p1++;
  if (*p1 == '\0')
    return FALSE;
  p2 = p1;
  while (((isdigit(*p2)) || (*p2 == '.')) && (*p2 != '\0'))
    p2++;
  if (*p2 == '\0')
    return FALSE;
  if (! (isspace(*p2)))
    return FALSE;
  *p2 = '\0';
  p2++;
  while ((! (isalnum(*p2))) && (*p2 != '\0'))
    p2++;
  if (*p2 == '\0')
    return FALSE;
  p3 = p2;
  while ((isalnum(*p3)) && (*p3 != '\0'))
    p3++;
  *p3 = '\0';
  if (oid)
    *oid = p1;
  if (key)
    *key = p2;
  return TRUE;
}

void ssh_f_configd_clean_oid_pattern(char *oid)
{
  char *p1, *p2;

  for (p1 = p2 = oid; *p1 != '\0'; p1++)
    {
      if ((isdigit(*p1)) || (*p1 == '.') || (*p1 == '*') || (*p1 == '?'))
        {
          *p2 = *p1;
          p2++;
        }
    }
  *p2 = '\0';
  return;
}

SshFConfigDOidKeyPair ssh_f_configd_oid_key_pair_alloc(const char *oid,
                                                       const char *key)
{
  SshFConfigDOidKeyPair r;

  r = ssh_xcalloc(1, sizeof (*r));
  r->oid = ssh_xstrdup(oid);
  r->key = ssh_xstrdup(key);
  r->val = NULL;
  return r;
}

void ssh_f_configd_oid_key_pair_free(SshFConfigDOidKeyPair pair)
{
  ssh_xfree(pair->oid);
  ssh_xfree(pair->key);
  ssh_xfree(pair->val);
  ssh_xfree(pair->val_raw);
  ssh_xfree(pair);
  return;
}

SshFConfigDProg ssh_f_configd_prog_alloc(SshFConfigDConf conf)
{
  SshFConfigDProg prog;

  prog = ssh_xcalloc(1, sizeof (*prog));
  prog->oid_map = ssh_adt_create_strmap();
  /* 
     XXX: This works, but it's not elegant.  Use

     #include "sshadt_conv.h"
     ssh_adt_xcreate_strmap(NULL_FNPTR, destructor)
     
  */
  
  if (DfpCreatePolicyVar(&(prog->oid_pattern_var)) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpCreatePolicyVar fails.", conf->av0);
      ssh_f_configd_terminate(NULL, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  ssh_buffer_init(prog->config_data);

  return prog;
}

SshFConfigDOIDHandler ssh_f_configd_find_handler(const char *oid)
{
  if (strcmp(oid, "1.3.6.1.4.1.2213.36.1.4") == 0)
    return sshd_conf_handle_1_3_6_1_4_1_2213_36_1_4;
  return NULL;
}


void ssh_f_configd_prog_free(SshFConfigDConf conf, SshFConfigDProg prog)
{
  SshADTHandle handle;
  SshFConfigDOidKeyPair pair;

  ssh_xfree(prog->name);
  ssh_xfree(prog->config_file);
  ssh_xfree(prog->config_file_bak);
  ssh_xfree(prog->pid_file);
  ssh_xfree(prog->oid_pattern);
  DfpClose(prog->oid_pattern_var);
  ssh_xfree(prog->oid_mapping_file);
  ssh_xfree(prog->oid_pattern_file);
  while ((handle = ssh_adt_enumerate_start(prog->oid_map)) != 
         SSH_ADT_INVALID)
    {
      pair = ssh_adt_map_lookup(prog->oid_map, handle);
      ssh_f_configd_oid_key_pair_free(pair);
      ssh_adt_delete(prog->oid_map, handle);
    }
  ssh_adt_destroy(prog->oid_map);
  ssh_buffer_uninit(prog->config_data);
}

SshFConfigDConf ssh_f_configd_conf_alloc(const char *av0)
{
  SshFConfigDConf conf;
  SshFConfigDProg prog;

  conf = ssh_xcalloc(1, sizeof (*conf));
  conf->av0 = ((av0 != NULL) ? ssh_xstrdup(av0) : "???");
  conf->exit_signal = 0;
  conf->log_entry_on_termination = FALSE;
  conf_for_signal_handler = conf;
  conf->batch_mode = FALSE;
  conf->debug_mode = FALSE;
  conf->pid_file = 
    ssh_xstrdup(SSH_F_CONFIGD_PID_FILE_DEFAULT);
  conf->pid_file_written = FALSE;
  conf->oid_mapping_dir =
    ssh_xstrdup(SSH_F_CONFIGD_OID_MAPPING_DIR_DEFAULT);
  conf->fsma_module_oid =
    ssh_xstrdup(SSH_F_CONFIGD_FSMA_MODULE_OID_DEFAULT);
  conf->fsma_session_active = FALSE;
  if (DfpCreatePolicyVar(&(conf->fsma_module_oid_var)) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpCreatePolicyVar fails.", conf->av0);
      ssh_f_configd_terminate(NULL, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  conf->prog_map = ssh_adt_create_strmap();
  /* 
     XXX: This works, but it's not elegant.  Consider using

     #include "sshadt_conv.h"
     ssh_adt_xcreate_strmap(NULL_FNPTR, destructor)
     
  */
  
  prog = ssh_f_configd_prog_alloc(conf);
  prog->name = ssh_xstrdup("ssh");
  prog->config_file = 
    ssh_xstrdup(SSH_F_CONFIGD_SSH2_CONFIG_FILE_DEFAULT);
  prog->pid = 0;
  prog->pid_file = NULL;
    ssh_xstrdup(SSH_F_CONFIGD_SSHD2_PID_FILE_DEFAULT);
  prog->oid_pattern =
    ssh_xstrdup(SSH_F_CONFIGD_SSH2_OID_PREFIX_DEFAULT);
  if (DfpSetOID(prog->oid_pattern_var, 
                prog->oid_pattern) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpSetOID fails.", conf->av0);
      ssh_f_configd_terminate(NULL, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  SSH_ASSERT(! (ssh_adt_strmap_exists(conf->prog_map, prog->name)));
  ssh_adt_strmap_add(conf->prog_map, prog->name, prog);
  prog = ssh_f_configd_prog_alloc(conf);
  prog->name = ssh_xstrdup("sshd");
  prog->config_file = 
    ssh_xstrdup(SSH_F_CONFIGD_SSHD2_CONFIG_FILE_DEFAULT);
  prog->pid = 0;
  prog->pid_file =
    ssh_xstrdup(SSH_F_CONFIGD_SSHD2_PID_FILE_DEFAULT);
  prog->oid_pattern =
    ssh_xstrdup(SSH_F_CONFIGD_SSHD2_OID_PREFIX_DEFAULT);
  if (DfpSetOID(prog->oid_pattern_var, 
                prog->oid_pattern) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpSetOID fails.", conf->av0);
      ssh_f_configd_terminate(NULL, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  SSH_ASSERT(! (ssh_adt_strmap_exists(conf->prog_map, prog->name)));
  ssh_adt_strmap_add(conf->prog_map, prog->name, prog);
  return conf;
}

void ssh_f_configd_conf_free(SshFConfigDConf conf)
{
  SshADTHandle handle;
  SshFConfigDOidKeyPair pair;
  SshFConfigDProg prog;

  while ((handle = ssh_adt_enumerate_start(conf->prog_map)) != 
         SSH_ADT_INVALID)
    {
      prog = ssh_adt_map_lookup(conf->prog_map, handle);
      ssh_f_configd_prog_free(conf, prog);
      ssh_adt_delete(conf->prog_map, handle);
    }
  ssh_adt_destroy(conf->prog_map);
  ssh_xfree(conf->pid_file);
  ssh_xfree(conf->oid_mapping_dir);
  ssh_xfree(conf->fsma_module_oid);
  DfpClose(conf->fsma_module_oid_var);
  if (conf_for_signal_handler == conf)
    conf_for_signal_handler = NULL;
  ssh_xfree(conf->av0);
  ssh_xfree(conf);
}

void ssh_f_configd_conf_finalize(SshFConfigDConf conf)
{
  FILE *f;
  char buf[128], *tmp, *oid, *key;
  SshFConfigDProg prog;
  SshFConfigDOidKeyPair pair;

  if (DfpSetOID(conf->fsma_module_oid_var, 
                conf->fsma_module_oid) != DFP_SUCCESS)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: DfpSetOID fails.", conf->av0);
      ssh_f_configd_terminate(NULL, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "ssh"));
  prog = ssh_adt_strmap_get(conf->prog_map, "ssh");
  ssh_dsprintf(&(prog->oid_pattern_file),
               "%s/%s",
               conf->oid_mapping_dir,
               "ssh2.oid-pattern");
  ssh_dsprintf(&(prog->oid_mapping_file),
               "%s/%s",
               conf->oid_mapping_dir,
               "ssh2.oid");
  ssh_dsprintf(&(prog->config_file_bak),
               "%s.bak",
               prog->config_file);
  if (! prog->configuration_disable)
    {
      f = fopen(prog->oid_pattern_file, "r");
      if (f != NULL)
        {
          tmp = fgets(buf, sizeof (buf), f);
          fclose(f);
          if (tmp != NULL)
            {
              ssh_f_configd_clean_oid_pattern(tmp);
              if (strlen(tmp) > 0)
                {
                  ssh_xfree(prog->oid_pattern);
                  prog->oid_pattern = ssh_xstrdup(tmp);
                  if (DfpSetOID(prog->oid_pattern_var, 
                                prog->oid_pattern) != DFP_SUCCESS)
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                                    "%s: DfpSetOID fails.", conf->av0);
                      ssh_f_configd_terminate(
                                          NULL, 
                                          SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
                    }
                }
              else
                {
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                      "%s: Invalid OID prefix in file \"%s\".  Using default.",
                      conf->av0, prog->oid_pattern_file);
                }
            }
          else
            {
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                  "%s: Can't open ssh OID prefix file \"%s\".  Using default.",
                  conf->av0, prog->oid_pattern_file);
            }
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
              "%s: Can't open ssh OID prefix file \"%s\".  Using default.",
              conf->av0, prog->oid_pattern_file);
        }
      f = fopen(prog->oid_mapping_file, "r");
      if (f != NULL)
        {
          while ((tmp = fgets(buf, sizeof (buf), f)) != NULL)
            {
              if (ssh_f_configd_parse_oid_keyword_pair(buf, &oid, &key))
                {
                  if (ssh_f_configd_oid_match(prog->oid_pattern, oid))
                    {
                      if (ssh_adt_strmap_exists(prog->oid_map, oid))
                        {
                          ssh_log_event(SSH_LOGFACILITY_DAEMON,
                                        SSH_LOG_WARNING, 
                                        "%s: Duplicate OID %s ignored.",
                                        conf->av0, oid);
                        }
                      else
                        {
                          pair = ssh_f_configd_oid_key_pair_alloc(oid, key);
                          pair->oid_handler = ssh_f_configd_find_handler(oid);
                          ssh_adt_strmap_add(prog->oid_map, oid, pair);
                        }
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: OID %s doesn not match to ssh OID pattern %s.",
                          conf->av0,
                          oid,
                          prog->oid_pattern);
                    }
                }
            }
          fclose(f);
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Can't open ssh OID file \"%s\".", 
                        conf->av0, prog->oid_mapping_file);
        }
    }
  SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "sshd"));
  prog = ssh_adt_strmap_get(conf->prog_map, "sshd");
  ssh_dsprintf(&(prog->oid_pattern_file),
               "%s/%s",
               conf->oid_mapping_dir,
               "sshd2.oid-pattern");
  ssh_dsprintf(&(prog->oid_mapping_file),
               "%s/%s",
               conf->oid_mapping_dir,
               "sshd2.oid");
  ssh_dsprintf(&(prog->config_file_bak), 
               "%s.bak", 
               prog->config_file);
  if (! prog->configuration_disable)
    {
      f = fopen(prog->oid_pattern_file, "r");
      if (f != NULL)
        {
          tmp = fgets(buf, sizeof (buf), f);
          fclose(f);
          if (tmp != NULL)
            {
              ssh_f_configd_clean_oid_pattern(tmp);
              if (strlen(tmp) > 0)
                {
                  ssh_xfree(prog->oid_pattern);
                  prog->oid_pattern = ssh_xstrdup(tmp);
                  if (DfpSetOID(prog->oid_pattern_var, 
                                prog->oid_pattern) != DFP_SUCCESS)
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                                    "%s: DfpSetOID fails.", conf->av0);
                      ssh_f_configd_terminate(
                                          NULL,
                                          SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
                    }
                }
              else
                {
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                      "%s: Invalid OID prefix in file \"%s\".  Using default.",
                      conf->av0, prog->oid_pattern_file);
                }
            }
          else
            {
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                "%s: Can't open sshd OID prefix file \"%s\".  Using default.",
                conf->av0, prog->oid_pattern_file);
            }
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
              "%s: Can't open sshd OID prefix file \"%s\".  Using default.",
              conf->av0, prog->oid_pattern_file);
        }
      f = fopen(prog->oid_mapping_file, "r");
      if (f != NULL)
        {
          while ((tmp = fgets(buf, sizeof (buf), f)) != NULL)
            {
              if (ssh_f_configd_parse_oid_keyword_pair(buf, &oid, &key))
                {
                  if (ssh_f_configd_oid_match(prog->oid_pattern, oid))
                    {
                      if (ssh_adt_strmap_exists(prog->oid_map, oid))
                        {
                          ssh_log_event(SSH_LOGFACILITY_DAEMON,
                                        SSH_LOG_WARNING,
                                        "%s: Duplicate OID %s ignored.",
                                        conf->av0, oid);
                        }
                      else
                        {
                          pair = ssh_f_configd_oid_key_pair_alloc(oid, key);
                          pair->oid_handler = ssh_f_configd_find_handler(oid);
                          ssh_adt_strmap_add(prog->oid_map, oid, pair);
                        }
                    }
                  else
                    {
                      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                          "%s: OID %s doesn not match to sshd OID pattern %s.",
                          conf->av0,
                          oid,
                          prog->oid_pattern);
                    }
                }
            }
          fclose(f);
        }
      else
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                        "%s: Can't open sshd OID file \"%s\".", 
                        conf->av0, prog->oid_mapping_file);
        }
    }
}

void ssh_f_configd_write_pid_file(SshFConfigDConf conf)
{
  FILE *f;
  f = fopen(conf->pid_file, "w");
  if (f != NULL)
    {
      fprintf(f, "%u\n", (unsigned int)(getpid()));
      fclose(f);
      conf->pid_file_written = TRUE;
    }
  else
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                    "%s: Can't write pid file \"%s\".", 
                    conf->av0,
                    conf->pid_file);
      conf->pid_file_written = FALSE;
    }
  return;
}

static void usage(const char *av0);
static void usage(const char *av0)
{
  fprintf(stderr, 
          "Usage: %s [-d debug_level_spec] [-r oid_mapping_directory] "
          "[-M module_oid] [-c ssh_config_file] [-C sshd_config_file] "
          "[-P sshd_pid_file] [-N] [-n] [-n] [-V]\n", 
          av0);
}

#define SSH_F_CONFIGD_GETOPT_STRING "d:c:o:r:C:P:O:R:VnNbM:p:"

int main(int argc, char **argv)
{
  SshFConfigDConf conf;
  int c;
  unsigned int n;
  FSMAERROR fe;
  const char *av0;
  SshFConfigDProg prog;
  SshADTHandle handle;
  Boolean all_disabled;

  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];
  conf = ssh_f_configd_conf_alloc(av0);
  ssh_log_register_callback(ssh_f_configd_log_stderr, conf);
  while ((c = ssh_getopt(argc, argv, SSH_F_CONFIGD_GETOPT_STRING, NULL)) != -1)
    {
      switch (c)
        {
        case 'd':
          conf->debug_mode = TRUE;
          ssh_debug_set_level_string(ssh_optarg);
          break;

        case 'b':
          conf->batch_mode = TRUE;
          break;

        case 'p':
          ssh_xfree(conf->pid_file);
          conf->pid_file = ssh_xstrdup(ssh_optarg);
          break;

        case 'c':
          SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "ssh"));
          prog = ssh_adt_strmap_get(conf->prog_map, "ssh");
          ssh_xfree(prog->config_file);
          prog->config_file = ssh_xstrdup(ssh_optarg);
          break;

        case 'r':
          ssh_xfree(conf->oid_mapping_dir);
          conf->oid_mapping_dir = ssh_xstrdup(ssh_optarg);
          break;

        case 'n':
          SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "ssh"));
          prog = ssh_adt_strmap_get(conf->prog_map, "ssh");
          prog->configuration_disable = TRUE;
          break;

        case 'C':
          SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "sshd"));
          prog = ssh_adt_strmap_get(conf->prog_map, "sshd");
          ssh_xfree(prog->config_file);
          prog->config_file = ssh_xstrdup(ssh_optarg);
          break;

        case 'P':
          SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "sshd"));
          prog = ssh_adt_strmap_get(conf->prog_map, "sshd");
          ssh_xfree(prog->pid_file);
          prog->pid_file = ssh_xstrdup(ssh_optarg);
          break;

        case 'N':
          SSH_ASSERT(ssh_adt_strmap_exists(conf->prog_map, "sshd"));
          prog = ssh_adt_strmap_get(conf->prog_map, "sshd");
          prog->configuration_disable = TRUE;
          break;

        case 'M':
          ssh_xfree(conf->fsma_module_oid);
          conf->fsma_module_oid = ssh_xstrdup(ssh_optarg);
          break;

        case 'V':
          printf("%s: SSH-F-ConfigD Version %s\n", 
                 conf->av0, SSH_F_CONFIGD_VERSION);
          ssh_f_configd_terminate(conf, 0);
          break;

        default:
          usage(conf->av0);
          ssh_f_configd_terminate(conf, -1);
        }
    }
  if ((! conf->batch_mode) && (! conf->debug_mode))
    {
      ssh_log_register_callback(ssh_f_configd_log, conf);
    }
  ssh_f_configd_conf_finalize(conf);
  conf->log_entry_on_termination = TRUE;
  all_disabled = TRUE;
  handle = ssh_adt_enumerate_start(conf->prog_map);
  while (handle != SSH_ADT_INVALID)
    {
      prog = ssh_adt_map_lookup(conf->prog_map, handle);
      if (! prog->configuration_disable)
        {
          n = (unsigned int)(ssh_adt_num_objects(prog->oid_map));
          SSH_DEBUG(3, ("%u oids found in %s configuration", n, prog->name));
          if (n < 1)
            {
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
            "%s: No ssh OIDs in configuration.  Disabling %s configuration.",
            conf->av0, prog->name);
              prog->configuration_disable = TRUE;
            }
          else
            {
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                            "%s: %u configurable OIDs found for `%s'.",
                            conf->av0, (unsigned int)n, prog->name);
              all_disabled = FALSE;
            }
        }
      handle = ssh_adt_enumerate_next(conf->prog_map, handle);
    }
  if (all_disabled)
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: No active configurations.  Exiting.\n", conf->av0);
      ssh_f_configd_terminate(conf, 0);
    }
  if ((! conf->batch_mode) && (! conf->debug_mode))
    {
      {
        pid_t child_pid;

        ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                      "%s: Forking to background.", conf->av0);
        switch (fork())
          {
          case -1:
            ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING,
                          "%s: Unable to fork to background.", conf->av0);
            ssh_f_configd_terminate(conf, 1);
            /*NOTREACHED*/
            break;

          case 0:
            {
              int fd;
              (void)freopen("/dev/null", "r", stdin);
              (void)freopen("/dev/null", "w", stdout);
              (void)freopen("/dev/null", "w", stderr);
#ifdef TIOCNOTTY
              fd = open("/dev/tty", O_RDWR | O_NOCTTY);
              if (fd >= 0) {
                ioctl(fd, TIOCNOTTY, NULL);
                close(fd);
              }
#endif /* TIOCNOTTY */
#ifdef HAVE_SETSID
              setsid();
#endif /* HAVE_SETSID */
              ssh_f_configd_write_pid_file(conf);
              ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_INFORMATIONAL,
                            "%s: Running in background.", conf->av0);
            }
            break;

          default:
            exit(0);
            /*NOTREACHED*/
          }
      }
    }
  else if (! conf->batch_mode)
    {
      ssh_f_configd_write_pid_file(conf);
    }
#ifdef HAVE_SIGNAL
#ifdef SIGINT
  signal(SIGINT, ssh_f_configd_kill_signal);
#endif /* SIGINT */
#ifdef SIGQUIT
  signal(SIGQUIT, ssh_f_configd_kill_signal);
#endif /* SIGQUIT */
#ifdef SIGILL
  signal(SIGILL, ssh_f_configd_kill_signal);
#endif /* SIGILL */
#ifdef SIGBUS
  signal(SIGBUS, ssh_f_configd_kill_signal);
#endif /* SIGBUS */
#ifdef SIGSEGV
  signal(SIGSEGV, ssh_f_configd_kill_signal);
#endif /* SIGSEGV */
#ifdef SIGTERM
  signal(SIGTERM, ssh_f_configd_kill_signal);
#endif /* SIGTERM */
#endif /* HAVE_SIGNAL */
  SSH_DEBUG(5, ("FSMA initialization"));
  fe = FSMAInitialize(FSMA_API_VERSION, NULL);
  if (! FSMAS_IS_SUCCESS(fe))
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: FSMAInitialize failed.", conf->av0);
      ssh_f_configd_terminate(conf, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  SSH_DEBUG(5, ("FSMA module registration"));
  fe = FSMARegisterModule(&(conf->fsma_session), conf->fsma_module_oid_var);
  if (! FSMAS_IS_SUCCESS(fe))
    {
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                    "%s: FSMARegisterModule failed.", conf->av0);
      ssh_f_configd_terminate(conf, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
    }
  else
    {
      conf->fsma_session_active = TRUE;
    }
  SSH_DEBUG(5, ("FSMA policy initialization"));
  DfpInitialize(NULL);
  if (! conf->batch_mode)
    {
      handle = ssh_adt_enumerate_start(conf->prog_map);
      while (handle != SSH_ADT_INVALID)
        {
          prog = ssh_adt_map_lookup(conf->prog_map, handle);
          SSH_DEBUG(5, ("FSMA policy OID registration (%s)", prog->name));
          if (! prog->configuration_disable)
            {
              fe = DfpRegisterOID(conf->fsma_session, prog->oid_pattern_var);
              if (! FSMAS_IS_SUCCESS(fe))
                {
                  ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                                "%s: DfpRegisterOID %s (%s) failed (%d).",
                                conf->av0,
                                prog->oid_pattern,
                                prog->name,
                                (int)fe);
                  ssh_f_configd_terminate(conf,
                                          SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
                }
            }
          handle = ssh_adt_enumerate_next(conf->prog_map, handle);
        }
    }
  handle = ssh_adt_enumerate_start(conf->prog_map);
  while (handle != SSH_ADT_INVALID)
    {
      prog = ssh_adt_map_lookup(conf->prog_map, handle);
      if (! prog->configuration_disable)
        {
          SSH_DEBUG(5, ("Collect initial policy for %s", prog->name));
          ssh_f_configd_ssh_config_data_process(conf, prog);
        }
      handle = ssh_adt_enumerate_next(conf->prog_map, handle);
    }
  if (! conf->batch_mode)
    {
      SSH_DEBUG(5, ("FSMA message pump initialization."));
      fe = FSMAInitMessagePump(conf->fsma_session, &(conf->fsma_pump));
      if (! FSMAS_IS_SUCCESS(fe))
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                        "%s: FSMAInitMessagePump failed.", conf->av0);
          ssh_f_configd_terminate(conf, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
        }
      SSH_DEBUG(5, ("FSMA message pump startup."));
      fe = FSMARunMessagePump(conf->fsma_pump, 
                              ssh_f_config_fsma_message,
                              conf);
      if (! FSMAS_IS_SUCCESS(fe))
        {
          ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_ERROR, 
                        "%s: FSMARunMessagePump failed.", conf->av0);
          ssh_f_configd_terminate(conf, SSH_F_CONFIGD_FSMA_FATAL_EXITVAL);
        }
      SSH_DEBUG(5, ("FSMA message pump stopped."));
    }
  SSH_DEBUG(5, ("FSMA policy OID unregister"));
  DfpUnregister(conf->fsma_session);
  SSH_DEBUG(5, ("FSMA module unregister"));
  conf->fsma_session_active = FALSE;
  FSMAUnregisterModule(conf->fsma_session);
  if (conf->pid_file_written)
    {
      remove(conf->pid_file);
      conf->pid_file_written = FALSE;
    }
  ssh_f_configd_conf_free(conf);
  ssh_f_configd_terminate(NULL, 0);
  /*NOTREACHED*/
  return -1;
}

void ssh_f_configd_reconfigure_programs(SshFConfigDConf conf)
{
  SshADTHandle handle;
  SshFConfigDProg prog;

  if (! conf->terminating)
    {
      handle = ssh_adt_enumerate_start(conf->prog_map);
      while (handle != SSH_ADT_INVALID)
        {
          prog = ssh_adt_map_lookup(conf->prog_map, handle);
          if (! prog->configuration_disable)
            {
              SSH_DEBUG(5, ("Collect policy data for %s", prog->name));
              ssh_f_configd_ssh_config_data_process(conf, prog);
            }
          handle = ssh_adt_enumerate_next(conf->prog_map, handle);
        }
    }
}

FSMAERROR ssh_f_config_fsma_message(FSMASESSION fsma_session, 
                                    FSMAMESSAGE msg,
                                    void *context)
{
  SshFConfigDConf conf = (SshFConfigDConf)context;
  MSGID fsma_msg_id;
  unsigned int version;

  SSH_ASSERT(conf->fsma_session == fsma_session);
  fsma_msg_id = FSMAGetMessageID(msg);
  SSH_DEBUG(5, ("got message %d from FSMA", (int)fsma_msg_id));
  switch (fsma_msg_id)
    {
    case FSMAM_PING:
      SSH_DEBUG(7, ("Received FSMAM_PING"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case FSMAM_GET_VERSION:
      SSH_DEBUG(7, ("Received FSMAM_GET_VERSION"));
      version = SSH_F_CONFIGD_VERSION_NUMERIC;
      FSMASetResponseData(msg, &version, sizeof (version));
      break;

    case CHM_READ_BASE_POLICY:
      SSH_DEBUG(7, ("Received CHM_READ_BASE_POLICY"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case CHM_SET_POLICY:
      SSH_DEBUG(7, ("Received CHM_SET_POLICY"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case CHM_SET_POLICY_GROUP:
      SSH_DEBUG(7, ("Received CHM_SET_POLICY_GROUP"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case CHN_POLICY_CHANGED:
      SSH_DEBUG(7, ("Received CHN_POLICY_CHANGED"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case CHM_PROCESS_NOTIFY:
      SSH_DEBUG(7, ("Received CHM_PROCESS_NOTIFY"));
      ssh_f_configd_reconfigure_programs(conf);
      break;

    case FSMAM_QUERY_END_SESSION:
      SSH_DEBUG(7, ("Received FSMAM_QUERY_END_SESSION"));
      conf->terminating = TRUE;
      FSMAStopMessagePump(conf->fsma_pump);
      break;

    case FSMAM_END_SESSION:
      SSH_DEBUG(7, ("Received FSMAM_END_SESSION"));
      conf->terminating = TRUE;
      FSMAStopMessagePump(conf->fsma_pump);
      break;

    case FSMAM_INVALID:
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Invalid FSMA message.", conf->av0);
      break;

    default:
      ssh_log_event(SSH_LOGFACILITY_DAEMON, SSH_LOG_WARNING, 
                    "%s: Unknown FSMA message type (%d).", 
                    conf->av0, (int)fsma_msg_id);
    }
  return FSMAS_SUCCESS;
}

/* eof (ssh-f-configd.c) */
