/*

  ssholdfsm.h

  Author: Antti Huima <huima@ssh.fi>

  Copyright (c) 1999-2001 SSH Communications Security, Finland
  All rights reserved.

  Created Thu Aug 26 12:21:07 1999.

  */

#ifndef SSHOLDFSM_H_INCLUDED
#define SSHOLDFSM_H_INCLUDED

/* Type definition of a finite state machine object. */
typedef struct ssh_oldfsm *SshOldFSM;

/* Type definition of a thread object. */
typedef struct ssh_oldfsm_thread *SshOldFSMThread;

/* These are the allowed return values from a step function. */
typedef enum {
  SSH_OLDFSM_CONTINUE,          /* Continue from the next state */
  SSH_OLDFSM_FINISH,            /* End of thread. */
  SSH_OLDFSM_SUSPENDED,         /* Waiting for an async call. */

  /* This doesn't need to be returned explicitly. */
  SSH_OLDFSM_WAIT_CONDITION     /* Waiting for a condition variable.
                                   This is automatically returned by
                                   SSH_OLDFSM_CONDITION_WAIT(...) and
                                   does not need to be explicitly
                                   returned by user code. */
} SshOldFSMStepStatus;

/* The type of step functions. */
typedef SshOldFSMStepStatus (* SshOldFSMStepCB)(SshOldFSMThread thread);

/* A destructor type for internal global data structures. The blob
   `data' itself will be automatically ssh_xfree'd after the
   destructor returns. */
typedef void (* SshOldFSMDestructor)(void *gdata);

/* A destructor type for thread-specific internal data structures. The
   blob `tdata' itself will be automatically ssh_xfree'd after the
   destructor returns. */
typedef void (* SshOldFSMThreadDestructor)(void *tdata);

/* Message handler type. */
typedef void (* SshOldFSMMessageHandler)(SshOldFSMThread thread,
                                         SshUInt32 message);

/* State array item type. */
typedef struct ssh_oldfsm_state_map_item {
  char *state_id;               /* Unique ID for the state. */
  char *descr;                  /* Description of the state,
                                   for debugging purposes. */
  SshOldFSMStepCB func;            /* A function used to run a step
                                   starting from the state. */
} *SshOldFSMStateMapItem, SshOldFSMStateMapItemStruct;

/* Create a new finite state machine. `states' must be constant and
   remain valid forever (a static array perhaps). */
SshOldFSM ssh_oldfsm_allocate(size_t internal_data_size,
                              SshOldFSMStateMapItem states,
                              int num_states,
                              SshOldFSMDestructor destructor);

/* Destroy the FSM when next reaching the event loop.
   This checks that there are no active threads. */
void ssh_oldfsm_destroy(SshOldFSM fsm);

/* Set the next state. */
void ssh_oldfsm_set_next(SshOldFSMThread thread, char *next_state);

/* Get the internal global data structure. */
void *ssh_oldfsm_get_gdata(SshOldFSMThread thread);

/* Get the internal global data structure. */
void *ssh_oldfsm_get_gdata_fsm(SshOldFSM fsm);

/* Get the internal thread-specific data structure. */
void *ssh_oldfsm_get_tdata(SshOldFSMThread thread);

/* Set the debugging name. */
void ssh_oldfsm_set_thread_name(SshOldFSMThread thread, const char *name);

/* Get the debugging name. */
const char *ssh_oldfsm_get_thread_name(SshOldFSMThread thread);

/* Get the state id of the current state of the thread. */
const char *ssh_oldfsm_get_thread_current_state(SshOldFSMThread thread);

/* Fork, i.e. create a new thread. `fsm' is the state the thread will
   run on, `internal_data_size' is the size of the internal data blob
   to be allocated, `first_state' is the state where the thread starts
   from. `message_handle' is the message handling function and
   `destructor' is the optional destructor function. */
SshOldFSMThread ssh_oldfsm_spawn(SshOldFSM fsm,
                                 size_t internal_data_size,
                                 char *first_state,
                                 SshOldFSMMessageHandler message_handler,
                                 SshOldFSMThreadDestructor destructor);

/* Revive a thread from an external callback. */
void ssh_oldfsm_continue(SshOldFSMThread thread);

/* Get the underlying FSM for a thread. */
SshOldFSM ssh_oldfsm_get_fsm(SshOldFSMThread thread);

/* Push a thread-specific data object for a thread. */
void ssh_oldfsm_push_tdata(SshOldFSMThread thread, void *tdata);

/* Pop a thread-specific data pointer (and return it, for that matter). */
void *ssh_oldfsm_pop_tdata(SshOldFSMThread thread);

/* Push a message handler. */
void ssh_oldfsm_push_ehandler(SshOldFSMThread thread,
                              SshOldFSMMessageHandler message_handler);

/* Pop a message handler. */
SshOldFSMMessageHandler ssh_oldfsm_pop_ehandler(SshOldFSMThread thread);

/* Messages. */

/* Throw a message to another thread that must belong to the
   same FSM.

   If `recipient' does not have a message handler then the call
   does nothing. */
void ssh_oldfsm_throw(SshOldFSMThread thread,
                      SshOldFSMThread recipient,
                      SshUInt32 message);

/* Kill a thread that was suspended. Calling this function is legal
   only if it can be guaranteed that the thread won't get any
   ssh_oldfsm_continue calls after this; that is, that the thread was not
   waiting for an external callback that couldn't be cancelled.

   If the thread `thread' was waiting for a condition variable, then
   the thread is automatically removed from the variable's waiting
   list. */
void ssh_oldfsm_kill_thread(SshOldFSMThread thread);

/* Return the topmost element of the conf stack but do not remove it. */
/* Condition variables. */
typedef struct ssh_oldfsm_cond_var *SshOldFSMCondition;

/* Create a new condition variable. */
SshOldFSMCondition ssh_oldfsm_condition_create(SshOldFSM fsm);

/* Destroy a condition variable. When a condition variable is destroyed,
   not threads may be waiting for it. Use SSH_OLDFSM_CONDITION_BROADCAST if
   there are some threads left and you want to release them prior
   to destroying. */
void ssh_oldfsm_condition_destroy(SshOldFSMCondition condition);

/* Do not call these functions directly, prefer the macros below instead. */
void ssh_oldfsm_condition_wait(SshOldFSMThread thread, SshOldFSMCondition cv);
void ssh_oldfsm_condition_signal(SshOldFSMThread thread,
                                 SshOldFSMCondition cv);
void ssh_oldfsm_condition_broadcast(SshOldFSMThread thread,
                                    SshOldFSMCondition cv);

/* The async function call streamlining functions. Do not call these
   directly! Use macros below instead. */
void ssh_oldfsm_set_callback_flag(SshOldFSMThread thread);
void ssh_oldfsm_drop_callback_flag(SshOldFSMThread thread);
Boolean ssh_oldfsm_get_callback_flag(SshOldFSMThread thread);

/* Wait for a condition. */
#define SSH_OLDFSM_CONDITION_WAIT(cv)           \
do                                              \
{                                               \
  ssh_oldfsm_condition_wait(thread, cv);        \
  return SSH_OLDFSM_WAIT_CONDITION;             \
}                                               \
while(0)

/* Signal a condition. */
#define SSH_OLDFSM_CONDITION_SIGNAL(cv)    \
  ssh_oldfsm_condition_signal(thread, cv)

/* Broadcast a condition. */
#define SSH_OLDFSM_CONDITION_BROADCAST(cv) \
  ssh_oldfsm_condition_broadcast(thread, cv)

/* Calculate the size of a state array. */
#define SSH_OLDFSM_NUM_STATES(array) (sizeof(array)/sizeof(array[0]))

/* Function header for a step function. */
#define SSH_OLDFSM_STEP(name) SshOldFSMStepStatus name(SshOldFSMThread thread)

/* Get global data item. */
#define SSH_OLDFSM_GDATA(gtype)    \
  gtype gdata = ssh_oldfsm_get_gdata(thread)

/* Get thread data. */
#define SSH_OLDFSM_TDATA(ttype)    \
  ttype tdata = ssh_oldfsm_get_tdata(thread)

/* Get both datas. */
#define SSH_OLDFSM_DATA(gtype, ttype)      \
  SSH_OLDFSM_GDATA(gtype); SSH_OLDFSM_TDATA(ttype)

/* Fork. */
#define SSH_OLDFSM_FORK(s,f,e,d) \
  ssh_oldfsm_spawn(ssh_oldfsm_get_fsm(thread), s, f, e, d)

/* Throw. */
#define SSH_OLDFSM_THROW(r, e) \
  ssh_oldfsm_throw(thread, r, e)

/* Next state. */
#define SSH_OLDFSM_SET_NEXT(n) ssh_oldfsm_set_next(thread, n)

/* State arrays made easy. See e.g. ssholdfsmstreams_states.h. */
#define SSH_OLDFSM_STATE(x,y,z) { x, y, z },

/* Call a function (in general, run a block) that will return a
   callback, either immediately or later. Can be used only inside step
   functions. Terminates the step function after the block. */
#define SSH_OLDFSM_ASYNC_CALL(x)                        \
do                                                      \
{                                                       \
  SSH_ASSERT(!(ssh_oldfsm_get_callback_flag(thread)));  \
  ssh_oldfsm_set_callback_flag(thread);                 \
  do                                                    \
    {                                                   \
      x;                                                \
    }                                                   \
  while(0);                                             \
  if (ssh_oldfsm_get_callback_flag(thread))             \
    return SSH_OLDFSM_SUSPENDED;                        \
  return SSH_OLDFSM_CONTINUE;                           \
}                                                       \
while(0)

/* This macro can be used inside a callback to revive the thread. Use
   in conjunction with calls made with SSH_OLDFSM_ASYNC_CALL. This macro
   does *NOT* return implicitly because the callback might want a
   value to be returned. */
#define SSH_OLDFSM_CONTINUE_AFTER_CALLBACK(thread)      \
do                                                      \
{                                                       \
  SSH_ASSERT(ssh_oldfsm_get_callback_flag(thread));     \
  ssh_oldfsm_drop_callback_flag(thread);                \
  ssh_oldfsm_continue(thread);                          \
}                                                       \
while(0)

#endif /* SSHFSM_H_INCLUDED */
