/*
 * Author: Tero Kivinen <kivinen@iki.fi>
 *         Kenneth Oksanen <cessu@iki.fi>
 *
 * Copyright (c) 2001 SSH Communications Security Oy <info@ssh.fi>
 */
/*
 *        Program: sshutil
 *        $Source: /ssh/CVS/src/lib/sshutil/sshcore/sshglobals.c,v $
 *        $Author: cessu $
 *
 *        Creation          : 14:04 Dec 19 2001 kivinen
 *        Last Modification : 18:05 Dec 19 2001 kivinen
 *        Last check in     : $Date: 2002/01/02 13:47:17 $
 *        Revision number   : $Revision: 1.2 $
 *        State             : $State: Exp $
 *        Version           : 1.107
 *        
 *
 *        Description       : Support for global variables
 *
 *
 *        $Log: sshglobals.c,v $ *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        $EndLog$
 */

#include "sshincludes.h"
#include "sshglobals.h"


#define SSH_DEBUG_MODULE "SshGlobals"


/* This is the hash table entry. The name is the constant string of the
   variable name (it is the first string given to the ssh_global_get).
   Immediately after the name pointer starts the actual value of the data. */
typedef struct SshGlobalsHashItemRec {
  struct SshGlobalsHashItemRec *next;
  SshUInt32 hash;
  const char *name;
} *SshGlobalsHashItem, SshGlobalsHashItemStruct;


/* Global entry point of the hash table. */
SshGlobalsHashItem *ssh_globals = NULL;

/* How many items does the hash table contain, and how many entries
   are in it now.  We rehash to double size whenever the number of
   items exceeds the number of hash table slots.  The number of
   entries must be a power of two.  Collisions are handled with
   separate chaining. */
SshUInt32 ssh_global_number_of_items = 0L;
SshUInt32 ssh_global_size_of_table = 0L;


static SshUInt32 ssh_global_hash_fun(const char *s)
{
  SshUInt32 h = 0xDEADBEEF;
  
  /* The `one-at-a-time' hash function from Robert J. Jenkins' web
     articles.  One of the better string hash functions I know of. */
  while (*s) {
    h += *s++;
    h += h << 10;
    h ^= h >> 6;
  }
  h += h << 3;
  h ^= h >> 11;
  h += h << 15;
  return h;
}


/* Find item from the hash table. */
static SshGlobalsHashItem ssh_global_hash_item_find(const char *str)
{
  SshGlobalsHashItem p;
  SshUInt32 h;

  if (ssh_globals == NULL)
    return NULL;

  h = ssh_global_hash_fun(str);
  /* Assert that the size of the table is a power of two. */
  SSH_ASSERT((ssh_global_size_of_table & (ssh_global_size_of_table - 1)) == 0);

  for (p = ssh_globals[h & (ssh_global_size_of_table - 1)];
       p != NULL;
       p = p->next)
    {
      if (h == p->hash && strcmp(str, p->name) == 0)
        return p;
    }
  return NULL;
}


/* Add new entry to hash table */
static void ssh_global_hash_item_add(SshGlobalsHashItem item)
{
  if (ssh_globals == NULL)
    {
      ssh_global_size_of_table = 8;
      ssh_globals = ssh_xcalloc(ssh_global_size_of_table,
                                sizeof(SshGlobalsHashItem));
    }
  else if (ssh_global_number_of_items > ssh_global_size_of_table)
    {
      /* Rehash the old table. */
      SshGlobalsHashItem *old_globals = ssh_globals;
      SshUInt32 old_size = ssh_global_size_of_table;
      SshUInt32 i;

      /* Double the size. */
      ssh_global_size_of_table *= 2;
      ssh_globals = ssh_xcalloc(ssh_global_size_of_table,
                                sizeof(SshGlobalsHashItem));
      for (i = 0; i < old_size; i++)
        /* Traverse all slots in the old table. */
        {
          SshGlobalsHashItem p, p_next;

          for (p = old_globals[i]; p != NULL; p = p_next) 
            /* All items in the chain. */
            {
              SshUInt32 new_i = p->hash & (ssh_global_size_of_table - 1);

              SSH_ASSERT((p->hash & (old_size - 1)) == i);

              /* Push the item first in the new slot.  As a side
                 effect the chain is reversed, but this shouldn't
                 matter. */
              p_next = p->next;
              p->next = ssh_globals[new_i];
              ssh_globals[new_i] = p;
            }
        }
      ssh_xfree(old_globals);
    }

  {
    SshUInt32 i;

    item->hash = ssh_global_hash_fun(item->name);
    i = item->hash & (ssh_global_size_of_table - 1);
    item->next = ssh_globals[i];
    ssh_globals[i] = item;
    ssh_global_number_of_items++;
  }
}

/* Function that returns pointer to the global variable based on the name of
   the global variable. If the variable is used before it is initialized (i.e
   the ssh_global_init_variable is not called before the first use of the
   ssh_global_get), then ssh_global_get might print out warning, and the value
   of the variable will be all zeros. */
void *ssh_global_get(const char *str, size_t variable_size)
{
  SshGlobalsHashItem item;

  item = ssh_global_hash_item_find(str);

  if (item == NULL)
    {
      ssh_warning("Use of uninitialized global variable %s", str);
      item = ssh_xcalloc(1, sizeof(SshGlobalsHashItemStruct) + variable_size);
      item->name = str;
      ssh_global_hash_item_add(item);
    }
  return (void *) (((char *) item) + sizeof(SshGlobalsHashItemStruct));
}


/* Initialize variable to have value of all zeros. This makes the variable to
   be known to the system, and ssh_global_get will assume not print out
   warnings about use of uninitialized variables. Call this function twice
   will print out warning. This returns always returns 0. */
int ssh_global_init_variable(const char *str, size_t variable_size)
{
  SshGlobalsHashItem item;

  item = ssh_global_hash_item_find(str);

  if (item != NULL)
    {
      ssh_warning("Duplicate initialization of the variable %s", str);
      return 0;
    }
  item = ssh_xcalloc(1, sizeof(SshGlobalsHashItemStruct) + variable_size);
  item->name = str;
  ssh_global_hash_item_add(item);
  return 0;
}


/* Initialize global variables system. Calling this will reset all global
   variables to uninitialized state. */
void ssh_global_init(void)
{
  ssh_globals = NULL;
}

/* Uninitialize global variables system. Calling this will reset all global
   variables to uninitialized state, and free all state allocated for the
   global variables. */
void ssh_global_uninit(void)
{
  SshGlobalsHashItem p, p_next;
  SshUInt32 i;

  if (ssh_globals == NULL)
    return;

  for (i = 0; i < ssh_global_size_of_table; i++)
    for (p = ssh_globals[i]; p != NULL; p = p_next)
      {
        p_next = p->next;
        ssh_xfree(p);
      }
  ssh_xfree(ssh_globals);

  ssh_globals = NULL;
  ssh_global_number_of_items = ssh_global_size_of_table = 0L;
}
