/*

sshunixuser.c

Authors: Tatu Ylonen   <ylo@ssh.fi>
         Sami Lehtinen <sjl@ssh.com>
        
Copyright (c) 1998,2001 SSH Communications Security, Finland
                        All rights reserved

Manipulating user information in SSH server (mostly password validation).
This is a simple implementation for generic unix platforms.

*/

#define SSH_ALLOW_CPLUSPLUS_KEYWORDS
#include "sshsessionincludes.h"
#include "sshuser.h"
#include "sshdsprintf.h"
#ifdef SSHDIST_SESSION_SIA
#ifdef HAVE_SIA
#include "sshsia.h"
#endif /* HAVE_SIA */
#endif /* SSHDIST_SESSION_SIA */

#ifdef HAVE_SCO_ETC_SHADOW
# include <sys/security.h>
# include <sys/audit.h>
# include <prot.h>
# ifdef HAVE_SHADOW_H
#  include <shadow.h>
# endif /* HAVE_SHADOW_H */
#else /* HAVE_SCO_ETC_SHADOW */
# ifdef HAVE_ETC_SHADOW
#  ifdef HAVE_SHADOW_H
#   include <shadow.h>
#  endif /* HAVE_SHADOW_H */
# endif /* HAVE_ETC_SHADOW */
#endif /* HAVE_SCO_ETC_SHADOW */
#ifdef HAVE_ETC_SECURITY_PASSWD_ADJUNCT
# include <sys/label.h>
# include <sys/audit.h>
# include <pwdadj.h>
#endif /* HAVE_ETC_SECURITY_PASSWD_ADJUNCT */
#ifdef HAVE_ULTRIX_SHADOW_PASSWORDS
# include <auth.h>
# include <sys/svcinfo.h>
#endif /* HAVE_ULTRIX_SHADOW_PASSWORDS */
#include <netdb.h>

#ifdef HAVE_LIBBSM
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif /* !HAVE_SYS_PARAM_H */
#ifdef HAVE_BSM_AUDIT_H
#include <bsm/audit.h>
#endif /* !HAVE_BSM_AUDIT_H */
#ifdef HAVE_BSM_LIBBSM_H
#include <bsm/libbsm.h>
#endif /* !HAVE_BSM_LIBBSM_H */

#endif /* HAVE_LIBBSM */

#ifdef _HPUX_SOURCE
#undef ctime
#include <sys/types.h>
#include <hpsecurity.h>
#include <prot.h>
#endif /* _HPUX_SOURCE */

#ifdef HAVE_USERSEC_H
#include <usersec.h>
#endif /* HAVE_USERSEC_H */

#ifdef HAVE_LOGINRESTRICTIONS
#include <login.h>
#endif /* HAVE_LOGINRESTRICTIONS */

#ifdef HAVE_GRP_H
#include <grp.h>
#endif /* HAVE_GRP_H */

#ifdef HAVE_INITGROUPS
#ifndef HAVE_PROTO_INITGROUPS
int initgroups (char *, gid_t);
#endif /* !HAVE_PROTO_INITGROUPS */
#endif /* HAVE_INITGROUPS */


#ifdef KERBEROS

#undef ctime
#undef free

/* The #%@!$(#$ krb5.h header redefines these.  ARRRGH! -ylo */
#undef SIZEOF_INT
#undef SIZEOF_LONG
#undef SIZEOF_SHORT
#undef HAVE_STDARG_H
#undef HAVE_SYS_TYPES_H
#include <krb5.h>

#endif /* KERBEROS */

#define SSH_DEBUG_MODULE "SshUnixUser"

extern char *crypt(const char *key, const char *salt);

/* Group structure. */
struct SshGroupRec
{
  gid_t gid;
  char *name;
};

/* Data type to hold machine-specific user data. */

struct SshUserRec
{
  char *name;
  char *group;
  char *dir;
  char *shell;
  char *correct_encrypted_passwd;
  uid_t uid;
  gid_t gid;
  Boolean password_needs_change;
  Boolean login_allowed;

  /* List all the groups, primary and supplementary, that the user is
     in. */
  SshGroup *groups;
  
#ifdef KERBEROS
  /* Set to TRUE if kcontext and krb_user below have been initialized. */
  Boolean krb_initialized;
  
  /* Kerberos context for this user authentication. */
  krb5_context kcontext;

  /* Kerberos principal name for this user.  Sometimes this can be different
     from the local user name.  See ssh_user_initialize for more details. */
  krb5_principal krb_user;

  /* This is a string representation of the kerberos user name. */
  char *krb_user_name;

  /* Ticket granting ticket to save in the user's credentials after login,
     or NULL if none. */
  krb5_creds *krb_tgt;
  
  /* Local username obtained from krb5_aname_to_localname from
     kerberos name */
  char *krb_local_username;
#endif /* KERBEROS */

};

/* Returns true if logging in as the specified user is permitted.  Returns
   false if login is not permitted (e.g., the account is expired). */

Boolean ssh_login_permitted(SshUser uc)
{
  char passwd[20];              /* Only for account lock check */
  const char *user = ssh_user_name(uc);
  
  strncpy(passwd, uc->correct_encrypted_passwd, sizeof(passwd));
  passwd[sizeof(passwd) - 1] = '\0';

#ifdef HAVE_LOGINRESTRICTIONS
  {
    char *msg;

    /* XXX memory leak here, since we must free() the msg */
    if (loginrestrictions(user, S_RLOGIN, NULL, &msg) < 0)
      {
        SSH_DEBUG(2, ("loginrestrictions S_RLOGIN failed: %.200s",
                      msg));
        return FALSE;
      }
  }
#endif /* HAVE_LOGINRESTRICTIONS */

#ifdef HAVE_PASSWDEXPIRED
  {
    char *message;
    int ret;

    if (ret = passwdexpired((char*)user, &message) != 0)
      {
        if (ret > 0)
          uc->password_needs_change = TRUE;
        else
          {
            SSH_DEBUG(2, ("passwdexpired failed: %.200s",
                           message));
            return FALSE;
          } 
      }
      /* XXX mem leak, must free message */
  }

#endif /* HAVE_PASSWDEXPIRED */

#ifdef HAVE_USERSEC_H
  {
    char *expiration, current_time[100], normalized[100];
    int rlogin_permitted;
    SshTime t;
    struct SshCalendarTimeRec tm[1];
    int account_is_locked;
    
    if (setuserdb(S_READ) < 0)
      {
        if (getuid() == 0) /* It's OK to fail here if we are not root */
          {
            SSH_DEBUG(2, ("setuserdb S_READ failed: %.200s.", 
                          strerror(errno)));
          }
        return FALSE;
      }
    if (getuserattr((char *)user, S_RLOGINCHK, &rlogin_permitted,
                    SEC_BOOL) < 0)
      {
        if (getuid() == 0) /* It's OK to fail here if we are not root */
          {
            SSH_DEBUG(2, ("getuserattr S_RLOGINCHK failed: %.200s",
                          strerror(errno)));
          }
        enduserdb();
        return FALSE;
      }
    if (getuserattr((char *)user, S_EXPIRATION, &expiration, SEC_CHAR) < 0)
      {
        SSH_DEBUG(2, ("getuserattr S_EXPIRATION failed: %.200s.", 
                      strerror(errno)));
        enduserdb();
        return FALSE;
      }
#ifdef S_LOCKED
    if (getuserattr(user, S_LOCKED, &account_is_locked, SEC_BOOL) < 0)
      {
        SSH_DEBUG(2, ("getuserattr S_LOCKED failed: %.200s.", 
                      strerror(errno)));
        enduserdb();
        return FALSE;
      }
    if (account_is_locked)
      {
        SSH_DEBUG(2, ("Account %.100s is locked.", user));
        enduserdb();
        return FALSE;
      }
#endif /* S_LOCKED */
    if (!rlogin_permitted)
      {
        SSH_DEBUG(2, ("Remote logins to account %.100s not permitted by "
                      "user profile.",
                      user));
        enduserdb();
        return FALSE;
      }
    if (strcmp(expiration, "0") == 0)
      {
        /* The account does not expire - return success immediately. */
        enduserdb();
        return TRUE;
      }
    if (strlen(expiration) != 10)
      {
        SSH_DEBUG(2, ("Account %.100s expiration date is in wrong format.", 
                      user));
        enduserdb();
        return FALSE;
      }
    t = ssh_time();
    ssh_calendar_time(t, tm, TRUE);
    ssh_snprintf(current_time, sizeof(current_time), "%04d%02d%02d%02d%02d",
                 tm->year, tm->month + 1, tm->monthday,
                 tm->hour, tm->minute);
    if (expiration[8] < '7') /* Assume year < 70 is 20YY. */
      strcpy(normalized, "20");
    else
      strcpy(normalized, "19");
    strcat(normalized, expiration + 8);
    strcat(normalized, expiration);
    normalized[12] = '\0';
    if (strcmp(normalized, current_time) < 0)
      {
        SSH_DEBUG(2, ("Account %.100s has expired - access denied.", user));
        enduserdb();
        return FALSE;
      }
    enduserdb();
  }
#endif /* HAVE_USERSEC_H */

#ifdef _HPUX_SOURCE
  if (iscomsec())
    {
      /* System is in trusted mode */
      struct pr_passwd *pr = getprpwnam((char *)user);
      
      if (!pr || locked_out(pr))
        {
          SSH_DEBUG(2, ("Account %.100s not valid - access denied.", user));
          endprpwent();
          return FALSE;
        }
      
      /* Check whether the password has expired and needs to be changed. */
      if (pr->uflg.fg_expire && pr->uflg.fg_schange)
        {
          if (pr->ufld.fd_expire > 0 &&
              (pr->ufld.fd_expire + pr->ufld.fd_schange) <= ssh_time())
            uc->password_needs_change = TRUE;
        }
      else
        {
          if (pr->sflg.fg_expire && pr->uflg.fg_schange &&
              pr->sfld.fd_expire > 0 &&
              ((pr->sfld.fd_expire + pr->ufld.fd_schange) <= ssh_time()))
            uc->password_needs_change = TRUE;
        }
      endprpwent();
    }
  else
    {
      /* The system is not in trusted mode. Check whether the password
         needs to be changed, using the traditional interface... */
      struct passwd *pwd = getpwnam((char *)user);
      SshTime current_time = ssh_time();
      long pw_expiry,
        pw_max, /* Max validity in days */
        pw_secs; /* Last change */
      
      if (!pwd)
        {
          SSH_DEBUG(2, ("Account %.100s not valid - access denied.", user));
          endpwent();
          return FALSE;
        }

      if (pwd->pw_age && strlen(pwd->pw_age) > 0)
        {
          pw_expiry = a64l(pwd->pw_age);
          pw_max = (pw_expiry & 077) * 7;
          pw_secs = (pw_expiry >> 12) * 7 * 24 * 60 * 60;
          
          if (pw_max >= 0 &&
              ((current_time - pw_secs) / 60 / 60 / 24) > pw_max)
            {
              SSH_DEBUG(2, ("Password expired, must be changed"));
              uc->password_needs_change = TRUE;
            }
        }
      
      endpwent();
    }
  
#endif /* _HPUX_SOURCE */
  
#ifdef HAVE_ETC_SHADOW
  {
    struct spwd *sp;
#ifdef HAVE_SETSPENT
    setspent();
#endif /* HAVE_SETSPENT */
    sp = (struct spwd *)getspnam(user);
#if defined(SECURE_RPC) && defined(NIS_PLUS)
    if (geteuid() == UID_ROOT && ssh_user_uid(uc) != UID_ROOT
        && (!sp || !sp->sp_pwdp || !strcmp(sp->sp_pwdp,"*NP*")))
      {
        if (seteuid(ssh_user_uid(uc)) >= 0)
          {
            sp = getspnam(user); /* retry as user */
            seteuid(UID_ROOT); 
          }
      }
#endif /* SECURE_RPC && NIS_PLUS */
    if (!sp)
      {
        /*
         * Some systems, e.g.: IRIX, may or may not have /etc/shadow.
         * Just check if there is one. If such system is also an YP
         * client, then valid password might already be present in passwd
         * structure. Just check if it's other than "x". Assume that
         * YP server is always right if this is the case.
         *                                      appro@fy.chalmers.se
         */
        struct stat sbf;
        
        if ((stat(SHADOW, &sbf) == 0) &&
            strcmp(uc->correct_encrypted_passwd, "x") == 0)
          {
            SSH_DEBUG(2, ("Can't find %.100s's shadow - access denied.", 
                          user));
            endspent();
            return FALSE;
          }
      }
    else
      {
        SshTime today = ssh_time()/24/60/60; /* what a day! */

#ifdef HAVE_STRUCT_SPWD_EXPIRE
        /* Check for expiration date */
        if (sp->sp_expire > 0 && today > sp->sp_expire)
          {
            SSH_DEBUG(2, ("Account %.100s has expired - access denied.", 
                          user));
            endspent();
            return FALSE;
          }
#endif /* HAVE_STRUCT_SPWD_EXPIRE */
        
#ifdef HAVE_STRUCT_SPWD_INACT
        /* Check for last login */
        if (sp->sp_inact > 0)
          {
            char buf[64];
            SshTime llt;
            
            llt = ssh_user_get_last_login_time(uc, buf, sizeof(buf));
            if (llt && (today - llt/24/60/60) > sp->sp_inact)
              {
                SSH_DEBUG(2, ("Account %.100s was inactive for more than %d "
                              "days.", user, sp->sp_inact));
                endspent();
                return FALSE;
              }
          }
#endif /* HAVE_STRUCT_SPWD_INACT */
        
        /* Check if password is valid */
        if (sp->sp_lstchg == 0 ||
            (sp->sp_max > 0 && today > sp->sp_lstchg + sp->sp_max))
          {
            SSH_DEBUG(2, ("Account %.100s's password is too old - forced "
                          "to change.", user));
            uc->password_needs_change = TRUE;
          }
        strncpy(passwd, sp->sp_pwdp, sizeof(passwd));
        passwd[sizeof(passwd) - 1] = '\0';
      }
    endspent();
  }
#endif /* HAVE_ETC_SHADOW */
  /*
   * Check if account is locked. Check if encrypted password starts
   * with "*LK*".
   */

  if (strncmp(passwd,"*LK*", 4) == 0)
    {
      SSH_DEBUG(2, ("Account %.100s is locked.", user));
      return FALSE;
    }
#ifdef CHECK_ETC_SHELLS
  {
    int  invalid = 1;
    char *shell = pwd->pw_shell, *etc_shell, *getusershell();
    
    if (!shell || !*shell)
      shell = DEFAULT_SHELL;
    
    while (invalid && (etc_shell = getusershell()))
      invalid = strcmp(etc_shell, shell);
    endusershell();
    
    if (invalid)
      {
        SSH_DEBUG(2, ("Account %.100s doesn't have valid shell", user));
        return FALSE;
      }
  }
#endif /* CHECK_ETC_SHELLS */

  return TRUE;
}

void ssh_user_record_login_failure(SshUser uc, const char *remote_host)
{
  SSH_PRECOND(uc != NULL);
  
#ifdef HAVE_LOGINFAILED
  /* record to the system's database that we have an unsucessful login */
  /* XXX this should be done on all systems, not just on those with
     loginfailed */
  /* Last parameter is `tty', but we haven't allocated one yet. Use
     dummy "ssh" value. */
  loginfailed(ssh_user_name(uc), remote_host, "ssh");
#endif /* HAVE_LOGINFAILED */

#ifdef _HPUX_SOURCE
  if (iscomsec())
    {
      /* System is in trusted mode */
      struct pr_passwd save_pr, *pr = getprpwnam((char *)ssh_user_name(uc));
      if (pr)
        {
          save_pr = *pr;
          pr = &save_pr;  /* Point to a copy, not the original */
          
          if (pr->uflg.fg_nlogins)
            pr->ufld.fd_nlogins++;
          else
            {
              pr->uflg.fg_nlogins = 1;
              pr->ufld.fd_nlogins = 1;
            }

          pr->uflg.fg_ulogin = 1;
          pr->ufld.fd_ulogin = ssh_time();

          putprpwnam((char *)ssh_user_name(uc), pr);
        }
      endprpwent();
    }
#endif /* _HPUX_SOURCE */
  
  uc->login_allowed = ssh_login_permitted(uc);
}


/* Forward declaration. */
SshUser ssh_user_initialize_with_pw(struct passwd *pw, Boolean privileged);

/* Allocates and initializes a context for the user.  The context is used
   to cache information about the particular user.  Returns NULL if the
   user does not exist. If `user' is NULL, use getuid(). 'privileged'
   should only be set, when the process is supposedly run with root
   privileges. If it is FALSE, ssh_user_initialize doesn't try to look for
   shadow passwords etc. */

SshUser ssh_user_initialize(const char *user, Boolean privileged)
{
  struct passwd *pw;

#ifdef KERBEROS
  krb5_context kcontext;
  krb5_principal krb_user;
  krb5_error_code kerr;
  char local_user[256], user_copy[256];
  char *krb_user_name;
  SshUser uc;

  /* Initialize a Kerberos context. */
  kerr = krb5_init_context(&kcontext);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_init_context failed %d", kerr));
      goto nokerberos;
    }
  
  /* Check if the name contains an explicit kerberos realm name.  Note that
     we support both '%' and '@' syntaxes.  The '%' syntax is sometimes used
     when a client program interprets user@host specially, and we still want
     to pass a kerberos name to the server. */
  if (user && (strchr(user, '@') || strchr(user, '%')))
    {
      char buf[256];

      /* The name is a kerberos name.  Map it to a local name, and parse
         the kerberos principal into krb_user.  First normalize the '%'
         syntax to the normal kerberos syntax. */
      ssh_snprintf(buf, sizeof(buf), "%s", user);
      if (strchr(buf, '%'))
        *strchr(buf, '%') = '\0';

      /* Try to parse the kerberos name. */
      kerr = krb5_parse_name(kcontext, buf, &krb_user);
      if (kerr)
        {
          krb5_free_context(kcontext);
          goto nokerberos;
        }

      /* Try to map the kerberos principal to a local user name. */
      kerr = krb5_aname_to_localname(kcontext, krb_user,
                                     sizeof(local_user), local_user);
      if (kerr)
        {
          krb5_free_principal(kcontext, krb_user);
          krb5_free_context(kcontext);
          goto nokerberos;
        }
    }
  else
    {
      /* If user is NULL, get the current user's name. */
      if (user == NULL)
        {
          pw = getpwuid(getuid());
          if (!pw)
            goto nokerberos;
          /* Copy the local name to our own buffer to make sure it remains
             valid during the parse call (in case it calls getpw*). */
          ssh_snprintf(user_copy, sizeof(user_copy), "%s", pw->pw_name);
          user = user_copy;
        }
      
      /* It does not look like a kerberos name.  However, still parse it
         as a kerberos principal name. */
      kerr = krb5_parse_name(kcontext, user, &krb_user);
      if (kerr)
        {
          krb5_free_context(kcontext);
          goto nokerberos;
        }

      /* Use the same name as the local user login name. */
      ssh_snprintf(local_user, sizeof(local_user), "%s", user);
    }
  /* local_user now contains the local user name, and krb_user the kerberos
     name. */

  /* To make sure we don't accidentally use this... */
  user = NULL;

  /* Get the string representation of the kerberos name. */
  krb5_unparse_name(kcontext, krb_user, &krb_user_name);
  
  /* Initialize the SshUser object using the local user name. */
  pw = getpwnam(local_user);
  uc = ssh_user_initialize_with_pw(pw, privileged);
  if (!uc)
    {
      free(krb_user_name);
      krb5_free_principal(kcontext, krb_user);
      krb5_free_context(kcontext);
      return NULL;
    }

  /* Store the kerberos data into the SshUser object. */
  uc->krb_local_username = ssh_xstrdup(local_user);
  uc->kcontext = kcontext;
  uc->krb_user = krb_user;
  uc->krb_user_name = krb_user_name;
  uc->krb_initialized = TRUE;

  /* Return the user object with the Kerberos data. */
  return uc;  

 nokerberos:
#endif /* KERBEROS */
  if (user == NULL)
    pw = getpwuid(getuid());
  else 
    pw = getpwnam(user);

  return ssh_user_initialize_with_pw(pw, privileged);
}

/* As above, but we explicitely want to use uid (instead of name). */
SshUser ssh_user_initialize_with_uid(uid_t uid, Boolean privileged)
{
  struct passwd *pw;

  pw = getpwuid(uid);

  return ssh_user_initialize_with_pw(pw, privileged);
}

/* As above, but uses struct passwd. This function does all the work. */
SshUser ssh_user_initialize_with_pw(struct passwd *pw, Boolean privileged)
{
  SshUser uc;

  char correct_passwd[200];

  if (!pw)
    return NULL;
  
  uc = ssh_xcalloc(1, sizeof(*uc));

  uc->name = ssh_xstrdup(pw->pw_name);
  uc->group = NULL;
  uc->dir = ssh_xstrdup(pw->pw_dir);
  uc->uid = pw->pw_uid;
  uc->gid = pw->pw_gid;

  if (strcmp(pw->pw_shell, "") == 0)
    {
      uc->shell = ssh_xstrdup("/bin/sh");
    }
  else
    {    
      uc->shell = ssh_xstrdup(pw->pw_shell);
    }
  
  if (privileged)
    {
      
      /* Save the encrypted password. */
      strncpy(correct_passwd, pw->pw_passwd, sizeof(correct_passwd));

#ifdef HAVE_SIA
#ifdef SSHDIST_SESSION_SIA
      /* pr->pw_passwd may not be the real encrypted password (it won't be
         under Enhanced Security), but we don't care because we never look at
         uc->correct_encrypted_passwd.  We let ssh_sia_validate_user() do all
         the work instead. */
#endif /* SSHDIST_SESSION_SIA */
#else /* HAVE_SIA */
      /* If we have shadow passwords, lookup the real encrypted password from
         the shadow file, and replace the saved encrypted password with the
         real encrypted password. */
#if defined(HAVE_SCO_ETC_SHADOW)
      {
        struct pr_passwd *pr = getprpwnam(ssh_user_name(uc));
        pr = getprpwnam(ssh_user_name(uc));
        if (pr)
          strncpy(correct_passwd, pr->ufld.fd_encrypt, sizeof(correct_passwd));
        endprpwent();
      }
#else /* defined(HAVE_SCO_ETC_SHADOW) */
#ifdef _HPUX_SOURCE
      {
        if (iscomsec())
          {
            struct pr_passwd *pr = getprpwnam((char *)ssh_user_name(uc));
            if (pr && pr->uflg.fg_encrypt)
              strncpy(correct_passwd, pr->ufld.fd_encrypt, sizeof(correct_passwd));
            endprpwent();
          }
      }
#else /* _HPUX_SOURCE */
#ifdef HAVE_ETC_SHADOW
      {
        struct spwd *sp;
#ifdef HAVE_SETSPENT
        setspent();
#endif /* HAVE_SETSPENT */
        sp = getspnam(ssh_user_name(uc));
#if defined(SECURE_RPC) && defined(NIS_PLUS)
        if (geteuid() == UID_ROOT && ssh_user_uid(uc) != UID_ROOT &&
            (!sp || !sp->sp_pwdp || !strcmp(sp->sp_pwdp,"*NP*")))
          if (seteuid(ssh_user_uid(uc)) >= 0)
            {
              sp = getspnam(ssh_user_name(uc)); /* retry as user */   
              seteuid(UID_ROOT);
            }
#endif /* SECURE_RPC && NIS_PLUS */
        if (sp)
          strncpy(correct_passwd, sp->sp_pwdp, sizeof(correct_passwd));
        endspent();
      }
#else /* HAVE_ETC_SHADOW */
#ifdef HAVE_ETC_SECURITY_PASSWD_ADJUNCT
      {
        struct passwd_adjunct *sp = getpwanam(ssh_user_name(uc));
        if (sp)
          strncpy(correct_passwd, sp->pwa_passwd, sizeof(correct_passwd));
        endpwaent();
      }
#else /* HAVE_ETC_SECURITY_PASSWD_ADJUNCT */
#ifdef HAVE_ETC_SECURITY_PASSWD /* AIX, at least.  Is there an easier way? */
      {
        FILE *f;
        char line[1024], looking_for_user[200], *cp;
        int found_user = 0;
        f = fopen("/etc/security/passwd", "r");
        if (f)
          {
            ssh_snprintf(looking_for_user, sizeof(looking_for_user), 
                         "%.190s:", ssh_user_name(uc));
            while (fgets(line, sizeof(line), f))
              {
                if (strchr(line, '\n'))
                  *strchr(line, '\n') = 0;
                if (strcmp(line, looking_for_user) == 0)
                  found_user = 1;
                else
                  if (line[0] != '\t' && line[0] != ' ')
                    found_user = 0;
                  else
                    if (found_user)
                      {
                        for (cp = line; *cp == ' ' || *cp == '\t'; cp++)
                          ;
                        if (strncmp(cp, 
                                    "password = ", 
                                    strlen("password = ")) == 0)
                          {
                            strncpy(correct_passwd, cp + strlen("password = "), 
                                    sizeof(correct_passwd));
                            correct_passwd[sizeof(correct_passwd) - 1] = 0;
                            break;
                          }
                      }
              }
            fclose(f);
          }
      }
#endif /* HAVE_ETC_SECURITY_PASSWD */
#endif /* HAVE_ETC_SECURITY_PASSWD_ADJUNCT */
#endif /* HAVE_ETC_SHADOW */
#endif /* _HPUX_SOURCE */
#endif /* HAVE_SCO_ETC_SHADOW */
#endif /* HAVE_SIA */

      uc->correct_encrypted_passwd = ssh_xstrdup(correct_passwd);

      uc->login_allowed = ssh_login_permitted(uc);
    }
  else /* !privileged */
    {
      uc->correct_encrypted_passwd = NULL;
      uc->login_allowed = TRUE;
      uc->password_needs_change = FALSE;
    }
  
  /* XXX should check password expirations (some systems already do this in
     ssh_login_permitted). */
  
  return uc;
}

/* Frees information about the user.  If ``undo'' is TRUE, undoes any
   cached state related to e.g. Kerberos and Secure RPC.  Returns
   FALSE if undo was requested, but was unable to undo everything; otherwise
   returns TRUE. */

Boolean ssh_user_free(SshUser uc, Boolean undo)
{
  /* XXX secure rpc state. */

#ifdef KERBEROS
  /* If there is kerberos data in the context, free it now. */
  if (uc->krb_initialized)
    {
      if (uc->krb_user_name)
        free(uc->krb_user_name);
      krb5_free_principal(uc->kcontext, uc->krb_user);
      krb5_free_context(uc->kcontext);
      if (uc->krb_tgt)
        krb5_free_creds(uc->kcontext, uc->krb_tgt);
    }
#endif /* KERBEROS */

  
  ssh_xfree(uc->name);
  ssh_xfree(uc->group);
  ssh_xfree(uc->dir);
  ssh_xfree(uc->shell);
  if (uc->correct_encrypted_passwd)
    ssh_xfree(uc->correct_encrypted_passwd);

  if (uc->groups)
    {
      int i;
      for (i = 0; uc->groups[i]; i++)
        {
          ssh_xfree(uc->groups[i]->name);
          ssh_xfree(uc->groups[i]);
        }
      ssh_xfree(uc->groups);
    }
  
  memset(uc, 'F', sizeof(*uc));
  ssh_xfree(uc);
  return TRUE;
}

/* Returns TRUE if logging in as the specified user is allowed. */

Boolean ssh_user_login_is_allowed(SshUser uc)
{
  return uc->login_allowed;
}

/* Returns TRUE if login is allowed with the given local password. */

Boolean ssh_user_validate_local_password(SshUser uc,
                                         const char *password,
                                         const char *remote_host)
{
  char *encrypted_password;
  const char *correct_passwd = uc->correct_encrypted_passwd;

#ifdef HAVE_ULTRIX_SHADOW_PASSWORDS
  {
    struct svcinfo *svp;
    struct passwd *pw;

    pw = getpwnam(uc->name);
    if (!pw)
      return FALSE;

    svp = getsvc();
    if (svp == NULL)
      {
        SSH_DEBUG(2, ("getsvc() failed in ultrix code in auth_passwd",
                      strerror(errno)));
        return FALSE;
      }
    if ((svp->svcauth.seclevel == SEC_UPGRADE &&
         strcmp(pw->pw_passwd, "*") == 0) ||
        svp->svcauth.seclevel == SEC_ENHANCED)
      return authenticate_user(pw, password, "/dev/ttypXX") >= 0;
  }
#endif /* HAVE_ULTRIX_SHADOW_PASSWORDS */

#ifdef SSHDIST_SESSION_SIA
#ifdef HAVE_SIA
  {
    int argc;
    char **argv;

    /* Passing a collection routine to ssh_sia_validate_user() here would
       be useless and could be harmful.

       It would be useless because at this point, stdin/stdout/stderr
       are all redirected to /dev/null.

       It would be harmful if the collection routine tried to print
       anything to stdout, say, a warning that the password is incorrect.
       At this point, nothing has been printed to stdout yet and it's
       redirected to /dev/null.  Printing to stdout now would initialize
       it as fully buffered (not line buffered) since /dev/null isn't a
       tty.  That would cause problems later on in the child, which gets
       a copy of the stdout structure.  When the child printed /etc/motd
       and the mail notice, the user wouldn't see them because they'd
       be stuck in the stdout buffer. */

    ssh_sia_get_args(&argc, &argv);
    if (ssh_sia_validate_user(NULL, argc, argv,
                              (char *)remote_host, uc->name,
                              NULL, 0, NULL, (char *)password) == SIASUCCESS)
      return TRUE;
    else
      return FALSE;
  }
#endif /* HAVE_SIA */
#endif /* SSHDIST_SESSION_SIA */

  /* Encrypt the candidate password using the proper salt. */
#if defined(HAVE_SCO_ETC_SHADOW)
  encrypted_password = bigcrypt((char *)password, 
                                (correct_passwd[0] && correct_passwd[1]) ?
                                (char *)correct_passwd : "xx");
#else /* HAVE_SCO_ETC_SHADOW */
# if defined(_HPUX_SOURCE)
  if (iscomsec())
    {
      encrypted_password = bigcrypt((char *)password, 
                                    (correct_passwd[0] && correct_passwd[1]) ?
                                    (char *)correct_passwd : "xx");
    }
  else
    {
      encrypted_password = crypt(password, 
                                 (correct_passwd[0] && correct_passwd[1]) ?
                                 correct_passwd : "xx");
    }
# else /* defined(_HPUX_SOURCE) */
  encrypted_password = crypt(password, 
                             (correct_passwd[0] && correct_passwd[1]) ?
                             correct_passwd : "xx");
# endif /* defined(_HPUX_SOURCE) */
#endif /* defined(HAVE_SCO_ETC_SHADOW) */
#ifdef _AIX41
  /* AIX authenticate loop, KG,04.00 */
  {
    int  result;
    int  reenter;
    char *message;

    /* keep re-calling authenticate while reenter is <> 0 */
    do {
      result = authenticate(uc->name, password, &reenter, &message);
    } while (reenter);

    /* if above loop exits, then we are able to return the result */
    return result == 0;
  }
#endif /* _AIX41 */






  
  if (strlen(correct_passwd) < 13)
    return FALSE;

  /* Authentication is accepted if the encrypted passwords are identical. */
  return strncmp(encrypted_password, correct_passwd,
                 strlen(correct_passwd)) == 0;
}

/* Returns TRUE if the user's password needs to be changed. */
Boolean ssh_user_password_must_be_changed(SshUser uc,
                                          char **prompt_return)
{
  if (uc->password_needs_change)
    *prompt_return = ssh_xstrdup("Your password has expired.");
  return uc->password_needs_change;
}

/* Changes the user's password.  Returns TRUE if the change was successful,
   FALSE if the change failed. */

Boolean ssh_user_change_password(SshUser uc,
                                 const char *old_password,
                                 const char *new_password)
{
  SSH_DEBUG(2, ("changing password not yet implemented"));
  return FALSE;
}

#ifdef KERBEROS

/* Verifies that a received ticket-granting ticket is valid.  This function
   draws on the corresponding code in SSH1 by Dug Song (originally based on
   MIT Kerberos appl/bsd/login.c). */

krb5_error_code ssh_user_kerberos_verify_tgt(krb5_context kcontext,
                                             krb5_ccache ccache)
{
  krb5_error_code kerr;
  krb5_principal server;
  krb5_auth_context kauth;
  krb5_keyblock *kb;
  krb5_data data;
  krb5_ticket *ticket;
  Boolean have_keys;

  /* Clear data that might otherwise get freed later. */
  server = NULL;
  kauth = NULL;
  kb = NULL;
  ticket = NULL;
  data.data = NULL;

  /* Get the server principal for the local host (host/<canonical-host>). */
  kerr = krb5_sname_to_principal(kcontext, 0, 0, KRB5_NT_SRV_HST, &server);
  if (kerr)
    goto done;

  /* Try to read the host/<hostname> key, if we have any. */
  kerr = krb5_kt_read_service_key(kcontext, NULL, server, 0,
                                  ENCTYPE_DES_CBC_CRC, &kb);
  if (kerr)
    have_keys = FALSE;
  else
    {
      have_keys = TRUE;
      krb5_free_keyblock(kcontext, kb);
    }

  /* Request to the KDC to construct a ticket. */
  kerr = krb5_mk_req(kcontext, &kauth, 0, "host",
                     krb5_princ_component(kcontext, server, 1)->data,
                     0, ccache, &data);
  if (kerr)
    {
      /* The KDC reported an error.  Reject, unless it is an unknown
         pricincipal for which we have no keys. */
      if (kerr == KRB5KDC_ERR_S_PRINCIPAL_UNKNOWN && !have_keys)
        {
          /* If we have keys, then the pricipal should also have been
             found in the KDC database.  Thus, if the KDC reports the
             pricipal as unknown, then this is most likely an attack.
             However, if we don't have keys for the pricipal, then
             there is no security anyhow, and the ticket does not need
             to be rejected (but we don't really know whether it is
             valid). */
          kerr = 0;
          goto done;
        }
      goto done;
    }
  
  /* Free the authentication context that we created above. */
  krb5_auth_con_free(kcontext, kauth);
  kauth = NULL;

  /* Try to use the ticket we just obtained.  If this results in an error,
     other than not found in keytab, then fail. */
  kerr = krb5_rd_req(kcontext, &kauth, &data, server, NULL, NULL, &ticket);
  if (kerr &&
      (have_keys || (kerr != KRB5_KT_NOTFOUND && kerr != ENOENT)))
    goto done;

  /* The ticket is valid.  Accept. */
  kerr = 0;
  /* Fall to next case... */
  
 done:
  /* We have made the decision (in `ret').  Release resources. */
  if (kauth)
    krb5_auth_con_free(kcontext, kauth);
  if (ticket)
    krb5_free_ticket(kcontext, ticket);
  if (data.data)
    free(data.data);
  krb5_free_principal(kcontext, server);

  /* Return the determined value. */
  return kerr;
}

#endif /* KERBEROS */

/* Returns the kerberos name for the user.  If kerberos is not used,
   then returns the user's normal login name. */

const char *ssh_user_kerberos_name(SshUser uc)
{
#ifdef KERBEROS
  return uc->krb_user_name;
#else /* KERBEROS */
  return ssh_user_name(uc);
#endif /* KERBEROS */
}

/* Tries to log in with the given kerberos password.  If successful,
   obtains a kerberos ticket for the user, and the ticket will be used
   for further access by the current process.  Returns TRUE on success. */

Boolean ssh_user_validate_kerberos_password(SshUser uc,
                                            const char *password)
{
#ifdef KERBEROS
  krb5_error_code kerr;
  krb5_principal server;
  krb5_creds creds;
  krb5_creds tgt_creds;
  krb5_timestamp current_time;
  krb5_ccache ccache;
  krb5_data *datap;
  char ccname[80];

  krb5_preauthtype preauth[3] = {
#ifdef KRB5_PADATA_ENC_UNIX_TIME
    KRB5_PADATA_ENC_UNIX_TIME,
#endif /* KRB5_PADATA_ENC_UNIX_TIME */
    KRB5_PADATA_ENC_TIMESTAMP,
    0 };
  
  SSH_DEBUG(3, ("trying kerberos password"));

  /* Sanity check: bail out on empty password to prevent the krb library from 
     asking it on the server side. */
  if (strlen(password) == 0)
    {
      SSH_TRACE(3, ("password is empty, failing"));
      return FALSE;
    }     
  
  /* Sanity check: the uc should contain kerberos data if we get here. */
  if (!uc->krb_initialized)
    {
      ssh_warning("ssh_user_validate_kerberos_password: uc not krb");
      return FALSE;
    }
  
  /* Initialize variables that are used in "fail". */
  server = NULL;
  ccache = NULL;
  memset(&creds, 0, sizeof(creds));
  
  /* Resolve the credentials cache path.  Make it unique to this process. */
  ssh_snprintf(ccname, sizeof(ccname), "FILE:/tmp/krb5cc_l%ld",
               (long)getpid());
  kerr = krb5_cc_resolve(uc->kcontext, ccname, &ccache);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_cc_resolve %d", kerr));
      goto fail;
    }
  
  /* Initialize the credentials cache for this process. */
  kerr = krb5_cc_initialize(uc->kcontext, ccache, uc->krb_user);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_cc_initialize %d", kerr));
      goto fail;
    }

  /* Build a principal name for the local ticket granting service. */
  datap = krb5_princ_realm(uc->kcontext, uc->krb_user);
  kerr = krb5_build_principal_ext(uc->kcontext, &server,
                                  datap->length, datap->data,
                                  KRB5_TGS_NAME_SIZE, KRB5_TGS_NAME,
                                  datap->length, datap->data, 0);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_cc_initialize %d", kerr));
      goto fail;
    }

  /* Initialize times for the credentials. */
  kerr = krb5_timeofday(uc->kcontext, &current_time);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_timeofday %d", kerr));
      goto fail;
    }
  creds.times.starttime = 0;
  creds.times.endtime = current_time + 10 * 3600; /* 10 hours */
  creds.times.renew_till = current_time + 7 * 24 * 3600; /* 7 days */

  /* Initialize principals for the credentials. */
  creds.client = uc->krb_user;
  creds.server = server;

  /* Try to authenticate user using password.  This essentially tries to get
     a ticket granting ticket for the user. */
  kerr = krb5_get_in_tkt_with_password(uc->kcontext,
                                       KDC_OPT_RENEWABLE | KDC_OPT_FORWARDABLE,
                                       0, NULL, preauth, password, ccache,
                                       &creds, 0);
  if (kerr)
    {
      SSH_DEBUG(2, ("krb5_get_in_tkt_with_password preauth %d", kerr));

      /* Failed to authenticate the user with preauthentication.  Try again
         without preauthentication. */
      kerr = krb5_get_in_tkt_with_password(uc->kcontext,
                                           KDC_OPT_RENEWABLE |
                                           KDC_OPT_FORWARDABLE,
                                           0, NULL, 0, password, ccache,
                                           &creds, 0);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_get_in_tkt_with_password no-preauth %d", kerr));
          goto fail;
        }
    }

  /* Verify the ticket we just obtained.  This returns 0 if the ticket was
     successfully verified, and otherwise returns a Kerberos error. */
  kerr = ssh_user_kerberos_verify_tgt(uc->kcontext, ccache);
  if (kerr)
    {
      SSH_DEBUG(2, ("could not validate Kerberos TGT in password auth (%d)",
                    kerr));
      goto fail;
    }

  /* Kerberos password is ok. */
  SSH_DEBUG(2, ("Kerberos password authentication successful!"));

  /* Retrieve the ticket granting ticket from the credentials cache
     into memory. */
  kerr = krb5_cc_retrieve_cred(uc->kcontext, ccache, 0, &creds, &tgt_creds);
  if (kerr)
    ssh_warning("Failed to retrieve TGT after kerberos password auth");
  else
    {
      ssh_user_kerberos_set_creds(uc, (void *)&tgt_creds);
      krb5_free_cred_contents(uc->kcontext, &tgt_creds);
    }

  /* Indicate that the password authentication was successful. */
  return TRUE;

 fail:
  /* Some error prevented authentication.  Free any resources we have
     allocated. */
  if (server)
    krb5_free_principal(uc->kcontext, server);
  if (ccache)
    krb5_cc_destroy(uc->kcontext, ccache);
  
  return FALSE;
#else /* KERBEROS */
  SSH_DEBUG(2, ("kerberos support not compiled in"));
  return FALSE;
#endif /* KERBEROS */
}

/* Tries to login with the given secure rpc password.  If successful,
   obtains a secure rpc key from the key server, and starts using that
   key for further communication.  Returns TRUE on success. */

Boolean ssh_user_validate_secure_rpc_password(SshUser uc,
                                              const char *password)
{
  SSH_DEBUG(2, ("not yet implemented"));
  return FALSE;
}

#ifdef CRAY
/*
 On a Cray, set the account number for the current process to the user's 
 default account.  If this is not done, the process will have an account 
 of zero and accounting (Cray System Accounting and/or SDSC Resource
 Management (realtime)) will not operate correctly.

 This routine also calls setjob to set up an Cray Job (also known 
 as a Session).  This is needed for CRI's Cray System Accounting 
 and SDSC's Resource Management accounting/management system.

 It also calls setlimit, to set up limits and permissions.
 
 Wayne Schroeder
 San Diego Supercomputer Center
 schroeder@sdsc.edu
 
*/

#include <udb.h>
#include <unistd.h>
#include <sys/category.h>
extern char *setlimits();

int ssh_cray_setup(uid_t uid, char *username)
{
  register struct udb *p;
  extern struct udb *getudb();
  int i, j;
  int accts[MAXVIDS];
  int naccts;
  int err, jid;
  char *sr;
  int pid;

  /* Find all of the accounts for a particular user */
  err = setudb();    /* open and rewind the Cray User DataBase */
  if(err != 0)
    {
      SSH_DEBUG(2, ("ssh_cray_setup: UDB open failure"));
      return(-1);
    }
  naccts = 0;
  while ((p = getudb()) != UDB_NULL) 
    {
      if (p->ue_uid == -1) break;
      if(uid == p->ue_uid) 
        {
          for(j = 0; p->ue_acids[j] != -1 && j < MAXVIDS; j++) 
            {
              accts[naccts] = p->ue_acids[j];
              naccts++;
            }
        }
    }
  endudb();        /* close the udb */
  if (naccts == 0 || accts[0] == 0)
    {
      SSH_DEBUG(2, ("ssh_cray_setup: No Cray accounts found"));
      return(-1);
    }
 
  /* Perhaps someday we'll prompt users who have multiple accounts
     to let them pick one (like CRI's login does), but for now just set 
     the account to the first entry. */
  if (acctid(0, accts[0]) < 0) 
    {
      SSH_DEBUG(2, ("ssh_cray_setup: System call acctid failed, accts[0]=%d",
                    accts[0]));
      return(-1);
    } 
 
  /* Now call setjob to create a new job(/session).  This assigns a new Session
     ID and session table entry to the calling process.  This process will be
     the first process in the job/session. */
  jid = setjob(uid, 0);
  if (jid < 0) 
    {
      SSH_DEBUG(2, ("ssh_cray_setup: System call setjob failure"));
      return(-1);
    }

  /* Now set limits, including CPU time for the (interactive) job and process,
     and set up permissions (for chown etc), etc.  This is via an internal CRI
     routine, setlimits, used by CRI's login. */

  pid = getpid();
  sr = setlimits(username, C_PROC, pid, UDBRC_INTER);
  if (sr != NULL) 
    {
      SSH_DEBUG(2, ("%.100s", sr));
      return(-1);
    }
  sr = setlimits(username, C_JOB, jid, UDBRC_INTER);
  if (sr != NULL) 
    {
      SSH_DEBUG(2, ("%.100s", sr));
      return(-1);
    }

  return(0);
}
#endif /* CRAY */

void ssh_user_close_fds(void)
{
  int max_fd = 0, i = 0;
#ifdef HAVE_GETDTABLESIZE
  max_fd =  getdtablesize();
#endif /* HAVE_GETDTABLESIZE */
#ifdef FD_SETSIZE
  max_fd = max_fd < FD_SETSIZE ? FD_SETSIZE : max_fd;
#endif /* FD_SETSIZE */
  if (max_fd <= 0)
    max_fd = 1000;

  SSH_DEBUG(3, ("max_fd: %d", max_fd));
  
  for (i = 3; i < max_fd; i++)
    close(i);
}

/* Switches the current process to the permissions and privileges of the
   specified user.  The process should not hold any confidential information
   at this point.  This returns FALSE if switching to the given user failed
   for some reason.  The return value of this function MUST BE CHECKED! */

Boolean ssh_user_become_generic(SshUser uc,
                                const char *chroot_dir,
                                SshUserFDCloseCB close_fds,
                                void *context)
{
  /* At this point, this process should no longer be holding any confidential
     information, as changing uid below will permit the user to attach with
     a debugger on some machines. */
  
#ifdef HAVE_SETLOGIN
  /* Set login name in the kernel.  Warning: setsid() must be called before
     this. */
  if (setlogin((char *)ssh_user_name(uc)) < 0)
    SSH_DEBUG(2, ("setlogin failed: %.100s", strerror(errno)));
#endif /* HAVE_SETLOGIN */
  
  /* Close any extra file descriptors.  Note that there may still be
     descriptors left by system functions.  They will be closed later. */
  endpwent();

#if 0
  endhostent();
#endif
  
  /* Close any extra open file descriptors so that we don't have them
     hanging around in clients.  Note that we want to do this after
     initgroups, because at least on Solaris 2.3 it leaves file descriptors
     open. */
  endgrent();

#ifdef HAVE_SIA
#ifdef SSHDIST_SESSION_SIA
  /* Close later on, after calling sia_become_user().  Closing now could
     break that call. */
#endif /* SSHDIST_SESSION_SIA */
#else /* HAVE_SIA */
  if (close_fds == NULL_FNPTR)
    ssh_user_close_fds();
  else
    (*close_fds)(context);
#endif /* HAVE_SIA */

#ifdef CRAY   /* set up accounting account number, job, limits, permissions  */
  if (ssh_cray_setup(ssh_user_uid(uc), ssh_user_name(uc)) < 0)
    {
      SSH_DEBUG(2, ("Failure in Cray job setup for user %d.",
                    (int)ssh_user_uid(uc)));
      return FALSE;
    }
#endif /* CRAY */

  /* Set uid, gid, and groups. */
  if (getuid() == UID_ROOT || geteuid() == UID_ROOT)
    { 
      if (setgid(ssh_user_gid(uc)) < 0)
        {
          SSH_DEBUG(2, ("setgid: %s", strerror(errno)));
          return FALSE;
        }
#ifdef HAVE_INITGROUPS
      /* Initialize the group list. */
      if (initgroups((char *)ssh_user_name(uc), ssh_user_gid(uc)) < 0)
        {
          SSH_DEBUG(2, ("initgroups: %s", strerror(errno)));
          return FALSE;
        }
#endif /* HAVE_INITGROUPS */
      endgrent();

#ifdef HAVE_LIBBSM
      {
        struct auditinfo au_info;
        memset(&au_info, 0, sizeof(au_info));
        if (getaudit(&au_info) < 0)
          {
            SSH_DEBUG(2, ("getauid %d: %s", (int)ssh_user_uid(uc),
                          strerror(errno)));
          }
        else
          {
            au_info.ai_auid = (au_id_t)ssh_user_uid(uc);
            if (au_user_mask(ssh_user_name(uc), &au_info.ai_mask) < 0)
              {
                SSH_DEBUG(2, ("au_user_mask failed for ``%s'' (auditing not "
                              "configured for system nor user)",
                              ssh_user_name(uc)));
              }
            else if (setaudit(&au_info) < 0)
              {
                SSH_DEBUG(2, ("setauid %d: %s", (int)ssh_user_uid(uc),
                              strerror(errno)));
              }
          }
      }
#endif /* HAVE_LIBBSM */

      /* chrooting at this point. */
      if (chroot_dir)
        {
          if (chroot(chroot_dir) < 0)
            {
              ssh_warning("Chroot to '%s' failed!", chroot_dir);
              return FALSE;
            }
        }
      
#ifdef HAVE_USERSEC_H
      /* On AIX, this "sets process credentials".  I am not sure what this
         includes, but it seems to be important.  This also does setuid
         (but we do it below as well just in case). */
      /* XXX does chroot work after this? If not, this should be done after
         chroot (and before setuid()). */
      if (setpcred(ssh_user_name(uc), NULL))
        SSH_DEBUG(2, ("setpcred %.100s: %.100s", strerror(errno)));
#endif /* HAVE_USERSEC_H */

#ifdef HAVE_SIA
#ifdef SSHDIST_SESSION_SIA
      /* Temporarily switch to the desired uid.  We'll permanently switch
         in ssh_user_become_real(). */
      if (seteuid(ssh_user_uid(uc)) < 0)
        {
          SSH_DEBUG(2, ("seteuid %d: %s", (int)ssh_user_uid(uc),
                        strerror(errno)));
          return FALSE;
        }
      if (geteuid() != ssh_user_uid(uc))
        {
          SSH_DEBUG(2, ("failed to set euid to %d.", (int)ssh_user_uid(uc)));
          return FALSE;
        }
#endif /* SSHDIST_SESSION_SIA */
#else /* HAVE_SIA */
#ifdef HAVE_SETLUID
  /* Set login uid, if we have setluid(). */
      if (setluid(ssh_user_uid(uc)) < 0)
        {
          SSH_DEBUG(2, ("setluid %d: %s", (int)ssh_user_uid(uc),
                        strerror(errno)));
          return FALSE;
        }
#endif /* HAVE_SETLUID */
      /* Permanently switch to the desired uid. */
      if (setuid(ssh_user_uid(uc)) < 0)
        {
          SSH_DEBUG(2, ("setuid %d: %s", (int)ssh_user_uid(uc),
                        strerror(errno)));
          return FALSE;
        }
      if (getuid() != ssh_user_uid(uc) || geteuid() != ssh_user_uid(uc))
        {
          SSH_DEBUG(2, ("failed to set uids to %d.", (int)ssh_user_uid(uc)));
          return FALSE;
        }
#endif /* HAVE_SIA */
    }
  
#ifdef KERBEROS
  /* If we have a ticket granting ticket, store that in the user's
     credentials. */
  if (uc->krb_initialized && uc->krb_tgt != NULL)
    {
      krb5_ccache ccache;
      krb5_error_code kerr;
      char ccname[256];
      char ccenvname[256], *env_ccache;

      ssh_snprintf(ccname, sizeof(ccname), "FILE:/tmp/krb5cc_p%s%d",
                   uc->krb_local_username, getpid());

      /* Construct the environment variable name to be used below */
      ssh_snprintf(ccenvname, sizeof(ccenvname),"KRB5CCNAME=%s",ccname);

      kerr = krb5_cc_resolve(uc->kcontext, ccname, &ccache);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_cc_default %d", kerr));
          goto krb_fail;
        }

      /* Initialize (and clear) the cache. */
      kerr = krb5_cc_initialize(uc->kcontext, ccache, uc->krb_user);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_cc_initialize %d", kerr));
          goto krb_fail;
        }

      /* Store the ticket granting ticket in the credentials. */
      kerr = krb5_cc_store_cred(uc->kcontext, ccache, uc->krb_tgt);
      if (kerr)
        {
          SSH_DEBUG(2, ("krb5_cc_store_cred %d", kerr));
          goto krb_fail;
        }

      /* Ok, we have successfully stored the TGT in the ccache. */
      SSH_DEBUG(2, ("TGT successfully stored in ccache"));
      
      /* Put the Cache name into the environment */
      SSH_DEBUG(2, ("Placing ticket cache %s into environment", ccenvname));
      env_ccache = ssh_xstrdup(ccenvname);
      putenv(env_ccache);

#if 0
      /* This is the spot to setup for DCE/DFS or AFS if possible */
      /* Map the kerberos principal to a local user name. */
      kerr = krb5_dfs_pag(uc->kcontext, 1,  uc->krb_user, uc->krb_local_username);
      SSH_DEBUG(2, ("krb_5dfs_pag returned %x", kerr));
#endif

    krb_fail:
      /* Free the ccache object. */
      krb5_cc_close(uc->kcontext, ccache);
    }
#endif /* KERBEROS */
      
  /* We are now running with the user's privileges. */
  return TRUE;
}

Boolean ssh_user_become(SshUser uc)
{
  return ssh_user_become_generic(uc, NULL, NULL_FNPTR, NULL);
}

/* Sets Kerberos credentials to be saved for the user.  The argument should
   be of type "krb5_creds *"; this will copy the creds into internal
   data structures. */

#ifdef KERBEROS
void ssh_user_kerberos_set_creds(SshUser uc, void *creds)
{
  krb5_error_code kerr;
  
  if (uc->krb_tgt)
    krb5_free_creds(uc->kcontext, uc->krb_tgt);
  uc->krb_tgt = NULL;
  kerr = krb5_copy_creds(uc->kcontext, (krb5_creds *)creds, &uc->krb_tgt);
  if (kerr)
    {
      SSH_DEBUG(2, ("Failed to save TGT."));
      uc->krb_tgt = NULL;
    }  
}
#endif /* KERBEROS*/

#ifdef SSHDIST_SESSION_SIA
#ifdef HAVE_SIA
/* Last chance to finish anything that ssh_user_become() left undone.  The
   difference between the two functions is that ssh_user_become() is called
   before the user's environment is set, while we're called after.

   Switches the current process to the permissions and privileges of the
   specified user.  The process should not hold any confidential information
   at this point.  This returns FALSE if switching to the given user failed
   for some reason.  The return value of this function MUST BE CHECKED! */
Boolean ssh_user_become_real(SshUser uc,
                             const char *remote_host,
                             const char *ttyname,
                             SshUserFDCloseCB close_fds,
                             void *context)
{
  int i;
  int argc;
  char **argv;

  SSH_DEBUG(7, ("before going-back-to-uid uid = %d and euid = %d", 
                (int)getuid(), (int)geteuid()));
  /* Switch back temporarily to our own uid (typically root, but
     not always) in case sia_become_user() needs to access protected
     account databases. */
  if (seteuid(getuid()) != 0)
    {
      SSH_DEBUG(2, ("seteuid %d: %s", (int)getuid(), strerror(errno)));
      return FALSE;
    }

  SSH_DEBUG(7, ("after seteuid-to-uid uid = %d and euid = %d", 
                (int)getuid(), (int)geteuid()));

  ssh_sia_get_args(&argc, &argv);
  /* Enhanced Security calls setluid() here in sia_become_user(). */
  if (sia_become_user(sia_collect_trm, argc, argv, (char *)remote_host,
                      (char *)ssh_user_name(uc), (char *)ttyname, 0, NULL,
                      NULL, SIA_BEU_REALLOGIN) != SIASUCCESS)
    {
      SSH_DEBUG(2, ("sia_become_user() failed for user %d.",
                    (int)ssh_user_uid(uc)));
      return FALSE;
    }

  /* Our sia calls are complete, so we can finally close any extra
     open file descriptors. */
  if (close_fds == NULL_FNPTR)
    ssh_user_close_fds();
  else
    (*close_fds)(context);

  /* We can now completely become the user.  We need to do this
     with setreuid() instead of setuid() because sia_become_user()
     has set our euid to the user's uid. */
  if (setreuid(ssh_user_uid(uc), ssh_user_uid(uc)) < 0)
    {
      SSH_DEBUG(2, ("setreuid %d: %s", (int)ssh_user_uid(uc),
                    strerror(errno)));
      return FALSE;
    }

  if (getuid() != ssh_user_uid(uc) || geteuid() != ssh_user_uid(uc))
    {
      SSH_DEBUG(2, ("failed to set uids to %d.", (int)ssh_user_uid(uc)));
      return FALSE;
    }

  SSH_DEBUG(7, ("after completion uid = %d and euid = %d", 
                (int)getuid(), (int)geteuid()));

  /* We are now running with the user's complete privileges. */
  return TRUE;
}
#endif /* HAVE_SIA */
#endif /* SSHDIST_SESSION_SIA */

/* Returns the login name of the user. */

const char *ssh_user_name(SshUser uc)
{
  return uc->name;
}

/* Returns the group name of the user. */
const char *ssh_group_name(SshUser uc)
{
  if (uc->group == NULL)
    {
      struct group *grp = getgrgid(ssh_user_gid(uc));
      
      if (grp != NULL)
        uc->group = ssh_xstrdup(grp->gr_name);
      else
        ssh_xdsprintf(&uc->group, "%lu", (unsigned long)(ssh_user_gid(uc)));
    }
  return uc->group;
}

/* Returns the uid of the user.  This is unix-specific. */

uid_t ssh_user_uid(SshUser uc)
{
  return uc->uid;
}

/* Returns the gid of the user.  This is unix-specific. */

gid_t ssh_user_gid(SshUser uc)
{
  return uc->gid;
}

/* Returns the user's home directory.  This is unix-specific. */

const char *ssh_user_dir(SshUser uc)
{
  return uc->dir;
}

/* Returns the user's shell.  This is unix-specific. */

const char *ssh_user_shell(SshUser uc)
{
  return uc->shell;
}

gid_t ssh_group_get_gid(SshGroup group)
{
  SSH_PRECOND(group != NULL);
  return group->gid;
}

const char *ssh_group_get_name(SshGroup group)
{
  SSH_PRECOND(group != NULL);
  return group->name;
}

/* Return the groups the user is in. Array is terminated with a NULL
   pointer.*/
SshGroup *ssh_user_get_groups(SshUser uc)
{
  SSH_PRECOND(uc != NULL);
  
  if (!uc->groups)
    {
#ifdef HAVE_GETGRENT
      int allocated = 10, count = 0, i = 0;
      struct group *grp;
      SshGroup *groups;
      const char *user = ssh_user_name(uc);
      char *hlp = NULL;
      
      groups = ssh_xcalloc(allocated, sizeof(SshGroup));
      groups[0] = ssh_xcalloc(1, sizeof(struct SshGroupRec));
      groups[0]->gid = ssh_user_gid(uc);
      groups[0]->name = ssh_xstrdup(ssh_group_name(uc));
      
      count++;
      
      setgrent();
      
      while((grp = getgrent()) != NULL)
        {
          if (count + 1 >= allocated)
            {
              allocated *= 2;
              groups = ssh_xrealloc(groups,
                                    allocated*sizeof(SshGroup));
            }
          
          /* Check whether 'user' is a member of group. */
          for(i = 0, hlp = grp->gr_mem[0]; hlp && strcmp(hlp, user);
              i++, hlp = grp->gr_mem[i])
            ;

          if(hlp)
            /* Don't add primary group twice. */
            if (groups[0]->gid != grp->gr_gid)
              {
                groups[count] =
                  (SshGroup) ssh_xcalloc(1, sizeof(struct SshGroupRec));
                groups[count]->gid = grp->gr_gid;
                if (grp->gr_name)
                  groups[count]->name = ssh_xstrdup(grp->gr_name);
                else
                  ssh_xdsprintf(&groups[count]->name, "%lu",
                               (unsigned long) grp->gr_gid);
                
                count++;
              }
        }

      endgrent();
      groups[count] = NULL;
      uc->groups = groups;
#else /* HAVE_GETGRENT */
      /* If we don't have getgrent(), we have to settle for the
         primary group. */
      uc->groups = ssh_xcalloc(2, sizeof(SshGroup));
      uc->groups[0] = ssh_xcalloc(1, sizeof(struct SshGroupRec));
      uc->groups[0]->gid = ssh_user_gid(uc);
      uc->groups[0]->name = ssh_xstrdup(ssh_group_name(uc));
#endif /* HAVE_GETGRENT */
    }
  return uc->groups;
}
