/*
 * Authors: Tero Kivinen  <kivinen@ssh.com>
 *          Tomi Salo     <ttsalo@ssh.com>
 *          Sami Lehtinen <sjl@ssh.com>
 * 
 * Copyright (c) 1996-2001 SSH Communications Security Corp, Helsinki, Finland
 * All rights reserved.
 */
/*
 *        Program: sshreadline
 *        $Source: /ssh/CVS/src/lib/sshreadline/sshreadline.h,v $
 *        $Author: sjl $
 *
 *        Creation          : 19:52 Mar 12 1997 kivinen
 *        Last Modification : 01:01 Mar 17 1997 kivinen
 *        Last check in     : $Date: 2002/04/29 10:36:11 $
 *        Revision number   : $Revision: 1.12.20.1 $
 *        State             : $State: Exp $
 *        Version           : 1.12
 *        
 *
 *        Description       : Readline library
 *
 *
 *        $Log: sshreadline.h,v $ *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        $EndLog$
 */

#ifndef SSHREADLINE_H
#define SSHREADLINE_H

#include "sshstream.h"

typedef struct ReadLineRec *SshReadLineCtx;

/*
 * Read line from user. The tty at file descriptor ``write_fd'' is put
 * to raw mode and data is read from ``read_fd'' until CR is
 * received. The ``prompt'' is used to prompt the input. ``def'' is
 * the initial data that is editable on the readline.
 *
 * When the line has been read, the function will call provided
 * ``callback'' once providing it the data read. The data is available
 * only during the callback execution.  ``context'' is given to the
 * ``callback'' as argument. ``line'' will be NULL in case of error.
 *
 */
typedef void (*SshRLCallback)(const char *line, void *context);
void ssh_readline_eloop(const char *prompt, const char *def,
                        SshReadLineCtx rl, SshRLCallback callback,
                        void *context);

typedef void (*SshRLExtCallback)(const char *line, unsigned int flags,
                                 void *context);

#define SSH_READLINE_CB_FLAG_TAB 0x1 /* The callback has been called
                                        because the user pressed
                                        tab. */

/* Version with extended callback type. ``flags'' should be set to indicate
   which flags are supported. */
void ssh_readline_eloop_ext(const char *prompt, const char *def,
                            SshReadLineCtx rl, SshRLExtCallback callback,
                            unsigned int flags, void *context);

/* Version that doesn't echo the passed input to the screen. */
void ssh_readpass_eloop(const char *prompt, const char *def,
                        SshReadLineCtx rl, SshRLCallback callback,
                        void *context);

/* This must be called _once_ before using any of the asynchronous
   ssh_read[line,pass}_eloop*() functions. Returns a context argument
   to be given to these functions. Returns NULL in case of
   initialization error. */
SshReadLineCtx ssh_readline_eloop_initialize(SshStream stream);

/* This should be used after you are finished with the "real"
   functions. This will free ``rl''. */
void ssh_readline_eloop_uninitialize(SshReadLineCtx rl);

/*
 * Read line from user. The tty at file descriptor ``write_fd'' is put
 * to raw mode and data is read from ``read_fd'' until CR is
 * received. The ``prompt'' is used to prompt the input. ``line'' is
 * pointer to char pointer and it should either contain NULL or the
 * mallocated string for previous value (that string is freed).  If
 * line can be successfully read the ``line'' argument contains the
 * new mallocated string.
 *
 * The ssh_readline will return the number of characters returned in
 * line buffer. If EOF or other error is noticed the return value is
 * -1.
 */
int ssh_readline(const char *prompt, char **line, int read_fd, int write_fd);


#endif /* SSHREADLINE_H */
