/*
 * Authors: Tero Kivinen  <kivinen@ssh.com>
 *          Tomi Salo     <ttsalo@ssh.com>
 *          Sami Lehtinen <sjl@ssh.com>
 *
 * Copyright (c) 1996-2001 SSH Communications Security Corp, Helsinki, Finland
 * All rights reserved.
 */
/*
 *        Program: sshreadline
 *        $Source: /ssh/CVS/src/lib/sshreadline/sshreadline.c,v $
 *        $Author: sjl $
 *
 *        Creation          : 19:47 Mar 12 1997 kivinen
 *        Last Modification : 12:37 Sep 18 2001 kivinen
 *        Last check in     : $Date: 2002/04/29 10:33:36 $
 *        Revision number   : $Revision: 1.52.4.2 $
 *        State             : $State: Exp $
 *        Version           : 1.669
 *        
 *
 *        Description       : Readline library
 *
 *
 *        $Log: sshreadline.c,v $ *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        
 *        $EndLog$
 */

#ifndef VXWORKS
/* some curses, such as Tru64's, use 'bool' */
#define SSH_ALLOW_CPLUSPLUS_KEYWORDS

#include "sshincludes.h"
#include "sshreadline.h"
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif /* HAVE_TERMIOS_H */
#ifdef HAVE_TERMIO_H
#include <termio.h>
#endif /* HAVE_TERMIO_H */
#ifdef HAVE_CURSES_H
#include <curses.h>
#endif /* HAVE_CURSES_H */
#ifdef HAVE_TERMINFO
# ifdef HAVE_USR_XPG4_INCLUDE_TERM_H
#  include </usr/xpg4/include/term.h>
# else /* HAVE_USR_XPG4_INCLUDE_TERM_H */
#  ifdef HAVE_TERM_H
#   include <term.h>
#  endif
# endif /* HAVE_USR_XPG4_INCLUDE_TERM_H */
#else /* HAVE_TERMINFO */
# ifdef HAVE_TERMCAP_H
#  include <termcap.h>
# endif /* HAVE_TERMCAP_H */
#endif /* HAVE_TERMINFO */
#include <sys/ioctl.h>
#include "sshfdstream.h"
#include "sshbuffer.h"
#include "sshadt_list.h"
#include "sshadt_conv.h"
#include "sshtimeouts.h"

#define SSH_DEBUG_MODULE "SshReadLine"

#define SSH_READLINE_MAX_UNDO_DEPTH 256

typedef enum {
  SSH_RL_WORD, SSH_RL_SENTENCE
} ReadLineUnit;

typedef enum {
  SSH_RL_BACKWARD, SSH_RL_FORWARD
} ReadLineDirection;

typedef struct TermRec {
  char *term_type;                     /* Terminal type */
  char *term_buffer;                   /* Termcap buffer */

                                       /* Terminfo | Termcap */
  int term_auto_margin;                /* am       | am      */
  char *term_cursor_key_app_mode_on;   /* smkx     | ks      */
  char *term_cursor_key_app_mode_off;  /* rmkx     | ke      */
  char *term_clear_to_end_of_line;     /* el       | ce      */
  char *term_set_cursor_column;        /* hpa(n)   | ch(n)   */
  char *term_delete_n_chars;           /* dch(n)   | DC(n)   */
  char *term_delete_char;              /* dch1     | dc      */
  char *term_insert_n_chars;           /* ich(n)   | IC(n)   */
  char *term_insert_char;              /* ich1     | ic      */
  char *term_move_cursor_down_n;       /* cud(n)   | DO(n)   */
  char *term_move_cursor_down;         /* cud1     | do      */
  char *term_move_cursor_left_n;       /* cub(n)   | LE(n)   */
  char *term_move_cursor_left;         /* cub1     | le      */
  char *term_move_cursor_right_n;      /* cuf(n)   | RI(n)   */
  char *term_move_cursor_right;        /* cuf1     | nd      */
  char *term_move_cursor_up_n;         /* cuu(n)   | UP(n)   */
  char *term_move_cursor_up;           /* cuu1     | up      */
  char *term_visual_bell;              /* flash    | vb      */
  char *key_down_arrow;                /* kcud1    | kd      */
  char *key_left_arrow;                /* kcub1    | kl      */
  char *key_right_arrow;               /* kcuf1    | kr      */
  char *key_up_arrow;                  /* kcuu1    | ku      */
} Term;

Term *ssh_last_term = NULL;

typedef enum {
  SSH_READLINE_NORMAL_KEYMAP,
  SSH_READLINE_CTRL_X_KEYMAP,
  SSH_READLINE_ESC_KEYMAP
} ReadLineKeyMap;

typedef struct ReadLineRec {
  int read_fd;
  int write_fd;
  const char *prompt;
  int prompt_len;
  char *line;
  int line_alloc;
  char *display_line;
  struct termios read_term;
  Boolean got_tty_modes;
  int write_fcntl_flags;
  char *yank;
  char **undo;
  int *undo_cursors;
  int max_undo_depth;
  int undo_length;
  int undo_position;
  int undo_direction;
  int cursor;
  int display_cursor;
  int row_length;
  int end;
  int mark;
  int last_command_cut;
  ReadLineKeyMap keymap;
  Term *tc;
  Boolean eloop;
  Boolean echo;
  SshStream stream;
  SshRLCallback callback;
  SshRLExtCallback ext_callback;
  Boolean extended_mode;
  unsigned int wanted_flags, flags;
  SshBuffer out_buffer;
  SshBuffer in_buffer;
  Boolean timeout_done;

  void *user_context; /* Context for the above callback */

  /* Command history state */
  SshADTContainer cmd_hist;
  /* handle to current history item, or SSH_ADT_INVALID if invalid.*/
  SshADTHandle ch;
  /* Whether the last line in the list was added after user pressed
     enter or because user is just browsing the command history. */
  Boolean line_entered_in_cmd_hist;
} *ReadLine, ReadLineStruct;

int ssh_rl_set_tty_modes_for_fd(int fd, ReadLine rl, int *fcntl_flags,
                                struct termios *fd_tio);
int ssh_rl_set_tty_modes(ReadLine rl);
int ssh_rl_restore_tty_modes(ReadLine rl);
void iocb(SshStreamNotification notify, void *context);

static RETSIGTYPE (*old_sigint_handler)(int) = NULL_FNPTR;
static RETSIGTYPE (*old_sigcont_handler)(int) = NULL_FNPTR;
ReadLine global_rl;

void ssh_readline_eloop_internal(const char *prompt, const char *def,
                                 Boolean echo, SshReadLineCtx rl,
                                 SshRLCallback callback,
                                 SshRLExtCallback ext_callback,
                                 unsigned int flags, void *context);

RETSIGTYPE ssh_rl_sigint_handler(int sig);
RETSIGTYPE ssh_rl_sigcont_handler(int sig);

void rl_reregister_signal_timeout(void *ctx)
{
  unsigned long signum = (unsigned long)ctx;
  if (signum == SIGCONT)
    old_sigcont_handler = signal(SIGCONT, ssh_rl_sigcont_handler);
  else if (signum == SIGINT)
    old_sigint_handler = signal(SIGINT, ssh_rl_sigint_handler);
    
}

RETSIGTYPE ssh_rl_sigint_handler(int sig)
{
  /* Restore terminal modes. */
  ssh_rl_restore_tty_modes(global_rl);

  (void)signal(SIGINT, old_sigint_handler);

  /* Resend the signal, with the old handler. */
  kill(getpid(), sig);

  /* The eventloop calls the signal handlers in the event loop, which
     would lead to a loop here, unless we used the following timeout to
     re-register the signal. (the timeouts are called after the signal
     handlers in the event loop) */
  ssh_register_timeout(0L, 0L, rl_reregister_signal_timeout,
                        (void*)SIGCONT);
}

RETSIGTYPE ssh_rl_sigcont_handler(int sig)
{
  int dummy;
  
  if (ssh_rl_set_tty_modes_for_fd(global_rl->read_fd, global_rl,
                                  &dummy,
                                  &global_rl->read_term) == -1)
    SSH_TRACE(1, ("Failed to set modes for read fd."));

  if (ssh_rl_set_tty_modes_for_fd(global_rl->write_fd, global_rl,
                                  &dummy,
                                  &global_rl->read_term) == -1)
    SSH_TRACE(1, ("Failed to set modes for write fd."));

  (void)signal(SIGCONT, old_sigcont_handler);

  /* Resend the signal, with the old handler. */
   kill(getpid(), sig); 

   ssh_register_timeout(0L, 0L, rl_reregister_signal_timeout,
                        (void*)SIGCONT);
}

int ssh_rl_set_tty_modes_for_fd(int fd, ReadLine rl, int *fcntl_flags,
                                struct termios *fd_tio)
{
  struct termios new_term;
  struct winsize win;
  int new_fcntl_flags;

  /* Set tty modes */
  if (tcgetattr(fd, fd_tio) < 0)
    {
      ssh_warning("tcgetattr failed in ssh_rl_set_tty_modes_for_fd: "
                  "fd %d: %.200s", fd, strerror(errno));
      return -1;
    }
  new_term = *fd_tio;
  new_term.c_iflag &= ~(ISTRIP);
  new_term.c_oflag &= ~(ONLCR);
  new_term.c_lflag &= ~(ECHO | ICANON);
  new_term.c_cc[VMIN] = 1;
  new_term.c_cc[VTIME] = 1;
  if (tcsetattr(fd, TCSAFLUSH, &new_term) < 0)
    {
      ssh_warning("tcsetattr failed in ssh_rl_set_tty_modes_for_fd: "
                  "fd %d: %.200s", fd, strerror(errno));
      return -1;
    }

  new_fcntl_flags = fcntl(fd, F_GETFL, 0);
  *fcntl_flags = new_fcntl_flags;

  if (!rl->eloop)
    {
#ifdef O_ASYNC
      fcntl(fd, F_SETFL, new_fcntl_flags & ~(O_ASYNC | O_NONBLOCK));
#else /* O_ASYNC */
      fcntl(fd, F_SETFL, new_fcntl_flags & ~(O_NONBLOCK));
#endif /* O_ASYNC */
    }
  else
    {
#if defined(O_NONBLOCK) && !defined(O_NONBLOCK_BROKEN)
      new_fcntl_flags |= O_NONBLOCK;
#else /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
      new_fcntl_flags |= O_NDELAY;
#endif /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
      (void)fcntl(fd, F_SETFL, new_fcntl_flags);
    }

  if (ioctl(fd, TIOCGWINSZ, &win) != -1)
    {
      rl->row_length = win.ws_col;
    }
  if (rl->row_length == 0)
    {
      rl->row_length = 80;
    }

  return 0;
}

/*
 * Set tty to raw mode. Return old flags in store_old_term.
 * Return 0 if success and -1 if failure.
 */
int ssh_rl_set_tty_modes(ReadLine rl)
{
  if (ssh_rl_set_tty_modes_for_fd(rl->write_fd, rl,
                                  &rl->write_fcntl_flags,
                                  &rl->read_term) == -1)
    return -1;

  rl->got_tty_modes = TRUE;

  return 0;
}

int ssh_rl_restore_tty_modes_for_fd(int fd, ReadLine rl, int fcntl_flags,
                                    struct termios *saved_tio)
{
  fcntl(fd, F_SETFL, fcntl_flags);

  if (tcsetattr(fd, TCSAFLUSH, saved_tio) < 0)
    {
      ssh_warning("tcsetattr failed in ssh_rl_restore_tty_modes_for_fd: "
                  "fd %d: %.200s", fd, strerror(errno));
      return -1;
    }
  return 0;
}

/*
 * Restore terminal modes
 * Return 0 if success and -1 if failure.
 */
int ssh_rl_restore_tty_modes(ReadLine rl)
{
  if (!rl->got_tty_modes)
    return 0;

  return ssh_rl_restore_tty_modes_for_fd(rl->write_fd, rl,
                                         rl->write_fcntl_flags,
                                         &rl->read_term);
}

/*
 * Store undo information
 */
void ssh_rl_store_undo(ReadLine rl)
{
  if (rl->undo_length == rl->max_undo_depth)
    {
      ssh_xfree(rl->undo[0]);
      memmove(rl->undo, rl->undo + 1, sizeof(const char *) *
              rl->max_undo_depth - 1);
      memmove(rl->undo_cursors, rl->undo_cursors + 1, sizeof(int) *
              rl->max_undo_depth - 1);
      rl->undo_length--;
    }
  rl->line[rl->end] = '\0';
  rl->undo[rl->undo_length] = ssh_xstrdup(rl->line);
  rl->undo_cursors[rl->undo_length] = rl->cursor;
  rl->undo_length++;
  rl->undo_position = -1;
  rl->undo_direction = -1;
}

/*
 * Delete the undo information
 */
void ssh_rl_delete_undo(ReadLine rl)
{
  int i;
  for(i = 0; i < rl->undo_length; i++)
    ssh_xfree(rl->undo[i]);
  rl->undo_length = 0;
  rl->undo_position = -1;
  rl->undo_direction = -1;
}

/*
 * Enlarge line buffer so it can hold length characters
 */
void ssh_rl_enlarge_to(ReadLine rl, int length)
{
  if (rl->line_alloc > length)
    return;

  rl->line_alloc = length + 10;
  rl->line = ssh_xrealloc(rl->line, rl->line_alloc);
  rl->display_line = ssh_xrealloc(rl->display_line, rl->line_alloc);
}

/*
 * Send string to terminal
 */
void ssh_rl_send_string(ReadLine rl, const char *txt,
                        int len)
{
  if (rl->eloop)
    {
      int ret;
      ssh_xbuffer_append(rl->out_buffer, txt, len);
      if ((ret = ssh_stream_write(rl->stream,
                                  ssh_buffer_ptr(rl->out_buffer),
                                  ssh_buffer_len(rl->out_buffer))))
        {
          if (ret == 0)
            ssh_warning("Write failed in ssh_rl_send_string: EOF received");
          if (ret > 0)
            ssh_buffer_consume(rl->out_buffer, ret);
        }
    }
  else
    {
      if (write(rl->write_fd, txt, len) != len)
        {
          ssh_warning("Write failed in ssh_rl_send_string: %.200s",
                      strerror(errno));
        }
    }
}

/*
 * Send terminal code
 */
void ssh_rl_send_code(ReadLine rl, const char *code)
{
  ssh_rl_send_string(rl, code, strlen(code));
}

/*
 * Send terminal code, with numeric argument
 */
void ssh_rl_send_code_n(ReadLine rl, const char *code,
                        int n)
{
#ifdef HAVE_TERMINFO
  char *buf;

  SSH_DEBUG_HEXDUMP(99, ("code before tparm():"), code, strlen(code));
  buf = tparm((char *)code, n, 0, 0, 0, 0, 0, 0, 0, 0);
  SSH_DEBUG_HEXDUMP(99, ("code after tparm():"), buf, strlen(buf));
  ssh_rl_send_string(rl, buf, strlen(buf));
#else /* HAVE_TERMINFO */
  char buffer[256];
  const char *fmt;
  int i;

  /* Skip padding */
  for( ; *code && isdigit(*code); code++)
    ;
  if (*code == '.')             /* one decimal place */
    code++;
  if (*code == '*')             /* Multiple by cnt */
    code++;

  i = 0;
  for(; *code; code++)
    {
      switch (*code)
        {
        case '%':
          code++;
          switch (*code)
            {
            case '%':
              buffer[i++] = '%';
              break;
            case 'd':
              fmt = "%d";
              goto format;
            case '2':
              fmt = "%02d";
              goto format;
            case '3':
              fmt = "%03d";
              goto format;
            case '+':
              code++;
              n += *code;
              /*FALLTHROUGH*/
            case '.':
              fmt = "%c";
            format:
              ssh_snprintf((buffer + i), sizeof(buffer) - i, fmt, n);
              i = strlen(buffer);
              break;
            case '>':
              code++;
              if(n > *code)
                n += *(code + 1);
              code++;
              break;
            case 'r':
              /* Ignored, because we only have one arg */
              break;
            case 'i':
              n++;
              break;
            case 'n':
              n ^= 0140;
              break;
            case 'B':
              n = (16 * (n / 10)) + (n % 10);
              break;
            case 'D':
              n = (n - 2 * (n % 16));
              break;
            }
          break;
        default:
          buffer[i++] = *code;
          break;
        }
      if (i > 200)
        {
          ssh_rl_send_string(rl, buffer, i);
          i = 0;
        }
    }
  SSH_DEBUG_HEXDUMP(99, ("code after parametrization:"), buffer, i);
  ssh_rl_send_string(rl, buffer, i);
#endif /* HAVE_TERMINFO */
}

/*
 * Move cursor to correct place
 */
void ssh_rl_move_cursor(ReadLine rl, int new_pos)
{
  int cursor_row, new_row, cursor_column, new_column;
  int i;

  if (new_pos == rl->display_cursor)
    return;

  new_row = new_pos / rl->row_length;
  cursor_row = rl->display_cursor / rl->row_length;
  new_column = new_pos % rl->row_length;
  cursor_column = rl->display_cursor % rl->row_length;

  /* On different row? */
  if (new_row != cursor_row)
    {
      if (new_row < cursor_row) /* Move up on screen */
        {
          if ((cursor_row - new_row > 1 &&
               rl->tc->term_move_cursor_up_n) ||
              (rl->tc->term_move_cursor_up_n &&
               !rl->tc->term_move_cursor_up))
            {
              ssh_rl_send_code_n(rl, rl->tc->term_move_cursor_up_n,
                                 cursor_row - new_row);
            }
          else
            {
              for(i = 0; i < cursor_row - new_row; i++)
                ssh_rl_send_code(rl, rl->tc->term_move_cursor_up);
            }
        }
      else /* Move down on screen */
        {
          if ((new_row - cursor_row > 1 &&
               rl->tc->term_move_cursor_down_n) ||
              (rl->tc->term_move_cursor_down_n &&
               !rl->tc->term_move_cursor_down))
            {
              ssh_rl_send_code_n(rl, rl->tc->term_move_cursor_down_n,
                                 new_row - cursor_row);
            }
          else
            {
              for(i = 0; i < new_row - cursor_row; i++)
                ssh_rl_send_code(rl, rl->tc->term_move_cursor_down);
            }
        }
    }
  if (new_column != cursor_column)
    {
      if (rl->tc->term_set_cursor_column)
        {
          ssh_rl_send_code_n(rl, rl->tc->term_set_cursor_column,
                             new_column);
        }
      else
        {
          if (new_column < cursor_column) /* Move left */
            {
              if ((cursor_column - new_column > 1 &&
                   rl->tc->term_move_cursor_left_n) ||
                  (rl->tc->term_move_cursor_left_n &&
                   !rl->tc->term_move_cursor_left))
                {
                  ssh_rl_send_code_n(rl, rl->tc->term_move_cursor_left_n,
                                     cursor_column - new_column);
                }
              else
                {
                  for(i = 0; i < cursor_column - new_column; i++)
                    ssh_rl_send_code(rl, rl->tc->term_move_cursor_left);
                }
            }
          else /* Move right */
            {
              if ((new_column - cursor_column > 1 &&
                   rl->tc->term_move_cursor_right_n) ||
                  (rl->tc->term_move_cursor_right_n &&
                   !rl->tc->term_move_cursor_right))
                {
                  ssh_rl_send_code_n(rl, rl->tc->term_move_cursor_right_n,
                                     new_column - cursor_column);
                }
              else
                {
                  for(i = 0; i < new_column - cursor_column; i++)
                    ssh_rl_send_code(rl, rl->tc->term_move_cursor_right);
                }
            }
        }
    }
  rl->display_cursor = new_pos;
}

/*
 * Write stuff to screen, fix the cursor location.
 */
void ssh_rl_write_string(ReadLine rl, const char *txt,
                         int len)
{
  int l;

  /* If we have auto margins or text fits on one line, just send it */
  if (rl->tc->term_auto_margin ||
      (rl->display_cursor % rl->row_length) + len < rl->row_length)
    {
      ssh_rl_send_string(rl, txt, len);
      rl->display_cursor += len;
      if ((rl->display_cursor % rl->row_length) == 0)
        {
          ssh_rl_send_string(rl, " ", 1);
          if (rl->tc->term_move_cursor_left)
            ssh_rl_send_code(rl, rl->tc->term_move_cursor_left);
          else if (rl->tc->term_move_cursor_left_n)
            ssh_rl_send_code_n(rl, rl->tc->term_move_cursor_left_n, 1);
          else
            ssh_rl_send_code_n(rl, rl->tc->term_set_cursor_column, 0);
        }
    }
  else
    {
      /* Send end of line */
      l = rl->row_length - (rl->display_cursor % rl->row_length);
      ssh_rl_send_string(rl, txt, l);
      rl->display_cursor += l - 1;
      txt += l;
      len -= l;
      /* Move to next line */
      ssh_rl_move_cursor(rl, rl->display_cursor + 1);
      for(; len > rl->row_length; len -= rl->row_length)
        {
          ssh_rl_send_string(rl, txt, rl->row_length);
          rl->display_cursor += rl->row_length - 1;
          txt += rl->row_length;
          /* Move to next line */
          ssh_rl_move_cursor(rl, rl->display_cursor + 1);
        }
      ssh_rl_send_string(rl, txt, len);
      rl->display_cursor += len;
    }
}

/*
 * Redraw display
 */
void ssh_rl_redraw_display(ReadLine rl)
{
  int common_start, common_end, line_len, display_len;
  int i;

  if (! rl->echo)
    return;

  line_len = rl->end;
  rl->line[rl->end] = '\0';
  display_len = strlen(rl->display_line);
  /* Skip common part */
  for(common_start = 0; common_start < line_len && common_start < display_len
        && rl->line[common_start] == rl->display_line[common_start];
      common_start++)
    ;
  if (common_start != line_len || common_start != display_len)
    {
      /* Find common end */
      for(common_end = 0; common_end < line_len && common_end < display_len
            && rl->line[line_len - common_end - 1]
            == rl->display_line[display_len - common_end - 1]; common_end++)
        ;
      if (common_start == line_len || common_start == display_len)
        common_end = 0;
      if (common_start + common_end > line_len)
        common_end = line_len - common_start;
      if (common_start + common_end > display_len)
        common_end = display_len - common_start;

      if (line_len < display_len) /* Some characters deleted */
        {
          /* Move cursor to start of different part */
          ssh_rl_move_cursor(rl, rl->prompt_len + common_start);

          if (line_len + rl->prompt_len < rl->row_length)
            {
              /* The line fits in one row, use delete capabilities */
              if ((display_len - line_len > 1 &&
                   rl->tc->term_delete_n_chars) ||
                  (rl->tc->term_delete_n_chars &&
                   !rl->tc->term_delete_char))
                {
                  ssh_rl_send_code_n(rl, rl->tc->term_delete_n_chars,
                                     display_len - line_len);
                }
              else if (rl->tc->term_delete_char)
                {
                  for(i = 0; i < display_len - line_len; i++)
                    ssh_rl_send_code(rl, rl->tc->term_delete_char);
                }
              else /* No terminal delete capability, draw till end of line */
                common_end = 0;
            }
          else /* More than one line, draw till end of line */
            common_end = 0;

          ssh_rl_write_string(rl, rl->line + common_start,
                              line_len - common_end - common_start);

          if (common_end == 0)
            {
              /* More than 1 line deleted */
              if (display_len - line_len > rl->row_length &&
                  rl->tc->term_clear_to_end_of_line)
                {
                  /* Clear first line */
                  ssh_rl_send_code(rl, rl->tc->term_clear_to_end_of_line);
                  for(i = ((rl->prompt_len + line_len) / rl->row_length + 1)
                        * rl->row_length;
                      i < display_len;
                      i += rl->row_length)
                    {
                      ssh_rl_move_cursor(rl, i);
                      ssh_rl_send_code(rl, rl->tc->term_clear_to_end_of_line);
                    }
                  ssh_rl_move_cursor(rl, i);
                  ssh_rl_send_code(rl, rl->tc->term_clear_to_end_of_line);
                }
              else
                {
                  /* Some junk may remain after last char */
                  if ((rl->display_cursor % rl->row_length) !=
                      rl->row_length - 1 &&
                      rl->tc->term_clear_to_end_of_line)
                    {
                      /* We are not at the end of line and clear to end of line
                         found, use it */
                      ssh_rl_send_code(rl, rl->tc->term_clear_to_end_of_line);
                    }
                  else
                    {
                      /* Either we are at the end of line, or clear to end of
                         line wasn't found. Use spaces */
                      for(i = 0; i < display_len - line_len; i += 10)
                        ssh_rl_write_string(rl, "          ", 10);
                      ssh_rl_write_string(rl, "          ",
                                          (display_len - line_len + 1) % 10);
                    }
                }
            }
        }
      else if (line_len > display_len) /* Some characters inserted */
        {
          /* Move cursor to start of different part */
          ssh_rl_move_cursor(rl, rl->prompt_len + common_start);

          if (line_len + rl->prompt_len < rl->row_length)
            {
              /* The line fits in one row, use insert capabilities */
              if ((line_len - display_len > 1 &&
                   rl->tc->term_insert_n_chars) ||
                  (rl->tc->term_insert_n_chars &&
                   !rl->tc->term_insert_char))
                {
                  ssh_rl_send_code_n(rl, rl->tc->term_insert_n_chars,
                                     line_len - display_len);
                }
              else if (rl->tc->term_insert_char)
                {
                  for(i = 0; i < line_len - display_len; i++)
                    ssh_rl_send_code(rl, rl->tc->term_insert_char);
                }
              else /* No terminal insert capability, draw till end of line */
                common_end = 0;
            }
          else /* More than one line, draw till end of line */
            common_end = 0;

          ssh_rl_write_string(rl, rl->line + common_start,
                              line_len - common_end - common_start);
        }
      else                      /* Some characters replaced, draw them */
        {
          /* Move cursor to start of different part */
          ssh_rl_move_cursor(rl, rl->prompt_len + common_start);
          ssh_rl_write_string(rl, rl->line + common_start,
                              line_len - common_end - common_start);
        }
    }
  ssh_rl_move_cursor(rl, rl->prompt_len + rl->cursor);
  strcpy(rl->display_line, rl->line);
}

/*
 * Find start/end of sentence/word from given direction
 */
int ssh_rl_find(ReadLine rl, int start_pos, ReadLineUnit unit,
                 ReadLineDirection dir)
{
  int orig_place = start_pos;
  if (dir == SSH_RL_BACKWARD)
    {
      if (start_pos > 0)
        start_pos--;
      if (unit == SSH_RL_WORD)
        {
          /* Find next possible place */
          for(; start_pos > 0 && !isalnum(rl->line[start_pos]); start_pos--)
            ;
          if (start_pos == 0)
            return start_pos;
          for(; start_pos > 0 && isalnum(rl->line[start_pos]); start_pos--)
            ;
          if (!isalnum(rl->line[start_pos]))
            start_pos++;
          return start_pos;
        }

      /* Find next possible place */
      for(; start_pos > 0 && isspace(rl->line[start_pos]); start_pos--)
        ;
      if (start_pos == 0)
        return start_pos;
      while (start_pos > 0 && isspace(rl->line[start_pos]))
        start_pos--;
      while (start_pos > 0 &&
             (rl->line[start_pos] == '.' || rl->line[start_pos] == '!' ||
              rl->line[start_pos] == '?' || rl->line[start_pos] == ':'))
        start_pos--;

      for(; start_pos > 0 && rl->line[start_pos] != '.' &&
            rl->line[start_pos] != '!' && rl->line[start_pos] != '?' &&
            rl->line[start_pos] != ':'; start_pos--)
        ;
      if (rl->line[start_pos] == '.' || rl->line[start_pos] == '!' ||
          rl->line[start_pos] == '?' || rl->line[start_pos] == ':')
        {
          start_pos++;
          for(;start_pos < orig_place && isspace(rl->line[start_pos]);
              start_pos++)
            ;
        }
      return start_pos;
    }
  /* Find next possible place */
  if (unit == SSH_RL_WORD)
    {
      for(; start_pos < rl->end && !isalnum(rl->line[start_pos]); start_pos++)
        ;
      if (start_pos == rl->end)
        return start_pos;
      for(; start_pos < rl->end && isalnum(rl->line[start_pos]); start_pos++)
        ;
      return start_pos;
    }
  for(; start_pos < rl->end && isspace(rl->line[start_pos]); start_pos++)
    ;
  if (start_pos == rl->end)
    return start_pos;
  for(; start_pos < rl->end && rl->line[start_pos] != '.' &&
        rl->line[start_pos] != '!' && rl->line[start_pos] != '?' &&
        rl->line[start_pos] != ':'; start_pos++)
    ;
  while (start_pos < rl->end &&
         (rl->line[start_pos] == '.' || rl->line[start_pos] == '!' ||
          rl->line[start_pos] == '?' || rl->line[start_pos] == ':'))
    start_pos++;
  while (start_pos < rl->end && isspace(rl->line[start_pos]))
    start_pos++;
  return start_pos;
}

/*
 * Downcase character
 */
void ssh_rl_downcase(ReadLine rl, int pos)
{
  rl->line[pos] = tolower(rl->line[pos]);
}

/*
 * Upcase character
 */
void ssh_rl_upcase(ReadLine rl, int pos)
{
  rl->line[pos] = toupper(rl->line[pos]);
}

/*
 * Run command between two positions
 */
void ssh_rl_run(ReadLine rl, int st, int en,
                void (*func)(ReadLine rl, int pos))
{
  int i;

  for(i = st; i < en; i++)
    {
      (*func)(rl, i);
    }
  return;
}

/*
 * Run command on region
 */
void ssh_rl_run_region(ReadLine rl, void (*func)(ReadLine rl, int pos))
{
  if (rl->mark == -1)
    return;
  if (rl->mark == rl->cursor)
    return;
  ssh_rl_store_undo(rl);
  if (rl->mark < rl->cursor)
    ssh_rl_run(rl, rl->mark, rl->cursor, func);
  else
    ssh_rl_run(rl, rl->cursor, rl->mark, func);
  rl->last_command_cut = -1;
  return;
}

void ssh_rl_cmd_hist_dump(ReadLine rl)
{
  SshADTHandle h;
  int i = 0;

  SSH_DEBUG(16, ("command history:"));
  for (h = ssh_adt_enumerate_start(rl->cmd_hist);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(rl->cmd_hist, h))
    {
      SSH_DEBUG(16, ("%d: `%s'", i, ssh_adt_get(rl->cmd_hist, h)));
      i++;
    }
}

/* Go "up" in command history, update readline params. */
void ssh_rl_cmd_hist_previous(ReadLine rl)
{
  SshADTHandle h = SSH_ADT_INVALID;
  Boolean at_end = FALSE;
  char *new_line = NULL;

  if (rl->cmd_hist == NULL)
    return;

  if (rl->ch == SSH_ADT_INVALID)
    {
      /* No items entered, send visual bell and return. */
      if (rl->tc->term_visual_bell)
        ssh_rl_send_code(rl, rl->tc->term_visual_bell);
      return;
    }

  at_end = (ssh_adt_next(rl->cmd_hist, rl->ch) == SSH_ADT_INVALID);

  rl->line[rl->end] = '\0';

  if (at_end)
    {
      if (strcmp(rl->line, ssh_adt_get(rl->cmd_hist, rl->ch)) != 0)
        {
          SSH_DEBUG(16, ("Adding line `%s' to end.",
                        rl->line));
          h = ssh_adt_insert_to(rl->cmd_hist, SSH_ADT_END,
                                 ssh_xstrdup(rl->line));
          SSH_VERIFY(h != SSH_ADT_INVALID);
          /* If last item was inserted in here, delete current. */
          if (rl->line_entered_in_cmd_hist == TRUE)
            {
              ssh_adt_delete(rl->cmd_hist, rl->ch);
            }
          rl->line_entered_in_cmd_hist = TRUE;
          rl->ch = h;
          if (SSH_DEBUG_ENABLED(16))
            ssh_rl_cmd_hist_dump(rl);
        }
    }

  h = ssh_adt_previous(rl->cmd_hist, rl->ch);

  if (h != SSH_ADT_INVALID)
    {
      /* If current item is different from the one in command history,
         replace it. */
      if (strcmp(rl->line, ssh_adt_get(rl->cmd_hist, rl->ch)) != 0)
        {
          SSH_DEBUG(16, ("Item different in command history, replacing."));
          SSH_VERIFY(ssh_adt_insert_at(rl->cmd_hist, SSH_ADT_AFTER,
                                       rl->ch, ssh_xstrdup(rl->line)) !=
                     SSH_ADT_INVALID);
          ssh_adt_delete(rl->cmd_hist, rl->ch);
          if (SSH_DEBUG_ENABLED(16))
            ssh_rl_cmd_hist_dump(rl);
        }

      rl->ch = h;
      new_line = ssh_adt_get(rl->cmd_hist, rl->ch);
      ssh_rl_enlarge_to(rl, strlen(new_line));
      strncpy(rl->line, new_line, strlen(new_line));
      rl->cursor = strlen(new_line);
      rl->end = rl->cursor;
    }
  else
    {
      /* At the top, send visual bell. */
      if (rl->tc->term_visual_bell)
        ssh_rl_send_code(rl, rl->tc->term_visual_bell);
    }
}

/* Go "down" in command history, update readline params. */
void ssh_rl_cmd_hist_next(ReadLine rl)
{
  SshADTHandle h = SSH_ADT_INVALID;

  if (rl->cmd_hist == NULL)
    return;

  /* If rl->ch is invalid, the h will also be, so visual bell will be
     output in either case. */
  if (rl->ch != SSH_ADT_INVALID)
    h = ssh_adt_next(rl->cmd_hist, rl->ch);

  if (h == SSH_ADT_INVALID)
    {
      if (rl->tc->term_visual_bell)
        ssh_rl_send_code(rl, rl->tc->term_visual_bell);
    }
  else
    {
      char *new_line = NULL;
      ssh_rl_delete_undo(rl);

      rl->line[rl->end] = '\0';

      /* If current item is different from the one in command history,
         replace it. */
      if (strcmp(rl->line, ssh_adt_get(rl->cmd_hist, rl->ch)) != 0)
        {
          SSH_DEBUG(16, ("Item different in command history, replacing."));
          SSH_VERIFY(ssh_adt_insert_at(rl->cmd_hist, SSH_ADT_AFTER,
                                       rl->ch, ssh_xstrdup(rl->line)) !=
                     SSH_ADT_INVALID);
          ssh_adt_delete(rl->cmd_hist, rl->ch);
          if (SSH_DEBUG_ENABLED(16))
            ssh_rl_cmd_hist_dump(rl);
        }

      new_line = ssh_adt_get(rl->cmd_hist, h);
      ssh_rl_enlarge_to(rl, strlen(new_line));
      strncpy(rl->line, new_line, strlen(new_line));
      rl->cursor = strlen(new_line);
      rl->end = rl->cursor;
      rl->ch = h;
    }
}

/* Timeout callback for waiting for rest of cursor key bytes to get through. */
void read_timeout(void *context)
{
  ReadLine rl = (ReadLine)context;
  rl->timeout_done = TRUE;
  iocb(SSH_STREAM_INPUT_AVAILABLE, rl);
}

/*
 * Readline main loop
 */
int ssh_rl_loop(ReadLine rl)
{
  char read_buf[64];
  int ret = 0, j;
  char *p;
  int tmp;
  int len, pos;
  int start1, start2, end1, end2;
  size_t read_len = 0;

  if (!rl->eloop)
    ssh_rl_store_undo(rl);

  do {
    ssh_rl_redraw_display(rl);
    read_len = 0;
    ret = 0;

    if (rl->eloop)
      {
        ssh_cancel_timeouts(read_timeout, SSH_ALL_CONTEXTS);
        do {
          ret = ssh_stream_read(rl->stream, read_buf + read_len,
                                sizeof(read_buf) - read_len);

          if (ret == 0)
            return -1;

          if (ret < 0 && read_len == 0)
            return 1;

          if (ret > 0)
            read_len += ret;
        } while (ret > 0 && read_len < sizeof(read_buf));
        SSH_DEBUG(20, ("ret: %ld, read_len: %d, buf_len: %ld",
                       ret, read_len, sizeof(read_buf)));
      }
    else
      {
      restart_blocking_read:
        ret = read(rl->read_fd, read_buf, sizeof(read_buf));

        if (ret <= 0)
          return -1;

        read_len = ret;
      }
    SSH_DEBUG_HEXDUMP(15, ("got %d bytes:", read_len), read_buf, read_len);

    ssh_xbuffer_append(rl->in_buffer, (unsigned char *)read_buf, read_len);

    for (; ssh_buffer_len(rl->in_buffer) > 0;
         ssh_buffer_consume(rl->in_buffer, 1))
      {
        Boolean partial_match = FALSE;
        /* Check for arrow keys first, as they have an escape, and if
           tested later, would be mistaken as a "normal" escape
           sequence. */
#define CHECK_FOR_CURSOR_KEY(var, where_to)                             \
        do {                                                            \
          if (ssh_buffer_len(rl->in_buffer) >= strlen((var)) &&         \
              strncmp((char *)ssh_buffer_ptr(rl->in_buffer), (var),     \
                      strlen((var))) == 0)                              \
            {                                                           \
              ssh_buffer_consume(rl->in_buffer, strlen((var)) - 1);     \
              goto where_to;                                            \
            }                                                           \
          /* Check for partial match. */                                \
          if (!partial_match &&                                         \
              ssh_buffer_len(rl->in_buffer) < strlen((var)) &&          \
              strncmp((char *)ssh_buffer_ptr(rl->in_buffer), (var),     \
                      ssh_buffer_len(rl->in_buffer)) == 0)              \
            {                                                           \
              SSH_DEBUG(18, ("partial match on cursor key."));          \
              partial_match = TRUE;                                     \
            }                                                           \
        } while (0)

        CHECK_FOR_CURSOR_KEY(rl->tc->key_left_arrow, left_arrow);
        CHECK_FOR_CURSOR_KEY(rl->tc->key_up_arrow, up_arrow);
        CHECK_FOR_CURSOR_KEY(rl->tc->key_right_arrow, right_arrow);
        CHECK_FOR_CURSOR_KEY(rl->tc->key_down_arrow, down_arrow);
        if (partial_match && !rl->timeout_done)
          {
            /* Need to get more character data to distinguish possible
               cursor key. */
            if (rl->eloop)
              {
                /* A small timeout (about 200 ms) should be enough for
                   the rest of the cursor key bytes to get through. If
                   not, well, tough luck. */
                SSH_DEBUG(18, ("entering timeout"));
                ssh_register_timeout(0L, 200000L, read_timeout, rl);
                return 1;
              }
            else
              {
                /* In principle, this could lead certain escape
                   sequences (which start as cursor keys) to not
                   execute before user inputs more, but in practise it
                   is a cheap price to pay for working cursor keys (at
                   least to the "normal" user...).*/
                SSH_DEBUG(18, ("restarting read()"));
                goto restart_blocking_read;
              }
          }
        rl->timeout_done = FALSE;

        if (rl->keymap == SSH_READLINE_NORMAL_KEYMAP)
          {
            switch(*(ssh_buffer_ptr(rl->in_buffer)))
              {
              case 000: /* Ctrl-Space: Set mark */
                rl->mark = rl->cursor;
                rl->last_command_cut = -1;
                break;
              case 001: /* Ctrl-A: Beginning of line */
                rl->cursor = 0;
                break;
              case 002: /* Ctrl-B: Backward character */
              left_arrow:
                if (rl->cursor > 0)
                  rl->cursor--;
                break;
              case 003: /* Ctrl-C: Ignored */
                ssh_rl_redraw_display(rl);
                break;
              case 004: /* Ctrl-D: Erase character right, or eof if beginning
                           of empty line. */
                if (rl->end == 0)
                  {
                    ssh_buffer_consume(rl->in_buffer, 1);
                    return -1;
                  }
                /* At end of line */
                if (rl->end == rl->cursor)
                  break;
                ssh_rl_store_undo(rl);
                memmove(rl->line + rl->cursor, rl->line + rl->cursor + 1,
                        rl->end - rl->cursor - 1);
                if (rl->mark == rl->cursor)
                  rl->mark = -1;
                else if (rl->mark > rl->cursor)
                  rl->mark--;
                rl->end--;
                rl->last_command_cut = -1;
                break;
              case 005: /* Ctrl-E: End of line */
                rl->cursor = rl->end;
                break;
              case 006: /* Ctrl-F: Forward character */
              right_arrow:
                if (rl->cursor != rl->end)
                  rl->cursor++;
                break;
              case 007: /* Ctrl-G: Ignored */
                break;
              case 010: /* Ctrl-H: Backspace */
              case 0177: /* Delete: Backspace */
                if (rl->cursor == 0)
                  break;
                ssh_rl_store_undo(rl);
                if (rl->cursor != rl->end)
                  memmove(rl->line + rl->cursor - 1, rl->line + rl->cursor,
                          rl->end - rl->cursor);
                if (rl->mark == rl->cursor - 1)
                  rl->mark = -1;
                else if (rl->mark > rl->cursor)
                  rl->mark--;
                rl->cursor--;
                rl->end--;
                rl->last_command_cut = -1;
                break;
              case 011: /* Ctrl-I: Tab */
                if (rl->extended_mode &&
                    rl->wanted_flags & SSH_READLINE_CB_FLAG_TAB)
                  {
                    rl->flags = rl->flags | SSH_READLINE_CB_FLAG_TAB;
                    ssh_buffer_consume(rl->in_buffer, 1);
                    return 0;
                  }
                else
                  {
                    ssh_rl_store_undo(rl);
                    len = (rl->cursor + rl->prompt_len) % 8;
                    len = 8 - len;
                    ssh_rl_enlarge_to(rl, rl->end + len);
                    if (rl->end != rl->cursor)
                      memmove(rl->line + rl->cursor + len,
                              rl->line + rl->cursor,
                              rl->end - rl->cursor);
                    if (rl->mark > rl->cursor)
                      rl->mark += len;
                    for(; len > 0; len--)
                      {
                        rl->line[rl->cursor] = ' ';
                        rl->end++;
                        rl->cursor++;
                      }
                    rl->last_command_cut = -1;
                  }
                break;
              delete_end_of_line:
              case 013: /* Ctrl-K: Delete to end of line */
                if (rl->cursor == rl->end)
                  break;
                ssh_rl_store_undo(rl);
                len = rl->end - rl->cursor;
                if (rl->yank)
                  {
                    if (rl->last_command_cut == rl->end)
                      {
                        rl->yank = ssh_xrealloc(rl->yank, len +
                                                strlen(rl->yank) + 1);
                        memmove(rl->yank + len, rl->yank,
                                strlen(rl->yank) + 1);
                        memmove(rl->yank, rl->line + rl->cursor, len);
                      }
                    else
                      {
                        ssh_xfree(rl->yank);
                        rl->yank = ssh_xmalloc(len + 1);
                        memmove(rl->yank, rl->line + rl->cursor, len);
                        rl->yank[len] = '\0';
                      }
                  }
                else
                  {
                    rl->yank = ssh_xmalloc(len + 1);
                    memmove(rl->yank, rl->line + rl->cursor, len);
                    rl->yank[len] = '\0';
                  }

                rl->end = rl->cursor;
                if (rl->mark > rl->cursor)
                  rl->mark = -1;
                rl->last_command_cut = rl->end;
                break;
              case 014: /* Ctrl-L: Redraw line */
                memset(rl->display_line, 1, rl->end);
                ssh_rl_move_cursor(rl, 0);
                ssh_rl_write_string(rl, rl->prompt, rl->prompt_len);
                break;
              case 012: /* Ctrl-J: Accept line */
              case 015: /* Ctrl-M: Accept line */
                /* Store the line into the end of command history
                   list.  Current position will move to the end of the
                   command history. */
                if (rl->cmd_hist != NULL)
                  {
                    Boolean at_end = FALSE;

                    rl->line[rl->end] = '\0';

                    if (rl->ch != SSH_ADT_INVALID)
                      at_end = (ssh_adt_next(rl->cmd_hist, rl->ch) ==
                                SSH_ADT_INVALID);

                    /* If last item was entered via "up" in command
                       history, delete it (effectively we replace it). */
                    if (at_end && rl->line_entered_in_cmd_hist == TRUE)
                        ssh_adt_delete(rl->cmd_hist, rl->ch);

                    rl->ch = ssh_adt_insert_to(rl->cmd_hist, SSH_ADT_END,
                                               ssh_xstrdup(rl->line));
                    SSH_VERIFY(rl->ch != SSH_ADT_INVALID);
                    rl->line_entered_in_cmd_hist = FALSE;
                  }
                ssh_buffer_consume(rl->in_buffer, 1);
                return 0;
              case 016: /* Ctrl-N: Next line */
              down_arrow:
                if (rl->cursor <= rl->end - rl->row_length)
                  rl->cursor += rl->row_length;
                else
                  ssh_rl_cmd_hist_next(rl);
                break;
              case 017: /* Ctrl-O: Ignored */
                break;
              case 020: /* Ctrl-P: Previous line */
              up_arrow:
                if (rl->cursor >= rl->row_length)
                  rl->cursor -= rl->row_length;
                else
                  ssh_rl_cmd_hist_previous(rl);
                break;
              case 021: /* Ctrl-Q: Ignored */
              case 022: /* Ctrl-R: Ignored */
              case 023: /* Ctrl-S: Ignored */
                break;
              case 024: /* Ctrl-T: toggle two chars */
                if (rl->cursor == 0)
                  break;
                if (rl->end <= 1)
                  break;
                if (rl->end != rl->cursor)
                  rl->cursor++;
                ssh_rl_store_undo(rl);
                tmp = rl->line[rl->cursor - 2];
                rl->line[rl->cursor - 2] = rl->line[rl->cursor - 1];
                rl->line[rl->cursor - 1] = tmp;
                rl->last_command_cut = -1;
                break;
              case 025: /* Ctrl-U: Kill line */
                ssh_rl_store_undo(rl);
                rl->cursor = 0;
                goto delete_end_of_line;
              case 026: /* Ctrl-V: Ignored */
                break;
              kill_region:
              case 027: /* Ctrl-W: kill-region */
                /* Mark not set kill word */
                if (rl->mark == -1 || rl->mark > rl->end)
                  {
                    if (rl->cursor == 0)
                      break;
                    j = rl->cursor;

                    /* Skip whitespace */
                    if (isspace(rl->line[j - 1]))
                      for(; j > 0; j--)
                        if (!isspace(rl->line[j - 1]))
                          break;

                    /* If the next char is alphanumeric, remove all
                       alphanumeric characters */
                    if (isalnum(rl->line[j - 1]))
                      {
                        for(; j > 0; j--)
                          if (!isalnum(rl->line[j - 1]))
                            break;
                      }
                    else   /* If the next char is not alphanumeric
                              remove up to next whitespace. */
                      {
                        for(; j > 0; j--)
                          if (isspace(rl->line[j - 1]))
                            break;
                      }
                    rl->mark = j;
                  }
                if (rl->mark == rl->cursor)
                  break;
                ssh_rl_store_undo(rl);
                if (rl->mark < rl->cursor)
                  {
                    pos = rl->mark;
                    rl->mark = rl->cursor;
                    rl->cursor = pos;
                  }
                len = rl->mark - rl->cursor;
                if (rl->yank)
                  {
                    if (rl->last_command_cut == rl->mark)
                      {
                        rl->yank = ssh_xrealloc(rl->yank, len +
                                                strlen(rl->yank) + 1);
                        memmove(rl->yank + len, rl->yank,
                                strlen(rl->yank) + 1);
                        memmove(rl->yank, rl->line + rl->cursor, len);
                      }
                    else if (rl->last_command_cut == rl->cursor)
                      {
                        rl->yank = ssh_xrealloc(rl->yank, len +
                                                strlen(rl->yank) + 1);
                        rl->yank[strlen(rl->yank) + len] = '\0';
                        memmove(rl->yank + strlen(rl->yank),
                                rl->line + rl->cursor, len);
                      }
                    else
                      {
                        ssh_xfree(rl->yank);
                        rl->yank = ssh_xmalloc(len + 1);
                        memmove(rl->yank, rl->line + rl->cursor, len);
                        rl->yank[len] = '\0';
                      }
                  }
                else
                  {
                    rl->yank = ssh_xmalloc(len + 1);
                    memmove(rl->yank, rl->line + rl->cursor, len);
                    rl->yank[len] = '\0';
                  }

                memmove(rl->line + rl->cursor, rl->line + rl->mark,
                        rl->end - rl->mark);
                rl->end -= len;
                rl->mark = -1;
                rl->last_command_cut = rl->cursor;
                break;
              case 030: /* Ctrl-X: extended command map */
                rl->keymap = SSH_READLINE_CTRL_X_KEYMAP;
                break;
              case 031: /* Ctrl-Y: yank */
                if (rl->yank && strlen(rl->yank) > 0)
                  {
                    ssh_rl_store_undo(rl);
                    len = strlen(rl->yank);
                    ssh_rl_enlarge_to(rl, rl->end + len);
                    memmove(rl->line + rl->cursor + len, rl->line + rl->cursor,
                            rl->end - rl->cursor);
                    memmove(rl->line + rl->cursor, rl->yank, len);
                    rl->mark = rl->cursor;
                    rl->end += len;
                    rl->cursor += len;
                    rl->last_command_cut = -1;
                  }
                break;
              case 032: /* Ctrl-Z: Ignored */
                break;
              case 033: /* Ctrl-[: Esc comamnd map */
                rl->keymap = SSH_READLINE_ESC_KEYMAP;
                break;
              case 034: /* Ctrl-\: Ignored */
                break;
              case 035: /* Ctrl-]: Ignored */
                break;
              case 036: /* Ctrl-^: Ignored */
                break;
              undo:
              case 037: /* Ctrl-_: Undo */
                if (rl->undo_length == 0)
                  break;
                if (rl->undo_position == -1)
                  {
                    ssh_rl_store_undo(rl);
                    rl->undo_position = rl->undo_length - 1;
                    rl->undo_direction = -1;
                  }
                rl->undo_position += rl->undo_direction;
                if (rl->undo_position == 0)
                  {
                    rl->undo_direction = 1;
                  }
                else if (rl->undo_position >= rl->undo_length - 1)
                  {
                    rl->undo_direction = -1;
                  }
                rl->cursor = rl->undo_cursors[rl->undo_position];
                ssh_rl_enlarge_to(rl, strlen(rl->undo[rl->undo_position]));
                strcpy(rl->line, rl->undo[rl->undo_position]);
                rl->end = strlen(rl->line);
                rl->mark = -1;
                rl->last_command_cut = -1;
                break;
              case 0200: case 0201: case 0202: case 0203:
              case 0204: case 0205: case 0206: case 0207:
              case 0210: case 0211: case 0212: case 0213:
              case 0214: case 0215: case 0216: case 0217:
              case 0220: case 0221: case 0222: case 0223:
              case 0224: case 0225: case 0226: case 0227:
              case 0230: case 0231: case 0232: case 0233:
              case 0234: case 0235: case 0236: case 0237:
                {
                  unsigned char *p = ssh_buffer_ptr(rl->in_buffer);
                  *p &= 0x7f;
                }
                goto esc_map;
                break;
              insert_char:
              default:
                ssh_rl_store_undo(rl);
                ssh_rl_enlarge_to(rl, rl->end + 1);
                if (rl->end != rl->cursor)
                  memmove(rl->line + rl->cursor + 1, rl->line + rl->cursor,
                          rl->end - rl->cursor);
                rl->line[rl->cursor] = (char)*(ssh_buffer_ptr(rl->in_buffer));
                if (rl->mark > rl->cursor)
                  rl->mark++;
                rl->end++;
                rl->cursor++;
                rl->last_command_cut = -1;
                break;
              }
          }
        else if (rl->keymap == SSH_READLINE_CTRL_X_KEYMAP)
          {
            rl->keymap = SSH_READLINE_NORMAL_KEYMAP;
            switch(*(ssh_buffer_ptr(rl->in_buffer)))
              {
              case 003: /* Ctrl-X Ctrl-C: Exit */
                ssh_buffer_consume(rl->in_buffer, 1);
                return 0;
              case 014: /* Ctrl-X Ctrl-L: Downcase region */
                ssh_rl_run_region(rl, ssh_rl_downcase);
                break;
              case 025: /* Ctrl-X Ctrl-U: Upcase region */
                ssh_rl_run_region(rl, ssh_rl_upcase);
                break;
              case 030: /* Ctrl-X Ctrl-X: Exchange point and mark */
                if (rl->mark == -1)
                  break;
                pos = rl->mark;
                rl->mark = rl->cursor;
                rl->cursor = pos;
                break;
              case 'h': /* Ctrl-X h: Mark whole buffer */
                rl->mark = 0;
                rl->cursor = rl->end;
                break;
              case 'u': /* Ctrl-X u: Undo */
                goto undo;
              default:
                break;
              }
          }
        else if (rl->keymap == SSH_READLINE_ESC_KEYMAP)
          {
          esc_map:
            rl->keymap = SSH_READLINE_NORMAL_KEYMAP;
            switch(*(ssh_buffer_ptr(rl->in_buffer)))
              {
              case 010: /* Esc Ctrl-h: Kill word backward */
              case 0177: /* Esc del: Kill word backward */
                rl->mark = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                       SSH_RL_BACKWARD);
                goto kill_region;
              case ' ': /* Esc Spc: Just one space */
              case '\\':/* Esc \\: delete spaces */
                if (!isspace(rl->line[rl->cursor]) &&
                    (rl->cursor == 0 || !isspace(rl->line[rl->cursor - 1])))
                  {
                    if ((char)*(ssh_buffer_ptr(rl->in_buffer)) == ' ')
                      goto insert_char;
                    break;
                  }
                ssh_rl_store_undo(rl);
                if (rl->cursor > 0 && isspace(rl->line[rl->cursor - 1]))
                  rl->cursor--;
                tmp = rl->cursor;
                for(; rl->cursor > 0 && isspace(rl->line[rl->cursor]);
                    rl->cursor--)
                  ;
                if (isspace(rl->line[rl->cursor]))
                  rl->cursor++;
                else
                  rl->cursor += 2;
                if ((char)*(ssh_buffer_ptr(rl->in_buffer)) == '\\')
                  rl->cursor--;
                for(; tmp < rl->end && isspace(rl->line[tmp]); tmp++)
                  ;
                memmove(rl->line + rl->cursor, rl->line + tmp,
                        rl->end - tmp);
                rl->end -= (tmp - rl->cursor);
                if (rl->mark >= rl->cursor &&
                    rl->mark <= tmp)
                  rl->mark = -1;
                else if (rl->mark > tmp)
                  rl->mark -= (tmp - rl->cursor);
                rl->last_command_cut = -1;
                break;
              case '<': /* Esc <: Beginning of buffer */
                rl->cursor = 0;
                break;
              case '>': /* Esc >: End of buffer */
                rl->cursor = rl->end;
                break;
              case '@': /* Esc @: Mark word */
                rl->mark = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                       SSH_RL_FORWARD);
                break;
              case 'a': /* Esc A: Backward sentence */
                rl->cursor = ssh_rl_find(rl, rl->cursor, SSH_RL_SENTENCE,
                                         SSH_RL_BACKWARD);
                break;
              case 'b': /* Esc B: Backward word */
                rl->cursor = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                         SSH_RL_BACKWARD);
                break;
              case 'c': /* Esc C: Capitalize word */
                pos = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD, SSH_RL_FORWARD);
                len = ssh_rl_find(rl, pos, SSH_RL_WORD, SSH_RL_BACKWARD);
                if (len < rl->cursor)
                  {
                    len = rl->cursor;
                  }
                ssh_rl_store_undo(rl);
                rl->line[len] = toupper(rl->line[len]);
                rl->last_command_cut = -1;
                rl->cursor = pos;
                break;
              case 'd': /* Esc D: Kill word */
                rl->mark = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                       SSH_RL_FORWARD);
                goto kill_region;
              case 'e': /* Esc e: Forward sentence */
                rl->cursor = ssh_rl_find(rl, rl->cursor, SSH_RL_SENTENCE,
                                         SSH_RL_FORWARD);
                break;
              case 'f': /* Esc f: Forward word */
                rl->cursor = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                         SSH_RL_FORWARD);
                break;
              case 'k': /* Esc k: Kill sentence */
                rl->mark = ssh_rl_find(rl, rl->cursor, SSH_RL_SENTENCE,
                                       SSH_RL_FORWARD);
                goto kill_region;
              case 'l': /* Esc l: Lowercase word */
                ssh_rl_store_undo(rl);
                pos = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD, SSH_RL_FORWARD);
                ssh_rl_run(rl, rl->cursor, pos, ssh_rl_downcase);
                rl->cursor = pos;
                rl->last_command_cut = -1;
                break;
              case 't': /* Esc t: Transpose words */
                ssh_rl_store_undo(rl);
                start1 = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD,
                                     SSH_RL_BACKWARD);
                end1 = ssh_rl_find(rl, start1, SSH_RL_WORD,
                                   SSH_RL_FORWARD);
                end2 = ssh_rl_find(rl, end1, SSH_RL_WORD, SSH_RL_FORWARD);
                start2 = ssh_rl_find(rl, end2, SSH_RL_WORD, SSH_RL_BACKWARD);
                if (start1 == start2)
                  {
                    start1 = ssh_rl_find(rl, start2, SSH_RL_WORD,
                                         SSH_RL_BACKWARD);
                    end1 = ssh_rl_find(rl, start1, SSH_RL_WORD,
                                       SSH_RL_FORWARD);
                  }
                if (start1 == start2)
                  break;
                p = ssh_xmalloc(start2 - start1);
                memmove(p, rl->line + start1, start2 - start1);
                memmove(rl->line + start1, rl->line + start2, end2 - start2);
                memmove(rl->line + start1 + end2 - start2,
                        p + end1 - start1, start2 - end1);
                memmove(rl->line + end2 - (end1 - start1),
                        p, end1 - start1);
                rl->last_command_cut = -1;
                rl->cursor = end2;
                break;
              case 'u': /* Esc u: Uppercase word */
                ssh_rl_store_undo(rl);
                pos = ssh_rl_find(rl, rl->cursor, SSH_RL_WORD, SSH_RL_FORWARD);
                ssh_rl_run(rl, rl->cursor, pos, ssh_rl_upcase);
                rl->cursor = pos;
                rl->last_command_cut = -1;
                break;
              }
          }
      }
  } while (TRUE);

  if (rl->eloop)
    ssh_rl_redraw_display(rl);

  return 1;
}

/*
 * Initialize terminal capability entries (either terminfo or older termcap)
 */
void ssh_rl_initialize_term_capabilities(ReadLine rl)
{
  Boolean use_builtin = FALSE;
#if defined(HAVE_LIBTERMCAP) || defined(HAVE_LIBNCURSES) || defined(HAVE_LIBXCURSES)
  char *term;

  if (ssh_last_term == NULL)
    {
      ssh_last_term = ssh_xcalloc(1, sizeof(Term));
#ifndef HAVE_TERMINFO
      ssh_last_term->term_buffer = ssh_xcalloc(1024, sizeof(char));
#endif /* HAVE_TERMINFO */
      ssh_last_term->term_type = NULL;
    }
  term = getenv("TERM");
  if (term == NULL)
    {
      ssh_last_term->term_type = term;
      use_builtin = TRUE;
    }
  else if (term != ssh_last_term->term_type)
    {
#ifdef HAVE_TERMINFO
      int err_ret;

      if (setupterm(term, 1, &err_ret) != OK)
        {
          ssh_warning("Setting up terminal for term `%s' failed.", term);
          switch (err_ret)
            {
            case 1:
              SSH_TRACE(2, ("reason: Terminal is a hardcopy, cannot be used "
                            "for curses."));
              break;
            case 0:
              SSH_TRACE(2, ("reason: Terminal entry could not be found, or "
                            "it is a generic type which has too little "
                            "information for curses applications to run."));
              break;
            case -1:
              SSH_TRACE(2, ("reason: Terminfo database could not be found."));
              break;
            default:
              SSH_TRACE(2, ("reason: Unknown error with setupterm()."));
              break;
            }
          use_builtin = TRUE;
          ssh_last_term->term_type = term;
        }
#else /* HAVE_TERMINFO */
      char tcbuffer[1024], *term_buffer_ptr;

      if (tgetent(tcbuffer, term) < 0)
        {
          ssh_warning("No termcap entry for `%.50s' found, "
                      "using vt100",
                      term);
          use_builtin = TRUE;
          ssh_last_term->term_type = term;
        }
#endif /* HAVE_TERMINFO */
      else
        {
          ssh_last_term->term_type = term;
#ifdef HAVE_TERMINFO
          ssh_last_term->term_auto_margin = tigetflag("am");;
#define GET_STRCAP(cap, var)                                               \
          do {                                                             \
            char *cap_str = tigetstr((cap));                               \
            if (cap_str != NULL && cap_str != (char *)-1)                  \
              {                                                            \
                SSH_DEBUG_HEXDUMP(7, (cap ":"), cap_str, strlen(cap_str)); \
                (var) = cap_str;                                           \
              }                                                            \
            else                                                           \
              {                                                            \
                SSH_DEBUG(7, ("term does not have cap \"%s\".", (cap)));   \
                (var) = NULL;                                              \
              }                                                            \
          } while (0)
          GET_STRCAP("smkx",  ssh_last_term->term_cursor_key_app_mode_on);
          GET_STRCAP("rmkx",  ssh_last_term->term_cursor_key_app_mode_off);
          GET_STRCAP("el",    ssh_last_term->term_clear_to_end_of_line);
          GET_STRCAP("hpa",   ssh_last_term->term_set_cursor_column);
          GET_STRCAP("dch",   ssh_last_term->term_delete_n_chars);
          GET_STRCAP("dch1",  ssh_last_term->term_delete_char);
          GET_STRCAP("ich",   ssh_last_term->term_insert_n_chars);
          GET_STRCAP("ich1",  ssh_last_term->term_insert_char);
          GET_STRCAP("cud",   ssh_last_term->term_move_cursor_down_n);
          GET_STRCAP("cud1",  ssh_last_term->term_move_cursor_down);
          GET_STRCAP("cub",   ssh_last_term->term_move_cursor_left_n);
          GET_STRCAP("cub1",  ssh_last_term->term_move_cursor_left);
          GET_STRCAP("cuf",   ssh_last_term->term_move_cursor_right_n);
          GET_STRCAP("cuf1",  ssh_last_term->term_move_cursor_right);
          GET_STRCAP("cuu",   ssh_last_term->term_move_cursor_up_n);
          GET_STRCAP("cuu1",  ssh_last_term->term_move_cursor_up);
          GET_STRCAP("flash", ssh_last_term->term_visual_bell);
          GET_STRCAP("kcud1", ssh_last_term->key_down_arrow);
          GET_STRCAP("kcub1", ssh_last_term->key_left_arrow);
          GET_STRCAP("kcuf1", ssh_last_term->key_right_arrow);
          GET_STRCAP("kcuu1", ssh_last_term->key_up_arrow);
#undef GET_STRCAP
#else /* HAVE_TERMINFO */
          term_buffer_ptr = ssh_last_term->term_buffer;
          ssh_last_term->term_auto_margin = tgetflag("am");
#define GET_STRCAP(cap, var)                                               \
          do {                                                             \
            char *cap_str = tgetstr((cap), &term_buffer_ptr);              \
            if (cap_str != NULL)                                           \
              {                                                            \
                SSH_DEBUG_HEXDUMP(7, (cap ":"), cap_str, strlen(cap_str)); \
                (var) = cap_str;                                           \
              }                                                            \
            else                                                           \
              {                                                            \
                SSH_DEBUG(7, ("term does not have cap \"%s\".", (cap)));   \
                (var) = NULL;                                              \
              }                                                            \
          } while (0)
          GET_STRCAP("ce", ssh_last_term->term_clear_to_end_of_line);
          GET_STRCAP("ch", ssh_last_term->term_set_cursor_column);
          GET_STRCAP("ks", ssh_last_term->term_cursor_key_app_mode_on);
          GET_STRCAP("ke", ssh_last_term->term_cursor_key_app_mode_off);
          GET_STRCAP("DC", ssh_last_term->term_delete_n_chars);
          GET_STRCAP("dc", ssh_last_term->term_delete_char);
          GET_STRCAP("IC", ssh_last_term->term_insert_n_chars);
          GET_STRCAP("ic", ssh_last_term->term_insert_char);
          GET_STRCAP("DO", ssh_last_term->term_move_cursor_down_n);
          GET_STRCAP("do", ssh_last_term->term_move_cursor_down);
          GET_STRCAP("LE", ssh_last_term->term_move_cursor_left_n);
          GET_STRCAP("le", ssh_last_term->term_move_cursor_left);
          GET_STRCAP("RI", ssh_last_term->term_move_cursor_right_n);
          GET_STRCAP("nd", ssh_last_term->term_move_cursor_right);
          GET_STRCAP("UP", ssh_last_term->term_move_cursor_up_n);
          GET_STRCAP("up", ssh_last_term->term_move_cursor_up);
          GET_STRCAP("vb", ssh_last_term->term_visual_bell);
          GET_STRCAP("kd", ssh_last_term->key_down_arrow);
          GET_STRCAP("kl", ssh_last_term->key_left_arrow);
          GET_STRCAP("kr", ssh_last_term->key_right_arrow);
          GET_STRCAP("ku", ssh_last_term->key_up_arrow);
#undef GET_STRCAP
#endif /* HAVE_TERMINFO */
          if ((!ssh_last_term->term_move_cursor_up &&
               !ssh_last_term->term_move_cursor_up_n) ||
              (!ssh_last_term->term_move_cursor_down &&
               !ssh_last_term->term_move_cursor_down_n) ||
              (!ssh_last_term->term_set_cursor_column &&
               ((!ssh_last_term->term_move_cursor_left &&
                 !ssh_last_term->term_move_cursor_left_n) ||
                (!ssh_last_term->term_move_cursor_right &&
                 !ssh_last_term->term_move_cursor_right_n))))
            {
              ssh_warning("Need basic cursor movement capability, using "
                          "vt100");
              use_builtin = TRUE;
            }
        }
    }
#else /* HAVE_LIBTERMCAP || HAVE_LIBNCURSES || HAVE_LIBXCURSES */
  if (ssh_last_term == NULL)
    {
      ssh_last_term = ssh_xcalloc(1, sizeof(Term));
      ssh_last_term->term_buffer = ssh_xcalloc(1024, sizeof(char));
      ssh_last_term->term_type = NULL;
      ssh_warning("Need basic cursor movement capability, using vt100");
      use_builtin = TRUE;
    }
#endif /* HAVE_LIBTERMCAP || HAVE_LIBNCURSES || HAVE_LIBXCURSES */
  if (use_builtin)
    {
      /* Built-in VT100 */
      ssh_last_term->term_auto_margin          = 1;
      ssh_last_term->term_clear_to_end_of_line = "\x1b[K";
      ssh_last_term->term_set_cursor_column    = NULL;
      ssh_last_term->term_delete_n_chars       = NULL;
      ssh_last_term->term_delete_char          = NULL;
      ssh_last_term->term_insert_n_chars       = NULL;
      ssh_last_term->term_insert_char          = NULL;
      ssh_last_term->term_move_cursor_down_n   = "\x1b[%dB";
      ssh_last_term->term_move_cursor_down     = "\x1b[B";
      ssh_last_term->term_move_cursor_left_n   = "\x1b[%dD";
      ssh_last_term->term_move_cursor_left     = "\x1b[D";
      ssh_last_term->term_move_cursor_right_n  = "\x1b[%dC";
      ssh_last_term->term_move_cursor_right    = "\x1b[C";
      ssh_last_term->term_move_cursor_up_n     = "\x1b[%dA";
      ssh_last_term->term_move_cursor_up       = "\x1b[A";
      ssh_last_term->key_down_arrow            = "\x1bOB";
      ssh_last_term->key_left_arrow            = "\x1bOD";
      ssh_last_term->key_right_arrow           = "\x1bOC";
      ssh_last_term->key_up_arrow              = "\x1bOA";
    }
  rl->tc = ssh_last_term;
}

void iocb(SshStreamNotification notify, void *context)
{
  ReadLine rl = (ReadLine)context;
  int lr;

  switch (notify)
    {
    case SSH_STREAM_DISCONNECTED:
      lr = -1;
      goto disconnect;
      break;

    case SSH_STREAM_CAN_OUTPUT:
      if (ssh_buffer_len(rl->out_buffer))
        {
          int ret;
          if ((ret = ssh_stream_write(rl->stream,
                                      ssh_buffer_ptr(rl->out_buffer),
                                      ssh_buffer_len(rl->out_buffer))))
            {
              if (ret == 0)
                ssh_warning("Write failed in ssh_rl_send_string: EOF "
                            "received.");
              if (ret > 0)
                ssh_buffer_consume(rl->out_buffer, ret);
            }
        }

      ssh_rl_redraw_display(rl);
      break;

    case SSH_STREAM_INPUT_AVAILABLE:
      if ((lr = ssh_rl_loop(rl)) <= 0)
        {
        disconnect:
          ssh_stream_set_callback(rl->stream, NULL_FNPTR, NULL);

#ifndef SSH_RL_ELOOP_SET_TTY_MODES_ONCE
          ssh_rl_restore_tty_modes(rl);
#endif /* SSH_RL_ELOOP_SET_TTY_MODES_ONCE */

          ssh_rl_delete_undo(rl);

          if (rl->tc->term_cursor_key_app_mode_off)
            ssh_rl_send_code(rl, rl->tc->term_cursor_key_app_mode_off);

          if (!rl->extended_mode)
            {
              if (lr == 0 && rl->callback)
                {
                  rl->line[rl->end] = '\0';
                  (*rl->callback)(rl->line, rl->user_context);
                }

              if (lr == -1)
                {
                  if (rl->callback)
                    (*rl->callback)(NULL, rl->user_context);
                }
            }
          else
            {
              if (lr == 0 && rl->ext_callback)
                {
                  rl->line[rl->end] = '\0';
                  (*rl->ext_callback)(rl->line, rl->flags,
                                      rl->user_context);
                }

              if (lr == -1)
                {
                  if (rl->ext_callback)
                    (*rl->ext_callback)(NULL, 0, rl->user_context);
                }
            }
        }
      break;
    }
}

/*
 * Read line from user. The tty at file descriptor ``write_fd'' is put
 * to raw mode and data is read from ``read_fd'' until CR is
 * received. The ``prompt'' is used to prompt the input. ``def'' is
 * the initial data that is editable on the readline.
 *
 * When the line has been read, the function will call provided
 * ``callback'' once providing it the data read. The data is available
 * only during the callback execution.  ``context'' is given to the
 * ``callback'' as argument. ``line'' will be NULL in case of error.
 *
 */
void ssh_readline_eloop(const char *prompt, const char *def,
                        SshReadLineCtx rl, SshRLCallback callback,
                        void *context)
{
  ssh_readline_eloop_internal(prompt, def, TRUE, rl, callback, NULL_FNPTR, 0,
                              context);
}

/* Version with extended callback type. ``flags'' should be set to indicate
   which flags are supported. */
void ssh_readline_eloop_ext(const char *prompt, const char *def,
                            SshReadLineCtx rl, SshRLExtCallback ext_callback,
                            unsigned int flags, void *context)
{
  ssh_readline_eloop_internal(prompt, def, TRUE, rl, NULL_FNPTR, ext_callback,
                              flags, context);
}

/* Version that doesn't echo the passed input to the screen. */
void ssh_readpass_eloop(const char *prompt, const char *def,
                        SshReadLineCtx rl, SshRLCallback callback,
                        void *context)
{
  ssh_readline_eloop_internal(prompt, def, FALSE, rl, callback, NULL_FNPTR, 0,
                              context);
}


void ssh_readline_eloop_internal(const char *prompt, const char *def,
                                 Boolean echo, SshReadLineCtx rl,
                                 SshRLCallback callback,
                                 SshRLExtCallback ext_callback,
                                 unsigned int flags, void *context)
{
  const char *tmp, *nl, *cr;

  SSH_PRECOND(rl != NULL);

  rl->echo = echo;
#ifndef SSH_RL_ELOOP_SET_TTY_MODES_ONCE
  if (ssh_rl_set_tty_modes(rl) < 0)
    {
      if (rl->callback)
        (*rl->callback)(NULL, rl->user_context);
      SSH_TRACE(2, ("Setting tty modes failed."));
      return;
    }
#endif /* SSH_RL_ELOOP_SET_TTY_MODES_ONCE */

  if (prompt == NULL)
    prompt = "";
  rl->prompt = prompt;
  tmp = strrchr(rl->prompt, '\n');
  if (tmp != NULL)
    rl->prompt = tmp + 1;
  tmp = strrchr(rl->prompt, '\r');
  if (tmp != NULL)
    rl->prompt = tmp + 1;
  rl->prompt_len = strlen(rl->prompt);

  rl->display_cursor = 0;
  tmp = prompt;
  while (1)
    {
      nl = strchr(tmp, '\n');
      cr = strchr(tmp, '\r');
      if (nl == NULL && cr == NULL)
        break;
      if (nl != NULL && cr != NULL)
        {
          if (nl < cr)
            cr = NULL;
          else
            nl = NULL;
        }
      if (nl != NULL)
        {
          ssh_rl_write_string(rl, tmp, nl - tmp);
          ssh_rl_send_string(rl, "\r\n", 2);
          rl->display_cursor = 0;
          tmp = nl + 1;
          if (*tmp == '\r')
            tmp++;
        }
      else
        {
          ssh_rl_write_string(rl, tmp, cr - tmp);
          if (*(cr + 1) == '\n')
            {
              ssh_rl_send_string(rl, "\r\n", 2);
              cr++;
            }
          else
            ssh_rl_send_string(rl, "\r", 1);
          rl->display_cursor = 0;
          tmp = cr + 1;
        }
    }

  ssh_rl_write_string(rl, tmp, rl->prompt_len);

  if (def)
    {
      rl->line_alloc = strlen(def) + 1;
      rl->line = ssh_xstrdup(def);
    }
  else
    {
      rl->line_alloc = 80;
      rl->line = ssh_xmalloc(rl->line_alloc);
      rl->line[0] = '\0';
    }

  ssh_xfree(rl->display_line);
  rl->display_line = ssh_xcalloc(rl->line_alloc, sizeof(char));
  rl->display_line[0] = '\0';
  rl->end = rl->cursor = strlen(rl->line);

  ssh_rl_redraw_display(rl);

  rl->extended_mode = FALSE;
  rl->callback = callback;
  rl->ext_callback = ext_callback;
  rl->wanted_flags = flags;     /* Note what flags the client accepts */
  rl->flags = 0;                /* Zero the flags for this call */
  if (rl->ext_callback)
    rl->extended_mode = TRUE;

  rl->user_context = context;

  if (rl->tc->term_cursor_key_app_mode_on)
    ssh_rl_send_code(rl, rl->tc->term_cursor_key_app_mode_on);

  ssh_stream_set_callback(rl->stream, iocb, (void *)rl);
}

/* This must be called _once_ before using any of the asynchronous
   ssh_read[line,pass}_eloop*() functions. Returns a context argument
   to be given to these functions. Returns NULL in case of
   initialization error. */
SshReadLineCtx ssh_readline_eloop_initialize(SshStream stream)
{
  ReadLine rl;

  rl = ssh_xcalloc(1, sizeof(*rl));

  SSH_DEBUG(2, ("Initializing ReadLine..."));
  rl->eloop = TRUE;
  rl->echo = TRUE;
  /* When calling these, the stream must be a valid fdstream. */
  rl->read_fd = ssh_stream_fd_get_readfd(stream);
  rl->write_fd = ssh_stream_fd_get_writefd(stream);
  rl->mark = -1;
  rl->last_command_cut = 0;
  rl->yank = NULL;
  rl->keymap = SSH_READLINE_NORMAL_KEYMAP;
  rl->max_undo_depth = SSH_READLINE_MAX_UNDO_DEPTH;
  rl->undo = ssh_xcalloc(rl->max_undo_depth, sizeof(char *));
  rl->undo_cursors = ssh_xcalloc(rl->max_undo_depth, sizeof(int));
  rl->undo_cursors[0] = 0;
  rl->undo_length = 0;
  rl->undo_position = -1;
  rl->undo_direction = -1;
  rl->row_length = 80;

  rl->cmd_hist = ssh_adt_create_generic(SSH_ADT_LIST,
                                        SSH_ADT_DESTROY,
                                        ssh_adt_callback_destroy_free,
                                        SSH_ADT_ARGS_END);
  SSH_VERIFY(rl->cmd_hist != NULL);
  rl->ch = SSH_ADT_INVALID;
  rl->line_entered_in_cmd_hist = TRUE;

  ssh_rl_initialize_term_capabilities(rl);
#ifdef SSH_RL_ELOOP_SET_TTY_MODES_ONCE
  if (ssh_rl_set_tty_modes(rl) < 0)
    {
      ssh_xfree(rl->undo_cursors);
      ssh_xfree(rl->undo);
      ssh_xfree(rl);
      return NULL;
    }
#endif /* SSH_RL_ELOOP_SET_TTY_MODES_ONCE */

  rl->out_buffer = ssh_xbuffer_allocate();
  rl->in_buffer = ssh_xbuffer_allocate();
  rl->stream = stream;
  rl->timeout_done = FALSE;

  old_sigint_handler = signal(SIGINT, ssh_rl_sigint_handler);
  old_sigcont_handler = signal(SIGCONT, ssh_rl_sigcont_handler);
  global_rl = rl;

  ssh_stream_set_callback(rl->stream, NULL_FNPTR, NULL);

  return rl;
}

/* This should be used after you are finished with the "real"
   functions. This will free ``rl''. */
void ssh_readline_eloop_uninitialize(SshReadLineCtx rl)
{
  int i;

  SSH_DEBUG(2, ("Uninitializing ReadLine..."));
  for(i = 0; i < rl->undo_length; i++)
    ssh_xfree(rl->undo[i]);
  rl->undo_length = 0;
  ssh_xfree(rl->yank);
  ssh_xfree(rl->undo);
  ssh_xfree(rl->undo_cursors);
  if (rl->line)
    {
      rl->line[rl->end] = '\0';
      ssh_rl_redraw_display(rl);
    }

#ifdef SSH_RL_ELOOP_SET_TTY_MODES_ONCE
  rm = ssh_rl_restore_tty_modes(rl);
#endif /* SSH_RL_ELOOP_SET_TTY_MODES_ONCE */
  (void)signal(SIGINT, old_sigint_handler);
  (void)signal(SIGCONT, old_sigcont_handler);

  if (rl->cmd_hist)
    ssh_adt_destroy(rl->cmd_hist);

  ssh_buffer_free(rl->out_buffer);
  ssh_buffer_free(rl->in_buffer);
  ssh_xfree(rl->line);
  ssh_xfree(rl->display_line);
  /* We can free this here, as ssh_readline_eloop_uninitialize() is
     usually called at the time we really don't want it anymore. */
  ssh_xfree(rl->tc->term_buffer);
  ssh_xfree(rl->tc);
  ssh_xfree(rl);
}

/*
 * Read line from user. The tty at file descriptor ``write_fd'' is put
 * to raw mode and data is read from ``read_fd'' until CR is
 * received. The ``prompt'' is used to prompt the input. ``line'' is
 * pointer to char pointer and it should either contain NULL or the
 * mallocated string for previous value (that string is freed).  If
 * line can be successfully read the ``line'' argument contains the
 * new mallocated string.
 *
 * The ssh_readline will return the number of characters returned in
 * line buffer. If EOF or other error is noticed the return value is
 * -1.
 */
int ssh_readline(const char *prompt, char **line, int read_fd, int write_fd)
{
  ReadLine rl;
  int i;
  int lr, rm;
  const char *tmp, *nl, *cr;

  if (line == NULL)
    ssh_fatal("Fatal: ssh_readline LINE is NULL");

  rl = ssh_xcalloc(1, sizeof(*rl));

  rl->eloop = FALSE;
  rl->echo = TRUE;
  rl->read_fd = read_fd;
  rl->write_fd = write_fd;
  rl->mark = -1;
  rl->last_command_cut = 0;
  rl->yank = NULL;
  rl->keymap = SSH_READLINE_NORMAL_KEYMAP;
  rl->max_undo_depth = SSH_READLINE_MAX_UNDO_DEPTH;
  rl->undo = ssh_xcalloc(rl->max_undo_depth, sizeof(char *));
  rl->undo_cursors = ssh_xcalloc(rl->max_undo_depth, sizeof(int));
  rl->undo[0] = ssh_xstrdup("");
  rl->undo_cursors[0] = 0;
  rl->undo_length = 0;
  rl->undo_position = 0;
  rl->undo_direction = -1;
  rl->row_length = 80;
  rl->in_buffer = ssh_xbuffer_allocate();
  rl->timeout_done = FALSE;
  if (prompt == NULL)
    prompt = "";
  rl->prompt = prompt;
  tmp = strrchr(rl->prompt, '\n');
  if (tmp != NULL)
    rl->prompt = tmp + 1;
  tmp = strrchr(rl->prompt, '\r');
  if (tmp != NULL)
    rl->prompt = tmp + 1;
  rl->prompt_len = strlen(rl->prompt);

  ssh_rl_initialize_term_capabilities(rl);
  if (ssh_rl_set_tty_modes(rl) < 0)
    {
      ssh_xfree(rl->undo_cursors);
      ssh_xfree(rl->undo);
      ssh_xfree(rl);
      if (*line != NULL)
        ssh_xfree(*line);
      *line = NULL;
      return -1;
    }

  rl->cmd_hist = NULL;
  rl->ch = SSH_ADT_INVALID;

  rl->display_cursor = 0;
  tmp = prompt;
  while (1)
    {
      nl = strchr(tmp, '\n');
      cr = strchr(tmp, '\r');
      if (nl == NULL && cr == NULL)
        break;
      if (nl != NULL && cr != NULL)
        {
          if (nl < cr)
            cr = NULL;
          else
            nl = NULL;
        }
      if (nl != NULL)
        {
          ssh_rl_write_string(rl, tmp, nl - tmp);
          ssh_rl_send_string(rl, "\r\n", 2);
          rl->display_cursor = 0;
          tmp = nl + 1;
          if (*tmp == '\r')
            tmp++;
        }
      else
        {
          ssh_rl_write_string(rl, tmp, cr - tmp);
          if (*(cr + 1) == '\n')
            {
              ssh_rl_send_string(rl, "\r\n", 2);
              cr++;
            }
          else
            ssh_rl_send_string(rl, "\r", 1);
          rl->display_cursor = 0;
          tmp = cr + 1;
        }
    }
  ssh_rl_write_string(rl, tmp, rl->prompt_len);

  if (*line != NULL)
    {
      rl->line_alloc = strlen(*line) + 1;
      rl->line = *line;
    }
  else
    {
      rl->line_alloc = 80;
      rl->line = ssh_xmalloc(rl->line_alloc);
      rl->line[0] = '\0';
    }

  rl->display_line = ssh_xmalloc(rl->line_alloc);
  rl->display_line[0] = '\0';
  rl->end = rl->cursor = strlen(rl->line);

  if (rl->tc->term_cursor_key_app_mode_on)
    ssh_rl_send_code(rl, rl->tc->term_cursor_key_app_mode_on);

  lr = ssh_rl_loop(rl);

  if (rl->tc->term_cursor_key_app_mode_off)
    ssh_rl_send_code(rl, rl->tc->term_cursor_key_app_mode_off);

  rm = ssh_rl_restore_tty_modes(rl);

  if ((lr < 0) || (rm < 0))
    {
      for(i = 0; i < rl->undo_length; i++)
        ssh_xfree(rl->undo[i]);
      ssh_xfree(rl->yank);
      ssh_xfree(rl->undo);
      ssh_xfree(rl->undo_cursors);
      ssh_xfree(rl->line);
      ssh_xfree(rl);
      *line = NULL;
      return -1;
    }
  for(i = 0; i < rl->undo_length; i++)
    ssh_xfree(rl->undo[i]);
  ssh_xfree(rl->yank);
  ssh_xfree(rl->undo);
  ssh_xfree(rl->undo_cursors);
  ssh_buffer_free(rl->in_buffer);
  *line = rl->line;             /* Line is not freed, it is returned */
  rl->line[rl->end] = '\0';
  ssh_xfree(rl);
  return strlen(*line);
}
#endif /* VXWORKS */
