/*

sshmp-integer-misc.c

Authors: Mika Kojo     <mkojo@ssh.fi>
         Patrick Irwin <irwin@ssh.fi> 

Copyright (C) 1996-2001 SSH Communications Security Corp
All rights reserved.

Created: Mon Dec 3  12:44:32  2001 [irwin]

*/

/* This library implements the computations in the ring of integers,
   that is in the set

   Z = {..., -2, -1, 0, 1, 2, ...}.

   Namely, the ring operations and some convenience routines.

   The more arithmetical routines (i.e. routines that also utilize
   local mod p features) are implemented elsewhere.  

   This module together with sshmp-int-core.c comprise the integer 
   library. The routines here are not necessary for RSA and 
   Diffie-Hellman with predefined keys.
*/

#include "sshincludes.h"
#include "sshmp.h"
#include "sshgetput.h"

#define SSH_DEBUG_MODULE "SshMPIntegerMisc"

SshWord ssh_mprz_get_word(SshMPIntegerConst op, unsigned int i)
{
  if (i >= op->n)
    return 0;
  return op->v[i];
}

SshSignedWord ssh_mprz_get_si(SshMPIntegerConst op)
{
  SshSignedWord si;
  if (op->n == 0)
    return 0;
  /* Figure the bits that can be used. */
  si = (SshSignedWord)(op->v[0] & (SSH_WORD_MASK >> 1));
  if (SSH_MP_GET_SIGN(op))
    return -si;
  return si;
}

























































































void ssh_mprz_set_si(SshMPInteger op, SshSignedWord n)
{
  if (n == 0)
    {
      op->n = 0;
      SSH_MP_NO_SIGN(op);
      return;
    }

  /* Check that we have enough space. */
  if (!ssh_mprz_realloc(op, 1))
    return;

  if (n < 0)
    {
      SSH_MP_SET_SIGN(op);
      n = -n;
    }
  else
    SSH_MP_NO_SIGN(op);
  /* Set the integer. */
  op->v[0] = (SshWord)n;
  op->n = 1;
}

int ssh_mprz_init_set_str(SshMPInteger ret, const char *str, unsigned int base)
{
  ssh_mprz_init(ret);
  return ssh_mprz_set_str(ret, str, base);
}

void ssh_mprz_init_set_si(SshMPInteger ret, SshSignedWord s)
{
  ssh_mprz_init(ret);
  ssh_mprz_set_si(ret, s);
}

int ssh_mprz_cmp_si(SshMPIntegerConst op, SshSignedWord s)
{
  int rv;
  SshWord sw = 0L;
  
  if (ssh_mprz_isnan(op))
    return 1;
  
  if (SSH_MP_GET_SIGN(op) || (s < 0))
    {
      if (SSH_MP_GET_SIGN(op) && (s >= 0))
        return -1;
      if (!SSH_MP_GET_SIGN(op) && (s < 0))
        return 1;
      /* Make s positive. */
      if (s < 0)
        sw = (SshWord)(-s);
    }
  else
    sw = (SshWord)s;
  rv = ssh_mpk_cmp_ui(op->v, op->n, sw);
  if (SSH_MP_GET_SIGN(op) && (s < 0))
    rv = -rv;
  return rv;
}

/* Division routine. */

void ssh_mprz_divrem(SshMPInteger q, SshMPInteger r,
                     SshMPIntegerConst op1,
                     SshMPIntegerConst op2)
{
  SSH_MP_WORKSPACE_DEFINE;
  SshWord *rem, *quot, *div;
  unsigned int rem_n, quot_n, bits;

  if (ssh_mprz_nanresult2(q, op1, op2))
    return; 

  if (ssh_mprz_cmp_ui(op2, 0) == 0)
    {
      ssh_mprz_makenan(q, SSH_MP_NAN_EDIVZERO);
      ssh_mprz_makenan(r, SSH_MP_NAN_EDIVZERO);
      return;
    }

  /* Check sizes first. */
  if (op1->n < op2->n)
    {
      ssh_mprz_set(r, op1);
      ssh_mprz_set_ui(q, 0);
      return;
    }

  if (op1->n == op2->n)
    {
      if (ssh_mpk_cmp(op1->v, op1->n, op2->v, op2->n) < 0)
        {
          ssh_mprz_set(r, op1);
          ssh_mprz_set_ui(q, 0);
          return;
        }
    }

  rem_n = op1->n + 1;
  quot_n = op1->n - op2->n + 1;

  /* Do some reallocation. */
  if (!ssh_mprz_realloc(q, op1->n - op2->n + 1))
    {
      ssh_mprz_makenan(r, SSH_MP_NAN_ENOMEM);
      return;
    }
  if (!ssh_mprz_realloc(r, op2->n))
    {
      ssh_mprz_makenan(q, SSH_MP_NAN_ENOMEM);
      return;
    }

  /* Allocate temporary space. */
  SSH_MP_WORKSPACE_ALLOC(rem, (rem_n + quot_n + op2->n));
  if (!rem)
    {
      ssh_mprz_makenan(r, SSH_MP_NAN_ENOMEM);
      return;
    }
  quot = rem + rem_n;
  div  = quot + quot_n;

  /* Clear and copy. */
  ssh_mpk_memzero(quot, quot_n);
  ssh_mpk_memcopy(rem, op1->v, op1->n);
  rem[op1->n] = 0;

  /* Normalize, this can be optimized. XXX */
  ssh_mpk_memcopy(div, op2->v, op2->n);

  bits = ssh_mpk_leading_zeros(div, op2->n);
  ssh_mpk_shift_up_bits(div, op2->n, bits);
  ssh_mpk_shift_up_bits(rem, rem_n, bits);

  /* Certify the length. */
  if (rem[rem_n - 1] == 0)
    rem_n--;

  /* Do the division iteration. */
  if (!ssh_mpk_div(quot, quot_n, rem, rem_n, div, op2->n))
    {
      SSH_MP_WORKSPACE_FREE(rem);
      ssh_mprz_makenan(q, SSH_MP_NAN_EDIVZERO);
      return;
    }
  /* Quotient is immediately correct. However, remainder must be
     denormalized. */
  ssh_mpk_shift_down_bits(rem, op2->n, bits);

  /* Now set the quotient. */
  ssh_mpk_memcopy(q->v, quot, quot_n);
  q->n = quot_n;

  /* Set the remainder. */
  ssh_mpk_memcopy(r->v, rem, op2->n);
  r->n = op2->n;

  /* Figure out quotient sign. */
  SSH_MP_XOR_SIGNS(q, op1, op2);

  /* Check sizes. */
  while (q->n && q->v[q->n - 1] == 0)
    q->n--;

  while (r->n && r->v[r->n - 1] == 0)
    r->n--;

  /* Handle the sign of the remainder. */
  if (SSH_MP_GET_SIGN(op1))
    SSH_MP_SET_SIGN(r);
  else
    SSH_MP_NO_SIGN(r);

  /* Make sure that zeros are positive :) */
  if (r->n == 0)
    SSH_MP_NO_SIGN(r);
  if (q->n == 0)
    SSH_MP_NO_SIGN(q);

  /* Free temporary storage. */
  SSH_MP_WORKSPACE_FREE(rem);
}

void ssh_mprz_div(SshMPInteger ret_q,
                  SshMPIntegerConst op1, SshMPIntegerConst op2)
{
  SshMPIntegerStruct t;

  if (ssh_mprz_nanresult2(ret_q, op1, op2))
    return;

  ssh_mprz_init(&t);
  ssh_mprz_divrem(ret_q, &t, op1, op2);
  ssh_mprz_clear(&t);
}

/* XXX: NaN; there is no way to distinguish between zero division, NaN
   argument and divident being zero. */
SshWord ssh_mprz_mod_ui(SshMPIntegerConst op, SshWord u)
{
  SSH_MP_WORKSPACE_DEFINE;
  SshWord *norm, rem, t;
  unsigned int r;

  if (u == 0 || ssh_mprz_isnan(op))
    return 0;

  if (op->n == 0)
    return 0;

  /* Handle the normalization of 'u'. */
  t = u;
  r = 0;
  SSH_MPK_COUNT_LEADING_ZEROS(r, t);
  t <<= r;

  /* Allocate and normalize. */
  SSH_MP_WORKSPACE_ALLOC(norm, op->n + 1);
  if (!norm)
    {
      return 0;
    }
  ssh_mpk_memcopy(norm, op->v, op->n);
  norm[op->n] = 0;

  ssh_mpk_shift_up_bits(norm, op->n + 1, r);
  rem = ssh_mpk_mod_ui(norm, op->n + 1, t);

  SSH_MP_WORKSPACE_FREE(norm);

  /* Correct remainder. */
  rem >>= r;

  return rem;
}

/* GMP like interface to mod_ui. Just for compatibility. */
SshWord ssh_mprz_mod_ui2(SshMPInteger ret, SshMPIntegerConst op,
                         SshWord u)
{
  SshWord t;

  if (ssh_mprz_nanresult1(ret, op))
    return 0;

  t = ssh_mprz_mod_ui(op, u);
  ssh_mprz_set_ui(ret, t);
  return t;
}

/* Clear a bit at position 'bit'. */
void ssh_mprz_clr_bit(SshMPInteger op, unsigned int bit)
{
  unsigned int i;

  if (ssh_mprz_isnan(op))
    return;

  /* Find out the word offset. */
  i = bit / SSH_WORD_BITS;
  if (i >= op->n)
    return;
  /* Find out the bit offset. */
  bit %= SSH_WORD_BITS;
  op->v[i] &= (~((SshWord)1 << bit));
  /* Normalize. */
  while (op->n > 0 && op->v[op->n-1] == 0)
    op->n--;
}

/* Scan the integer starting from given position 'bitpos' for bit
   with value 'bitval'. Returns new bit position where the bitval
   differs from what was given. */
unsigned int ssh_mprz_scan_bit(SshMPIntegerConst op,
                               unsigned int bitpos,
                               unsigned int bitval)
{
  unsigned int bit = bitpos;

  while (ssh_mprz_get_bit(op, bit) == bitval)
    bit++;
  return bit;
}


/* Converting a stream of 8 bit octets to multiple precision integer. */

void ssh_buf_to_mp_lsb(SshMPInt x, const unsigned char *cp, size_t len)
{
  size_t i;
  unsigned long limb;

  ssh_mp_set_ui(x, 0);
  for (i = len - 3; i > 4; i -= 4)
    {
      limb = SSH_GET_32BIT(cp + i - 1);
      ssh_mp_mul_2exp(x, x, 32);
      ssh_mp_add_ui(x, x, limb);
    }
  for (; i > 0; i--)
    {
      ssh_mp_mul_2exp(x, x, 8);
      ssh_mp_add_ui(x, x, cp[i - 1]);
    }
}

#if 0
/* Operation of above functions is identical so use them. These functions
   might be used somewhere so we don't want to delete anything yet. */
void mp_linearize_msb_first(unsigned char *buf, unsigned int len,
                            SshMPInt value)
{
  ssh_mp_to_buf(buf, len, value);
}

void mp_unlinearize_msb_first(SshMPInt value, const unsigned char *buf,
                              unsigned int len)
{
  ssh_buf_to_mp(value, buf, len);
}
#endif



/* Remark. This is proprietary format for linearization of integers. This
   is efficient both in space and time. This does not encode the
   length of the integer itself. */
unsigned char *ssh_mprz_get_binary(size_t *bin_length, SshMPIntegerConst op)
{
  size_t l,k,lb,t,i;
  Boolean sign;
  unsigned char *bin;

  if (ssh_mprz_cmp_ui(op,0) == 0)
    {
      /* Create the zero object. */
      if ((bin = ssh_malloc(1)) == NULL)
        return NULL;
      bin[0] = 0;
      *bin_length = 1;
      return bin;
    }

  /* Get size of the absolute value. */
  l = ssh_mpk_size_in_bits(op->v, op->n);

  if (SSH_MP_GET_SIGN(op))
    sign = TRUE;
  else
    sign = FALSE;

  /* Compute the byte size. */
  k = (l + 7)/8;

  /* Count the number of bytes needed for the length. */
  for (lb = 1, t = k >> 6; t != 0; t >>= 7, lb++)
    ;

  /* Allocate the binary object. */
  if ((bin = ssh_malloc(k+lb)) == NULL)
    return NULL;

  /* Generate the first byte. */
  bin[0] =
    (sign == TRUE ? 0x80 : 0x00) |
    (lb > 1 ? 0x40 : 0x00) |
    ((k >> (7 * (lb - 1))) & 0x3f);
  /* Generate the following length bytes. */
  for (i = 1; i < lb; i++)
    {
      bin[i] =
        (lb - 1 > i ? 0x80 : 0x00) |
        ((k >> (7 * (lb - 1 - i))) & 0x7f);
    }
  /* Encode the absolute value. */
  ssh_mprz_get_buf(bin+lb, k, op);

  /* Return the created binary object. */
  *bin_length = k+lb;
  return bin;
}

size_t ssh_mprz_set_binary(SshMPInteger ret, const unsigned char *bin,
                           size_t bin_length)
{
  size_t l = 0L, i = 1L;
  Boolean sign;

  if (bin == NULL)
    return (size_t)-1;

  /* Handle the zero first. */
  if (bin_length == 0)
    return (size_t)-1;

  if (bin[0] & 0x80)
    sign = TRUE;
  else
    sign = FALSE;

  /* Determine the length. */
  l = bin[0] & 0x3f;

  /* Check for a zero. */
  if (l == 0 && (bin[0] & 0x40) == 0)
    {
      ssh_mprz_set_ui(ret, 0);
      return 1;
    }

  /* Check for an extension. */
  if (bin[0] & 0x40)
    {
      for (i = 1; i < bin_length; i++)
        {
          l = (l << 7) | (bin[i] & 0x7f);
          if (!(bin[i] & 0x80))
            {
              i++;
              break;
            }
        }
      if (i >= bin_length)
        return (size_t)-1;
    }

  /* Check if possible. */
  if (bin_length < l + i)
    return (size_t)-1;

  /* Do the decoding. */
  ssh_mprz_set_buf(ret, bin + i, l);
  if (sign)
    SSH_MP_SET_SIGN(ret);

  /* Return the read length. */
  return i + l;
}

/* Random number routines. */

/* 'Find' a random number of 'bits' bits. */
void ssh_mprz_rand(SshMPInteger op, unsigned int bits)
{
  unsigned int i, k;

  /* Compute the word and bit positions. */
  k = bits / SSH_WORD_BITS;
  bits %= SSH_WORD_BITS;

  if (!ssh_mprz_realloc(op, k + 1))
    return;

  /* Generate enough random bits. */
  for (i = 0; i < k + 1; i++)
    op->v[i] = ssh_rand();

  /* Don't do any shifting? */
  if (bits == 0)
    {
      op->n = k;
      while (op->n && op->v[op->n - 1] == 0)
        op->n--;
      SSH_MP_NO_SIGN(op);
      return;
    }

  /* Trivial shifting, masking... */
  op->v[k] = op->v[k] & (((SshWord)1 << bits) - 1);
  op->n = k + 1;

  while (op->n && op->v[op->n - 1] == 0)
    op->n--;
  SSH_MP_NO_SIGN(op);
}


/* Slow but allows one to build either very sparse (in 2-adic sense :) or
   very dense random numbers. */
void ssh_mprz_rand_w(SshMPInteger op, unsigned int bits, unsigned int weigth)
{
  unsigned int i, j;
  SshWord k, n0, n1;

  /* Clear the number before anything else. */
  ssh_mprz_set_ui(op, 0);

  for (i = op->n; i < op->m; i++)
    op->v[i] = 0;
  /*  XXX: ssh_mprz_clear_extra(op);*/
  
  /* We need some kind of an algorithm, which reasonably well builds
     sparse random numbers. They should be almost uniformly
     distributed. */

  /* One easy algorithm is to consider the probability that a bit is
     set, denoted by p = weigth/bits. However, we cannot work with
     floats so we have to do some tricks. */

  for (i = 0, j = 0; i < bits; i++)
    {
      /* Get a value from 0 <= k < 2^32. */
      k = ssh_rand();

      /* Convert k to comparable value. */

      SSH_MPK_LONG_MUL(n1, n0, k, (SshWord)bits);
      if (n1 <= (SshWord)weigth)
        ssh_mprz_set_bit(op, i);
    }
}

/* Slow, but so simple to write that I had to do it. */
void ssh_mprz_pow(SshMPInteger ret,
                  SshMPIntegerConst g, SshMPIntegerConst e)
{
  SshMPIntegerStruct temp;
  unsigned int bits, i;

  if (ssh_mprz_nanresult2(ret, g, e))
    return;

  /* Check the sign. */
  if (ssh_mprz_cmp_ui(e, 0) < 0)
    {
      ssh_mprz_makenan(ret, SSH_MP_NAN_ENEGPOWER);
      return;
    }

  /* Trivial cases. */
  if (ssh_mprz_cmp_ui(e, 0) == 0)
    {
      ssh_mprz_set_ui(ret, 1);
      return;
    }

  if (ssh_mprz_cmp_ui(e, 1) == 0)
    {
      ssh_mprz_set(ret, g);
      return;
    }

  ssh_mprz_init(&temp);
  ssh_mprz_set(&temp, g);

  /* Compute the size of the exponent. */
  bits = ssh_mpk_size_in_bits(e->v, e->n);

  for (i = bits - 1; i; i--)
    {
      ssh_mprz_square(&temp, &temp);
      if (ssh_mprz_get_bit(e, i - 1))
        ssh_mprz_mul(&temp, &temp, g);
    }

  ssh_mprz_set(ret, &temp);
  ssh_mprz_clear(&temp);
}

void ssh_mprz_pow_ui_exp(SshMPInteger ret, SshMPIntegerConst g, SshWord e)
{
  SshMPIntegerStruct temp;

  if (ssh_mprz_nanresult1(ret, g))
    return;

  /* Trivial cases. */
  if (e == 0)
    {
      ssh_mprz_set_ui(ret, 1);
      return;
    }

  if (e == 1)
    {
      ssh_mprz_set(ret, g);
      return;
    }

  ssh_mprz_init(&temp);
  ssh_mprz_set(&temp, g);
  ssh_mprz_set_ui(ret, 1);

  while (e)
    {
      if (e & 1)
        ssh_mprz_mul(ret, ret, &temp);
      e >>= 1; if (!e) break;
      ssh_mprz_square(&temp, &temp);
    }

  ssh_mprz_clear(&temp);
}

/* XXX Write optimized binary versions of the next two routines. That is,
   implement the binary gcd routines. They are reasonably simple, but
   I don't think they will be much faster thus it's easier to get
   along with these. */

/* Naive versions of these routines, which are fairly rarely used. */
void ssh_mprz_gcd(SshMPInteger d,
                  SshMPIntegerConst a, SshMPIntegerConst b)
{
  SshMPIntegerStruct a0, b0, r;

  if (ssh_mprz_nanresult2(d, a, b))
    return;

  ssh_mprz_init(&a0);
  ssh_mprz_init(&b0);
  ssh_mprz_init(&r);

  ssh_mprz_set(&a0, a);
  ssh_mprz_set(&b0, b);

  /* Standard gcd, however, we should implemented much faster ways also. */
  while (ssh_mprz_cmp_ui(&b0, 0) != 0)
    {
      ssh_mprz_mod(&r, &a0, &b0);
      ssh_mprz_set(&a0, &b0);
      ssh_mprz_set(&b0, &r);
    }

  ssh_mprz_set(d, &a0);

  ssh_mprz_clear(&a0);
  ssh_mprz_clear(&b0);
  ssh_mprz_clear(&r);
}

/* Compute (d, u, v) given (a, b) such that au + bv = d. */
void ssh_mprz_gcdext(SshMPInteger d, SshMPInteger u, SshMPInteger v,
                     SshMPIntegerConst a, SshMPIntegerConst b)
{
  SshMPIntegerStruct v1, v3, t1, t3, d0, u0, x;

  /* XXX: NaN */
  if (ssh_mprz_nanresult2(d, a, b))
    return;

  if (ssh_mprz_cmp_ui(b, 0) == 0)
    {
      ssh_mprz_set(d, a);
      ssh_mprz_set_ui(v, 0);
      ssh_mprz_set_ui(u, 1);
    }

  ssh_mprz_init(&v1);
  ssh_mprz_init(&v3);
  ssh_mprz_init(&t1);
  ssh_mprz_init(&t3);
  ssh_mprz_init(&u0);
  ssh_mprz_init(&d0);
  ssh_mprz_init(&x);

  ssh_mprz_set_ui(&u0, 1);
  ssh_mprz_set(&d0, a);
  ssh_mprz_set_ui(&v1, 0);
  ssh_mprz_set(&v3, b);

  /* Check for zero value using the internal size, which is bit ugly. */
  while (v3.n != 0)
    {
      /* Standard extended GCD algorithm inner loop. See for example
         Cohen's book. */
      ssh_mprz_divrem(&x, &t3, &d0, &v3);
      ssh_mprz_mul(&t1, &x, &v1);
      ssh_mprz_sub(&t1, &u0, &t1);
      ssh_mprz_set(&u0, &v1);
      ssh_mprz_set(&d0, &v3);
      ssh_mprz_set(&v1, &t1);
      ssh_mprz_set(&v3, &t3);
    }

  /* Compute v. */
  ssh_mprz_mul(&t1, a, &u0);
  ssh_mprz_sub(&t1, &d0, &t1);
  ssh_mprz_divrem(&v1, &v3, &t1, b);

  ssh_mprz_set(d, &d0);
  ssh_mprz_set(u, &u0);
  ssh_mprz_set(v, &v1);

  ssh_mprz_clear(&v1);
  ssh_mprz_clear(&v3);
  ssh_mprz_clear(&t1);
  ssh_mprz_clear(&t3);
  ssh_mprz_clear(&d0);
  ssh_mprz_clear(&u0);
  ssh_mprz_clear(&x);
}

/* Simple, but hopefully reasonably efficient. This is almost directly
   from Cohen's book. Improve if more speed is needed, one could open
   things up a bit, but this seems reasonably efficient. */
void ssh_mprz_sqrt(SshMPInteger sqrt_out, SshMPIntegerConst op)
{
  SshMPIntegerStruct x, y, r, t;
  int bits;

  if (ssh_mprz_nanresult1(sqrt_out, op))
    return;

  /* Check impossible cases. */
  if (ssh_mprz_cmp_ui(op, 0) <= 0)
    {
      /* Should we terminate? Perhaps we return the integer part of this
         operation. */
      ssh_mprz_set_ui(sqrt_out, 0);
      return;
    }

  ssh_mprz_init(&x);
  ssh_mprz_init(&y);
  ssh_mprz_init(&r);
  ssh_mprz_init(&t);

  /* Find a nice estimate for n. */
  bits = ssh_mpk_size_in_bits(op->v, op->n);

  /* This should be fairly correct estimate. */
  ssh_mprz_set_bit(&x, (bits + 2)/2);

  /* Loop until a nice value found. */
  while (1)
    {
      /* Compute the newtonian step. */
      ssh_mprz_divrem(&t, &r, op, &x);
      ssh_mprz_add(&t, &t, &x);
      ssh_mprz_div_2exp(&y, &t, 1);

      if (ssh_mprz_cmp(&y, &x) < 0)
        ssh_mprz_set(&x, &y);
      else
        break;
    }

  /* Finished. */
  ssh_mprz_set(sqrt_out, &x);

  ssh_mprz_clear(&x);
  ssh_mprz_clear(&y);
  ssh_mprz_clear(&r);
  ssh_mprz_clear(&t);
}

/* Basic bit operations, for integers. These are simple, but useful
   sometimes. */
void ssh_mprz_and(SshMPInteger ret,
                  SshMPIntegerConst op1, SshMPIntegerConst op2)
{
  unsigned int i;

  if (ssh_mprz_nanresult2(ret, op1, op2))
    return;

  /* Swap. */
  if (op1->n > op2->n)
    {
      SshMPIntegerConst t;
      t = op1;
      op1 = op2;
      op2 = t;
    }

  /* Reallocate. */
  if (!ssh_mprz_realloc(ret, op1->n))
    return;

  /* This can be written more optimally. */
  for (i = 0; i < op1->n; i++)
    ret->v[i] = op1->v[i] & op2->v[i];

  ret->n = op1->n;
  while (ret->n && ret->v[ret->n - 1] == 0)
    ret->n--;
  SSH_MP_NO_SIGN(ret);
}

void ssh_mprz_or(SshMPInteger ret,
                 SshMPIntegerConst op1, SshMPIntegerConst op2)
{
  unsigned int i;

  if (ssh_mprz_nanresult2(ret, op1, op2))
    return;

  /* Swap. */
  if (op1->n > op2->n)
    {
      SshMPIntegerConst t;
      t = op1;
      op1 = op2;
      op2 = t;
    }

  /* Reallocate. */
  if (!ssh_mprz_realloc(ret, op2->n))
    return;

  /* This can be written more optimally. */
  for (i = 0; i < op1->n; i++)
    ret->v[i] = op1->v[i] | op2->v[i];
  for (; i < op2->n; i++)
    ret->v[i] = op2->v[i];

  ret->n = op2->n;
  while (ret->n && ret->v[ret->n - 1] == 0)
    ret->n--;
  SSH_MP_NO_SIGN(ret);
}

void ssh_mprz_xor(SshMPInteger ret,
                  SshMPIntegerConst op1, SshMPIntegerConst op2)
{
  unsigned int i;

  if (ssh_mprz_nanresult2(ret, op1, op2))
    return;

  /* Swap. */
  if (op1->n > op2->n)
    {
      SshMPIntegerConst t;
      t = op1;
      op1 = op2;
      op2 = t;
    }

  /* Reallocate. */
  if (!ssh_mprz_realloc(ret, op1->n))
    return;

  /* This can be written more optimally. */
  for (i = 0; i < op1->n; i++)
    ret->v[i] = op1->v[i] ^ op2->v[i];
  for (; i < op2->n; i++)
    ret->v[i] = op2->v[i];

  ret->n = op2->n;
  while (ret->n && ret->v[ret->n - 1] == 0)
    ret->n--;
  SSH_MP_NO_SIGN(ret);
}

void ssh_mprz_com(SshMPInteger ret, SshMPIntegerConst op)
{
  unsigned int i;

  if (ssh_mprz_nanresult1(ret, op))
    return;

  /* Reallocate. */
  if (!ssh_mprz_realloc(ret, op->n))
    return;

  /* This can be written more optimally. */
  for (i = 0; i < op->n; i++)
    ret->v[i] = ~op->v[i];

  ret->n = op->n;
  while (ret->n && ret->v[ret->n - 1] == 0)
    ret->n--;
  SSH_MP_NO_SIGN(ret);
}

/* Encode given SshMPInteger in SSH2 style.  return length of the
   buffer (which this allocates) presenting the number, or zero in
   case of failure. */
int
ssh_mprz_encode_ssh2style(SshMPIntegerConst mp,
                          unsigned char *buf, size_t len)
{
  SshMPIntStruct temp;
  unsigned int i;
  unsigned char *four = buf;
  size_t buf_len;

  /* This code is written along the lines of the code in ber.c */
  switch (ssh_mprz_cmp_ui(mp, 0))
    {
    case 0: /* Handle the zero case. */

      if (len >= 4)
        {
          four = buf;
          four[0] = four[1] = four[2] = four[3] = 0;
        }
      return 4;

    case 1: /* Handle the positive case. */

      buf_len = ssh_mp_get_size(mp, 2);
      /* If highest bit set add one empty octet, then correct octet count. */
      if ((buf_len & 7) == 0)
        buf_len += 8;
      buf_len = (buf_len + 7)/8;

      if ((4 + buf_len) > len)
        return 4 + buf_len;

      /* Put the length and integer value. */
      SSH_PUT_32BIT(buf, buf_len);
      ssh_mprz_get_buf(buf+4, buf_len, mp);
      return 4 + buf_len;

    case -1: /* Handle negative case. */

      ssh_mp_init(&temp);
      /* Compute temp = (-value - 1) = -(value + 1). E.g. -1 -> 0, which
         then can be complemented. */
      ssh_mp_set_ui(&temp, 0);
      ssh_mp_sub(&temp, &temp, mp);
      ssh_mp_sub_ui(&temp, &temp, 1);
      /* Compute the correct length in base 2. */
      buf_len = ssh_mp_get_size(&temp, 2);

      /* Check the highest bit case. Note that here we actually want the
         highest bit be set (after complementing). */
      if ((buf_len & 7) == 0)
        buf_len += 8;
      buf_len = (buf_len + 7)/8;

      if ((buf_len + 4) > len)
        {
          ssh_mp_clear(&temp);
          return buf_len + 4;
        }

      SSH_PUT_32BIT(buf, buf_len);
      ssh_mprz_get_buf(buf+4, buf_len, mp);

      /* Doing the complementing. Currently the ssh_mp_to_buf doesn't
         know how to do it. */
      for (i = 0; i < buf_len; i++)
        buf[i + 4] ^= 0xff;
      return buf_len + 4;

    default:
      return 0;
    }
}

int
ssh_mprz_decode_ssh2style(const unsigned char *buf, size_t len,
                          SshMPInteger mp)
{
  size_t byte_size;
  const unsigned char *bufptr;
  int i;

  if (len < 4)
    return 0;
  byte_size = SSH_GET_32BIT(buf);

  if (byte_size == 0)
    {
      ssh_mprz_set_ui(mp, 0);
      return 4;
    }

  if ((byte_size + 4) > len)
    return 0;

  bufptr = buf + 4;
  if (bufptr[0] & 0x80)
    {
      unsigned char *tmp;

      if ((tmp = ssh_memdup(bufptr, byte_size)) != NULL)
        {
          for (i = 0; i < byte_size; i++)
            tmp[i] ^= 0xff;

          ssh_mprz_set_buf(mp, tmp, byte_size);
          ssh_mprz_add_ui(mp, mp, 1);
          ssh_mprz_neg(mp, mp);
          ssh_free(tmp);
        }
      else
        return 0;
    }
  else
    {
      ssh_mprz_set_buf(mp, bufptr, byte_size);
    }
  return byte_size + 4;
}

/* sshmp-integer-misc.c */
