/*

sshfilexfer.h

Authors: Tatu Ylonen   <ylo@ssh.com>
         Timo Rinne    <tri@ssh.com>
         Sami Lehtinen <sjl@ssh.com>

Copyright (c) 1998,2001,2002 SSH Communications Security, Finland
                             All rights reserved

This interface implements a simple file sharing protocol across a
SshStream channel.  It can be used in implementing file transfer, file
sharing, or other data copying over any stream.

*/

#ifndef SSHFILEXFER_H
#define SSHFILEXFER_H

#include "sshstream.h"
#include "sshoperation.h"

#ifdef WIN32
#define off_t SshUInt64
#endif /* WIN32 */

/* Data type for the file transfer client. */
typedef struct SshFileClientRec *SshFileClient;

/* Attribute flags.  These are used to specify which attributes are
   present. */
#define SSH_FILEXFER_ATTR_SIZE          0x00000001
#define SSH_FILEXFER_ATTR_UIDGID        0x00000002
#define SSH_FILEXFER_ATTR_PERMISSIONS   0x00000004
#define SSH_FILEXFER_ATTR_ACMODTIME     0x00000008
#define SSH_FILEXFER_ATTR_EXTENDED      0x80000000

/* Data structure for representing file attributes. */

typedef struct SshFileAttributesRec {
  /* Flags that specify which attributes are present. */
  unsigned int flags;

  /* Size of the file. */
  off_t size;

  /* User id owning the file. */
  uid_t uid;

  /* Group id owning the file. */
  gid_t gid;

  /* Time of last modification. */
  SshTime mtime;

  /* Time of last access. */
  SshTime atime;
  
  /* Permissions of the file.  This is in unix format. */
  unsigned long permissions;
} *SshFileAttributes, SshFileAttributesStruct;

/* Data type for a file handle. */
typedef struct SshFileHandleRec *SshFileHandle;

/* Data type for status returns by various functions in this module.
   These values are typically passed to the callback when the operation
   completes. */
typedef enum {
  /* The operation completed successfully. */
  SSH_FX_OK = 0,

  /* The operation failed because of trying to read at end of file. */
  SSH_FX_EOF,
  
  /* The requested file does not exist. */
  SSH_FX_NO_SUCH_FILE,

  /* Insufficient privileges to perform the operation. */
  SSH_FX_PERMISSION_DENIED,

  /* The requested operation failed for some other reason. */
  SSH_FX_FAILURE,
  
  /* A badly formatted message was received.  This indicates an error or
     incompatibility in the protocol implementation. */
  SSH_FX_BAD_MESSAGE,

  /* Connection has not been established (yet) */
  SSH_FX_NO_CONNECTION,
  
  /* Connection to the server was lost, and the operation could not be
     performed. */
  SSH_FX_CONNECTION_LOST,

  /* Operation is unsupported by the fileserver. This is actually
     checked by the client, and the offending call is never sent to the
     server. The server may, however, send this if it doesn't support an
     EXTENDED request, or the platform hasn't got some feature (even
     though a specific message would be supported by the protocol
     version. */
  SSH_FX_OP_UNSUPPORTED,

  /* Internal (not defined in the protocol) error messages. */
  /* Requested operation failed because memory allocation failed in
     the client library. */
  SSH_FX_OUT_OF_MEMORY = 129

} SshFileClientError;

/***********************************************************************
 * Common functions for client and server
 ***********************************************************************/

/* Duplicate a SshFileAttributes-structure.
 */
SshFileAttributes ssh_file_attributes_dup(SshFileAttributes attributes);

/***********************************************************************
 * Client-side functions
 ***********************************************************************/

typedef void (*SshFileConnCompleteCallback)(void *context);

/* All of the following callbacks have a ``error_msg'' and ``lang_tag''
   field, which are used in returning a human-readable error message
   from the server to the client. Note that these will be set to NULL,
   if the server doesn't provide these with the SSH_FXP_STATUS message
   (i.e. the server is older than version 3). Also, they will be NULL if
   no error occurred (except with messages expecting SSH_FXP_STATUS on
   success).*/
/* Callback function for returning only the status of the command. */
typedef void (*SshFileStatusCallback)(SshFileClientError error,
                                      const char *error_msg,
                                      const char *lang_tag,
                                      void *context);

/* Callback function for returning file handles. */
typedef void (*SshFileHandleCallback)(SshFileClientError error,
                                      SshFileHandle handle,
                                      const char *error_msg,
                                      const char *lang_tag,
                                      void *context);

/* Callback function for returning data. */
typedef void (*SshFileDataCallback)(SshFileClientError error,
                                    const unsigned char *data,
                                    size_t len,
                                    const char *error_msg,
                                    const char *lang_tag,
                                    void *context);

/*
 * Callback function for returning file names. 
 * 
 * long_name      "long" description of the file.. interpretation
 *                is host specific. 
 *
 * All returned values should be copied is to be used outside the
 * callback, as they are freed after this callback returns.
 */
typedef void (*SshFileNameCallback)(SshFileClientError error,
                                    const char *name,
                                    const char *long_name,
                                    SshFileAttributes attrs,
                                    const char *error_msg,
                                    const char *lang_tag,
                                    void *context);

/* Callback function for returning file attributes. The returned
   attributes should be copied if they are to be used outside the
   callback, as they are freed after this callback returns. */
typedef void (*SshFileAttributeCallback)(SshFileClientError error,
                                         SshFileAttributes attributes,
                                         const char *error_msg,
                                         const char *lang_tag,
                                         void *context);

/* This function wraps a communications channel into a file transfer
   client.  This takes over the stream, and it should no longer be used
   directly.  This returns an object that represents the file transfer
   client.  `completion_cb' (if not NULL_FNPTR) is called, when the
   server has sent its version, after which real requests can be
   started. If application does not want to wait for this to send
   requests, it must handle SSH_FX_NO_CONNECTION properly (which is
   returned for all requests before we have received server
   version). After the callback has been called, it is guaranteed that
   SSH_FX_NO_CONNECTION will not be returned.
*/
SshFileClient ssh_file_client_wrap(SshStream stream,
                                   SshFileConnCompleteCallback completion_cb,
                                   void *context);

/* Closes the file transfer client.  Any outstanding requests are silently
   terminated without calling their callbacks. */
void ssh_file_client_destroy(SshFileClient client);

/* Sends a request to open a file, and calls the given callback when
   complete.  The callback will be called either during this call or
   any time later.  Attributes may be NULL to use default values. */
SshOperationHandle ssh_file_client_open(SshFileClient client,
                                        const char *name,
                                        unsigned int flags,
                                        SshFileAttributes attributes,
                                        SshFileHandleCallback callback,
                                        void *context);

/* Sends a read request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_read(SshFileHandle handle,
                                        off_t offset,
                                        size_t len,
                                        SshFileDataCallback callback,
                                        void *context);

/* Sends a write request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_write(SshFileHandle handle,
                                         off_t offset,
                                         const unsigned char *buf,
                                         size_t len,
                                         SshFileStatusCallback callback,
                                         void *context);

/* Sends a close request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_close(SshFileHandle handle,
                                         SshFileStatusCallback callback,
                                         void *context);

/* Sends a stat request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_stat(SshFileClient client,
                                        const char *name,
                                        SshFileAttributeCallback callback,
                                        void *context);

/* Sends a lstat request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_lstat(SshFileClient client,
                                         const char *name,
                                         SshFileAttributeCallback callback,
                                         void *context);

/* Sends an fstat request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later. */
SshOperationHandle ssh_file_client_fstat(SshFileHandle handle,
                                         SshFileAttributeCallback callback,
                                         void *context);

/* Sends a setstat request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later.
   Setstat requests can be used to implement e.g. chown and chmod. */
SshOperationHandle ssh_file_client_setstat(SshFileClient client,
                                           const char *name,
                                           SshFileAttributes attributes,
                                           SshFileStatusCallback callback,
                                           void *context);

/* Sends an fsetstat request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later.
   Fsetstat requests can be used to implement e.g. fchown and fchmod. */
SshOperationHandle ssh_file_client_fsetstat(SshFileHandle handle,
                                            SshFileAttributes attributes,
                                            SshFileStatusCallback callback,
                                            void *context);

/* Sends an opendir request, and calls the given callback when complete.  The
   callback will be called either during this call or any time later.
   The path should point to a directory.  An empty string refers to the
   current directory. */
SshOperationHandle ssh_file_client_opendir(SshFileClient client,
                                           const char *name,
                                           SshFileHandleCallback callback,
                                           void *context);

/* Sends a readdir request, and calls the given callback when
   complete.  The callback will be called either during this call or
   any time later.  This returns one name at a time.  Only the last
   component of the name is returned (i.e., the name stored in the
   directory). When all files of directory have been fetched, returns
   NULL. Also returns NULL on error. */
SshOperationHandle ssh_file_client_readdir(SshFileHandle handle,
                                           SshFileNameCallback callback,
                                           void *context);

/* Sends a request to remove the given file.  This cannot be used to
   remove directories.  The callback will be called either during this call
   or any time later. */
SshOperationHandle ssh_file_client_remove(SshFileClient client,
                                          const char *name,
                                          SshFileStatusCallback callback,
                                          void *context);

/* Sends a requst to create the named directory, and calls the given callback
   when complete.  The callback will be called either during this call
   or any time later.  Attrs may be NULL to use default values. */
SshOperationHandle ssh_file_client_mkdir(SshFileClient client,
                                         const char *name,
                                         SshFileAttributes attrs,
                                         SshFileStatusCallback callback,
                                         void *context);

/* Sends a request to remove the given directory.  This cannot be used to
   remove normal files.  The callback will be called either during this
   call or any time later. */
SshOperationHandle ssh_file_client_rmdir(SshFileClient client,
                                         const char *name,
                                         SshFileStatusCallback callback,
                                         void *context);

/* Sends a request to rename the given file ('oldpath') to a new name
   ('newpath'). These can point to different directories, but must be
   on same filesystem (the standard rename() system call is performed
   in the server). The callback will be called either during this call
   or any time later. NOTE: If a file is renamed on top of existing
   file (not directory), the old file will (depending on system)
   probably be overwritten, so checks should be performed by the
   client. NOTE2: This call will only work for file servers with
   protocol version of 2 or more (ssh-2.2.0 and above). (this call
   will return SSH_FX_OP_UNSUPPORTED if server is too old.)  */
SshOperationHandle ssh_file_client_rename(SshFileClient client,
                                          const char *oldpath,
                                          const char *newpath,
                                          SshFileStatusCallback callback,
                                          void *context);

/* Sends a request to get the value of the symbolic link specified with
   ``path'' (standard readlink() is performed on the server, usually).
   NOTE: works only against fileservers with protocol version of 3 or
   more (ssh-3.2.0 and above).
*/
SshOperationHandle ssh_file_client_readlink(SshFileClient client,
                                            const char *path,
                                            SshFileNameCallback callback,
                                            void *context);

/* Sends a request to create a symbolic link ``linkpath'' with a value
   of ``targetpath'' (standard symlink() is performed on the server,
   usually).  NOTE: works only against fileservers with protocol version
   of 3 or more (ssh-3.2.0 and above).
*/
SshOperationHandle ssh_file_client_symlink(SshFileClient client,
                                           const char *linkpath,
                                           const char *targetpath,
                                           SshFileStatusCallback callback,
                                           void *context);

SshOperationHandle ssh_file_client_hash(SshFileHandle handle,
                                        SshUInt64 offset,
                                        SshUInt32 len,
                                        const char *hash_list,
                                        SshFileDataCallback callback,
                                        void *context);

/* Asks the server side to resolve a path */
SshOperationHandle ssh_file_client_realpath(SshFileClient client,
                                            const char *path,
                                            SshFileNameCallback callback,
                                            void *context);

/* Get the negotiated version number. This can be used only after the
   connection has been established. */
SshUInt32 ssh_file_client_get_version(SshFileClient client);

/* Query for the data of server extension ``ext_name''. Returns TRUE if
   the extension was found, FALSE otherwise. If extension was not
   defined, ``ext_data'' and ``ext_data_len'' are not touched. Note that
   you must not modify ``ext_data'' in any way. ``ext_data'' will remain
   valid until ``client'' is destroyed. Naturally, this will only return
   useful values after the connection to the server is completed. */
Boolean ssh_file_client_get_extension_data(SshFileClient client,
                                           const char *ext_name,
                                           unsigned char **ext_data,
                                           size_t *ext_data_len);

/***********************************************************************
 * Server-side functions
 ***********************************************************************/

/* Data type representing the server object. */
typedef struct SshFileServerRec *SshFileServer;

/* Wraps the given communications channel into a file transfer server.
   The server is automatically destroyed when the connection is closed. */
SshFileServer ssh_file_server_wrap(SshStream stream,
                                   int log_facility);
















/* Internal definitions */

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

#endif /* SSHFILEXFER_H */
