/*

  gentest.c

  Author: Mika Kojo <mkojo@ssh.fi>

  Copyright (C) 1996, 2000, 2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Created: Fri Nov  1 05:37:55 1996 [mkojo]

  Testing those gen- prefixed files.

  */
#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshtimemeasure.h"
#include "readfile.h"
#include "sshmp.h"
#include "sshdsprintf.h"

/********************** Random number tests. **********************/

/* Bit frequency tests. */

unsigned int rnd_bytes[256];
unsigned int rnd_freq[8][8];
unsigned int rnd_bits[8];

void rnd_set_freq(int i)
{
  rnd_bits[i]++;
  if (rnd_freq[i][0])
    {
      if (rnd_freq[i][0] < 7)
        rnd_freq[i][rnd_freq[i][0]]++;
      else
        rnd_freq[i][7]++;
      rnd_freq[i][0] = 0;
    }
}

void rnd_add_freq(void)
{
  int i;

  for (i = 0; i < 8; i++)
    rnd_freq[i][0]++;
}

void rnd_test_bits(void)
{
  int i, hi, lo, average, j, byte, error = 0;
  double av;

  printf("Running random number bit tests...\n");

  for (i = 0; i < 8; i++)
    {
      rnd_bits[i] = 0;
      for (j = 0; j < 8; j++)
        rnd_freq[i][j] = 0;
    }
  for (j = 0; j < 256; j++)
    rnd_bytes[j] = 0;

  for (i = 0; i < 1000000; i++)
    {
      if ((i & 0xffff) == 0)
        {
          printf(".");
          fflush(stdout);
        }

      byte = ssh_random_get_byte() & 0xff;

      rnd_bytes[byte]++;

      if (byte & 128)
        rnd_set_freq(7);
      if (byte & 64)
        rnd_set_freq(6);
      if (byte & 32)
        rnd_set_freq(5);
      if (byte & 16)
        rnd_set_freq(4);
      if (byte & 8)
        rnd_set_freq(3);
      if (byte & 4)
        rnd_set_freq(2);
      if (byte & 2)
        rnd_set_freq(1);
      if (byte & 1)
        rnd_set_freq(0);

      rnd_add_freq();
    }

  printf("\nRandom number generation validation suggests: \n");

  for (j = 0, hi = 0, lo = i, average = 0; j < 256; j++)
    {
      if (rnd_bytes[j] < lo)
        lo = rnd_bytes[j];
      if (rnd_bytes[j] > hi)
        hi = rnd_bytes[j];
    }

  if (hi > 5000 || lo < 3000)
    {
      printf("\nNote: byte distribution is off the set limits.\n");
      error++;
    }

  printf("Plain byte distribution: %d tries: %d highest, %d lowest.\n",
         i, hi, lo);

  printf("Single bit distributions, and counts in time.\n");

  for (j = 0; j < 8; j++)
    {
      av = ((double)rnd_bits[j]) / (double)i;

      printf("bit %d av. %f  %5d %5d %5d %5d %5d %5d . %5d\n", j,
             av,
             rnd_freq[j][1], rnd_freq[j][2], rnd_freq[j][3], rnd_freq[j][4],
             rnd_freq[j][5], rnd_freq[j][6], rnd_freq[j][7]);

      /* Simple checks for too good results. */
      if (av == 0.5 ||
          (rnd_freq[j][1] == 250000 || rnd_freq[j][2] == 125000 ||
          rnd_freq[j][3] == 62500  || rnd_freq[j][4] == 31250 ||
          rnd_freq[j][5] == 15625))
        {
          printf("\n Note: bit distributions are too good. "
                 "Please check these results.\n");
        }

      /* Checks for too poor results. */

      if (av < 0.30 || av > 0.70)
        {
          printf("\n Note: average bit distribution is off"
                 " the set limits.\n");
          error++;
        }

      if ((rnd_freq[j][1] < 200000 || rnd_freq[j][1] > 290000) ||
          (rnd_freq[j][2] < 100000 || rnd_freq[j][2] > 150000) ||
          (rnd_freq[j][3] <  50000 || rnd_freq[j][3] >  70000) ||
          (rnd_freq[j][4] <  25000 || rnd_freq[j][4] >  35000) ||
          (rnd_freq[j][5] <  12000 || rnd_freq[j][5] >  18000) ||
          (rnd_freq[j][6] <   6000 || rnd_freq[j][6] >   10000) ||
          (rnd_freq[j][7] <   6000 || rnd_freq[j][7] >   10000))
        {
          printf("\n Note: bit distributions in time are "
                 "off the set limits.\n");
          error++;
        }
    }

  printf("\n");


  if (error)
    {
      printf("\nATTENTION: Simple byte and bit tests have shown that \n"
             "this random number generation is performing \"badly\".\n"
             "Please run again few times to see if this error will occur\n"
             "again.\n"
             "If this error occurs often please contact SSH.\n");
      ssh_fatal("Warning: possible failure in random number generation.");
    }

}

/* Missing sequence tests. */

void rnd_test_missing_bits(void)
{
  unsigned char *bit_table;
  unsigned int i, l, m;
  double average;
  int error = 0;

#define SET_BIT(pos) (bit_table[pos / 8] |= (1 << (pos & 0x7)))
#define GET_BIT(pos) (bit_table[pos / 8] & (1 << (pos & 0x7)))

  printf("Following set of tests give the number of missing sequences\n"
         "after n iterations.\n");

  /* Sequence of 2 */

  /* Use 8 bit characters. */

#define SIZE 256*256/8
#define LOOPS SIZE * 16

  printf("Running test for sequences of 2 (table size %d bytes).\n", SIZE);

  bit_table = ssh_xmalloc(SIZE);
  memset(bit_table, 0, SIZE);

  for (i = 0; i < LOOPS; i++)
    {
      if (i > 0 && (i & 0xfff) == 0)
        {
          printf(".");
          fflush(stdout);
        }
      l = ssh_random_get_byte() * 256 + ssh_random_get_byte();
      SET_BIT(l);
    }

  printf("\nChecking missing sequences...\n");

  for (i = 0, m = 0; i < SIZE * 8; i++)
    {
      if (!GET_BIT(i))
        m++;
    }

  average = ((double)m / (SIZE*8));
  if (average < 0.11 || average > 0.15)
    {
      printf("\n NOTE: Possible error detected.\n");
      error++;
    }

  printf("After %d runs: \n"
         "%d of %d missing (average %f after %d full iterations).\n",
         LOOPS, m, SIZE * 8,
         average, LOOPS/(SIZE*8));

  ssh_xfree(bit_table);
#undef SIZE
#undef LOOPS

  /* Sequence of 3 */

  /* Use 7 bit characters. */

#define SIZE 128*128*128/8
#define LOOPS SIZE * 16

  printf("Running test for sequences of 3 (table size %d bytes).\n", SIZE);

  bit_table = ssh_xmalloc(SIZE);
  memset(bit_table, 0, SIZE);

  for (i = 0; i < LOOPS; i++)
    {
      if (i > 0 && (i & 0x1ffff) == 0)
        {
          printf(".");
          fflush(stdout);
        }
      l = (ssh_random_get_byte() >> 1) * (128*128) +
        (ssh_random_get_byte() >> 1) * 128 +
        (ssh_random_get_byte() >> 1);
      SET_BIT(l);
    }

  printf("\nChecking missing sequences...\n");

  for (i = 0, m = 0; i < SIZE * 8; i++)
    {
      if (!GET_BIT(i))
        m++;
    }

  average = ((double)m / (SIZE*8));
  if (average < 0.11 || average > 0.15)
    {
      printf("\n NOTE: Possible error detected.\n");
      error++;
    }

  printf("After %d runs: \n"
         "%d of %d missing (average %f after %d full iterations).\n",
         LOOPS, m, SIZE * 8,
         average, LOOPS/(SIZE*8));

  ssh_xfree(bit_table);
#undef SIZE
#undef LOOPS

  /* Sequence of 4 */

  /* Use 5 bit characters. */

#define SIZE 32*32*32*32/8
#define LOOPS SIZE * 16

  printf("Running test for sequences of 4 (table size %d bytes).\n", SIZE);

  bit_table = ssh_xmalloc(SIZE);
  memset(bit_table, 0, SIZE);

  for (i = 0; i < LOOPS; i++)
    {
      if (i > 0 && (i & 0xffff) == 0)
        {
          printf(".");
          fflush(stdout);
        }
      l = (ssh_random_get_byte() >> 3) * (32*32*32) +
        (ssh_random_get_byte() >> 3) * (32*32) +
        (ssh_random_get_byte() >> 3) * 32 +
        (ssh_random_get_byte() >> 3);
      SET_BIT(l);
    }

  printf("\nChecking missing sequences...\n");

  for (i = 0, m = 0; i < SIZE * 8; i++)
    {
      if (!GET_BIT(i))
        m++;
    }

  average = ((double)m / (SIZE*8));
  if (average < 0.11 || average > 0.15)
    {
      printf("\n NOTE: Possible error detected.\n");
      error++;
    }

  printf("After %d runs: \n"
         "%d of %d missing (average %f after %d full iterations).\n",
         LOOPS, m, SIZE * 8,
         average, LOOPS/(SIZE*8));

  ssh_xfree(bit_table);
#undef SIZE
#undef LOOPS

  /* Sequence of 5 */

  /* Use 4 bit characters. */

#define SIZE 16*16*16*16*16/8
#define LOOPS SIZE * 16

  printf("Running test for sequences of 5 (table size %d bytes).\n", SIZE);

  bit_table = ssh_xmalloc(SIZE);
  memset(bit_table, 0, SIZE);

  for (i = 0; i < LOOPS; i++)
    {
      if (i > 0 && (i & 0xffff) == 0)
        {
          printf(".");
          fflush(stdout);
        }
      l = (ssh_random_get_byte() >> 4) * (16*16*16*16) +
        (ssh_random_get_byte() >> 4) * (16*16*16) +
        (ssh_random_get_byte() >> 4) * (16*16) +
        (ssh_random_get_byte() >> 4) * 16 +
        (ssh_random_get_byte() >> 4);
      SET_BIT(l);
    }

  printf("\nChecking missing sequences...\n");

  for (i = 0, m = 0; i < SIZE * 8; i++)
    {
      if (!GET_BIT(i))
        m++;
    }

  average = ((double)m / (SIZE*8));
  if (average < 0.11 || average > 0.15)
    {
      printf("\n NOTE: Possible error detected.\n");
      error++;
    }

  printf("After %d runs: \n"
         "%d of %d missing (average %f after %d full iterations).\n",
         LOOPS, m, SIZE * 8,
         average, LOOPS/(SIZE*8));

  ssh_xfree(bit_table);
#undef SIZE
#undef LOOPS

  /* Sequence of 5 */

  /* Use 3 bit characters. */

#define SIZE 8*8*8*8*8*8/8
#define LOOPS SIZE * 16

  printf("Running test for sequences of 6 (table size %d bytes).\n", SIZE);

  bit_table = ssh_xmalloc(SIZE);
  memset(bit_table, 0, SIZE);

  for (i = 0; i < LOOPS; i++)
    {
      if (i > 0 && (i & 0x3fff) == 0)
        {
          printf(".");
          fflush(stdout);
        }
      l = (ssh_random_get_byte() >> 5) * (8*8*8*8*8) +
        (ssh_random_get_byte() >> 5) * (8*8*8*8) +
        (ssh_random_get_byte() >> 5) * (8*8*8) +
        (ssh_random_get_byte() >> 5) * (8*8) +
        (ssh_random_get_byte() >> 5) * 8 +
        (ssh_random_get_byte() >> 5);
      SET_BIT(l);
    }

  printf("\nChecking missing sequences...\n");

  for (i = 0, m = 0; i < SIZE * 8; i++)
    {
      if (!GET_BIT(i))
        m++;
    }

  average = ((double)m / (SIZE*8));
  if (average < 0.11 || average > 0.15)
    {
      printf("\n NOTE: Possible error detected.\n");
      error++;
    }

  ssh_xfree(bit_table);
#undef SIZE
#undef LOOPS

  printf("\n");

  if (error > 2)
    {
      printf("\nATTENTION: Random statistics checks failed %d times.\n"
             "It is however possible that there is nothing wrong with it\n"
             "but to be safe run this test again.\n"
             "\nContact SSH if this error appears more than few times.\n",
             error);
      ssh_fatal("Warning: possible failure in random number generation.");
    }
}

void test_random(int flag)
{
  /* Run the simple bit testing. */
  printf(" - bit tests.\n");
  rnd_test_bits();

  /* Run the missing sequence tests. */
  printf(" - sequence tests.\n");
  rnd_test_missing_bits();

  /* Random tests ends. */
}
