
#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshcryptocore/namelist.h"

SshCryptoStatus
ssh_pk_group_set_scheme(SshPkGroup group, void *scheme, SshPkSchemeFlag flag)
{
  switch (flag)
    {
    case SSH_PK_SCHEME_SIGN:
    case SSH_PK_SCHEME_ENCRYPT:
      /* XXX case SSH_PK_SCHEME_UDH: */
      /* Lets just ignore these, not considered errorneous. Main reason for
         this is the fact that some of these might want to add some
         information to the action_make context and we don't want to
         restrict that. */
      break;
    case SSH_PK_SCHEME_DH:
      group->diffie_hellman = scheme;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;
}

/* Generate the full name of a particular pk group. */
char *ssh_pk_group_name(SshPkGroup group)
{
  SshNameTree tree;
  SshNameNode node;
  char *tmp = NULL;

  ssh_ntree_allocate(&tree);
  if (tree)
    {
      node = ssh_ntree_add_child(tree, NULL, group->type->name);
      if (group->diffie_hellman)
        {
          node = ssh_ntree_add_next(tree, node, "dh");
          ssh_ntree_add_child(tree, node, group->diffie_hellman->name);
        }
      ssh_ntree_generate_string(tree, &tmp);
      ssh_ntree_free(tree);
    }
  return tmp;
}

SshCryptoStatus ssh_pk_group_get_scheme_name(SshPkGroup group,
                                             const char **name,
                                             SshPkSchemeFlag flag)
{
  switch (flag)
    {
    case SSH_PK_SCHEME_DH:
      *name = group->diffie_hellman->name;
      break;
    default:
      return SSH_CRYPTO_LIBRARY_CORRUPTED;
      break;
    }
  return SSH_CRYPTO_OK;
}

/* Function to retrieve a comma separated list of supported predefined
   groups for this particular key type. */

char *
ssh_public_key_get_predefined_groups(const char *key_type)
{
  SshNameTree tree;
  SshNameNode node;
  SshNameTreeStatus nstat;
  char *tmp;
  unsigned int i;

  /* Generate a name tree from key type. */
  ssh_ntree_allocate(&tree);
  if (!tree)
      return NULL;
  nstat = ssh_ntree_parse(key_type, tree);
  if (nstat != SSH_NTREE_OK)
    {
      ssh_ntree_free(tree);
      return NULL;
    }
  node = ssh_ntree_get_root(tree);
  if (node == NULL ||
      (tmp = ssh_nnode_get_identifier(node)) == NULL)
    {
      ssh_ntree_free(tree);
      return NULL;
    }

  /* Free the allocated tree now; we are not going to need it later. */
  ssh_ntree_free(tree);

  for (i = 0; ssh_pk_type_slots[i]->name; i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, tmp) == 0)
        {
          if (ssh_pk_type_slots[i]->pk_group_get_predefined_groups != NULL_FNPTR)
            {
              ssh_free(tmp);
              return (*ssh_pk_type_slots[i]->pk_group_get_predefined_groups)();
            }
        }
    }
  ssh_free(tmp);
  return NULL;
}

/* Parameter functions named here as ssh pk group (standing for
   ssh public key group). */

SshCryptoStatus
ssh_pk_group_generate(SshPkGroup *group,
                      const char *group_type, ...)
{
  SshCryptoStatus status;
  unsigned int i;
  const SshPkAction *action;
  SshPkGroup pk_group;
  void *context = NULL;
  void *scheme;
  SshPkFormat format;
  SshNameTree tree;
  SshNameNode node, child;
  SshNameTreeStatus nstat;
  const char *name, *r;
  char consumed[128], *tmp;
  va_list ap;

  /* Parse given group type. */
  ssh_ntree_allocate(&tree);
  if (!tree)
      return SSH_CRYPTO_NO_MEMORY;
  nstat = ssh_ntree_parse(group_type, tree);
  if (nstat != SSH_NTREE_OK)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_UNKNOWN_GROUP_TYPE;
    }
  node = ssh_ntree_get_root(tree);
  if (node == NULL)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_UNKNOWN_GROUP_TYPE;
    }

  tmp = ssh_nnode_get_identifier(node);
  if (tmp == NULL)
    {
      ssh_ntree_free(tree);
      return SSH_CRYPTO_NO_MEMORY;
    }
  for (i = 0;
       ssh_pk_type_slots[i] != NULL && ssh_pk_type_slots[i]->name;
       i++)
    {
      if (strcmp(ssh_pk_type_slots[i]->name, tmp) != 0)
        continue;
      ssh_free(tmp);
      tmp = NULL;

      /* Type matches i.e. we've found our key type, so continue with
         finding schemes and parameters. */

      node = ssh_nnode_get_child(node);

      /* Allocate group context. */
      if ((pk_group = ssh_malloc(sizeof(*pk_group))) != NULL)
        {
          pk_group->type = ssh_pk_type_slots[i];
          pk_group->diffie_hellman = NULL;
          context = (*pk_group->type->pk_group_action_init)();
          if (context == NULL)
            {
              ssh_free(pk_group);
              return SSH_CRYPTO_OPERATION_FAILED;
            }

          /* Run through all preselected schemes in the group_type. */
          status = SSH_CRYPTO_OK;
          while (node)
            {
              tmp = ssh_nnode_get_identifier(node);
              action = ssh_pk_find_scheme_action(pk_group->type->action_list,
                                                 tmp,
                                                 SSH_PK_FLAG_PK_GROUP);
              ssh_free(tmp);
              tmp = NULL;
              if (!action)
                {
                  status = SSH_CRYPTO_SCHEME_UNKNOWN;
                  break;
                }
              child = ssh_nnode_get_child(node);
              if (child == NULL)
                /* We are not yet confident that there does not exists
                   a method of this name. Thus because for some
                   schemes it is easier to just write the scheme
                   class, we try to match for a fixed name. */
                tmp = ssh_strdup(SSH_PK_USUAL_NAME);
              else
                tmp = ssh_nnode_get_identifier(child);

              /* Find the scheme of that name. */
              scheme = ssh_pk_find_generic(tmp,
                                           action->type,
                                           action->type_size);
              ssh_free(tmp);
              tmp = NULL;

              if (scheme == NULL)
                {
                  status = SSH_CRYPTO_SCHEME_UNKNOWN;
                  break;
                }

              /* Call action_scheme if not set to NULL. */
              if (((SshPkGen *)scheme)->action_scheme != NULL_FNPTR)
                (*((SshPkGen *)scheme)->action_scheme)(context);

              /* Set the corresponding scheme to the group. */
              status = ssh_pk_group_set_scheme(pk_group, scheme,
                                               action->scheme_flag);

              if (status != SSH_CRYPTO_OK)
                break;

              /* Move to the next scheme. */
              node = ssh_nnode_get_next(node);
            }
        }
      else
        status = SSH_CRYPTO_OPERATION_FAILED;

      ssh_ntree_free(tree);
      if (status != SSH_CRYPTO_OK)
        {
          if (pk_group != NULL)
            {
              (*pk_group->type->pk_group_action_free)(context);
              ssh_free(pk_group);
            }
          return status;
        }

      /* Start reading the vararg list. */
      consumed[0] = '\000';
      while (TRUE)
        {
          va_start(ap, group_type);
          PROCESS(ap, consumed);

          format = va_arg(ap, SshPkFormat);
          strcat(consumed, "i");
          if (format == SSH_PKF_END)
            break;

          /* Search name from command lists. */
          action = ssh_pk_find_action(format,
                                      pk_group->type->action_list,
                                      SSH_PK_FLAG_PK_GROUP);
          if (!action)
            {
              /* Free the action context. */
              (*pk_group->type->pk_group_action_free)(context);
              ssh_free(pk_group);
              va_end(ap);
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          /* Supported only scheme selection and special operations. */
          switch (action->flags & (SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_SPECIAL))
            {
            case SSH_PK_FLAG_SCHEME:
              name = va_arg(ap, const char *);
              strcat(consumed, "p");
              scheme = ssh_pk_find_generic(name, action->type,
                                           action->type_size);
              if (scheme == NULL)
                {
                  (*pk_group->type->pk_group_action_free)(context);
                  ssh_free(pk_group);
                  va_end(ap);
                  return SSH_CRYPTO_SCHEME_UNKNOWN;
                }

              /* Call action_scheme if not set to NULL. */
              if (((SshPkGen *)scheme)->action_scheme != NULL_FNPTR)
                (*((SshPkGen *)scheme)->action_scheme)(context);

              /* Set the corresponding scheme to the group. */
              status = ssh_pk_group_set_scheme(pk_group, scheme,
                                               action->scheme_flag);

              if (status != SSH_CRYPTO_OK)
                {
                  (*pk_group->type->pk_group_action_free)(context);
                  ssh_free(pk_group);
                  va_end(ap);
                  return status;
                }
              break;

            case SSH_PK_FLAG_SPECIAL:
              /* Assume no wrappings. */
              if (action->flags & SSH_PK_FLAG_WRAPPED)
                {
                  if (action->action_put)
                    ssh_fatal("ssh_pk_group_generate: cannot wrap.");
                  va_end(ap);
                  (*pk_group->type->pk_group_action_free)(context);
                  ssh_free(pk_group);
                  return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
                }

              r = (*action->action_put)(context, ap, NULL, format);

              if (r == NULL)
                {
                  (*pk_group->type->pk_group_action_free)(context);
                  ssh_free(pk_group);
                  va_end(ap);
                  return SSH_CRYPTO_LIBRARY_CORRUPTED;
                }
              else
                strcat(consumed, r);
              break;
            default:
              ssh_fatal("ssh_pk_group_generate: internal error.");
              break;
            }
          va_end(ap);
        }

      /* Make the key and remove context. (One could incorporate making
         and freeing, however this way things seem to work also). */
      pk_group->context =
        (*pk_group->type->pk_group_action_make)(context);
      (*pk_group->type->pk_group_action_free)(context);

      /* Quit unhappily. */
      if (pk_group->context == NULL)
        {
          ssh_free(pk_group);
          va_end(ap);
          return SSH_CRYPTO_OPERATION_FAILED;
        }

      /* Quit happily. */
      *group = pk_group;
      va_end(ap);

      return SSH_CRYPTO_OK;
    }
  ssh_free(tmp);

  ssh_ntree_free(tree);
  va_end(ap);

  return SSH_CRYPTO_UNKNOWN_GROUP_TYPE;
}

void
ssh_pk_group_free(SshPkGroup group)
{
  if (group == NULL || group->context == NULL)
    ssh_fatal("ssh_pk_group_free: undefined group.");
  if (group->type->pk_group_free)
    (*group->type->pk_group_free)(group->context);
  group->context = NULL;
  ssh_free(group);
}

SshCryptoStatus
ssh_pk_group_select_scheme(SshPkGroup group, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  void *scheme;
  SshPkFormat format;
  const char *name;
  va_list ap;

  if (group->type == NULL)
    return SSH_CRYPTO_KEY_UNINITIALIZED;

  va_start(ap, group);

  while ((format = va_arg(ap, SshPkFormat)) != SSH_PKF_END)
    {
      action = ssh_pk_find_action(format, group->type->action_list,
                                  SSH_PK_FLAG_SCHEME |
                                  SSH_PK_FLAG_PK_GROUP);
      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      /* Find the new scheme. */
      name = va_arg(ap, const char *);
      scheme = ssh_pk_find_generic(name, action->type,
                                   action->type_size);
      /* Check that scheme exists. */
      if (scheme == NULL)
        {
          va_end(ap);
          return SSH_CRYPTO_SCHEME_UNKNOWN;
        }

      status = ssh_pk_group_set_scheme(group, scheme, action->scheme_flag);
      if (status != SSH_CRYPTO_OK)
        {
          va_end(ap);
          return status;
        }
    }
  va_end(ap);
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_pk_group_get_info(SshPkGroup group, ...)
{
  SshCryptoStatus status;
  const SshPkAction *action;
  SshPkFormat format;
  const char **name_ptr, *r;
  char consumed[128];
  va_list ap;

  consumed[0] = '\000';
  while (TRUE)
    {
      va_start(ap, group);
      PROCESS(ap, consumed);

      format = va_arg(ap, SshPkFormat);
      strcat(consumed, "i");
      if (format == SSH_PKF_END)
        break;

      /* Seek for the action. */
      action = ssh_pk_find_action(format,
                                  group->type->action_list,
                                  SSH_PK_FLAG_PK_GROUP);
      if (!action)
        {
          va_end(ap);
          return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
        }

      switch (action->flags & (SSH_PK_FLAG_SCHEME | SSH_PK_FLAG_SPECIAL |
                               SSH_PK_FLAG_KEY_TYPE))
        {
        case SSH_PK_FLAG_KEY_TYPE:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          *name_ptr = strchr(group->type->name, ':');
          if (*name_ptr)
            (*name_ptr)++;
          else
            *name_ptr = group->type->name;
          break;

        case SSH_PK_FLAG_SCHEME:
          name_ptr = va_arg(ap, const char **);
          strcat(consumed, "p");
          status = ssh_pk_group_get_scheme_name(group,
                                                name_ptr,
                                                action->scheme_flag);
          if (status != SSH_CRYPTO_OK)
            {
              va_end(ap);
              return status;
            }
          break;

        case SSH_PK_FLAG_SPECIAL:
          if (action->flags & SSH_PK_FLAG_WRAPPED)
            {
              if (action->action_get)
                ssh_fatal("ssh_pk_group_get_info: cannot wrap.");
              va_end(ap);
              return SSH_CRYPTO_UNSUPPORTED_IDENTIFIER;
            }

          r = (*action->action_get)(group->context, ap, NULL, format);
          if (r == NULL)
            {
              va_end(ap);
              return SSH_CRYPTO_LIBRARY_CORRUPTED;
            }
          strcat(consumed, r);
          break;

        default:
          ssh_fatal("ssh_pk_group_get_info: internal error.");
          break;
        }
      va_end(ap);
    }

  va_end(ap);
  return SSH_CRYPTO_OK;
}

SshCryptoStatus
ssh_pk_group_precompute(SshPkGroup group)
{
  if (group == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  if (group->type->pk_group_precompute)
    (*group->type->pk_group_precompute)(group->context);
  return SSH_CRYPTO_OK;
}


SshCryptoStatus
ssh_pk_group_generate_randomizer(SshPkGroup group)
{
  if (group == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;

  if (group->type->pk_group_generate_randomizer)
    (*group->type->pk_group_generate_randomizer)(group->context);
  return SSH_CRYPTO_OK;
}

unsigned int
ssh_pk_group_count_randomizers(SshPkGroup group)
{
  if (group == NULL)
    return SSH_CRYPTO_OPERATION_FAILED;
  if (group->type->pk_group_count_randomizers)
    return (*group->type->pk_group_count_randomizers)(group->context);
  return 0;
}
