/*
  File: rsa-operation.c

  Authors:
        Mika Kojo <mkojo@ssh.fi>
        Tatu Ylonen <ylo@cs.hut.fi>
        Tero T Mononen <tmo@ssh.fi>

  Description:

        Take on the RSA operations and key definition, modified after
        Tatu Ylonen's original SSH implementation.

        Description of the RSA algorithm can be found e.g. from the
        following sources:

  - Bruce Schneier: Applied Cryptography.  John Wiley & Sons, 1994.
  - Jennifer Seberry and Josed Pieprzyk: Cryptography: An Introduction to
    Computer Security.  Prentice-Hall, 1989.
  - Man Young Rhee: Cryptography and Secure Data Communications.  McGraw-Hill,
    1994.
  - R. Rivest, A. Shamir, and L. M. Adleman: Cryptographic Communications
    System and Method.  US Patent 4,405,829, 1983.
  - Hans Riesel: Prime Numbers and Computer Methods for Factorization.
    Birkhauser, 1994.

  Copyright:
        Copyright (c) 1995-2001 SSH Communications Security Corp, Finland.
        All rights reserved.
*/

#include "sshincludes.h"
#include "sshmp.h"
#include "sshgenmp.h"
#include "sshcrypt.h"
#include "sshgetput.h"
#include "sshpk.h"
#include "rsa.h"
#include "sshasn1.h"

#ifdef WITH_RSA

#define SSH_DEBUG_MODULE "SshCryptoRSA"






#define SSH_RSA_MINIMUM_PADDING 10
#define SSH_RSA_MAX_BYTES       65535









void *ssh_rsa_make_private_key_of_all(SshMPInt p, SshMPInt q, SshMPInt n,
                                      SshMPInt e, SshMPInt d, SshMPInt u)
{
  SshRSAPrivateKey *private_key = ssh_malloc(sizeof(*private_key));

  if (private_key)
    {
      ssh_mp_init_set(&private_key->e, e);
      ssh_mp_init_set(&private_key->d, d);
      ssh_mp_init_set(&private_key->n, n);
      ssh_mp_init_set(&private_key->u, u);
      ssh_mp_init_set(&private_key->p, p);
      ssh_mp_init_set(&private_key->q, q);
      private_key->bits = ssh_mp_bit_size(n);
    }
  return (void *)private_key;
}

void ssh_rsa_private_key_init(SshRSAPrivateKey *private_key)
{
  /* Initialize with zeroes. */
  ssh_mp_init_set_ui(&private_key->e, 0);
  ssh_mp_init_set_ui(&private_key->d, 0);
  ssh_mp_init_set_ui(&private_key->n, 0);
  ssh_mp_init_set_ui(&private_key->u, 0);
  ssh_mp_init_set_ui(&private_key->p, 0);
  ssh_mp_init_set_ui(&private_key->q, 0);
  private_key->bits = 0;
}


/* Making the key from this bunch of information. */

void *ssh_rsa_public_key_make_action(void *context)
{
  SshRSAInitCtx *ctx = context;
  SshRSAPublicKey *public_key;

  if (ssh_mp_cmp_ui(&ctx->e, 0) == 0 ||
      ssh_mp_cmp_ui(&ctx->n, 0) == 0)
    return NULL;

  if ((public_key = ssh_malloc(sizeof(*public_key))) != NULL)
    {
      ssh_mp_init_set(&public_key->e, &ctx->e);
      ssh_mp_init_set(&public_key->n, &ctx->n);

      /* Compute the size of the public key. */
      public_key->bits = ssh_mp_bit_size(&public_key->n);
    }
  return public_key;
}


/* Try to handle the given data in a reasonable manner. */
void *ssh_rsa_private_key_define_action(void *context)
{
  SshRSAInitCtx *ctx = context;

  if (ssh_mp_cmp_ui(&ctx->d, 0) != 0 &&
      ssh_mp_cmp_ui(&ctx->p, 0) != 0 &&
      ssh_mp_cmp_ui(&ctx->q, 0) != 0 &&
      ssh_mp_cmp_ui(&ctx->e, 0) != 0 &&
      ssh_mp_cmp_ui(&ctx->n, 0) != 0 &&
      ssh_mp_cmp_ui(&ctx->u, 0) != 0)
    {
      return
        ssh_rsa_make_private_key_of_all(&ctx->p, &ctx->q,
                                        &ctx->n, &ctx->e,
                                        &ctx->d, &ctx->u);
    }
  return NULL;
}

void ssh_rsa_private_key_derive_public_key(const void *private_key,
                                           void **public_key)
{
  const SshRSAPrivateKey *prv = private_key;
  SshRSAPublicKey *pub;

  if ((pub = ssh_malloc(sizeof(*pub))) != NULL)
    {
      ssh_mp_init_set(&pub->n, &prv->n);
      ssh_mp_init_set(&pub->e, &prv->e);

      /* Size of the public key modulus. */
      pub->bits = prv->bits;
    }
  *public_key = (void *)pub;
}

/* These are return values for ssh_rsa_private. */
typedef enum
{
  SSH_RSA_OP_OK,
  SSH_RSA_OP_FAILED
} SshRSAOpStatus;



































































































































/* Performs a private-key RSA operation (encrypt/decrypt).  The
   computation is done using the Chinese Remainder Theorem, which is
   faster than direct modular exponentiation. */
static SshRSAOpStatus ssh_rsa_private(SshMPInt input,
                                      SshMPInt output,
                                      const SshRSAPrivateKey *prv)
{






















  SshMPIntStruct dp, dq, p2, q2, k;

  /* Initialize temporary variables. */
  ssh_mp_init(&dp);
  ssh_mp_init(&dq);
  ssh_mp_init(&p2);
  ssh_mp_init(&q2);
  ssh_mp_init(&k);

  /* Compute dp = d mod p-1. */
  ssh_mp_sub_ui(&dp, &prv->p, 1);
  ssh_mp_mod(&dp, &prv->d, &dp);

  /* Compute dq = d mod q-1. */
  ssh_mp_sub_ui(&dq, &prv->q, 1);
  ssh_mp_mod(&dq, &prv->d, &dq);

  /* Compute p2 = (input mod p) ^ dp mod p. */
  ssh_mp_mod(&p2, input, &prv->p);
  ssh_mp_powm(&p2, &p2, &dp, &prv->p);

  /* Compute q2 = (input mod q) ^ dq mod q. */
  ssh_mp_mod(&q2, input, &prv->q);
  ssh_mp_powm(&q2, &q2, &dq, &prv->q);

  /* Compute k = ((q2 - p2) mod q) * u mod q. */
  ssh_mp_sub(&k, &q2, &p2);
  ssh_mp_mul(&k, &k, &prv->u);
  ssh_mp_mod(&k, &k, &prv->q);

  /* Compute output = p2 + p * k. */
  ssh_mp_mul(output, &prv->p, &k);
  ssh_mp_add(output, output, &p2);

  /* Clear temporary variables. */
  ssh_mp_clear(&dp);
  ssh_mp_clear(&dq);
  ssh_mp_clear(&p2);
  ssh_mp_clear(&q2);
  ssh_mp_clear(&k);

  if (ssh_mprz_isnan(output))
    return SSH_RSA_OP_FAILED;
  else
  return SSH_RSA_OP_OK;

}

/* Performs a public-key RSA operation (encrypt/decrypt). */

void ssh_rsa_public(SshMPInt input, SshMPInt output,
                    const SshRSAPublicKey *pub)
{




















  /* Exponentiate. */
  ssh_mp_powm(output, input, &pub->e, &pub->n);


}


/* MGF1 for the OAEP. */
void ssh_rsa_mgf1(const SshHashDefStruct *hash,
                  const unsigned char *seed, size_t seed_len,
                  unsigned char *mask, size_t mask_len)
{
  SshUInt32      i, steps;
  void          *context;
  unsigned char  digest[SSH_MAX_HASH_DIGEST_LENGTH];

  if ((context = ssh_malloc((*hash->ctxsize)())) == NULL)
    {
      /* XXX: recover from out-of-memory */
      return;
    }

  for (i = 0, steps = 0; i < mask_len; i += hash->digest_length, steps++)
    {
      unsigned char counter[4];
      size_t avail;

      SSH_PUT_32BIT(counter, steps);

      (*hash->reset_context)(context);
      (*hash->update)(context, seed, seed_len);
      (*hash->update)(context, counter, 4);
      (*hash->final)(context, digest);

      /* Now copy the digest to the mask. */
      avail = mask_len - i;
      if (avail >= hash->digest_length)
        memcpy(mask + i, digest, hash->digest_length);
      else
        memcpy(mask + i, digest, avail);
    }
  ssh_free(context);
}

/* OAEP encode.

   Note: in future a suitable library of the mask generation functions
   should be written.
 */
Boolean ssh_rsa_oaep_encode_with_mgf1(const SshHashDefStruct *hash,
                                      const unsigned char *msg,
                                      size_t msg_len,
                                      const unsigned char *param,
                                      size_t param_len,
                                      unsigned char *emsg, size_t emsg_len)
{
  void *context;
  unsigned char *db;
  unsigned char seed[SSH_MAX_HASH_DIGEST_LENGTH];
  size_t db_len, i;

  /* Check that the size constraints are satisfied. */
  if (msg_len > emsg_len - 2*hash->digest_length - 1)
    return FALSE;

  /* This is: emLen - ||M|| - 2hLen - 1  + hLen + 1 + ||M|| =
     emLen - hLen. */
  db_len = emsg_len - hash->digest_length;
  if ((db = ssh_calloc(1, db_len)) == NULL)
    return FALSE;

  /* Compute a hash of the params. */
  if ((context = ssh_malloc((*hash->ctxsize)())) == NULL)
    {
      ssh_free(db);
      return FALSE;
    }

  (*hash->reset_context)(context);
  (*hash->update)(context, param, param_len);
  (*hash->final)(context, db);

  /* Add the "01" before the last msg_len bytes. */
  db[db_len - msg_len - 1] = 0x1;

  /* Now throw in the msg. */
  memcpy(db + db_len - msg_len, msg, msg_len);

  /* Generate a random octet string. */
  for (i = 0; i < hash->digest_length; i++) seed[i] = ssh_random_get_byte();

  /* Now use the MGF1. */
  ssh_rsa_mgf1(hash,
               seed, hash->digest_length,
               emsg + hash->digest_length, db_len);

  /* Xor. */
  for (i = 0; i < db_len; i++)
    emsg[hash->digest_length + i] ^= db[i];
  memset(db, 0, db_len);

  /* Use MGF1 again. */
  ssh_rsa_mgf1(hash,
               emsg + hash->digest_length, db_len,
               emsg, hash->digest_length);

  /* Xor the seed. */
  for (i = 0; i < hash->digest_length; i++)
    emsg[i] ^= seed[i];
  memset(seed, 0, hash->digest_length);

  /* Now free the allocated information. */
  ssh_free(context);
  ssh_free(db);

  return TRUE;
}

/* OAEP decode using MGF1. */
Boolean ssh_rsa_oaep_decode_with_mgf1(const SshHashDefStruct *hash,
                                      const unsigned char *emsg,
                                      size_t emsg_len,
                                      const unsigned char *param,
                                      size_t param_len,
                                      unsigned char **msg, size_t *msg_len)
{
  void *context;
  unsigned char  seed[SSH_MAX_HASH_DIGEST_LENGTH];
  unsigned char  phash[SSH_MAX_HASH_DIGEST_LENGTH];
  unsigned char *db;
  size_t         db_len, i;
  Boolean        rv = FALSE;

  if (emsg_len < 2*hash->digest_length + 1)
    return FALSE;

  /* Allocate enough working buffers. */
  db_len = emsg_len - hash->digest_length;
  if ((db = ssh_malloc(db_len)) == NULL)
    return FALSE;

  /* Use the mgf. */
  ssh_rsa_mgf1(hash, emsg + hash->digest_length, db_len,
               seed, hash->digest_length);
  /* Now xor. */
  for (i = 0; i < hash->digest_length; i++)
    seed[i] ^= emsg[i];

  /* Use the mgf again. */
  ssh_rsa_mgf1(hash,
               seed, hash->digest_length,
               db, db_len);
  /* Now xor again. */
  for (i = 0; i < db_len; i++)
    db[i] ^= emsg[hash->digest_length + i];

  /* Compute a hash of the parameters. */
  if ((context = ssh_malloc((*hash->ctxsize)())) == NULL)
    {
      ssh_free(db);
      return FALSE;
    }

  (*hash->reset_context)(context);
  (*hash->update)(context, param, param_len);
  (*hash->final)(context, phash);

  /* Do the check. */
  if (memcmp(db, phash, hash->digest_length) != 0)
    goto failed;

  for (i = hash->digest_length; i < db_len; i++)
    {
      if (db[i] != 0)
        {
          if (db[i] != 0x1)
            goto failed;
          break;
        }
    }
  if (i >= db_len)
    goto failed;

  /* Now we must have db[i] == 0x1. */
  *msg_len = db_len - i - 1;
  if ((*msg = ssh_malloc(*msg_len)) == NULL)
    {
      rv = FALSE;
      goto failed;
    }
  memcpy(*msg, db + i + 1, *msg_len);
  rv = TRUE;

failed:
  ssh_free(context);
  ssh_free(db);

  return rv;
}

/* This function is specific to PKCS-1v2 and might need revision
   after possible updates. However, there is little we can help
   here. */
unsigned char *
ssh_rsa_pkcs1v2_default_explicit_param(const SshHashDefStruct *hash,
                                       size_t *param_len)
{
  SshAsn1Status  status;
  SshAsn1Context context;
  SshAsn1Node    node;
  unsigned char *param;

  /* Initialize the ASN.1 library. */
  if ((context = ssh_asn1_init()) == NULL)
    {
      *param_len = 0;
      return NULL;
    }

  /* Useful OIDs. */
#define SSH_RSA_MGF1_OID    "1.2.840.113549.1.1.8"
#define SSH_RSA_PSOURCE_OID "1.2.840.113549.1.1.9"

  status = ssh_asn1_create_node(context, &node,
                                "(sequence ()"
                                "  (sequence ()"
                                "    (object-identifier ())"
                                "    (null ()))"                 /* Hash */
                                "  (sequence ()"
                                "    (object-identifier ())"     /* MGF */
                                "    (sequence ()"
                                "      (object-identifier ())"
                                "      (null ())))"              /* Hash */
                                "  (sequence ()"
                                "    (object-identifier ())"     /* psource */
                                "    (octet-string ())))",
                                hash->asn1_oid,
                                SSH_RSA_MGF1_OID,
                                hash->asn1_oid,
                                SSH_RSA_PSOURCE_OID,
                                NULL, 0);
  if (status != SSH_ASN1_STATUS_OK)
    {
      ssh_asn1_free(context);
      return NULL;
    }

  /* Now encode the node. */

  status = ssh_asn1_encode_node(context, node);
  if (status != SSH_ASN1_STATUS_OK)
    {
      ssh_asn1_free(context);
      return NULL;
    }

  status =
    ssh_asn1_node_get_data(node, &param, param_len);
  if (status != SSH_ASN1_STATUS_OK)
    {
      ssh_asn1_free(context);
      return NULL;
    }

  ssh_asn1_free(context);

  return param;
}

Boolean
ssh_rsa_public_key_none_decrypt(const void *public_key,
                                const unsigned char *ciphertext,
                                size_t ciphertext_len,
                                unsigned char *plaintext,
                                size_t plaintext_len,
                                size_t *plaintext_length_return)
{
  const SshRSAPublicKey *pub = public_key;
  SshMPIntStruct t1, t2;

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Unlinearize. */
  ssh_buf_to_mp(&t1, ciphertext, ciphertext_len);
  if (ssh_mp_cmp(&t1, &pub->n) >= 0 ||
      ssh_mp_cmp_ui(&t1, 0) <= 0)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      return FALSE;
    }

  /* Allocate temporary buffer. */
  *plaintext_length_return = (pub->bits + 7)/8;
  if (plaintext_len < *plaintext_length_return)
    return FALSE;

  /* Retrieve the original for of the ciphertext. */
  ssh_rsa_public(&t1, &t2, pub);

  /* Unlinearize. */
  ssh_mp_to_buf(plaintext, *plaintext_length_return, &t2);

  /* Clear multiple precision integers. */
  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);
  return TRUE;
}

Boolean
ssh_rsa_private_key_none_encrypt(const void *private_key,
                                 const unsigned char *data,
                                 size_t data_len,
                                 unsigned char *signature_buffer,
                                 size_t signature_buffer_len,
                                 size_t *signature_length_return)
{
  const SshRSAPrivateKey *prv = private_key;
  SshMPIntStruct t1, t2;
  int rsa_status;

  *signature_length_return = (prv->bits + 7)/8;

  if (*signature_length_return != data_len)
    return FALSE;
  if (*signature_length_return > signature_buffer_len)
    return FALSE;

  /* Compute signature. */

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Unlinearize. */
  ssh_buf_to_mp(&t1, data, data_len);

  /* Private key operation. */
  rsa_status = ssh_rsa_private(&t1, &t2, prv);
  if (rsa_status == SSH_RSA_OP_FAILED)
  {
    ssh_mp_clear(&t1);
    ssh_mp_clear(&t2);

    return FALSE;
  }
  if (rsa_status == SSH_RSA_OP_OK)
    /* Linearize. */
    ssh_mp_to_buf(signature_buffer, *signature_length_return, &t2);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  return TRUE;
}

/* The routines using the RGF's. */

Boolean
ssh_rsa_public_key_encrypt(const void *public_key,
                           const unsigned char *plaintext,
                           size_t plaintext_len,
                           unsigned char *ciphertext_buffer,
                           size_t ciphertext_buffer_len,
                           size_t *ciphertext_len_return,
                           SshRGFHash rgf)
{
  const SshRSAPublicKey *pub = public_key;
  SshMPIntStruct t1, t2;
  unsigned char *buf;

  /* Get size. */
  *ciphertext_len_return = (pub->bits + 7)/8;

  /* Check lengths. */
  if (*ciphertext_len_return < plaintext_len)
    return FALSE;
  if (*ciphertext_len_return > ciphertext_buffer_len)
    return FALSE;

  /* Allocate large enough buffer. */
  if ((buf = ssh_malloc(*ciphertext_len_return)) == NULL)
    return FALSE;

  /* Now run the RGF. */

  if (ssh_rgf_hash_encrypt(rgf,
                           plaintext, plaintext_len,
                           buf, *ciphertext_len_return) != SSH_RGF_OK)
    {
      ssh_free(buf);
      return FALSE;
    }

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Convert to integer. */
  ssh_buf_to_mp(&t2, buf, *ciphertext_len_return);
  ssh_free(buf);

  /* Public encrypt. */
  ssh_rsa_public(&t2, &t1, pub);

  /* Linearize again. */
  ssh_mp_to_buf(ciphertext_buffer, *ciphertext_len_return, &t1);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  return TRUE;
}

Boolean ssh_rsa_private_key_decrypt(const void *private_key,
                                    const unsigned char *ciphertext,
                                    size_t ciphertext_len,
                                    unsigned char *plaintext_buffer,
                                    size_t plaintext_buffer_len,
                                    size_t *plaintext_length_return,
                                    SshRGFHash rgf)
{
  const SshRSAPrivateKey *prv = private_key;
  SshMPIntStruct t1, t2;
  size_t prv_size = (prv->bits + 7)/8;
  unsigned char *decrypted_msg, *output_msg;
  size_t output_msg_len;

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Unlinearize. */
  ssh_buf_to_mp(&t1, ciphertext, ciphertext_len);

  /* Private key operation. */
  if (ssh_rsa_private(&t1, &t2, prv) == SSH_RSA_OP_FAILED)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      return FALSE;
    }

  /* Linearize. */
  if ((decrypted_msg = ssh_malloc(prv_size)) == NULL)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      return FALSE;
    }

  ssh_mp_to_buf(decrypted_msg, prv_size, &t2);

  if (ssh_rgf_hash_decrypt(rgf,
                           decrypted_msg, prv_size, prv_size,
                           &output_msg, &output_msg_len) != SSH_RGF_OK)
    {
      ssh_free(decrypted_msg);
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      return FALSE;
    }

  ssh_free(decrypted_msg);
  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  /* Now check the output size. */
  if (output_msg_len > plaintext_buffer_len)
    {
      ssh_free(output_msg);
      return FALSE;
    }

  memcpy(plaintext_buffer, output_msg, output_msg_len);
  *plaintext_length_return = output_msg_len;
  ssh_free(output_msg);

  return TRUE;
}

Boolean ssh_rsa_public_key_verify(const void *public_key,
                                  const unsigned char *signature,
                                  size_t signature_len,
                                  SshRGFHash hash)
{
  const SshRSAPublicKey *pub = public_key;
  SshMPIntStruct t1, t2;
  unsigned char *buf, *output_buf;
  size_t buf_len, output_buf_len, len;
  Boolean rv = FALSE;

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Unlinearize. */
  ssh_buf_to_mp(&t1, signature, signature_len);
  if (ssh_mprz_isnan(&t1) ||
      ssh_mp_cmp(&t1, &pub->n) >= 0 ||
      ssh_mp_cmp_ui(&t1, 0) <= 0)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      ssh_rgf_hash_free(hash);
      goto failed;
    }

  /* Allocate temporary buffer. */
  buf_len = (pub->bits + 7)/8;
  if ((buf = ssh_malloc(buf_len)) == NULL)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      ssh_rgf_hash_free(hash);
      goto failed;
    }

  /* Retrieve the original for of the signature. */
  ssh_rsa_public(&t1, &t2, pub);

  /* Unlinearize. */
  len = ssh_mp_to_buf(buf, buf_len, &t2);

  /* Clear multiple precision integers. */
  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  if (len == 0 ||
      ssh_rgf_hash_verify(hash, buf, buf_len, buf_len,
                          &output_buf, &output_buf_len) != SSH_RGF_OK)
    {
      ssh_free(buf);
      goto failed;
    }

  ssh_free(buf);
  if (output_buf != NULL)
    ssh_free(output_buf);
  rv = TRUE;
failed:

  return rv;
}

Boolean
ssh_rsa_private_key_sign(const void *private_key,
                         SshRGFHash hash,
                         unsigned char *signature_buffer,
                         size_t signature_buffer_len,
                         size_t *signature_length_return)
{
  const SshRSAPrivateKey *prv = private_key;
  unsigned char *output_msg;
  SshMPIntStruct t1, t2;
  int rsa_status;

  *signature_length_return = (prv->bits + 7)/8;

  if (*signature_length_return > signature_buffer_len)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  if ((output_msg = ssh_malloc(*signature_length_return)) == NULL)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  /* Build the to be signed data. */
  if (ssh_rgf_hash_sign(hash,
                        output_msg, *signature_length_return) != SSH_RGF_OK)
    {
      ssh_free(output_msg);
      return FALSE;
    }

  /* Compute signature. */

  ssh_mp_init(&t1);
  ssh_mp_init(&t2);

  /* Unlinearize. */
  ssh_buf_to_mp(&t1, output_msg, *signature_length_return);
  ssh_free(output_msg);

  /* Private key operation. */
  rsa_status = ssh_rsa_private(&t1, &t2, prv);
  if (rsa_status == SSH_RSA_OP_FAILED)
    {
      ssh_mp_clear(&t1);
      ssh_mp_clear(&t2);
      return FALSE;
    }
  /* Linearize. */
  ssh_mp_to_buf(signature_buffer, *signature_length_return, &t2);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);

  return TRUE;
}

/* Compute sizes needed in each operation. */

size_t ssh_rsa_public_key_max_encrypt_input_len(const void *public_key)
{
  const SshRSAPublicKey *pub = public_key;
  size_t len = ((pub->bits + 7)/8 - 3 - SSH_RSA_MINIMUM_PADDING);

  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

/* Note: this should be changed to take the hash function as an argument.
   Indeed, even more nicely take the MGF as an argument. */
size_t ssh_rsa_public_key_max_oaep_encrypt_input_len(const void *public_key)
{
  const SshRSAPublicKey *pub = public_key;
  size_t len = ((pub->bits + 7)/8 - 2 - 2*SSH_MAX_HASH_DIGEST_LENGTH);

  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_rsa_public_key_max_none_encrypt_input_len(const void *public_key)
{
  const SshRSAPublicKey *pub = public_key;
  size_t len = ((pub->bits + 7)/8);

  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_rsa_private_key_max_decrypt_input_len(const void *private_key)
{
  const SshRSAPrivateKey *prv = private_key;

  return (prv->bits + 7)/8;
}

size_t ssh_rsa_private_key_max_signature_input_len(const void *private_key)
{
  return (size_t)-1;
}

size_t
ssh_rsa_private_key_max_signature_unhash_input_len(const void *private_key)
{
  const SshRSAPrivateKey *prv = private_key;
  size_t len = ((prv->bits + 7)/8 - 3 - SSH_RSA_MINIMUM_PADDING);

  if (len > 0 && len < SSH_RSA_MAX_BYTES)
    return len;
  return 0;
}

size_t ssh_rsa_public_key_max_encrypt_output_len(const void *public_key)
{
  const SshRSAPublicKey *pub = public_key;

  return (pub->bits + 7)/8;
}

size_t ssh_rsa_private_key_max_decrypt_output_len(const void *private_key)
{
  const SshRSAPrivateKey *prv = private_key;

  return (prv->bits + 7)/8;
}

size_t ssh_rsa_private_key_max_signature_output_len(const void *private_key)
{
  const SshRSAPrivateKey *prv = private_key;

  return (prv->bits + 7)/8;
}

#endif /* WITH_RSA */

/* rsa.c */
