/*

  Authors: Antti Huima   <huima@ssh.fi>
           Mika Kojo     <mkojo@ssh.fi>
           Patrick Irwin <irwin@ssh.fi>

  Copyright (C) 2001 SSH Communications Security Corp., Finland
  All rights reserved.

  Created: Mon Nov  26 09:32:16  2001 [irwin]

  This file contains generic functions for generating 
  multiple-precision primes.

  */

#include "sshincludes.h"
#include "sshmp.h"
#include "sshgenmp.h"
#include "sshcrypt.h"
#include "libmonitor.h"
#include "sshgetput.h"

#define SSH_GENMP_MAX_PRIME        16000
#define SSH_GENMP_MAX_SIEVE_MEMORY 8192
#define SSH_MAX_PRIMES_IN_TABLE 1051

#define SSH_DEBUG_MODULE "SshGenMPPrime"

/* Generate traditional prime. */

/* XXX: this may fail. In failure, the ret is set to NaN. */
void ssh_mp_random_prime(SshMPInt ret, unsigned int bits)
{
  SshMPIntStruct start, aux;
  SshSieveStruct sieve;
  unsigned int num_primes, p, i;
  SshWord *moduli = NULL, *prime_table = NULL;
  SshWord difference;

  /* Progress monitoring. */
  unsigned int progress_counter = 0;

  /* Initialize the prime search. */
  ssh_mp_init(&start);
  ssh_mp_init(&aux);

  if (ssh_mprz_isnan(&start) || ssh_mprz_isnan(&aux))
    {
    failure_nosieve:
      ssh_mp_clear(&start);
      ssh_mp_clear(&aux);
      ssh_mprz_makenan(ret, SSH_MP_NAN_ENOMEM);
      return;
    }

  if (bits < 16)
    {
      SshWord temp;

      /* Check from the prime sieve. */
      if (!ssh_sieve_allocate_ui(&sieve, (1 << bits), (1 << bits)))
        goto failure_nosieve;

      /* Do not choose 2. */
      num_primes = ssh_sieve_prime_count(&sieve) - 1;

      ssh_mp_random_integer(&aux, bits);
      if (ssh_mprz_isnan(&aux))
        goto failure;

      temp = ssh_mp_get_ui(&aux) % num_primes;

      for (p = 2; p; p = ssh_sieve_next_prime(p, &sieve), temp--)
        if (temp == 0)
          {
            ssh_mp_set_ui(ret, p);
            break;
          }
      if (temp != 0)
        ssh_fatal("ssh_mp_random_prime: could not find small prime.");

      ssh_mp_clear(&start);
      ssh_mp_clear(&aux);
      return;
    }

  /* Generate the prime sieve, this takes very little time. */
  if (!ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                             SSH_GENMP_MAX_SIEVE_MEMORY))
    goto failure_nosieve;

  /* Don't count 2. */
  num_primes = ssh_sieve_prime_count(&sieve)-1;

  /* Generate a simply indexed prime table. */
  if ((prime_table = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    goto failure;

  for (p = 2, i = 0; p; p = ssh_sieve_next_prime(p, &sieve), i++)
    prime_table[i] = p;

  /* Allocate moduli table. */
  if ((moduli = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    goto failure;

 retry:

  /* Pick a random integer of the appropriate size. */
  ssh_mp_random_integer(&start, bits);
  if (ssh_mprz_isnan(&start))
    goto failure;

  /* Set the highest bit. */
  ssh_mp_set_ui(&aux, 1);
  ssh_mp_mul_2exp(&aux, &aux, bits - 1);
  ssh_mp_or(&start, &start, &aux);
  /* Set the lowest bit to make it odd. */
  ssh_mp_set_ui(&aux, 1);
  ssh_mp_or(&start, &start, &aux);

  /* Initialize moduli of the small primes with respect to the given
     random number. */
  for (i = 0; i < num_primes; i++)
    moduli[i] = ssh_mp_mod_ui(&start, prime_table[i]);

  /* Look for numbers that are not evenly divisible by any of the small
     primes. */
  for (difference = 0; ; difference += 2)
    {
      unsigned int i;

      if (difference > 0x70000000)
        {
          /* Might never happen... */
          goto retry;
        }

      /* Check if it is a multiple of any small prime.  Note that this
         updates the moduli into negative values as difference grows. */
      for (i = 1; i < num_primes; i++)
        {
          while (moduli[i] + difference >= prime_table[i])
            moduli[i] -= prime_table[i];
          if (moduli[i] + difference == 0)
            break;
        }
      if (i < num_primes)
        continue; /* Multiple of a known prime. */

      /* Progress information. */
      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Compute the number in question. */
      ssh_mp_add_ui(ret, &start, difference);

      if (ssh_mprz_isnan(ret))
        goto failure;

      /* Perform Miller-Rabin strong pseudo primality tests */
      if (ssh_mp_is_probable_prime(ret, 20))
        break;
    }

  /* Found a (probable) prime.  It is in ret. */

  /* Sanity check: does it still have the high bit set (we might have
     wrapped around)? */
  ssh_mp_div_2exp(&aux, ret, bits - 1);
  if (ssh_mp_get_ui(&aux) != 1)
    {
      goto retry;
    }

  /* Free the small prime moduli; they are no longer needed. Also free
     start, aux and sieve. */
  ssh_free(moduli);
  ssh_free(prime_table);
  ssh_mp_clear(&start);
  ssh_mp_clear(&aux);
  ssh_sieve_free(&sieve);

  /* Return value already set in ret. */
  return;

 failure:
  ssh_sieve_free(&sieve);
  ssh_free(moduli);
  ssh_free(prime_table);
  ssh_mp_clear(&start);
  ssh_mp_clear(&aux);
  ssh_mprz_makenan(ret, SSH_MP_NAN_ENOMEM);
}

/* Generate a random prime within the [min, max] interval. We observe
   that the process can just choose a random number modulo (max - min)
   and then start from there. If it goes beyond max-1 then it
   cycles. */

void ssh_mp_random_prime_within_interval(SshMPInt ret,
                                         SshMPInt min, SshMPInt max)
{
  SshMPIntStruct pprime, temp, aux;
  SshSieveStruct sieve;
  SshWord *moduli = NULL, *prime_table = NULL, difference, max_difference, num_primes, p;
  unsigned int i, bits;

  /* Progress monitoring. */
  unsigned int progress_counter = 0;

  /* Verify the interval. */
  if (ssh_mp_cmp(min, max) >= 0)
    ssh_fatal("ssh_mp_random_prime_within_interval: interval invalid.");

  /* Initialize temps. */
  ssh_mp_init(&pprime);
  ssh_mp_init(&temp);
  ssh_mp_init(&aux);

  /* Allocate a sieve. */
  if (!ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                             SSH_GENMP_MAX_SIEVE_MEMORY))
    {
      ssh_mp_clear(&pprime);
      ssh_mp_clear(&aux);
      ssh_mp_clear(&temp);
      ssh_mprz_makenan(ret, SSH_MP_NAN_ENOMEM);
      return;
    }

  /* Don't count 2. */
  num_primes = ssh_sieve_prime_count(&sieve) - 1;

  /* Make a table of the primes. */
  if ((prime_table = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    goto failure;

  for (p = 2, i = 0; p; p = ssh_sieve_next_prime(p, &sieve), i++)
    prime_table[i] = p;

  /* Allocate moduli table. */
  if ((moduli = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    {
      goto failure;
    }

retry:

  /* Generate the random number within the interval. */
  ssh_mp_sub(&temp, max, min);
  bits = ssh_mp_get_size(&temp, 2);

  /* Generate suitable random number (some additional bits for perhaps
     more uniform distribution, these really shouldn't matter). */
  ssh_mp_random_integer(&aux, bits + 10);
  /* Compute. */
  ssh_mp_mod(&aux, &aux, &temp);
  ssh_mp_add(&pprime, &aux, min);

  /* Fix it as odd. */
  ssh_mp_set_bit(&pprime, 0);

  /* Compute the max difference. */
  ssh_mp_sub(&aux, max, &pprime);
  if (ssh_mp_cmp_ui(&aux, 0) < 0)
    goto retry;

  /* Get it. */
  max_difference = ssh_mp_get_ui(&aux);

  if (ssh_mprz_isnan(&pprime) || ssh_mprz_isnan(&aux))
    goto failure;

  /* Now we need to set up the moduli table. */
  for (i = 0; i < num_primes; i++)
    moduli[i] = ssh_mp_mod_ui(&pprime, prime_table[i]);

  /* Look for numbers that are not evenly divisible by any of the small
     primes. */
  for (difference = 0; ; difference += 2)
    {
      unsigned int i;

      if (difference > max_difference)
        /* Although we could just wrap around, we currently choose to
           just start from the scratch again. */
        goto retry;

      /* Check if it is a multiple of any small prime.  Note that this
         updates the moduli into negative values as difference grows. */
      for (i = 1; i < num_primes; i++)
        {
          while (moduli[i] + difference >= prime_table[i])
            moduli[i] -= prime_table[i];
          if (moduli[i] + difference == 0)
            break;
        }
      if (i < num_primes)
        continue; /* Multiple of a known prime. */

      /* Progress information. */
      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Compute the number in question. */
      ssh_mp_add_ui(ret, &pprime, difference);

      /* Perform Miller-Rabin strong pseudo primality tests */
      if (ssh_mprz_isnan(ret) || ssh_mp_is_probable_prime(ret, 20))
        break;
    }

  /* Found a (probable) prime.  It is in ret. */

  /* Sanity check, are we in the interval. */
  if (!ssh_mprz_isnan(ret) &&
      (ssh_mp_cmp(ret, min) <= 0 || ssh_mp_cmp(ret, max) >= 0))
    goto retry;

  /* Free the small prime moduli; they are no longer needed. */
  ssh_sieve_free(&sieve);
  ssh_free(moduli);
  ssh_free(prime_table);

  ssh_mp_clear(&pprime);
  ssh_mp_clear(&aux);
  ssh_mp_clear(&temp);
  /* Return value already set in ret. */
  return;

 failure:
  ssh_sieve_free(&sieve);
  ssh_free(moduli);
  ssh_free(prime_table);
  ssh_mp_clear(&pprime);
  ssh_mp_clear(&aux);
  ssh_mp_clear(&temp);
  ssh_mprz_makenan(ret, SSH_MP_NAN_ENOMEM);
}

/* The P1363 prime generation (from working draft i.e. might change in
   future). */

/* Generate random prime number using explicitly set limits. */

void ssh_mp_random_prime_within_limits(SshMPInt ret,
                                       int min_bits, int max_bits)
{
  SshMPIntStruct pprime, temp;
  SshSieveStruct sieve;
  SshWord *moduli, *prime_table, difference, num_primes, p;
  unsigned int i, len;
  Boolean divisible;

  /* Progress monitoring. */
  unsigned int progress_counter = 0;

  /* Verify that limits are in correct order. */
  if (min_bits >= max_bits)
    {
      /* Assume we still want random prime so get it but use more bits
         rather than less. */

      min_bits = max_bits;
      max_bits = min_bits + 2;
    }

  ssh_mp_init(&pprime);
  ssh_mp_init(&temp);

  /* Allocate a sieve. */
  ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                        SSH_GENMP_MAX_SIEVE_MEMORY);
  /* Don't count 2. */
  num_primes = ssh_sieve_prime_count(&sieve) - 1;

  /* Make a table of the primes. */
  if ((prime_table = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    return;
  for (p = 2, i = 0; p; p = ssh_sieve_next_prime(p, &sieve), i++)
    prime_table[i] = p;

  /* Allocate moduli table. */
  if ((moduli = ssh_malloc(num_primes * sizeof(SshWord))) == NULL)
    return;

retry:

  /* Get a random integer within limits. (Should not be too difficult,
     could be done also by setting the highest bit, but that approach was
     taken in the above code so doing this differently). */
  do {
    ssh_mp_random_integer(&pprime, max_bits);
    len = ssh_mp_get_size(&pprime, 2);
  } while (len < min_bits);

  /* If even the make it odd. */
  if ((ssh_mp_get_ui(&pprime) & 0x1) == 0)
    ssh_mp_add_ui(&pprime, &pprime, 1);

  /* Initialize moduli of the small primes with respect to the given
     random number. */
  for (i = 0; i < num_primes; i++)
    moduli[i] = ssh_mp_mod_ui(&pprime, prime_table[i]);

  /* Look for numbers that are not evenly divisible by any of the small
     primes. */
  difference = 0;

  while (1)
    {
      /* Set the divisible flag. */
      divisible = FALSE;

      /* In now and them add the difference to the probable prime. */
      if (difference > 1000)
        {
          ssh_mp_add_ui(&pprime, &pprime, difference);
          difference = 0;

          len = ssh_mp_get_size(&pprime, 2);
          if (len > max_bits)
            {
              ssh_mp_set_ui(&temp, 1);
              ssh_mp_mul_2exp(&temp, &temp, max_bits);
              ssh_mp_sub(&pprime, &pprime, &temp);

              ssh_mp_div_2exp(&temp, &temp, max_bits - min_bits);
              ssh_mp_add(&pprime, &pprime, &temp);
              ssh_mp_sub_ui(&pprime, &pprime, 1);

              /* Check that the probable prime is odd. */
              if ((ssh_mp_get_ui(&pprime) & 0x1) == 0)
                ssh_mp_add_ui(&pprime, &pprime, 1);

              /* Compute again the moduli table. */
              for (i = 0; i < num_primes; i++)
                moduli[i] = ssh_mp_mod_ui(&pprime, prime_table[i]);
            }
        }

      /* Check if it is a multiple of any small prime. */
      for (i = 0; i < num_primes; i++)
        {
          /* Check for this round. */
          if (moduli[i] == 0)
            divisible = TRUE;
          /* Compute for the next round. */
          moduli[i] += 2;
          if (moduli[i] >= prime_table[i])
            moduli[i] -= prime_table[i];
        }

      /* Add the difference by 2. */
      difference += 2;

      /* Multiple of known prime. */
      if (divisible)
        continue;

      /* Acknowledge application. */
      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Set to ret and check if gone over the max limit. */
      ssh_mp_add_ui(&pprime, &pprime, difference);
      difference = 0;

      /* Check the length. */
      len = ssh_mp_get_size(&pprime, 2);
      if (len > max_bits)
        {
          /* compute: pprime - 2^max_bits + 2^min_bits - 1 */
          ssh_mp_set_ui(&temp, 1);
          ssh_mp_mul_2exp(&temp, &temp, max_bits);
          ssh_mp_sub(&pprime, &pprime, &temp);
          ssh_mp_set_ui(&temp, 1);
          ssh_mp_mul_2exp(&temp, &temp, min_bits);
          ssh_mp_add(&pprime, &pprime, &temp);
          ssh_mp_sub_ui(&pprime, &pprime, 1);

          /* Check that the probable prime is odd. */
          if ((ssh_mp_get_ui(&pprime) & 0x1) == 0)
            ssh_mp_add_ui(&pprime, &pprime, 1);

          /* Compute again the moduli table. */
          for (i = 0; i < num_primes; i++)
            moduli[i] = ssh_mp_mod_ui(&pprime, prime_table[i]);
          continue;
        }

      /* Compute the number in question. */
      ssh_mp_set(ret, &pprime);

      /* Perform Miller-Rabin strong pseudo primality tests */
      if (ssh_mp_is_probable_prime(ret, 15))
        break;
    }

  /* Found a (probable) prime.  It is in ret. */

  /* Sanity check. */
  len = ssh_mp_get_size(ret, 2);
  if (len < min_bits || len > max_bits)
    goto retry;

  /* Free the small prime moduli; they are no longer needed. */
  ssh_free(moduli);
  ssh_free(prime_table);
  ssh_sieve_free(&sieve);

  ssh_mp_clear(&pprime);
  ssh_mp_clear(&temp);
  /* Return value already set in ret. */
}

/* Generate random prime number using explicitly set limits and
   a congruence condition. ret = a (mod r). This operation is
   rather slow. */

void ssh_mp_random_prime_with_congruence(SshMPInt ret,
                                         int min_bits, int max_bits,
                                         SshMPInt r, SshMPInt a)
{
  SshMPIntStruct pprime, temp, w, r2;
  unsigned int len;

  unsigned int progress_counter = 0;

  /* Verify that limits are in correct order. */
  if (min_bits >= max_bits)
    {
      /* Assume we still want random prime so get it but use more bits
         rather than less. */

      min_bits = max_bits;
      max_bits = min_bits + 2;
    }

  ssh_mp_init(&pprime);
  ssh_mp_init(&temp);
  ssh_mp_init(&w);
  ssh_mp_init(&r2);

retry:

  /* Get a random integer within limits. (Should not be too difficult,
     could be done also by setting the highest bit, but that approach was
     taken in the above code so doing this differently). */
  do {
    ssh_mp_random_integer(&pprime, max_bits);
    len = ssh_mp_get_size(&pprime, 2);
  } while (len < min_bits);

  ssh_mp_mul_ui(&r2, r, 2);
  ssh_mp_mod(&w, &pprime, &r2);

  ssh_mp_add(&pprime, &pprime, &r2);
  ssh_mp_add(&pprime, &pprime, a);
  ssh_mp_sub(&pprime, &pprime, &w);

  /* If even the make it odd. */
  if ((ssh_mp_get_ui(&pprime) & 0x1) == 0)
    ssh_mp_add(&pprime, &pprime, r);

  while (1)
    {
      ssh_mp_add(&pprime, &pprime, &r2);

      /* Check the length. */
      len = ssh_mp_get_size(&pprime, 2);
      if (len > max_bits)
        {
          /* compute: pprime - 2^max_bits + 2^min_bits - 1 */
          ssh_mp_set_ui(&temp, 1);
          ssh_mp_mul_2exp(&temp, &temp, max_bits);
          ssh_mp_sub(&pprime, &pprime, &temp);
          ssh_mp_set_ui(&temp, 1);
          ssh_mp_mul_2exp(&temp, &temp, min_bits);
          ssh_mp_add(&pprime, &pprime, &temp);
          ssh_mp_sub_ui(&pprime, &pprime, 1);

          /* Check that the probable prime is odd. */
          if ((ssh_mp_get_ui(&pprime) & 0x1) == 0)
            ssh_mp_add_ui(&pprime, &pprime, 1);

          ssh_mp_mod(&w, &pprime, &r2);

          ssh_mp_add(&pprime, &pprime, &r2);
          ssh_mp_add(&pprime, &pprime, a);
          ssh_mp_sub(&pprime, &pprime, &w);
          continue;
        }

      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Check for primality. */

      /* Compute the number in question. */
      ssh_mp_set(ret, &pprime);

      /* Perform Miller-Rabin strong pseudo primality tests */
      if (ssh_mp_is_probable_prime(ret, 15))
        break;
    }

  /* Sanity check. */
  len = ssh_mp_get_size(ret, 2);
  if (len < min_bits || len > max_bits)
    {
      goto retry;
    }
  ssh_mp_clear(&pprime);
  ssh_mp_clear(&temp);
  ssh_mp_clear(&w);
  ssh_mp_clear(&r2);
  /* Return value already set in ret. */
}

/* Generate strong random primes P1363 style. Where prime 'prime' satisfies
   prime = 1 (mod r), prime = -1 (mod s), r = 1 (mod t) and r, s, t are all
   large primes. Also 'div' = r. */

void ssh_mp_strong_p1363_random_prime(SshMPInt prime, SshMPInt div,
                                      int big_bits, int small_bits)
{
  SshMPIntStruct t, r, s, u, v, a, temp;
  unsigned int lt_bits, lr_bits, ls_bits;

  if (small_bits < 160 || big_bits < 320)
    ssh_fatal("error: discrete log might be too easy with primes (%d, %d).\n",
              big_bits, small_bits);

  if (small_bits > big_bits)
    big_bits = small_bits + 10;

  /* Assume that small_bits > 160. */
  lr_bits = small_bits;
  lt_bits = lr_bits - 10;
  ls_bits = small_bits;

  /* Initialize integers. */
  ssh_mp_init(&t);
  ssh_mp_init(&r);
  ssh_mp_init(&s);
  ssh_mp_init(&u);
  ssh_mp_init(&v);
  ssh_mp_init(&a);
  ssh_mp_init(&temp);

  ssh_mp_set_ui(&temp, 1);

  ssh_mp_random_prime_within_limits(&t, lt_bits - 1, lt_bits);
  ssh_mp_random_prime_with_congruence(&r, lr_bits - 1, lr_bits, &t, &temp);
  ssh_mp_random_prime_within_limits(&s, ls_bits - 1, ls_bits);

  /* Invert s (mod r) and r (mod s). */
  ssh_mp_mod_invert(&u, &s, &r);
  ssh_mp_mod_invert(&v, &r, &s);

  /* Compute a = su - rv (mod rs) */
  ssh_mp_mul(&a, &s, &u);
  ssh_mp_mul(&temp, &r, &v);
  ssh_mp_sub(&a, &a, &temp);

  ssh_mp_mul(&temp, &r, &s);
  ssh_mp_mod(&a, &a, &temp);

  ssh_mp_random_prime_with_congruence(prime, big_bits - 1, big_bits, &temp,
                                      &a);

  ssh_mp_set(div, &r);

  /* Free integers. */
  ssh_mp_clear(&t);
  ssh_mp_clear(&r);
  ssh_mp_clear(&s);
  ssh_mp_clear(&u);
  ssh_mp_clear(&v);
  ssh_mp_clear(&a);
  ssh_mp_clear(&temp);
}

/* Generate a strong random prime. That is, p = q * c + 1, where p and
   q are prime and c > 1.

   Here we use the idea that given random 2^n-1 < x < 2^n, we can
   compute y = x (mod 2q), and then p = x - y + 1 + 2tq. Given this
   method the probability that we get values that are not in the
   correct range is reasonably small. */

void ssh_mp_random_strong_prime(SshMPInt prime,
                                SshMPInt order,
                                int prime_bits, int order_bits)
{
  SshMPIntStruct aux, aux2, u;
  SshSieveStruct sieve;
  SshWord *table_q, *table_u, *prime_table, p;
  unsigned long i, j, table_count, upto;
  Boolean flag;

  unsigned int progress_counter = 0;

  /* Check for bugs. */
  if (prime_bits < order_bits)
    ssh_fatal("ssh_mp_random_strong_prime: "
              "requested prime less than the group order!");

  /* Keep the running in place. */
  if (prime_bits - order_bits - 1 > 24)
    upto = 1 << 24;
  else
    upto = 1 << (prime_bits - order_bits - 1);

  ssh_mp_init(&aux);
  ssh_mp_init(&aux2);
  ssh_mp_init(&u);

  /* There seems to be no real reason to generate this as a strong
     prime. */
  ssh_mp_random_prime(order, order_bits);

  /* Generate the sieve again (it was already generated in the random
     prime generation code), but it shouldn't be too slow. */
  ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                        SSH_GENMP_MAX_SIEVE_MEMORY);
  /* Compute the number of primes. */
  table_count = ssh_sieve_prime_count(&sieve) - 1;

  /* Generate a suitable table of primes. */
  if ((prime_table = ssh_malloc(table_count * sizeof(SshWord))) == NULL)
    return;

  for (p = 2, i = 0; p; p = ssh_sieve_next_prime(p, &sieve), i++)
    prime_table[i] = p;

  /* Reduce group order. Remember the factor 2. */
  if ((table_q = ssh_malloc(table_count * sizeof(SshWord) * 2)) == NULL)
    {
      ssh_free(prime_table);
      return;
    }
  table_u = table_q + table_count;
  for (i = 0; i < table_count; i++)
    {
      table_q[i] =
        (ssh_mp_mod_ui(order, prime_table[i]) * 2) % prime_table[i];
    }

  /* In case we don't find one quickly enough. */
retry:

  /* Generate a random integer large enough. */
  ssh_mp_random_integer(&u, prime_bits);

  /* Set the highest bit on. */
  ssh_mp_set_ui(&aux, 1);
  ssh_mp_mul_2exp(&aux, &aux, prime_bits - 1);
  ssh_mp_or(&u, &u, &aux);

  /* Compute the initial value for the prime. */
  ssh_mp_set(&aux, order);
  ssh_mp_mul_2exp(&aux, &aux, 1);
  ssh_mp_mod(&aux2, &u, &aux);
  ssh_mp_sub(&u, &u, &aux2);
  ssh_mp_add_ui(&u, &u, 1);

  /* Now check whether the value is still large enough. */
  if (ssh_mp_get_size(&u, 2) <= prime_bits - 1)
    goto retry;

  /* Now compute the residues of the 'probable prime'. */
  for (j = 0; j < table_count; j++)
    table_u[j] = ssh_mp_mod_ui(&u, prime_table[j]);

  /* Set the 2*q for  later. */
  ssh_mp_mul_2exp(&aux2, order, 1);

  /* Loop through until a prime is found. */
  for (i = 0; i < upto; i++)
    {
      unsigned long cur_p, value;

      flag = TRUE;
      for (j = 1; j < table_count; j++)
        {
          cur_p = prime_table[j];
          value = table_u[j];

          /* Check if the result seems to indicate divisible value. */
          if (value >= cur_p)
            value -= cur_p;
          if (value == 0)
            flag = FALSE;
          /* For the next round compute. */
          table_u[j] = value + table_q[j];
        }

      if (flag != TRUE)
        continue;

      /* Acknowledge application that again one possibly good value was
         found. */
      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Compute the proposed prime. */
      ssh_mp_set(prime, &u);
      ssh_mp_mul_ui(&aux, &aux2, i);
      ssh_mp_add(prime, prime, &aux);

      /* Check that the size of the prime is within range. */
      if (ssh_mp_get_size(prime, 2) > prime_bits)
        goto retry;

      /* Miller-Rabin */
      if (ssh_mp_is_probable_prime(prime, 20))
        break;
    }

  if (i >= upto)
    goto retry;

  /* Free the moduli tables. */
  ssh_free(table_q);
  ssh_free(prime_table);
  ssh_sieve_free(&sieve);

  /* Free temporary memory. */
  ssh_mp_clear(&aux);
  ssh_mp_clear(&aux2);
  ssh_mp_clear(&u);
}

/* Method for computing a prime that is resistant against p-1 and p+1
   methods of factoring.

   As suggested by John Krueger at sci.crypt (31 Jul 1997).

   The improvement made over Kruegers method is to compute chinese remainder
   theorem so that

     x =  1 mod q1
     x = -1 mod q2
     x =  1 mod 2

   where 1 <= x < q1*q2*2.

   Last conqruence, of course, asserts that we don't need to change q1 and
   q2, i.e. there should be number of form

     t*(q1*q2*2) + x

   which is prime for some t. Hopefully t need not be too large.

   */

void ssh_mp_random_safe_prime(SshMPInt p,
                              SshMPInt q1,
                              SshMPInt q2,
                              unsigned int bits)
{
  SshMPIntStruct t1, t2, t3, y1, y2, y3, m1, m2, m3, q3, qq;
  SshWord *table_v, *table_u, *prime_table, small_prime;
  SshSieveStruct sieve;
  unsigned int table_count, i, j;
  unsigned int upto = (1 << 30);
  Boolean flag;

  unsigned int progress_counter = 0;

  /* Initialize a few temporary variables. */
  ssh_mp_init(&t1);
  ssh_mp_init(&t2);
  ssh_mp_init(&t3);
  ssh_mp_init(&m1);
  ssh_mp_init(&m2);
  ssh_mp_init(&m3);
  ssh_mp_init(&y1);
  ssh_mp_init(&y2);
  ssh_mp_init(&y3);
  ssh_mp_init(&qq);
  ssh_mp_init(&q3);

  /* Generate the sieve, but it shouldn't be too slow. */
  ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                        SSH_GENMP_MAX_SIEVE_MEMORY);
  /* Compute the number of primes. */
  table_count = ssh_sieve_prime_count(&sieve) - 1;

  /* Generate a suitable table of primes. */
  if ((prime_table = ssh_malloc(table_count * sizeof(SshWord))) == NULL)
    return;
  for (small_prime = 3, i = 0; small_prime;
       small_prime = ssh_sieve_next_prime(small_prime, &sieve), i++)
    prime_table[i] = small_prime;

  /* Allocate tables. */
  if ((table_v = ssh_malloc(table_count * sizeof(table_v[0]) * 2)) == NULL)
    {
      ssh_free(prime_table);
      return;
    }
  table_u = table_v + SSH_MAX_PRIMES_IN_TABLE;

  /* Using chinese remainder theorem generate t1 = 1 mod q1, t1 = -1 mod q2.
     Also we'd like to make sure that t1 = 1 mod 2. */

  /* Generate two large primes. */
  ssh_mp_random_prime(q1, (bits/2));
  ssh_mp_random_prime(q2, (bits/2));

  /* Compute modulus. */
  ssh_mp_mul(&m3, q1, q2);

  /* q3 = 2, thus q1*q2 mod 2 == 1. */
  if ((ssh_mp_get_ui(&m3) & 0x1) == 0)
    ssh_fatal("ssh_mp_random_safe_prime: prime equals to 2.");

  ssh_mp_mul_ui(&qq, &qq, 2);

  ssh_mp_mul_ui(&m1, q2, 2);
  ssh_mp_mul_ui(&m2, q1, 2);

  ssh_mp_set_ui(&q3, 2);

  /* Compute inverses. */
  ssh_mp_mod_invert(&y1, &m1, q1);
  ssh_mp_mod_invert(&y2, &m2, q2);

  /* Compute first part. */
  ssh_mp_mul(&t1, &m1, &y1);

  /* Compute second part. */
  ssh_mp_mul(&t2, &m2, &y2);
  ssh_mp_sub_ui(&t3, q1, 1);
  ssh_mp_mul(&t2, &t2, &t3);
  ssh_mp_mod(&t2, &t2, &qq);

  /* Combine. */
  ssh_mp_add(&t1, &t1, &t2);
  ssh_mp_add(&t1, &t1, &m3);
  ssh_mp_mod(&t1, &t1, &qq);

  /* We never should have to deal with cases like this. */
  if ((ssh_mp_get_ui(&t1) & 0x1) == 0)
    ssh_fatal("ssh_mp_random_safe_prime: should never be divisible by 2!");

  /* Next search for number of form l + t1 which is a prime where
     l = c*qq.

     We can again use small primes to get rid of values that are not
     prime, and then Fermats little theorem etc. */

  /* Following generate a table where

     v[i] = t1 % p[i],
     u[i] = qq % p[i],

     which can be used for quick checks. */

  /* For simplicity we'd like to work only with values > qq. */
  ssh_mp_add(&t1, &t1, &qq);

  /* Compute table values. */
  for (i = 0; i < table_count; i++)
    {
      table_v[i] = ssh_mp_mod_ui(&t1, prime_table[i]);
      table_u[i] = ssh_mp_mod_ui(&qq, prime_table[i]);
    }

  /* Search for a prime. */
  for (i = 0; i < upto; i++)
    {
      flag = TRUE;
      for (j = 0; j < table_count; j++)
        {
          /* Check if the result seems to indicate divisible value. */
          if (table_v[j] == 0)
            flag = FALSE;
          /* For the next round compute. */
          table_v[j] += table_u[j];
          if (table_v[j] >= prime_table[j])
            table_v[j] -= prime_table[j];
        }

      if (flag != TRUE)
        continue;

      ssh_crypto_progress_monitor(SSH_CRYPTO_PRIME_SEARCH,
                                  ++progress_counter);

      /* Compute the proposed prime. */
      ssh_mp_mul_ui(p, &qq, i);
      ssh_mp_add(p, p, &t1);

      /* Miller-Rabin */
      if (ssh_mp_is_probable_prime(p, 20))
        break;
    }

  /* Free tables. */
  ssh_free(table_v);
  ssh_free(prime_table);
  ssh_sieve_free(&sieve);

  ssh_mp_clear(&t1);
  ssh_mp_clear(&t2);
  ssh_mp_clear(&t3);
  ssh_mp_clear(&m1);
  ssh_mp_clear(&m2);
  ssh_mp_clear(&m3);
  ssh_mp_clear(&y1);
  ssh_mp_clear(&y2);
  ssh_mp_clear(&y3);
  ssh_mp_clear(&qq);
  ssh_mp_clear(&q3);
}

/* Basic modular enhancements. Due the nature of extended euclids algorithm
   it sometimes returns integers that are negative. For our cases positive
   results are better. */

int ssh_mp_mod_invert(SshMPInt op_dest, const SshMPInt op_src,
                      const SshMPInt modulo)
{
  int status;

  status = ssh_mp_invert(op_dest, op_src, modulo);

  if (ssh_mp_cmp_ui(op_dest, 0) < 0)
    ssh_mp_add(op_dest, op_dest, modulo);

  return status;
}

/* Check whether op_src is of order op_ord (mod modulo). Not used and not
   tested. */

int ssh_mp_is_order(const SshMPInt op_ord, const SshMPInt op_src,
                    const SshMPInt modulo)
{
  SshMPIntStruct t, t1;
  SshSieveStruct sieve;
  SshWord p;
  int is_order = 1;

  /* Initialize t and t1 */
  ssh_mp_init(&t);
  ssh_mp_init(&t1);

  ssh_mp_powm(&t, op_src, op_ord, modulo);
  if (ssh_mp_cmp_ui(&t, 1) != 0)
    {
      is_order = 0;
      goto end;
    }

  ssh_sieve_allocate_ui(&sieve, SSH_GENMP_MAX_PRIME,
                        SSH_GENMP_MAX_SIEVE_MEMORY);

  /* Trial division factoring algorithm... this shouldn't need better (?) */
  ssh_mp_set(&t, op_ord);
  for (p = 2; p; p = ssh_sieve_next_prime(p, &sieve))
    {
      /* Check whether op_src is divisible by a prime... */
      if (ssh_mp_mod_ui(&t, p) == 0)
        {
          /* This really isn't necessary but speeds up possibly a bit. */
          do
            {
              ssh_mp_div_ui(&t, &t, p);
            }
          while (ssh_mp_mod_ui(&t1, p) == 0);

          ssh_mp_powm_expui(&t, op_src, p, modulo);
          if (ssh_mp_cmp_ui(&t, 1) == 0)
            {
              is_order = 0;
              break;
            }
        }
    }
  ssh_sieve_free(&sieve);

end:

  ssh_mp_clear(&t);
  ssh_mp_clear(&t1);

  /* Could be of the order of op_ord */
  return is_order;
}

/* Find a random generator of order 'order' modulo 'modulo'. */

Boolean ssh_mp_random_generator(SshMPInt g, SshMPInt order, SshMPInt modulo)
{
  SshMPIntStruct aux, t;
  int bits;

  ssh_mp_init(&aux);
  ssh_mp_init(&t);

  ssh_mp_sub_ui(&aux, modulo, 1);
  ssh_mp_mod(&t, &aux, order);

  if (ssh_mp_cmp_ui(&t, 0) != 0)
    {
      ssh_mp_clear(&aux);
      ssh_mp_clear(&t);
      return FALSE;
    }

  ssh_mp_div_q(&t, &aux, order);
  bits = ssh_mp_get_size(modulo, 2);

  while (1)
    {
      ssh_mp_random_integer(g, bits);
      ssh_mp_mod(g, g, modulo);
      ssh_mp_powm(g, g, &t, modulo);

      if (ssh_mp_cmp_ui(g, 1) != 0)
        break;
    }

  /* Check. */
  ssh_mp_powm(&aux, g, order, modulo);
  if (ssh_mp_cmp_ui(&aux, 1) != 0)
    {
      ssh_mp_clear(&aux);
      ssh_mp_clear(&t);
      return FALSE;
    }

  ssh_mp_clear(&aux);
  ssh_mp_clear(&t);

  return TRUE;
}


/* genmp-prime.c */
