#include "sshincludes.h"
#include "sshmp.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshcstack.h"
#include "dlfix.h"
#include "dlglue.h"
#include "dl-internal.h"
#include "sshgenmp.h"

SshDLStackRandomizer *
ssh_cstack_SshDLStackRandomizer_constructor(void *context);

/* Precompute randomizer with parameters only, private key and public key. */

Boolean ssh_dlp_param_generate_randomizer(void *parameters)
{
  /* Allocate stack element with constructor! */
  SshDLStackRandomizer *stack =
    ssh_cstack_SshDLStackRandomizer_constructor(NULL);
  SshDLParam param = parameters;

  if (!stack)
    return FALSE;

retry:
  /* Add information to stack. */
  if (param->exponent_entropy)
    ssh_mp_mod_random_entropy(&stack->k, &param->q,
                              param->exponent_entropy);
  else
    ssh_mp_mod_random(&stack->k, &param->q);
  if (ssh_mp_cmp_ui(&stack->k, 0) == 0)
    goto retry;
  if (param->base_defined == FALSE)
    ssh_mp_powm(&stack->gk, &param->g, &stack->k, &param->p);
  else
    ssh_mprz_powm_with_precomp(&stack->gk, &stack->k, param->base);

  /* Push to stack list, in parameter context. No it is visible for
     all, private keys, public keys and parameters. */
  ssh_cstack_push(&param->stack, stack);
  return TRUE;
}


void *ssh_dlp_param_generate(int bits, int small_bits)
{
  SshDLParam param, temp;

  if ((param = ssh_malloc(sizeof(*param))) == NULL)
    return NULL;

  ssh_dlp_init_param(param);

  ssh_mp_random_strong_prime(&param->p, &param->q, bits, small_bits);

  if (ssh_mp_random_generator(&param->g, &param->q, &param->p) != TRUE)
    {
      ssh_dlp_clear_param(param);
      ssh_free(param);
      return NULL;
    }

  temp = ssh_dlp_param_list_add(param);
  if (temp)
    {
      ssh_dlp_clear_param(param);
      ssh_free(param);
      param = temp;
    }

  return (void *)param;
}

void *ssh_dlp_private_key_action_generate(void *context)
{
  SshDLPInitCtx *ctx = context;
  SshDLParam param;

  /* First generate paramters. */
  if (!ctx->predefined)
    {
      if (ssh_mp_cmp_ui(&ctx->p, 0) == 0 ||
          ssh_mp_cmp_ui(&ctx->q, 0) == 0 ||
          ssh_mp_cmp_ui(&ctx->g, 0) == 0)
        {
          if (ctx->size)
            {
              unsigned int q_size;

              /* For DSA force subprime size to 160 bits, for others
                 make it half of size. That should depend on selected
                 policy, but seems to be pretty good tradeoff here. */
              if ((ctx->flag & DLP_FLAG_DSA) == DLP_FLAG_DSA)
                q_size = 160;
              else
                q_size = ctx->size / 2;

              param = ssh_dlp_param_generate(ctx->size, q_size);
            }
          else
            return NULL;
        }
      else
        {
          if ((param = ssh_dlp_param_create(&ctx->p, &ctx->q, &ctx->g))
              == NULL)
            return NULL;
        }
    }
  else
    {
      if ((param = ssh_dlp_param_create_predefined(ctx->predefined)) == NULL)
        return NULL;
    }

  /* Then maybe generate private key components. */
  if (ssh_mp_cmp_ui(&ctx->x, 0) == 0 || ssh_mp_cmp_ui(&ctx->y, 0) == 0)
    {
      /* Generate secret key. Note, here we definitely don't want to
         use the restriction of random number size for the exponent.
         It would be a poor practice, some attack could find the
         discrete log faster that way.

         Well, that isn't the main point however, just that in
         Diffie-Hellman and signatures you are mainly using for short
         term security, but private keys might last for a long
         while. Thus for sake of clarity we don't do any restrictions
         here. */
      if (ssh_mp_cmp_ui(&ctx->x, 0) == 0)
        ssh_mp_mod_random(&ctx->x, &param->q);

      /* Compute public key. */
      if (param->base_defined)
        ssh_mprz_powm_with_precomp(&ctx->y, &ctx->x, param->base);
      else
        ssh_mp_powm(&ctx->y, &param->g, &ctx->x, &param->p);
    }

  return ssh_dlp_action_make(context, param, 2);
}
