#include "sshincludes.h"
#include "sshmp.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshcstack.h"
#include "dlfix.h"
#include "dlglue.h"
#include "sshencode.h"
#include "dl-internal.h"
#include "sshgenmp.h"

#ifdef SSHDIST_CRYPT_DSA

/* DSA - Digital Signature Algorithm */

Boolean
ssh_dlp_dsa_public_key_verify(const void *public_key,
                              const unsigned char *signature,
                              size_t signature_len,
                              SshRGFHash hash)
{
  const SshDLPublicKey *pub_key = public_key;
  unsigned int len = ssh_mp_byte_size(&pub_key->param->q);
  unsigned int vlen;
  SshMPIntStruct v, w, s, r, e, invs, u1, u2;
  unsigned char digest[SSH_MAX_HASH_DIGEST_LENGTH];
  /* Assume failure. */
  Boolean rv = FALSE;

  if (signature_len & 1)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  vlen = signature_len / 2;
  if (vlen > len)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  ssh_mp_init(&v);
  ssh_mp_init(&w);
  ssh_mp_init(&e);
  ssh_mp_init(&s);
  ssh_mp_init(&r);
  ssh_mp_init(&u1);
  ssh_mp_init(&u2);
  ssh_mp_init(&invs);

  /* Verify the signature. */
  if (ssh_rgf_hash_sign(hash, digest, len) != SSH_RGF_OK)
    {
      rv = FALSE;
      goto failed;
    }

  /* Reduce to correct length. */
  ssh_buf_to_mp(&e, digest, len);
  ssh_mp_mod(&e, &e, &pub_key->param->q);

  /* Convert and reduce signature. */
  ssh_buf_to_mp(&r, signature, vlen);
  if (ssh_mp_cmp(&r, &pub_key->param->q) >= 0 ||
      ssh_mp_cmp_ui(&r, 0) <= 0)
    {
      rv = FALSE;
      goto failed;
    }

  ssh_buf_to_mp(&s, signature + vlen, vlen);
  if (ssh_mp_cmp(&s, &pub_key->param->q) >= 0 ||
      ssh_mp_cmp_ui(&s, 0) <= 0)
    {
      rv = FALSE;
      goto failed;
    }

  /* Compute verification parameters:

  g^(k(m + rx)^-1 * m) * g^(x*k(m + rx)^-1 * r)) =
     g^k((m + rx)^-1 * m + (m + rx)^-1 * x * r) =
     g^k((m + rx)^-1 * (m + rx)) = g^k.

   */

  ssh_mp_mod_invert(&invs, &s, &pub_key->param->q);
  ssh_mp_mul(&u1, &invs, &e);
  ssh_mp_mod(&u1, &u1, &pub_key->param->q);
  ssh_mp_mul(&u2, &invs, &r);
  ssh_mp_mod(&u2, &u2, &pub_key->param->q);

  /* Exponentiate. */
  ssh_mp_powm_gg(&v, &pub_key->param->g, &u1,
                 &pub_key->y, &u2, &pub_key->param->p);
  ssh_mp_mod(&v, &v, &pub_key->param->p);
  ssh_mp_mod(&v, &v, &pub_key->param->q);

  /* Check validy. If and only if v = r then successful. */
  if (ssh_mp_cmp(&v, &r) == 0)
    rv = TRUE;

failed:
  /* Clean memory. */
  ssh_mp_clear(&v);
  ssh_mp_clear(&w);
  ssh_mp_clear(&e);
  ssh_mp_clear(&s);
  ssh_mp_clear(&r);
  ssh_mp_clear(&invs);
  ssh_mp_clear(&u1);
  ssh_mp_clear(&u2);

  return rv;
}

size_t
ssh_dlp_dsa_private_key_max_signature_input_len(const void *private_key)
{
  return (size_t)-1;
}

size_t
ssh_dlp_dsa_private_key_max_signature_output_len(const void *private_key)
{
  const SshDLPrivateKey *prv_key = private_key;
  return ssh_mp_byte_size(&prv_key->param->q) * 2;
}

Boolean
ssh_dlp_dsa_private_key_sign(const void *private_key,
                             SshRGFHash hash,
                             unsigned char *signature_buffer,
                             size_t ssh_buffer_len,
                             size_t *signature_length_return)
{
  const SshDLPrivateKey *prv_key = private_key;
  SshDLStackRandomizer *stack;
  SshMPIntStruct k, e, r, invk, s;
  unsigned int len = ssh_mp_byte_size(&prv_key->param->q);
  unsigned char digest[SSH_MAX_HASH_DIGEST_LENGTH];

  if (ssh_buffer_len < len * 2)
    {
      ssh_rgf_hash_free(hash);
      return FALSE;
    }

  if (ssh_rgf_hash_sign(hash, digest, len) != SSH_RGF_OK)
    {
      return FALSE;
    }

  ssh_mp_init(&k);
  ssh_mp_init(&e);
  ssh_mp_init(&r);
  ssh_mp_init(&invk);
  ssh_mp_init(&s);

  /* Reduce */
  ssh_buf_to_mp(&e, digest, len);
  ssh_mp_mod(&e, &e, &prv_key->param->q);

retry0:

  stack = (SshDLStackRandomizer *)
    ssh_cstack_pop(&prv_key->param->stack, SSH_DLP_STACK_RANDOMIZER);

  /* Check if in stack. */
  if (!stack)
    {
      /* In case we hit to failure cases. */
    retry1:

      /* Find the randomizer. The use of restrictions for the size of
         the exponent work here. However, you should be very careful
         with it. */
      if (prv_key->param->exponent_entropy)
        ssh_mp_mod_random_entropy(&k, &prv_key->param->q,
                                  prv_key->param->exponent_entropy);
      else
        ssh_mp_mod_random(&k, &prv_key->param->q);

      if (ssh_mp_cmp_ui(&k, 0) == 0)
        goto retry1;

      /* Check if we have done any precomputation. */
      if (prv_key->param->base_defined)
        ssh_mp_powm_with_base(&r, &k, prv_key->param->base);
      else
        ssh_mp_powm(&r, &prv_key->param->g, &k, &prv_key->param->p);
    }
  else
    {
      ssh_mp_set(&k, &stack->k);
      ssh_mp_set(&r, &stack->gk);
      /* This is legal, uses the destructor we have defined. */
      ssh_cstack_free(stack);
    }

  /* Compute: r = (g^(k mod q) mod p) mod q */
  ssh_mp_mod(&r, &r, &prv_key->param->q);
  if (ssh_mp_cmp_ui(&r, 0) == 0)
    goto retry0;

  /* Invert. */
  ssh_mp_mod_invert(&invk, &k, &prv_key->param->q);

  /* Compute signature s = k^-1(e + xr). */
  ssh_mp_mul(&s, &r, &prv_key->x);
  ssh_mp_add(&s, &s, &e);
  ssh_mp_mul(&s, &s, &invk);
  ssh_mp_mod(&s, &s, &prv_key->param->q);

  if (ssh_mp_cmp_ui(&s, 0) == 0)
    goto retry0;

  /* Linearize signature. */
  ssh_mp_to_buf(signature_buffer, len, &r);
  ssh_mp_to_buf(signature_buffer + len, len, &s);
  *signature_length_return = len * 2;

  /* Clear temps. */
  ssh_mp_clear(&k);
  ssh_mp_clear(&e);
  ssh_mp_clear(&r);
  ssh_mp_clear(&invk);
  ssh_mp_clear(&s);

  return TRUE;
}

void ssh_dlp_dsa_nist(void *context)
{
  SshDLPInitCtx *ctx = context;
  ctx->flag |= DLP_FLAG_DSA;
}
#endif /* SSHDIST_CRYPT_DSA */
