#include "sshincludes.h"
#include "sshmp.h"
#include "sshcrypt.h"
#include "sshpk.h"
#include "sshcstack.h"
#include "dlfix.h"
#include "dlglue.h"
#include "sshencode.h"
#include "dl-internal.h"
#include "sshgetput.h"
#include "sshgenmp.h"

/************************ Key exchange **************************/

#ifdef SSHDIST_CRYPT_DH

void *ssh_dlp_mp_out(SshMPInt k)
{
  unsigned char *buf;
  unsigned int len = ssh_mp_byte_size(k);

  if ((buf = ssh_malloc(len + 4)) != NULL)
    {
      SSH_PUT_32BIT(buf, len);
      ssh_mp_to_buf(buf + 4, len, k);
    }
  return buf;
}

void ssh_dlp_mp_in(SshMPInt k, void *ptr)
{
  unsigned char *buf = ptr;
  unsigned int len;

  len = SSH_GET_32BIT(buf);
  ssh_buf_to_mp(k, buf + 4, len);
}

/* Diffie-Hellman */
size_t
ssh_dlp_diffie_hellman_exchange_length(const void *parameters)
{
  const SshDLParamStruct *param = parameters;
  return ssh_mp_byte_size(&param->p);
}

size_t
ssh_dlp_diffie_hellman_shared_secret_length(const void *parameters)
{
  const SshDLParamStruct *param = parameters;
  return ssh_mp_byte_size(&param->p);
}

void ssh_dlp_diffie_hellman_internal_generate(SshMPInt ret,
                                              SshDLParam param,
                                              SshMPInt k)
{
  SshDLStackRandomizer *stack_r;

  stack_r = (SshDLStackRandomizer *)ssh_cstack_pop(&param->stack,
                                                   SSH_DLP_STACK_RANDOMIZER);
  if (!stack_r)
    {
      /* This is the main place where the entropy limitation will
         be very useful. Usually Diffie-Hellman session keys are for
         short term use, and are not used for stuff that needs to
         be secure forever. Thus smaller amount of entropy is suitable. */
      if (param->exponent_entropy)
        ssh_mp_mod_random_entropy(k, &param->q,
                                  param->exponent_entropy);
      else
        ssh_mp_mod_random(k, &param->q);

      ssh_mp_powm(ret, &param->g, k, &param->p);
    }
  else
    {
      ssh_mp_set(ret, &stack_r->gk);
      ssh_mp_set(k, &stack_r->k);
      ssh_cstack_free(stack_r);
    }
}

Boolean ssh_dlp_diffie_hellman_generate(const void *parameters,
                                        void **diffie_hellman,
                                        unsigned char *exchange,
                                        size_t exchange_length,
                                        size_t *return_length)
{
  const SshDLParamStruct *param = parameters;
  SshMPIntStruct e;
  SshMPIntStruct k;
  unsigned int len = ssh_mp_byte_size(&param->p);

  if (exchange_length < len)
    return FALSE;

  ssh_mp_init(&k);
  ssh_mp_init(&e);

  ssh_dlp_diffie_hellman_internal_generate(&e, (SshDLParam )param, &k);

  /* Linearize. */
  ssh_mp_to_buf(exchange, len, &e);
  *return_length = len;

  ssh_mp_clear(&e);

  *diffie_hellman = ssh_dlp_mp_out(&k);
  ssh_mp_clear(&k);

  return TRUE;
}

Boolean 
ssh_dlp_diffie_hellman_internal_final(SshMPInt ret,
                                      SshMPInt input,
                                      const SshDLParamStruct *param,
                                      SshMPInt k)

{
  SshMPIntStruct t;
  /* Perform some checks. */
  if (ssh_mp_cmp_ui(ret, 0) <= 0 ||
      ssh_mp_cmp(ret, &param->p) >= 0)
    return FALSE;

  /* Get a temporary variable. */
  ssh_mp_init(&t);

  /* Remark. We probably should add here the more general subgroup
     checks. The subgroup check could be interleaved with the actual
     Diffie-Hellman part. However, that would definitely be slower
     than just one exponentiation. */

  ssh_mp_square(&t, ret);
  ssh_mp_mod(&t, &t, &param->p);
  /* Check for trivial subgroup of 2. */
  if (ssh_mp_cmp_ui(&t, 1) == 0)
    {
      ssh_mp_clear(&t);
      return FALSE;
    }
  ssh_mp_clear(&t);

  /* Diffie-Hellman part. */
  ssh_mp_powm(ret, ret, k, &param->p);
  return TRUE;
}

Boolean ssh_dlp_diffie_hellman_final(const void *parameters,
                                     void *diffie_hellman,
                                     const unsigned char *exchange,
                                     size_t exchange_length,
                                     unsigned char *secret,
                                     size_t secret_length,
                                     size_t *return_length)
{
  const SshDLParamStruct *param = parameters;
  SshMPIntStruct v, k;
  unsigned int len = ssh_mp_byte_size(&param->p);

  if (secret_length < len)
    return FALSE;

  ssh_mp_init(&v);
  ssh_mp_init(&k);

  /* Import the secret. */
  ssh_dlp_mp_in(&k, diffie_hellman);
  ssh_buf_to_mp(&v, exchange, exchange_length);

  /* Compute v further. */
  if (ssh_dlp_diffie_hellman_internal_final(&v, &v, param, &k) == FALSE)
    {
      ssh_mp_clear(&v);
      ssh_mp_clear(&k);
      ssh_free(diffie_hellman);
      return FALSE;
    }

  ssh_free(diffie_hellman);
  ssh_mp_clear(&k);

  /* Linearize. */
  ssh_mp_to_buf(secret, len, &v);
  *return_length = len;

  /* Clear memory. */
  ssh_mp_clear(&v);
  return TRUE;
}

/* Unified Diffie-Hellman (used after first part of standard Diffie-Hellman) */

size_t
ssh_dlp_unified_diffie_hellman_shared_secret_length(const void *parameters)
{
  const SshDLParamStruct *param = parameters;
  return ssh_mp_byte_size(&param->p) * 2;
}

Boolean ssh_dlp_unified_diffie_hellman_final(const void *public_key,
                                             const void *private_key,
                                             void *diffie_hellman,
                                             const unsigned char *exchange,
                                             size_t exchange_length,
                                             unsigned char *secret,
                                             size_t secret_length,
                                             size_t *return_length)
{
  const SshDLPrivateKey *prv_key = private_key;
  const SshDLPublicKey *pub_key = public_key;
  SshMPIntStruct v, w, k;
  unsigned int len = ssh_mp_byte_size(&prv_key->param->p);

  if (exchange_length < len)
    return FALSE;
  if (secret_length < len)
    return FALSE;

  ssh_mp_init(&v);
  ssh_mp_init(&k);

  ssh_dlp_mp_in(&k, diffie_hellman);

  /* Diffie-Hellman in its basic form. */
  ssh_buf_to_mp(&v, exchange, len);

  if (ssh_dlp_diffie_hellman_internal_final(&v, &v,
                                            prv_key->param,
                                            &k) != TRUE)
    {
      ssh_mp_clear(&v);
      ssh_mp_clear(&k);
      ssh_free(diffie_hellman);
      return FALSE;
    }

  ssh_free(diffie_hellman);
  ssh_mp_clear(&k);

  ssh_mp_init(&w);

  /* Unified Diffie-Hellman part. */
  ssh_mp_powm(&w, &pub_key->y, &prv_key->x, &prv_key->param->p);

  /* Linearize (this _could_ feature some sort of hashing but we assume
     it could be left for higher level). */
  ssh_mp_to_buf(secret, len, &v);
  ssh_mp_to_buf(secret + len, len, &w);

  *return_length = len * 2;

  ssh_mp_clear(&v);
  ssh_mp_clear(&w);

  return TRUE;
}

#endif /* SSHDIST_CRYPT_DH */
