/*

File: ansirand.c

Author: Patrick Irwin <irwin@ssh.fi>

Copyright (C) 2001-2002 SSH Communications Security Corp
All rights reserved.

Created: Tue Nov 13  14:54:31 2001 [irwin]

*/

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshgetput.h"
#include "sshhash/sshhash.h" 
#include "sshhash/sha256.h"
#include "sshcipher/des.h"

#ifdef WITH_ANSI_RNG

#define SSH_DEBUG_MODULE "AnsiRand"

/* This module implements the ANSI X9.17 Random Number Generator (RNG).
   Output blocks (consisting of 8 bytes) are obtained using three triple 
   DES (3-DES) encryptions. The RNG requires as its seed material a 24 byte 
   3-DES key and a special 8 byte seed block. In total this gives 32 
   bytes of seed material, which must be externally supplied noise of high 
   entropy. The security of the RNG is dependent on the quality of this 
   external noise. 
   
   To produce each output block, a timestamp is needed as input, it is 
   encrypted with the secret key and then XORed with the present seed 
   value, and finally encrypted again to give the output block. The seed 
   is updated by XORing the output block with the encrypted timestamp and 
   encrypting. See the ANSI standards for a more detailed description.

   The interfaces conform with those of the SSH random number generator, 
   genrand.c, namely the ssh_random_add_noise(), ssh_random_stir(), 
   ssh_random_get_byte() and ssh_random_free() functions. However the 
   details of the two random number generators are quite different.
   External noise is added using the function ssh_random_noise(). Unlike 
   the SSH random number generator, the added noise does not become 
   effective until ssh_random_stir() is called, which reseeds the RNG. 
   
   To preserve the interfaces between the two diffrerent RNG's, the 
   ssh_random_add_noise() function takes as input an arbitrarily long 
   buffer. This is then hashed (using SHA-256) to give a digest of 32 
   bytes which is then used to seed the 3-DES key and the 8 byte seed block. 
*/

#define KEY_BYTES    24
#define BLOCK_BYTES   8

/* SshRandomStateRec represents a generic random state structure. */

typedef struct SshRandomStateRec { 
  /* Holds the hash digest from SHA-256. */
  unsigned char digest[KEY_BYTES + BLOCK_BYTES];
  /* The first KEY_BYTES bytes of state are used to key 3-DES, the
     remaining BLOCK_BYTES bytes is the seed variable. */
  unsigned char state[KEY_BYTES + BLOCK_BYTES];
  unsigned char enc_time[BLOCK_BYTES]; 
  unsigned char   output[BLOCK_BYTES]; 
  size_t next_available_byte;
  void  *cipher; 
  void  *hash;   
} *SshRandomState;


/* The global random state. */
SshRandomState ssh_global_random_state = NULL;

SshRandomState ssh_random_get_global(void)
{
  if (ssh_global_random_state == NULL)
    {
      size_t ctx_len, cipher_len, hash_len;
      
      ctx_len    = sizeof(*ssh_global_random_state);
      cipher_len = ssh_des3_ctxsize();
      hash_len   = ssh_sha256_ctxsize();
      
      /* First allocate memory for the random state context. */ 
      if ((ssh_global_random_state = ssh_malloc(ctx_len)) == NULL)
        return NULL;
      
      ssh_global_random_state->cipher = NULL;
      ssh_global_random_state->hash   = NULL;
      
      /* Allocate memory for the 3-DES cipher context. */
      if ((ssh_global_random_state->cipher = ssh_malloc(cipher_len)) 
          == NULL)
        goto failure; 
      
      /* Allocate memory for the SHA-256 hash context. */
      if ((ssh_global_random_state->hash = ssh_malloc(hash_len)) == NULL)
        goto failure; 
      
      memset(ssh_global_random_state->digest, 0, BLOCK_BYTES + KEY_BYTES);
      memset(ssh_global_random_state->state, 0, BLOCK_BYTES + KEY_BYTES);
      memset(ssh_global_random_state->output,   0, BLOCK_BYTES);
      memset(ssh_global_random_state->enc_time, 0, BLOCK_BYTES); 
      
      /* Initialize the 3-DES context with the all zero key. */ 
      ssh_des3_init(ssh_global_random_state->cipher, 
                    ssh_global_random_state->state, 
                    KEY_BYTES, TRUE); 
      
      /* Initialize the SHA256 context. */ 
      ssh_sha256_reset_context(ssh_global_random_state->hash);
      
      ssh_random_stir();
    }
  
  return ssh_global_random_state;
  
 failure:
  ssh_free(ssh_global_random_state->cipher);
  ssh_free(ssh_global_random_state);
  ssh_global_random_state = NULL;
  return NULL; 
}

/* Cryptographically strong random number functions. */

/* XOR noise directly into the state array, i must be less 
   than BLOCK_BYTES + KEY_BYTES / 4 (= 8).   */
void ssh_random_xor_noise(size_t i, SshUInt32 value)
{
  SshRandomState state;
  
  state = ssh_random_get_global();
  
  if (4 * i >= BLOCK_BYTES + KEY_BYTES)
    ssh_fatal("ssh_random_xor_noise: internal error.");
 
  value ^= SSH_GET_32BIT(state->state + 4 * i);
  SSH_PUT_32BIT(state->state + 4 * i, value);
}

void ssh_random_acquire_light_environmental_noise(void)
{
#if !defined(WINDOWS) && !defined(DOS) && !defined(macintosh) && !defined(VXWORKS)
  {
    int f;
    SSH_DEBUG(10, ("Starting read from /dev/random."));
    /* If /dev/random is available, read some data from there in non-blocking
       mode and mix it into the pool. */
    f = open("/dev/random", O_RDONLY);
    if (f >= 0)
      {
        unsigned char buf[32];
        int len;
        /* Set the descriptor into non-blocking mode. */
#if defined(O_NONBLOCK) && !defined(O_NONBLOCK_BROKEN)
        fcntl(f, F_SETFL, O_NONBLOCK);
#else /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
        fcntl(f, F_SETFL, O_NDELAY);
#endif /* O_NONBLOCK && !O_NONBLOCK_BROKEN */
        len = read(f, buf, sizeof(buf));
        close(f);
        SSH_DEBUG(10, ("Read %ld bytes from /dev/random.", len));
        if (len > 0)
          ssh_random_add_noise(buf, len);
      }
    else
      {
        SSH_DEBUG(10, ("Opening /dev/random failed."));
      }
  }
#endif /* !WINDOWS, !DOS, !macintosh, !VXWORKS */

  /* Get miscellaneous noise from various system parameters and statistics. */
  ssh_random_xor_noise(0, (SshUInt32)ssh_time());
#ifdef HAVE_CLOCK
  ssh_random_xor_noise(1, (SshUInt32)clock());
#endif /* HAVE_CLOCK */
#ifdef HAVE_GETTIMEOFDAY
  {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    ssh_random_xor_noise(2, (SshUInt32)tv.tv_usec);
    ssh_random_xor_noise(3, (SshUInt32)tv.tv_sec);
  }
#endif /* HAVE_GETTIMEOFDAY */
#ifdef HAVE_TIMES
  {
    struct tms tm;
    ssh_random_xor_noise(4, (SshUInt32)times(&tm));
    ssh_random_xor_noise(5, (SshUInt32)(tm.tms_utime ^
                                        (tm.tms_stime << 8) ^
                                        (tm.tms_cutime << 16) ^
                                        (tm.tms_cstime << 24)));
  }
#endif /* HAVE_TIMES */
#ifdef HAVE_GETRUSAGE
  {
    struct rusage ru, cru;
    getrusage(RUSAGE_SELF, &ru);
    getrusage(RUSAGE_CHILDREN, &cru);
    ssh_random_xor_noise(6, (SshUInt32)(ru.ru_utime.tv_usec +
                                        cru.ru_utime.tv_usec));
    ssh_random_xor_noise(7, (SshUInt32)(ru.ru_stime.tv_usec +
                                        cru.ru_stime.tv_usec));
    ssh_random_xor_noise(0, (SshUInt32)(ru.ru_maxrss + cru.ru_maxrss));
    ssh_random_xor_noise(1, (SshUInt32)(ru.ru_ixrss + cru.ru_ixrss));
    ssh_random_xor_noise(2, (SshUInt32)(ru.ru_idrss + cru.ru_idrss));
    ssh_random_xor_noise(3, (SshUInt32)(ru.ru_minflt + cru.ru_minflt));
    ssh_random_xor_noise(4, (SshUInt32)(ru.ru_majflt + cru.ru_majflt));
    ssh_random_xor_noise(5, (SshUInt32)(ru.ru_nswap + cru.ru_nswap));
    ssh_random_xor_noise(6, (SshUInt32)(ru.ru_inblock + cru.ru_inblock));
    ssh_random_xor_noise(7, (SshUInt32)(ru.ru_oublock + cru.ru_oublock));
    ssh_random_xor_noise(0, (SshUInt32)((ru.ru_msgsnd ^ ru.ru_msgrcv ^
                                          ru.ru_nsignals) +
                                         (cru.ru_msgsnd ^ cru.ru_msgrcv ^
                                          cru.ru_nsignals)));
    ssh_random_xor_noise(1, (SshUInt32)(ru.ru_nvcsw + cru.ru_nvcsw));
    ssh_random_xor_noise(2, (SshUInt32)(ru.ru_nivcsw + cru.ru_nivcsw));
  }
#endif /* HAVE_GETRUSAGE */
#if !defined(WINDOWS) && !defined(DOS)
#ifdef HAVE_GETPID
  ssh_random_xor_noise(3, (SshUInt32)getpid());
#endif /* HAVE_GETPID */
#ifdef HAVE_GETPPID
  ssh_random_xor_noise(4, (SshUInt32)getppid());
#endif /* HAVE_GETPPID */
#ifdef HAVE_GETUID
  ssh_random_xor_noise(5, (SshUInt32)getuid());
#endif /* HAVE_GETUID */
#ifdef HAVE_GETGID
  ssh_random_xor_noise(6, (SshUInt32)(getgid() << 16));
#endif /* HAVE_GETGID */
#ifdef HAVE_GETPGRP
  ssh_random_xor_noise(7, (SshUInt32)getpgrp());
#endif /* HAVE_GETPGRP */
#endif /* WINDOWS */
#ifdef _POSIX_CHILD_MAX
  ssh_random_xor_noise(0, (SshUInt32)(_POSIX_CHILD_MAX << 16));
#endif /* _POSIX_CHILD_MAX */
#if defined(CLK_TCK) && !defined(WINDOWS) && !defined(DOS)
  ssh_random_xor_noise(1, (SshUInt32)(CLK_TCK << 16));
#endif /* CLK_TCK && !WINDOWS */
#ifdef VXWORKS
  /* XXX - Get better noise from cipher chips RNG unit (HiFN, MPC180,...) */
#endif /* VXWORKS */
  
  return;
}

/* Mixes the bytes from the buffer into the state array. The pool should 
   be stirred after a sufficient amount of noise has been added. The noise 
   added from this function will not be utilized until the ssh_random_stir() 
   function has been called. 
*/
void ssh_random_add_noise(const void *buf, size_t bytes)
{
  size_t i;
  const unsigned char *input = buf;
  SshRandomState state;
  
  state = ssh_random_get_global();

  /* First hash the input to 32 bytes and put it into state->digest. */ 
  ssh_sha256_update(state->hash, input, bytes);
  ssh_sha256_final(state->hash, state->digest);
  
  /* XOR it to the entropy state pool. */ 
  for (i = 0; i < BLOCK_BYTES + KEY_BYTES; i++)  
    state->state[i] ^= state->digest[i];
  
  /* Reset the hash context and clean. */
  ssh_sha256_reset_context(state->hash);
  memset(state->digest, 0, sizeof(*state->digest));

  return;
}

/* Reseed the RNG. Noise added by calling ssh_random_add_noise() does not
   become effective until ssh_random_stir() is called. 
*/ 
void ssh_random_stir()
{
  SshRandomState state;

  state = ssh_random_get_global();
  
  ssh_random_acquire_light_environmental_noise();
 
  /* Reset the 3-DES key. */ 
  ssh_des3_init(state->cipher, state->state, KEY_BYTES, TRUE);
  /* Reset the hash context. */
  ssh_sha256_reset_context(state->hash);
  
  state->next_available_byte = BLOCK_BYTES;
  
  return;
}

/* Returns a random byte. */  
unsigned int ssh_random_get_byte(void)
{
  int i; 
  SshRandomState state;
  SshUInt64 time;
  
  if ((state = ssh_random_get_global()) == NULL)
    ssh_fatal("ssh_random_get_byte: not enough space for random state.");
  
  if (state->next_available_byte >= BLOCK_BYTES)
    {    
      state->next_available_byte = 0;
      
      /* Get the current timestamp. */
      time = (SshUInt64) ssh_time();
      SSH_PUT_64BIT(state->enc_time, time);
      
      /* Encrypt the timestamp. */ 
      ssh_des3_ecb(state->cipher, state->enc_time, state->enc_time, 
                   BLOCK_BYTES, NULL);
      
      /* Update the output block. */ 
      for (i = 0; i < BLOCK_BYTES; i++)
        state->output[i] = state->enc_time[i] ^ state->state[KEY_BYTES + i]; 

      /* Encrypt again. */ 
      ssh_des3_ecb(state->cipher, state->output, 
                   state->output, BLOCK_BYTES, NULL);
      
      /* Update the seed block. */ 
      for (i = 0; i < BLOCK_BYTES; i++)
        state->state[KEY_BYTES + i] = state->enc_time[i] ^ state->output[i]; 

      ssh_des3_ecb(state->cipher, state->state + KEY_BYTES, 
                   state->state + KEY_BYTES, BLOCK_BYTES, NULL);
    }
  
 return state->output[state->next_available_byte++];
}

/* Zeroes and frees any data structures associated with the random number
   generator. This should be called when the state is no longer needed to
   remove any sensitive data from memory. */

void ssh_random_free(void)
{
  if (ssh_global_random_state)
    {
      memset(ssh_global_random_state->hash, 0, ssh_sha256_ctxsize()); 
      memset(ssh_global_random_state->cipher, 0, ssh_des3_ctxsize()); 
      memset(ssh_global_random_state, 0, sizeof(*ssh_global_random_state));
      
      ssh_free(ssh_global_random_state->hash);
      ssh_free(ssh_global_random_state->cipher);
      ssh_free(ssh_global_random_state);

      ssh_global_random_state = NULL;
    }
}

#endif /* WITH_ANSI_RNG */
