/*

sshserver.h

  Authors:
        Tatu Ylnen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SSH server functionality for processing a connection.

*/

#ifndef SSHSERVER_H
#define SSHSERVER_H

#include "sshcrypt.h"
#include "sshconfig.h"
#include "sshcommon.h"
#include "sshauth.h"

/* Applications clean-up function. */
typedef void (*SshServerCleanupCB)(void *context);

/* Data type for an SSH server object.  The server object processes one
   connection (potentially multiple sessions/channels). */

typedef struct SshServerRec
{
  /* Connection data for both client and server. */
  SshCommon common;

  /* Configuration options that apply to the server. */
  SshConfig config;
  
  /* Authentication methods. */
  SshAuthServerMethod methods;

  /* The SshUser object for the authenticated user will get stored here
     before SSH_CROSS_AUTHENTICATED is received.  This will be freed
     by SshCommon. */
  SshUser uc;

  /* The authenticated user name of the applicant in the client
     side. (got from "hostbased"-authentication.) */
  char *authenticated_client_user_name;
  
  /* Authenticated host name, or empty if none. (got from
     "hostbased"-authentication.) */
  char *authenticated_client_host;

  /* Applications clean-up function, called in ssh_server_destroy(). */
  SshServerCleanupCB clean_up;

  SshOperationHandle ext_auth_handle;
  
  /* Last user name used for authentication (sshd2, auth_policy_proc()). */
  char *last_user;
} SshServerStruct, *SshServer;

/* Takes a stream, and creates an SSH server for processing that
   connection.  This closes the stream and returns NULL (without
   calling the destroy function) if an error occurs.  This does not
   free the given server key.  The random state is required to stay
   valid until the server has been destroyed.  ``config'' must remain
   valid until the server is destroyed; it is not automatically freed.
     `stream'        the connection stream
     `config'        configuration data (not freed, must remain valid)
     `client_host_name'
                     name of the client host (reverse mapped, or if that
                     can't or won't be done, IP-address)
     `random_state'  random number generator state
     `disconnect'    function to call on disconnect
     `debug'         function to call on debug message (may be NULL)
     `version_check' version check callback (may be NULL)
     `protocol_version' protocol version string to be sent to the client
     `clean_up'      called during ssh_server_destroy() to let the
                     application cancel timeouts etc, which could cause
                     race conditions on disconnect (may be NULL)
     `context'       context to pass to the callbacks
   The object should be destroyed from the ``disconnect'' callback. */
SshServer ssh_server_wrap(SshStream stream,



                          SshConfig config,
                          const char *client_host_name,




                          SshConnDisconnectProc disconnect,
                          SshConnDebugProc debug,
                          SshVersionCallback version_check,

                          SshAuthPolicyProc auth_policy_proc,
                          SshCommonAuthenticationNotify authentication_notify,
                          SshServerCleanupCB clean_up,
                          const char *protocol_version,
                          void *context);

/* Forcibly destroys the given connection. */
void ssh_server_destroy(SshServer server);

#endif /* SSHSERVER_H */
