/*

  sshpamserver.h

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  PAM (Pluggable Authentication Modules) authentication, server side.

  Packet type definitions for the protocol between sshd2 and
  the PAM subprocess.
*/

#ifndef _SSH_PAM_INTERNAL_H_
#define _SSH_PAM_INTERNAL_H_

#if defined(DAEMON_WITH_PAM)

#include "sshpacketstream.h"

/* Packet types from sshd2 to check-user. */

/* This packet starts the PAM at the server side. ssh-pam-user-check
   calls pam_start() after this.

   Packet payload:
     string  service name (usually "sshd2")
     string  user (which is authenticated)     

   PAM subprocess replies to this with either SSH_PAM_OP_SUCCESS, or
   SSH_PAM_OP_ERROR.
*/
#define SSH_PAM_START             (SshPacketType)1

/* Request pam_authenticate().

   Packet payload:
     uint32  flags
*/
#define SSH_PAM_AUTHENTICATE      (SshPacketType)2

/* Request pam_acct_mgmt().

   Packet payload:
     uint32  flags
*/
#define SSH_PAM_ACCT_MGMT         (SshPacketType)3

/* Request pam_open_session().

   Packet payload:
     uint32  flags
*/
#define SSH_PAM_OPEN_SESSION      (SshPacketType)4

/* Request pam_close_session().

   Packet payload:
     uint32 flags
*/
#define SSH_PAM_CLOSE_SESSION     (SshPacketType)5

/* Request pam_setcred().

   Packet payload:
     uint32  flags
*/
#define SSH_PAM_SETCRED           (SshPacketType)6

/* Request pam_chauthtok().

   Packet payload:
     uint32  flags
*/
#define SSH_PAM_CHAUTHTOK         (SshPacketType)7

/* Request pam_end(). After acknowledging this message,
   PAM subprocess will exit.

   Packet payload:
     uint32  pam_status
*/
#define SSH_PAM_END               (SshPacketType)8

/* Request pam_set_item().

   Packet payload:
     uint32  item_type
     string  item
*/
#define SSH_PAM_SET_ITEM          (SshPacketType)9

/* Request pam_get_item(). On success, returns
   SSH_PAM_OP_SUCCESS_WITH_PAYLOAD.

   Packet payload:
     uint32 item_type
*/
#define SSH_PAM_GET_ITEM          (SshPacketType)10

/* Packet types from check-user to sshd2. */

/* Responses to a message. */

/* Indicates success. No payload. */
#define SSH_PAM_OP_SUCCESS        (SshPacketType)50

/* Indicates success. No payload. */
#define SSH_PAM_OP_SUCCESS_WITH_PAYLOAD (SshPacketType)51

/* Some error occurred during PAM operation.

   Packet payload:
     byte    PAM error number
     string  PAM error message
*/
#define SSH_PAM_OP_ERROR          (SshPacketType)55

/* Some error occurred durng other than PAM operation.
   
   Packet payload:
     byte    error number
     string  error message
*/
#define SSH_PAM_ERROR             (SshPacketType)56

typedef enum
{
  SSH_PAM_PROTOCOL_ERROR = 0
} SshPAMError;

/* Conversation function message. These should be passed to the client.

   Packet payload:
     uint32  number of messages
     byte    message type
     string  message

     ...     more messages (message type-message pairs), up to number of
             messages
*/
#define SSH_PAM_CONVERSATION_MSG  (SshPacketType)60

/* Response message from client.

   Packet payload:
     uint32  number of responses
     byte    response retcode
     string  response
     ...     more responses (response retcode-response pairs), up to
             number of responses

   Number of responses MUST be the same as was passed with
   SSH_PAM_CONVERSATION_MSG, as the conversation function expects
   that. The index of a response corresponds with the index of the
   message it is answering. Message types, that don't require
   responses, should still be responded with empty messages and
   response retcode of 0. This is to stay consistent with the PAM-API.
*/
#define SSH_PAM_CONVERSATION_RESP (SshPacketType)61

/* This function should be called in the child process. It converses
   with the parent via a pipe, with the protocol described
   above. Returns FALSE if everything went ok, TRUE if an error
   occurs. */
Boolean ssh_pam_conv(void);

#endif /* DAEMON_WITH_PAM */
#endif /* _SSH_PAM_INTERNAL_H_ */
