/*
  sshfilecopyi.h

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Definitions common to the sshfilecopy*.* and sshfc_*.* files.
 */

#ifndef SSHFILECOPYI_H
#define SSHFILECOPYI_H

#include "sshdsprintf.h"
#include "sshfsm.h"
#include "sshtimeouts.h"

#define SSH_FC_MAX_BUFFER_SIZE 0x40000
#define SSH_FC_READ_MAX 0x8000

#ifdef __SUNPRO_C
#pragma error_messages (off,E_STATEMENT_NOT_REACHED)
#endif /* __SUNPRO_C */

/* Record for filenames and attributes, needed in the transfer. */
struct SshFileCopyFileRec
{
  SshFileAttributes attributes;
  SshFileHandle handle;
  char *name;
  char *long_name;
  /* If non-NULL, this file is a directory. Contains the files in this
     directory. */
  SshFileCopyLocation dir_entries;
};

/* mapcar function which rewinds SshFileCopyLocations in a list. */
void *location_list_rewind(void *item, void *ctx);

/* Unified callback interface.

   XXX Change to use normal functions, instead of statics, so that these
   could be re-used.

   XXX simplify API.
*/
typedef enum {
  SSH_FC_STATUS_CALLBACK,
  SSH_FC_HANDLE_CALLBACK,
  SSH_FC_DATA_CALLBACK,
  SSH_FC_NAME_CALLBACK,
  SSH_FC_ATTRIBUTE_CALLBACK
} SshFileCopyCallbackType;

char *fcc_format_error(const char *fmt, ...);
SshFileCopyError fcc_fx_err_to_fc_err(SshFileClientError error);

typedef void (*FCCommonStatusCB)(SshFileClientError error,
                                 SshFileCopyFile file,
                                 const char *error_msg, SshFSMThread thread);
typedef void (*FCCommonHandleCB)(SshFileClientError error,
                                 SshFileCopyFile file,
                                 SshFileHandle handle, const char *error_msg,
                                 SshFSMThread thread);
typedef void (*FCCommonDataCB)(SshFileClientError error, SshFileCopyFile file,
                               const unsigned char *data, size_t data_len,
                               const char *error_msg, SshFSMThread thread);
typedef void (*FCCommonNameCB)(SshFileClientError error, SshFileCopyFile file,
                               const char *name, const char *long_name,
                               SshFileAttributes attrs, const char *error_msg,
                               SshFSMThread thread);
typedef void (*FCCommonAttributeCB)(SshFileClientError error,
                                    SshFileCopyFile file,
                                    SshFileAttributes attrs,
                                    const char *error_msg,
                                    SshFSMThread thread);


#define FCC_TDATA(ttype)                        \
ttype tdata = (ttype)ssh_fsm_get_tdata(thread)
#define FCC_GDATA(gtype)                        \
gtype gdata = (gtype)ssh_fsm_get_gdata(thread)

#define FCC_DATA(gtype, ttype)                  \
FCC_GDATA(gtype);                               \
FCC_TDATA(ttype)

#define FCC_ERROR(error, varcall)                                       \
do                                                                      \
{                                                                       \
  char *str = fcc_format_error varcall;                                 \
  (*gdata->error_callback)((error), str, gdata->callback_context);      \
  ssh_xfree(str);                                                       \
}                                                                       \
while(0)

#define FCC_CLEAN_CALLBACKS(tdata)              \
do                                              \
{                                               \
  (tdata)->status_cb = NULL_FNPTR;              \
  (tdata)->handle_cb = NULL_FNPTR;              \
  (tdata)->data_cb = NULL_FNPTR;                \
  (tdata)->name_cb = NULL_FNPTR;                \
  (tdata)->attrs_cb = NULL_FNPTR;               \
}                                               \
while(0)

#define FCC_START_GEN(tdata, file, source)                              \
do                                                                      \
{                                                                       \
  FCC_CLEAN_CALLBACKS(tdata);                                           \
  (tdata)->current_state = ssh_fsm_get_thread_current_state(thread);    \
  (tdata)->current_file = file;                                         \
  (tdata)->dealing_with_source = source;                                \
}                                                                       \
while(0)

#define FCC_START_STATUS(tdata, file, source, cb)       \
do                                                      \
{                                                       \
  FCC_START_GEN((tdata), (file), (source));             \
  (tdata)->status_cb = (cb);                            \
}                                                       \
while(0)

#define FCC_START_HANDLE(tdata, file, source, cb)       \
do                                                      \
{                                                       \
  FCC_START_GEN((tdata), (file), (source));             \
  (tdata)->handle_cb = (cb);                            \
}                                                       \
while(0)

#define FCC_START_DATA(tdata, file, source, cb) \
do                                              \
{                                               \
  FCC_START_GEN((tdata), (file), (source));     \
  (tdata)->data_cb = (cb);                      \
}                                               \
while(0)

#define FCC_START_NAME(tdata, file, source, cb) \
do                                              \
{                                               \
  FCC_START_GEN((tdata), (file), (source));     \
  (tdata)->name_cb = (cb);                      \
}                                               \
while(0)

#define FCC_START_ATTRS(tdata, file, source, cb)        \
do                                                      \
{                                                       \
  FCC_START_GEN((tdata), (file), (source));             \
  (tdata)->attrs_cb = (cb);                             \
}                                                       \
while(0)

/*

  XXX
  This kind of macro is required. 

  #define FCC_CLEANUP(error, message) */

/* #define FCC_GEN_BF_CALLBACK(prefix, gtype, ttype */

#define FCC_GEN_BF_CALLBACK_PROTO(prefix)                       \
void prefix ## _transfer_callback(SshFileClientError error,     \
                                 SshFileCopyCallbackType type,  \
                                 /* handle callback. */         \
                                 SshFileHandle handle,          \
                                 /* data callback. */           \
                                 const unsigned char *data,     \
                                 size_t data_len,               \
                                 /* name callback */            \
                                 const char *name,              \
                                 const char *long_name,         \
                                 /* attrs and name callbacks */ \
                                 SshFileAttributes attrs,       \
                                 /* all callbacks. */           \
                                 const char *error_msg,         \
                                 const char *lang_tag,          \
                                 void *context)


#define FCC_GEN_BF_CALLBACK(prefix, gtype, ttype) \
FCC_GEN_BF_CALLBACK_PROTO(prefix) \
{ \
  SshFSMThread thread = (SshFSMThread) context; \
  char buf[256]; \
  size_t buf_len = sizeof(buf); \
  SshFileCopyFile file; \
  gtype gdata; \
  ttype tdata; \
  SshFileCopyError fc_error; \
 \
  SSH_PRECOND(thread != NULL); \
  gdata = (gtype)ssh_fsm_get_gdata(thread); \
  SSH_PRECOND(gdata != NULL); \
  tdata = (ttype)ssh_fsm_get_tdata(thread); \
  SSH_PRECOND(tdata != NULL); \
  file = tdata->current_file; \
  SSH_PRECOND(file != NULL); \
 \
  if (error != SSH_FX_OK) \
    SSH_DEBUG(2, ("Received error %s%s%s%d%s.", \
                  error_msg ? "`" : "", \
                  error_msg ? error_msg : "", \
                  error_msg ? "' (" : "", \
                  error, \
                  error_msg ? ")" : "")); \
 \
  tdata->op_handle = NULL; \
 \
  switch (error) \
    { \
    case SSH_FX_OK: \
      ssh_snprintf(buf, buf_len, "%s (%s): OK %s%s%s", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst", \
                   error_msg ? "(server msg: '" : "", \
                   error_msg ? error_msg : "", \
                   error_msg ? "')" : ""); \
      goto fcc_err_common; \
    case SSH_FX_EOF: \
      ssh_snprintf(buf, buf_len, "%s (%s): received EOF %s%s%s", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst", \
                   error_msg ? "(server msg: '" : "", \
                   error_msg ? error_msg : "", \
                   error_msg ? "')" : ""); \
      goto fcc_err_common; \
    case SSH_FX_NO_SUCH_FILE: \
      ssh_snprintf(buf, buf_len, "%s (%s): no such file %s%s%s", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst", \
                   error_msg ? "(server msg: '" : "", \
                   error_msg ? error_msg : "", \
                   error_msg ? "')" : ""); \
      goto fcc_err_common; \
    case SSH_FX_PERMISSION_DENIED: \
      ssh_snprintf(buf, buf_len, "%s (%s): permission denied %s%s%s", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst", \
                   error_msg ? "(server msg: '" : "", \
                   error_msg ? error_msg : "", \
                   error_msg ? "')" : ""); \
      goto fcc_err_common; \
    case SSH_FX_FAILURE: \
      ssh_snprintf(buf, buf_len, "%s (%s): unspecified failure %s%s%s", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst", \
                   error_msg ? "(server msg: '" : "", \
                   error_msg ? error_msg : "", \
                   error_msg ? "')" : ""); \
    fcc_err_common: \
      switch (type) \
        { \
        case SSH_FC_STATUS_CALLBACK: \
          SSH_ASSERT(tdata->status_cb != NULL_FNPTR); \
          (*tdata->status_cb)(error, file, buf, thread); \
          break; \
        case SSH_FC_HANDLE_CALLBACK: \
          SSH_ASSERT(tdata->handle_cb != NULL_FNPTR); \
          (*tdata->handle_cb)(error, file, handle, buf, thread); \
          break; \
        case SSH_FC_DATA_CALLBACK: \
          SSH_ASSERT(tdata->data_cb != NULL_FNPTR); \
          (*tdata->data_cb)(error, file, data, data_len, buf, thread); \
          break; \
        case SSH_FC_NAME_CALLBACK: \
          SSH_ASSERT(tdata->name_cb != NULL_FNPTR); \
          (*tdata->name_cb)(error, file, name, long_name, attrs, \
                            buf, thread); \
          break; \
        case SSH_FC_ATTRIBUTE_CALLBACK: \
          SSH_ASSERT(tdata->attrs_cb != NULL_FNPTR); \
          (*tdata->attrs_cb)(error, file, attrs, buf, thread); \
          break; \
        } \
      break; \
    case SSH_FX_NO_CONNECTION: \
      ssh_fatal(#prefix "_transfer_callback: Got " \
                "SSH_FX_NO_CONNECTION, even though we wait for " \
                "the version. (this is a bug)"); \
      return; \
    case SSH_FX_CONNECTION_LOST: \
      fc_error = SSH_FC_ERROR_CONNECTION_LOST; \
      ssh_snprintf(buf, buf_len, "%s (%s): connection lost", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst"); \
      goto fcc_unrec_err_common; \
    case SSH_FX_OP_UNSUPPORTED: \
      fc_error = SSH_FC_ERROR_PROTOCOL_MISMATCH; \
      ssh_snprintf(buf, buf_len, "%s (%s): op unsupported", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst"); \
      goto fcc_unrec_err_common; \
    case SSH_FX_BAD_MESSAGE: \
      fc_error = SSH_FC_ERROR_PROTOCOL_MISMATCH; \
      ssh_snprintf(buf, buf_len, "%s (%s): bad message", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst"); \
      goto fcc_unrec_err_common; \
    case SSH_FX_OUT_OF_MEMORY: \
      fc_error = SSH_FC_ERROR_PROTOCOL_MISMATCH; \
      ssh_snprintf(buf, buf_len, "%s (%s): out of memory", \
                   ssh_file_copy_file_get_name(file), \
                   tdata->dealing_with_source ? "src" : "dst"); \
    fcc_unrec_err_common: \
      FCC_CLEANUP(fc_error, error_msg); \
      return; \
    } \
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread); \
}


#define FCC_GEN_STATUS_PROTO(prefix)                            \
static void transfer_status_cb(SshFileClientError error,        \
                               const char *error_msg,           \
                               const char *lang_tag,            \
                               void *context)

#define FCC_GEN_STATUS(prefix)                                  \
FCC_GEN_STATUS_PROTO(prefix)                                    \
{                                                               \
  prefix ## _transfer_callback(error, SSH_FC_STATUS_CALLBACK,   \
                              NULL, NULL, 0L, NULL, NULL, NULL, \
                              error_msg, lang_tag, context);    \
}

#define FCC_GEN_HANDLE_PROTO(prefix)                            \
static void transfer_handle_cb(SshFileClientError error,        \
                               SshFileHandle handle,            \
                               const char *error_msg,           \
                               const char *lang_tag,            \
                               void *context)

#define FCC_GEN_HANDLE(prefix)                                          \
FCC_GEN_HANDLE_PROTO(prefix)                                            \
{                                                                       \
  prefix ## _transfer_callback(error, SSH_FC_HANDLE_CALLBACK, handle,   \
                              NULL, 0L, NULL, NULL, NULL, error_msg,    \
                              lang_tag, context);                       \
}

#define FCC_GEN_DATA_PROTO(prefix)                      \
static void transfer_data_cb(SshFileClientError error,  \
                             const unsigned char *data, \
                             size_t len,                \
                             const char *error_msg,     \
                             const char *lang_tag,      \
                             void *context)

#define FCC_GEN_DATA(prefix)                                            \
FCC_GEN_DATA_PROTO(prefix)                                              \
{                                                                       \
  prefix ## _transfer_callback(error, SSH_FC_DATA_CALLBACK, NULL,       \
                              data, len, NULL, NULL, NULL,              \
                              error_msg, lang_tag, context);            \
}

#define FCC_GEN_NAME_PROTO(prefix)                              \
static void transfer_name_cb(SshFileClientError error,          \
                             const char *name,                  \
                             const char *long_name,             \
                             SshFileAttributes attributes,      \
                             const char *error_msg,             \
                             const char *lang_tag,              \
                             void *context)

#define FCC_GEN_NAME(prefix)                                            \
FCC_GEN_NAME_PROTO(prefix)                                              \
{                                                                       \
  prefix ## _transfer_callback(error, SSH_FC_NAME_CALLBACK, NULL,       \
                              NULL, 0L, name, long_name, attributes,    \
                              error_msg, lang_tag, context);            \
}

#define FCC_GEN_ATTR_PROTO(prefix)                              \
static void transfer_attribute_cb(SshFileClientError error,     \
                                  SshFileAttributes attributes, \
                                  const char *error_msg,        \
                                  const char *lang_tag,         \
                                  void *context)

#define FCC_GEN_ATTR(prefix)                                             \
FCC_GEN_ATTR_PROTO(prefix)                                               \
{                                                                        \
  prefix ## _transfer_callback(error, SSH_FC_ATTRIBUTE_CALLBACK,         \
                              NULL, NULL, 0L, NULL, NULL,                \
                              attributes, error_msg, lang_tag, context); \
}

#endif /* SSHFILECOPYI_H */
