/*
  sshfc_transfer.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions that perform the actual transfer.
 */

#include "ssh2includes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"
#include "sshbuffer.h"
#include "sshtimemeasure.h"
#include "sshfc_transferi.h"




#define SSH_DEBUG_MODULE "SshFCTransfer"

/**********
 * ssh_file_copy_transfer_files() related stuff.
 **********/

/* Forward declarations. */
SSH_FSM_STEP(transfer_check_src_conn);
SSH_FSM_STEP(transfer_check_dest_conn);
SSH_FSM_STEP(transfer_conn_failed);
SSH_FSM_STEP(transfer_set_next_source_item);
SSH_FSM_STEP(transfer_set_next_source);
SSH_FSM_STEP(transfer_stat_next_file);
SSH_FSM_STEP(transfer_mkdir);
SSH_FSM_STEP(transfer_lstat_dest_dir);
SSH_FSM_STEP(transfer_updir);
SSH_FSM_STEP(transfer_chmod_destdir);
SSH_FSM_STEP(transfer_lstat_src_dir);
SSH_FSM_STEP(transfer_rm_src_dir);
SSH_FSM_STEP(transfer_parse_raw_source);
SSH_FSM_STEP(transfer_stat_source);
SSH_FSM_STEP(transfer_stat_dest_location);
SSH_FSM_STEP(transfer_stat_dest);
SSH_FSM_STEP(transfer_rm_source);
SSH_FSM_STEP(transfer_open_dest);
SSH_FSM_STEP(transfer_open_source);
SSH_FSM_STEP(transfer_chmod_dest_before_transfer);
SSH_FSM_STEP(transfer_close_src);
SSH_FSM_STEP(transfer_close_dest);
SSH_FSM_STEP(transfer_one_done);
SSH_FSM_STEP(transfer_done);
SSH_FSM_STEP(transfer_abort);
SSH_FSM_STEP(transfer_cleanup);

/*
  Prototypes for

    SSH_FSM_STEP(transfer_set_acmod_time);
    SSH_FSM_STEP(transfer_wake_parent_and_finish);

  are in sshfc_transferi.h .
*/

#define FCC_CLEANUP(error, message)                             \
do                                                              \
{                                                               \
  (*gdata->completion_callback)((error), (message),             \
                                gdata->callback_context);       \
  /* stop transfer and abort */                                 \
  ssh_fsm_set_next(gdata->main_thread, transfer_cleanup);       \
  ssh_fsm_continue(gdata->main_thread);                         \
}                                                               \
while(0)

FCC_GEN_BF_CALLBACK(fct, SshFileCopyTransferContext, TransferThreadContext)
FCC_GEN_STATUS(fct)
FCC_GEN_HANDLE(fct)
FCC_GEN_ATTR(fct)

void transfer_thread_destructor(SshFSM fsm, void *tdata)
{
  TransferThreadContext thread_context = (TransferThreadContext) tdata;

  SSH_PRECOND(tdata != NULL);

  /* XXX More cases. */
  if (thread_context->op_handle)
    {
      ssh_operation_abort(thread_context->op_handle);
      thread_context->op_handle = NULL;
    }
  if (thread_context->parent)
    {
      TransferThreadContext parent_tdata;
      parent_tdata = ssh_fsm_get_tdata(thread_context->parent);
      parent_tdata->child = NULL;
    }

  if (thread_context->child)
    {
      TransferThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(thread_context->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(thread_context->child);
      thread_context->child = NULL;
    }

  ssh_xfree(thread_context->source_file_name);
  thread_context->source_file_name = NULL;

#if 0
  if (thread_context->current_dest_file)
    {
      ssh_file_copy_file_destroy(thread_context->current_dest_file);
      thread_context->current_dest_file = NULL;
    }
#endif
}

void fct_conn_completion_cb(SshFileClient client, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext)ssh_fsm_get_gdata(thread);
  TransferThreadContext tdata =
    (TransferThreadContext)ssh_fsm_get_tdata(thread);

  tdata->op_handle = NULL;

  if (!client)
    SSH_FSM_SET_NEXT(transfer_conn_failed);
  
  if (tdata->dealing_with_source)
    gdata->source->client = client;
  else
    gdata->destination->client = client;
  
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(transfer_check_src_conn)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  tdata->dealing_with_source = TRUE;

  SSH_FSM_SET_NEXT(transfer_check_dest_conn);

  if (gdata->source->client)
    {
      SSH_DEBUG(4, ("Source connection OK."));
      return SSH_FSM_CONTINUE;
    }
  SSH_DEBUG(2, ("Establishing source connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->source,
                                           fct_conn_completion_cb,
                                           thread));

}

SSH_FSM_STEP(transfer_check_dest_conn)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  tdata->dealing_with_source = FALSE;

  SSH_FSM_SET_NEXT(transfer_set_next_source_item);
  
  if (gdata->destination->client)
    {
      SSH_DEBUG(4, ("Destination connection OK."));
      return SSH_FSM_CONTINUE;
    }
  SSH_DEBUG(2, ("Establishing destination connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->destination,
                                           fct_conn_completion_cb,
                                           thread));
}

SSH_FSM_STEP(transfer_conn_failed)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  const char *msg;

  if (tdata->dealing_with_source)
    msg = "Connecting to source failed.";
  else
    msg = "Connecting to destination failed.";
    
  (*gdata->completion_callback)(SSH_FC_ERROR_CONNECTION_FAILED,
                                msg, gdata->callback_context);
  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_set_next_source_item)
{
  TransferThreadContext child_thread_context;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(thread == gdata->main_thread);

  SSH_DEBUG(3, ("Setting next source file item..."));

  if (ssh_dllist_is_current_valid(gdata->file_list))
    {
      child_thread_context = ssh_xcalloc(1, sizeof(*child_thread_context));
      tdata->child = ssh_fsm_thread_create(gdata->fsm,
                                           transfer_set_next_source,
                                           NULL_FNPTR,
                                           transfer_thread_destructor,
                                           child_thread_context);
      SSH_VERIFY(tdata->child != NULL);
      child_thread_context->parent = thread;
      child_thread_context->current_item =
        (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);
      child_thread_context->current_location =
        child_thread_context->current_item->files;
      ssh_dllist_fw(gdata->file_list, 1);
    }
  else
    {
      SSH_DEBUG(3, ("No more source files in this list."));
      SSH_FSM_SET_NEXT(transfer_done);
      return SSH_FSM_CONTINUE;
    }

  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(transfer_set_next_source)
{
  TransferThreadContext child_thread_context;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(tdata->current_location != NULL);
  SSH_PRECOND(tdata->current_location->file_list != NULL);

  if (ssh_dllist_is_current_valid(tdata->current_location->file_list))
    {
      if (tdata->current_location->raw)
        {
          SSH_DEBUG(2, ("Source file is \"raw\", and it needs to "
                        "be parsed."));
          SSH_FSM_SET_NEXT(transfer_parse_raw_source);
          return SSH_FSM_CONTINUE;
        }
      else
        {
          SshFileCopyFile file;
          SshFileAttributes attributes;

          file = (SshFileCopyFile) ssh_dllist_current
            (tdata->current_location->file_list);

          if ((attributes = ssh_file_copy_file_get_attributes(file)) != NULL)
            {
              if (attributes->flags & SSH_FILEXFER_ATTR_PERMISSIONS)
                {
                  if ((attributes->permissions & S_IFMT) == S_IFDIR)
                    {
                      if (!file->dir_entries && gdata->attrs->recurse_dirs)
                        {
                          ssh_fatal
                            ("transfer_set_next_source: "
                             "Directory recursion is enabled, "
                             "but directory %s hasn't been "
                             "recursed. Fatal error, submit "
                             "bug report (read README).",
                             ssh_file_copy_file_get_name(file));
                        }

                      if (!file->dir_entries || !gdata->attrs->recurse_dirs)
                        {
                          char *error_message;

                          /* Directory hasn't been recursed,
                             ie. recursing isn't enabled. */

                          ssh_xdsprintf(&error_message, "File %s is a "
                                       "directory, and recursion is not "
                                       "enabled. ",
                                       ssh_file_copy_file_get_name(file));

                          (*gdata->error_callback)(SSH_FC_ERROR,
                                                   error_message,
                                                   gdata->callback_context);

                          ssh_xfree(error_message);
                          goto drop_file;
                        }

                      child_thread_context =
                        ssh_xcalloc(1, sizeof(*child_thread_context));
                      tdata->child =
                        ssh_fsm_thread_create(gdata->fsm,
                                              transfer_mkdir,
                                              NULL_FNPTR,
                                              transfer_thread_destructor,
                                              child_thread_context);
                      SSH_VERIFY(tdata->child != NULL);
                      child_thread_context->current_location =
                        file->dir_entries;
                      child_thread_context->current_item = tdata->current_item;

                      /* We have to advance this here, because the child
                         doesn't have access to this list. */
                      ssh_dllist_fw(tdata->current_location->file_list, 1);

                      goto common_continue;
                    }
#ifdef S_IFLNK
                  else if ((attributes->permissions & S_IFMT) == S_IFLNK)
                    {
                      /* The file should stat()ted to see whether it
                         points to a regular file. */
                      SSH_DEBUG(3, ("File (%s) is a symbolic link.",
                                    ssh_file_copy_file_get_name
                                    (file)));
                      goto stat_file;
                    }
#else /* S_IFLNK */











#endif /* S_IFLNK */
                  else if ((attributes->permissions & S_IFMT) != S_IFREG)
                    {
                      /* File is not a regular, symbolic link or a
                         directory. It should be discarded. */
                      char *error_message;

                      /* Directory hasn't been recursed,
                         ie. recursing isn't enabled. */

                      ssh_xdsprintf(&error_message, "File %s not a "
                                   "regular file, directory or symlink.",
                                   ssh_file_copy_file_get_name(file));

                      (*gdata->error_callback)(SSH_FC_ERROR,
                                               error_message,
                                               gdata->callback_context);

                      ssh_xfree(error_message);
                      goto drop_file;
                    }
                }
              else
                {
                  /* File should be statted again. */
                  SSH_DEBUG(3, ("File's (%s) attributes don't contain "
                                "permissions.",
                                ssh_file_copy_file_get_name(file)));
                  goto stat_file;
                }
            }
          else
            {
              SSH_DEBUG(3, ("File (%s) doesn't contain attributes.",
                            ssh_file_copy_file_get_name(file)));
            stat_file:
              /* The file should be statted again to get the
                 attributes. */
              SSH_DEBUG(3, ("Commencing stat on file..."));

              SSH_FSM_SET_NEXT(transfer_stat_next_file);
              if (!tdata->source)
                tdata->source = gdata->source;

              return SSH_FSM_CONTINUE;
            }


          SSH_DEBUG(2, ("Next source file is %s/%s .",
                        tdata->current_location->file->name,
                        file->name));
          child_thread_context = ssh_xcalloc(1, sizeof(*child_thread_context));
          tdata->child = ssh_fsm_thread_create(gdata->fsm,
                                               transfer_stat_source,
                                               NULL_FNPTR,
                                               transfer_thread_destructor,
                                               child_thread_context);
          SSH_VERIFY(tdata->child != NULL);
          child_thread_context->current_location = tdata->current_location;
          child_thread_context->current_item = tdata->current_item;

        common_continue:
          child_thread_context->parent = thread;
          child_thread_context->source = gdata->source;
          child_thread_context->source_file = ssh_file_copy_file_dup(file);

          if (tdata->source_file)
            {
              char *temp_filename, *ph, *basedir;
              ph = (char *)ssh_file_copy_file_get_name(file);
              basedir = tdata->source_file ?
                (char *)ssh_file_copy_file_get_name(tdata->source_file) : "";
              ssh_xdsprintf(&temp_filename, "%s%s%s",
                           *basedir ? basedir : "" ,
                           (*ph && *basedir &&
                            basedir[strlen(basedir) - 1] != '/') ? "/" : "",
                           *ph ? ph : "");
              ssh_file_copy_file_register_filename
                (child_thread_context->source_file, temp_filename);
            }

          if (tdata->source_dir)
            child_thread_context->source_dir = tdata->source_dir;
          else
            child_thread_context->source_dir = tdata->current_location->file;

          child_thread_context->current_dest_file =
            ssh_file_copy_file_allocate();

          return SSH_FSM_SUSPENDED;

        drop_file:
          ssh_dllist_fw(tdata->current_location->file_list, 1);
          return SSH_FSM_CONTINUE;
        }
    }
  else
    {
      if (ssh_dllist_length(tdata->current_location->file_list) < 1)
        {
          if (!gdata->attrs->recurse_dirs)
            {
              char *error_message;

              /* Directory hasn't been recursed,
                 ie. recursing isn't enabled. */

              ssh_xdsprintf(&error_message, "File %s is a "
                           "directory, and recursion is not "
                           "enabled. ",
                           ssh_file_copy_file_get_name
                           (tdata->current_location->file));

              (*gdata->error_callback)(SSH_FC_ERROR,
                                       error_message,
                                       gdata->callback_context);

              ssh_xfree(error_message);
            }
          else
            {
              SSH_TRACE(2, ("Directory %s was empty",
                            ssh_file_copy_file_get_name
                            (tdata->current_location->file)));
            }
        }

      SSH_DEBUG(4, ("Current location list is at it's end. Moving "
                    "to next..."));
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      return SSH_FSM_CONTINUE;
    }

  SSH_NOTREACHED;
  return SSH_FSM_FINISH;
}

void fct_stat_next_file_cb(SshFileClientError error,
                           SshFileCopyFile file,
                           SshFileAttributes attrs,
                           const char *error_msg, SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      if ((attrs->permissions & S_IFMT) == S_IFDIR && !file->dir_entries)
        {
          /* Just a sanity check, I can't really see how we could end up
             here. */
          FCC_ERROR(SSH_FC_OK, ("Directory %s is excluded from copying.",
                                ssh_file_copy_file_get_name(file)));
          ssh_dllist_fw(tdata->current_location->file_list, 1);
        }
      else
        {
          file->attributes = ssh_file_attributes_dup(attrs);
        }
    }
  else
    {
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("stat_next_file: `%s' ", error_msg));
      ssh_dllist_fw(tdata->current_location->file_list, 1);
    }
}
                           
SSH_FSM_STEP(transfer_stat_next_file)
{
  SshFileCopyFile file;
  char *temp_filename, *ph, *basedir;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(tdata->source != NULL);

  file = ssh_dllist_current(tdata->current_location->file_list);

  SSH_ASSERT(file != NULL);

  if (tdata->source_file)
    {
      ph = (char *)ssh_file_copy_file_get_name(file);
      basedir = tdata->source_file ? (char *)ssh_file_copy_file_get_name
        (tdata->source_file) : "";

      ssh_xdsprintf(&temp_filename, "%s%s%s",
                   *basedir ? basedir : "" ,
                   (*ph && *basedir && basedir[strlen(basedir) - 1] != '/' ?
                    "/" : ""),
                   *ph ? ph : "");
    }
  else
    {
      temp_filename = ssh_xstrdup(ssh_file_copy_file_get_name(file));
    }

  if (tdata->source_dir)
    basedir = (char *)ssh_file_copy_file_get_name(tdata->source_dir);
  else
    basedir =
      (char *)ssh_file_copy_file_get_name(tdata->current_location->file);

  ssh_xdsprintf(&(tdata->source_file_name), "%s%s%s",
               *basedir ? basedir : "",
               *basedir && basedir[strlen(basedir) - 1] != '/' ?
               "/" : "", temp_filename);

  ssh_xfree(temp_filename);

  if (file->attributes)
    {
      ssh_xfree(file->attributes);
      file->attributes = NULL;
    }

  SSH_DEBUG(3, ("Statting file %s...", tdata->source_file_name));

  /* XXX can the macros be made more simple? */
  FCC_START_ATTRS(tdata, file, TRUE, fct_stat_next_file_cb);
  
  SSH_FSM_SET_NEXT(transfer_set_next_source);

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (tdata->source->client, tdata->source_file_name,
                      transfer_attribute_cb, thread));
}

void fct_mkdir_cb(SshFileClientError error, SshFileCopyFile file,
                  const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      SSH_DEBUG(3, ("Successfully created directory %s.",
                    ssh_file_copy_file_get_name(file)));
    }
  else if (error == SSH_FX_FAILURE)
    {
      /* When creating directories, this error occurs if the directory
         already existed. So, the destination directory is statted to
         see if it exists, to avoid giving the user false messages. */
      SSH_FSM_SET_NEXT(transfer_lstat_dest_dir);
    }
  else
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("mkdir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_mkdir)
{
  char *temp_filename;
  char *dest_file;




  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  dest_file = (char *)ssh_file_copy_file_get_name(gdata->dest_location->file);


















  ssh_xdsprintf(&temp_filename, "%s%s%s",
               *dest_file ? dest_file : "",
               *dest_file && dest_file[strlen(dest_file) - 1] != '/' ?
               "/" : "",
               ssh_file_copy_file_get_name(tdata->source_file));
  ssh_file_copy_file_register_filename(tdata->current_dest_file,
                                       temp_filename);

  /* XXX if destination file is not a directory, it should've been
     detected before this. */
  FCC_START_STATUS(tdata, tdata->current_dest_file, FALSE, fct_mkdir_cb);
  SSH_FSM_SET_NEXT(transfer_updir);  
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_mkdir
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name(tdata->current_dest_file),
                      NULL, transfer_status_cb, thread));
}

void fct_lstat_dest_dir_cb(SshFileClientError error,
                           SshFileCopyFile file,
                           SshFileAttributes attrs,
                           const char *error_msg,
                           SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);
  
  if (error == SSH_FX_OK)
    {
      /* File (dir) exists. */
      SSH_TRACE(2, ("Destination directory %s already existed.",
                    ssh_file_copy_file_get_name(file)));
    }
  else
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("lstat_dest_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_lstat_dest_dir)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  FCC_START_ATTRS(tdata, tdata->current_dest_file, FALSE,
                  fct_lstat_dest_dir_cb);
  SSH_FSM_SET_NEXT(transfer_updir);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name
                      (tdata->current_dest_file),
                      transfer_attribute_cb,
                      thread));
}

SSH_FSM_STEP(transfer_updir)
{
  TransferThreadContext child_tdata;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (gdata->attrs->unlink_source && gdata->attrs->preserve_attributes &&
      !(tdata->source_file->attributes
        && tdata->source_file->attributes->flags &
           SSH_FILEXFER_ATTR_PERMISSIONS
        && tdata->source_file->attributes->flags &
        SSH_FILEXFER_ATTR_ACMODTIME))
    {
      tdata->after_lstat_src_dir_state = transfer_updir;
      SSH_FSM_SET_NEXT(transfer_lstat_src_dir);
      return SSH_FSM_CONTINUE;
    }

  if (tdata->after_lstat_src_dir_state)
    {
      /* so that we don't end to an infinite loop. (lstat_src_dir
         checks for the existence of
         tdata->after_lstat_src_dir_state)*/
      tdata->after_lstat_src_dir_state = NULL_FNPTR;
    }

  SSH_FSM_SET_NEXT(transfer_chmod_destdir);

  child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
  tdata->child = ssh_fsm_thread_create(gdata->fsm,
                                       transfer_set_next_source,
                                       NULL_FNPTR,
                                       transfer_thread_destructor,
                                       child_tdata);
  SSH_VERIFY(tdata->child != NULL);
  *child_tdata = *tdata;

  child_tdata->source_file = ssh_file_copy_file_dup(tdata->source_file);
  if (tdata->source_file_name)
    child_tdata->source_file_name = ssh_xstrdup(tdata->source_file_name);
  child_tdata->current_dest_file =
    ssh_file_copy_file_dup(tdata->current_dest_file);

  child_tdata->child = NULL;
  child_tdata->parent = thread;

  return SSH_FSM_SUSPENDED;
}

void fct_chmod_dest_dir_cb(SshFileClientError error,
                           SshFileCopyFile file,
                           const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error != SSH_FX_OK)
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("chmod_dest_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_chmod_destdir)
{
  SshFileAttributesStruct attrs;
  SshFileAttributes orig_attrs;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (!gdata->attrs->preserve_attributes)
    {
      SSH_FSM_SET_NEXT(transfer_rm_src_dir);
      return SSH_FSM_CONTINUE;
    }








  orig_attrs = ssh_file_copy_file_get_attributes(tdata->source_file);

  if (!(orig_attrs &&
        orig_attrs->flags & SSH_FILEXFER_ATTR_PERMISSIONS &&
        orig_attrs->flags & SSH_FILEXFER_ATTR_ACMODTIME))
    {
      /* Attrs doesn't exist, or it isn't complete. We have to lstat
         the file again. */
      SSH_FSM_SET_NEXT(transfer_lstat_src_dir);
      return SSH_FSM_CONTINUE;
    }

  attrs = *orig_attrs;

  /* Remove size from flags. */
  attrs.flags &= ~SSH_FILEXFER_ATTR_SIZE;
  /* Remove uidgid from flags. */
  attrs.flags &= ~SSH_FILEXFER_ATTR_UIDGID;














  FCC_START_STATUS(tdata, tdata->current_dest_file, FALSE,
                   fct_chmod_dest_dir_cb);
  SSH_FSM_SET_NEXT(transfer_rm_src_dir);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_setstat
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name(tdata->current_dest_file),
                      &attrs, transfer_status_cb, thread));
}

void fct_lstat_src_dir_cb(SshFileClientError error,
                          SshFileCopyFile file,
                          SshFileAttributes attrs,
                          const char *error_msg,
                          SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      ssh_file_copy_file_register_attributes(tdata->source_file,
                                             ssh_file_attributes_dup(attrs));
    }
  else if (error == SSH_FX_NO_SUCH_FILE && gdata->attrs->unlink_source)
    {
      /* This should not happen, but here is the code for the situation
         anyway. */
      SSH_FSM_SET_NEXT(transfer_rm_src_dir);
      FCC_ERROR(SSH_FC_ERROR_NO_SUCH_FILE, ("lstat_src_dir: %s, error_msg"));
    }
  else
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("lstat_src_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_lstat_src_dir)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  const char *basedir;

  ssh_xfree(tdata->source_file_name);

  basedir = ssh_file_copy_file_get_name(tdata->source_dir);
  ssh_xdsprintf(&(tdata->source_file_name), "%s%s%s",
                *basedir ? basedir : "",
                *basedir && basedir[strlen(basedir) - 1] != '/' ? "/" : "",
                ssh_file_copy_file_get_name(tdata->source_file));

  FCC_START_ATTRS(tdata, tdata->source_file, TRUE,
                  fct_lstat_src_dir_cb);
  if (tdata->after_lstat_src_dir_state)
    SSH_FSM_SET_NEXT(tdata->after_lstat_src_dir_state);
  else
    SSH_FSM_SET_NEXT(transfer_chmod_destdir);

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                     (tdata->source->client, tdata->source_file_name,
                      transfer_attribute_cb, thread));
}

void fct_rm_src_dir_cb(SshFileClientError error,
                       SshFileCopyFile file,
                       const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      SSH_DEBUG(2, ("Successfully removed source directory `%s'",
                    ssh_file_copy_file_get_name(file)));
    }
  else
    {
      /* For future changes; we're already going to this state. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("rm_src_dir: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_rm_src_dir)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (!gdata->attrs->unlink_source)
    {
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      return SSH_FSM_CONTINUE;
    }

  if (!tdata->source_file_name)
    {
      const char *basedir;

      basedir = ssh_file_copy_file_get_name(tdata->source_dir);
      ssh_xdsprintf(&(tdata->source_file_name), "%s%s%s",
                   *basedir ? basedir : "",
                   *basedir && basedir[strlen(basedir) - 1] != '/' ? "/" : "",
                   ssh_file_copy_file_get_name(tdata->source_file));
    }

  FCC_START_STATUS(tdata, tdata->source_file, TRUE,
                  fct_rm_src_dir_cb);
  SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_rmdir
                     (tdata->source->client, tdata->source_file_name,
                      transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_wake_parent_and_finish)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
 
  if (tdata->parent)
    ssh_fsm_continue(tdata->parent);

  return SSH_FSM_FINISH;
}

void fct_parse_raw_stat_cb(SshFileClientError error,
                           SshFileCopyFile file,
                           SshFileAttributes attrs,
                           const char *error_msg,
                           SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      SshFileCopyFileListItem item;
      SshFileCopyFile cur_file;
      SshFileCopyLocation to_be_deleted;

      /* Drop raw location, and make a new one. Put it back to the
         list in the place of the current one. */
      /* XXX This probably needs clean-up. Works now, though... */
      item = tdata->current_item;

      SSH_ASSERT(item != NULL);
      SSH_ASSERT(item->files->raw);

      SSH_ASSERT(ssh_dllist_is_current_valid(item->files->file_list));
      cur_file = (SshFileCopyFile) ssh_dllist_current(item->files->file_list);

      /* XXX `cur_file' seems to be equivalent to `file'.  Will be fixed
         when I clean up the structures. */
      to_be_deleted = item->files;

      item->files = ssh_file_copy_location_allocate(TRUE);

      ssh_file_copy_location_add_file(item->files, attrs, cur_file->name);

      ssh_dllist_rewind(item->files->file_list);
      ssh_file_copy_location_destroy(to_be_deleted);
      tdata->current_location = item->files;
    }
  else
    {
      /* Some error occurred. Drop file, and report an error. */
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("parse_raw_stat: %s", error_msg));
      ssh_dllist_fw(tdata->current_location->file_list, 1);
    }
}

SSH_FSM_STEP(transfer_parse_raw_source)
{
  SshFileCopyFile file;

  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_PRECOND(tdata->current_location->raw);
  SSH_PRECOND(tdata->current_location->source);

  /* stat file */
  SSH_ASSERT(ssh_dllist_is_current_valid
             (tdata->current_location->file_list));

  file =
    (SshFileCopyFile) ssh_dllist_current(tdata->current_location->file_list);

  SSH_ASSERT(file != NULL);
  
  FCC_START_ATTRS(tdata, file, TRUE, fct_parse_raw_stat_cb);
  SSH_FSM_SET_NEXT(transfer_set_next_source);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->source->client, ssh_file_copy_file_get_name(file),
                      transfer_attribute_cb, thread));
}

void fct_stat_source_cb(SshFileClientError error,
                        SshFileCopyFile file,
                        SshFileAttributes attrs,
                        const char *error_msg,
                        SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      /* Store attributes */
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("stat: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_stat_source)
{
  char *basedir;

  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  basedir = (char *)ssh_file_copy_file_get_name(tdata->source_dir);

  ssh_xfree(tdata->source_file_name);
  tdata->source_file_name = NULL;
  ssh_xdsprintf(&(tdata->source_file_name), "%s%s%s",
               *basedir ? basedir : "",
               *basedir && basedir[strlen(basedir) - 1] != '/' ? "/" : "",
               ssh_file_copy_file_get_name(tdata->source_file));


  /* Note that this doesn't affect us, if we just SSH_FSM_CONTINUE. This
     needs be here (before SSH_FSM_SET_NEXT), because otherwise it would
     get the wrong current state. */
  FCC_START_ATTRS(tdata, tdata->source_file, TRUE, fct_stat_source_cb);

  if (gdata->dest_location->file->attributes == NULL)
    SSH_FSM_SET_NEXT(transfer_stat_dest_location);
  else
    SSH_FSM_SET_NEXT(transfer_open_dest);
  /* If file has attributes, move to next state. */
  if (tdata->source_file->attributes == NULL ||
      !(tdata->source_file->attributes->flags &
        SSH_FILEXFER_ATTR_SIZE) ||
      !(tdata->source_file->attributes->flags &
        SSH_FILEXFER_ATTR_PERMISSIONS) ||
      ((gdata->attrs->preserve_attributes == TRUE) &&
       (!(tdata->source_file->attributes->flags &
          SSH_FILEXFER_ATTR_ACMODTIME)))
#ifdef S_IFLNK
      || ((tdata->source_file->attributes->permissions & S_IFMT) ==
          S_IFLNK)
#endif /* S_IFLNK */
      )
    {
      tdata->dealing_with_source = TRUE;

      if (tdata->source_file->attributes)
        {
          ssh_xfree(tdata->source_file->attributes);
          tdata->source_file->attributes = NULL;
        }

      SSH_DEBUG(3, ("Statting source file %s...", tdata->source_file_name));

      SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                         (tdata->source->client,
                          tdata->source_file_name,
                          transfer_attribute_cb, thread));
    }
  else
    {
      return SSH_FSM_CONTINUE;
    }
}

void fct_stat_dest_location_cb(SshFileClientError error,
                               SshFileCopyFile file,
                               SshFileAttributes attrs,
                               const char *error_msg,
                               SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      /* Store attributes */
      ssh_file_copy_file_register_attributes
        (gdata->dest_location->file, ssh_file_attributes_dup(attrs));
    }
  else if (error == SSH_FX_NO_SUCH_FILE)
    {
      /* Doesn't really matter. */
      SSH_DEBUG(2, ("Destination file didn't exist."));
      ssh_file_copy_file_register_attributes(gdata->dest_location->file, NULL);
    }
  else
    {
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      /* Drop file. */
      ssh_dllist_fw(tdata->current_location->file_list, 1);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("stat_dest_before_rm: %s", error_msg));
    }
}

void fct_stat_dest_cb(SshFileClientError error,
                      SshFileCopyFile file,
                      SshFileAttributes attrs,
                      const char *error_msg,
                      SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_OK)
    {
      /* Store attributes */
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else if (error != SSH_FX_NO_SUCH_FILE)
    {
      /* We don't care if the destination file didn't previously
         exist. */
      /* XXX We should probably free something here. */
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("stat: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_stat_dest_location)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_DEBUG(3, ("Statting destination file %s before trying to remove it...",
                ssh_file_copy_file_get_name(gdata->dest_location->file)));

  FCC_START_ATTRS(tdata, gdata->dest_location->file, FALSE,
                  fct_stat_dest_location_cb);
  SSH_FSM_SET_NEXT(transfer_open_dest);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name(gdata->dest_location->file),
                      transfer_attribute_cb, thread));
}

SSH_FSM_STEP(transfer_stat_dest)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  
  SSH_DEBUG(3, ("Statting destination file %s...",
                ssh_file_copy_file_get_name(tdata->current_dest_file)));

  FCC_START_ATTRS(tdata, tdata->current_dest_file, FALSE, fct_stat_dest_cb);

  SSH_FSM_SET_NEXT(transfer_chmod_dest_before_transfer);

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name(tdata->current_dest_file),
                      transfer_attribute_cb, thread));
}

void fct_rm_src_cb(SshFileClientError error,
                   SshFileCopyFile file,
                   const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);
#ifdef DEBUG_LIGHT
  FCC_TDATA(TransferThreadContext);
#endif /* DEBUG_LIGHT */
  
  if (error == SSH_FX_OK)
    SSH_DEBUG(3, ("Source file '%s' removed successfully.",
                  tdata->source_file_name));
  else
    FCC_ERROR(fcc_fx_err_to_fc_err(error), ("rm_src: %s", error_msg));
}

SSH_FSM_STEP(transfer_rm_source)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  FCC_START_STATUS(tdata, tdata->source_file, TRUE, fct_rm_src_cb);
  SSH_FSM_SET_NEXT(transfer_close_dest);

  if (!gdata->attrs->unlink_source)
    {
      return SSH_FSM_CONTINUE;
    }

  SSH_TRACE(2, ("Removing source file %s...", tdata->source_file_name));

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_remove
                     (tdata->source->client, tdata->source_file_name,
                      transfer_status_cb, thread));
}

void fct_open_cb(SshFileClientError error,
                 SshFileCopyFile file,
                 SshFileHandle handle, const char *error_msg,
                 SshFSMThread thread)
{
  FCC_DATA(SshFileCopyTransferContext, TransferThreadContext);

  if (error == SSH_FX_OK)
    {
      /* Store handle */
      SSH_DEBUG(4, ("File `%s' opened successfully.",
                    ssh_file_copy_file_get_name(file)));
      file->handle = handle;
    }
  else
    {
      /* Drop file, and report error. */
      ssh_dllist_fw(tdata->current_location->file_list, 1);
      SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("open: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_open_dest)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  char *temp_filename;

#ifndef ALWAYS_REMOVE_DEST_FILE
  if (gdata->attrs->destination_must_be_dir &&
      (!gdata->dest_location->file->attributes ||
       ((gdata->dest_location->file->attributes->permissions & S_IFMT)
        != S_IFDIR)))
    {
      FCC_ERROR(SSH_FC_ERROR_DEST_NOT_DIR,
                ("destination (%s) is not a directory.",
                 ssh_file_copy_file_get_name(gdata->dest_location->file)));
      /* XXX Kill other threads, etc. etc. */
      return SSH_FSM_FINISH;
    }

  if (gdata->attrs->destination_must_be_dir ||
      (gdata->dest_location->file->attributes &&
       ((gdata->dest_location->file->attributes->permissions & S_IFMT)
        == S_IFDIR)))
    {
      char *dest_file =
        (char *)ssh_file_copy_file_get_name(gdata->dest_location->file);


















      ssh_xdsprintf(&temp_filename, "%s%s%s",
                   *dest_file ? dest_file : "",
                   *dest_file && dest_file[strlen(dest_file) - 1] != '/' ?
                   "/" : "",
                   ssh_file_copy_file_get_name(tdata->source_file));

    }
  else
    {
      temp_filename = ssh_xstrdup(ssh_file_copy_file_get_name
                                  (gdata->dest_location->file));
    }

  ssh_file_copy_file_register_filename(tdata->current_dest_file,
                                       temp_filename);
#endif /* ALWAYS_REMOVE_DEST_FILE */
  SSH_DEBUG(4, ("Opening file %s...",
                ssh_file_copy_file_get_name(tdata->current_dest_file)));

  FCC_START_HANDLE(tdata, tdata->current_dest_file, FALSE, fct_open_cb);
  SSH_FSM_SET_NEXT(transfer_open_source);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_open
                     (gdata->destination->client,
                      ssh_file_copy_file_get_name(tdata->current_dest_file),
                      O_WRONLY | O_CREAT | O_TRUNC,
                      NULL, transfer_handle_cb, thread));
}

SSH_FSM_STEP(transfer_open_source)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  
  SSH_DEBUG(4, ("Opening file %s...", tdata->source_file_name));

  FCC_START_HANDLE(tdata, tdata->source_file, TRUE, fct_open_cb);
  SSH_FSM_SET_NEXT(transfer_stat_dest);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_open
                     (tdata->source->client, tdata->source_file_name,
                      O_RDONLY, NULL, transfer_handle_cb, thread));
}

void fct_chmod_dest_before_cb(SshFileClientError error,
                              SshFileCopyFile file,
                              const char *error_msg, SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyTransferContext);

  if (error == SSH_FX_FAILURE &&
      ssh_file_client_get_version(gdata->destination->client) <= 0)
    {
      /* If the server is of older breed, these warnings are probably
         bogus. (atleast if the server doesn't have futimes function,
         but has utimes function) */
      return;
    }
  if (error != SSH_FX_OK)
    {
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("chmod_dest_before_transfer: %s", error_msg));
    }
}

SSH_FSM_STEP(transfer_chmod_dest_before_transfer)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributesStruct attrs;

  memset(&attrs, 0, sizeof(attrs));

  SSH_ASSERT(tdata->current_dest_file->attributes != NULL);
  SSH_ASSERT(tdata->current_dest_file->attributes->flags &
             SSH_FILEXFER_ATTR_PERMISSIONS);
  SSH_ASSERT(tdata->source_file->attributes != NULL);
  SSH_ASSERT(tdata->source_file->attributes->flags &
             SSH_FILEXFER_ATTR_PERMISSIONS);

  /* These are here because of the SSH_FSM_CONTINUE in the following
     if-clauses. */
  FCC_START_STATUS(tdata, tdata->current_dest_file, FALSE,
                   fct_chmod_dest_before_cb);
  SSH_FSM_SET_NEXT(transfer_start);

  if ((tdata->current_dest_file->attributes->permissions & S_IFMT) !=
      S_IFREG)
    {
      SSH_DEBUG(2, ("Destination file is not a regular file, not changing "
                    "attributes."));
      return SSH_FSM_CONTINUE;
    }


  if (gdata->attrs->preserve_attributes)
    {
      if (tdata->source_file->attributes->flags &
          SSH_FILEXFER_ATTR_PERMISSIONS)
        {
          SSH_DEBUG(4, ("Setting permissions."));
          attrs.permissions =
            tdata->source_file->attributes->permissions;
          attrs.flags |= SSH_FILEXFER_ATTR_PERMISSIONS;
        }
    }
  else
    {
      /* This has to be done anyway, so if these operations are not
         supported, it still valid to tell that to the user. */
      if (tdata->source_file->attributes->permissions & 0111)
        {
          /* Umask. */
          attrs.permissions =
            tdata->current_dest_file->attributes->permissions;

          /* exec-bit. */
          attrs.permissions |=
            (tdata->source_file->attributes->permissions & 0111);

          attrs.flags |= SSH_FILEXFER_ATTR_PERMISSIONS;
        }
      else
        {
          /* The source file doesn't have exec bits set, so we don't
             have to touch it. */
          return SSH_FSM_CONTINUE;
        }
    }







  
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat
                     (tdata->current_dest_file->handle,
                      &attrs, transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_set_acmod_time)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;
  SshFileAttributesStruct attrs;

  memset(&attrs, 0, sizeof(attrs));

  /* These are here because of the SSH_FSM_CONTINUE in the following
     if-clauses. */
  FCC_START_STATUS(tdata, tdata->current_dest_file, FALSE,
                   fct_chmod_dest_before_cb);
  SSH_FSM_SET_NEXT(transfer_close_src);

  if ((tdata->current_dest_file->attributes->permissions & S_IFMT) !=
      S_IFREG || !gdata->attrs->preserve_attributes)
    {
      SSH_DEBUG(2, ("Destination file is not a regular file, not changing "
                    "attributes."));
      return SSH_FSM_CONTINUE;
    }

  if (tdata->source_file->attributes->flags &
      SSH_FILEXFER_ATTR_ACMODTIME)
    {
      SSH_DEBUG(4, ("Setting modification time."));
      attrs.mtime = tdata->source_file->attributes->mtime;
      attrs.atime = tdata->source_file->attributes->atime;
      attrs.flags |= SSH_FILEXFER_ATTR_ACMODTIME;
    }
  else
    {
      SSH_DEBUG(2, ("Source file attributes do not contain times, skipping."));
      return SSH_FSM_CONTINUE;
    }

  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_fsetstat
                     (tdata->current_dest_file->handle,
                      &attrs, transfer_status_cb, thread));
}

SSH_FSM_STEP(transfer_close_src)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_TRACE(3, ("Closing source file %s...",
                ssh_file_copy_file_get_name(tdata->source_file)));

  SSH_FSM_SET_NEXT(transfer_rm_source);

  ssh_file_client_close(tdata->source_file->handle,
                        NULL_FNPTR,
                        NULL);

  tdata->source_file->handle = NULL;
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_close_dest)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  SSH_TRACE(3, ("Closing dest file %s...",
                ssh_file_copy_file_get_name
                (tdata->current_dest_file)));

  SSH_FSM_SET_NEXT(transfer_one_done);

  ssh_file_client_close(tdata->current_dest_file->handle,
                        NULL_FNPTR,
                        NULL);

  tdata->current_dest_file->handle = NULL;

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_one_done)
{
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  ssh_dllist_fw(tdata->current_location->file_list, 1);

  SSH_TRACE(1, ("Finished with file %s.",
                ssh_file_copy_file_get_name
                (tdata->current_dest_file)));

  if (tdata->current_dest_file)
    {
      ssh_file_copy_file_destroy(tdata->current_dest_file);
      tdata->current_dest_file = NULL;
    }
  if (tdata->source_file)
    {
      ssh_file_copy_file_destroy(tdata->source_file);
      tdata->source_file = NULL;
    }

  SSH_FSM_SET_NEXT(transfer_wake_parent_and_finish);
  return SSH_FSM_CONTINUE;
}

typedef struct CompletionTimeoutCtxRec
{
  SshFileCopyTransferReadyCallback completion_callback;
  void *callback_context;
} *CompletionTimeoutCtx;

/* The calling of the completion callback from the bottom of the event
   loop is a fix for a persistent bug in Windows. */
void completion_timeout(void *context)
{
  CompletionTimeoutCtx ctx = (CompletionTimeoutCtx) context;

  SSH_PRECOND(ctx != NULL);

  (*ctx->completion_callback)(SSH_FC_OK, "", ctx->callback_context);

  memset(ctx, 'F', sizeof(*ctx));
  ssh_xfree(ctx);
}

SSH_FSM_STEP(transfer_done)
{
  CompletionTimeoutCtx completion_timeout_ctx = NULL;
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  
  completion_timeout_ctx = ssh_xcalloc(1, sizeof(*completion_timeout_ctx));
  completion_timeout_ctx->completion_callback = gdata->completion_callback;
  completion_timeout_ctx->callback_context = gdata->callback_context;

  ssh_register_timeout(0L, 0L, completion_timeout, completion_timeout_ctx);

  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_abort)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  /* Abort the transfer. */

  SSH_TRACE(2, ("Aborting transfer..."));

  ssh_cancel_timeouts(progress_timeout_cb, gdata);

  gdata->operation_handle = NULL;

  if (gdata->abort_notify_cb)
    {
      SSH_DEBUG(3, ("Calling abort_notify callback..."));
      (*gdata->abort_notify_cb)(gdata->callback_context);
    }
  SSH_FSM_SET_NEXT(transfer_cleanup);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(transfer_cleanup)
{
  SshFileCopyTransferContext gdata = (SshFileCopyTransferContext)fsm_context;
  TransferThreadContext tdata = (TransferThreadContext)thread_context;

  if (tdata->child)
    {
      TransferThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(tdata->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(tdata->child);
      tdata->child = NULL;
    }

  if (gdata->reader)
    {
      ssh_fsm_kill_thread(gdata->reader);
      gdata->reader = NULL;
    }

  if (gdata->writer)
    {
      ssh_fsm_kill_thread(gdata->writer);
      gdata->writer = NULL;
    }

  if (gdata->mangler)
    {
      ssh_fsm_kill_thread(gdata->mangler);
      gdata->mangler = NULL;
    }

  ssh_fsm_destroy(gdata->fsm);

  if (gdata->operation_handle)
    ssh_operation_unregister(gdata->operation_handle);

  if (gdata->in_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->in_blocking);
      ssh_fsm_condition_destroy(gdata->in_blocking);
      gdata->in_blocking = NULL;
    }
  if (gdata->out_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->out_blocking);
      ssh_fsm_condition_destroy(gdata->out_blocking);
      gdata->out_blocking = NULL;
    }
  if (gdata->mangle_blocking)
    {
      /* Everybody should be killed already, this is just for sanity. */
      SSH_FSM_CONDITION_BROADCAST(gdata->mangle_blocking);
      ssh_fsm_condition_destroy(gdata->mangle_blocking);
      gdata->mangle_blocking = NULL;
    }

  if (gdata->buf_queue)
    ssh_adt_destroy(gdata->buf_queue);

  ssh_time_measure_free(gdata->transfer_timer);
  ssh_time_measure_free(gdata->total_transfer_timer);

  ssh_xfree(gdata);

  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct transfer_states_array[] =
{
  { "fct_check_src_conn", "Check source connection", transfer_check_src_conn },
  { "fct_check_dest_conn", "Check destination connection",
    transfer_check_dest_conn },
  { "fct_conn_failed", "Connection has failed", transfer_conn_failed },

  { "fct_stat_source", "Stat source file", transfer_stat_source },
  { "fct_stat_dest", "Stat destination file", transfer_stat_dest },
  { "fct_stat_dest_location", "Stat destination location.",
    transfer_stat_dest_location },
  { "fct_remove_source", "Remove source file", transfer_rm_source },
  { "fct_open_dest", "Open destination file", transfer_open_dest },
  { "fct_open_source", "Open source file", transfer_open_source },
  { "fct_chmod_dest_before", "Change permissions of dest file before transfer",
    transfer_chmod_dest_before_transfer },
  { "fct_set_acmod_time", "Change access and modification times on file",
    transfer_set_acmod_time },
  
#include "sshfc_trcore_states.h"

  { "fct_mkdir", "Make a new directory", transfer_mkdir },
  { "fct_lstat_dest_dir", "lstat destination directory",
    transfer_lstat_dest_dir },
  { "fct_updir", "Start going through files of subdirectory", transfer_updir },
  { "fct_chmod_dest_dir",
    "Change permissions and fileattributes of destination directory",
    transfer_chmod_destdir },
  { "fct_remove_source_dir", "Remove source directory", transfer_rm_src_dir },
  { "fct_lstat_source_dir", "lstat source directory",
    transfer_lstat_src_dir },

  { "fct_next_file_item", "Set next source file item",
    transfer_set_next_source_item },
  { "fct_next_file", "Set next source file",
    transfer_set_next_source },
  { "fct_stat_next_file", "Stat file during next_file",
    transfer_stat_next_file },
  { "fct_parse_raw_source", "Parse raw source location",
    transfer_parse_raw_source },
  { "fct_close_source", "Close source file", transfer_close_src },
  { "fct_close_dest", "Close destination file", transfer_close_dest },
  { "fct_one_transfer_done", "Finalize after transfer of one file",
    transfer_one_done },
  { "fct_transfer_done", "Transfer done", transfer_done },

  { "fct_wake_parent_and_finish", "Wake parent and finish",
    transfer_wake_parent_and_finish },
  { "fct_abort", "Abort file transfer exchange", transfer_abort }
};

void *subdir_rewind(void *item, void *ctx)
{
  SshFileCopyFile file = (SshFileCopyFile) item;

  SSH_PRECOND(file != NULL);

  if (file->dir_entries)
    {
      ssh_dllist_mapcar(file->dir_entries->file_list,
                        subdir_rewind, NULL);
      ssh_dllist_rewind(file->dir_entries->file_list);
    }

  return item;
}

void *location_list_rewind(void *item, void *ctx)
{
  SshFileCopyFileListItem list_item = (SshFileCopyFileListItem) item;

  SSH_PRECOND(list_item->files->file_list != NULL);

  ssh_dllist_mapcar(list_item->files->file_list, subdir_rewind, NULL);
  ssh_dllist_rewind(list_item->files->file_list);

  return item;
}

void *subdir_count(void *item, void *ctx)
{
  SshFileCopyFile file = (SshFileCopyFile) item;
  int *count = (int *) ctx;

  SSH_PRECOND(file != NULL);
  SSH_PRECOND(count != NULL);

  if (file->dir_entries)
    {
      *count +=ssh_dllist_length(file->dir_entries->file_list);
    }

  return item;
}

void *location_list_count(void *item, void *ctx)
{
  SshFileCopyFileListItem list_item = (SshFileCopyFileListItem) item;
  int *count = (int *) ctx;

  SSH_PRECOND(list_item->files->file_list != NULL);
  SSH_PRECOND(count != NULL);

  *count +=ssh_dllist_length(list_item->files->file_list);

  ssh_dllist_mapcar(list_item->files->file_list, subdir_count, count);

  return item;
}

void *location_list_mapcar_subdir_rewind(void *item, void *ctx)
{
  SshFileCopyFile file = (SshFileCopyFile) item;

  SSH_PRECOND(file != NULL);

  if (strcmp("", ssh_file_copy_file_get_name(file)) == 0)
    return item;

  if (file->dir_entries)
    {
      ssh_dllist_rewind(file->dir_entries->file_list);
      ssh_dllist_mapcar(file->dir_entries->file_list,
                        location_list_mapcar_subdir_rewind, ctx);
    }

  return item;
}

void *location_list_rewind_mapcar(void *item, void *ctx)
{
  SshFileCopyFileListItem list_item = (SshFileCopyFileListItem) item;

  SSH_PRECOND(list_item->files->file_list != NULL);

  ssh_dllist_rewind(list_item->files->file_list);
  ssh_dllist_mapcar(list_item->files->file_list,
                    location_list_mapcar_subdir_rewind, ctx);

  return item;
}

void ssh_file_copy_file_list_rewind(SshDlList file_list)
{
  ssh_dllist_rewind(file_list);
  ssh_dllist_mapcar(file_list, location_list_rewind_mapcar, NULL);
}

typedef struct FileListMapcarContextRec
{
  SshFileCopyFileListMapCarFunc func;
  void *context;
} *FileListMapcarContext;

void *location_list_mapcar_subdir(void *item, void *ctx)
{
  SshFileCopyFile file = (SshFileCopyFile) item;
  FileListMapcarContext context = (FileListMapcarContext) ctx;

  SSH_PRECOND(file != NULL);
  SSH_PRECOND(context != NULL);
  SSH_PRECOND(context->func != NULL_FNPTR);

  if (strcmp("", ssh_file_copy_file_get_name(file)) != 0)
    (*context->func)(file, context->context);

  if (file->dir_entries)
    {
      ssh_dllist_mapcar(file->dir_entries->file_list,
                        location_list_mapcar_subdir, context);
    }

  return item;
}

void *location_list_mapcar(void *item, void *ctx)
{
  FileListMapcarContext context = (FileListMapcarContext) ctx;
  SshFileCopyFileListItem list_item = (SshFileCopyFileListItem) item;

  SSH_PRECOND(context != NULL);
  SSH_PRECOND(context->func != NULL_FNPTR);
  SSH_PRECOND(list_item->files->file_list != NULL);

  if (list_item->files->file)
    (*context->func)(list_item->files->file, context->context);

  ssh_dllist_mapcar(list_item->files->file_list,
                    location_list_mapcar_subdir, context);

  return item;
}

void ssh_file_copy_file_list_mapcar(SshDlList file_list,
                                    SshFileCopyFileListMapCarFunc func,
                                    void *context)
{
  FileListMapcarContext ctx;
  ctx = ssh_xcalloc(1, sizeof(*ctx));
  ctx->func = func;
  ctx->context = context;

  ssh_dllist_rewind(file_list);

  ssh_dllist_mapcar(file_list, location_list_mapcar, ctx);
}

void mapcar_count_func(SshFileCopyFile file, void *context)
{
  int *count = (int *) context;
  SSH_PRECOND(context != NULL);
  SSH_PRECOND(file != NULL);
  (*count)++;
}

void transfer_operation_abort_cb(void *operation_context)
{
  SshFileCopyTransferContext gdata =
    (SshFileCopyTransferContext) operation_context;
  TransferThreadContext tdata;

  if (gdata->main_thread == NULL)
    return;
  
  tdata = (TransferThreadContext)ssh_fsm_get_tdata(gdata->main_thread);

  ssh_operation_abort(tdata->op_handle);
  tdata->op_handle = NULL;
  
  ssh_fsm_set_next(gdata->main_thread, transfer_abort);
  ssh_fsm_continue(gdata->main_thread);
}

SshOperationHandle
ssh_file_copy_transfer_files(SshFileCopyConnection source,
                             SshDlList file_list,
                             SshFileCopyConnection destination,
                             SshFileCopyLocation dest_location,
                             SshFileCopyTransferAttrs attrs,
                             SshFileCopyTransferReadyCallback ready_cb,
                             SshFileCopyErrorCallback error_cb,
                             SshFileCopyTransferProgressCallback
                             progress_cb,
                             SshFileCopyTransferAbortNotifyCallback
                             abort_notify_cb,
                             void *context)
{
  SshFileCopyTransferContext transfer_context;
  TransferThreadContext child_tdata;
  int count;

  SSH_PRECOND(source != NULL);
  SSH_PRECOND(file_list != NULL);
  SSH_PRECOND(destination != NULL);
  SSH_PRECOND(dest_location != NULL);
  SSH_PRECOND(ready_cb != NULL_FNPTR);
  SSH_PRECOND(error_cb != NULL);

  /* Count files in file_list (if more than one, destination must
     be directory). */
  count = 0;
  ssh_file_copy_file_list_mapcar(file_list,
                                 mapcar_count_func,
                                 &count);


  SSH_DEBUG(2, ("File list has %d files.", count));

  if (count <= 0)
    {
      (*ready_cb)(SSH_FC_OK, "No files to transfer.", context);
      return NULL;
    }

  if (attrs == NULL)
    {
      /* Use default attrs. */
      attrs = ssh_xcalloc(1, sizeof(*attrs));
    }

  if (count > 2)
    {
      /* Ignore the first file items basedir. If we have multiple
         file_items, this doesn't matter, as then we have multiple
         files, too. */
      attrs->destination_must_be_dir = TRUE;
    }

  transfer_context = ssh_xcalloc(1, sizeof(*transfer_context));
  transfer_context->fsm = ssh_fsm_create(transfer_context);
  SSH_VERIFY(transfer_context->fsm != NULL);
  ssh_fsm_register_debug_names(transfer_context->fsm, transfer_states_array,
                               SSH_FSM_NUM_STATES(transfer_states_array));
                               
  transfer_context->source = source;
  transfer_context->file_list = file_list;
  transfer_context->destination = destination;
  transfer_context->dest_location = dest_location;
  transfer_context->attrs = attrs;
  transfer_context->completion_callback = ready_cb;
  transfer_context->error_callback = error_cb;
  transfer_context->progress_callback = progress_cb;
  transfer_context->abort_notify_cb = abort_notify_cb;
  transfer_context->callback_context = context;

  if (attrs->transfer_mode & (SSH_FC_TRANSFER_MODE_ASCII |
                              SSH_FC_TRANSFER_MODE_AUTO))
    {
      if (attrs->transfer_mode & SSH_FC_TRANSFER_MODE_AUTO)
        SSH_ASSERT(attrs->ascii_extensions != NULL);
      if (!(attrs->transfer_mode & SSH_FC_TRANSFER_MODE_NL_CONV_ASK))
        {
          SSH_ASSERT(attrs->source_newline != NULL);
          SSH_ASSERT(attrs->dest_newline != NULL);
        }
      else
        {
          SSH_ASSERT(attrs->newline_query_cb != NULL);
        }
    }

  /* Rewind all source file lists. */
  ssh_file_copy_file_list_rewind(file_list);
  
  child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
  transfer_context->main_thread =
    ssh_fsm_thread_create(transfer_context->fsm,
                          transfer_check_src_conn,
                          NULL_FNPTR, NULL_FNPTR /* XXX mem leaks ? */,
                          child_tdata);
  SSH_VERIFY(transfer_context->main_thread != NULL);
  
  ssh_fsm_set_thread_name(transfer_context->main_thread, "main_thread");
  transfer_context->transfer_timer = ssh_time_measure_allocate();
  transfer_context->total_transfer_timer = ssh_time_measure_allocate();

  transfer_context->operation_handle =
    ssh_operation_register(transfer_operation_abort_cb, transfer_context);

  return transfer_context->operation_handle;
}
