/*
  sshfc_recurse.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Functions that perform the recursing of subdirectories.
 */

#include "sshincludes.h"
#include "sshfilecopy.h"
#include "sshfilecopyi.h"








#define SSH_DEBUG_MODULE "SshFCRecurse"

/**********
 * ssh_file_copy_recurse_dirs() related stuff.
 **********/

#define SSH_FCR_MAX_SYMLINK_LEVEL_DEPTH 64

/* Global data. */
typedef struct SshFileCopyRecurseContextRec
{
  SshFSM fsm;

  SshFileCopyErrorCallback error_callback;
  SshFileCopyRecurseReadyCallback completion_callback;
  void *callback_context;

  SshFSMThread main_thread;

  /* File list given by application. */
  SshDlList file_list;

  /* Optional function to be executed on files, given by
     application. */
  SshFileCopyRecurseFileFunc func;

  /* Recurse attrs, given by application. */
  SshFileCopyRecurseAttrs attrs;

  /* New file list to be returned to application. */
  SshDlList new_file_list;

  /* New list item, which is already in the new_file_list, but a
     pointer is kept here for ease of implementation.  */
  SshFileCopyFileListItem new_item;

  /* New directory, which we are currently recursing. */
  SshFileCopyFile new_dir;

  /* Helper variable, used to retain pointer to a dynamically
     malloced string, which is used to lstat a file. The string is
     freed in a callback. */
  char *filename_to_be_statted;

  /* Connection handling. */
  SshFileCopyConnection source;
  SshFileCopyLocation location;
  
  /* Error message given to application, when fatal error encountered. */
  char *error_message;

} *SshFileCopyRecurseContext;

typedef struct RecurseThreadContextRec
{
  /* Parent thread, that will be waken when child is ready. */
  SshFSMThread parent;
  /* Child thread of the thread. */
  SshFSMThread child;

  /* The dir we are currently working on. */
  SshFileCopyFile current_dir;

  /* File, which we got from the last call to readdir. */
  SshFileCopyFile new_file;

  /* The count of symlinks we have traversed (in depth). */
  int symlink_level;

  /* For unified callback handling macros. */
  Boolean dealing_with_source;
  
  SshOperationHandle op_handle;
  SshFSMStepCB current_state;
  SshFileCopyFile current_file;
  FCCommonStatusCB status_cb;
  FCCommonHandleCB handle_cb;
  FCCommonDataCB data_cb;
  FCCommonNameCB name_cb;
  FCCommonAttributeCB attrs_cb;
} *RecurseThreadContext;

/* Main thread. */
void recurse_child_thread_destructor(SshFSM fsm, void *tdata)
{
  RecurseThreadContext thread_context = (RecurseThreadContext) tdata;

  SSH_PRECOND(thread_context != NULL);

  if (thread_context->op_handle)
    {
      ssh_operation_abort(thread_context->op_handle);
      thread_context->op_handle = NULL;
    }
  if (thread_context->parent)
    {
      RecurseThreadContext parent_tdata;
      parent_tdata = ssh_fsm_get_tdata(thread_context->parent);
      parent_tdata->child = NULL;
    }

  if (thread_context->child)
    {
      RecurseThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(thread_context->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(thread_context->child);
      thread_context->child = NULL;
    }
#if 0
  if (child_tdata->current_dir)
    ssh_file_copy_file_destroy(child_tdata->current_dir);
  /* XXX Causes weird errors. Check with atom. */
  if (child_tdata->new_file)
    ssh_file_copy_file_destroy(child_tdata->new_file);
#endif
}

/* Forward declarations. */
SSH_FSM_STEP(recurse_check_conn);
SSH_FSM_STEP(recurse_next_file);
SSH_FSM_STEP(recurse_parse_raw);
SSH_FSM_STEP(recurse_lstat_file);
SSH_FSM_STEP(recurse_opendir);
SSH_FSM_STEP(recurse_readdir);
SSH_FSM_STEP(recurse_process_file);
SSH_FSM_STEP(recurse_readdir_lstat_file);
SSH_FSM_STEP(recurse_current_dir_at_end);
SSH_FSM_STEP(recurse_wake_parent_and_finish);
SSH_FSM_STEP(recurse_done);
SSH_FSM_STEP(recurse_cleanup);

#define FCC_CLEANUP(error, message)                             \
do                                                              \
{                                                               \
  (*gdata->completion_callback)((error), (message),             \
                                NULL, NULL,                     \
                                gdata->callback_context);       \
  /* stop transfer and abort */                                 \
  ssh_fsm_set_next(gdata->main_thread, recurse_cleanup);        \
  ssh_fsm_continue(gdata->main_thread);                         \
}                                                               \
while(0)

FCC_GEN_BF_CALLBACK(fcr, SshFileCopyRecurseContext, RecurseThreadContext)
FCC_GEN_HANDLE(fcr)
FCC_GEN_NAME(fcr)
FCC_GEN_ATTR(fcr)

void fcr_conn_completion_cb(SshFileClient client, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshFileCopyRecurseContext gdata =
    (SshFileCopyRecurseContext)ssh_fsm_get_gdata(thread);
  RecurseThreadContext tdata = (RecurseThreadContext)ssh_fsm_get_tdata(thread);

  tdata->op_handle = NULL;

  if (!client)
    {
      (*gdata->completion_callback)(SSH_FC_ERROR_CONNECTION_FAILED,
                                    "Connecting to source failed",
                                    NULL, NULL,
                                    gdata->callback_context);
      ssh_fsm_set_next(gdata->main_thread, recurse_cleanup);
    }
  gdata->source->client = client;
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(recurse_check_conn)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_FSM_SET_NEXT(recurse_next_file);
  
  if (gdata->source->client)
    {
      SSH_DEBUG(4, ("Source connection OK."));
      return SSH_FSM_CONTINUE;
    }

  SSH_DEBUG(2, ("Establishing source connection."));
  SSH_FSM_ASYNC_CALL(tdata->op_handle =
                     ssh_file_copy_connect(gdata->source,
                                           fcr_conn_completion_cb,
                                           thread));

}

SSH_FSM_STEP(recurse_next_file)
{
  RecurseThreadContext child_tdata;
  char *temp_filename;
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_PRECOND(thread == gdata->main_thread);

 restart:
  if (ssh_dllist_is_current_valid(gdata->file_list))
    {
      SshFileCopyFileListItem item;

      item = (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);

      if (item->files->raw)
        {
          SSH_DEBUG(2, ("File is \"raw\", and it needs to "
                        "be parsed."));
          SSH_FSM_SET_NEXT(recurse_parse_raw);
          return SSH_FSM_CONTINUE;
        }

      if (!gdata->func && gdata->new_item == NULL)
        {
          gdata->new_item = ssh_file_copy_file_list_item_allocate();
          gdata->new_item->original_filename =
            ssh_xstrdup(item->original_filename);
          gdata->new_item->files->file =
            ssh_file_copy_file_dup(item->files->file);
          gdata->new_item->files->raw = FALSE;
          SSH_VERIFY(ssh_dllist_add_item(gdata->new_file_list,
                                         gdata->new_item,
                                         SSH_DLLIST_END) ==
                     SSH_DLLIST_OK);

        }

      if (ssh_dllist_is_current_valid(item->files->file_list))
        {
          SshFileCopyFile file;

          file =
            (SshFileCopyFile) ssh_dllist_current(item->files->file_list);

          /* If basedir is not "" generate filename. */
          if (*(ssh_file_copy_file_get_name(item->files->file)) != '\0')
            {
              const char *current_basedir;

              current_basedir =
                ssh_file_copy_file_get_name(item->files->file);

              ssh_dsprintf(&temp_filename, "%s%s%s",
                           current_basedir,
                           current_basedir[strlen(current_basedir) - 1]
                           == '/' ? "" : "/",
                           ssh_file_copy_file_get_name(file));
            }
          else
            {
              temp_filename =
                ssh_xstrdup(ssh_file_copy_file_get_name(file));
            }

          if (file->attributes == NULL ||
              !(file->attributes->flags & SSH_FILEXFER_ATTR_PERMISSIONS))
            {
              SSH_DEBUG(3, ("File %s needs to be statted.",
                            ssh_file_copy_file_get_name(file)));

              gdata->filename_to_be_statted = temp_filename;
              SSH_FSM_SET_NEXT(recurse_lstat_file);
              return SSH_FSM_CONTINUE;
            }

          ssh_dllist_fw(item->files->file_list, 1);

          if ((file->attributes->permissions & S_IFMT) == S_IFDIR ||
              ((file->attributes->permissions & S_IFMT) == S_IFLNK &&
               gdata->attrs->follow_symlinks))
            {
              if (gdata->func &&
                  !(*gdata->func)(temp_filename,
                                  ssh_file_copy_file_get_long_name(file),
                                  ssh_file_copy_file_get_attributes(file),
                                  gdata->attrs,
                                  gdata->callback_context))
                {
                  ssh_xfree(temp_filename);
                  /* This directory was rejected by func for some
                     reason. Skip it. */
                  return SSH_FSM_CONTINUE;
                }

              SSH_DEBUG(3, ("File %s is a directory. Starting "
                            "recursion...",
                            ssh_file_copy_file_get_name(file)));

              child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
              tdata->child =
                ssh_fsm_thread_create(gdata->fsm, recurse_opendir, NULL_FNPTR,
                                      recurse_child_thread_destructor,
                                      child_tdata);
              SSH_VERIFY(tdata->child != NULL);
              child_tdata->parent = thread;

              tdata->current_dir = file;

              child_tdata->current_dir = ssh_file_copy_file_dup(file);
              ssh_file_copy_file_register_filename(child_tdata->current_dir,
                                                   temp_filename);
              if (!gdata->func)
                {
                  gdata->new_dir =
                    ssh_file_copy_file_dup(file);

                  gdata->new_dir->dir_entries =
                    ssh_file_copy_location_allocate(TRUE);

                  gdata->new_dir->dir_entries->parent_dir =
                    gdata->new_item->files;
                  gdata->new_dir->dir_entries->file = gdata->new_dir;

                  SSH_VERIFY(ssh_dllist_add_item
                             (gdata->new_item->files->file_list,
                              gdata->new_dir,
                              SSH_DLLIST_END) == SSH_DLLIST_OK);

                  child_tdata->current_dir->dir_entries =
                    gdata->new_dir->dir_entries;
                }

              return SSH_FSM_SUSPENDED;
            }
          else
            {
              SSH_DEBUG(3, ("File %s is not a directory. Adding it "
                            "to list...",
                            ssh_file_copy_file_get_name(file)));

              if (gdata->func)
                {
                  (void)(*gdata->func)(temp_filename,
                                       ssh_file_copy_file_get_long_name(file),
                                       ssh_file_copy_file_get_attributes(file),
                                       gdata->attrs,
                                       gdata->callback_context);
                }
              else
                {
                  /* Add to list. */
                  SSH_VERIFY(ssh_dllist_add_item
                             (gdata->new_item->files->file_list,
                              ssh_file_copy_file_dup(file),
                              SSH_DLLIST_END) == SSH_DLLIST_OK);
                }

              ssh_xfree(temp_filename);
              return SSH_FSM_CONTINUE;
            }
        }
      else
        {
          if (ssh_dllist_length(item->files->file_list) < 1)
            {
              /* XXX When the basedir is the only file, we should still
                 recurse through it. */
              temp_filename = ssh_xstrdup
                (ssh_file_copy_file_get_name(item->files->file));

              /* XXX This needs to be fixed. */
              SSH_ASSERT(item->files->file->attributes != NULL);
              SSH_ASSERT(item->files->file->attributes->flags &
                         SSH_FILEXFER_ATTR_PERMISSIONS);

              if ((item->files->file->attributes->permissions & S_IFMT) ==
                  S_IFDIR ||
                  ((item->files->file->attributes->permissions & S_IFMT) ==
                  S_IFLNK && gdata->attrs->follow_symlinks))
                {
                  if (gdata->func &&
                      !(*gdata->func)(temp_filename,
                                      ssh_file_copy_file_get_long_name
                                      (item->files->file),
                                      ssh_file_copy_file_get_attributes
                                      (item->files->file),
                                      gdata->attrs,
                                      gdata->callback_context))
                    {
                      ssh_xfree(temp_filename);
                      /* This directory was rejected by func for some
                         reason. Skip it. */
                      return SSH_FSM_CONTINUE;
                    }
                  SSH_DEBUG(3, ("File %s is a directory. Starting "
                                "recursion...",
                                ssh_file_copy_file_get_name
                                (item->files->file)));

                  child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
                  tdata->child =
                    ssh_fsm_thread_create(gdata->fsm,
                                          recurse_opendir, NULL_FNPTR,
                                          recurse_child_thread_destructor,
                                          child_tdata);
                  SSH_VERIFY(tdata->child != NULL);
                  
                  child_tdata->parent = thread;

                  tdata->current_dir = item->files->file;

                  child_tdata->current_dir =
                    ssh_file_copy_file_dup(item->files->file);
                  ssh_file_copy_file_register_filename
                    (child_tdata->current_dir,
                     temp_filename);

                  if (!gdata->func)
                    {
                      gdata->new_dir =
                        ssh_file_copy_file_dup(tdata->current_dir);

                      gdata->new_dir->dir_entries =
                        gdata->new_item->files;

                      gdata->new_dir->dir_entries->parent_dir =
                        NULL;

                      child_tdata->current_dir->dir_entries =
                        gdata->new_dir->dir_entries;
                    }
                }
              else
                {
                  /* XXX Error handling here. */
                  SSH_NOTREACHED;
                }

              ssh_dllist_fw(gdata->file_list, 1);
              return SSH_FSM_SUSPENDED;
            }

          SSH_DEBUG(4, ("Current location list is at it's end. Moving "
                        "to next..."));
          ssh_dllist_fw(gdata->file_list, 1);
          gdata->new_item = NULL;
          goto restart;
        }
    }
  else
    {
      SSH_DEBUG(3, ("No more files in this list."));
      SSH_FSM_SET_NEXT(recurse_done);
      return SSH_FSM_CONTINUE;
    }

  return SSH_FSM_SUSPENDED;
}

void fcr_parse_stat_cb(SshFileClientError error,
                       SshFileCopyFile file,
                       SshFileAttributes attrs,
                       const char *error_msg,
                       SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyRecurseContext);
  SshFileCopyFileListItem item;
  
  item = (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);

  if (error == SSH_FX_OK)
    {
      SshFileCopyLocation to_be_deleted;

      to_be_deleted = item->files;

      SSH_ASSERT(item != NULL);
      SSH_ASSERT(item->files->raw);

      item->files = ssh_file_copy_location_allocate(TRUE);

      ssh_file_copy_location_add_file(item->files,
                                      attrs,
                                      ssh_file_copy_file_get_name(file));

      ssh_dllist_rewind(item->files->file_list);
      ssh_file_copy_location_destroy(to_be_deleted);
    }
  else
    {
      SshDlListNode node;
      node = ssh_dllist_current_node(gdata->file_list);
      ssh_dllist_delete_node(node);
      ssh_file_copy_file_list_item_destroy(item);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_parse_raw: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_parse_raw)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;
  SshFileCopyFileListItem item;
  SshFileCopyFile file;

  item = (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);

  SSH_ASSERT(item != NULL);
  SSH_ASSERT(item->files->raw);
  SSH_ASSERT(item->files->source);

  /* stat file */
  SSH_ASSERT(ssh_dllist_is_current_valid(item->files->file_list));
  file = (SshFileCopyFile) ssh_dllist_current(item->files->file_list);

  SSH_ASSERT(file != NULL);

  FCC_START_ATTRS(tdata, file, TRUE, fcr_parse_stat_cb);
  SSH_FSM_SET_NEXT(recurse_next_file);  
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_stat
                        (gdata->source->client,
                         ssh_file_copy_file_get_name(file),
                         transfer_attribute_cb, thread));
}

void fcr_lstat_cb(SshFileClientError error,
                  SshFileCopyFile file,
                  SshFileAttributes attrs,
                  const char *error_msg,
                  SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyRecurseContext);
  ssh_xfree(gdata->filename_to_be_statted);
  gdata->filename_to_be_statted = NULL;

  if (error == SSH_FX_OK)
    {
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else
    {
      SshFileCopyFileListItem item;
      item = (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);
      ssh_dllist_fw(item->files->file_list, 1);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_lstat: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_lstat_file)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;
  SshFileCopyFile file;
  SshFileCopyFileListItem item;

  SSH_ASSERT(ssh_dllist_is_current_valid(gdata->file_list));
  item = (SshFileCopyFileListItem) ssh_dllist_current(gdata->file_list);
  SSH_ASSERT(ssh_dllist_is_current_valid(item->files->file_list));
  file = (SshFileCopyFile) ssh_dllist_current(item->files->file_list);

  FCC_START_ATTRS(tdata, file, TRUE, fcr_lstat_cb);
  SSH_FSM_SET_NEXT(recurse_next_file);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                     (gdata->source->client, gdata->filename_to_be_statted,
                      transfer_attribute_cb, thread));
}

void fcr_opendir_cb(SshFileClientError error,
                    SshFileCopyFile file,
                    SshFileHandle handle, const char *error_msg,
                    SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyRecurseContext);

  if (error == SSH_FX_OK)
    {
      file->handle = handle;
    }
  else
    {
      if (error == SSH_FX_FAILURE &&
          (file->attributes->permissions & S_IFMT) == S_IFLNK &&
          gdata->attrs->follow_symlinks)
        {
          /* We tried to follow a symlink, which we assumed to be
             a directory.*/
          SSH_TRACE(2, ("file '%s', a symlink, apparently wasn't a "
                        "directory. Tata!",
                        ssh_file_copy_file_get_name(file)));
          SSH_FSM_SET_NEXT(recurse_wake_parent_and_finish);
          return;
        }
      SSH_FSM_SET_NEXT(recurse_wake_parent_and_finish);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_opendir: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_opendir)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_PRECOND(tdata->current_dir != NULL);
  SSH_PRECOND(tdata->parent != NULL);
  SSH_PRECOND(tdata->current_dir->attributes != NULL);
  SSH_PRECOND(tdata->current_dir->attributes->flags &
              SSH_FILEXFER_ATTR_PERMISSIONS);
  SSH_PRECOND((tdata->current_dir->attributes->permissions & S_IFMT) ==
              S_IFDIR ||
              (gdata->attrs->follow_symlinks ||
               (tdata->current_dir->attributes->permissions & S_IFMT) ==
               S_IFLNK));

  if (tdata->symlink_level > SSH_FCR_MAX_SYMLINK_LEVEL_DEPTH)
    {
      char *error_string;

      ssh_dsprintf(&error_string, "%s: Max symlink count for depth exceeded "
                   "(we have very probably encountered a symlink loop. See "
                   "man-page for scp2).",
                   ssh_file_copy_file_get_name(tdata->current_dir));
      /* As opendir is always called in a new child, we just report
         the error as non-fatal to the application, and wake up the
         parent. */
      SSH_FSM_SET_NEXT(recurse_wake_parent_and_finish);
      /* Report error. */
      (*gdata->error_callback)(SSH_FC_ERROR_ELOOP, error_string,
                               gdata->callback_context);
      ssh_xfree(error_string);
      return SSH_FSM_CONTINUE;
    }

  if (gdata->attrs->follow_symlinks ||
      (tdata->current_dir->attributes->permissions & S_IFMT) == S_IFLNK)
    {
      tdata->symlink_level++;
      SSH_DEBUG(4, ("symlink_level: %d", tdata->symlink_level));
    }

  SSH_DEBUG_INDENT;

  SSH_TRACE(3, ("Opening directory %s...",
                ssh_file_copy_file_get_name(tdata->current_dir)));

  FCC_START_HANDLE(tdata, tdata->current_dir, TRUE, fcr_opendir_cb);
  SSH_FSM_SET_NEXT(recurse_readdir);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_opendir
                     (gdata->source->client,
                      ssh_file_copy_file_get_name(tdata->current_dir),
                      transfer_handle_cb, thread));
}

void fcr_readdir_cb(SshFileClientError error, SshFileCopyFile file,
                    const char *name, const char *long_name,
                    SshFileAttributes attrs, const char *error_msg,
                    SshFSMThread thread)
{
  FCC_DATA(SshFileCopyRecurseContext, RecurseThreadContext);




  if (error == SSH_FX_OK)
    {
      if (strcmp(name, ".") == 0 || strcmp(name, "..") == 0)
        {
          SSH_FSM_SET_NEXT(recurse_readdir);
          return;
        }
      tdata->new_file = ssh_file_copy_file_allocate();
      ssh_file_copy_file_register_filename(tdata->new_file,
                                           ssh_xstrdup(name));

      ssh_file_copy_file_register_long_name(tdata->new_file,
                                            ssh_xstrdup(long_name));





      ssh_file_copy_file_register_attributes(tdata->new_file,
                                             ssh_file_attributes_dup(attrs));
    }
  else if (error == SSH_FX_EOF)
    {
      /* We are at the end of the directory. Store gathered list, and
         wake up parent and finish. */
      SSH_FSM_SET_NEXT(recurse_current_dir_at_end);
    }
  else
    {
      /* XXX Drop current dir. */
      SSH_FSM_SET_NEXT(recurse_readdir);
      FCC_ERROR(fcc_fx_err_to_fc_err(error), ("fcr_readdir: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_readdir)
{
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  SSH_PRECOND(tdata->current_dir != NULL);
  SSH_PRECOND(tdata->current_dir->handle != NULL);
  SSH_PRECOND(tdata->parent != NULL);

  FCC_START_NAME(tdata, tdata->current_dir, TRUE, fcr_readdir_cb);
  SSH_FSM_SET_NEXT(recurse_process_file);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_readdir
                     (tdata->current_dir->handle, transfer_name_cb, thread));
}

SSH_FSM_STEP(recurse_process_file)
{
  Boolean func_ret = FALSE;
  char *temp_filename;
  RecurseThreadContext child_tdata;
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  if (!(tdata->new_file->attributes->flags & SSH_FILEXFER_ATTR_PERMISSIONS))
    {
      SSH_FSM_SET_NEXT(recurse_readdir_lstat_file);
      return SSH_FSM_CONTINUE;
    }

  SSH_FSM_SET_NEXT(recurse_readdir);

  if (*(ssh_file_copy_file_get_name(tdata->current_dir)) != '\0')
    {
      const char *current_basedir;

      current_basedir =
        ssh_file_copy_file_get_name(tdata->current_dir);

      ssh_dsprintf(&temp_filename, "%s%s%s",
                   current_basedir,
                   current_basedir[strlen(current_basedir) - 1]
                   == '/' ? "" : "/",
                   ssh_file_copy_file_get_name(tdata->new_file));
    }
  else
    {
      temp_filename =
        ssh_xstrdup(ssh_file_copy_file_get_name(tdata->new_file));
    }

  if (gdata->func)
    {
      func_ret =
        (*gdata->func)(temp_filename,
                       ssh_file_copy_file_get_long_name(tdata->new_file),
                       ssh_file_copy_file_get_attributes(tdata->new_file),
                       gdata->attrs,
                       gdata->callback_context);
    }
  else
    {
      SSH_VERIFY(ssh_dllist_add_item
                 (tdata->current_dir->dir_entries->file_list,
                  tdata->new_file, SSH_DLLIST_END) == SSH_DLLIST_OK);
    }

  if ((tdata->new_file->attributes->permissions & S_IFMT) == S_IFDIR ||
      ((tdata->new_file->attributes->permissions & S_IFMT) == S_IFLNK &&
       gdata->attrs->follow_symlinks))
    {
      if (!gdata->func)
        {
          tdata->new_file->dir_entries = ssh_file_copy_location_allocate(TRUE);
          tdata->new_file->dir_entries->parent_dir =
            tdata->current_dir->dir_entries;
          tdata->new_file->dir_entries->file = tdata->new_file;
        }
      else if (func_ret == FALSE)
        {
          SSH_DEBUG(3, ("SshFileRecurseFunc returned FALSE, so we "
                        "don't want to traverse this directory."));
          ssh_xfree(temp_filename);
          ssh_file_copy_file_destroy(tdata->new_file);
          return SSH_FSM_CONTINUE;
        }

      child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
      tdata->child = ssh_fsm_thread_create(gdata->fsm,
                                           recurse_opendir, NULL_FNPTR,
                                           recurse_child_thread_destructor,
                                           child_tdata);
      SSH_VERIFY(tdata->child != NULL);

      child_tdata->parent = thread;
      child_tdata->current_dir = ssh_file_copy_file_dup(tdata->new_file);
      child_tdata->symlink_level = tdata->symlink_level;

      ssh_file_copy_file_register_filename(child_tdata->current_dir,
                                           temp_filename);
      return SSH_FSM_SUSPENDED;
    }
  else
    {
      ssh_xfree(temp_filename);
    }

  return SSH_FSM_CONTINUE;
}

void fcr_readdir_lstat_cb(SshFileClientError error,
                          SshFileCopyFile file,
                          SshFileAttributes attrs,
                          const char *error_msg,
                          SshFSMThread thread)
{
  FCC_GDATA(SshFileCopyRecurseContext);

  ssh_xfree(gdata->filename_to_be_statted);
  gdata->filename_to_be_statted = NULL;

  if (error == SSH_FX_OK)
    {
      ssh_file_copy_file_register_attributes(file,
                                             ssh_file_attributes_dup(attrs));
    }
  else
    {
      SSH_FSM_SET_NEXT(recurse_readdir);
      FCC_ERROR(fcc_fx_err_to_fc_err(error),
                ("fcr_readdir_lstat: %s", error_msg));
    }
}

SSH_FSM_STEP(recurse_readdir_lstat_file)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  if (*(ssh_file_copy_file_get_name(tdata->current_dir)) != '\0')
    {
      const char *current_basedir;

      current_basedir =
        ssh_file_copy_file_get_name(tdata->current_dir);

      ssh_dsprintf(&gdata->filename_to_be_statted, "%s%s%s",
                   current_basedir,
                   current_basedir[strlen(current_basedir) - 1]
                   == '/' ? "" : "/",
                   ssh_file_copy_file_get_name(tdata->new_file));
    }
  else
    {
      gdata->filename_to_be_statted =
        ssh_xstrdup(ssh_file_copy_file_get_name(tdata->new_file));
    }

  FCC_START_ATTRS(tdata, tdata->new_file, TRUE, fcr_readdir_lstat_cb);
  SSH_FSM_SET_NEXT(recurse_process_file);
  SSH_FSM_ASYNC_CALL(tdata->op_handle = ssh_file_client_lstat
                     (gdata->source->client, gdata->filename_to_be_statted,
                      transfer_attribute_cb, thread));
}

SSH_FSM_STEP(recurse_current_dir_at_end)
{
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;
  ssh_file_client_close(tdata->current_dir->handle, NULL_FNPTR, NULL);
  tdata->current_dir->handle = NULL;
  SSH_FSM_SET_NEXT(recurse_wake_parent_and_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(recurse_wake_parent_and_finish)
{
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  if (tdata->parent)
    ssh_fsm_continue(tdata->parent);

  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(recurse_done)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;

  (*gdata->completion_callback)(SSH_FC_OK,
                                "Recursion complete.",
                                gdata->source,
                                gdata->new_file_list,
                                gdata->callback_context);
  ssh_fsm_destroy(gdata->fsm);

  /* XXX What else should be freed ? */
  ssh_xfree(gdata);
  return SSH_FSM_FINISH;
}

SSH_FSM_STEP(recurse_cleanup)
{
  SshFileCopyRecurseContext gdata = (SshFileCopyRecurseContext)fsm_context;
  RecurseThreadContext tdata = (RecurseThreadContext)thread_context;

  /* XXX testing */
  if (tdata->child)
    {
      RecurseThreadContext child_tdata;
      child_tdata = ssh_fsm_get_tdata(tdata->child);
      child_tdata->parent = NULL;
      ssh_fsm_kill_thread(tdata->child);
      tdata->child = NULL;
    }
  ssh_fsm_destroy(gdata->fsm);

#if 0
  if (gdata->operation_handle)
    ssh_operation_unregister(gdata->operation_handle);
#endif
  
  ssh_xfree(gdata);

  return SSH_FSM_FINISH;
}

SshFSMStateDebugStruct recurse_states_array[] =
{ { "fcr_check_conn", "Check connection", recurse_check_conn },
  { "fcr_next_file", "Set next file", recurse_next_file },
  { "fcr_lstat", "lstat file", recurse_lstat_file },
  { "fcr_parse_raw", "Parse \"raw\" file", recurse_parse_raw },
  { "fcr_recurse_done", "We're done recursing", recurse_done },

  /* Child threads check directories "recursively". */
  { "fcr_opendir", "Open a directory", recurse_opendir },
  { "fcr_readdir", "Take next directory element", recurse_readdir },
  { "fcr_process_file", "Process new file", recurse_process_file },
  { "fcr_readdir_lstat_file", "lstat a file if it doesn't have permissions",
    recurse_readdir_lstat_file },
  { "fcr_current_dir_at_end", "We are finished processing the current dir",
    recurse_current_dir_at_end },
  { "fcr_wake_parent_and_finish", "Wake parent and finish",
    recurse_wake_parent_and_finish }
};

void ssh_file_copy_recurse_dirs(SshFileCopyConnection source,
                                SshDlList file_list,
                                SshFileCopyRecurseAttrs attrs,
                                SshFileCopyRecurseFileFunc func,
                                SshFileCopyRecurseReadyCallback ready_cb,
                                SshFileCopyErrorCallback error_cb,
                                void *context)
{
  SshFileCopyRecurseContext recurse_context;
  RecurseThreadContext tdata;

  SSH_PRECOND(source != NULL);
  SSH_PRECOND(file_list != NULL);
  SSH_PRECOND(ready_cb != NULL_FNPTR);
  SSH_PRECOND(error_cb != NULL_FNPTR);

  recurse_context = ssh_xcalloc(1, sizeof(*recurse_context));
  recurse_context->fsm = ssh_fsm_create(recurse_context);
  SSH_VERIFY(recurse_context->fsm != NULL);
  ssh_fsm_register_debug_names(recurse_context->fsm, recurse_states_array,
                               SSH_FSM_NUM_STATES(recurse_states_array));
  recurse_context->completion_callback = ready_cb;
  recurse_context->error_callback = error_cb;
  recurse_context->callback_context = context;
  recurse_context->source = source;
  recurse_context->func = func;

  if (!attrs)
    attrs = ssh_xcalloc(1, sizeof(*attrs));

  recurse_context->attrs = attrs;

  ssh_dllist_rewind(file_list);
  ssh_dllist_mapcar(file_list, location_list_rewind, NULL);

  recurse_context->file_list = file_list;

  if (!recurse_context->func)
    recurse_context->new_file_list = ssh_dllist_allocate();

  tdata = ssh_xcalloc(1, sizeof(*tdata));
  recurse_context->main_thread = ssh_fsm_thread_create(recurse_context->fsm,
                                                       recurse_check_conn,
                                                       NULL_FNPTR,
                                                       NULL_FNPTR, /* XXX mem
                                                                      leaks? */
                                                       tdata);
  SSH_VERIFY(recurse_context->main_thread != NULL);
  ssh_fsm_set_thread_name(recurse_context->main_thread, "main_thread");
}
