/*

  sshd2.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/











#include "ssh2includes.h"
#include "sshtcp.h"
#include "sshudp.h"
#include "sshnameserver.h"
#include "sshsignals.h"
#include "sshfdstream.h"
#include "sshcrypt.h"
#include "sshbuffer.h"
#include "sshtimeouts.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "sshcipherlist.h"
#include "sshuserfiles.h"
#include "ssheloop.h"
#include "sshmsgs.h"
#include "sshgetopt.h"
#include "auths-common.h"
#include "sshencode.h"
#include "sshdsprintf.h"
#include "sshappcommon.h"
#include "sshtimemeasure.h"
#include "sshsnlist.h"
#include "sshhostkeyio.h"
#include "sshglobals.h"
#include "sshfsm.h"
#include "sshfsmstreams.h"

#include "sigchld.h"
#include "sshunixpipestream.h"
#include "sshunixptystream.h"
#ifndef VXWORKS
#include <syslog.h>
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */
#endif /* VXWORKS */




















































#ifdef HAVE_LIBWRAP
#include <tcpd.h>
int allow_severity = SSH_LOG_INFORMATIONAL;
int deny_severity = SSH_LOG_WARNING;
#endif /* HAVE_LIBWRAP */

#ifdef SSHDIST_SESSION_SIA
#ifdef HAVE_SIA
#include "sshsia.h"
#endif /* HAVE_SIA */
#endif /* SSHDIST_SESSION_SIA */

#ifdef HAVE_SCO_ETC_SHADOW
#include <sys/types.h>
#include <sys/security.h>
#include <sys/audit.h>
#include <prot.h>
#endif /* HAVE_SCO_ETC_SHADOW */
































#define SSHD_PROGRAM_ARGUMENTS  "d:D:vf:g:h:io:p:q"


#define SSH_DEBUG_MODULE "Sshd2"

#ifdef __SUNPRO_C
#pragma error_messages (off,E_STATEMENT_NOT_REACHED)
#endif /* __SUNPRO_C */





















/* Program name, without path. */
const char *av0;

typedef struct SshServerData
{
  SshConfig config;
  Boolean debug;
  Boolean debug_dont_fork;
  SshTcpListener listener;
  SshUdpListener broadcast_listener;
  SshTime broadcast_last;
  int broadcasts_per_second;
  int connections;
  SshUser user;
  Boolean ssh_fatal_called;
  SshTimeMeasure idle_timer;
  SshUInt64 bytes_from_client;
  char *pid_file;






  



} *SshServerData;

typedef struct SshServerConnectionRec
{
  SshServerData shared;
  SshServer server;
  char *remote_ip;
  char *remote_host_name;
  /* For wrapping the connection to SshServer. */
  SshStream stream;
} *SshServerConnection;

void ssh_server_connection_destroy(SshServerConnection connection)
{
  /* SshServerData is not destroyed here, it is destroyed by main() when
     we return from the event loop. */
  ssh_server_destroy(connection->server);
  ssh_xfree(connection->remote_ip);
  ssh_xfree(connection->remote_host_name);
  memset(connection, 'F', sizeof(*connection));
  ssh_xfree(connection);
}

/* Forward declaration. */
void ssh_idle_timeout_cb(void *context);
void server_hostkeys_initialized(SshHostKeyStatus status, void *context);
















void server_disconnect(int reason, Boolean locally_generated,
                       const char *msg, void *context)
{
  SshServerConnection c = context;

  SSH_TRACE(0, ("locally_generated = %s",
                locally_generated ? "TRUE" : "FALSE"));

  switch(reason)
    {
    case SSH_DISCONNECT_HOST_NOT_ALLOWED_TO_CONNECT:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Disallowed connect from denied host. '%s'",
                    msg);
      break;
    case SSH_DISCONNECT_PROTOCOL_ERROR:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Protocol error in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_KEY_EXCHANGE_FAILED:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "Key exchange failed in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_RESERVED:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "RESERVED (this is an illegal code) in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_MAC_ERROR:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "MAC failed in %s, disconnecting: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_COMPRESSION_ERROR:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "compression error in %s, disconnecting: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_SERVICE_NOT_AVAILABLE:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "service not available%s: '%s'",
                    locally_generated ? "" : " (an odd error code to come "
                    "from the client)",
                    msg);
      break;
    case SSH_DISCONNECT_PROTOCOL_VERSION_NOT_SUPPORTED:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "protocol version not supported in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_HOST_KEY_NOT_VERIFIABLE:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_WARNING,
                    "host key not verifiable in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_CONNECTION_LOST:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "connection lost: '%s'", msg);
      break;
    case SSH_DISCONNECT_BY_APPLICATION:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "disconnected by application in %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_TOO_MANY_CONNECTIONS:
      if (!(*c->server->common->compat_flags->
            deprecated_disconnect_codes_draft_incompatibility))
        {
          ssh_log_event(c->server->config->log_facility,
                        SSH_LOG_INFORMATIONAL,
                        "too many connections %s: '%s'",
                        locally_generated ? "" : "(an odd error code to come "
                        "from the client)",
                        msg);
        }
      else
        {
          /* This is not a valid reason code according to draft. We
             process this anyway, because ssh-versions ssh-2.0.13 and
             older sent this. (It was
             SSH_DISCONNECT_AUTHENTICATION_ERROR.) */
          SSH_TRACE(1, ("Received disconnection reason code 12, which "
                        "does not conform with the draft."));
          ssh_log_event(c->server->config->log_facility,
                        SSH_LOG_WARNING,
                        "User authentication failed: '%s'",
                        msg);
        }
      break;
    case SSH_DISCONNECT_AUTH_CANCELLED_BY_USER:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "authentication cancelled by user: '%s'", msg);
      break;
    case SSH_DISCONNECT_NO_MORE_AUTH_METHODS_AVAILABLE:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "no more authentication methods on %s: '%s'",
                    locally_generated ? "local" : "remote", msg);
      break;
    case SSH_DISCONNECT_ILLEGAL_USER_NAME:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "illegal user name %s: '%s'",
                    locally_generated ? "" : "(an odd error code to come "
                    "from the client)", msg);
      break;
    default:
      ssh_log_event(c->server->config->log_facility,
                    SSH_LOG_ERROR,
                    "Unknown reason code '%d' for disconnect. msg: '%s'",
                    reason, msg);
      SSH_TRACE(1, ("Unknown reason code '%d' for disconnect. msg: '%s'",
                    reason, msg));
      break;
    }

  /* Destroy the server connection object. */
  ssh_cancel_timeouts(ssh_idle_timeout_cb, SSH_ALL_CONTEXTS);





  ssh_server_connection_destroy(c);
}

void server_debug(int type, const char *msg, void *context)
{
  ssh_debug("server_debug: %s", msg);
}

/* Checks the remote version number, and execs a compatibility program as
   appropriate. */

Boolean ssh_server_version_check(const char *version, void *context)
{

  SshServerConnection c = (SshServerConnection)context;
  char *args[100], *aa;
  char buf[200];
  int i, arg;
  extern char **environ;

  if (strncmp(version, "SSH-1.", 6) == 0 &&
      strncmp(version, "SSH-1.99", 8) != 0 &&
      c->server->config->ssh1compatibility == TRUE &&
      c->server->config->ssh1_path != NULL &&
      c->server->config->ssh1_argv != NULL)
    {
      SSH_TRACE(0, ("Executing %s for ssh1 compatibility.",
                    c->server->config->ssh1_path));

      arg = 0;
      args[arg++] = "sshd";
      args[arg++] = "-i";
      args[arg++] = "-V";
      ssh_snprintf(buf, sizeof(buf), "%s\n", version); /* add newline */
      args[arg++] = buf;
      for (i = 1; c->server->config->ssh1_argv[i]; i++)
        {
          if (arg >= sizeof(args)/sizeof(args[0]) - 2)
            ssh_fatal("Too many arguments for compatibility ssh1.");
          aa = c->server->config->ssh1_argv[i];
          if (strcmp(aa, "-b") == 0 ||
              strcmp(aa, "-g") == 0 ||
              strcmp(aa, "-h") == 0 ||
              strcmp(aa, "-k") == 0 ||
              strcmp(aa, "-p") == 0)
            {
              args[arg++] = aa;
              if (c->server->config->ssh1_argv[i + 1])
                args[arg++] = c->server->config->ssh1_argv[++i];
            }
          else
            if (strcmp(aa, "-d") == 0)
              {
                args[arg++] = aa;
                if (c->server->config->ssh1_argv[i + 1])
                  i++; /* Skip the level. */
              }
            else if (strcmp(aa, "-q") == 0 ||
                     strcmp(aa, "-i") == 0)
              {
                args[arg++] = aa;
              }
            else  if (strcmp(aa, "-f") == 0)
              {
                if (c->server->config->ssh1_argv[i + 1])
                  i++; /* Skip the argument. */
                if (c->server->config->sshd1_config_file_path)
                  {
                    args[arg++] = aa;
                    args[arg++] = c->server->config->sshd1_config_file_path;
                  }
              }
        }
      args[arg++] = NULL;

      /* Set the input file descriptor to be fd 0. */
      if (c->server->config->ssh1_fd != 0)
        {
          if (dup2(c->server->config->ssh1_fd, 0) < 0)
            ssh_fatal("Making ssh1 input fd 0 (dup2) failed: %s",
                      strerror(errno));
          if (dup2(c->server->config->ssh1_fd, 1) < 0)
            ssh_fatal("Making ssh1 input fd 1 (dup2) failed: %s",
                      strerror(errno));
          close(c->server->config->ssh1_fd);
        }

      /* Exec the ssh1 server. */
      execve(c->server->config->ssh1_path, args, environ);
      ssh_fatal("Executing ssh1 in compatibility mode failed.");
      /*NOTREACHED*/
    }

  return TRUE;
}








































































































































typedef struct HeadingMatchCtxRec {
  char *user;
  char *uid_str;
  char *remote_host;
  char *remote_ip;
  size_t num_groups;
  char **groups;
  char **gid_strs;
} HeadingMatchCtxStruct, *HeadingMatchCtx;

Boolean match_heading_cb(const char *pattern, SshMetaConfig metaconfig,
                         void *context)
{
  HeadingMatchCtx match_ctx = (HeadingMatchCtx) context;
  SshPatternHolderStruct holder;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();

  SSH_PRECOND(match_ctx->remote_host != NULL);
  SSH_PRECOND(match_ctx->remote_ip != NULL);
  
  memset(&holder, 0, sizeof(holder));
  holder.regex_syntax = metaconfig->regex_syntax;

  if (match_ctx->user != NULL)
    {
      holder.pattern = (char *)pattern;

      SSH_ASSERT(match_ctx->uid_str != NULL);
      return ssh_match_user_groups(match_ctx->user, match_ctx->uid_str,
                                   match_ctx->num_groups,
                                   match_ctx->groups,
                                   match_ctx->gid_strs,
                                   match_ctx->remote_host,
                                   match_ctx->remote_ip,
                                   &holder, rex_ctx);
    }
  holder.pattern = (char *)pattern;
  return ssh_match_host_id(match_ctx->remote_host, match_ctx->remote_ip,
                           &holder, rex_ctx);
}

/*
 * External authorization program interface.
 */
typedef struct ExternalAuthCtxRec {
  Boolean result;
  Boolean need_passwd_change;
  Boolean got_failure;
  char *error_msg;

  char *user;
  SshServer server;
  SshAuthPolicyCompletionCB completion_cb;
  void *completion_ctx;
  SshBuffer in_buf;
  SshBuffer out_buf;
  SshBuffer stderr_buf;
  SshStream stderr_stream;
  SshStream stdio_stream;
  SshOperationHandle handle;
  SshFSM fsm;
  SshFSMThread main;
  SshFSMThread stream_handler;
  SshFSMCondition out_buf_grown;
  SshFSMCondition read_more;
  SshFSMCondition in_buf_shrunk;
  SshUInt32 stub_flags;
} ExternalAuthCtxStruct, *ExternalAuthCtx;

void authorization_stderr_iocb(SshStreamNotification notification,
                               void *context)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) context;
  
  if (notification == SSH_STREAM_INPUT_AVAILABLE)
    {
      unsigned char buf[10];
      int ret ;

      do {
        ret = ssh_stream_read(ctx->stderr_stream, buf, 1);
        if (ret > 0)
          {
            if (buf[0] == '\n')
              {
                ssh_debug("ext_auth: %.*s", ssh_buffer_len(ctx->stderr_buf),
                          ssh_buffer_ptr(ctx->stderr_buf));
                ssh_buffer_consume(ctx->stderr_buf,
                                   ssh_buffer_len(ctx->stderr_buf));
              }
            else
              {
                ssh_xbuffer_append(ctx->stderr_buf, buf, 1);
              }
          }
        else if (ret == 0)
          {
            SSH_DEBUG(2, ("Got EOF from stderr stream."));
          }
      } while (ret > 0);
    }
  else
    {
      SSH_DEBUG(0, ("Got notification %d from stderr stream.", notification));
    }
}

SSH_FSM_STEP(authorization_start);
SSH_FSM_STEP(authorization_get_result);
SSH_FSM_STEP(authorization_finish);
SSH_FSM_STEP(authorization_finish_finalize);

void authorization_abort_cb(void *context)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) context;
  SSH_DEBUG(2, ("Aborting authorization."));
  ctx->handle = NULL;
  ctx->completion_cb = NULL_FNPTR;
  if (ctx->stderr_stream)
    ssh_stream_set_callback(ctx->stderr_stream, NULL_FNPTR, NULL);
  if (ctx->main)
    {
      ssh_fsm_set_next(ctx->main, authorization_finish);
      ssh_fsm_continue(ctx->main);
    }
}

SSH_FSM_STEP(authorization_start)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) fsm_context;
  char line[1024];
  SSH_DEBUG(4, ("Sending parameters to authorization program."));
  ssh_snprintf(line, sizeof(line), "user_name:%s\n", ctx->user);
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "host_ip:%s\n",
               ctx->server->common->local_ip);
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "host_name:%s\n",
               ctx->server->common->local_hostname);
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "remote_host_ip:%s\n",
               ctx->server->common->remote_ip);
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  ssh_snprintf(line, sizeof(line), "remote_host_name:%s\n",
               ctx->server->common->remote_host);
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  if (ctx->server->authenticated_client_user_name)
    {
      ssh_snprintf(line, sizeof(line), "remote_user_name:%s\n",
                   ctx->server->authenticated_client_user_name);
      ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
    }
  ssh_snprintf(line, sizeof(line), "end_of_params\n");
  ssh_xbuffer_append(ctx->out_buf, (unsigned char *)line, strlen(line));
  SSH_FSM_CONDITION_SIGNAL(ctx->out_buf_grown);
  SSH_FSM_SET_NEXT(authorization_get_result);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(authorization_get_result)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) fsm_context;
  unsigned char *data;
  char line[1024] = "\0";
  size_t len, i;
  Boolean newline_found = FALSE;

  data = ssh_buffer_ptr(ctx->in_buf);
  len = ssh_buffer_len(ctx->in_buf);

  if (len == 0)
    goto skip_and_check_for_eof;
  
  for (i = 0; i < len ; i++)
    if (data[i] == '\n')
      break;

  if (i >= sizeof(line))
    {
      ssh_warning("Got too long line from external authorization program "
                  "(len = %d). Authorization for user %s failed.",
                  i + 1, ctx->user);
      SSH_FSM_SET_NEXT(authorization_finish);
      return SSH_FSM_CONTINUE;
    }
  if (i < len)
    {
      newline_found = TRUE;
      memmove(line, data, i + 1);
      line[i] = '\0';
      ssh_buffer_consume(ctx->in_buf, i + 1);
      SSH_FSM_CONDITION_SIGNAL(ctx->in_buf_shrunk);
    }
  else
    {
      /* No newline found. */
      SSH_FSM_CONDITION_SIGNAL(ctx->in_buf_shrunk);
      SSH_FSM_CONDITION_WAIT(ctx->read_more);
    }
#define ERROR_CODE_ID "error_code:"
#define ERROR_MSG_ID "error_msg:"
  
  SSH_DEBUG(2, ("Got line `%s' from authorization program.", line));
  if (strcasecmp(line, "failure") == 0)
    {
      ctx->result = FALSE;
      ctx->got_failure = TRUE;
    }
  else if (strcasecmp(line, "success") == 0)
    {
      if (!ctx->got_failure)
        {
          ctx->result = TRUE;
          SSH_DEBUG(2, ("Got success from authorization program, finishing."));
          SSH_FSM_SET_NEXT(authorization_finish);
          return SSH_FSM_CONTINUE;
        }
      else
        {
          SSH_DEBUG(2, ("Ignoring \"success\" after \"failure\"."));
        }
    }
  else if (strncasecmp(line, ERROR_CODE_ID, strlen(ERROR_CODE_ID)) == 0)
    {
      char *error_code;
      error_code = &line[strlen(ERROR_CODE_ID)];
      SSH_DEBUG(2, ("Got error_code `%s'.", error_code));
      if (strcmp(error_code, "password_too_old") == 0)
        ctx->need_passwd_change = TRUE;
      else if (strcmp(error_code, "password_too_old") == 0)
        SSH_DEBUG(4, ("Got \"generic error\" code."));
      else
        SSH_DEBUG(4, ("Error code not recognized, ignoring."));
    }
  else if (strncasecmp(line, ERROR_MSG_ID, strlen(ERROR_MSG_ID)) == 0)
    {
      char *error_msg;
      error_msg = &line[strlen(ERROR_MSG_ID)];
      SSH_DEBUG(2, ("Got error_msg `%s'.", error_msg));
      ctx->error_msg = ssh_xstrdup(error_msg);
    }
  else
    {
      SSH_DEBUG(2, ("Line ignored."));
    }
  
 skip_and_check_for_eof:
  if (ssh_buffer_len(ctx->in_buf) == 0 || !newline_found)
    {
      if (ctx->stub_flags & SSH_STREAMSTUB_EOF_RECEIVED)
        {
          SSH_DEBUG(4, ("EOF received from authorization program."));
          SSH_FSM_SET_NEXT(authorization_finish);
        }
      else
        {
          SSH_DEBUG(4, ("No line yet from authorization program."));
          SSH_FSM_CONDITION_WAIT(ctx->read_more);
        }
    }
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(authorization_finish)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) fsm_context;
  char *cont;
  if (ctx->handle)
    ssh_operation_unregister(ctx->handle);
  ctx->server->ext_auth_handle = NULL;
  /* Free everything, call completion callback. */

  if (ctx->need_passwd_change)
    {
      /* Set forced command to PasswdPath. */
      ssh_xfree(ctx->server->common->forced_command);
      ctx->server->common->forced_command =
        ssh_xstrdup(ctx->server->config->passwd_path);
    }
  if (ctx->result || ctx->need_passwd_change)
    cont = NULL;
  else
    cont = ssh_xstrdup("");

  SSH_DEBUG(2, ("External authorization was a %s%s.", ctx->result ? "success" :
                "failure", ctx->need_passwd_change ?
                " (user's password needs to be changed)" : ""));
  if (ctx->completion_cb)
    (*ctx->completion_cb)(cont, ctx->completion_ctx);

  SSH_FSM_SET_NEXT(authorization_finish_finalize);

  /* XXX The error message should be shown to the user. */
  ssh_xfree(ctx->error_msg);
  ssh_xfree(ctx->user);
  ssh_stream_destroy(ctx->stderr_stream);

  if (!(ctx->stub_flags & SSH_STREAMSTUB_FINISHED))
    SSH_FSM_THROW(ctx->stream_handler, SSH_STREAMSTUB_ABORT);
  else
    return SSH_FSM_CONTINUE;
  SSH_FSM_WAIT_THREAD(ctx->stream_handler);
}

SSH_FSM_STEP(authorization_finish_finalize)
{
  ExternalAuthCtx ctx = (ExternalAuthCtx) fsm_context;
  ssh_buffer_free(ctx->in_buf);
  ssh_buffer_free(ctx->out_buf);
  ssh_buffer_free(ctx->stderr_buf);
  ssh_fsm_condition_destroy(ctx->out_buf_grown);
  ssh_fsm_condition_destroy(ctx->read_more);
  ssh_fsm_condition_destroy(ctx->in_buf_shrunk);
  ssh_fsm_destroy(ctx->fsm);
  ssh_xfree(ctx);
  SSH_DEBUG(5, ("Finished."));
  return SSH_FSM_FINISH;
}

SshOperationHandle do_external_authorization(ExternalAuthCtx ctx)
{
  char *argv[10], *shell;
  

  switch (ssh_pipe_create_and_fork(&ctx->stdio_stream, &ctx->stderr_stream))





    {

    case SSH_PIPE_CHILD_OK:
      shell = "/bin/sh";
      argv[0] = shell;
      argv[1] = "-c";
      argv[2] = ctx->server->config->external_authorization_prog;
      argv[3] = NULL;
      execv(shell, argv);
      ssh_warning("Failed to execute %s, sys_err `%s'", shell,
                  strerror(errno));
      exit(254);  
      break;

    case SSH_PIPE_PARENT_OK:
      SSH_DEBUG(2, ("Starting external authorization."));
      ctx->result = FALSE;
      ctx->in_buf = ssh_xbuffer_allocate();
      ctx->out_buf = ssh_xbuffer_allocate();
      ctx->stderr_buf = ssh_xbuffer_allocate();
      ctx->fsm = ssh_fsm_create(ctx);
      SSH_ASSERT(ctx->fsm != NULL);
      ctx->main = ssh_fsm_thread_create(ctx->fsm, authorization_start,
                                          NULL_FNPTR, NULL_FNPTR, NULL);
      SSH_ASSERT(ctx->main != NULL);
      ctx->read_more = ssh_fsm_condition_create(ctx->fsm);
      SSH_ASSERT(ctx->read_more != NULL);
      ctx->in_buf_shrunk = ssh_fsm_condition_create(ctx->fsm);
      SSH_ASSERT(ctx->in_buf_shrunk != NULL);
      ctx->out_buf_grown = ssh_fsm_condition_create(ctx->fsm);
      SSH_ASSERT(ctx->out_buf_grown != NULL);
      ctx->stream_handler = ssh_streamstub_spawn(ctx->fsm,
                                                 ctx->stdio_stream,
                                                 ctx->in_buf,
                                                 ctx->out_buf,
                                                 1024,
                                                 ctx->read_more,
                                                 ctx->in_buf_shrunk,
                                                 NULL,
                                                 ctx->out_buf_grown,
                                                 NULL,
                                                 &ctx->stub_flags);
      SSH_ASSERT(ctx->stream_handler != NULL);
      ssh_stream_set_callback(ctx->stderr_stream, authorization_stderr_iocb,
                              ctx);
      ctx->handle = ssh_operation_register(authorization_abort_cb, ctx);
      SSH_ASSERT(ctx->handle != NULL);
      return ctx->handle;
      break;
    case SSH_PIPE_ERROR:
      ssh_warning("Failed to create pipes when trying to execute external "
                  "authorization program `%s'. Failing authorization for "
                  "user `%s'.",
                  ctx->server->config->external_authorization_prog, ctx->user);
      (*ctx->completion_cb)("", ctx->completion_ctx);
      ssh_xfree(ctx->user);
      ssh_xfree(ctx);
      break;
    }
  return NULL;
}

/* This function will be used to see what authentications we can use
   for the user, and to provide a common place to put access
   control-information. */
void auth_policy_proc(const char *user,
                      const char *service,
                      const char *client_ip,
                      const char *client_port,
                      const char *completed_authentications,
                      const char *disabled_authentications,
                      SshAuthPolicyCompletionCB completion_proc,
                      void *completion_ctx,
                      void *context)
{
  SshServer server = (SshServer)context;
  SshConfig config = server->config;
  char *cp = NULL;







  SSH_DEBUG(2, ("user '%s' service '%s' client_ip '%s' client_port '%s' "
                "completed '%s'",
                user, service, client_ip, client_port,
                completed_authentications));

  if (server->last_user &&
      strcmp(server->last_user, user) != 0)
    {
      /* User has changed. */
      ssh_xfree(server->last_user);
      server->last_user = NULL;
    }

  /* User specific config. */
  if (server->last_user == NULL)
    {
      SshADTHandle h;
      SshADTContainer c = config->user_configurations;
      SshUser uc = ssh_user_initialize(user, FALSE);
      char *uid_str;
      size_t num_groups = 0;
      int i;
      char **group_strs = NULL, **gid_strs = NULL;
      Boolean match = FALSE;
      SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
      HeadingMatchCtxStruct match_ctx;

      SshGroup *groups = NULL;

      SshUser conf_uc = ssh_user_initialize(NULL, FALSE);

      memset(&match_ctx, 0, sizeof(match_ctx));
      
      /* First time, or user has changed. */
      server->last_user = ssh_xstrdup(user);

      if (uc)
        ssh_xdsprintf(&uid_str, "%lu", ssh_user_uid(uc));
      else
        uid_str = ssh_xstrdup("NO_SUCH_USER");
        
      match_ctx.user = (char *)user;
      match_ctx.uid_str = uid_str;
      match_ctx.remote_host = server->common->remote_host;
      match_ctx.remote_ip = (char *)client_ip;


      if (uc)
        {
          groups = ssh_user_get_groups(uc);
          SSH_VERIFY(groups != NULL);
          for (i = 0; groups[i]; i++)
            ;
          num_groups = i;
          group_strs = ssh_xcalloc(i, sizeof(char *));
          gid_strs = ssh_xcalloc(i, sizeof(char *));
          SSH_DEBUG(2, ("Number of groups: %d.", i));
          for (i = 0; groups[i]; i++)
            {
              SSH_DEBUG(2, ("Adding group: %s, %d.",
                            ssh_group_get_name(groups[i]),
                            ssh_group_get_gid(groups[i])));
              group_strs[i] = ssh_xstrdup(ssh_group_get_name(groups[i]));
              ssh_xdsprintf(&gid_strs[i], "%ld", ssh_group_get_gid(groups[i]));
            }
        }
      else
        {
          SSH_TRACE(2, ("User '%s' doesn't exist, using bogus group "
                        "\"NO_SUCH_USER\".", user));
          num_groups = 1;
          group_strs = ssh_xcalloc(num_groups, sizeof(char *));
          gid_strs = ssh_xcalloc(num_groups, sizeof(char *));
          group_strs[0] = ssh_xstrdup("NO_SUCH_USER");
          gid_strs[0] = ssh_xstrdup("NO_SUCH_USER");
        }

      match_ctx.num_groups = num_groups;
      match_ctx.groups = group_strs;
      match_ctx.gid_strs = gid_strs;

      /* Check whether user name matches the pattern. */
      for (h = ssh_adt_enumerate_start(c);
           h != SSH_ADT_INVALID;
           h = ssh_adt_enumerate_next(c, h))
        {
          SshSubConfig user_config = ssh_adt_get(c, h);
          match = ssh_match_user_groups(user, uid_str,
                                        num_groups, group_strs, gid_strs,
                                        server->common->remote_host,
                                        client_ip,
                                        &user_config->pattern, rex_ctx);

          /* If so, load the configuration file (log missing or
             non-readable config files) */
          /* XXX In principle, per-user configuration should be flushed
             if user changes. This is currently not a big problem, and
             therefore it is not yet implemented. */
          if (match)
            {
              char *config_file;
              SSH_DEBUG(2, ("user `%s' matched with `%s'.",
                            user, user_config->pattern.pattern));
              SSH_DEBUG(2, ("Reading per-user config file `%s'...",
                            user_config->config_file));
              SSH_ASSERT(conf_uc != NULL);

              config_file = ssh_userdir_prefix(conf_uc, config,
                                               user_config->config_file,
                                               SSH_SERVER_DIR,
                                               FALSE);



              if (!ssh_config_read_file_ext(conf_uc, config, user,
                                            match_heading_cb, &match_ctx,
                                            config_file,
                                            SSH_CONFIG_TYPE_SERVER_USER))
                {
                  ssh_warning("Non-existent or malformed per-user config "
                              "file `%s' for user `%s'. User not allowed to "
                              "login.",
                              user_config->config_file, user);
                  cp = ssh_xstrdup("");
                  (*completion_proc)(cp, completion_ctx);
                  return;
                }
              ssh_xfree(config_file);
            }
          else
            {
              SSH_DEBUG(2, ("user `%s' didn't match with `%s'.",
                            user, user_config->pattern.pattern));
            }
        }

      if (uc)
        ssh_user_free(uc, FALSE);
      if (num_groups > 0)
        {
          for (i = 0; i < num_groups; i++)
            {
              ssh_xfree(group_strs[i]);
              ssh_xfree(gid_strs[i]);
            }
          ssh_xfree(group_strs);
          ssh_xfree(gid_strs);
        }
      ssh_xfree(uid_str);
      ssh_user_free(conf_uc, FALSE);
    }

















































































































































































  /* Check whether user has passed all required authentications*/
  if (completed_authentications != NULL &&
      strlen(completed_authentications) > 0)
    {
      SshADTContainer required = server->config->required_authentications;
      SshADTHandle h;
      char *current = NULL;

      SSH_DEBUG(3, ("Using old-style authentication policy configuration."));

      /* If some authentication method has been passed, and there is
         no list for required authentications, the user is
         authenticated. */
      if (required != NULL)
        {
          /* Go trough the list to see if we find a match for every method
             needed. */
          for (h = ssh_adt_enumerate_start(required);
               h != SSH_ADT_INVALID;
               h = ssh_adt_enumerate_next(required, h))
            {
              current = ssh_adt_get(required, h);
              if (strstr(completed_authentications, current) == NULL)
                goto construct;
            }
        }
      
      /* If all were found from completed_authentications, the user is
         authenticated. */
      
      if (server->config->external_authorization_prog &&
          strlen(server->config->external_authorization_prog) > 0)
        {
          ExternalAuthCtx authorization_ctx =
            ssh_xcalloc(1, sizeof(*authorization_ctx));
          /* If the external authorization program is defined, execute
             it. The external authorization program has final say
             whether the user is allowed (authorized) to log in, or
             not. */
          SSH_ASSERT(user != NULL);
          authorization_ctx->user = ssh_xstrdup(user);
          authorization_ctx->server = server;
          authorization_ctx->completion_cb = completion_proc;
          authorization_ctx->completion_ctx = completion_ctx;
          server->ext_auth_handle =
            do_external_authorization(authorization_ctx);
        }
      else
        {
          (*completion_proc)(NULL, completion_ctx);
        }
      return;
    }

  /* Otherwise, construct a list of the authentications that can continue.
     All supported authentication methods are included in the list. */
 construct:
  {
    SshADTContainer allowed, required;
    SshBufferStruct buffer;

    allowed = server->config->allowed_authentications;
    required = server->config->required_authentications;

    if (ssh_adt_num_objects(required) > 0 || ssh_adt_num_objects(allowed) > 0)
      {
        SshADTContainer auths = ssh_adt_num_objects(required) > 0 ?
          required : allowed;
        SshADTHandle h = SSH_ADT_INVALID;
        Boolean first = TRUE;

        ssh_buffer_init(&buffer);
        for (h = ssh_adt_enumerate_start(auths);
             h != SSH_ADT_INVALID;
             h = ssh_adt_enumerate_next(auths, h))
          {
            char *current = NULL;
            current = ssh_adt_get(auths, h);
            SSH_ASSERT(current != NULL);

            if (strstr(disabled_authentications, current) != NULL)
              continue;

            if (strstr(completed_authentications, current) == NULL)
              {
                if (first)
                  first = FALSE;
                else
                  ssh_xbuffer_append(&buffer, (unsigned char *)",", 1);

                ssh_xbuffer_append(&buffer, (unsigned char *)current,
                                   strlen(current));

              }
          }
        ssh_xbuffer_append(&buffer, (unsigned char *)"\0", 1);
        cp = ssh_xstrdup(ssh_buffer_ptr(&buffer));
        ssh_buffer_uninit(&buffer);
      }
    else
      {
        cp = ssh_xstrdup("");
      }
  }

  SSH_DEBUG(2, ("output: %s", cp));

  (*completion_proc)(cp, completion_ctx);
}

/* Forward declaration. */
void ssh_login_grace_time_exceeded(void *context);

void ssh_idle_timeout_cb(void *context)
{
  SshServerConnection connection = (SshServerConnection) context;
  SshCommon common = connection->server->common;
  SshUInt64 sent, received;
  SshUInt64 seconds;
  unsigned long next_timeout;

  /* get statistics from all the channels */







  ssh_conn_channel_statistic(common->conn, NULL, &sent, &received);

  SSH_DEBUG(3, ("Idle check: bytes from client %ld, to client %ld",
                (unsigned long)received, (unsigned long)sent));

  /* If there has been no data from the client during the time specified
     in config, let's disconnect. */
  ssh_time_measure_get_value(connection->shared->idle_timer,
                             &seconds, NULL);

  if (common->config->idle_timeout > 0 &&
      connection->shared->bytes_from_client <= received &&
      seconds >= common->config->idle_timeout)
    {
      char s[] = "Idle timeout exceeded.";

      ssh_log_event(common->config->log_facility,
                    SSH_LOG_WARNING, "%s", s);
      SSH_DEBUG(1, ("Idle timeout exceeded after %ld seconds.", seconds));
      /* We send disconnect and exit. */





      ssh_conn_send_disconnect(common->conn,
                               SSH_DISCONNECT_BY_APPLICATION,
                               s);


      ssh_cancel_timeouts(ssh_idle_timeout_cb, SSH_ALL_CONTEXTS);
      ssh_server_connection_destroy(connection);
      return;
    }

  /* remember current values */
  if (connection->shared->bytes_from_client != received)
    {
      connection->shared->bytes_from_client = received;
      ssh_time_measure_reset(connection->shared->idle_timer);
    }

  next_timeout = common->config->idle_timeout / 10;
  if (next_timeout <= 0)
    next_timeout = 1L;

  ssh_register_timeout(next_timeout, 0L, ssh_idle_timeout_cb, context);
}

/* This is called, when we have an authenticated user ready to continue. */
void server_authentication_notify(const char *user, Boolean successful,
                                  void *context)
{
  SshServerConnection connection = (SshServerConnection) context;
  SshCommon common = connection->server->common;

  /* We unregister the (possible) grace time callback. */
  ssh_cancel_timeouts(ssh_login_grace_time_exceeded, SSH_ALL_CONTEXTS);






  
  if (successful)
    {
      ssh_log_event(common->config->log_facility,
                    SSH_LOG_NOTICE,
                    "User %s, coming from %s, authenticated.",
                    user, common->remote_host);

      if (common->config->idle_timeout > 0)
        {
          connection->shared->idle_timer = ssh_time_measure_allocate();
          ssh_time_measure_start(connection->shared->idle_timer);
          ssh_register_timeout((long) (common->config->idle_timeout / 10),
                               0L, ssh_idle_timeout_cb, connection);
        }
    }
}


/* Callback to handle the closing of connections in the "mother"
   process */
void child_returned(pid_t pid, int status, void *context)
{
  SshServerData data = (SshServerData) context;

  SSH_DEBUG(2, ("Child with pid '%d' returned with status '%d'.",
                pid, status));

  if (data->config->max_connections)
    {
      data->connections--;

      SSH_DEBUG(4, ("%d connections now open. (max %d)",
                    data->connections, data->config->max_connections));
    }
}

/* This callback gets called, if LoginGraceTime is exceeded. */
void ssh_login_grace_time_exceeded(void *context)
{
  SshServerConnection connection = (SshServerConnection) context;
  char s[] = "LoginGraceTime exceeded.";

  ssh_log_event(connection->server->config->log_facility,
                SSH_LOG_WARNING,
                "%s", s);

  /* We send disconnect, and exit. If LoginGraceTime is exceeded,
     there might be some kind of DoS-attack going on. */
#if 0
  /* XXX Argh, major problem. Because the auth-layer doesn't do any kind
     of buffering, sending this will cause a situation, where the conn
     layer won't be destroyed because of buffered data (it wants to
     flush its buffers). The auth-layer should process disconnect
     messages, because otherwise we can't send disconnect messages down
     before authentication is finished.

     XXX FIXME FIXME FIXME XXX
  */
#ifndef VXWORKS





  ssh_conn_send_disconnect(connection->server->common->conn,
                           SSH_DISCONNECT_BY_APPLICATION,
                           s);

#endif /* VXWORKS */
#endif





  ssh_cancel_timeouts(ssh_login_grace_time_exceeded, SSH_ALL_CONTEXTS);
  ssh_server_connection_destroy(connection);
}

/* This callback is called, when a stream needs to be destroyed
   with a small timeout. */
void destroy_stream_callback(void *context)
{
  SshStream stream = (SshStream)context;

  SSH_DEBUG(2, ("Destroying stream."));
  ssh_stream_destroy(stream);
  return;
}

void server_clean_up(void *context)
{
  ssh_cancel_timeouts(ssh_login_grace_time_exceeded, SSH_ALL_CONTEXTS);
}

/* This is supposed to be used before the stream has been wrapped to the
   protocol object. */
void server_output_disconnect(SshStream stream,
                              SshServerData data,
                              const char *error_msg,
                              SshUInt32 reason_code)
{
  SshBuffer buffer1, buffer2;
  char lang[] = "en";
  char *version_string;
  unsigned char pad[8];
  size_t pad_len;

  memset(pad, 0, sizeof (pad));
  buffer1 = ssh_xbuffer_allocate();
  buffer2 = ssh_xbuffer_allocate();

  /* Maximum number of connections is exceeded.  We construct
     a disconnect packet and send it after the syntethic version
     number string.  We have to do all this manually, because
     transport layer doesn't exist yet.  This is close to pure
     madness, but end result is at least somewhat elegant. */

  /* First encode the disconnect packet. */
  ssh_encode_buffer(buffer1,
                    /* SSH_MSG_DISCONNECT */
                    SSH_FORMAT_CHAR,
                    (unsigned int) SSH_MSG_DISCONNECT,
                    /* uint32 reason code */
                    SSH_FORMAT_UINT32, reason_code,
                    /* string description */
                    SSH_FORMAT_UINT32_STR,
                    error_msg, strlen(error_msg),
                    /* string language tag */
                    SSH_FORMAT_UINT32_STR,
                    lang, strlen(lang),
                    SSH_FORMAT_END);

  /* Construct a version string. */
  ssh_dsprintf(&version_string,
               (data->config->ssh1compatibility ?
                "SSH-1.99-%.200s" : "SSH-2.0-%.200s"),
               data->config->protocol_version_string);
  ssh_xbuffer_append(buffer2, (unsigned char *)version_string,
                     strlen(version_string));
  /* No CR in ssh1 compatible mode. */
  if (! data->config->ssh1compatibility)
    ssh_xbuffer_append(buffer2, (unsigned char *)"\r", 1);
  ssh_xbuffer_append(buffer2, (unsigned char *)"\n", 1);
  ssh_xfree(version_string);
  /* Assume granularity that is enough for server. */
  pad_len = 8 - ((ssh_buffer_len(buffer1) + 4 + 1) % 8);
  /* At least 4 bytes of padding. */
  if (pad_len == 0)
    pad_len = 8;
  
  /* Encode complete disconnect packet after the version id string. */
  ssh_encode_buffer(buffer2,
                    SSH_FORMAT_UINT32,
                    (SshUInt32)(ssh_buffer_len(buffer1) + 1 + pad_len),
                    SSH_FORMAT_CHAR,
                    (unsigned int)pad_len,
                    SSH_FORMAT_DATA,
                    ssh_buffer_ptr(buffer1), ssh_buffer_len(buffer1),
                    SSH_FORMAT_DATA,
                    pad, pad_len,
                    SSH_FORMAT_END);
  ssh_buffer_free(buffer1);
  ssh_stream_write(stream, ssh_buffer_ptr(buffer2),
                   ssh_buffer_len(buffer2));
  ssh_buffer_free(buffer2);
}

/* XXX kludge. Something is left in the event-loop. Fix.  Timeout
   needed, because there is a race condition with the fork(). If no
   timeout, child may exit before parent has registered a SIGCHLD
   handler for the child. Should be fixed in ssh_sigchld_register()
   (keep list of children with no registered callbacks, and if at a
   later time a callback is registered to a child that has already
   exited, call the handler immediately).
*/
void server_abort_cb(void *context)
{
  ssh_event_loop_abort();
}

void new_connection_start_finalize(SshTcpError error, const char *result,
                                   void *context)
{
  SshServerConnection c = (SshServerConnection)context;
  SshConfig config = c->shared->config;
  SshADTHandle h;
  SshADTContainer l = config->host_configurations;
  Boolean match = FALSE;



  HeadingMatchCtxStruct match_ctx;
  SshUser conf_uc = ssh_user_initialize(NULL, FALSE);

  memset(&match_ctx, 0, sizeof(match_ctx));
  
  if (error == SSH_TCP_OK)
    {
      SSH_DEBUG(3, ("remote hostname is \"%s\".", result));
      c->remote_host_name = ssh_xstrdup(result);
    }
  else
    {
      if (config->try_reverse_mapping)
        {
          ssh_warning("DNS lookup failed for \"%s\".", c->remote_ip);
        }
      else
        {
          SSH_DEBUG(2, ("DNS lookup disabled in the configuration"));
        }
      c->remote_host_name = ssh_xstrdup(c->remote_ip);
    }

  /* Host specific config. */
  match_ctx.remote_host = c->remote_host_name;
  match_ctx.remote_ip = c->remote_ip;
  
  for (h = ssh_adt_enumerate_start(l);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(l, h))
    {
      SshSubConfig host_config = ssh_adt_get(l, h);
      match = ssh_match_host_id(c->remote_host_name, c->remote_ip,
                                &host_config->pattern,
                                ssh_app_get_global_regex_context());

      /* If so, load the configuration file (log missing or
         non-readable config files) */
      if (match)
        {
          char *config_file;
          SSH_DEBUG(2, ("Address %s[%s] matched with `%s'.",
                        c->remote_host_name, c->remote_ip,
                        host_config->pattern.pattern));
          SSH_DEBUG(2, ("Reading per-host config file `%s'...",
                        host_config->config_file));
          SSH_ASSERT(conf_uc != NULL);

          config_file = ssh_userdir_prefix(conf_uc, config,
                                           host_config->config_file,
                                           SSH_SERVER_DIR, FALSE);



          if (!ssh_config_read_file_ext(conf_uc, config, NULL,
                                        match_heading_cb, &match_ctx,
                                        config_file,
                                        SSH_CONFIG_TYPE_SERVER_HOST))
            {
              ssh_warning("Non-existent or malformed per-host config "
                          "file `%s' for address %s[%s]. Connection not "
                          "allowed.",
                          host_config->config_file, c->remote_host_name,
                          c->remote_ip);
              /* XXX disconnect */
              server_output_disconnect(c->stream, c->shared,
                                       "Configuration error.",
                                       SSH_DISCONNECT_BY_APPLICATION);
              /* XXX */
              ssh_register_timeout(1L, 0L, server_abort_cb, c);
              return;
            }
          ssh_xfree(config_file);
        }
      else
        {
          SSH_DEBUG(4, ("Address %s[%s] didn't match with `%s'.",
                        c->remote_host_name, c->remote_ip,
                        host_config->pattern.pattern));
        }
    }
  ssh_user_free(conf_uc, FALSE);

  /* Reading banner message file dynamically. */
  ssh_config_read_banner_message(config);
      
  SSH_TRACE(2, ("Wrapping stream with ssh_server_wrap..."));


















  c->server = ssh_server_wrap(c->stream, config,
                              c->remote_host_name,
                              server_disconnect,
                              server_debug,
                              ((config->ssh1compatibility &&
                                config->ssh1_path != NULL) ?
                               ssh_server_version_check : NULL_FNPTR),
                              auth_policy_proc,
                              server_authentication_notify,
                              server_clean_up,
                              config->protocol_version_string,
                              (void *)c);







  SSH_TRACE(2, ("done."));
  if (config->login_grace_time > 0)
    ssh_register_timeout((long)config->login_grace_time, 0L,
                         ssh_login_grace_time_exceeded, c);













  c->stream = NULL;
}

/* This function is called whenever we receive a new connection. */
void new_connection_callback(SshTcpError error, SshStream stream,
                             void *context)
{
  SshServerData data = (SshServerData)context;
  SshServerConnection c;
  pid_t ret;
  const char *s;
  char buf[256];

  if (error != SSH_TCP_NEW_CONNECTION)
    {
      ssh_warning("new_connection_callback: unexpected error %d", (int)error);
      return;
    }

  if (!ssh_tcp_get_remote_address(stream, buf, sizeof(buf)))
    {
      ssh_warning("Failed to fetch remote ip address.");
      ssh_snprintf(buf, sizeof(buf), "UNKNOWN");
    }

  ssh_log_event(data->config->log_facility, SSH_LOG_INFORMATIONAL,
                "connection from \"%s\"", buf);

  SSH_DEBUG(2, ("new_connection_callback"));

  /* check for MaxConnections */


  if (data->config->max_connections)
    {
      SSH_TRACE(0, ("connections now open: %d", data->connections));
      if (data->connections >= data->config->max_connections)
        {
          server_output_disconnect(stream, data,
                                   "Too many connections.",
                                   SSH_DISCONNECT_TOO_MANY_CONNECTIONS);
          /* We destroy the stream with a little timeout in order
             to give some time to the protocol to send the disconnect
             message.  If the message can't be sent in this time window,
             it's obvious that we are under some kind of DoS attack
             and it's OK just to destroy the stream. */
          ssh_register_timeout(0L, 20000L,
                               destroy_stream_callback, (void *)stream);
          ssh_log_event(data->config->log_facility, SSH_LOG_WARNING,
                        "Refusing connection from \"%s\". Too many "
                        "open connections (max %d, now open %d).",
                        buf, data->config->max_connections,
                        data->connections);
          /* return from this callback. */
          return;
        }
    }


  /* Set socket to nodelay mode if configuration suggests this. */
  ssh_tcp_set_nodelay(stream, data->config->no_delay);
  /* Set socket to keepalive mode if configuration suggests this. */
  ssh_tcp_set_keepalive(stream, data->config->keep_alive);
  /* Set socket to linger mode. */
  ssh_tcp_set_linger(stream, TRUE);


  /* Fork to execute the new child, unless in debug mode (and
     ``debug_dont_fork'' is TRUE). */
#if 1
  if ((data->debug && data->debug_dont_fork) || data->config->inetd_mode)
    ret = 0;
  else
    ret = fork();
#else /* 0 or 1*/
  /* Simulate that fork() fails. */
  ret = -1;
#endif /* 0 or 1 */
  if (ret == 0)
    {
      /* Child. */
      /* Destroy the listener. */
      /* For VxWorks, enable multiple simultaneous file-transfers */
#ifndef VXWORKS
      if (data->listener)
        ssh_tcp_destroy_listener(data->listener);
      data->listener = NULL;
      if (data->broadcast_listener)
        {
          ssh_udp_listener_mark_forked(data->broadcast_listener);
          ssh_udp_destroy_listener(data->broadcast_listener);
          data->broadcast_listener = NULL;
        }
#endif /* VXWORKS */
      /* Save the file descriptor.  It is only used if we exec ssh1 for
         compatibility mode. */
      data->config->ssh1_fd = ssh_stream_fd_get_readfd(stream);

#ifdef HAVE_LIBWRAP
      {
        struct request_info req;
        static RETSIGTYPE (*old_handler)(int sig) = NULL;

        old_handler = signal(SIGCHLD, SIG_DFL);
        if (old_handler == SIG_ERR)
          {
            ssh_warning("new_connection_callback: Could not set "
                        "SIGCHLD signal handler.");
            old_handler = SIG_IGN;
          }

        request_init(&req, RQ_DAEMON, av0, RQ_FILE,
                     ssh_stream_fd_get_readfd(stream), NULL);
        fromhost(&req); /* validate client host info */
        if (!hosts_access(&req))
          {
            ssh_warning("Denied connection from %s by tcp wrappers.",
                        eval_client(&req));
            ssh_log_event(data->config->log_facility, SSH_LOG_WARNING,
                          "Denied connection from %s by tcp wrappers.",
                          eval_client(&req));
            refuse(&req);/* If connection is not allowed, clean up and exit.*/
          }

        if (signal(SIGCHLD, old_handler) == SIG_ERR && old_handler != SIG_IGN)
          {
            ssh_warning("new_connection_callback: Could not reset "
                        "SIGCHLD signal handler.");
          }
      }
#endif /* HAVE_LIBWRAP */

      /* Create a context structure for the connection. */
      c = ssh_xcalloc(1, sizeof(*c));
      c->shared = data;
      c->remote_ip = ssh_xstrdup(buf);
      c->stream = stream;
      if (data->config->try_reverse_mapping)
        ssh_tcp_get_host_by_addr(c->remote_ip, new_connection_start_finalize,
                                 c);
      else
        new_connection_start_finalize(SSH_TCP_FAILURE, c->remote_ip, c);
    }
  else
    {
      /* Parent */
      if (ret == -1)
        {
          s = "Forking a server for a new connection failed.";
          ssh_warning(s);
          ssh_log_event(data->config->log_facility, SSH_LOG_WARNING, s);
          ssh_stream_write(stream, (const unsigned char *)s, strlen(s));
          ssh_stream_write(stream, (const unsigned char *)"\r\n", 2);
          ssh_stream_destroy(stream);
          return;
        }

      ssh_stream_fd_mark_forked(stream);
      ssh_stream_destroy(stream);

      /* Stir the random state so that future connections get a
         different seed. */
      ssh_random_stir();

      /* Update the random seed file on disk. */
      ssh_randseed_update(data->user, data->config);

      if (data->config->max_connections)
        {
          data->connections++;
          SSH_DEBUG(4, ("Registering sigchld-handler for child '%d'.", ret));
          SSH_DEBUG(4, ("%d connections now open. (max %d)",
                        data->connections, data->config->max_connections));
          ssh_sigchld_register(ret, child_returned, data);
        }
    }














  ssh_debug("new_connection_callback returning");
}

void broadcast_callback(SshUdpListener listener, void *context)
{
  SshServerData data = (SshServerData)context;
  SshUdpError err;
  char remote_address[64];
  char remote_port[32];
  unsigned char packet[2048];
  size_t packet_len, consumed;
  char *remote_version = NULL, *remote_command = NULL;
  SshBufferStruct buffer[1];
  SshTime broadcast_time = 0L;

  ssh_buffer_init(buffer);
  while (1)
    {
      err = ssh_udp_read(listener,
                         remote_address, sizeof (remote_address),
                         remote_port, sizeof (remote_port),
                         packet, sizeof (packet),
                         &packet_len);
      broadcast_time = ssh_time();
      switch (err)
        {
        case SSH_UDP_OK:
          if (broadcast_time != data->broadcast_last)
            {
              data->broadcast_last = broadcast_time;
              data->broadcasts_per_second = 0;
            }
          else
            {
              data->broadcasts_per_second++;
            }
          if (data->broadcasts_per_second <
              data->config->broadcasts_allowed_per_second)
            {
              SSH_TRACE(0, ("broadcast: addr=%s port=%s len=%d",
                            remote_address,
                            remote_port,
                            packet_len));

              consumed = ssh_decode_array(packet, packet_len,
                                          SSH_FORMAT_UINT32_STR,
                                          &remote_command, NULL,
                                          SSH_FORMAT_UINT32_STR,
                                          &remote_version, NULL,
                                          SSH_FORMAT_END);
              if (consumed == 0)
                {
                  SSH_TRACE(0, ("malformed udp query"));
                  break;
                }
              SSH_TRACE(0, ("received datagram cmd=\"%s\" ver=\"%s\"",
                            remote_command,
                            remote_version));
              if (strcmp(remote_command, "server-query") == 0)
                {
                  ssh_buffer_clear(buffer);
                  ssh_encode_buffer(buffer,
                                    SSH_FORMAT_UINT32_STR,
                                    "server-reply",
                                    strlen ("server-reply"),
                                    SSH_FORMAT_UINT32_STR,
                                    SSH2_VERSION_STRING,
                                    strlen(SSH2_VERSION_STRING),
                                    SSH_FORMAT_UINT32_STR,
                                    data->config->port,
                                    strlen(data->config->port),
                                    SSH_FORMAT_END);
                  SSH_TRACE(0, ("sending reply"));
                  ssh_udp_send(listener,
                               remote_address,
                               remote_port,
                               ssh_buffer_ptr(buffer),
                               ssh_buffer_len(buffer));
                  break;
                }
              else if (strcmp(remote_command, "server-key-query") == 0)
                {
                  /* Not supported.  XXX */
                  break;
                }
              else
                {
                  break;
                }
            }
          else
            {
              SSH_TRACE(0, ("ignored (flood) broadcast: addr=%s port=%s "
                            "len=%d",
                            remote_address,
                            remote_port,
                            packet_len));
              break;
            }
        case SSH_UDP_HOST_UNREACHABLE:
        case SSH_UDP_PORT_UNREACHABLE:
          break;

        case SSH_UDP_NO_DATA:
          ssh_buffer_uninit(buffer);
          return;
        case SSH_UDP_INVALID_ARGUMENTS:
          SSH_TRACE(2, ("ssh_udp_read() returned with "
                        "SSH_UDP_INVALID_ARGUMENTS"));
          break;
        }
      ssh_xfree(remote_version);
      ssh_xfree(remote_command);
      remote_version = NULL;
      remote_command = NULL;
    }
  SSH_NOTREACHED;
}


void server_ssh_debug(const char *msg, void *context)
{
  SshServerData data = (SshServerData)context;

  if (data->config && data->config->quiet_mode)
    return;

  if (data->debug)
    fprintf(stderr, "debug[%d]: %s\r\n", getpid(), msg);
}

void server_ssh_warning(const char *msg, void *context)
{
  SshServerData data = (SshServerData)context;

  if (data->config && data->config->quiet_mode)
    return;

  ssh_log_event(data->config->log_facility, SSH_LOG_WARNING,
                "WARNING: %s", msg);

  fprintf(stderr, "%s[%d]: WARNING: %s\r\n", av0, getpid(), msg);
}


void server_ssh_fatal(const char *msg, void *context)
{
  SshServerData data = (SshServerData)context;
  data->ssh_fatal_called = TRUE;

  ssh_log_event(data->config->log_facility, SSH_LOG_ERROR, "FATAL ERROR: %s",
                msg);

  fprintf(stderr, "%s[%d]: FATAL: %s\r\n", av0, getpid(), msg);
  exit(255);
}

Boolean restart;


/* This is the logging callback */

void server_ssh_log(SshLogFacility facility, SshLogSeverity severity,
                    const char *msg, void *context)
{
  SshServerData data = (SshServerData)context;
  SshConfig config = data->config;
  int fac, sev;
  static int logopen = 0;
  static int logopt;
  static int logfac;

  if (!logopen)
    {
      logopt = LOG_PID;
#ifdef LOG_PERROR
      if (config->verbose_mode)
        logopt |= LOG_PERROR;
#endif /* LOG_PERROR */
      logfac = ssh_app_log_facility(config->log_facility);

      openlog(av0, logopt, logfac);
      logopen = 1;
    }

  /* Configuring for QuietMode and FascistLogging is an 'apparent
     user error', but if FascistLogging is enabled, we log
     everything. ssh_fatal()s are also logged.
     */
  if ((!config->quiet_mode || config->fascist_logging) ||
      data->ssh_fatal_called)
    {
      fac = ssh_app_log_facility(facility);
      sev = ssh_app_log_severity(severity);
      if( fac != -1 && sev != -1)
        {
          syslog(((fac != logfac) ? fac : 0) | sev, "%s", msg);
#ifndef LOG_PERROR
          /* Print it also to stderr. XXX */
#endif /* LOG_PERROR */
        }
    }
}

/* signal callback for SIGHUP*/

void sighup_handler(int signal, void *context)
{
  SshServerData data = (SshServerData) context;

  if (signal != SIGHUP)
    {
      SSH_DEBUG(0, ("Invalid signal received by SIGHUP-handler."));
      ssh_log_event(data->config->log_facility, SSH_LOG_WARNING,
                    "Invalid signal received by SIGHUP-handler.");
      return;
    }

  /* We cannot call fork() and exec() here directly, because we are in
     a signal handler. It seems that eventloop must be uninitialized
     for this to work. */
  restart = TRUE;

  ssh_event_loop_abort();
}


/* check whether parameter with options is correctly specified */

Boolean parameter_defined(const char param, int num, char **elements)
{
  int optidx;

  for (optidx = 1; optidx < num ; optidx++)
    {
      if (elements[optidx][0] == '-' || elements[optidx][0] == '+')
        if (elements[optidx][1] == param)
          if (elements[optidx + 1][0] != '-' && elements[optidx + 1][0] != '+')
            return TRUE;
    }

  return FALSE;
}

































































































































/*
 *
 *  SSH2 server main()
 *
 */

int main(int argc, char **argv)
{
  SshServerData data;
  SshUser user;
  char *conf_dir = NULL;
  char config_fn[1024];
  struct stat st;















  /* Save program name. */
  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];

  /* Initializations */
  restart = FALSE;

#ifdef HAVE_SCO_ETC_SHADOW
  set_auth_parameters(argc, argv);
#endif /* HAVE_SCO_ETC_SHADOW */

#ifdef SSHDIST_SESSION_SIA
#ifdef HAVE_SIA
  ssh_sia_initialize(argc, argv);
#endif /* HAVE_SIA */
#endif /* SSHDIST_SESSION_SIA */

  data = ssh_xcalloc(1, sizeof(*data));
  user = ssh_user_initialize(NULL, TRUE);

  data->ssh_fatal_called = FALSE;
  data->debug_dont_fork = TRUE;
  data->connections = 0;

  ssh_global_init();




  ssh_event_loop_initialize();

#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  ssh_pk_provider_register(&ssh_pk_if_modn_generator);
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */
#ifdef SSHDIST_CRYPT_DSA
  ssh_pk_provider_register(&ssh_pk_dl_modp_generator);
#endif /* SSHDIST_CRYPT_DSA */

  /* Create config context. */
  data->config = ssh_server_create_config();


  /* Register debug, fatal, and warning callbacks. */
  ssh_debug_register_callbacks(server_ssh_fatal, server_ssh_warning,
                               server_ssh_debug, (void *)data);

  /* Register log callback */
  ssh_log_register_callback(server_ssh_log, (void *)data);

  /* If -d is the first flag, we set debug level here.  It is reset
     later, but something may be lost, if we leave it 'til that. */
  if ((argc >= 3) && ((strcmp("-d", argv[1]) == 0) ||
                      (strcmp("-D", argv[1]) == 0)))
    {
      ssh_debug_set_level_string(argv[2]);
      if (strcmp("0", argv[2]) != 0)
        data->debug = TRUE;
      else
        data->debug = FALSE;
    }
  else if ((argc >= 2) && (strcmp("-v", argv[1]) == 0))
    {
      ssh_debug_set_level_string("2");
      data->debug = TRUE;
    }







































  /* Save command line options for ssh1 compatibility code. */
  data->config->ssh1_argv = argv;
  data->config->ssh1_argc = argc;

  /* Save information about current user. */
  data->user = user;


  /* Prevent core dumps to avoid revealing sensitive information. */
  ssh_signals_prevent_core(TRUE, data);
  ssh_register_signal(SIGPIPE, NULL_FNPTR, NULL);

  /* register SIGHUP for restart callback */
  ssh_register_signal(SIGHUP, sighup_handler, data);

  /* Register SIGCHLD signal handler, to kill those darn zombies */
  ssh_sigchld_initialize();



























































































  /* Read the standard server configuration file. if one wasn't specified
     on the commandline. */
  if (!parameter_defined('f', argc, argv))
    {

      if (ssh_user_uid(user) == 0 )
        {
          conf_dir = ssh_xstrdup(SSH_SERVER_DIR);
        }
      else
        {
          if ((conf_dir = ssh_userdir(user, data->config, TRUE)) == NULL)
            ssh_fatal("No ssh2 user directory");
        }

      ssh_snprintf(config_fn, sizeof(config_fn), "%s/%s",
                   conf_dir, SSH_SERVER_CONFIG_FILE);
      ssh_xfree(conf_dir);







      if ((stat(config_fn, &st) == 0))
        {
          if (!ssh_config_read_file(user, data->config, NULL, config_fn,
                                    SSH_CONFIG_TYPE_SERVER_GLOBAL))
            ssh_fatal("Failed to read config file %s", config_fn);
        }
      else
        {
          ssh_warning("Default configuration file \"%s\" does not exist.",
                      config_fn);
        }
    }

  ssh_opterr = 0;

  /* Parse the command line parameters. */
  while (1)
    {
      int option;

      option = ssh_getopt(argc, argv, SSHD_PROGRAM_ARGUMENTS, NULL);

      if (option == -1)
        {
          if (ssh_optind < argc)
            ssh_fatal("Extra arguments in command line");
          break;
        }

      /* Do we have a flag here ? */

      switch (option)
        {
          /* Debug mode */
        case 'D':
          data->debug_dont_fork = FALSE;
          /* FALLTHROUGH */
        case 'd':
          if (!ssh_optval)
            ssh_fatal("Illegal -d parameter.");
          data->config->verbose_mode = TRUE;
          ssh_debug_set_level_string(ssh_optarg);
          break;

          /* Verbose mode (= -d 2) */
        case 'v':
          data->config->verbose_mode = TRUE;
          ssh_debug_set_level_string("2");
          break;

          /* An additional configuration file */
        case 'f':
          if (!ssh_optval)
            ssh_fatal("Illegal -f parameter.");

          if (stat(ssh_optarg, &st) < 0)
            ssh_fatal("Alternate config file \"%s\" does not exist.",
                      ssh_optarg);

          if (!ssh_config_read_file(user, data->config, NULL, ssh_optarg,
                                    SSH_CONFIG_TYPE_SERVER_GLOBAL))
            ssh_fatal("Failed to read config file \"%s\"", ssh_optarg);
          break;

          /* Specify the login grace period */
        case 'g':
          if (!ssh_optval)
            ssh_fatal("Illegal -g parameter.");
          data->config->login_grace_time = atoi(ssh_optarg);
          if (data->config->login_grace_time < 0)
            ssh_fatal("Illegal login grace time '%s'", ssh_optarg);
          break;

          /* specify the host key file */
        case 'h':
          if (!ssh_optval)
            ssh_fatal("%s: Illegal -h parameter.", av0);
          ssh_host_key_add_private_key(ssh_optarg,
                                       data->config->host_keys_ctx);
          break;

          /* is inetd enabled ? */
        case 'i':
          data->config->inetd_mode = (ssh_optval != 0);
          break;

          /* Give one line of configuration data directly */
        case 'o':
          if (!ssh_optval)
            ssh_fatal("Illegal -o parameter.");

          /* You can set parameters also for subconfigs here. */
          if (ssh_config_parse_line_and_set_params(data->config, ssh_optarg,
                                                   SSH_CONFIG_TYPE_SERVER_ALL))
            ssh_fatal("Illegal -o parameter \"%s\"", ssh_optarg);

          break;

          /* Specify the port */
        case 'p':
          if (!ssh_optval)
            ssh_fatal("Illegal -p parameter.");

          ssh_xfree(data->config->port);
          data->config->port = ssh_xstrdup(ssh_optarg);
          break;

          /* Quiet mode */
        case 'q':
          data->config->quiet_mode = (ssh_optval != 0);
          break;









        default:
          ssh2_version(av0);
          if (ssh_optmissarg)
            fprintf(stderr, "Option -%c needs an argument\n", ssh_optopt);
          else
            fprintf(stderr, "Unknown option -%c\n", ssh_optopt);
          exit(1);
        }
    }

























  /* Display version, unless run in inetd mode. */
  if (!data->config->inetd_mode)
    ssh2_version(av0);

  data->debug = data->config->verbose_mode;

  /* Try to read in hostkeys and continue in the callback */
  ssh_host_key_read_keys(data->config->host_keys_ctx,
                         data->config, NULL, TRUE, FALSE,
                         server_hostkeys_initialized, data);












  ssh_debug("Running event loop");
  ssh_event_loop_run();

  ssh_signals_reset();




  ssh_global_uninit();


  if (data->listener)
    {
      if (data->pid_file)
        remove(data->pid_file);

      ssh_tcp_destroy_listener(data->listener);
      data->listener = NULL;

      if (data->broadcast_listener)
        ssh_udp_destroy_listener(data->broadcast_listener);
      data->broadcast_listener = NULL;
    }
  else if (restart)
    {
      restart = FALSE;
    }

  if (restart)
    {
      int ret;

      ssh_log_event(data->config->log_facility, SSH_LOG_WARNING,
                    "restarting...");
      ret = fork();

      if (ret == 0)
        {
          /* Child */
          execvp(argv[0], argv);
          ssh_fatal("Restart (exec) failed on SIGHUP. "
                    "(error message \"%s\")",
                    strerror(errno));
        }
      else
        {
          /* Parent */
          if (ret == -1)
            {
              /* Fork failed */
              ssh_fatal("Restart (fork) failed on SIGHUP.");
            }
          else
            {
              exit(0);
            }
        }
    }












  ssh_random_free();
  ssh_app_free_global_regex_context();

  /* free the server configuration data */
  ssh_config_free(data->config);

  ssh_debug("Exiting event loop");
  ssh_event_loop_uninitialize();

  ssh_user_free(data->user, TRUE);
  ssh_xfree(data->pid_file);
  ssh_xfree(data);

  return 0;
}

void server_hostkeys_initialized(SshHostKeyStatus status,
                                 void *context)
{
  SshServerData data = (SshServerData)context;
  SshUser user = data->user;
  FILE *f;

  if (status == SSH_HOSTKEY_ERROR)
    ssh_fatal("Unable to load any hostkeys");
  














  /* load the random seed */
  ssh_randseed_open(user, data->config);

  /* Finalize the initialization. */
  ssh_config_init_finalize(data->config);


























































































  ssh_debug("Becoming server.");


  /* Check if we are being called from inetd. */
  if (data->config->inetd_mode)
    {
      SshStream stream;

      ssh_log_event(data->config->log_facility,
                    SSH_LOG_WARNING,
                    "Starting daemon in inetd mode.");
      /* We are being called from inetd.  Take stdio to be the connection
         and proceed with the new connection. */
      stream = ssh_stream_fd_stdio();
      ssh_debug("processing stdio connection");
      new_connection_callback(SSH_TCP_NEW_CONNECTION, stream, (void *)data);
      ssh_debug("got_connection returned");
    }
  else
    {
      /* Start as daemon. */

      ssh_debug("Creating listener");
      data->listener = ssh_tcp_make_listener(data->config->listen_address,
                                             data->config->port,
                                             NULL,
                                             new_connection_callback,
                                             (void *)data);
      if (data->listener == NULL)
        ssh_fatal("Creating listener failed: port %s probably already in use!",
                  data->config->port);
      ssh_debug("Listener created");

      if (data->config->broadcasts_allowed_per_second > 0)
        {
          data->broadcast_listener = ssh_udp_make_listener(SSH_IPADDR_ANY_IPV4,
                                                           SSH_DEFAULT_PORT,
                                                           NULL,
                                                           NULL,
                                                           broadcast_callback,
                                                           (void *)data);
          if (data->broadcast_listener == NULL)
            ssh_warning("Creating broadcast listener failed: port %s probably"
                        "already in use!", data->config->port);
        }
      else
        {
          ssh_debug("no udp listener created.");
        }
      ssh_log_event(data->config->log_facility,
                    SSH_LOG_WARNING,
                    "Listener created on port %s.",
                    data->config->port);

#ifndef VXWORKS
      /* If not debugging, fork into background. */
      if (!data->debug)
        {
#ifdef HAVE_DAEMON
          if (daemon(0, 0) < 0)
            ssh_fatal("daemon(): %.100s", strerror(errno));
#else /* HAVE_DAEMON */
#ifdef TIOCNOTTY
          int fd;
#endif /* TIOCNOTTY */
          /* Running as a daemon; fork to background. */
          if (fork() != 0)
            {
              /* Parent */
              exit(0);
            }

          /* Redirect stdin, stdout, and stderr to /dev/null. */
          freopen("/dev/null", "r", stdin);
          freopen("/dev/null", "w", stdout);
          freopen("/dev/null", "w", stderr);

          /* Disconnect from the controlling tty. */
#ifdef TIOCNOTTY
          fd = open("/dev/tty", O_RDWR|O_NOCTTY);
          if (fd >= 0)
            {
              (void)ioctl(fd, TIOCNOTTY, NULL);
              close(fd);
            }
#endif /* TIOCNOTTY */
#ifdef HAVE_SETSID
#ifdef ultrix
          setpgrp(0, 0);
#else /* ultrix */
          if (setsid() < 0)
            ssh_log_event(data->config->log_facility, SSH_LOG_NOTICE,
                          "setsid: %.100s", strerror(errno));
#endif /* ultrix */
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
        }
#endif /* VXWORKS */
    }










  /* Save our process id in the pid file. */
  ssh_xdsprintf(&data->pid_file, SSHD_PIDDIR "/sshd2_%s.pid",
                data->config->port);

  SSH_DEBUG(5, ("Trying to create pidfile %s", data->pid_file));
  f = fopen(data->pid_file, "w");
  if (f == NULL)
    {
      ssh_xfree(data->pid_file);
      ssh_xdsprintf(&data->pid_file, SSH_SERVER_DIR "/sshd2_%s.pid",
                    data->config->port);
      SSH_DEBUG(5, ("Trying to create pidfile %s", data->pid_file));
      f = fopen(data->pid_file, "w");
    }
  if (f != NULL)
    {
      SSH_DEBUG(5, ("Writing pidfile %s", data->pid_file));
      fprintf(f, "%ld\n", (long)getpid());
      fclose(f);
    }

  ssh_log_event(data->config->log_facility,
                SSH_LOG_WARNING,
                "Daemon is running.");

}






















































































































































































































































































































