/*
  sshd-check-conf.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Description: Program to check configuration of sshd2.

  Created: Sun May 27 12:52:20 2001

  Copyright (C) 2001,2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#include "ssh2includes.h"
#include "sshconfig.h"
#include "sshgetopt.h"
#include "sshuserfiles.h"
#include "sshdsprintf.h"
#include "ssheloop.h"
#include "sshregex.h"
#include "sshinet.h"
#include "sshtcp.h"
#include "sshnameserver.h"
#include "sshappcommon.h"
#include "auths-common.h"
#include "sigchld.h"
#include "sshfsm.h"
#include "sshreadline.h"
#include "sshtty.h"
#include "sshfdstream.h"

#define SSH_DEBUG_MODULE "SshdCheckConf"

void check_conf(const char *user, const char *remote_name,
                const char *remote_ip, SshConfig config)
{
  const char *ip = NULL;
  SshUser uc = ssh_user_initialize(user, FALSE);

  if (uc == NULL)
    {
      Boolean nan = FALSE;
      char *cp;

      SSH_TRACE(2, ("User couldn't be initialized with name, assuming "
                    "given username is a UID.."));

      for (cp = (char *)user; *cp; cp++)
        if (!isdigit(*cp))
          nan = TRUE;

      if (nan)
        {
          SSH_TRACE(2, ("Given string is neither a UID or a username."));
        }
      else
        {
          uc = ssh_user_initialize_with_uid(atoi(user), FALSE);
          if (uc)
            user = ssh_user_name(uc);
          else
            SSH_TRACE(2, ("User still couldn't be initialized, user probably "
                          "doesn't exist."));
        }
    }

  SSH_ASSERT(user != NULL);
  SSH_ASSERT(remote_name != NULL);
  SSH_ASSERT(remote_ip != NULL);
  SSH_ASSERT(config != NULL);

  if (ssh_inet_is_valid_ip_address(remote_ip))
    ip = remote_ip;
  else
    ip = "UNKNOWN";

  ssh_informational("\r\nVerifying %s@%s[%s]...\r\n", user, remote_name, ip);

  if (ssh_server_auth_check_host_generic(remote_name, ip,
                                         config->denied_hosts,
                                         config->allowed_hosts,
                                         config->require_reverse_mapping))
    ssh_informational("  Logins from %s[%s] denied.\r\n", remote_name,
                      ip);
  else
    ssh_informational("  Logins from %s[%s] allowed.\r\n", remote_name,
                      ip);

  if (ssh_server_auth_check_host_generic(remote_name, ip,
                                         config->denied_shosts,
                                         config->allowed_shosts,
                                         FALSE))
    ssh_informational("  Hostbased cannot be used from %s[%s].\r\n",
                      remote_name, ip);
  else
    ssh_informational("  Hostbased can be used from %s[%s].\r\n", remote_name,
                      ip);

  if (ssh_server_auth_check_user_generic(uc, user,
                                         "is allowed to login",
                                         "is denied login",
                                         remote_name, ip,
                                         config->allowed_users,
                                         config->denied_users,
                                         config->allowed_groups,
                                         config->denied_groups))
    {
      ssh_informational("  Login by user %s denied.\r\n", user);
    }
  else
    {
      ssh_informational("  Login by user %s allowed.\r\n", user);

      if (ssh_server_auth_check_user_generic(uc, user,
                                             "won't be chrooted",
                                             "will be chrooted",
                                             remote_name, ip,
                                             NULL,
                                             config->chroot_users,
                                             NULL,
                                             config->chroot_groups))
        ssh_informational("  User %s will be chrooted.\r\n", user);
      else
        ssh_informational("  User %s will not be chrooted.\r\n", user);

      if (ssh_server_auth_check_user_generic
          (uc, user,
           "is allowed TCP forwarding",
           "is denied TCP forwarding",
           remote_name, ip,
           config->allowed_tcp_forwarding_users,
           config->denied_tcp_forwarding_users,
           config->allowed_tcp_forwarding_groups,
           config->denied_tcp_forwarding_groups))
        ssh_informational("  TCP forwarding by user %s denied.\r\n", user);
      else
        ssh_informational("  TCP forwarding by user %s allowed.\r\n", user);
    }
}

int render_indent(char *buf, int buf_size, int precision, void *datum)
{
  /* This kludge is here to get rid of compilation warnings on 64-bit
     platforms. */
  unsigned long ind_num = (unsigned long)datum;
  int indentation = (int)ind_num;
  int i;
  if (indentation > buf_size)
    indentation = buf_size + 1;
  
  for (i = 0; i < indentation; i++)
    buf[i] = ' ';
  
  return indentation;
}

#define INFO ssh_informational

void dump_string_list(char *name, SshADTContainer c, int indentation,
                      int width)
{
  SshADTHandle h;
  
  if (!c || ssh_adt_num_objects(c) < 1)
    return;
  
  h = ssh_adt_enumerate_start(c);
  SSH_ASSERT(h != SSH_ADT_INVALID);
  INFO("%@%s = %s", render_indent, indentation, name,
       (char *)ssh_adt_get(c, h));
  for (h = ssh_adt_enumerate_next(c, h);
       h != SSH_ADT_INVALID; h = ssh_adt_enumerate_next(c, h))
    {
      INFO(",%s", (char *)ssh_adt_get(c, h));
    }
  INFO("\r\n");
}

static Boolean check_for_changed_rex_syntax(int *last, SshRegexSyntax current,
                                            int indentation, int width,
                                            Boolean prepend_newline)
{
  Boolean rex_syntax_changed = FALSE;

  if (current != *last)
    {
      *last = current;
      rex_syntax_changed = TRUE;
    }
  if (rex_syntax_changed)
    {
      const char *syntax_str;
      switch (*last)
        {
        case SSH_REGEX_SYNTAX_SSH:
          syntax_str = "ssh";
          break;
        case SSH_REGEX_SYNTAX_EGREP:
          syntax_str = "egrep";
          break;
        case SSH_REGEX_SYNTAX_ZSH_FILEGLOB:
          syntax_str = "zsh_fileglob";
          break;
        default:
          syntax_str = "unknown # (? should not happen)";
          break;
        }
      if (prepend_newline)
        INFO("\r\n");
      INFO("%@# regex-syntax %s\r\n", render_indent, indentation,
           syntax_str);
    }
  return rex_syntax_changed;
}

void dump_pattern_list(char *name, SshADTContainer c, int indentation,
                       int width, int *last_regex_syntax)
{
  SshADTHandle h;
  SshPatternHolder holder;
  Boolean first = TRUE, output_something = FALSE;
  
  if (!c || ssh_adt_num_objects(c) < 1)
    return;
  
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      holder = ssh_adt_get(c, h);
      if (check_for_changed_rex_syntax(last_regex_syntax, holder->regex_syntax,
                                       indentation, width, output_something))
        {
          first = TRUE;
          output_something = FALSE;
        }
      if (first)
        {
          INFO("%@%s = %s", render_indent, indentation, name, holder->pattern);
          first = FALSE;
        }
      else
        {
          INFO(",%s", holder->pattern);
        }
      if (!output_something)
        output_something = TRUE;
    }
  INFO("\r\n");
}

/* string */
#define DUMP_STRING(name, string)                                        \
do                                                                       \
{                                                                        \
  if (string)                                                            \
    INFO("%@%s = %s\r\n", render_indent, indentation, (name), (string)); \
}                                                                        \
while(0)

/* bool */
#define DUMP_BOOL(name, bool_val)                               \
do                                                              \
{                                                               \
  INFO("%@%s = %s\r\n", render_indent, indentation, (name),     \
       (bool_val) ? "yes" : "no");                              \
}                                                               \
while(0)

/* string_list */
#define DUMP_STRING_LIST(name, c)                       \
do                                                      \
{                                                       \
  dump_string_list((name), (c), indentation, width);    \
}                                                       \
while(0)

/* number */
#define DUMP_NUM(name, num)                                             \
do                                                                      \
{                                                                       \
  INFO("%@%s = %ld\r\n", render_indent, indentation, (name), (num));    \
}                                                                       \
while(0)

/* pattern_list */
#define DUMP_PATTERN_LIST(name, c)                                        \
do                                                                        \
{                                                                         \
  dump_pattern_list((name), (c), indentation, width, &last_regex_syntax); \
}                                                                         \
while(0)

/* logfacility */
#define DUMP_LOG_FACILITY(name, fac)                                     \
do                                                                       \
{                                                                        \
  char *fac_name = ssh_config_log_facility_name(fac);                    \
  if (fac_name)                                                          \
    INFO("%@%s = %s\r\n", render_indent, indentation, (name), fac_name); \
  ssh_xfree(fac_name);                                                   \
}                                                                        \
while(0)

#define DUMP_PERMIT_ROOT(name, permit)                                  \
do                                                                      \
{                                                                       \
  const char *str;                                                      \
  switch (permit)                                                       \
    {                                                                   \
    case SSH_ROOTLOGIN_FALSE:                                           \
      str = "no";                                                       \
      break;                                                            \
    case SSH_ROOTLOGIN_TRUE:                                            \
      str = "yes";                                                      \
      break;                                                            \
    case SSH_ROOTLOGIN_NOPWD:                                           \
      str = "nopwd";                                                    \
      break;                                                            \
    default:                                                            \
      str = "unknown (? should not happen)";                            \
    }                                                                   \
  INFO("%@%s = %s\r\n", render_indent, indentation, (name), str);       \
}                                                                       \
while(0)

/* XXX cert variables, hostkeys */

void dump_config(SshConfig config, int indentation, int width)
{
  int last_regex_syntax = -1;
  SshADTHandle h;
  SshADTContainer c;
  
  SSH_PRECOND(config != NULL);
  INFO("# General\r\n");
  DUMP_STRING("Port", config->port);
  DUMP_STRING("Ciphers", config->ciphers);
  DUMP_STRING("MACs", config->macs);
  DUMP_STRING("ProtocolVersionString", config->protocol_version_string);
  /* XXX ListenAddress */
  DUMP_NUM("MaxConnections", config->max_connections);
  /* XXX HostKeyFile */
  /* XXX PublicHostKeyFile */
  DUMP_STRING("RandomSeedFile", config->random_seed_file);
  /* XXX PGPPublicKeyFile */
  /* XXX PGPSecretKeyFile */
  DUMP_BOOL("RequireReverseMapping", config->require_reverse_mapping);
  DUMP_BOOL("ResolveClientHostName", config->try_reverse_mapping);
  DUMP_BOOL("KeepAlive", config->keep_alive);
  DUMP_BOOL("NoDelay", config->no_delay);
  DUMP_LOG_FACILITY("SysLogFacility", config->log_facility);
  DUMP_LOG_FACILITY("SftpSysLogFacility", config->sftp_server_log_facility);
  DUMP_BOOL("FascistLogging", config->fascist_logging);
  DUMP_NUM("RekeyIntervalSeconds", config->rekey_interval_seconds);
  DUMP_STRING("BannerMessageFile", config->banner_message_file_path);
  DUMP_BOOL("QuietMode", config->quiet_mode);
  DUMP_BOOL("VerboseMode", config->verbose_mode);
  DUMP_NUM("MaxBroadcastsPerSecond", config->broadcasts_allowed_per_second);

  INFO("\r\n# Compatibility\r\n");
  DUMP_BOOL("Ssh1Compatibility", config->ssh1compatibility);
  DUMP_STRING("Sshd1Path", config->ssh1_path);
  DUMP_STRING("Sshd1ConfigFile", config->sshd1_config_file_path);
  
  INFO("\r\n# Authentication and authorization\r\n");
  DUMP_STRING_LIST("AllowedAuthentications", config->allowed_authentications);
  DUMP_STRING_LIST("RequiredAuthentications",
                   config->required_authentications);
  DUMP_PATTERN_LIST("AllowHosts", config->allowed_hosts);
  DUMP_PATTERN_LIST("DenyHosts", config->denied_hosts);
  DUMP_PATTERN_LIST("AllowUsers", config->allowed_users);
  DUMP_PATTERN_LIST("DenyUsers", config->denied_users);
  DUMP_PATTERN_LIST("AllowGroups", config->allowed_groups);
  DUMP_PATTERN_LIST("DenyGroups", config->denied_groups);
  DUMP_PATTERN_LIST("AllowSHosts", config->allowed_shosts);
  DUMP_PATTERN_LIST("DenySHosts", config->denied_shosts);
  DUMP_BOOL("IgnoreRhosts", config->ignore_rhosts);
  DUMP_BOOL("IgnoreRootRhosts", config->ignore_root_rhosts);
#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
  DUMP_NUM("AuthKbdInt.NumOptional", config->auth_kbd_int_optional_needed);
  DUMP_STRING_LIST("AuthKbdInt.Optional", config->auth_kbd_int_optional);
  DUMP_STRING_LIST("AuthKbdInt.Required", config->auth_kbd_int_required);
  DUMP_NUM("AuthKbdInt.Retries", config->auth_kbd_int_retries);
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */
  DUMP_NUM("LoginGraceTime", config->login_grace_time);

  DUMP_PERMIT_ROOT("PermitRootLogin", config->permit_root_login);
  DUMP_STRING("AuthorizationFile", config->authorization_file);
  DUMP_BOOL("UserKnownHosts", config->user_known_hosts);
  DUMP_BOOL("PermitEmptyPasswords", config->permit_empty_passwords);
  DUMP_BOOL("HostbasedAuthForceClientHostnameDNSMatch",
            config->hostbased_force_client_hostname_dns_match);
  DUMP_NUM("PasswordGuesses", config->password_guesses);
  DUMP_NUM("AuthInteractiveFailureTimeout",
           config->auth_interactive_failure_timeout);
  DUMP_BOOL("StrictModes", config->strict_modes);
#ifdef SSH_SERVER_WITH_SECURID
  DUMP_NUM("SecurIdGuesses", config->securid_guesses);
#endif /* SSH_SERVER_WITH_SECURID */







  DUMP_STRING("ExternalAuthorizationProgram",
              config->external_authorization_prog);
  DUMP_STRING("PasswdPath", config->passwd_path);
              
  INFO("\r\n# Forwardings\r\n");
  DUMP_BOOL("ForwardX11", config->x11_forwarding);
  DUMP_BOOL("ForwardAgent", config->agent_forwarding);
  DUMP_BOOL("AllowTcpForwarding", config->allow_tcp_forwarding);
  DUMP_PATTERN_LIST("AllowTcpForwardingForUsers",
                    config->allowed_tcp_forwarding_users);
  DUMP_PATTERN_LIST("DenyTcpForwardingForUsers",
                    config->denied_tcp_forwarding_users);
  DUMP_PATTERN_LIST("AllowTcpForwardingForGroups",
                    config->allowed_tcp_forwarding_groups);
  DUMP_PATTERN_LIST("DenyTcpForwardingForGroups",
                    config->denied_tcp_forwarding_groups);
  c = config->forward_acls;
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      SshForwardACL forw_acl = ssh_adt_get(c, h);
      check_for_changed_rex_syntax(&last_regex_syntax,
                                   forw_acl->user_pattern.regex_syntax,
                                   indentation, width, FALSE);
      INFO("%@ForwardACL = %s %s %s %s %s\r\n", render_indent, indentation,
           forw_acl->allow ? "allow" : "deny",
           forw_acl->local ? "local" : "remote",
           forw_acl->user_pattern.pattern,
           forw_acl->forward_pattern.pattern,
           forw_acl->originator_pattern.pattern ?
           forw_acl->originator_pattern.pattern : "");
    }  
  DUMP_STRING("XauthPath", config->xauth_path);
  
  INFO("\r\n# Miscellaneous user setup\r\n");
  DUMP_STRING("UserConfigDirectory", config->user_conf_dir);
  DUMP_BOOL("PrintMOTD", config->print_motd);
  DUMP_BOOL("CheckMail", config->check_mail);
  DUMP_NUM("IdleTimeout", config->idle_timeout);
  DUMP_PATTERN_LIST("SettableEnvironmentVars", config->settable_env_vars);
  DUMP_PATTERN_LIST("ChrootUsers", config->chroot_users);
  DUMP_PATTERN_LIST("ChrootGroups", config->chroot_groups);
  
  /* Subsystems. */
  c = config->subsystems;  
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      char *key = ssh_adt_get(c, h);
      char *path = ssh_adt_map_lookup(c, h);
      INFO("%@subsystem-%s = %s\r\n", render_indent, indentation, key, path);
    }
  INFO("\r\n");
}

/**************************************************************/
/* The state machine */

/* Structures. */
typedef struct CheckGDataRec
{
  SshFSM fsm;
  char *config_file;
  SshConfig config;
  char *current;
  Boolean debug_mode;
} CheckGDataStruct, *CheckGData;

typedef struct StepTDataRec
{
  SshADTContainer list;
  SshADTHandle h;
} StepTDataStruct, *StepTData;

typedef struct ReadlineTDataRec
{
  SshOperationHandle op_handle;
  char *prompt;
  char *line;
  SshStream stdio_stream;
  SshReadLineCtx rl_ctx;
} ReadlineTDataStruct, *ReadlineTData;

typedef struct CheckTDataRec
{
  SshOperationHandle op_handle;
  char *user;
  char *host_or_ip;
  Boolean ip_lookup;
  char *lookup_result;
} CheckTDataStruct, *CheckTData;

/* Destructors. */

void check_thread_destructor(SshFSM fsm, void *ctdata)
{
  CheckTData tdata = (CheckTData) ctdata;

  if (tdata->op_handle)
    {
      ssh_operation_abort(tdata->op_handle);
      tdata->op_handle = NULL;
    }
  ssh_xfree(tdata->user);
  ssh_xfree(tdata->host_or_ip);
  ssh_xfree(tdata->lookup_result);
  ssh_xfree(tdata);
}

void step_thread_destructor(SshFSM fsm, void *ctdata)
{
  StepTData tdata = (StepTData) ctdata;
  if (tdata->list)
    ssh_adt_destroy(tdata->list);
  ssh_xfree(tdata);
}

void readline_thread_destructor(SshFSM fsm, void *ctdata)
{
  ReadlineTData tdata = (ReadlineTData) ctdata;

  if (tdata->op_handle)
    ssh_operation_abort(tdata->op_handle);
  ssh_xfree(tdata->line);
  ssh_xfree(tdata->prompt);
  ssh_stream_destroy(tdata->stdio_stream);
  ssh_readline_eloop_uninitialize(tdata->rl_ctx);
  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);
  ssh_xfree(tdata);
}

SSH_FSM_STEP(check_readline);
SSH_FSM_STEP(check_readline_process);
SSH_FSM_STEP(check_step_list);
SSH_FSM_STEP(check_finish);

SSH_FSM_STEP(check_begin);
SSH_FSM_STEP(check_lookup);
SSH_FSM_STEP(check_complete);

SshFSMStateDebugStruct check_states[] =
{
  { "check_step_list", "Step forward in list.", check_step_list },

  { "check_readline", "Read a line.", check_readline },
  { "check_readline_process", "Process read line.", check_readline_process },

  { "check_finish", "Finish FSM.", check_finish },

  { "check_begin", "Initialize check.", check_begin },
  { "check_lookup", "Start lookup of item.", check_lookup },
  { "check_complete", "Perform check.", check_complete }
};

void rl_callback(const char *line, void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  ReadlineTData tdata = (ReadlineTData) ssh_fsm_get_tdata(thread);;

  ssh_xfree(tdata->line);

  if (line == NULL)
    tdata->line = NULL;
  else
    tdata->line = ssh_xstrdup(line);

  fprintf(stderr, "\r\n");
  
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(check_readline)
{
  ReadlineTData tdata = (ReadlineTData) thread_context;
  SSH_FSM_SET_NEXT(check_readline_process);
  SSH_FSM_ASYNC_CALL(ssh_readline_eloop(tdata->prompt, "", tdata->rl_ctx,
                                        rl_callback, thread));
}

SSH_FSM_STEP(check_readline_process)
{
  char *cp = NULL, *p = NULL;
  SshFSMThread ct;
  CheckTData child_tdata = NULL;
  CheckGData gdata = (CheckGData) fsm_context;
  ReadlineTData tdata = (ReadlineTData) thread_context;

  SSH_FSM_SET_NEXT(check_readline);

  SSH_TRACE(2, ("Got line `%s'.", tdata->line ? tdata->line : "NULL"));

  if (tdata->line == NULL)
    {
      SSH_FSM_SET_NEXT(check_finish);
      return SSH_FSM_CONTINUE;
    }

  cp = tdata->line;

  /* Why is it that stripping whitespace always looks (and is) a
     terrible kludge? Here we go... */
  for (cp = tdata->line; *cp != '\0' && isspace(*cp); cp++)
    ;

  if (strlen(cp) <= 0)
    return SSH_FSM_CONTINUE;

  for (p = cp + strlen(cp) - 1; p > cp && isspace(*p); p--)
    ;

  *(p + 1) = '\0';
  /* You may look again. */

  if (strlen(cp) <= 0)
    return SSH_FSM_CONTINUE;

  SSH_TRACE(4, ("stripped line: `%s'", cp));

  if (strcasecmp(cp, "quit") == 0)
    {
      SSH_FSM_SET_NEXT(check_finish);
      return SSH_FSM_CONTINUE;
    }
  else if (strcasecmp(cp, "dump") == 0)
    {
      if (gdata->config)
        dump_config(gdata->config, 0, 80);
      else
        ssh_informational("No config data to dump; input <user@host> "
                          "first.\r\n");
    }
  else
    {
      ssh_xfree(gdata->current);
      gdata->current = ssh_xstrdup(cp);

      child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
      ct = ssh_fsm_thread_create(gdata->fsm, check_begin, NULL_FNPTR,
                                 check_thread_destructor, child_tdata);
      SSH_FSM_WAIT_THREAD(ct);
    }

  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(check_step_list)
{
  SshFSMThread ct;
  CheckTData child_tdata = NULL;
  CheckGData gdata = (CheckGData) fsm_context;
  StepTData tdata = (StepTData) thread_context;

  if (tdata->h == NULL)
    tdata->h = ssh_adt_enumerate_start(tdata->list);
  else
    tdata->h = ssh_adt_enumerate_next(tdata->list, tdata->h);

  if (tdata->h == SSH_ADT_INVALID)
    {
      SSH_TRACE(1, ("List at end."));
      SSH_FSM_SET_NEXT(check_finish);
      return SSH_FSM_CONTINUE;
    }

  ssh_xfree(gdata->current);
  gdata->current = ssh_xstrdup(ssh_adt_get(tdata->list, tdata->h));

  child_tdata = ssh_xcalloc(1, sizeof(*child_tdata));
  ct = ssh_fsm_thread_create(gdata->fsm, check_begin, NULL_FNPTR,
                             check_thread_destructor, child_tdata);
  SSH_FSM_WAIT_THREAD(ct);
}

SSH_FSM_STEP(check_finish)
{
  CheckGData gdata = (CheckGData) fsm_context;

  if (gdata->config)
    ssh_config_free(gdata->config);

  ssh_fsm_destroy(gdata->fsm);
  ssh_xfree(gdata->current);
  ssh_xfree(gdata->config_file);

  return SSH_FSM_FINISH;
}

/* Child states. */
SSH_FSM_STEP(check_begin)
{
  char *current = NULL, *user = NULL, *host = NULL;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
  SshRegexMatcher rex;
  CheckGData gdata = (CheckGData) fsm_context;
  CheckTData tdata = (CheckTData) thread_context;

  SSH_FSM_SET_NEXT(check_lookup);

  current = gdata->current;

  rex = ssh_regex_create(rex_ctx,
                         "^{([-@]+)@}?(.*)$",
                         SSH_REGEX_SYNTAX_SSH);
  if (!rex)
    ssh_fatal("Unable to create regex for commandline parameter parsing.");

  if (ssh_regex_match_cstr(rex, current))
    {
      user = ssh_regex_get_submatch(rex, 1);
      host = ssh_regex_get_submatch(rex, 2);
    }
  else
    {
      ssh_warning("Regex didn't match `%s'", current);
      ssh_regex_free(rex);
      return SSH_FSM_FINISH;
    }

  tdata->user = ssh_xstrdup(user ? user : "UNKNOWN");
  tdata->host_or_ip =  ssh_xstrdup(host ? host : "UNKNOWN");

  ssh_regex_free(rex);

  return SSH_FSM_CONTINUE;
}

void addr_lookup_cb(SshTcpError error, const char *result,
                    void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  CheckTData tdata = (CheckTData) ssh_fsm_get_tdata(thread);

  tdata->op_handle = NULL;

  SSH_DEBUG(4, ("Received error code %d", error));

  if (error != SSH_TCP_OK)
    {
      tdata->lookup_result = NULL;
      ssh_warning("DNS lookup failed for %s.", tdata->host_or_ip);
    }
  else
    {
      tdata->lookup_result = ssh_xstrdup(result);
      SSH_TRACE(3, ("Lookup result: %s", result));
    }

  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

SSH_FSM_STEP(check_lookup)
{
  CheckTData tdata = (CheckTData) thread_context;

  SSH_FSM_SET_NEXT(check_complete);

  if (ssh_inet_is_valid_ip_address(tdata->host_or_ip))
    {
      tdata->ip_lookup = FALSE;
      /* Host name is a valid IP-address, so we need hostname. */
      SSH_FSM_ASYNC_CALL(tdata->op_handle =
                         ssh_tcp_get_host_by_addr(tdata->host_or_ip,
                                                  addr_lookup_cb,
                                                  (void *)thread));
    }
  else
    {
      tdata->ip_lookup = TRUE;
      /* Host name is not a valid IP-address, so we are assuming that
         we were given valid host name, and we try to resolve host's
         IP-address(es). */
      SSH_FSM_ASYNC_CALL(tdata->op_handle =
                         ssh_tcp_get_host_addrs_by_name(tdata->host_or_ip,
                                                        addr_lookup_cb,
                                                        (void *)thread));
    }
}

/* XXX Copied from sshd2.c . Implement something more generic. */
typedef struct HeadingMatchCtxRec {
  char *user;
  char *uid_str;
  char *remote_host;
  char *remote_ip;
  size_t num_groups;
  char **groups;
  char **gid_strs;
} HeadingMatchCtxStruct, *HeadingMatchCtx;

Boolean match_heading_cb(const char *pattern, SshMetaConfig metaconfig,
                         void *context)
{
  HeadingMatchCtx match_ctx = (HeadingMatchCtx) context;
  SshPatternHolderStruct holder;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();

  SSH_PRECOND(match_ctx->remote_host != NULL);
  SSH_PRECOND(match_ctx->remote_ip != NULL);
  
  memset(&holder, 0, sizeof(holder));
  holder.regex_syntax = metaconfig->regex_syntax;

  if (match_ctx->user != NULL)
    {
      holder.pattern = (char *)pattern;

      SSH_ASSERT(match_ctx->uid_str != NULL);
      return ssh_match_user_groups(match_ctx->user, match_ctx->uid_str,
                                   match_ctx->num_groups,
                                   match_ctx->groups,
                                   match_ctx->gid_strs,
                                   match_ctx->remote_host,
                                   match_ctx->remote_ip,
                                   &holder, rex_ctx);
    }
  holder.pattern = (char *)pattern;
  return ssh_match_host_id(match_ctx->remote_host, match_ctx->remote_ip,
                           &holder, rex_ctx);
}

SSH_FSM_STEP(check_complete)
{
  size_t num_groups = 0;
  char *cp = NULL, *remote_host = NULL, *remote_ip = NULL, *uid_str;
  char **group_strs = NULL, **gid_strs = NULL;
  CheckGData gdata = (CheckGData) fsm_context;
  CheckTData tdata = (CheckTData) thread_context;
  SshUser conf_uc, uc;
  SshADTContainer c;
  SshADTHandle h;
  Boolean match;
  HeadingMatchCtxStruct match_ctx;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
#ifndef WIN32
  SshGroup *groups = NULL;
#endif /* WIN32 */
  int i;

  memset(&match_ctx, 0, sizeof(match_ctx));
  
  if (tdata->ip_lookup)
    {
      char *p = NULL;

      cp = tdata->lookup_result ? tdata->lookup_result : "UNKNOWN";

      /* Check config for first IP-address only.  */
      /* XXX document it. */
      p = strchr(cp, ',');
      if (p)
        {
          ssh_warning("System name '%s' resolved to multiple IP-addresses "
                      "('%s'); using the first one. If you want to check "
                      "the rest, you need specify the IP-addresses "
                      "separately by hand.", tdata->host_or_ip, cp);
          *p = '\0';
        }
      remote_ip = cp;
      remote_host = tdata->host_or_ip;
    }
  else
    {
      cp = tdata->lookup_result ? tdata->lookup_result : "UNKNOWN";

      remote_ip = tdata->host_or_ip;
      remote_host = cp;
    }

  if (gdata->config)
    ssh_config_free(gdata->config);

  gdata->config = ssh_server_create_config();

  conf_uc = ssh_user_initialize(NULL, FALSE);
  uc = ssh_user_initialize(tdata->user, FALSE);
  
  if (!ssh_config_read_file(conf_uc, gdata->config, NULL,
                            gdata->config_file,
                            SSH_CONFIG_TYPE_SERVER_GLOBAL))
    ssh_fatal("Failed to read config file \"%s\"", gdata->config_file);

  if (!gdata->config->try_reverse_mapping)
    remote_host = remote_ip;

  c = gdata->config->host_configurations;

  match_ctx.remote_host = remote_host;
  match_ctx.remote_ip = remote_ip;

  /* Host specific config. */
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      SshSubConfig host_config = ssh_adt_get(c, h);
      match = ssh_match_host_id(remote_host, remote_ip, &host_config->pattern,
                                rex_ctx);

      /* If so, load the configuration file (log missing or
         non-readable config files) */
      if (match)
        {
          char *config_file;
          SSH_DEBUG(2, ("Address %s[%s] matched with `%s'.",
                        remote_host, remote_ip,
                        host_config->pattern.pattern));
          SSH_DEBUG(2, ("Reading per-host config file `%s'...",
                        host_config->config_file));
          config_file = ssh_userdir_prefix(conf_uc, gdata->config,
                                           host_config->config_file,
                                           SSH_SERVER_DIR, FALSE);
          if (!ssh_config_read_file_ext(conf_uc, gdata->config, NULL,
                                        match_heading_cb, &match_ctx,
                                        config_file,
                                        SSH_CONFIG_TYPE_SERVER_HOST))
            ssh_warning("Non-existent or malformed per-host config "
                        "file `%s' for address %s[%s]",
                        host_config->config_file, remote_host, remote_ip);
          ssh_xfree(config_file);
        }
      else
        {
          SSH_DEBUG(4, ("Address %s[%s] didn't match with `%s'.",
                        remote_host, remote_ip, host_config->pattern.pattern));
        }
    }

  c = gdata->config->user_configurations;
  
  if (uc)
    ssh_xdsprintf(&uid_str, "%lu", ssh_user_uid(uc));
  else
    uid_str = ssh_xstrdup("NO_SUCH_USER");
        
  match_ctx.user = tdata->user;
  match_ctx.uid_str = uid_str;

#ifndef WIN32
  if (uc)
    {
      groups = ssh_user_get_groups(uc);
      SSH_VERIFY(groups != NULL);
      for (i = 0; groups[i]; i++)
        ;
      num_groups = i;
      group_strs = ssh_xcalloc(i, sizeof(char *));
      gid_strs = ssh_xcalloc(i, sizeof(char *));
      SSH_DEBUG(2, ("Number of groups: %d.", i));
      for (i = 0; groups[i]; i++)
        {
          SSH_DEBUG(2, ("Adding group: %s, %d.",
                        ssh_group_get_name(groups[i]),
                        ssh_group_get_gid(groups[i])));
          group_strs[i] = ssh_xstrdup(ssh_group_get_name(groups[i]));
          ssh_xdsprintf(&gid_strs[i], "%ld", ssh_group_get_gid(groups[i]));
        }
    }
  else
    {
      SSH_TRACE(2, ("User '%s' doesn't exist, using bogus group "
                    "\"NO_SUCH_USER\".", tdata->user));
      num_groups = 1;
      group_strs = ssh_xcalloc(num_groups, sizeof(char *));
      gid_strs = ssh_xcalloc(num_groups, sizeof(char *));
      group_strs[0] = ssh_xstrdup("NO_SUCH_USER");
      gid_strs[0] = ssh_xstrdup("NO_SUCH_USER");
    }
#endif /* WIN32 */
  match_ctx.num_groups = num_groups;
  match_ctx.groups = group_strs;
  match_ctx.gid_strs = gid_strs;

  /* Check whether user name matches the pattern. */
  for (h = ssh_adt_enumerate_start(c);
       h != SSH_ADT_INVALID;
       h = ssh_adt_enumerate_next(c, h))
    {
      SshSubConfig user_config = ssh_adt_get(c, h);
      match = ssh_match_user_groups(tdata->user, uid_str,
                                    num_groups, group_strs, gid_strs,
                                    remote_host, remote_ip,
                                    &user_config->pattern, rex_ctx);

      /* If so, load the configuration file (log missing or
         non-readable config files) */
      /* XXX In principle, per-user configuration should be flushed
         if user changes. This is currently not a big problem, and
         therefore it is not yet implemented. */
      if (match)
        {
          SshUser conf_uc = ssh_user_initialize(NULL, FALSE);
          char *config_file;
          SSH_DEBUG(2, ("user `%s' matched with `%s'.",
                        tdata->user, user_config->pattern.pattern));
          SSH_DEBUG(2, ("Reading per-user config file `%s'...",
                        user_config->config_file));

          config_file = ssh_userdir_prefix(conf_uc, gdata->config,
                                           user_config->config_file,
                                           SSH_SERVER_DIR,
                                           FALSE);
          if (!ssh_config_read_file_ext(conf_uc, gdata->config, tdata->user,
                                        match_heading_cb, &match_ctx,
                                        config_file,
                                        SSH_CONFIG_TYPE_SERVER_USER))
            ssh_warning("Non-existent or malformed per-user config "
                        "file `%s' for user `%s'",
                        user_config->config_file, tdata->user);
          ssh_user_free(conf_uc, FALSE);
          ssh_xfree(config_file);
        }
      else
        {
          SSH_DEBUG(2, ("user `%s' didn't match with `%s'.",
                        tdata->user, user_config->pattern.pattern));
        }
    }

  if (uc)
    ssh_user_free(uc, FALSE);
  if (num_groups > 0)
    {
      for (i = 0; i < num_groups; i++)
        ssh_xfree(group_strs[i]);
      for (i = 0; i < num_groups; i++)
        ssh_xfree(gid_strs[i]);
      ssh_xfree(group_strs);
      ssh_xfree(gid_strs);
    }
  ssh_xfree(uid_str);

  check_conf(tdata->user, remote_host, remote_ip, gdata->config);
  
  return SSH_FSM_FINISH;
}

/**************************************************************/

void sigint_handler(int signal, void *context)
{
  SSH_ASSERT(signal == SIGINT);
  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);
  exit(0);
}

void fatal_callback(const char *msg, void *context)
{
  fprintf(stderr, "FATAL: %s\r\n", msg);
  ssh_leave_raw_mode(-1);
  ssh_leave_non_blocking(-1);
  exit(255);
}

void warning_callback(const char *msg, void *context)
{
  fprintf(stderr, "Warning: %s\r\n", msg);
}

void debug_callback(const char *msg, void *context)
{
  CheckGData gdata = (CheckGData) context;
  if (gdata->debug_mode)
    fprintf(stderr, "debug: %s\r\n", msg);
}

void sshd_check_conf_usage(const char *prog_name)
{
  ssh_informational("Usage: %s [-d debug_level] [-v] [-V] [-h] "
                    "[-f config_file] [...]\n", prog_name);
}

int main(int argc, char **argv)
{
  SshUser user;
  char *config_file = NULL;
  struct stat st;
  Boolean readline_mode = TRUE, debug_mode = FALSE;
  char *av0;
  CheckGData gdata = NULL;
  SshFSMThread cthread = NULL;

  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];

  user = ssh_user_initialize(NULL, FALSE);

  if (!user)
    ssh_fatal("Couldn't initialize current user.");

  ssh_event_loop_initialize();

  /* Register SIGCHLD signal handler. (so that SshUserFile SIGCHLDs
     are catched.) */
  ssh_sigchld_initialize();

  while (1)
    {
      int option;

      option = ssh_getopt(argc, argv, "Vvd:f:h", NULL);

      if (option == -1)
        break;

      switch (option)
        {
        case 'V':
          ssh2_version(av0);
          return 0;
        case 'v':
          ssh_debug_set_global_level(2);
          debug_mode = TRUE;
          break;
        case 'd':
          if (!ssh_optval)
            ssh_fatal("Bad -d parameter.");
          ssh_debug_set_level_string(ssh_optarg);
          debug_mode = TRUE;
          break;
        case 'f':
          if (!ssh_optval)
            ssh_fatal("Illegal -f parameter.");

          if (stat(ssh_optarg, &st) < 0)
            ssh_fatal("Alternate config file \"%s\" does not exist.",
                      ssh_optarg);

          config_file = ssh_xstrdup(ssh_optarg);
          break;
        case 'h':
          sshd_check_conf_usage(av0);
          return 0;
        default:
          if (ssh_optmissarg)
            ssh_fatal("Option -%c needs an argument\n", ssh_optopt);
          else
            ssh_fatal("Unknown option -%c\n", ssh_optopt);
          sshd_check_conf_usage(av0);
          return 1;
        }
    }

  /* If we have command line arguments, process them instead of going
     to interactive mode. */
  if (ssh_optind < argc)
    readline_mode = FALSE;

  if (config_file == NULL)
    {
      char *conf_dir = NULL;

      if (ssh_user_uid(user) == 0 )
        {
          conf_dir = ssh_xstrdup(SSH_SERVER_DIR);
        }
      else
        {
          if ((conf_dir = ssh_userdir(user, NULL, FALSE)) == NULL)
            ssh_fatal("No ssh2 user directory");
        }

      ssh_xdsprintf(&config_file, "%s/%s", conf_dir, SSH_SERVER_CONFIG_FILE);
      ssh_xfree(conf_dir);
    }

  SSH_TRACE(1, ("Main configuration file used is `%s'.", config_file));

  gdata = ssh_xcalloc(1, sizeof(*gdata));
  gdata->fsm = ssh_fsm_create(gdata);
  gdata->config_file = config_file;
  gdata->debug_mode = debug_mode;

  ssh_debug_register_callbacks(fatal_callback, warning_callback,
                               debug_callback, gdata);
  
  ssh_fsm_register_debug_names(gdata->fsm, check_states,
                               SSH_FSM_NUM_STATES(check_states));
  if (!readline_mode)
    {
      /* Start FSM, and walk through all the rest of the command
         line. */
      int i = 0;
      StepTData ctdata = NULL;

      SshADTContainer list =
        ssh_adt_create_generic(SSH_ADT_LIST,
                               SSH_ADT_DESTROY,
                               ssh_adt_callback_destroy_free,
                               SSH_ADT_ARGS_END);

      for (i = ssh_optind; i < argc ; i++)
        SSH_VERIFY(ssh_adt_insert(list, ssh_xstrdup(argv[i])) !=
                   SSH_ADT_INVALID);

      ctdata = ssh_xcalloc(1, sizeof(*ctdata));
      cthread = ssh_fsm_thread_create(gdata->fsm, check_step_list, NULL_FNPTR,
                                      step_thread_destructor, ctdata);
      ctdata->list = list;
    }
  else
    {
      ReadlineTData ctdata = NULL;

      /* Start FSM, and goto readline-mode. */
      ctdata = ssh_xcalloc(1, sizeof(*ctdata));
      cthread = ssh_fsm_thread_create(gdata->fsm, check_readline, NULL_FNPTR,
                                      readline_thread_destructor, ctdata);
      ssh_enter_non_blocking(-1);
      ssh_enter_raw_mode(-1, TRUE);
      /* XXX */
      ssh_register_signal(SIGINT, sigint_handler, NULL);
      ctdata->stdio_stream = ssh_stream_fd_stdio();
      ctdata->rl_ctx = ssh_readline_eloop_initialize(ctdata->stdio_stream);

      /* This ain't pretty. */
      {
        int fd = fileno(stderr);
        int flags = fcntl(fd, F_GETFL, 0);
#ifdef O_ASYNC
        fcntl(fd, F_SETFL, flags & ~(O_ASYNC | O_NONBLOCK));
#else /* O_ASYNC */
        fcntl(fd, F_SETFL, flags & ~(O_NONBLOCK));
#endif /* O_ASYNC */
      }
      ssh_xdsprintf(&ctdata->prompt, "%s> ", av0);

      if (!ctdata->rl_ctx)
        {
          ssh_stream_destroy(ctdata->stdio_stream);
          ssh_leave_non_blocking(-1);
          ssh_leave_raw_mode(-1);
          ssh_fatal("Couldn't initialize ReadLine.");
        }
    }

  ssh_event_loop_run();

  ssh_app_free_global_regex_context();
  ssh_event_loop_uninitialize();
  ssh_xfree(gdata);

  return 0;
}
