/*

sshconfig.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Processing configuration data in SSH (both client and server).
  
*/

#include "ssh2includes.h"
#include "sshconfig.h"
#include "sshuser.h"

#include "sshuserfile.h"

#include "sshkeyfile.h"
#include "sshuserfiles.h"
#include "sshcipherlist.h"
#include "sshsnlist.h"
#include "sshappcommon.h"
#include "sshglob.h"
#include "sshdsprintf.h"
#include "sshhostkey.h"

#include "sshbuffer.h"



#include "sshmsgs.h"
#include "sshtcp.h"
#include "sshdlex.h"
#include "sshadt_strmap.h"

#define SSH_DEBUG_MODULE "SshConfig"

void ssh_free_forward(SshForward fwd);

/* Free the "vars" and "vals" arrays */

void ssh_free_varsvals(int n, char **vars, char **vals)
{
  int i;

  for (i = 0; i < n; i++)
    {
      ssh_xfree(vars[i]);
      ssh_xfree(vals[i]);
    }
  ssh_xfree(vars);
  ssh_xfree(vals);
}

typedef void *(*SshParameterAllocator)(char *token, void *context);

/* Parses a given comma-separated list to tokens, which are stored in
   a SshADTContainer. The list is allocated and returned by this
   function. On error, returns TRUE. */
Boolean ssh_config_parse_list_check_validity(const char *string,
                                             SshADTContainer container,
                                             SshParameterAllocator allocator,
                                             void *alloc_context,
                                             SshParameterValidityProc function,
                                             void *context)
{
  char *rest;
  char *current;

  SSH_PRECOND(string != NULL);

  rest = (char *)string;

  while (strlen(rest) != 0 &&
         (current = ssh_app_param_list_get_next(rest)) != NULL)
    {
      char *temp;
      int i, j;

      rest += strlen(current);
      if (*rest == ',')
        rest++;

      /* strip whitespaces and non-printable characters */
      temp = ssh_xcalloc(strlen(current) + 1, sizeof(char));

      for (i = 0, j = 0; i < strlen(current) ; i ++)
        if(isascii(current[i]) && isprint(current[i]) && !isspace(current[i]))
          {
            temp[j] = current[i];
            j++;
          }

      temp[j] = '\0';
      ssh_xfree(current);

      /* strip escapes preceding list separator. */
      SSH_DEBUG(8, ("current: %s", temp));      
      current = ssh_xstrdup(temp);
      for (i = 0, j = 0; i < strlen(temp) ; i++,j++)
        {
          if (temp[i] == ',' && ssh_glob_isescaped(&temp[i], temp))
            current[--j] = temp[i];
          else
            current[j] = temp[i];
        }
      current[j] = '\0';

      SSH_DEBUG(8, ("stripped: %s", current));
      ssh_xfree(temp);

      /* If validity function is given, invoke it to check the current
         parameter.*/
      if (function)
        if ((*function)(current, context))
          {
            ssh_xfree(current);
            ssh_adt_clear(container);
            return TRUE;
          }

      if (!allocator)
        SSH_VERIFY(ssh_adt_insert(container, (void *)current) !=
                   SSH_ADT_INVALID);
      else
        SSH_VERIFY(ssh_adt_insert(container,
                                  (*allocator)(current, alloc_context)) !=
                   SSH_ADT_INVALID);
    }
  return FALSE;
}

/* List is not valid if NULL is returned. */
SshADTContainer
ssh_config_parse_list_check_validity_alloc(const char *string,
                                           SshParameterAllocator allocator,
                                           void *alloc_context,
                                           SshParameterValidityProc function,
                                           void *val_context)
{
  SshADTContainer container =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  SSH_VERIFY(container != NULL);
  
  if (ssh_config_parse_list_check_validity(string, container,
                                           allocator, alloc_context,
                                           function, val_context))
    ssh_adt_clear(container);

  if (ssh_adt_num_objects(container) < 1)
    {
      ssh_adt_destroy(container);
      container = NULL;
    }
  return container;
}

SshADTContainer ssh_config_parse_list_alloc(const char *string,
                                            SshParameterAllocator allocator,
                                            void *alloc_context)
{
  return ssh_config_parse_list_check_validity_alloc
    (string, allocator, alloc_context, NULL_FNPTR, NULL);
}

void ssh_config_parse_list(const char *string,
                           SshParameterAllocator allocator,
                           void *alloc_context,
                           SshADTContainer container)
{
  (void)ssh_config_parse_list_check_validity(string, container,
                                             allocator, alloc_context,
                                             NULL_FNPTR, NULL);
}

void subconfig_destroy_callback(void *obj, void *context)
{
  SshSubConfig subconfig = (SshSubConfig)obj;
  SSH_PRECOND(subconfig != NULL);
  ssh_xfree(subconfig->pattern.pattern);
  ssh_xfree(subconfig->config_file);
  ssh_xfree(subconfig);
}

struct LogFacility
{
  SshLogFacility facility;
  char *fac_name;
} logfacilities[] =
{
  {SSH_LOGFACILITY_AUTH, "AUTH"},
  {SSH_LOGFACILITY_SECURITY, "SECURITY"},
  {SSH_LOGFACILITY_DAEMON, "DAEMON"},
  {SSH_LOGFACILITY_USER, "USER"},
  {SSH_LOGFACILITY_MAIL, "MAIL"},
  {SSH_LOGFACILITY_LOCAL0, "LOCAL0"},
  {SSH_LOGFACILITY_LOCAL1, "LOCAL1"},
  {SSH_LOGFACILITY_LOCAL2, "LOCAL2"},
  {SSH_LOGFACILITY_LOCAL3, "LOCAL3"},
  {SSH_LOGFACILITY_LOCAL4, "LOCAL4"},
  {SSH_LOGFACILITY_LOCAL5, "LOCAL5"},
  {SSH_LOGFACILITY_LOCAL6, "LOCAL6"},
  {SSH_LOGFACILITY_LOCAL7, "LOCAL7"},
  {-1, NULL}
};

Boolean auth_param_validity(const char *param, void *context)
{
  if (strcasecmp(param, SSH_AUTH_PUBKEY) == 0)
    return FALSE;

  if (strcasecmp(param, SSH_AUTH_PASSWD) == 0)
    return FALSE;

#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#if defined (SSH_SERVER_WITH_KEYBOARD_INTERACTIVE) || defined(SSH_CLIENT_WITH_KEYBOARD_INTERACTIVE)
  if (strcasecmp(param, SSH_AUTH_KBD_INTERACTIVE) == 0)
    return FALSE;
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE || SSH_CLIENT_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */

#if defined (SSH_SERVER_WITH_SECURID) || defined(SSH_CLIENT_WITH_SECURID)

  if (strcasecmp(param, SSH_AUTH_SECURID) == 0)
    return FALSE;

#endif /* SSH_SERVER_WITH_SECURID || SSH_CLIENT_WITH_SECURID */


#if defined (DAEMON_WITH_PAM) || defined(CLIENT_WITH_PAM)
  if (strcasecmp(param, SSH_AUTH_PAM) == 0)
    return FALSE;
#endif /* DAEMON_WITH_PAM || CLIENT_WITH_PAM */

  if (strcasecmp(param, SSH_AUTH_HOSTBASED) == 0)
    return FALSE;

#ifdef KERBEROS
  if (strcasecmp(param, SSH_AUTH_KERBEROS) == 0)
    return FALSE;

  if (strcasecmp(param, SSH_AUTH_KERBEROS_TGT) == 0)
    return FALSE;
#endif /* KERBEROS */



  return TRUE;
}

SshPatternHolder ssh_config_pattern_alloc(char *token,
                                          SshMetaConfig metaconfig)
{
  SshPatternHolder holder;
  
  SSH_PRECOND(token != NULL);
  SSH_PRECOND(metaconfig != NULL);

  holder = ssh_xcalloc(1, sizeof(*holder));

  holder->pattern = token;
  holder->regex_syntax = metaconfig->regex_syntax;

  return holder;
}

void ssh_config_pattern_destructor(SshPatternHolder holder, void *context)
{
  SSH_PRECOND(holder != NULL);

  ssh_xfree(holder->pattern);
  ssh_xfree(holder);
}

void config_pattern_destructor(void *obj, void *context)
{
  SshPatternHolder holder = (SshPatternHolder) obj;
  ssh_config_pattern_destructor(holder, context);
}

void *config_pattern_alloc(char *token,
                           void *context)
{
  SshMetaConfig metaconfig = (SshMetaConfig) context;
  return (void *)ssh_config_pattern_alloc(token, metaconfig);
}

char *ssh_config_remove_whitespace(const char *str)
{
  char *new_str;
  int i = 0, j = 0, len = 0;

  len = strlen(str);
  
  new_str = ssh_xcalloc(len + 1, sizeof(char));
  for (i = 0; i < len ; i++)
    {
      if (!isspace(str[i]))
        {
          new_str[j] = str[i];
          j++;
        }
    }
  return new_str;
}

/* Return FALSE if success, TRUE otherwise. */
typedef Boolean (*SshConfigSetParamCB)(/* The function should not
                                          normally directly modify
                                          ``config'' (although there are
                                          exceptions...), it is for
                                          getting values of other
                                          configuration variables
                                          only.*/
                                       SshConfig config,
                                       SshMetaConfig metaconfig,
                                       /* For debugging purposes, mainly. */
                                       const char *var,
                                       /* The value as a string...*/
                                       const char *val_str,
                                       /* .. an integer...*/
                                       int val_num,
                                       /* ... and a Boolean. */
                                       Boolean val_bool,
                                       /* Pointer to the variable, whose
                                          actual type the function must
                                          have. */
                                       void *varp);

typedef struct SshConfigSetParamRec {
  char *variable;   /* Normal case (i.e. all not lowercase). */
  void *var_to_set; /* Pointer to approriate field in the SshConfig struct. */
  int type_mask;    /* In what cases is this variable usable? */
  SshConfigSetParamCB callback;
} SshConfigSetParamStruct, *SshConfigSetParam;

typedef struct SshConfigSetParamAliasRec {
  char *alias;
  char *variable;
} SshConfigSetParamAliasStruct, *SshConfigSetParamAlias;

#define SET_PARAM_CB(param)                                                  \
Boolean ssh_config_set_param_ ## param(SshConfig config,                     \
                                       SshMetaConfig metaconfig,             \
                                       const char *var, const char *val_str, \
                                       int val_num, Boolean val_bool,        \
                                       void *varp)

#define ADD_PARAM_REC(var_name, varp, mask, param)              \
do                                                              \
{                                                               \
  SshConfigSetParam item = ssh_xcalloc(1, sizeof(*item));       \
                                                                \
  item->variable = ssh_xstrdup(var_name);                       \
  item->var_to_set = (varp);                                    \
  item->type_mask = mask;                                       \
  item->callback = ssh_config_set_param_ ## param;              \
  ssh_adt_strmap_set(config->set_param, var_name, item);        \
}                                                               \
while(0)

#define ADD_PARAM_ALIAS(alias_name, var_name)                           \
do                                                                      \
{                                                                       \
  SshConfigSetParamAlias alias = ssh_xcalloc(1, sizeof(*alias));        \
                                                                        \
  alias->alias = ssh_xstrdup(alias_name);                               \
  alias->variable = ssh_xstrdup(var_name);                              \
  ssh_adt_strmap_set(config->set_param_aliases, alias_name, alias);     \
}                                                                       \
while(0)

SET_PARAM_CB(string)
{
  char **variable_p = (char **)varp;
  SSH_PRECOND(variable_p != NULL);
  SSH_DEBUG(4, ("Setting variable '%s' to '%s'.", var, val_str));
  ssh_xfree(*variable_p);
  *variable_p = ssh_xstrdup(val_str);
  return FALSE;
}

SET_PARAM_CB(bool)
{
  Boolean *variable_p = (Boolean *)varp;
  SSH_PRECOND(variable_p != NULL);
  SSH_DEBUG(4, ("Setting variable '%s' to '%s'.", var,
                val_bool ? "TRUE" : "FALSE"));
  *variable_p = val_bool;
  return FALSE;
}

SET_PARAM_CB(bool_inverse)
{
  Boolean *variable_p = (Boolean *)varp;  
  SSH_PRECOND(variable_p != NULL);
  SSH_DEBUG(4, ("Setting variable '%s' to '%s'.", var,
                val_bool ? "TRUE" : "FALSE"));
  *variable_p = val_bool ? FALSE : TRUE;
  return FALSE;
}

SET_PARAM_CB(authentications)
{
  SshADTContainer c = *((SshADTContainer *)varp);
  ssh_adt_clear(c);
  if (ssh_config_parse_list_check_validity(val_str, c, NULL_FNPTR, NULL,
                                           auth_param_validity, NULL))
    {
      ssh_warning("Parsing of value for %s failed.", var);
      return TRUE;
    }
  else
    {
      return FALSE;
    }
}

SET_PARAM_CB(long_int)
{
  SshUInt32 *value = (SshUInt32 *)varp;
  *value = strtoul(val_str, NULL, 0);
  return FALSE;
}

SET_PARAM_CB(num)
{
  int *value = (int *)varp;
  if (val_num < 0)
    {
      ssh_warning("Ignoring illegal value %d for %s", val_num, var);
      return TRUE;
    }
  *value = val_num;
  return FALSE;
}

SET_PARAM_CB(basic_list_insert)
{
  SshADTContainer c = *((SshADTContainer *)varp);
  SSH_VERIFY(ssh_adt_insert(c, ssh_xstrdup(val_str)) != SSH_ADT_INVALID);
  return FALSE;
}

SET_PARAM_CB(simple_list_parse)
{
  SshADTContainer c = *((SshADTContainer *)varp);
  ssh_adt_clear(c);
  ssh_config_parse_list(val_str, NULL_FNPTR, NULL, c);
  return FALSE;
}

SET_PARAM_CB(pattern_list_parse)
{
  SshADTContainer *c = (SshADTContainer *)varp;
  if (*c == NULL)
    {
      *c = ssh_adt_create_generic(SSH_ADT_LIST,
                                  SSH_ADT_DESTROY,
                                  config_pattern_destructor,
                                  SSH_ADT_ARGS_END);
      SSH_VERIFY(*c != NULL);
    }
  ssh_config_parse_list(val_str, config_pattern_alloc,
                        metaconfig, *c);
  return FALSE;
}

SET_PARAM_CB(log_facility)
{
  SshLogFacility *log_fac = (SshLogFacility *)varp;
  int i;
  
  for (i = 0; logfacilities[i].fac_name != NULL; i++)
    {
      if (strcasecmp(logfacilities[i].fac_name, val_str) == 0)
        {
          *log_fac = logfacilities[i].facility;
          return FALSE;
        }
    }
  ssh_warning("Unknown %s \"%s\".", var, val_str);
  return TRUE;
}

SET_PARAM_CB(port)
{
  char **port = (char **)varp;
  if (val_num >= 1 && val_num < 65536)
    {
      ssh_xfree(*port);
      *port = ssh_xstrdup(val_str);
    }
  else
    {
      ssh_warning("Ignoring illegal port number %s", val_str);
      return TRUE;
    }
  return FALSE;
}

SET_PARAM_CB(compression_level)
{
  int *level = (int *)varp;
  if (val_num > 9 || val_num < 1)
    {
      ssh_warning("Ignoring illegal value %d for %s", val_num, var);
      return TRUE;
    }
  *level = val_num;
  return FALSE;
}

SET_PARAM_CB(algs)
{
  char **list = (char **)varp;
  char *stdlist, *native, *new_val;
  Boolean cipher = FALSE;
  SSH_DEBUG(5, ("Got config alglist \"%s\" for %s", val_str, var));
  ssh_xfree(*list);
  new_val = ssh_config_remove_whitespace(val_str);

  if (strcasecmp(var, "ciphers") == 0)
    cipher = TRUE;

  if (cipher)
    {
      native = ssh_cipher_get_supported_native();
      stdlist = SSH_STD_CIPHERS;
    }
  else
    {
      native = ssh_mac_get_supported();
      stdlist = SSH_STD_MACS;
    }
  
  if (strcasecmp(new_val, "any") == 0 ||
      ((cipher && (strcasecmp(new_val, "anycipher") == 0)) ||
       (!cipher && (strcasecmp(new_val, "anymac") == 0))))
    {
      char *hlp1, *hlp2;

      hlp1 = ssh_xstrdup(native);
      if ((cipher && (strcasecmp(new_val, "anycipher") == 0)) ||
          (!cipher && (strcasecmp(new_val, "anymac") == 0)))
        {
          hlp2 = hlp1;
          hlp1 = ssh_cipher_list_exclude(hlp2, "none");
          ssh_xfree(hlp2);
        }
      *list = ssh_snlist_intersection(stdlist, hlp1);
      hlp2 = ssh_cipher_list_exclude(*list, "none");
      ssh_xfree(*list);
      ssh_xdsprintf(list, "%s,%s", hlp2, hlp1);
      ssh_xfree(hlp1);
      ssh_xfree(hlp2);
      if (cipher)
        hlp1 = ssh_cipher_list_canonicalize(*list);
      else
        hlp1 = ssh_hash_list_canonicalize(*list);
      ssh_xfree(*list);
      *list = hlp1;
    }
  else if (strcasecmp(new_val, "anystd") == 0 ||
           ((cipher && (strcasecmp(new_val, "anystdcipher") == 0)) ||
            (!cipher && (strcasecmp(new_val, "anystdmac") == 0))))
    {
      char *hlp = ssh_xstrdup(native);
      *list = ssh_snlist_intersection(stdlist, hlp);
      ssh_xfree(hlp);
      if ((cipher && (strcasecmp(new_val, "anystdcipher") == 0)) ||
          (!cipher && (strcasecmp(new_val, "anystdmac") == 0)))
        {
          hlp = *list;
          *list = ssh_cipher_list_exclude(hlp, "none");
          ssh_xfree(hlp);
        }
    }
  else
    {
      if (cipher)
        *list = ssh_cipher_list_canonicalize(new_val);
      else
        *list = ssh_hash_list_canonicalize(new_val);
    }
  ssh_xfree(native);
  ssh_xfree(new_val);
  SSH_DEBUG(5, ("Final alglist \"%s\"", *list));
  return FALSE;
}

SET_PARAM_CB(timeout)
{
  SshUInt32 *num = (SshUInt32 *)varp;
  SshInt32 timeout_value = ssh_config_parse_timeout(val_str);

  if (timeout_value < 0)
    {
      ssh_warning("Illegal %s value '%s'.", var, val_str);
      return TRUE;
    }
          
  *num = timeout_value;
  SSH_DEBUG(5, ("%s is set to %qd.", var, *num));
  return FALSE;
}

SET_PARAM_CB(verbose)
{
  Boolean *variable_p = (Boolean *)varp;
  SSH_PRECOND(variable_p != NULL);
  SSH_DEBUG(2, ("Setting variable '%s' to '%s'.", var,
                val_bool ? "TRUE" : "FALSE"));
  *variable_p = val_bool;
  if (val_bool)
    ssh_debug_set_level_string("2");
  else
    ssh_debug_clear_modules();
  
  return FALSE;
}

SET_PARAM_CB(agent_compat)
{
  SshAgentSsh1CompatMode *compat_mode = (SshAgentSsh1CompatMode *)varp;
  Boolean error = FALSE;
  
  if (strcasecmp(val_str, "none") == 0)
    *compat_mode = SSH_AGENT_COMPAT_NONE;
  else if (strcasecmp(val_str, "traditional") == 0)
    *compat_mode = SSH_AGENT_COMPAT_TRADITIONAL;
  else if (strcasecmp(val_str, "ssh2") == 0)
    *compat_mode = SSH_AGENT_COMPAT_SSH2;
  else
    error = TRUE;

  if (!error)
    return FALSE;
  
  ssh_warning("Bad %s definition \"%s\"", var, val_str);
  return TRUE;
}

SET_PARAM_CB(strict_hk_check)
{
  SshStrictHostKeyChecking *mode = (SshStrictHostKeyChecking *)varp;

  if (strcasecmp(val_str, "ask") == 0)
    *mode = SSH_STRICT_HOSTKEY_CHECKING_ASK;
  else if (val_bool == TRUE)
    *mode = SSH_STRICT_HOSTKEY_CHECKING_YES;
  else
    *mode = SSH_STRICT_HOSTKEY_CHECKING_NO;

  return FALSE;
}

SET_PARAM_CB(escape_char)
{
  char **echar = (char **)varp;
  ssh_xfree(*echar);
  if (strcasecmp(val_str, "none") == 0)
    *echar = ssh_xstrdup("");
  else
    *echar = ssh_xstrdup(val_str);
  return FALSE;
}

SET_PARAM_CB(go_background)
{
  Boolean *gobg = (Boolean *)varp;
  
  if (strcasecmp(val_str, "oneshot") == 0)
    {
      *gobg = TRUE;
      config->one_shot_forwarding = TRUE;
      return FALSE;
    }
  *gobg = val_bool;
  return FALSE;
}

SET_PARAM_CB(permit_root_login)
{
  SshPermitRootLogin *permit = (SshPermitRootLogin *)varp;
  if (strcasecmp(val_str, "nopwd") == 0)
    *permit = SSH_ROOTLOGIN_NOPWD;
  else if (val_bool == TRUE)
    *permit = SSH_ROOTLOGIN_TRUE;
  else
    *permit = SSH_ROOTLOGIN_FALSE;

  return FALSE;  
}

SET_PARAM_CB(listen_address)
{
  char **listen_address = (char **)varp;
  ssh_xfree(*listen_address);
  if (strcasecmp(val_str, "any") == 0)
    *listen_address = ssh_xstrdup(SSH_IPADDR_ANY);
  else
    *listen_address = ssh_xstrdup(val_str);          
  return FALSE;
}

SET_PARAM_CB(max_connections)
{
  int *max_connections = (int *)varp;

  if (val_num < 0)
    {
      ssh_warning("Ignoring illegal value %d for %s", val_num, var);
      return TRUE;
    }
  
#ifndef SSH2_MAX_CONNECTIONS
  *max_connections = val_num;
#else /* ! SSH2_MAX_CONNECTIONS */
#if SSH2_MAX_CONNECTIONS == 0
  *max_connections = val_num;
#else /* SSH2_MAX_CONNECTIONS == 0 */
  if ((val_num > SSH2_MAX_CONNECTIONS) || (val_num <= 0))
    {
      if (val_num > 0)
        ssh_warning("Maximum of %d connections requested while "
                    "only %d allowed.",
                    *max_connections, SSH2_MAX_CONNECTIONS);
      else
        ssh_warning("Unlimited connections requested while only "
                    "%d allowed.",
                    SSH2_MAX_CONNECTIONS);
      *max_connections = SSH2_MAX_CONNECTIONS;
    }
  else
    {
      *max_connections = val_num;
    }
#endif /* SSH2_MAX_CONNECTIONS == 0 */
#endif /* ! SSH2_MAX_CONNECTIONS */















  return FALSE;
}

SET_PARAM_CB(subconfig)
{
  SshADTContainer *subconfig_list = (SshADTContainer *)varp;
  SshRegexMatcher rex =
    ssh_regex_create(ssh_app_get_global_regex_context(),
                     "^([^ \t]+)[ \t]+(.+)$", SSH_REGEX_SYNTAX_EGREP);
  SshSubConfig subconfig;
  Boolean match;
  SshADTHandle h;
  SSH_VERIFY(rex != NULL);
  SSH_PRECOND(subconfig_list != NULL);
  
  match = ssh_regex_match_cstr(rex, val_str);
  if (match)
    {
      subconfig = ssh_xcalloc(1, sizeof(*subconfig));
      subconfig->pattern.pattern =
        ssh_xstrdup(ssh_regex_get_submatch(rex, 1));
      subconfig->pattern.regex_syntax = metaconfig->regex_syntax;
      subconfig->config_file = ssh_xstrdup(ssh_regex_get_submatch(rex, 2));
      h = ssh_adt_insert_to(*subconfig_list, SSH_ADT_END, subconfig);
      SSH_VERIFY(h != SSH_ADT_INVALID);
      SSH_DEBUG(2, ("Got pattern `%s', file '%s' for subconfig %s.",
                    subconfig->pattern.pattern, subconfig->config_file,
                    var));
    }
  
  ssh_regex_free(rex);
  return !match;
}

SET_PARAM_CB(local_forward)
{
  SshForward *fw = (SshForward *)varp;
#ifdef SSHDIST_SSH2_SOCKS_FILTER
  SshRegexMatcher rex =
    ssh_regex_create(ssh_app_get_global_regex_context(),
                     "^socks~/{{~[([-~]]+)~]|([-~:]+)}:}?([[:digit:]]+)$",
                     SSH_REGEX_SYNTAX_SSH);
  SSH_ASSERT(rex != NULL);
  if (ssh_regex_match_cstr(rex, val_str))
    {
      SshForward new_fw = ssh_xcalloc(1, sizeof (*new_fw));
      const char *la;
      la = ssh_regex_get_submatch(rex, 1);
      if (la == NULL || strlen(la) == 0)
        la = ssh_regex_get_submatch(rex, 2);
      if (la == NULL || strlen(la) == 0)
        la = SSH_IPADDR_ANY;
      new_fw->local_addr = ssh_xstrdup(la);
      SSH_DEBUG(2, ("socks forwarding local address `%s'.", la));
      new_fw->port = ssh_xstrdup(ssh_regex_get_submatch(rex, 3));
      new_fw->connect_to_host = ssh_xstrdup("dynamic");
      new_fw->connect_to_port = ssh_xstrdup("dynamic");
      new_fw->protocol = ssh_xstrdup("socks");
      new_fw->next = *fw;
      *fw = new_fw;
      ssh_regex_free(rex);
      return FALSE;
    }
  ssh_regex_free(rex);
#endif /* SSHDIST_SSH2_SOCKS_FILTER */
  if (ssh_parse_forward(fw, val_str))
    {
      ssh_warning("Bad %s definition \"%s\"", var, val_str);
      return TRUE;
    }
  return FALSE;
}

SET_PARAM_CB(basic_forward)
{
  SshForward *fw = (SshForward *)varp;
  if (ssh_parse_forward(fw, val_str))
    {
      ssh_warning("Bad %s definition \"%s\"", var, val_str);
      return TRUE;
    }
  return FALSE;
}

SET_PARAM_CB(forward_acl)
{
  SshRegexMatcher r;
  SshADTContainer *c = (SshADTContainer *)varp;
  
  r = ssh_regex_create(ssh_app_get_global_regex_context(),
                       "([aA][lL][lL][oO][wW]|[dD][eE][nN][yY])~s+"
                       "([lL][oO][cC][aA][lL]|[rR][eE][mM][oO][tT][eE])~s+"
                       "(~S+)~s+(~S+){(~S+(~s+)|.*)}",
                       SSH_REGEX_SYNTAX_SSH);
  SSH_VERIFY(r != NULL);

  if (ssh_regex_match_cstr(r, val_str))
    {
      SshForwardACL forw_acl = ssh_xcalloc(1, sizeof(*forw_acl));
      char *str;
      
      str = ssh_regex_get_submatch(r, 1);
      SSH_ASSERT(str != NULL);
      if (strcasecmp(str, "allow") == 0)
        forw_acl->allow = TRUE;
      str = ssh_regex_get_submatch(r, 2);
      SSH_ASSERT(str != NULL);
      if (strcasecmp(str, "local") == 0)
        forw_acl->local = TRUE;
      str = ssh_regex_get_submatch(r, 3);
      SSH_ASSERT(str != NULL);
      forw_acl->user_pattern.regex_syntax = metaconfig->regex_syntax;
      forw_acl->user_pattern.pattern = ssh_xstrdup(str);
      str = ssh_regex_get_submatch(r, 4);
      SSH_ASSERT(str != NULL);
      forw_acl->forward_pattern.regex_syntax = metaconfig->regex_syntax;
      forw_acl->forward_pattern.pattern = ssh_xstrdup(str);
      str = ssh_regex_get_submatch(r, 5);
      if (str && strlen(str) > 0)
        {
          forw_acl->originator_pattern.regex_syntax = metaconfig->regex_syntax;
          forw_acl->originator_pattern.pattern = ssh_xstrdup(str);
        }
      SSH_ASSERT(*c != NULL);
      SSH_VERIFY(ssh_adt_insert(*c, forw_acl) != SSH_ADT_INVALID);
      ssh_regex_free(r);
      return FALSE;
    }

  ssh_regex_free(r);
  ssh_warning("Bad %s definition \"%s\"", var, val_str);
  return TRUE;
}

/* This should be one of the few destructive param cbs. */
SET_PARAM_CB(clear_forwards)
{
  SshForward current_forward;

  if (val_bool == TRUE)
    {
      while (config->remote_forwards != NULL)
        {
          current_forward = config->remote_forwards;
          config->remote_forwards = config->remote_forwards->next;
          ssh_free_forward(current_forward);
        }

      while (config->local_forwards != NULL)
        {
          current_forward = config->local_forwards;
          config->local_forwards = config->local_forwards->next;
          ssh_free_forward(current_forward);
        }
    }
  return FALSE;
}

SET_PARAM_CB(add_priv_key)
{
  SshHostKeysContext *ctx = (SshHostKeysContext *)varp;
  ssh_host_key_add_private_key(val_str, *ctx);
  return FALSE;
}

SET_PARAM_CB(add_pub_key)
{
  SshHostKeysContext *ctx = (SshHostKeysContext *)varp;
  ssh_host_key_add_public_key(val_str, *ctx);
  return FALSE;
}


























































































































































































































































































































































































































SET_PARAM_CB(ignore)
{
  SSH_DEBUG(2, ("Silently ignoring %s.", var));
  return FALSE;
}

SET_PARAM_CB(deprecated)
{
  ssh_warning("Configuration option %s is deprecated.", var);
  return FALSE;
}

void detach_set_param_rec(void *value, void *context)
{
  SshConfigSetParam item = (SshConfigSetParam) value;
  ssh_xfree(item->variable);
  ssh_xfree(value);
}

void detach_alias_rec(void *value, void *context)
{
  SshConfigSetParamAlias alias = (SshConfigSetParamAlias) value;
  ssh_xfree(alias->alias);
  ssh_xfree(alias->variable);
  ssh_xfree(alias);
}

void forward_acl_destroy_cb(void *obj, void *context)
{
  SshForwardACL forw_acl = (SshForwardACL)obj;
  ssh_xfree(forw_acl->user_pattern.pattern);
  ssh_xfree(forw_acl->forward_pattern.pattern);
  ssh_xfree(forw_acl->originator_pattern.pattern);
  ssh_xfree(forw_acl);
}

int compare_param_str_cb(const void *obj1, const void *obj2, void *ctx)
{
  const char *s1 = obj1, *s2 = obj2;
  return strcasecmp(s1, s2);
}

SshUInt32 hash_param_str_cb(const void *obj, void *ctx)
{
  SshUInt32 hash = 0;
  const char *c1 = obj;
  char *c2 = ssh_xcalloc(strlen(c1) + 1 , sizeof(char)), *delet;
  int i;
  for (i = 0 ; c1[i]; i++)
    c2[i] = tolower(c1[i]);

  delet = c2;
  
  while (*c2)
    {
      hash = (hash << 17) + (hash >> 15) + *c2;
      c2++;
    }
  ssh_xfree(delet);
  return hash;
}

/* Allocates and initializes a config structure */
SshConfig ssh_config_init(Boolean client)
{
  SshConfig config;



  config = ssh_xcalloc(1, sizeof(*config));
  config->client = client;

  config->set_param = ssh_adt_create_generic
    (SSH_ADT_MAP,
     SSH_ADT_MAP_DETACH,   detach_set_param_rec,
     SSH_ADT_HASH,         hash_param_str_cb,
     SSH_ADT_DUPLICATE,    ssh_adt_callback_duplicate_str,
     SSH_ADT_DESTROY,      ssh_adt_callback_destroy_free,
     SSH_ADT_COMPARE,      compare_param_str_cb,
     SSH_ADT_ARGS_END);
  SSH_VERIFY(config->set_param != NULL);
  config->set_param_aliases = ssh_adt_create_generic
    (SSH_ADT_MAP,
     SSH_ADT_MAP_DETACH,   detach_alias_rec,
     SSH_ADT_HASH,         hash_param_str_cb,
     SSH_ADT_DUPLICATE,    ssh_adt_callback_duplicate_str,
     SSH_ADT_DESTROY,      ssh_adt_callback_destroy_free,
     SSH_ADT_COMPARE,      compare_param_str_cb,
     SSH_ADT_ARGS_END);
  SSH_VERIFY(config->set_param_aliases != NULL);

  /* Configuration option that does nothing. */
  ADD_PARAM_REC("NoOp", NULL, SSH_CONFIG_TYPE_SERVER_ALL, ignore);
  ADD_PARAM_REC("NoOpClient", NULL, SSH_CONFIG_TYPE_CLIENT, ignore);
  
  /*
   * Configuration variables that are valid everywhere.
   */
  /* This behaves slightly differently depending on where it is used. In
     global config, will make the daemon stay on the terminal and not go
     to the background, and it will only give one connection (and then
     destroys the listener). If set on the other files, just starts
     spewing debug messages, probably to the user too. */
  ADD_PARAM_REC("VerboseMode", &config->verbose_mode,
                SSH_CONFIG_TYPE_ALL, verbose);
  ADD_PARAM_REC("UserConfigDirectory", &config->user_conf_dir,
                SSH_CONFIG_TYPE_ALL, string);

  ADD_PARAM_REC("PGPPublicKeyFile", &config->pgp_public_key_file,
                /* XXX this feature is not documented. */
                SSH_CONFIG_TYPE_ALL, string);
  ADD_PARAM_REC("PGPSecretKeyFile", &config->pgp_secret_key_file,
                /* XXX this feature is not documented. */
                SSH_CONFIG_TYPE_ALL, string);

  ADD_PARAM_REC("AllowedAuthentications", &config->allowed_authentications,
                SSH_CONFIG_TYPE_ALL, authentications);
  ADD_PARAM_REC("QuietMode", &config->quiet_mode,
                SSH_CONFIG_TYPE_ALL, bool);
  /* XXX not yet implemented */
  ADD_PARAM_REC("RekeyIntervalBytes", &config->rekey_interval_bytes,
                SSH_CONFIG_TYPE_ALL, long_int);
  ADD_PARAM_REC("RekeyIntervalSeconds", &config->rekey_interval_seconds,
                SSH_CONFIG_TYPE_ALL, long_int);
  ADD_PARAM_REC("ForwardX11", &config->x11_forwarding,
                SSH_CONFIG_TYPE_ALL, bool);
  ADD_PARAM_REC("ForwardAgent", &config->agent_forwarding,
                SSH_CONFIG_TYPE_ALL, bool);

  /*
   * Configuration variables that are valid for the client and the
   * server (at least in the global config)..
   */
  ADD_PARAM_REC("Ciphers", &config->ciphers,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, algs);
  ADD_PARAM_REC("MACs", &config->macs,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, algs);

  ADD_PARAM_REC("Ssh1Compatibility", &config->ssh1compatibility,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, bool);

  ADD_PARAM_REC("RandomSeedFile", &config->random_seed_file,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                string);
  ADD_PARAM_REC("Port", &config->port,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                port);
  ADD_PARAM_REC("KeepAlive", &config->keep_alive,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                bool);
  ADD_PARAM_REC("NoDelay", &config->no_delay,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                bool);
  ADD_PARAM_REC("XauthPath", &config->xauth_path,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                string);
  /* Client uses socks for connections.  Server uses socks for LDAP
     CRL queries. XXX document to sshd2_config.5 and remove this
     note. */
  ADD_PARAM_REC("SocksServer", &config->socks_server,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                string);
  ADD_PARAM_REC("UseSOCKS5", &config->use_socks5,
                SSH_CONFIG_TYPE_CLIENT | SSH_CONFIG_TYPE_SERVER_GLOBAL,
                bool);

  /*
   * Configuration variables only for the server. More generic variables
   * are closer to the top. (Windows-server specific are at the bottom,
   * as are other more special distribution specific
   * variables. Reasoning is that even though the variables could be put
   * to any of the configuration files, they are not really generic in
   * the sense I use here.)
   */
  /* Server configuration variables valid everywhere. */

#define ADD_PARAM_PATTERN_LIST(param, listp, add_flag)                        \
  ADD_PARAM_REC((param), (listp),                                             \
                SSH_CONFIG_TYPE_SERVER_GLOBAL | SSH_CONFIG_TYPE_SERVER_HOST | \
                (add_flag), pattern_list_parse)

  ADD_PARAM_REC("AuthorizationFile", &config->authorization_file,
                SSH_CONFIG_TYPE_SERVER_ALL, string);
  ADD_PARAM_REC("FascistLogging", &config->fascist_logging,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("RequiredAuthentications",
                &config->required_authentications,
                SSH_CONFIG_TYPE_SERVER_ALL, authentications);
  ADD_PARAM_REC("SysLogFacility", &config->log_facility,
                SSH_CONFIG_TYPE_SERVER_ALL, log_facility);
  ADD_PARAM_REC("SftpSysLogFacility", &config->sftp_server_log_facility,
                SSH_CONFIG_TYPE_SERVER_ALL, log_facility);
  ADD_PARAM_REC("UserKnownHosts", &config->user_known_hosts,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("IgnoreRhosts", &config->ignore_rhosts,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("IgnoreRootRhosts", &config->ignore_root_rhosts,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("PermitEmptyPasswords", &config->permit_empty_passwords,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("StrictModes", &config->strict_modes,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("PrintMOTD", &config->print_motd,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("CheckMail", &config->check_mail,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("AllowTcpForwarding", &config->allow_tcp_forwarding,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("HostbasedAuthForceClientHostnameDNSMatch",
                &config->hostbased_force_client_hostname_dns_match,
                SSH_CONFIG_TYPE_SERVER_ALL, bool);
  ADD_PARAM_REC("PasswdPath", &config->passwd_path,
                SSH_CONFIG_TYPE_SERVER_ALL, string);
  ADD_PARAM_REC("PasswordGuesses", &config->password_guesses,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
  ADD_PARAM_REC("AuthInteractiveFailureTimeout",
                &config->auth_interactive_failure_timeout,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
  ADD_PARAM_REC("IdleTimeout", &config->idle_timeout,
                SSH_CONFIG_TYPE_SERVER_ALL, timeout);
  ADD_PARAM_REC("AuthPublicKey.MinSize", &config->auth_publickey_min_size,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
  ADD_PARAM_REC("AuthPublicKey.MaxSize", &config->auth_publickey_max_size,
                SSH_CONFIG_TYPE_SERVER_ALL, num);








  ADD_PARAM_ALIAS("AuthPublicKey.Cert.MinSize", "NoOp");
  ADD_PARAM_ALIAS("AuthPublicKey.Cert.MaxSize", "NoOp");

#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
  ADD_PARAM_REC("AuthKbdInt.Retries", &config->auth_kbd_int_retries,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
  ADD_PARAM_REC("AuthKbdInt.Required", &config->auth_kbd_int_required,
                SSH_CONFIG_TYPE_SERVER_ALL, simple_list_parse);
  ADD_PARAM_REC("AuthKbdInt.Optional", &config->auth_kbd_int_optional,
                SSH_CONFIG_TYPE_SERVER_ALL, simple_list_parse);
  ADD_PARAM_REC("AuthKbdInt.NumOptional",
                &config->auth_kbd_int_optional_needed,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
  ADD_PARAM_REC("AuthKbdInt.Plugin",
                &config->auth_kbd_int_plugin_prog,
                SSH_CONFIG_TYPE_SERVER_ALL, string);
#else /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
  ADD_PARAM_ALIAS("AuthKbdInt.Retries", "NoOp");
  ADD_PARAM_ALIAS("AuthKbdInt.Required", "NoOp");
  ADD_PARAM_ALIAS("AuthKbdInt.Optional", "NoOp");
  ADD_PARAM_ALIAS("AuthKbdInt.NumOptional", "NoOp");
  ADD_PARAM_ALIAS("AuthKbdInt.Plugin", "NoOp");
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */
#ifdef SSH_SERVER_WITH_SECURID
  ADD_PARAM_REC("SecurIdGuesses", &config->securid_guesses,
                SSH_CONFIG_TYPE_SERVER_ALL, num);
#else /* SSH_SERVER_WITH_SECURID */
  ADD_PARAM_ALIAS("SecurIdGuesses", "NoOp");
#endif /* SSH_SERVER_WITH_SECURID */
  ADD_PARAM_PATTERN_LIST("AllowShosts", &config->allowed_shosts,
                         SSH_CONFIG_TYPE_SERVER_USER);
  ADD_PARAM_PATTERN_LIST("DenyShosts", &config->denied_shosts,
                         SSH_CONFIG_TYPE_SERVER_USER);
  ADD_PARAM_PATTERN_LIST("SettableEnvironmentVars", &config->settable_env_vars,
                         SSH_CONFIG_TYPE_SERVER_USER);
  /* Server configuration variables valid in the global file and the
     host specific config. */
  ADD_PARAM_PATTERN_LIST("AllowUsers", &config->allowed_users, 0);
  ADD_PARAM_PATTERN_LIST("DenyUsers", &config->denied_users, 0);
  ADD_PARAM_PATTERN_LIST("AllowGroups", &config->allowed_groups, 0);
  ADD_PARAM_PATTERN_LIST("DenyGroups", &config->denied_groups, 0);
  ADD_PARAM_PATTERN_LIST("AllowTcpForwardingForUsers",
                         &config->allowed_tcp_forwarding_users, 0);
  ADD_PARAM_PATTERN_LIST("DenyTcpForwardingForUsers",
                         &config->denied_tcp_forwarding_users, 0);
  ADD_PARAM_PATTERN_LIST("AllowTcpForwardingForGroups",
                         &config->allowed_tcp_forwarding_groups, 0);
  ADD_PARAM_PATTERN_LIST("DenyTcpForwardingForGroups",
                         &config->denied_tcp_forwarding_groups, 0);
  ADD_PARAM_REC("ExternalAuthorizationProgram",
                &config->external_authorization_prog,
                SSH_CONFIG_TYPE_SERVER_GLOBAL | SSH_CONFIG_TYPE_SERVER_HOST,
                string);
  ADD_PARAM_REC("ForwardACL", &config->forward_acls,
                SSH_CONFIG_TYPE_SERVER_GLOBAL | SSH_CONFIG_TYPE_SERVER_HOST,
                forward_acl);
  ADD_PARAM_PATTERN_LIST("ChrootUsers", &config->chroot_users, 0);
  ADD_PARAM_PATTERN_LIST("ChrootGroups", &config->chroot_groups, 0);
  ADD_PARAM_REC("Sshd1Path", &config->ssh1_path,
                SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, string);
  ADD_PARAM_REC("Sshd1ConfigFile", &config->sshd1_config_file_path,
                SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, string);
  ADD_PARAM_REC("BannerMessageFile", &config->banner_message_file_path,
                SSH_CONFIG_TYPE_SERVER_GLOBAL |
                SSH_CONFIG_TYPE_SERVER_HOST, string);
  ADD_PARAM_REC("ProtocolVersionString", &config->protocol_version_string,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, string);
  ADD_PARAM_REC("LoginGraceTime", &config->login_grace_time,
                SSH_CONFIG_TYPE_SERVER_GLOBAL | SSH_CONFIG_TYPE_SERVER_HOST,
                num);
  ADD_PARAM_REC("PermitRootLogin", &config->permit_root_login,
                SSH_CONFIG_TYPE_SERVER_GLOBAL | SSH_CONFIG_TYPE_SERVER_HOST,
                permit_root_login);
  /* Server configuration variables valid only in the global file. */
  ADD_PARAM_REC("MaxBroadcastsPerSecond",
                &config->broadcasts_allowed_per_second,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, num);
  ADD_PARAM_REC("MaxConnections", &config->max_connections,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, max_connections);
  ADD_PARAM_REC("RequireReverseMapping", &config->require_reverse_mapping,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, bool);
  ADD_PARAM_REC("ResolveClientHostName", &config->try_reverse_mapping,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, bool);
  ADD_PARAM_REC("ListenAddress", &config->listen_address,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, listen_address);
  ADD_PARAM_REC("HostKeyFile", &config->host_keys_ctx,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, add_priv_key);
  ADD_PARAM_REC("PublicHostKeyFile", &config->host_keys_ctx,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, add_pub_key);
  ADD_PARAM_REC("UserSpecificConfig", &config->user_configurations,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, subconfig);
  ADD_PARAM_REC("HostSpecificConfig", &config->host_configurations,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, subconfig);
  ADD_PARAM_REC("AllowHosts", &config->allowed_hosts,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, pattern_list_parse);
  ADD_PARAM_REC("DenyHosts", &config->denied_hosts,
                SSH_CONFIG_TYPE_SERVER_GLOBAL, pattern_list_parse);

  /* Server configuration variables only available in special
     distributions. */































































  
  /*
   * Configuration variables only for the client.
   */
  ADD_PARAM_REC("IdentityFile", &config->identity_file,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("Host", &config->host_to_connect, SSH_CONFIG_TYPE_CLIENT,
                string);
  ADD_PARAM_REC("User", &config->login_as_user, SSH_CONFIG_TYPE_CLIENT,
                string);
  ADD_PARAM_REC("Compression", &config->compression,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("CompressionLevel", &config->compression_level,
                SSH_CONFIG_TYPE_CLIENT, compression_level);
  ADD_PARAM_REC("Batchmode", &config->batch_mode,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("AuthenticationNotify", &config->authentication_notify,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("PasswordPrompt", &config->password_prompt,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("DontReadStdin", &config->dont_read_stdin,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("Ssh1Path", &config->ssh1_path,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("SshSignerPath", &config->signer_path,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("GatewayPorts", &config->gateway_ports,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("LocalForward", &config->local_forwards,
                SSH_CONFIG_TYPE_CLIENT, local_forward);
  ADD_PARAM_REC("RemoteForward", &config->remote_forwards,
                SSH_CONFIG_TYPE_CLIENT, basic_forward);  
  ADD_PARAM_REC("ClearAllForwardings", NULL, SSH_CONFIG_TYPE_CLIENT,
                clear_forwards);
  ADD_PARAM_REC("ForcePTTYAllocation", &config->force_ptty_allocation,
                SSH_CONFIG_TYPE_CLIENT, bool);
  /* XXX */
  ADD_PARAM_REC("TryEmptyPassword", &config->try_empty_password,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("DefaultDomain", &config->default_domain,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("AuthenticationSuccessMsg", &config->auth_success_msg,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("DebugLogFile", &config->debug_log_file_name,
                SSH_CONFIG_TYPE_CLIENT, string);
  ADD_PARAM_REC("Ssh1AgentCompatibility", &config->ssh_agent_compat,
                SSH_CONFIG_TYPE_CLIENT, agent_compat);
  ADD_PARAM_REC("StrictHostkeyChecking", &config->strict_host_key_checking,
                SSH_CONFIG_TYPE_CLIENT, strict_hk_check);
  ADD_PARAM_REC("SetRemoteEnv", &config->remote_env, SSH_CONFIG_TYPE_CLIENT,
                basic_list_insert);
  ADD_PARAM_REC("EscapeChar", &config->escape_char, SSH_CONFIG_TYPE_CLIENT,
                escape_char);
  ADD_PARAM_REC("GoBackground", &config->go_background, SSH_CONFIG_TYPE_CLIENT,
                go_background);
  ADD_PARAM_REC("NumberOfPasswordPrompts", &config->number_of_password_prompts,
                SSH_CONFIG_TYPE_CLIENT, num);
  ADD_PARAM_REC("TrustX11Applications", &config->x11_application_trusted,
                SSH_CONFIG_TYPE_CLIENT, bool);
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
  ADD_PARAM_REC("Ssh1InternalEmulation", &config->ssh1_emulation_internal,
                SSH_CONFIG_TYPE_CLIENT, bool);
  ADD_PARAM_REC("Ssh1MaskPasswordLength",
                &config->ssh1_no_ignore_packets_in_password_auth,
                SSH_CONFIG_TYPE_CLIENT, bool_inverse);
#else /* WITH_INTERNAL_SSH1_EMULATION */
  ADD_PARAM_ALIAS("Ssh1InternalEmulation", "NoOpClient");
  ADD_PARAM_ALIAS("Ssh1MaskPasswordLength", "NoOpClient");
#endif /* WITH_INTERNAL_SSH1_EMULATION */  
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */









































  /*
   * Configuration variables that are deprecated or ignored.
   */
  /* Silently ignored. */
  ADD_PARAM_ALIAS("FallbackToRsh", "NoOpClient");
  ADD_PARAM_ALIAS("UseRsh", "NoOpClient");
  /* Deprecated (but still recognized). A warning will be output. */
  ADD_PARAM_REC("PubkeyAuthentication", NULL, SSH_CONFIG_TYPE_CLIENT |
                SSH_CONFIG_TYPE_SERVER_GLOBAL, deprecated);
  ADD_PARAM_REC("SshPAMClientPath", NULL, SSH_CONFIG_TYPE_SERVER_GLOBAL,
                deprecated);

  /* Aliases. */
  ADD_PARAM_ALIAS("RSAAuthentication", "PubkeyAuthentication");
  ADD_PARAM_ALIAS("PasswordAuthentication", "PubkeyAuthentication");
  ADD_PARAM_ALIAS("AllowX11Forwarding", "ForwardX11");
  ADD_PARAM_ALIAS("X11Forwarding", "ForwardX11");
  ADD_PARAM_ALIAS("AllowAgentForwarding", "ForwardAgent");
  ADD_PARAM_ALIAS("Verbose", "VerboseMode");
  
  config->host_keys_ctx = ssh_host_key_list_init();
  
  if (!client)
    {
      config->user_configurations =
        ssh_adt_create_generic(SSH_ADT_LIST,
                               SSH_ADT_DESTROY,
                               subconfig_destroy_callback,
                               SSH_ADT_ARGS_END);
      SSH_VERIFY(config->user_configurations != NULL);
      config->host_configurations =
        ssh_adt_create_generic(SSH_ADT_LIST,
                               SSH_ADT_DESTROY,
                               subconfig_destroy_callback,
                               SSH_ADT_ARGS_END);
      SSH_VERIFY(config->host_configurations != NULL);
    }
  else
    {
      config->remote_env =
        ssh_adt_create_generic(SSH_ADT_LIST,
                               SSH_ADT_DESTROY,
                               ssh_adt_callback_destroy_free,
                               SSH_ADT_ARGS_END);
      SSH_VERIFY(config->remote_env != NULL);
    }
  
  config->random_seed_file = ssh_xstrdup(SSH_RANDSEED_FILE);
  config->pgp_public_key_file = ssh_xstrdup(SSH_PGP_PUBLIC_KEY_FILE);
  config->pgp_secret_key_file = ssh_xstrdup(SSH_PGP_SECRET_KEY_FILE);

  config->x11_forwarding = TRUE;
  config->x11_forwarding = TRUE;
  config->allow_tcp_forwarding = TRUE;
  config->agent_forwarding = TRUE;

  config->x11_application_trusted = TRUE;
#ifdef XAUTH_PATH
  config->xauth_path = ssh_xstrdup(XAUTH_PATH);
#else /* XAUTH_PATH */
  config->xauth_path = NULL;
#endif /* XAUTH_PATH */
  config->force_ptty_allocation = FALSE;
  config->verbose_mode = FALSE;
  config->compression = FALSE;
  config->compression_level = -1;

  config->allowed_authentications =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  config->required_authentications =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  SSH_VERIFY(config->allowed_authentications != NULL);
  SSH_VERIFY(config->required_authentications != NULL);

  /* "hostbased"-authentication method is not enabled by default. */
  if (client)
    {
      ssh_config_parse_list((char *)
#ifdef KERBEROS
                            SSH_AUTH_KERBEROS_TGT ","
                            SSH_AUTH_KERBEROS ","
#endif /* KERBEROS */
                            SSH_AUTH_PUBKEY ","
#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
                            SSH_AUTH_KBD_INTERACTIVE ","
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */
                            SSH_AUTH_PASSWD,
                            NULL_FNPTR, NULL,
                            config->allowed_authentications);
    }
  /* Server. */

  else
    {
      /* "hostbased"-authentication method is not enabled by default. */
      if (ssh_adt_num_objects(config->allowed_authentications) == 0)
        {
          ssh_config_parse_list((char *)
#ifdef KERBEROS
                                SSH_AUTH_KERBEROS_TGT ","
                                SSH_AUTH_KERBEROS ","
#endif /* KERBEROS */
                                SSH_AUTH_PUBKEY "," SSH_AUTH_PASSWD,
                                NULL_FNPTR, NULL,
                                config->allowed_authentications);
        }
    }

  config->user_known_hosts = TRUE;
  config->port = ssh_xstrdup(SSH_DEFAULT_PORT);


  config->ciphers = NULL;












  config->macs = NULL;
  config->user_conf_dir = ssh_xstrdup(SSH_USER_CONFIG_DIRECTORY);








  config->identity_file = ssh_xstrdup(SSH_IDENTIFICATION_FILE);
  config->authorization_file = ssh_xstrdup(SSH_AUTHORIZATION_FILE);







  config->external_authorization_prog = NULL;
  
  config->passwd_path = ssh_xstrdup(PASSWD_PATH);
  config->password_prompt = ssh_xstrdup("%U's password: ");
  config->password_guesses = 3;
  config->number_of_password_prompts = 3;

  config->settable_env_vars = NULL;
  
  config->rekey_interval_bytes = 0L;
  config->rekey_interval_seconds = 3600L;

#ifndef SSH2_MAX_CONNECTIONS
  config->max_connections = 0;
#else /* ! SSH2_MAX_CONNECTIONS */
  config->max_connections = SSH2_MAX_CONNECTIONS;
#endif /* ! SSH2_MAX_CONNECTIONS */

  config->host_to_connect = NULL;
  config->login_as_user = NULL;
  config->local_forwards = NULL;
  config->remote_forwards = NULL;

  config->allowed_hosts = NULL;
  config->denied_hosts = NULL;
  config->allowed_shosts = NULL;
  config->denied_shosts = NULL;

  config->allowed_users = NULL;
  config->denied_users = NULL;
  config->allowed_groups = NULL;
  config->denied_groups = NULL;

  config->require_reverse_mapping = FALSE;
  config->try_reverse_mapping = TRUE;

  config->log_facility = SSH_LOGFACILITY_AUTH;
  config->sftp_server_log_facility = -1;
  config->debug_log_file_name = NULL;
  
  config->batch_mode = FALSE;
  config->authentication_notify = FALSE;
  config->strict_host_key_checking = SSH_STRICT_HOSTKEY_CHECKING_ASK;
  config->escape_char = ssh_xstrdup("~");
  config->go_background = FALSE;
  config->one_shot_forwarding = FALSE;
  config->dont_read_stdin = FALSE;
  config->gateway_ports = FALSE;

  config->ignore_rhosts = FALSE;
  config->ignore_root_rhosts = -1;
  config->permit_root_login = SSH_ROOTLOGIN_TRUE;
  config->permit_empty_passwords = TRUE;
  config->try_empty_password = FALSE;
  config->strict_modes = TRUE;
  config->quiet_mode = FALSE;
  config->fascist_logging = FALSE;
  config->print_motd = TRUE;
  config->check_mail = TRUE;
  config->auth_success_msg = TRUE;
  config->keep_alive = TRUE;
  config->no_delay = FALSE;
  config->inetd_mode = FALSE;
  config->hostbased_force_client_hostname_dns_match = FALSE;
  config->listen_address = ssh_xstrdup(SSH_IPADDR_ANY);
  config->login_grace_time = 600;
  config->chroot_users = NULL;
  config->chroot_groups = NULL;
  config->allowed_tcp_forwarding_users = NULL;
  config->denied_tcp_forwarding_users = NULL;
  config->denied_tcp_forwarding_groups = NULL;
  config->allowed_tcp_forwarding_groups = NULL;
  config->forward_acls =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY, forward_acl_destroy_cb,
                           SSH_ADT_ARGS_END);
  config->socks_server = NULL;
  config->use_socks5 = FALSE;
  config->subsystems = ssh_adt_xcreate_strmap(NULL_FNPTR,
                                              ssh_adt_callback_destroy_free);
  config->broadcasts_allowed_per_second = 0;

  config->ssh1compatibility = FALSE;
#ifdef SSH1_COMPATIBILITY
  config->ssh1_path = ssh_xstrdup(client ? SSH1_PATH : SSHD1_PATH);
#else /* SSH1_COMPATIBILITY */
  config->ssh1_path = NULL;
#endif /* SSH1_COMPATIBILITY */
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
  config->ssh1_emulation_internal = TRUE;
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
  config->sshd1_config_file_path = NULL;
  config->ssh_agent_compat = SSH_AGENT_COMPAT_NONE;

  config->signer_path = ssh_xstrdup(SSH_SIGNER_PATH);

  config->default_domain = NULL;
  
#ifdef SSH_SERVER_WITH_SECURID
  config->securid_guesses = 3;
#endif /* SSH_SERVER_WITH_SECURID */

  config->auth_interactive_failure_timeout = 2L;

  config->auth_publickey_min_size = 0;
  config->auth_publickey_max_size = 0;
  
#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
  config->auth_kbd_int_optional_needed = 0;
  config->auth_kbd_int_optional =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  config->auth_kbd_int_required =
    ssh_adt_create_generic(SSH_ADT_LIST,
                           SSH_ADT_DESTROY,
                           ssh_adt_callback_destroy_free,
                           SSH_ADT_ARGS_END);
  SSH_VERIFY(config->auth_kbd_int_optional != NULL);
  SSH_VERIFY(config->auth_kbd_int_required != NULL);
  config->auth_kbd_int_retries = 3;
  config->auth_kbd_int_plugin_prog = NULL;
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */





  config->banner_message_file_path = ssh_xstrdup(SSH_BANNER_MSG_FILE);
  config->banner_msg = NULL;

  config->idle_timeout = SSH_SERVER_DEFAULT_IDLE_TIMEOUT;

  config->protocol_version_string = ssh_xstrdup(SSH2_PROTOCOL_VERSION_STRING);
  
























































  return config;
}

/* This should be called after initializing the config-struct
   (ie. after command-line variables have been parsed. This checks,
   that some required members are initialized properly.)*/
/* XXX with user and host specific config, the line where this should be
   called has blurred. Solution? */
void ssh_config_init_finalize(SshConfig config)
{
  /* Common. */
  
  if (config->ciphers == NULL)
    {
      char *hlp;

      hlp = ssh_cipher_get_supported_native();
      config->ciphers = ssh_snlist_intersection(SSH_STD_CIPHERS, hlp);
      ssh_xfree(hlp);
      hlp = config->ciphers;
      config->ciphers = ssh_cipher_list_exclude(hlp, "none");
      ssh_xfree(hlp);
    }

  if (config->macs == NULL)
    {
      char *hlp;

      hlp = ssh_mac_get_supported();
      config->macs = ssh_snlist_intersection(SSH_STD_MACS, hlp);
      ssh_xfree(hlp);
      hlp = config->macs;
      config->macs = ssh_cipher_list_exclude(hlp, "none");
      ssh_xfree(hlp);      
    }






















  /* Server. */
  if (!config->client)
    {
      /* If IgnoreRootRhosts isn't defined at this stage, assign it to
         the same as IgnoreRhosts. */
      if (config->ignore_root_rhosts == -1)
        config->ignore_root_rhosts = config->ignore_rhosts;




















    }

}

void ssh_free_forward(SshForward fwd)
{
  SSH_PRECOND(fwd != NULL);
  ssh_xfree(fwd->local_addr);
  ssh_xfree(fwd->port);
  ssh_xfree(fwd->connect_to_host);
  ssh_xfree(fwd->connect_to_port);
  ssh_xfree(fwd->protocol);
  ssh_xfree(fwd);
}




































/* Frees client configuration data. */
void ssh_config_free(SshConfig config)
{
  SshForward ptr = NULL;

  ssh_adt_destroy(config->set_param);
  ssh_adt_destroy(config->set_param_aliases);
  
  /* free all allocated memory */
  ssh_xfree(config->random_seed_file);
  ssh_xfree(config->pgp_public_key_file);
  ssh_xfree(config->pgp_secret_key_file);

  ssh_xfree(config->port);
  ssh_xfree(config->ciphers);
  ssh_xfree(config->macs);
  ssh_xfree(config->user_conf_dir);





  ssh_xfree(config->identity_file);
  ssh_xfree(config->authorization_file);
  ssh_xfree(config->escape_char);
  ssh_xfree(config->listen_address);
  ssh_xfree(config->external_authorization_prog);
  ssh_xfree(config->passwd_path);
  ssh_xfree(config->password_prompt);

  ssh_adt_destroy(config->user_configurations);
  ssh_adt_destroy(config->host_configurations);

  ssh_xfree(config->host_to_connect);
  ssh_xfree(config->login_as_user);

  for (ptr = config->local_forwards; ptr;)
    {
      SshForward to_be_deleted = ptr;
      ptr = ptr->next;
      ssh_free_forward(to_be_deleted);
    }

  for (ptr = config->remote_forwards; ptr;)
    {
      SshForward to_be_deleted = ptr;
      ptr = ptr->next;
      ssh_free_forward(to_be_deleted);
    }

  ssh_adt_destroy(config->allowed_hosts);
  ssh_adt_destroy(config->denied_hosts);
  
  ssh_adt_destroy(config->allowed_shosts);
  ssh_adt_destroy(config->denied_shosts);  

  ssh_adt_destroy(config->allowed_users);
  ssh_adt_destroy(config->denied_users);
  ssh_adt_destroy(config->allowed_groups);
  ssh_adt_destroy(config->denied_groups);

  ssh_adt_destroy(config->allowed_authentications);
  ssh_adt_destroy(config->required_authentications);

#ifdef SSHDIST_SSH2_AUTH_KBDINTERACTIVE
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
  ssh_adt_destroy(config->auth_kbd_int_optional);
  ssh_adt_destroy(config->auth_kbd_int_required);
  ssh_xfree(config->auth_kbd_int_plugin_prog);
#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
#endif /* SSHDIST_SSH2_AUTH_KBDINTERACTIVE */

  ssh_adt_destroy(config->chroot_users);
  ssh_adt_destroy(config->chroot_groups);

  ssh_adt_destroy(config->allowed_tcp_forwarding_users);
  ssh_adt_destroy(config->denied_tcp_forwarding_users);
  ssh_adt_destroy(config->denied_tcp_forwarding_groups);
  ssh_adt_destroy(config->allowed_tcp_forwarding_groups);
  ssh_adt_destroy(config->forward_acls);
  
  ssh_adt_destroy(config->remote_env);
  ssh_adt_destroy(config->settable_env_vars);

  ssh_xfree(config->debug_log_file_name);  
  ssh_xfree(config->socks_server);

  ssh_adt_destroy(config->subsystems);
  
  ssh_xfree(config->ssh1_path);
  ssh_xfree(config->sshd1_config_file_path);
  
  ssh_xfree(config->signer_path);
  ssh_xfree(config->default_domain);

  ssh_xfree(config->xauth_path);
  
  ssh_xfree(config->banner_message_file_path);
  ssh_xfree(config->banner_msg);

  ssh_xfree(config->protocol_version_string);





















  /* XXX kludge; sanitize. */
  if (config->host_keys_ctx)
    {
      ssh_host_key_list_free(config->host_keys_ctx);
      config->host_keys_ctx = NULL;
    }








  memset(config, 0, sizeof(*config));
  ssh_xfree(config);
}


/* Returns default configuration information for the server. */

SshConfig ssh_server_create_config()
{
  return ssh_config_init(FALSE);
}

/* Returns default configuration information for the client. */

SshConfig ssh_client_create_config()
{
  return ssh_config_init(TRUE);
}

/* Set the variable corresponding to `var' to `val' in config */

Boolean ssh_config_set_parameter(SshConfig config, SshMetaConfig metaconfig,
                                 char *var, char *val, int config_type)
{
  Boolean bool_val;
  SshRegexSyntax rex_syntax;
  int config_version_major, config_version_minor, num;
  char *real_var;
  SshConfigSetParam item;
  SshConfigSetParamAlias alias;

  SSH_PRECOND(metaconfig != NULL);
  
  if (config->client == TRUE)
    SSH_PRECOND(config_type & SSH_CONFIG_TYPE_CLIENT);
  else
    SSH_PRECOND(config_type & SSH_CONFIG_TYPE_SERVER_ALL);

  config_version_major = metaconfig->version_major;
  config_version_minor = metaconfig->version_minor;
  rex_syntax = metaconfig->regex_syntax;
  
  switch (val[0])
    {
    case 'y':  /* for "yes" */
    case 'Y':
    case 't':  /* for "true" */
    case 'T':
    case 'k':  /* for kylla [finnish] :-) */
    case 'K':

      bool_val = TRUE;
      break;

    default:
      bool_val = FALSE;
    }

  num = atoi(val);

  /* Aliases. */
  if ((alias = ssh_adt_strmap_get(config->set_param_aliases, var)) == NULL)
    real_var = var;
  else
    real_var = alias->variable;
  
  if ((item = ssh_adt_strmap_get(config->set_param, real_var)) != NULL)
    {
      if (item->type_mask & config_type)
        {
          SSH_DEBUG(6, ("Got match with SetParam array variable '%s' "
                        "and '%s'. %s%s%s", item->variable, real_var,
                        alias ? "(Specified option " : "",
                        alias ? alias->alias : "",
                        alias ? " is an alias to it)" : ""));
          return (*item->callback)(config, metaconfig,
                                   alias ?
                                   alias->alias : item->variable,
                                   val, num, bool_val,
                                   item->var_to_set);
        }
    }  
  
  /* These parameters are only for the server */
  if (config->client == FALSE)
    {      
      /* Parse subsystem definitions */
      /* This applies to all server configurations. This remains here,
         because the configuration variable name is part of this
         configuration. */
      if (strncasecmp(var, SUBSYSTEM_PREFIX, SUBSYSTEM_PREFIX_LEN) == 0)
        {
          SshADTContainer c = config->subsystems;
          char *subsys_key = &var[SUBSYSTEM_PREFIX_LEN];
          int i;
          for (i = 0; subsys_key[i]; i++)
            subsys_key[i] = tolower(subsys_key[i]);
          
          if (strlen(val) < 1)
            {
              ssh_adt_strmap_remove(c, subsys_key);
              return FALSE;
            }

          ssh_adt_strmap_set(c, subsys_key, ssh_xstrdup(val));
          return FALSE;
        }
    }

  ssh_warning("Unrecognized configuration parameter '%s'.", var);
  return TRUE;
}

/* Checks whether linebuf is a heading. */
char *ssh2_config_line_heading_separator(char *linebuf)
{
  char *hlp;

  hlp = &linebuf[strlen(linebuf) - 1];
  
  if (*hlp == ':')
    return hlp;
  else
    return NULL;
}

void ssh2_config_remove_quotes(char *str)
{
  char *hlp1, *hlp2;
  int in_quotes = 0;
  int quoted = 0;

  hlp1 = hlp2 = str;

  while (*hlp1)
    {
      switch (*hlp1)
        {
        case '"':
          if (quoted)
            {
              *hlp2 = *hlp1;
              hlp2++;
            }
          in_quotes = !in_quotes;
          break;


        case '\\':
          if ((!in_quotes) || quoted)
            {
              *hlp2 = *hlp1;
              hlp2++;
              quoted = 0;
            }
          else
            {
              quoted = 1;
            }
          break;


        default:
          *hlp2 = *hlp1;
          hlp2++;
          quoted = 0;
        }
      hlp1++;
    }
  *hlp2 = '\0';
}

/* Parse a configuration/authorization file into an array of variable
   name <-> value pairs. Return the number of variables (if no variables
   could be read, returns 0) or -1 if file couldn't be opened. Pointers
   to tables of pointers to null-terminated strings are placed at
   *variables and *values. If ``match_cb'' is not NULL_FNPTR, this
   function will ignore ``instance'' and will use the callback to check
   whether a heading matches. If ``match_cb'' is NULL_FNPTR and instance
   is NULL, will warn about headings, and the stanzas following headings
   will be ignored. */
int ssh2_parse_config_ext(SshUser user, const char *instance,
                          SshConfigMatchCB match_cb,
                          void *match_ctx,
                          const char *path,
                          char ***variables, char ***values,
                          SshMetaConfig metaconfig,
                          Boolean remove_quotes)
{
  int i = 0;

  SshUserFile f;
  char **vars, **vals, *var = NULL, *val = NULL, *hlp;
  char linebuf[1024];
  size_t n;
  int j, k;
  int line, ch;
  Boolean matching;
  Boolean doing_metaconfig = TRUE;
  Boolean metaconf_initialized = FALSE;
  int version_major = 1, version_minor = 0;
  /* The default, if version is old, this is set ZSH_FILEGLOB later. */
  SshRegexSyntax rex_syntax = SSH_REGEX_SYNTAX_EGREP;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
  SshRegexMatcher rl = ssh_regex_create(rex_ctx,
                                        "^(#[#~s]+[A:Z+0:9+~-]+~s.*)~n",
                                        SSH_REGEX_SYNTAX_SSH);
  SshRegexMatcher r_rex = ssh_regex_create(rex_ctx,
                                           "^#[#~s]+REGEX-SYNTAX+~s+(~S+).*$",
                                           SSH_REGEX_SYNTAX_SSH);
  SSH_PRECOND(rl != NULL);
  SSH_PRECOND(r_rex != NULL);
  SSH_PRECOND(variables != NULL);
  SSH_PRECOND(values != NULL);

  ssh_userfile_init(ssh_user_name(user), ssh_user_uid(user),
                    ssh_user_gid(user), NULL_FNPTR, NULL);
  
  if ((f = ssh_userfile_open
       (

        user == NULL ? getuid() :

        ssh_user_uid(user), path, O_RDONLY, 0755)) ==
      NULL)
    {
      SSH_TRACE(0, ("Unable to open %s", path));
      ssh_regex_free(r_rex);
      ssh_regex_free(rl);
      ssh_userfile_uninit();
      return -1;
    }

  line = 0;
  i = 0;
  n = 16;
  matching = TRUE;
  vars = ssh_xcalloc(n, sizeof(char *));
  vals = ssh_xcalloc(n, sizeof(char *));

  while (ssh_userfile_gets(linebuf, sizeof(linebuf) - 1, f) != NULL)
    {
      line++;

      /* skip the starting white spaces and comment lines */
      for (j = 0;; j++)
        {
          ch = linebuf[j];

          if (ch == '\0' || (ch == '#' && !doing_metaconfig))
            goto skip;

          if (ch == '#')
            {
              /* If we are doing metaconfig... */
              if (line == 1)
                {
                  SshRegexMatcher r = ssh_regex_create(rex_ctx,
                                                       "^#.*VERSION~s+(~d+)"
                                                       "~.(~d+).*$",
                                                       SSH_REGEX_SYNTAX_SSH);
                  SSH_ASSERT(r != NULL);
                  if (ssh_regex_match_cstr(r, &linebuf[j]))
                    {
                      char *maj_ver_str = (char *)ssh_regex_get_submatch(r, 1);
                      char *min_ver_str = (char *)ssh_regex_get_submatch(r, 2);
                      SSH_ASSERT(maj_ver_str != NULL);
                      SSH_ASSERT(min_ver_str != NULL);
                      version_major = atoi(maj_ver_str);
                      version_minor = atoi(min_ver_str);
                      if (version_major < 1 || version_minor < 0)
                        {
                          SSH_TRACE(3, ("%s: Invalid version number "
                                        "(%d.%d).", path,
                                        version_major, version_minor));
                          version_major = 1;
                          version_minor = 0;
                          ssh_regex_free(r);
                          goto metaconfig_error;
                        }
                      metaconf_initialized = TRUE;
                    }
                  else
                    {
                      SSH_DEBUG(2, ("Version not found on first line, "
                                    "assuming configuration to be old "
                                    "style."));
                      doing_metaconfig = FALSE;
                    }
                  ssh_regex_free(r);
                }
              else if (ssh_regex_match_cstr(rl, &linebuf[j]))
                {
                  SSH_DEBUG(4, ("Got metaconfig line `%s'.",
                                ssh_regex_get_submatch(rl, 1)));
                  if (ssh_regex_match_cstr(r_rex, &linebuf[j]))
                    {
                      char *rex_str = ssh_regex_get_submatch(r_rex, 1);
                      SSH_ASSERT(rex_str != NULL);
                      if (!strcasecmp(rex_str, "ssh"))
                        rex_syntax = SSH_REGEX_SYNTAX_SSH;
                      else if (!strcasecmp(rex_str, "egrep"))
                        rex_syntax = SSH_REGEX_SYNTAX_EGREP;
                      else if (!strcasecmp(rex_str, "zsh_fileglob") ||
                               !strcasecmp(rex_str, "traditional"))
                        rex_syntax = SSH_REGEX_SYNTAX_ZSH_FILEGLOB;
                      else
                        goto metaconfig_error;

                      SSH_TRACE(3, ("Metaconfig specifies regex style '%s'.",
                                    rex_syntax == SSH_REGEX_SYNTAX_SSH ?
                                    "SSH" :
                                    rex_syntax == SSH_REGEX_SYNTAX_EGREP ?
                                    "EGREP" : "ZSH_FILEGLOB"));
                    }
                }
              else
                {
                  SSH_DEBUG(2, ("Metaconfig parsing stopped at line %d.",
                                line));
                  doing_metaconfig = FALSE;
                }
              goto skip;
            metaconfig_error:
              ssh_warning("%s: Invalid metaconfig line (%d: %s)",
                          path, line, linebuf);
              doing_metaconfig = FALSE;
              goto skip;
            }

          if (!isspace(ch))
            {
              doing_metaconfig = FALSE;
              break;
            }
        }
      /* Remove trailing whitespace. */
      for (k = strlen(linebuf) - 1; k >= 0; k--)
        {
          if (!isspace(linebuf[k]))
            {
              linebuf[k + 1] = '\0';
              break;
            }
        }

      if (k <= 0)
        goto skip;
      
      if (version_major == 1 && version_minor == 0 && !metaconf_initialized)
        {
          /* Here we set defaults for metaconfig parameters, if
             conf version is too old (or version is missing) . */
          SSH_TRACE(3, ("Configuration file `%s' is old-style. (1.0)", path));
          rex_syntax = SSH_REGEX_SYNTAX_ZSH_FILEGLOB;
        }

      if (doing_metaconfig)
        {
          doing_metaconfig = FALSE;
          metaconf_initialized = TRUE;
        }
      
      /* see if this is a heading or not.. */
      if ((hlp = ssh2_config_line_heading_separator(&linebuf[j])) != NULL)
        {
          SSH_DEBUG(3, ("Found heading '%s'.", &linebuf[j]));
          *hlp = '\0';

          if (instance == NULL && match_cb == NULL_FNPTR)
            {
              ssh_warning("%s: %d: File has heading '%s', even though "
                          "headings (and configuration options following "
                          "them) in this file will never parsed.", path, line,
                          &linebuf[j]);
              matching = FALSE;
              continue;
            }
          else
            {
              if (match_cb != NULL_FNPTR)
                {
                  SshMetaConfigStruct meta;
                  meta.version_major = version_major;
                  meta.version_minor = version_minor;
                  meta.regex_syntax = rex_syntax;
                  matching = (*match_cb)(&linebuf[j], &meta, match_ctx);
                }
              else
                {
                  SshRegexMatcher matcher = NULL;
                  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();
                  
                  matcher = ssh_regex_create(rex_ctx, &linebuf[j], rex_syntax);

                  if (!matcher)
                    {
                      ssh_warning("%s: %d: Illegal regex pattern '%s'.", path,
                                  line, &linebuf[j]);
                      matching = FALSE;
                    }
                  else
                    {
                      matching = ssh_regex_match_cstr(matcher, instance);
                      if (matching)
                        {
                          int m_index = 0;
                          size_t mlen = 0;
                          SSH_DEBUG(3, ("'%s' matches (at least partially) "
                                        "with instance '%s'", &linebuf[j],
                                        instance));
                  
                          (void) ssh_regex_access_submatch(matcher, 0,
                                                           &m_index, &mlen);
                          if (m_index != 0 || mlen != strlen(instance))
                            {
                              /* Did not match whole instance. */
                              SSH_DEBUG(3, ("'%s' didn't match instance '%s' "
                                            "completely", &linebuf[j],
                                            instance));
                              matching = FALSE;
                            }
                        }
                      ssh_regex_free(matcher);
                    }
                }
              continue;
            }
        }
      /* ok, it must be a variable definition */
      if (!matching)
        goto skip;

      if (ssh_config_parse_line(linebuf, &var, &val))
        {
          ssh_warning("%s: %d: parsing line failed.", path, line);
          goto skip;
        }
      
      vars[i] = var;
      vals[i] = val;
      
      if (remove_quotes)
        ssh2_config_remove_quotes(vals[i]);

      i++;

      /* get more space if needed */
      if (i >= n)
        {
          n = 2 * n;
          vars = ssh_xrealloc(vars, n * sizeof(char *));
          vals = ssh_xrealloc(vals, n * sizeof(char *));
        }
    skip:
      ;
    }

  ssh_userfile_close(f);
  *variables = vars;
  *values = vals;

  ssh_regex_free(rl);
  ssh_regex_free(r_rex);
  
  if (metaconfig)
    {
      metaconfig->version_major = version_major;
      metaconfig->version_minor = version_minor;
      metaconfig->regex_syntax = rex_syntax;
    }

  ssh_userfile_uninit();

  return i;
}

/* Parse a configuration/authorization file into an array of variable
   name <-> value pairs. Return the number of variables or -1 on
   error. Pointers to tables of pointers to null-terminated strings are
   placed at *variables and *values. ``metaconfig'' will store
   "configuration for the configuration" and can be NULL, if caller
   doesn't care. Stanzas with headings matching ``instance'' will be
   parsed. If ``instance'' is NULL, this function will warn about
   headings, and the stanzas following headings will be ignored. */
int ssh2_parse_config(SshUser user, const char *instance, const char *path,
                      char ***variables, char ***values,
                      SshMetaConfig metaconfig,
                      Boolean remove_quotes)
{
  return ssh2_parse_config_ext(user, instance, NULL_FNPTR, NULL, path,
                               variables, values, metaconfig, remove_quotes);
}
/* Parse a line of input */
Boolean ssh_config_parse_line(char *line, char **var, char **val)
{
  SshRegexMatcher r;
  char *rex_str = "^~s*([~S-=]+){~s*=~s*|~s+}(.*)$";
  char *n_var = NULL, *n_val = NULL;
  
  r = ssh_regex_create(ssh_app_get_global_regex_context(),
                       rex_str, SSH_REGEX_SYNTAX_SSH);

  SSH_VERIFY(r != NULL);

  if (!ssh_regex_match_cstr(r, line))
    {
      SSH_DEBUG(4, ("Regex didn't match."));
      ssh_regex_free(r);
      return TRUE;
    }

  n_var = ssh_regex_get_submatch(r, 1);
  n_val = ssh_regex_get_submatch(r, 2);

  SSH_DEBUG(6, ("n_var = `%s', n_val = `%s'", n_var, n_val));
  
  SSH_ASSERT(n_var != NULL);
  SSH_ASSERT(n_val != NULL);

  *var = ssh_xstrdup(n_var);
  *val = ssh_xstrdup(n_val);

  ssh_regex_free(r);

  return FALSE;  
}

/* Parse line of input and set the parameters in the SshConfig struct.
   Returns TRUE on failure, FALSE otherwise. */
Boolean ssh_config_parse_line_and_set_params(SshConfig config, char *line,
                                             int config_type)
{
  char *var = NULL, *val = NULL;
  SshMetaConfigStruct metaconfig;
  Boolean ret = TRUE;
  
  metaconfig.version_major = 1;
  metaconfig.version_minor = 1;
  metaconfig.regex_syntax = SSH_REGEX_SYNTAX_EGREP;
  
  if (ssh_config_parse_line(line, &var, &val))
    return TRUE;
  
  ret = ssh_config_set_parameter(config, &metaconfig, var, val, config_type);

  ssh_xfree(var);
  ssh_xfree(val);
  
  return ret;
}


/* Reads config data from the given file.  Returns FALSE if an error
   occurs (displays error messages with ssh_warning.)  If ``match_cb''
   is not NULL_FNPTR, this function will ignore ``instance'' and will
   use the callback to check whether a heading matches. If ``match_cb''
   is NULL_FNPTR and instance is NULL, will warn about headings, and the
   stanzas following headings will be ignored. */
Boolean ssh_config_read_file_ext(SshUser user, SshConfig config,
                                 const char *instance,
                                 SshConfigMatchCB match_cb, void *match_ctx,
                                 const char *filename,
                                 int config_type)
{
  SshUser user_data;
  SshMetaConfigStruct metaconfig;
  char **vars, **vals;
  int i, n;
  Boolean something_failed = FALSE, ret = FALSE;
  
  if (filename == NULL || strlen(filename) == 0)
    return FALSE;

  if (user == NULL)
    user_data = ssh_user_initialize(NULL, FALSE);
  else
    user_data = user;

  memset(&metaconfig, 0, sizeof(metaconfig));
  
  n = ssh2_parse_config_ext(user_data, instance, match_cb, match_ctx,
                            filename, &vars, &vals, &metaconfig, TRUE);

  if (n < 0)
    {
      if (user_data != user)
        ssh_user_free(user_data, FALSE);
      return FALSE;
    }

  /* ok, now fill in the fields */

  for (i = 0; i < n; i++)
    {
      ret = ssh_config_set_parameter(config, &metaconfig, vars[i], vals[i],
                                     config_type);
      if (!something_failed && ret)
        something_failed = ret;
    }

  ret = TRUE;
  
  if (something_failed)
    {
      ssh_warning("Failed to parse some variables from config file '%s'.",
                  filename);
      ret = FALSE;
    }
  
  ssh_free_varsvals(n, vars, vals);

  if (user_data != user)
    ssh_user_free(user_data, FALSE);

  return ret;
}

Boolean ssh_config_read_file(SshUser user, SshConfig config,
                             const char *instance, const char *filename,
                             int config_type)
{
  return ssh_config_read_file_ext(user, config, instance, NULL_FNPTR, NULL,
                                  filename, config_type);
}


/* Parse forwarding definitions. Format is
   [protocol/][localhost:]port:remotehost:remoteport */
Boolean ssh_parse_forward(SshForward *fws, const char *s)
{
  SshForward fw;
  char *lp, *lh, *rh, *rp, *pr;
  size_t l;
  SshRegexContext rc = ssh_app_get_global_regex_context();
  SshRegexMatcher r;

  l = strlen(s);
  r = ssh_regex_create(rc,
                       "^([^:/]+)/\\[([^\\]]+)\\]:([^:]+):\\[([^\\]]+)\\]:"
                       "([^:]+)$", SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 1 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 2 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/\\[([^\\]]+)\\]:([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 3 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 5));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 4 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 5 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:/]+)/([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 6 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^\\[([^\\]]+)\\]:([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 7 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 8 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^\\[([^\\]]+)\\]:([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 9 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 4));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 10 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):\\[([^\\]]+)\\]:([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 11 in forward parser.");
    }
  r = ssh_regex_create(rc,
                       "^([^:]+):([^:]+):([^:]+)$",
                       SSH_REGEX_SYNTAX_EGREP);
  if (r != NULL)
    {
      if (ssh_regex_match(r, s, l))
        {
          pr = ssh_xstrdup("tcp");
          lh = ssh_xstrdup(SSH_IPADDR_ANY);
          lp = ssh_xstrdup(ssh_regex_get_submatch(r, 1));
          rh = ssh_xstrdup(ssh_regex_get_submatch(r, 2));
          rp = ssh_xstrdup(ssh_regex_get_submatch(r, 3));
          ssh_regex_free(r);
          goto done;
        }
      ssh_regex_free(r);
    }
  else
    {
      ssh_warning("unable to create regexp 12 in forward parser.");
    }
  SSH_DEBUG(5, ("forward parsing failed"));
  return TRUE;

 done:
  SSH_DEBUG(5, ("fw: lh=\"%s\" lp=\"%s\" rh=\"%s\" rp=\"%s\" pr=\"%s\"",
                lh, lp, rh, rp, pr));
  fw = ssh_xcalloc(1, sizeof (*fw));
  fw->local_addr = lh;
  fw->port = lp;
  fw->connect_to_host = rh;
  fw->connect_to_port = rp;
  fw->protocol = pr;
  fw->next = *fws;
  *fws = fw;
  return FALSE;
}

SshInt32 ssh_config_parse_timeout(const char *value)
{
  char ch, *str;
  SshInt32 num;
  
  str = ssh_xstrdup(value);
          
  ch = str[strlen(str) - 1];

  if (!isdigit(ch))
    {
      str[strlen(str) - 1] = '\0';
      ch = tolower(ch);
    }
          
  num = atoi(str);
          
  if (num < 0)
    {
      num = -1;
      goto error;
    }

  if (!isdigit(ch))
    {
      switch (ch)
        {
        case 'w': /* Weeks. */
          num *= 7 * 24 * 60 * 60;
          break;
        case 'd': /* Days. */
          num *= 24 * 60 * 60;
          break;
        case 'h': /* Hours. */
          num *= 60 * 60;
          break;
        case 'm': /* Minutes. */
          num *= 60;
          break;
        case 's': /* Seconds. */
          /* Nothing. */
          break;
        default:
          num = -1;
        }
    }

 error:
  ssh_xfree(str);
  return num;
}

char *ssh_config_log_facility_name(SshLogFacility fac)
{
  int i;

  for (i = 0; logfacilities[i].fac_name; i++)
    {
      if (logfacilities[i].facility == fac)
        return ssh_xstrdup(logfacilities[i].fac_name);
    }
  return NULL;
}

void ssh_config_read_banner_message(SshConfig config)
{
  SshUserFile f;
  uid_t uid = 999;

  uid = getuid();


  if ((f = ssh_userfile_open(uid, config->banner_message_file_path,
                             O_RDONLY, 0755)) != NULL)
    {
      SshBufferStruct buf;
      char linebuf[1024];

      uid = getuid();


      ssh_buffer_init(&buf);

      while (ssh_userfile_gets(linebuf, sizeof(linebuf) - 1, f) != NULL)
        ssh_xbuffer_append(&buf, linebuf, strlen(linebuf));

      if (ssh_buffer_len(&buf) > 0)
        {
          size_t len;

          ssh_xbuffer_append(&buf, "\0", 1);
          
          config->banner_msg = ssh_buffer_steal(&buf, &len);
          if (config->banner_msg == NULL)
            {
              ssh_warning("Unable to get banner message from buffer; "
                          "memory allocation failed.");
            }
          else
            {
              /* If the banner message is too big, truncate it to fit
                 a protocol packet. */
              if (len > SSH_MAX_PAYLOAD_LENGTH - 1)
                {
                  len = SSH_MAX_PAYLOAD_LENGTH - 1;              
                  config->banner_msg[len] = '\0';
                }
            }
        }

      ssh_buffer_uninit(&buf);

      ssh_userfile_close(f);
    }
}
