/*

sshchssh1agent.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Code for ssh1 authentication agent forwarding channels for SSH2 
  servers and clients.

*/

#include "ssh2includes.h"
#include "sshfilterstream.h"
#include "sshtimeouts.h"
#include "sshgetput.h"
#include "sshtcp.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "sshcommon.h"
#include "sshagentint.h"

#ifdef SSH_CHANNEL_SSH1_AGENT

#include "sshchssh1agent.h"

#define SSH_DEBUG_MODULE "Ssh2ChannelSsh1Agent"

#define AGENT_WINDOW_SIZE        10000
#define AGENT_PACKET_SIZE         1024

typedef struct SshChannelTypeSsh1AgentRec
{
  /* Pointer to the SshCommon object. */
  SshCommon common;

  /* Data for the proxy agent on the server side. */
  char *agent_path;
  SshLocalListener agent_listener;

  /* Flag indicating that agent forwarding has been requested. */
  Boolean agent_requested;
} *SshChannelTypeSsh1Agent;

typedef struct SshChannelSsh1AgentSessionRec
{
  SshCommon common;
} *SshChannelSsh1AgentSession;

/***********************************************************************
 * Glue functions for creating/destroying the channel type context.
 ***********************************************************************/

/* This function is called once when a SshCommon object is created. */

void *ssh_channel_ssh1_agent_create(SshCommon common)
{
  SshChannelTypeSsh1Agent ct;
  
  ct = ssh_xcalloc(1, sizeof(struct SshChannelTypeSsh1AgentRec));
  ct->common = common;
  return ct;
}

/* This function is called once when an SshCommon object is being
   destroyed.  This should destroy all agent channels and listeners and
   free the context. */

void ssh_channel_ssh1_agent_destroy(void *context)
{
  SshChannelTypeSsh1Agent ct = (SshChannelTypeSsh1Agent)context;

  /* Destroy all existing channels.
     XXX not implemented. */

  /* Destroy the listener. */
  if (ct->agent_listener)
    {
      ssh_local_destroy_listener(ct->agent_listener);

      /* Destroy the socket. */
      remove(ct->agent_path);
    }

  /* Free the path name. */
  ssh_xfree(ct->agent_path);

  /* Destroy the channel type context. */
  ssh_xfree(ct);
}

/* This function is called once for each session channel that is created.
   This should initialize per-session state for agent forwarding.  The
   argument points to a void pointer that will be given as argument to
   the following functions.  It can be used to store the per-session
   state. */

void ssh_channel_ssh1_agent_session_create(SshCommon common,
                                          void **session_placeholder)
{
  SshChannelSsh1AgentSession session;

  /* Allocate a session context. */
  session = ssh_xcalloc(1, sizeof(*session));
  session->common = common;

  *session_placeholder = (void *)session;
}
                                        
/* This function is called once whenever a session channel is destroyed.
   This should free any agent forwarding state related to the session. */

void ssh_channel_ssh1_agent_session_destroy(void *session_placeholder)
{
  SshChannelSsh1AgentSession session = (SshChannelSsh1AgentSession)session_placeholder;

  /* Free the session context. */
  ssh_xfree(session);
}

/* Returns the channel type context from the SshCommon object. */

SshChannelTypeSsh1Agent ssh_channel_ssh1_agent_ct(SshCommon common)
{
  return (SshChannelTypeSsh1Agent)ssh_common_get_channel_type_context(common,
                                                          "auth-ssh1-agent");
}

/***********************************************************************
 * Functions that are used in the SSH server end.  These receive
 * incoming agent connections, and cause channel open requests to be
 * sent to the SSH client.
 ***********************************************************************/

/* Function to be called when a forwarded agent connection is closed.  Note
   that this takes SshCommon as context, as the forwarded connection might
   outlive the session in which it was created. */

void ssh_channel_ssh1_agent_connection_destroy(void *context)
{
  SshCommon common = (void *)context;

  /* Inform the SshCommon object that a channel has been destroyed.  Note
     that this may cause the SshCommon object to be destroyed, and the
     destroy function for agent forwarding to be called. */
  ssh_common_destroy_channel(common);
}

/* Processes an incoming connection to the agent.  This is called when
   a connection is received at the agent listener.  This will open the
   agent channel to the client. */

void ssh_channel_ssh1_agent_connection(SshStream stream, void *context)
{
  SshCommon common = (SshCommon)context;

  SSH_DEBUG(5, ("incoming connection to agent proxy"));
   
  /* Increase the number of open channels.  This will be decremented in
     the destroy function. */
  ssh_common_new_channel(common);

  /* Send a channel open message.  If this fails, the stream will be
     automatically closed. */








  ssh_conn_send_channel_open(common->conn, "auth-ssh1-agent",
                             stream, TRUE, TRUE,
                             AGENT_WINDOW_SIZE, AGENT_PACKET_SIZE,
                             NULL, 0, NULL_FNPTR,
                             ssh_channel_ssh1_agent_connection_destroy,
                             (void *)common, NULL_FNPTR, NULL);

}

/***********************************************************************
 * Functions that are used at the SSH client end for connecting to the
 * real authentication agent.
 ***********************************************************************/

typedef struct SshAgentConnectionRec
{
  SshCommon common;
  int channel_id;



  SshConnOpenCompletionProc completion;

  void *context;
} *SshAgentConnection;

/* This filter function will insert a SSH_AGENT_FORWARDING_NOTICE
   message into the channel to the agent.  */

void
ssh_agent_ssh1_forwarding_notice_insert_filter(void *context,
                                               SshFilterGetCB get_data,
                                               SshFilterCompletionCB completed,
                                               void *internal_context)
{
  SshBuffer msg_buffer = (SshBuffer)context; /* Message is given in this buffer */
  SshBuffer data_buf;
  size_t offset, orig_data_len;
  Boolean eof_received;
  unsigned char *append_ptr;

  if (ssh_buffer_len(msg_buffer) == 0)
    {
      (*completed)(internal_context, SSH_FILTER_SHORTCIRCUIT);
      return;
    }

  (*get_data)(internal_context, &data_buf, &offset, &eof_received);

  /* Make room for the message to be inserted */
  orig_data_len = ssh_buffer_len(data_buf);
  ssh_xbuffer_append_space(data_buf, &append_ptr, ssh_buffer_len(msg_buffer));
  memmove(ssh_buffer_ptr(data_buf) + ssh_buffer_len(msg_buffer),
          ssh_buffer_ptr(data_buf), orig_data_len);

  /* Insert the message */
  memcpy(ssh_buffer_ptr(data_buf), ssh_buffer_ptr(msg_buffer),
         ssh_buffer_len(msg_buffer));

  /* Empty the msg_buffer so that we can shortcircuit the filter on the
     next call. */
  ssh_buffer_consume(msg_buffer, ssh_buffer_len(msg_buffer));

  /* Pass the whole thing */
  (*completed)(internal_context, SSH_FILTER_ACCEPT(ssh_buffer_len(data_buf)));
}

void
ssh_agent_ssh1_forwarding_filter_destroy(void *context)
{
  ssh_buffer_free((SshBuffer)context);
}


/* This function is called when a connecting to a real agent has been
   completed (possibly with error). */

void ssh_channel_open_ssh1_agent_connected(SshStream stream,
                                          void *context)
{
  SshAgentConnection a = (SshAgentConnection)context;
  SshStream filter_stream;
  SshBuffer buffer;
  unsigned char *cp;

  if (stream == NULL)
    {
      SSH_DEBUG(1, ("Connecting to the real agent failed."));
      (*a->completion)(SSH_OPEN_CONNECT_FAILED,
                       NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                       NULL, a->context);
      ssh_xfree(a);
      return;
    }

  SSH_DEBUG(5, ("connection to real agent established"));

  /* Increment the number of channels. */
  ssh_common_new_channel(a->common);

  if (a->common->config->ssh_agent_compat == SSH_AGENT_COMPAT_SSH2)
    {
      /* We are required to send a FORWARDING_NOTICE to the agent to inform it
         that the connection is actually forwarded.  Format that packet now. */
      buffer = ssh_xbuffer_allocate();
      ssh_encode_buffer(buffer,
                        SSH_FORMAT_DATA, "1234", (size_t)4,
                        SSH_FORMAT_CHAR,
                        (unsigned int) SSH_AGENT_FORWARDING_NOTICE,
                        SSH_FORMAT_UINT32_STR,
                          a->common->server_host_name,
                          strlen(a->common->server_host_name),
                        SSH_FORMAT_UINT32_STR,
                          a->common->remote_ip, strlen(a->common->remote_ip),
                        SSH_FORMAT_UINT32,
                        (SshUInt32) atol(a->common->remote_port),
                        SSH_FORMAT_END);
      cp = ssh_buffer_ptr(buffer);
      SSH_PUT_32BIT(cp, ssh_buffer_len(buffer) - 4);

      /* Create a filter stream for inserting the message into the channel. */
      filter_stream =
        ssh_stream_filter_create(stream, 10240,
                                 ssh_agent_ssh1_forwarding_notice_insert_filter,
                                 NULL_FNPTR, 
                                 ssh_agent_ssh1_forwarding_filter_destroy, 
                                 buffer);
    }

  /* Create the channel. */
  (*a->completion)(SSH_OPEN_OK, stream, TRUE, TRUE, AGENT_WINDOW_SIZE, NULL, 0,
                   NULL_FNPTR, ssh_channel_ssh1_agent_connection_destroy,
                   (void *)a->common, a->context);
  ssh_xfree(a);
}

/***********************************************************************
 * Receiving an open request for an agent channel.  This call typically
 * happens in the SSH client, and this will contact the local real
 * authentication agent.
 ***********************************************************************/

/* Processes an open request for an agent channel. */

void ssh_channel_ssh1_agent_open(const char *type, int channel_id,
                                 const unsigned char *data, size_t len,



                                 SshConnOpenCompletionProc completion,

                                 void *completion_context, void *context)
{
  SshCommon common = (SshCommon)context;
  SshChannelTypeSsh1Agent ct;
  SshAgentConnection a;

  SSH_DEBUG(5, ("agent channel open request received"));

  ct = ssh_channel_ssh1_agent_ct(common);
  
  if (len != 0)
    {
      SSH_DEBUG(0, ("Bad agent channel open request"));
      (*completion)(SSH_OPEN_CONNECT_FAILED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }

  /* Do not allow agent opens at the server. */
  if (!common->client)
    {
      ssh_warning("Refused attempted agent connection to server.");
      (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }

  /* Do not allow agent opens if we didn't request agent forwarding. */
  if (!ct->agent_requested)
    {
      ssh_warning("Refused attempted agent connection when forwarding not requested.");
      (*completion)(SSH_OPEN_ADMINISTRATIVELY_PROHIBITED,
                    NULL, FALSE, FALSE, 0, NULL, 0, NULL_FNPTR, NULL_FNPTR,
                    NULL, completion_context);
      return;
    }
  
  /* Create a context argument for connecting. */
  a = ssh_xcalloc(1, sizeof(*a));
  a->common = common;
  a->channel_id = channel_id;
  a->completion = completion;
  a->context = completion_context;

  /* Try to connect to the real agent. */

  ssh_agenti_connect(ssh_channel_open_ssh1_agent_connected, TRUE, (void *)a);








}

/***********************************************************************
 * Processing a request to start agent forwarding at server end.
 ***********************************************************************/

/* Processes a received agent forwarding request.  This creates a listener
   at the server end for incoming agent connections. */

Boolean ssh_channel_ssh1_agent_process_request(void *session_placeholder,
                                              const unsigned char *data,
                                              size_t len)
{
  SshChannelTypeSsh1Agent ct;
  SshChannelSsh1AgentSession session;

  SSH_DEBUG(5, ("request to start ssh1 agent forwarding received"));

  session = (SshChannelSsh1AgentSession)session_placeholder;

  if (!session->common->config->agent_forwarding)
    {
      ssh_log_event(session->common->config->log_facility,
                    SSH_LOG_INFORMATIONAL,
                    "SSH1 agent forwarding request "
                    "denied in configuration.");
      return FALSE;
    }

  ct = ssh_channel_ssh1_agent_ct(session->common);
  
  if (ct->agent_path != NULL)
    return TRUE; /* We've alread created a fake listener. */


  ct->agent_listener =
    ssh_agenti_create_listener(ssh_user_uid(ct->common->user_data),
                               &ct->agent_path, 
                               ssh_channel_ssh1_agent_connection,
                               TRUE,
                               ct->common);
  return TRUE;



}

/* Sending a request to start authentication agent forwarding. */

void ssh_channel_ssh1_agent_send_request(SshCommon common, 
                                        int session_channel_id)
{
  SshChannelTypeSsh1Agent ct;
  
  SSH_DEBUG(5, ("sending request to start ssh1 agent forwarding"));

  ct = ssh_channel_ssh1_agent_ct(common);






  ssh_conn_send_channel_request(common->conn, session_channel_id,
                                "auth-ssh1-agent-req", NULL_FNPTR, 0,
                                NULL_FNPTR, NULL);


  ct->agent_requested = TRUE;
}

/* Returns the authentication agent path, or NULL if agent is not set.  The
   returned value will remain valid until the SshCommon object is destroyed. */

const char *ssh_channel_ssh1_agent_get_path(SshCommon common)
{
  SshChannelTypeSsh1Agent ct;
  
  ct = ssh_channel_ssh1_agent_ct(common);
  return ct->agent_path;
}

#endif /* SSH_CHANNEL_SSH1_AGENT */
