/*

  ssh2.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/
/*
  Exit values:

    Exits because of fatal signal:

    128 + signal number (e.g. 139 for SIGSEGV)

#define SSH_DISCONNECT_HOST_NOT_ALLOWED_TO_CONNECT      1
#define SSH_DISCONNECT_PROTOCOL_ERROR                   2
#define SSH_DISCONNECT_KEY_EXCHANGE_FAILED              3
#define SSH_DISCONNECT_RESERVED                         4
#define SSH_DISCONNECT_MAC_ERROR                        5
#define SSH_DISCONNECT_COMPRESSION_ERROR                6
#define SSH_DISCONNECT_SERVICE_NOT_AVAILABLE            7
#define SSH_DISCONNECT_PROTOCOL_VERSION_NOT_SUPPORTED   8
#define SSH_DISCONNECT_HOST_KEY_NOT_VERIFIABLE          9
#define SSH_DISCONNECT_CONNECTION_LOST                 10
#define SSH_DISCONNECT_BY_APPLICATION                  11
#define SSH_DISCONNECT_TOO_MANY_CONNECTIONS            12
#define SSH_DISCONNECT_AUTH_CANCELLED_BY_USER          13
#define SSH_DISCONNECT_NO_MORE_AUTH_METHODS_AVAILABLE  14
#define SSH_DISCONNECT_ILLEGAL_USER_NAME               15

    64 + disconnect code (e.g. 66 for PROTOCOL_ERROR)

    255 ssh_fatal
    254 failed to exec something (generic catch-all in the libraries for
        failures to fork or exec).

    1   generic error, usually because invalid command line options or
        malformed configuration.
    2   connecting host failed
*/


















#include "ssh2includes.h"

#include "sshunixptystream.h"
#include "sshtty.h"
#include "sshsignals.h"





#include "sshclient.h"
#include "sshtimeouts.h"
#include "sshfilterstream.h"
#include "sshtcp.h"
#include "sshfdstream.h"
#include "sshcrypt.h"
#include "sshbuffer.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshconfig.h"
#include "sshuserfiles.h"
#include "ssheloop.h"
#include "sshstdiofilter.h"
#include "sshgetopt.h"
#include "sshmiscstring.h"
#include "sshappcommon.h"
#include "sshdsprintf.h"

#ifndef VXWORKS
#include <syslog.h>
#endif /* VXWORKS */
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */











#include "sshglobals.h"



#define SSH_DEBUG_MODULE "Ssh2"

/* Define this if you want ssh2 to sleep() 30 seconds after startup
   (for debugging purposes. For example scp2 etc.) */
/* #define SLEEP_AFTER_STARTUP */

/* Define this if you want the child process to suspend after fork()
   when ssh2 is invoked with the '-f' parameter. Again, this is purely
   for debugging purposes. */
/* #define SUSPEND_AFTER_FORK */









#ifdef HAVE_LIBWRAP
/* These are needed, because admins might want to restrict connections
   to forwarded ports with tcp wrappers. */
#include <tcpd.h>
int allow_severity = SSH_LOG_INFORMATIONAL;
int deny_severity = SSH_LOG_WARNING;
#endif /* HAVE_LIBWRAP */

/* Program name, without path. */
const char *av0;


/* The callback context to pass to ssh_tcp_connect */
typedef struct SshTcpConnectContextRec
{
  SshClientData data;
  char *host;
  SshTcpConnectParamsStruct* tcp_connect_params;
  SshADTContainer socks_server_list;
} *SshTcpConnectContext;

/* The callback context to pass to connect_done_finalize */
typedef struct SshConnectDoneFinalizeCtxRec
{
  SshClientData data;
  SshStream stream;
} *SshConnectDoneFinalizeCtx;

void client_disconnect(int reason, Boolean locally_generated,
                       const char *msg, void *context)
{
  SshClientData data = (SshClientData)context;
  
  SSH_TRACE(1, ("locally_generated = %s",
                locally_generated ? "TRUE" : "FALSE"));
#define PRINT_DISCONNECT_MSG(app_msg, msg)                      \
  do {                                                          \
    ssh_informational("Disconnected; " app_msg "%s%s%s.\r\n",   \
                      (((msg) && (msg)[0]) ? " (" : ""),        \
                      (((msg) && (msg)[0]) ? (msg)  : ""),      \
                      (((msg) && (msg)[0]) ? ")"  : ""));       \
  } while (0)

  switch(reason)
    {
    case SSH_DISCONNECT_CONNECTION_LOST:
      PRINT_DISCONNECT_MSG("connection lost", msg);
      break;
    case SSH_DISCONNECT_BY_APPLICATION:
      PRINT_DISCONNECT_MSG("by application", msg);
      break;
    case SSH_DISCONNECT_PROTOCOL_ERROR:
      PRINT_DISCONNECT_MSG("protocol error", msg);
      break;
    case SSH_DISCONNECT_SERVICE_NOT_AVAILABLE:
      PRINT_DISCONNECT_MSG("service not available", msg);
      break;
    case SSH_DISCONNECT_MAC_ERROR:
      PRINT_DISCONNECT_MSG("MAC error", msg);
      break;
    case SSH_DISCONNECT_COMPRESSION_ERROR:
      PRINT_DISCONNECT_MSG("compression error", msg);
      break;
    case SSH_DISCONNECT_HOST_NOT_ALLOWED_TO_CONNECT:
      PRINT_DISCONNECT_MSG("host not allowed to connect", msg);
      break;
    case SSH_DISCONNECT_RESERVED:
      if (data->client && data->client->common &&
          !(*data->client->common->compat_flags->
            deprecated_disconnect_codes_draft_incompatibility))
        {
          PRINT_DISCONNECT_MSG("SSH_DISCONNECT_RESERVED; this is an illegal "
                               "code", msg);
        }
      else
        {
          PRINT_DISCONNECT_MSG("host authentication failed", msg);
        }
      break;
    case SSH_DISCONNECT_PROTOCOL_VERSION_NOT_SUPPORTED:
      PRINT_DISCONNECT_MSG("protocol version not supported", msg);
      break;
    case SSH_DISCONNECT_HOST_KEY_NOT_VERIFIABLE:
      PRINT_DISCONNECT_MSG("host key not verifiable", msg);
      break;
    case SSH_DISCONNECT_TOO_MANY_CONNECTIONS:
      if (data->client && data->client->common &&
          !(*data->client->common->compat_flags->
            deprecated_disconnect_codes_draft_incompatibility))
        {
          PRINT_DISCONNECT_MSG("too many connections", msg);
        }
      else
        {
          /* This is not a valid reason code according to draft. We
             process this anyway, because ssh-versions ssh-2.0.13 and
             older sent this. (It was
             SSH_DISCONNECT_AUTHENTICATION_ERROR.) */
          SSH_TRACE(1, ("Received disconnection reason code 12, which "
                        "does not conform with the draft."));
          PRINT_DISCONNECT_MSG("authentication error", msg);
        }
      break;
    case SSH_DISCONNECT_KEY_EXCHANGE_FAILED:
      PRINT_DISCONNECT_MSG("key exchange or algorithm negotiation failed",
                           msg);
      break;
    case SSH_DISCONNECT_AUTH_CANCELLED_BY_USER:
      PRINT_DISCONNECT_MSG("authentication cancelled by user", msg);
      break;
    case SSH_DISCONNECT_NO_MORE_AUTH_METHODS_AVAILABLE:
      PRINT_DISCONNECT_MSG("no more authentication methods available", msg);
      break;
    case SSH_DISCONNECT_ILLEGAL_USER_NAME:
      PRINT_DISCONNECT_MSG("illegal user name", msg);
      break;
    default:
      ssh_informational("Disconnected; unknown disconnect code %d%s%s%s.\r\n",
                        reason,
                        ((msg && msg[0]) ? " (" : ""),
                        ((msg && msg[0]) ? msg  : ""),
                        ((msg && msg[0]) ? ")"  : ""));
      break;
    }
#undef PRINT_DISCONNECT_MSG
  if (data->client)
    {
      ssh_client_destroy(data->client);
      data->client = NULL;
    }
  data->exit_status = 64 + reason;
}

static int client_debug_output(FILE *stream, const char *format, ...)
{
  int bytes, flags, fd;
  va_list ap;


  /* stream might be in non-blocking mode, so we set it to blocking for
     as long as we write our message. */
  fd = fileno(stream);
  flags = fcntl(fd, F_GETFL, 0);
#ifdef O_ASYNC
  fcntl(fd, F_SETFL, flags & ~(O_ASYNC | O_NONBLOCK));
#else /* O_ASYNC */
  fcntl(fd, F_SETFL, flags & ~(O_NONBLOCK));
#endif /* O_ASYNC */

  va_start(ap, format);
  bytes = vfprintf(stream, format, ap);
  va_end(ap);

  fcntl(fd, F_SETFL, flags);

  return bytes;
}

void client_debug(int type, const char *msg, void *context)
{
  SshClientData data = (SshClientData)context;

  if (data->config->quiet_mode)
    return;

  switch (type)
    {
    case SSH_DEBUG_DEBUG:
      if (data->debug)
        client_debug_output(stderr, "%s\r\n", msg);
      break;

    case SSH_DEBUG_DISPLAY:
      client_debug_output(stderr, "%s\r\n", msg);
      break;

    default:
      client_debug_output(stderr, "UNKNOWN DEBUG DATA TYPE %d: %s\r\n",
                          type, msg);
      break;
    }
  clearerr(stderr); /*XXX*/
}

void client_banner_msg(const char *msg, void *context)
{
  client_debug(SSH_DEBUG_DISPLAY, msg, context);
}

void client_ssh_debug(const char *msg, void *context)
{
  SshClientData data = (SshClientData)context;
  FILE *fp;

  if (data->config->debug_log_file_name &&
      strcmp(data->config->debug_log_file_name, "") != 0)
    {
      fp = fopen(data->config->debug_log_file_name, "a");
      if (fp == NULL)
        {
          fprintf(stderr, "Couldn't open debug log file `%s' for writing.",
                  data->config->debug_log_file_name);
          ssh_xfree(data->config->debug_log_file_name);
          data->config->debug_log_file_name = NULL;
        }
      else
        {
          fprintf(fp, "debug: %s\n", msg);
          fclose(fp);
        }
    }

  if (data->config->quiet_mode)
    return;

  if (data->debug)
    client_debug_output(stderr, "debug: %s\r\n", msg);

  clearerr(stderr); /*XXX*/
}

void client_ssh_warning(const char *msg, void *context)
{
  SshClientData data = (SshClientData)context;
  if (data->config->quiet_mode)
    return;

  client_debug_output(stderr, "warning: %s\r\n", msg);
}

void client_ssh_fatal(const char *msg, void *context)
{
  static Boolean tried = FALSE;
  SshClientData data = (SshClientData)context;

  /* This is a sanity check, so that no unitilization funcs, which
     could cause a SIGSEGV etc., cause an infinite loop. */
  if (tried)
    goto loop_detected;
  else
    tried = TRUE;

  client_debug_output(stderr, "%s: FATAL: %s\r\n", av0, msg);

  if (data && data->client && data->client->rl)
    {
      ssh_readline_eloop_uninitialize(data->client->rl);
      data->client->rl = NULL;
    }
  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);

 loop_detected:
  exit(255);
}

int client_ssh_informational(const char *msg, void *context)
{
  SshClientData data = (SshClientData)context;

  if (data->config->quiet_mode)
    return 0;

  return client_debug_output(stderr, "%s", msg);
}

#ifdef MEMLEAKS_AT_SIGINT
void memleak_sigint_handler(int sig, void *context)
{





}
#endif /* MEMLEAKS_AT_SIGINT */


void ssh_fatal_signal_handler(int sig, void *context)
{
  SshClientData data = (SshClientData)context;

  client_debug_output(stderr, "Received signal %d. (no core)\r\n", sig);
  if (data && data->client && data->client->rl)
    {
      ssh_readline_eloop_uninitialize(data->client->rl);
      data->client->rl = NULL;
    }

  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);

  /* To get a meaningful exit value. We can't use exit(255-sig), because
     SIGHUP (=1) would have same return value as lot of catch-22:s in
     the code. */
  exit(sig + 128);
}

void sigpipe_nonfatal_cb(int sig, void *context)
{
  SshClientData data = (SshClientData)context;

  ssh_unregister_signal(SIGPIPE);

  SSH_TRACE(2, ("Received SIGPIPE."));

  if (data->client)
    {
      if (data->client->rl)
        {
          SSH_DEBUG(2, ("Uninitializing readline..."));
          ssh_readline_eloop_uninitialize(data->client->rl);
          data->client->rl = NULL;
        }
      SSH_DEBUG(2, ("Destroying client..."));
      ssh_client_destroy(data->client);
      data->client = NULL;
    }
  data->exit_status = 0;
}


void session_close(SshUInt32 exit_status, void *context)
{
  SshClientData data = (void *)context;
  int ret = 0;
  char *remote_host = NULL;
  SshCommon common = data->client->common;

  /* We save the number of channels, because if num_channels is 0 we
     eventually destroy the common structure, and using
     common->num_channels later would be an error. */
  unsigned int num_channels = common->num_channels;

  data->exit_status = exit_status;
  if (common->server_host_name)
    remote_host = ssh_xstrdup(common->server_host_name);

  ssh_debug("Got session close with exit_status=%d", (int)exit_status);

  if (num_channels == 0)
    {
      if (data->client)
        {
          ssh_debug("destroying client struct...");
          ssh_client_destroy(data->client);
          data->client = NULL;
        }
    }


  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);

  /* If there are forwarded channels open, we fork to background to wait
     for them to complete. */
  if (data->config->go_background &&
      (data->config->local_forwards || data->config->remote_forwards))
    {
      /* XXX if all local or remote forwardings have failed, we should
         exit. */
      ssh_warning("ssh2[%d]: waiting for connections to forwarded "
                  "ports in the background.",
                  getpid());
      return;
    }

  if (num_channels != 0)
    {
      ssh_debug("Forking... parent pid = %d", getpid());

      ret = fork();
      if (ret == -1)
        {
          ssh_warning("Fork failed.");
        }
      else if (ret != 0)
        {
          if (data && data->client && data->client->rl)
            {
              ssh_readline_eloop_uninitialize(data->client->rl);
              data->client->rl = NULL;
            }
          ssh_leave_non_blocking(-1);
          ssh_leave_raw_mode(-1);
          exit(0);
        }
      ssh_debug("num_channels now %d", common->num_channels);
#ifdef HAVE_DAEMON
      if (daemon(0, 1) < 0)
        ssh_fatal("daemon(): %.100s", strerror(errno));
#else /* HAVE_DAEMON */
#ifdef HAVE_SETSID
#ifdef ultrix
      setpgrp(0, 0);
#else /* ultrix */
      if (setsid() < 0)
        ssh_warning("setsid: %.100s", strerror(errno));
#endif /* ultrix */
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON*/
      ssh_warning("ssh2[%d]: number of forwarded channels still "
                  "open, forked to background to wait for completion.",
                  getpid());
    }
  else
    {
      if (remote_host && isatty(fileno(stdout)) &&
          !data->command && !data->is_subsystem)
        {
          ssh_informational("Connection to %s closed.\n", remote_host);
          ssh_xfree(remote_host);
        }
    }

















}






















void ssh_stream_sink_filter(void *context,
                            SshFilterGetCB get_data,
                            SshFilterCompletionCB completed,
                            void *internal_context)
{
  size_t offset, received_len;
  SshBuffer data;
  Boolean eof_received;

  (*get_data)(internal_context, &data, &offset, &eof_received);

  received_len = ssh_buffer_len(data) - offset;
  ssh_buffer_consume(data, received_len);

  (*completed)(internal_context, SSH_FILTER_HOLD);
}

void ssh_stream_sink_filter_destroy(void *context)
{
  ssh_leave_raw_mode(-1);
  return;
}



/* Stream creator func, which the client-interface calls, if
   authentication is successful before calling the notify
   callback. Or in the case of ssh1-emulation code, after successful
   authentication. */
void client_stdio_stream_wrapper(Boolean ssh1_compat,
                                 SshClient client)
{

  ssh_enter_non_blocking(-1);
  if (ssh1_compat || client->common->no_session_channel == FALSE)
    {
      /* Create streams for stdio/stderr. */
      /* XXX */
      if ((*client->config->escape_char != '\0') && isatty(fileno(stdin)))
        {
          client->stdio_stream =
            ssh_stream_filter_create(client->stdio_stream,
                                     1024,
                                     ssh_stdio_output_filter,
                                     ssh_stdio_input_filter,
                                     ssh_stdio_filter_destroy,
                                     (void *)client);
        }

      client->stderr_stream = ssh_stream_fd_wrap2(-1, 2, FALSE);
    }
  else
    {
      client->stdio_stream =
        ssh_stream_filter_create(client->stdio_stream,
                                 1024,
                                 ssh_stdio_output_filter,
                                 ssh_stdio_input_filter,
                                 ssh_stdio_filter_destroy,
                                 (void *)client);
      client->stdio_stream =
        ssh_stream_filter_create(client->stdio_stream,
                                 1024,
                                 ssh_stream_sink_filter,
                                 ssh_stream_sink_filter,
                                 ssh_stream_sink_filter_destroy,
                                 NULL);
      client->stderr_stream = NULL;
      ssh_enter_raw_mode(-1, FALSE);
    }





















}

#ifdef SSH_CHANNEL_TCPFWD
void remote_forward_completion(Boolean success, void *context)
{
  SshForward fwd = (SshForward)context;

  if (!success)
    ssh_warning("Remote forward %s:%s:%s failed. Operation was denied by "
                "the server.", fwd->port, fwd->connect_to_host,
                fwd->connect_to_port);
  else
    SSH_TRACE(2, ("Remote forward request %s:%s:%s succeeded.",
                  fwd->port, fwd->connect_to_host, fwd->connect_to_port));
}
#endif /* SSH_CHANNEL_TCPFWD */

void client_authentication_notify(const char *user, Boolean successful,
                                  void *context)
{
  int ret = 0;
  SshClientData data = (SshClientData)context;

#ifdef SSH_CHANNEL_TCPFWD
  SshForward fwd;
#endif /* SSH_CHANNEL_TCPFWD */


  if (data->client->rl)
    {
      ssh_readline_eloop_uninitialize(data->client->rl);
      data->client->rl = NULL;
      if (data->user_input_stream &&
          data->user_input_stream != data->main_stdio_stream)
        {
          SSH_TRACE(2, ("Returning user input stream to original values."));

          ssh_leave_non_blocking(ssh_stream_fd_get_readfd
                                 (data->user_input_stream));
          ssh_leave_raw_mode(ssh_stream_fd_get_readfd
                             (data->user_input_stream));
          ssh_stream_destroy(data->user_input_stream);
          data->user_input_stream = NULL;
        }
    }


  if (data->config->authentication_notify)
    {
      ssh_client_parent_notify_authentication(successful,
                                              data->client->stdio_stream);
    }

  if (data->config->dont_read_stdin)
    {
      freopen("/dev/null", "r", stdin);
    }

  if (!successful)
    {
      ssh_warning("Authentication failed.");
    }
  else
    {
      if (isatty(fileno(stdout)) && data->config->auth_success_msg)
        ssh_informational("Authentication successful.\r\n");


      /* If we are requested to go to background, do it now. */
      if (data->config->go_background)
        {
          ret = fork();
          if (ret == -1)
            {
              ssh_warning("Fork failed.");
            }
          else if (ret != 0)
            {
              if (data->main_stdio_stream)
                {
                  ssh_stream_destroy(data->main_stdio_stream);
                  data->main_stdio_stream = NULL;
                }
              exit(0);
            }

#ifdef HAVE_DAEMON
          if (daemon(0, 1) < 0)
            ssh_fatal("daemon(): %.100s", strerror(errno));
#else /* HAVE_DAEMON */
# ifdef HAVE_SETSID
#  ifdef ultrix
          setpgrp(0, 0);
#  else /* ultrix */
          if (setsid() < 0)
            ssh_warning("setsid: %.100s", strerror(errno));
#  endif /* ultrix */
# endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON*/
        }

#ifdef SUSPEND_AFTER_FORK
      kill(getpid(), SIGSTOP);
#endif /* SUSPEND_AFTER_FORK */



#ifdef SSH_CHANNEL_TCPFWD
      for (fwd = data->config->local_forwards; fwd; fwd = fwd->next)
        if (!ssh_client_local_tcp_ip_forward(data->client,
                                             data->config->gateway_ports ?
                                             /* XXX IPv6? */
                                             fwd->local_addr : "127.0.0.1",
                                             fwd->port, fwd->connect_to_host,
                                             fwd->connect_to_port,
                                             fwd->protocol))
          ssh_warning("Local TCP/IP forwarding for port %s failed.",
                      fwd->port);

      for (fwd = data->config->remote_forwards; fwd; fwd = fwd->next)
        {
          char *bind_addr;
          if (strcmp(fwd->local_addr, SSH_IPADDR_ANY) == 0)
            bind_addr = "0.0.0.0";
          else
            bind_addr = fwd->local_addr;
          ssh_client_remote_tcp_ip_forward(data->client,
                                           bind_addr,
                                           fwd->port, fwd->connect_to_host,
                                           fwd->connect_to_port,
                                           fwd->protocol,
                                           remote_forward_completion,
                                           fwd);
        }
#endif /* SSH_CHANNEL_TCPFWD */

      SSH_TRACE(3, ("forward_x11 = %s, forward_agent = %s",
                    data->config->x11_forwarding ? "TRUE" : "FALSE",
                    data->config->agent_forwarding ? "TRUE" : "FALSE"));

      if ((!data->command && !data->allocate_pty) &&
          data->client->stdio_stream)
        data->client->stdio_stream = NULL;

      if (!(data->no_session_channel && !data->command))
        {
          ssh_client_start_session(data->client,
                                   data->client->stdio_stream,
                                   data->client->stderr_stream,
                                   TRUE,
                                   data->is_subsystem,
                                   data->command,
                                   data->allocate_pty,
                                   data->term,
                                   data->config->remote_env,
                                   data->config->x11_forwarding,
                                   data->config->agent_forwarding,

                                   NULL_FNPTR,



                                   session_close,
                                   (void *)data);
          /* So that we don't touch this anymore, it is now handled by
             session channel code. */
          data->config->remote_env = NULL;
        }
    }
}





































void connect_done_finalize(void *context)

{
  SshConnectDoneFinalizeCtx ctx = (SshConnectDoneFinalizeCtx) context;

  /* Start the client */



















  ctx->data->client = ssh_client_wrap(ctx->stream, ctx->data->config,
                                      ctx->data->user_data,
                                      ctx->data->config->host_to_connect,
                                      ctx->data->config->login_as_user,
                                      ctx->data->command,
                                      ctx->data->term,
                                      client_stdio_stream_wrapper,
                                      ctx->data->main_stdio_stream,
                                      ctx->data->allocate_pty,
                                      client_disconnect,
                                      client_debug,
                                      client_banner_msg,
                                      client_authentication_notify,
                                      session_close,
                                      (void *)ctx->data);

























  if (ctx->data->config->batch_mode == FALSE)
    {
      if (isatty(fileno(stdin)) && !ctx->data->config->dont_read_stdin)
        {
          ctx->data->user_input_stream = ctx->data->main_stdio_stream;
        }
      else
        {
          FILE *fi;

          SSH_TRACE(2, ("Opening /dev/tty for queries."));
          if ((fi = fopen("/dev/tty", "r")) != NULL)
            {
              ctx->data->user_input_stream =
                ssh_stream_fd_wrap2(fileno(fi), dup(fileno(stderr)), TRUE);
            }
          else
            {
              ssh_debug("No controlling terminal. Can't initialize "
                        "readline for confirmations.");
            }
        }

      if (ctx->data->user_input_stream &&
          (ctx->data->client->rl = ssh_readline_eloop_initialize
           (ctx->data->user_input_stream)) == NULL)
        {
          fprintf(stderr, "Couldn't initialize readline. Maybe termcap "
                  "entries are wrong or missing.\nPlease contact your system "
                  "administrator.\n");
        }
    }


  /* This is done, because in ssh_common_* functions we don't know anything
     about the SshClient* structures. no_session_channel's value must
     however be known there.*/
  ctx->data->client->common->no_session_channel =
    ctx->data->no_session_channel;






  ssh_xfree(ctx);
}

void connect_done(SshTcpError error, SshStream stream, void *context)
{
  SshTcpConnectContext connect_context = (SshTcpConnectContext)context;
  SshClientData data = connect_context->data;
  SshConnectDoneFinalizeCtx finalize_ctx;






  if (error != SSH_TCP_OK)
    {

      ssh_warning("Connecting to %s failed: %s",
                  data->config->host_to_connect, ssh_tcp_error_string(error));
      exit(2);





































    }











  /* Set socket to nodelay mode if configuration suggests this. */
  ssh_tcp_set_nodelay(stream, data->config->no_delay);
  /* Set socket to keepalive mode if configuration suggests this. */
  ssh_tcp_set_keepalive(stream, data->config->keep_alive);
  /* Set socket to linger mode. */
  ssh_tcp_set_linger(stream, TRUE);


  /* Save the file descriptor for ssh1 compatibility code. */
  data->config->ssh1_fd = ssh_stream_fd_get_readfd(stream);

  /* Background process, doesn't allocate pty and doesn't read stdin. */
  if (data->config->go_background)
    {
      data->no_session_channel = TRUE;
      data->allocate_pty = FALSE;
      data->config->dont_read_stdin = TRUE;
    }

  /* Create main stdio-stream here, wrap it later. (needed with
     ssh_readline_eloop()) */
  data->main_stdio_stream = ssh_stream_fd_stdio();


  finalize_ctx = ssh_xcalloc(1, sizeof(struct SshConnectDoneFinalizeCtxRec));
  finalize_ctx->data = data;
  finalize_ctx->stream = stream;






  connect_done_finalize(finalize_ctx);

}

static void finalize_password_prompt(char **prompt, char *host, char *user)
{
  char *tmp;

  tmp = ssh_replace_in_string(*prompt, "%H", host);
  ssh_xfree(*prompt);
  *prompt = tmp;
  tmp = ssh_replace_in_string(*prompt, "%U", user);
  ssh_xfree(*prompt);
  *prompt = tmp;
}




void ssh2_help(const char *name)

{



  char *ciphs, *macs;
  
  ssh2_version(name);
#define INFO ssh_informational
  INFO("Copyright (c) 1995-2002 SSH Communications Security Corp\n");
  INFO("SSH is a registered trademark and Secure Shell is a trademark of\n"
       "SSH Communications Security Corp (www.ssh.com).\n");

  INFO("All rights reserved.  See LICENSE file for usage and distribution "
       "terms.\n");



  INFO("\n");
  INFO("Usage: %s [options] [user@]host[#port] [command]\n", name);
  INFO("\n");
  INFO("Options:\n");
  INFO("  -l login_name  Log in using this user name.\n");

  INFO("  -n             Redirect input from /dev/null.\n");

  INFO("  +a             Enable authentication agent forwarding.\n");
  INFO("  -a             Disable authentication agent forwarding.\n");
  INFO("  +x             Enable X11 connection forwarding (treat X11 clients "
       "as\n"
       "                 UNTRUSTED).\n");
  INFO("  +X             Enable X11 connection forwarding (treat X11 clients "
       "as\n"
       "                 TRUSTED).\n");
  INFO("  -x             Disable X11 connection forwarding.\n");
  INFO("  -i file        Identity file for public key authentication\n");
  INFO("  -F file        Read an alternative configuration file.\n");
  INFO("  -t             Tty; allocate a tty even if command is given.\n");
  INFO("  -v             Verbose; display verbose debugging messages. Equal "
       "to '-d 2'\n");
  INFO("  -d level       Set debug level.\n");
  INFO("  -V             Display version string.\n");
  INFO("  -q             Quiet; don't display any warning messages.\n");

  INFO("  -f[o]          Fork into background after authentication.\n");
  INFO("                 With optional 'o' argument, goes to \"one-shot\" "
       "mode.\n");
  INFO("  -e char        Set escape character; 'none' = disable "
       "(default: ~).\n");

  INFO("  -c cipher      Select encryption algorithm. Multiple -c options "
       "are \n"
       "                 allowed and a single -c flag can have only one "
       "cipher.\n");
  INFO("  -m MAC         Select MAC algorithm. Multiple -m options are \n"
       "                 allowed and a single -m flag can have only one "
       "MAC.\n");
  INFO("  -p port        Connect to this port.  Server must be on the same "
       "port.\n");
  INFO("  -S             Don't request a session channel. \n");
  INFO("  -L listen-port:host:port   Forward local port to remote address\n");
  INFO("  -R listen-port:host:port   Forward remote port to local address\n");
  INFO("                 These cause ssh to listen for connections on a port, "
       "and\n"
       "                 forward them to the other side by connecting to "
       "host:port.\n");
  INFO("  -g             Gateway ports, i.e. remote hosts may connect to "
       "locally\n"
       "                 forwarded ports.\n");
  INFO("  +g             Don't gateway ports.\n");
  INFO("  +C             Enable compression.\n");
  INFO("  -C             Disable compression.\n");

  INFO("  -4             Use IPv4 to connect.\n");
  INFO("  -6             Use IPv6 to connect.\n");







  INFO("  -o 'option'    Process the option as if it was read from a "
       "configuration\n"
       "                 file.\n");
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION

#ifdef WITH_INTERNAL_SSH1_EMULATION
  INFO("  -1[ti]         Choose ssh1-protocol fallback type. Mandatory argument. "
       "'i'\n"
       "                 means internal emulation, 't' means traditional "
       "(call ssh1\n"
       "                 executable).\n");
#endif /* WITH_INTERNAL_SSH1_EMULATION */

#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
  INFO("  -h             Display this help.\n");
  INFO("\n");
  INFO("Command can be either:\n");
  INFO("  remote_command [arguments] ...    Run command in remote host.\n");
  INFO("  -s service                        Enable a service in remote "
       "server.\n");
  INFO("\n");











  ciphs = ssh_cipher_get_supported_native();
  INFO("Supported ciphers:\n");
  INFO("  %s\n", ciphs);
  ssh_xfree(ciphs);
            
  macs = ssh_mac_get_supported();
  INFO("Supported MAC algorithms:\n");
  INFO("  %s\n", macs);
  ssh_xfree(macs);



























}

/*
 *  This function digs out the first non-option parameter, ie. the host to
 * connect to.
 */
char *ssh_get_host_name_from_command_line(int argc, char **argv)
{
  struct SshGetOptDataRec getopt_data;

  ssh_getopt_init_data(&getopt_data);
  getopt_data.reset = 1;
  getopt_data.allow_plus = 1;
  getopt_data.err = 0;

  while (ssh_getopt(argc, argv, SSH2_GETOPT_ARGUMENTS, &getopt_data) != -1)
    /*NOTHING*/;
  if ((argc <= getopt_data.ind) || (argv[getopt_data.ind] == NULL))
    return NULL;
  else
    return ssh_xstrdup(argv[getopt_data.ind]);
}


/*
 *
 *  SSH2 main
 *
 */
int main(int argc, char **argv)
{
  int i = 0;
  char *host = NULL, *user = NULL, *userdir = NULL, *socks_server = NULL,
    *command = NULL, *port = NULL;
  SshClientData data;
  SshUser tuser;
  char temp_s[1024];
  Boolean have_c_arg, have_m_arg;
  SshTcpConnectParamsStruct tcp_connect_params;
  int exit_status = 0;
  SshTcpConnectContext connect_context;
  struct stat st;















  /* The metaconfig used with command-line arguments. */
  SshMetaConfigStruct metaconfig;

  metaconfig.version_major = 1;
  metaconfig.version_minor = 1;
  metaconfig.regex_syntax = SSH_REGEX_SYNTAX_EGREP;







#ifdef SLEEP_AFTER_STARTUP
  sleep(30);
#endif /* SLEEP_AFTER_STARTUP */

  have_c_arg = FALSE;
  have_m_arg = FALSE;

  /* Save program name. */
  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];

  ssh_register_program_name(av0);

  /* Register cryptographic providers. */
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  ssh_pk_provider_register(&ssh_pk_if_modn);
  /* Needed at least for pgp-keys. */
  ssh_pk_provider_register(&ssh_pk_if_modn_generator);
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */
#ifdef SSHDIST_CRYPT_DSA
  ssh_pk_provider_register(&ssh_pk_dl_modp);
  /* Needed at least for pgp-keys. */
  ssh_pk_provider_register(&ssh_pk_dl_modp_generator);
#endif /* SSHDIST_CRYPT_DSA */

  /* Initializations. */
  tuser = ssh_user_initialize(NULL, FALSE);
  data = ssh_xcalloc(1, sizeof(*data));
  ssh_event_loop_initialize();

  /* Initialize config with built-in defaults. */
  data->config = ssh_client_create_config();
  data->is_subsystem = FALSE;
  data->no_session_channel = FALSE;
  data->exit_status = 0;

  /* Save arguments for ssh1 compatibility. */
  data->config->ssh1_argv = argv;
  data->config->ssh1_argc = argc;

  /* Register debug, fatal, and warning callbacks. */
  ssh_debug_register_callbacks(client_ssh_fatal, client_ssh_warning,
                               client_ssh_debug, (void *)data);

  ssh_informational_callback_register(client_ssh_informational,
                                      (void *)data);

  /* If -d is the first flag, we set debug level here.  It is reset
     later, but something may be lost if we leave it 'til there. */
  if ((argc >= 3) && (strcmp("-d", argv[1]) == 0))
    {
      ssh_debug_set_level_string(argv[2]);
      if (strcmp("0", argv[2]) != 0)  /* (XXX: is there really a
                                         difference between '-d 0' and
                                         '-d"*=0"'?) */
        data->debug = TRUE;
      else
        data->debug = FALSE;
    }
  else if ((argc >= 2) && (strcmp("-v", argv[1]) == 0))
    {
      ssh_debug_set_level_string("2");
      data->debug = TRUE;
    }

  /* Only dump help if -h option is explicitly given (not with empty
     parameter list).  This way we have time to parse the license file
     first. */
  if (argc == 1)
    {
      ssh_warning("No arguments.");
      fprintf(stderr, "Type %s -h for help.\n", av0);
      exit(0);
    }


  /* Prevent core dumps from revealing sensitive information. */
  ssh_signals_prevent_core(TRUE, data);

  /* Register fatal signals to prevent messing up the terminal. */
  ssh_register_signal(SIGTERM, ssh_fatal_signal_handler, data);
  ssh_register_signal(SIGHUP, ssh_fatal_signal_handler, data);

  /* We need to register this, as otherwise readline would not be
     uninitialized, and the terminal would be left in raw mode. */
  ssh_register_signal(SIGPIPE, sigpipe_nonfatal_cb, data);
#ifdef TEST_TRANSFER_ABORT
  ssh_register_signal(SIGINT, NULL, NULL);
#endif /* TEST_TRANSFER_ABORT */


  ssh_global_init();




  host = ssh_get_host_name_from_command_line(argc, argv);
  if (host)
    {
      char *p;

      /* check whether form 'user@host' is used */
      if ((p = strchr(host, '@')) != NULL)
        {
          char *to_be_deleted;
          /* If so, cut string */
          *p = '\0';
          p++;

          user = ssh_xstrdup(host);
          /* make 'host' to point to the real hostname */
          to_be_deleted = host;
          host = ssh_xstrdup(p);
          ssh_xfree(to_be_deleted);
        }

      if ((port = strchr(host, '#')) != NULL)
        {
          *port = '\0';
          port++;
          port = ssh_xstrdup(port);
        }

      data->config->host_to_connect = ssh_xstrdup(host);
    }













































































































  /* Try to read host specific settings from the global configuration
     file */
  if ((stat(SSH_CLIENT_GLOBAL_CONFIG_FILE, &st) == 0) &&
      !ssh_config_read_file(tuser, data->config,
                            data->config->host_to_connect ?
                            data->config->host_to_connect : "",
                            SSH_CLIENT_GLOBAL_CONFIG_FILE,
                            SSH_CONFIG_TYPE_CLIENT))
    ssh_fatal("Error reading config file %s.",
              SSH_CLIENT_GLOBAL_CONFIG_FILE);

  /* Try to read in the user configuration file. */

  userdir = ssh_userdir(tuser, data->config, TRUE);
  if (userdir == NULL)
    {
      ssh_debug("Failed to create user ssh directory.");
    }
  else
    {
      ssh_snprintf(temp_s, sizeof(temp_s), "%s/%s",
                   userdir, SSH_CLIENT_CONFIG_FILE);
      ssh_xfree(userdir);

      ssh_config_read_file(tuser, data->config,
                           data->config->host_to_connect ?
                           data->config->host_to_connect : "",
                           temp_s, SSH_CONFIG_TYPE_CLIENT);
    }

  if (!user)
    {
      if (!data->config->login_as_user)
        data->config->login_as_user = ssh_xstrdup(ssh_user_name(tuser));
    }
  else
    {
      ssh_xfree(data->config->login_as_user);
      data->config->login_as_user = ssh_xstrdup(user);
      ssh_xfree(user);
    }

  ssh_xfree(host);
  host = NULL;
  ssh_opterr = 0;
  ssh_optallowplus = 1;

  /* Interpret the command line parameters. */
  while (1)
    {
      int option;

      option = ssh_getopt(argc, argv, SSH2_GETOPT_ARGUMENTS, NULL);

      if ((option == -1) && (host == NULL))
        {
          host = argv[ssh_optind];
          if (!host)
            {
              ssh_warning("You didn't specify a host name.");
              fprintf(stderr, "Type %s -h for help.\n", av0);
              exit(1);
            }
          ssh_optind++;
          SSH_DEBUG(3, ("remote host = \"%s\"", host));
          ssh_optreset = 1;
          option = ssh_getopt(argc, argv, SSH2_GETOPT_ARGUMENTS, NULL);
        }
      if (option == -1)
        {
          /* Rest ones are the command and arguments. */
          if (argc <= ssh_optind)
            {
              if (!(data->is_subsystem))
                {
                  command = NULL;
                }
            }
          else
            {
              if (data->is_subsystem)
                {
                  ssh_fatal("No command allowed with subsystem.");
                }
              command = ssh_xstrdup(argv[ssh_optind]);
              for (i = 1; i < (argc - ssh_optind); i++)
                {
                  char *newcommand;

                  newcommand = ssh_string_concat_3(command,
                                                   " ",
                                                   argv[ssh_optind + i]);
                  ssh_xfree(command);
                  command = newcommand;
                }
              SSH_DEBUG(3, ("remote command = \"%s\"", command));
              if (!(*command))
                {
                  /* Empty command string equals to no command at all. */
                  ssh_xfree(command);
                  command = NULL;
                }
            }
          break;
        }

      SSH_DEBUG(5, ("ssh_getopt(...) -> %d '%c'", option, option));
      SSH_DEBUG(5, (" ssh_opterr = %d", ssh_opterr));
      SSH_DEBUG(5, (" ssh_optind = %d", ssh_optind));
      SSH_DEBUG(5, (" ssh_optval = %d", ssh_optval));
      SSH_DEBUG(5, (" ssh_optopt = %d", ssh_optopt));
      SSH_DEBUG(5, (" ssh_optreset = %d", ssh_optreset));
      SSH_DEBUG(5, (" ssh_optarg = %p \"%s\"",
                    ssh_optarg, ssh_optarg ? ssh_optarg : "NULL"));
      SSH_DEBUG(5, (" ssh_optmissarg = %d", ssh_optmissarg));
      SSH_DEBUG(5, (" ssh_optargnum = %d", ssh_optargnum));
      SSH_DEBUG(5, (" ssh_optargval = %d", ssh_optargval));

      switch (option)
        {
          /* Forward agent */
        case 'a':
          data->config->agent_forwarding = !(ssh_optval);
          break;

          /* add a cipher name to the list */
        case 'c':
          {
            char *cname;

            if (!ssh_optval)
              ssh_fatal("Illegal -c parameter.");

            cname = ssh_cipher_get_native_name(ssh_optarg);

            if (cname == NULL)
              ssh_fatal("Cipher %s is not supported.", ssh_optarg);

            if (!have_c_arg)
              {
                have_c_arg = TRUE;
                if (data->config->ciphers != NULL)
                  {
                    ssh_xfree(data->config->ciphers);
                    data->config->ciphers = NULL;
                  }
              }
            if (data->config->ciphers == NULL)
              {
                data->config->ciphers = ssh_xstrdup(cname);
              }
            else
              {
                char *hlp = NULL;

                ssh_dsprintf(&hlp, "%s,%s", data->config->ciphers,
                             cname);
                ssh_xfree(data->config->ciphers);
                data->config->ciphers = hlp;
              }
          }
          SSH_DEBUG(3, ("Cipherlist is \"%s\"", data->config->ciphers));
          break;

          /* add a MAC name to the list */
        case 'm':
          {
            char *macname = NULL;

            if (!ssh_optval)
              ssh_fatal("Illegal -m parameter.");

            if (ssh_mac_supported(ssh_optarg))
              macname = ssh_xstrdup(ssh_optarg);

            if (macname == NULL)
              ssh_fatal("MAC %s is not supported.", ssh_optarg);

            if (!have_m_arg)
              {
                have_m_arg = TRUE;
                if (data->config->macs != NULL)
                  {
                    ssh_xfree(data->config->macs);
                    data->config->macs = NULL;
                  }
              }
            if (data->config->macs == NULL)
              {
                data->config->macs = ssh_xstrdup(macname);
              }
            else
              {
                char *hlp = NULL;

                ssh_dsprintf(&hlp, "%s,%s",data->config->macs,
                             macname);
                ssh_xfree(data->config->macs);
                data->config->macs = hlp;
              }
          }
          SSH_DEBUG(3, ("Maclist is \"%s\"", data->config->macs));
          break;

          /* Compression */
        case 'C':

          /* XXX.

             The compression level that can be passed to ssh with
             +C[0..9] is not documented yet.  The reason is that the
             client can only control the compression level on the
             client->server stream, not in the other direction.  The
             protocol only sends the compression information 'zlib' or
             'none'.

             One possible solution is to use the compression level
             bits from the client->server stream (see RFC 1950, p.5,
             FLEVEL).  This is not supported by the zlib interface,
             though, and it requires that the server doesn't send any
             data before the client has sent these bits.

             Another solution is to fix the protocol, but the secsh
             group hasn't been too interested so far. */
          
          data->config->compression = !(ssh_optval);
          if (ssh_optarg != NULL)
            {
              int num = atoi(ssh_optarg);
              if (num > 9 || num < 1 || ssh_optval)
                {
                  ssh_fatal("Illegal argument '%s' for %cC.", 
                            ssh_optarg, ssh_optval ? '-' : '+');
                }
              data->config->compression_level = num;
            }
          break;

          /* Verbose mode */
        case 'v':
          data->config->verbose_mode = TRUE;
          ssh_debug_set_level_string("2");
          break;

          /* Debug level. */
        case 'd':
          if (!ssh_optval)
            ssh_fatal("Bad -d parameter.");
          data->config->verbose_mode = (ssh_optval != 0);
          ssh_debug_set_level_string(ssh_optarg);
          break;

          /* use ipv4 */
        case '4':
          data->config->protocol_mask |= SSH_IP_TYPE_MASK_IP4;
          break;

          /* use ipv6 */
        case '6':
          data->config->protocol_mask |= SSH_IP_TYPE_MASK_IP6;
          break;

          /* specify escape character */
        case 'e':
          if (!ssh_optval)
            ssh_fatal("Illegal -e parameter.");

          ssh_xfree(data->config->escape_char);
          if (strcmp(ssh_optarg, "none") == 0)
            data->config->escape_char = ssh_xstrdup("");
          else
            data->config->escape_char = ssh_xstrdup(ssh_optarg);

          break;
























          /* a "go background" flag */
        case 'f':
          data->config->go_background = (ssh_optval != 0);
          if (data->config->go_background)
            {
              if (ssh_optarg != NULL && strcmp(ssh_optarg,"o") == 0)
                {
                  /* One-shot forwarding. */
                  data->config->one_shot_forwarding = TRUE;
                }
              else if (ssh_optarg != NULL)
                {
                  ssh_fatal("Illegal argument '%s' for -f.", ssh_optarg);
                }
            }
          break;

          /* read in an alternative configuration file */
        case 'F':
          if (!ssh_optval)
            ssh_fatal("Illegal -F parameter.");

          if (stat(ssh_optarg, &st) < 0)
            ssh_fatal("Alternate config file \"%s\" does not exist.",
                      ssh_optarg);

          if (!ssh_config_read_file(tuser, data->config,
                                    data->config->host_to_connect ?
                                    data->config->host_to_connect : "",
                                    ssh_optarg, SSH_CONFIG_TYPE_CLIENT))
            ssh_fatal("Failed to read config file \"%s\"", ssh_optarg);
          break;

          /* specify the identity file */
        case 'i':
          if (!ssh_optval)
            ssh_fatal("Illegal -i parameter.");
          /* XXX Check that the file exists, and if not, issue warning. */
          ssh_xfree(data->config->identity_file);
          data->config->identity_file = ssh_xstrdup(ssh_optarg);
          break;
























          /* specify a login name */
        case 'l':
          if (!ssh_optval)
            ssh_fatal("Illegal -l parameter.");

          ssh_xfree(data->config->login_as_user);
          data->config->login_as_user = ssh_xstrdup(ssh_optarg);
          break;

          /* Specify a local forwarding */
        case 'L':
#ifdef SSH_CHANNEL_TCPFWD
          if (!ssh_optval)
            ssh_fatal("Illegal -L parameter.");

          if (ssh_config_set_parameter(data->config, &metaconfig,
                                       "LocalForward", ssh_optarg,
                                       SSH_CONFIG_TYPE_CLIENT))
            ssh_fatal("Bad local forward definition \"%s\"", ssh_optarg);
#else /* SSH_CHANNEL_TCPFWD */
          ssh_fatal("TCP forwarding disabled.");
#endif /* SSH_CHANNEL_TCPFWD */
          break;

          /* don't read stdin ? */
        case 'n':
          data->config->dont_read_stdin = (ssh_optval != 0);
          break;

          /* Give one line of configuration data directly. */
        case 'o':
          if (!ssh_optval)
            ssh_fatal("Illegal -o parameter.");

          if (ssh_config_parse_line_and_set_params(data->config, ssh_optarg,
                                                   SSH_CONFIG_TYPE_CLIENT))
            ssh_fatal("Illegal -o parameter \"%s\"", ssh_optarg);

          break;

          /* specify the login port */
        case 'p':
          if (!ssh_optval)
            ssh_fatal("Illegal -p parameter.");
          ssh_xfree(data->config->port);
          data->config->port = ssh_xstrdup(ssh_optarg);
          break;

          /* use privileged port. This option is only recognized for
             backwards compatibility with ssh1 */
        case 'P':
          break;

          /* quiet mode */
        case 'q':
          data->config->quiet_mode = (ssh_optval != 0);
          break;

          /* Is this a subsystem ? */
        case 's':
          if (data->is_subsystem)
            {
              ssh_fatal("No multiple -s flags allowed.");
            }
          data->is_subsystem = (ssh_optval != 0);
          command = ssh_xstrdup(ssh_optarg);
          break;

        case 'S':
          data->no_session_channel = (ssh_optval != 0);
          break;

          /* Force ptty allocation ? */
        case 't':
          data->config->force_ptty_allocation = (ssh_optval != 0);
          break;

          /* X11 forwarding */
        case 'x':
          data->config->x11_forwarding = (ssh_optval == 0);
          data->config->x11_application_trusted = FALSE;
          break;

          /* Set X11 client application trust status (also
             enables/disables X11 forwarding. */
        case 'X':
          data->config->x11_forwarding = (ssh_optval == 0);
          data->config->x11_application_trusted = (ssh_optval == 0);
          break;

#ifdef SSH_CHANNEL_TCPFWD
          /* Specify a remote forwarding */
        case 'R':
          if (!ssh_optval)
            ssh_fatal("Illegal -R parameter.");

          if (ssh_parse_forward(&(data->config->remote_forwards), ssh_optarg))
            ssh_fatal("Bad remote forward definition \"%s\"", ssh_optarg);
#else /* SSH_CHANNEL_TCPFWD */
          ssh_fatal("TCP forwarding disabled.");
#endif /* SSH_CHANNEL_TCPFWD */
          break;

        case 'h':



          ssh2_help(av0);

          exit(0);
          break;

          /* Specify 8-bit clean. This option is only recognized for
             backwards compatibility with ssh1, and is passed to
             rsh/rlogin if falling back to them. (ssh2 doesn't fall
             back to rsh; it wouldn't be secure (and it would be
             against the draft))*/
        case '8':
          break;

          /* Gateway ports?  If yes, remote hosts may connect to
             locally forwarded ports. */
        case 'g':
          data->config->gateway_ports = (ssh_optval != 0);
          break;

          /* XXX kerberos tgt passing (surrently only recognized to
             support all ssh1's command-line options.*/
        case 'k':
          /* XXX*/
          break;

        case 'V':
          ssh2_version(av0);















          exit(0);
          break;

        case 'w':
          data->config->try_empty_password = (ssh_optval == 0);
          break;

          /* SSH1 compatibility mode selection. */
        case '1':
#ifdef SSHDIST_SSH2_INTERNAL_SSH1_EMULATION
#ifdef WITH_INTERNAL_SSH1_EMULATION
          switch(*ssh_optarg)
            {
            case 't': /* Traditional. */
              data->config->ssh1_emulation_internal = FALSE;
              if (!data->config->ssh1_path)
                {
                  ssh_warning("\"-1t\" without path for ssh1 executable.");
                  break;
                }
              data->config->ssh1compatibility = (ssh_optval != 0);
              break;
            case 'i':
              data->config->ssh1_emulation_internal = (ssh_optval != 0);
              break;
            default:
              ssh_warning("Illegal argument for option %c%c.",
                          ssh_optval != 0 ? '-' : '+', ssh_optopt);
              break;
            }
#endif /* WITH_INTERNAL_SSH1_EMULATION */
#endif /* SSHDIST_SSH2_INTERNAL_SSH1_EMULATION */
          break;

        default:
          if (ssh_optmissarg)
            {
              ssh_fatal("Option -%c needs an argument\n", ssh_optopt);
            }
          else
            {
              ssh_fatal("Unknown option -%c\n", ssh_optopt);
            }
          exit(1);
        }
    }








  /* Initializations */

  host = data->config->host_to_connect;

  /* if user specified #port on the command line, that will override
     anything read in from config files or -p */
  if (port)
    {
      ssh_xfree(data->config->port);
      data->config->port = ssh_xstrdup(port);
      ssh_xfree(port);
      port = NULL;
    }


#ifndef MEMLEAKS_AT_SIGINT
  if (data->config->authentication_notify)
    {
      /* If we are running as child for sftp2 or scp2, ignore SIGINT,
         so that the parent may continue doing something with a
         connection, even if the user aborted the transfer. */
      ssh_register_signal(SIGINT, NULL_FNPTR, NULL);
    }
#else /* !MEMLEAKS_AT_SIGINT */
  ssh_register_signal(SIGINT, memleak_sigint_handler, NULL);
#endif /* !MEMLEAKS_AT_SIGINT */


  finalize_password_prompt(&data->config->password_prompt, host,
                           data->config->login_as_user);

  ssh_randseed_open(tuser, data->config);

  data->user_data = tuser;
  data->command = command;

  /* If we don't receive our input from a terminal, don't handle
     escape chars, as they would interfere. */
  if (!isatty(fileno(stdin)))
    {
      ssh_xfree(data->config->escape_char);
      data->config->escape_char = ssh_xstrdup("");
    }

  data->allocate_pty = (((command == NULL) &&
                         (isatty(fileno(stdin)))) ||
                        data->config->force_ptty_allocation);

  if ((data->term = getenv("TERM")) == NULL)
    data->term = ssh_xstrdup("vt100");
  else
    data->term = ssh_xstrdup(data->term);

  data->debug = data->config->verbose_mode;

  /* Finalize initialization. */
  ssh_config_init_finalize(data->config);

  /* Figure out the name of the socks server, if any.  It can specified
     at run time using the SSH_SOCKS_SERVER environment variable, or at
     compile time using the SOCKS_DEFAULT_SERVER define.  The environment
     variable overrides the compile-time define. */
  socks_server = getenv("SSH_SOCKS_SERVER");















































#ifdef SOCKS_DEFAULT_SERVER
  if (!socks_server)
    socks_server = SOCKS_DEFAULT_SERVER;
#endif /* SOCKS_DEFAULT_SERVER */


  if (socks_server && strcmp(socks_server, "") == 0)
    socks_server = NULL;

  /* Configuration file variable SockServer overrides environment
     variable. */
  if (data->config->socks_server &&
      strcmp(data->config->socks_server, "") != 0)
    socks_server = data->config->socks_server;

  /* Connect to the remote host. */
  ssh_debug("Connecting to %s, port %s%s%s%s", host, data->config->port,
            socks_server ? ", using SOCKS server URL \"" : "...",
            socks_server ? socks_server : "",
            socks_server ? "\"..." : " (SOCKS not used)");

  memset(&tcp_connect_params, 0, sizeof(tcp_connect_params));
  tcp_connect_params.socks_server_url = socks_server;
  if (data->config->use_socks5)
    tcp_connect_params.socks_type = SSH_TCP_SOCKS5;
  else
    tcp_connect_params.socks_type = SSH_TCP_SOCKS4;
    
  tcp_connect_params.connection_attempts = 5;
  tcp_connect_params.protocol_mask = data->config->protocol_mask;

  /* create the tcp connect callback context */
  connect_context = ssh_xmalloc(sizeof(*connect_context));

  connect_context->data = data;
  connect_context->tcp_connect_params = &tcp_connect_params;
  connect_context->host = host;



  connect_context->socks_server_list = NULL;

  
  ssh_tcp_connect(host, data->config->port, &tcp_connect_params,
                  connect_done, (void *)connect_context);
  SSH_DEBUG(2, ("Entering event loop."));
  ssh_event_loop_run();


  ssh_signals_reset();


  ssh_xfree(connect_context);




  ssh_global_uninit();

  /* Update random seed file. */
  ssh_randseed_update(tuser, data->config);

  ssh_random_free();

  ssh_event_loop_uninitialize();


  ssh_leave_non_blocking(-1);
  ssh_leave_raw_mode(-1);


  ssh_user_free(tuser, FALSE);
  exit_status = data->exit_status;





  ssh_app_free_global_regex_context();
  ssh_config_free(data->config);
  ssh_xfree(data->term);
  /* XXX free user, command, host ? */
  ssh_xfree(data);






  return exit_status;
}
