/*

  sftp-server2.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  File server that is executed by sshd.

*/

#include "ssh2includes.h"
#include "ssheloop.h"
#include "sshfdstream.h"
#include "sshfilexfer.h"
#include "sshsignals.h"
#include "sshappcommon.h"
#include "sshgetopt.h"
#include "sshuser.h"
#include <syslog.h>
#ifdef NEED_SYS_SYSLOG_H
#include <sys/syslog.h>
#endif /* NEED_SYS_SYSLOG_H */

#define SSH_DEBUG_MODULE "SshSftpServer"

/* Program name, without path. */
const char *av0;

typedef struct SftpServerDataRec
{
  int log_facility;
} *SftpServerData, SftpServerDataStruct;

/* Logging callback. Mostly stolen from sshd2.c . */
void sftp_log_cb(SshLogFacility facility, SshLogSeverity severity,
                 const char *msg, void *context)
{
  SftpServerData data = (SftpServerData) context;
  int fac, sev;
  static Boolean logopen = FALSE;
  static int logopt;
  static int logfac;

  if (!logopen)
    {
      logopt = LOG_PID;
      logfac = ssh_app_log_facility(data->log_facility);

      openlog(av0, logopt, logfac);
      logopen = TRUE;
    }

  fac = ssh_app_log_facility(facility);
  sev = ssh_app_log_severity(severity);
  if (fac != -1 && sev != -1)
    syslog(((fac != logfac) ? fac : 0) | sev, "%s", msg);
}

int main(int argc, char **argv)
{
  SshFileServer server;
  SftpServerDataStruct data;
  char *log_fac;
  pid_t sshd2_pid;
  SshUser uc;
  char   user_name[64];
  
  memset(&data, 0, sizeof(data));
  
  /* Save program name. */
  if (strchr(argv[0], '/'))
    av0 = strrchr(argv[0], '/') + 1;
  else
    av0 = argv[0];

  data.log_facility = -1;

  /* This environment variable is _always_ overrun by sshd2, so the user
     has no control over it if they have no shell access. (if the user
     has shell access, they can use their own sftp-server)

     So the log generated by sftp-server is more informative than the
     absolute truth.

     XXX Logging in a chrooted environment is a _major_ pain in the
     neck.
  */
  if ((log_fac = getenv(SSH_SFTP_LOG_FACILITY)) != NULL)
    data.log_facility = atoi(log_fac);

  SSH_DEBUG(2, ("Log facility: %d", data.log_facility));
  
  ssh_event_loop_initialize();
  ssh_signals_prevent_core(TRUE, NULL);

  if (data.log_facility >= 0)
    ssh_log_register_callback(sftp_log_cb, (void *)&data);

  uc = ssh_user_initialize_with_uid(getuid(), FALSE);
  if (uc)
    strncpy(user_name, ssh_user_name(uc), sizeof(user_name));
  else
    /* Use UID as user's name, if ssh_user_initialize() fails. */
    ssh_snprintf(user_name, sizeof(user_name), "%d", getuid());

  if (uc)
    {
      ssh_user_free(uc, FALSE);
      uc = NULL;
    }

  sshd2_pid = getppid();

  if (data.log_facility >= 0)
    ssh_log_event(data.log_facility, SSH_LOG_NOTICE,
                  "Starting SFTP-subsystem for user '%s' parent sshd2 "
                  "PID [%ld].", user_name, sshd2_pid);

  server = ssh_file_server_wrap(ssh_stream_fd_stdio(),
                                data.log_facility);
  ssh_event_loop_run();
  ssh_signals_reset();
  if (data.log_facility >= 0)
    ssh_log_event(data.log_facility, SSH_LOG_NOTICE,
                  "Stopping SFTP-subsystem for user '%s' parent sshd2 "
                  "PID [%ld].", user_name, sshd2_pid);

  ssh_event_loop_uninitialize();

  return 0;
}

