/*

trkex.c

  Authors:
        Tatu Ylonen <ylo@ssh.fi>
        Markku-Juhani Saarinen <mjos@ssh.fi>
        Timo J. Rinne <tri@ssh.fi>
        Sami Lehtinen <sjl@ssh.fi>

  Copyright (C) 1996-2001 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

  Key exchange methods.

*/

/*
 * $Id: trkex.c,v 1.3.4.2 2002/05/07 20:51:04 sjl Exp $
 * $Log: trkex.c,v $ * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * $EndLog$
 */

#include "sshincludes.h"
#include "sshbufaux.h"
#include "sshgetput.h"
#include "sshmsgs.h"
#include "trcommon.h"
#include "trkex.h"
#include "sshencode.h"
#include "ssh2pubkeyencode.h"
#include "sshcipherlist.h"
#include "sshdebug.h"
#include "ssh2compat.h"

#define SSH_DEBUG_MODULE "SshProtoTrKex"

/* Uncomment this, if you wish to be sure that the signature operation
   is really performed asynchronously. */
/* #define TEST_SIGN_TIMEOUT */

/* forward definitions */

Boolean ssh_kex_derive_keys(SshTransportCommon tr);
void ssh_kex_keycheck_callback(Boolean result, void *ctx);

void ssh_kexdh_server_make_kex2_cb(SshCryptoStatus status,
                                   const unsigned char *signature_buffer,
                                   size_t signature_buffer_len,
                                   void *context);
#ifdef TEST_SIGN_TIMEOUT
#include "sshtimeouts.h"
void trkex_sign_do_timeout_cb(SshCryptoStatus status,
                              const unsigned char *signature_buffer,
                              size_t signature_buffer_len,
                              void *context);
void trkex_timeout_cb(void *context);
#endif /* TEST_SIGN_TIMEOUT */

/* Prepares the client side for key exchange. */

void ssh_tr_client_init_kex(SshTransportCommon tr,
                            const char *service_name,
                            const char *server_host_name,
                            SshKeyCheckCallback key_check,
                            void *key_check_context)
{
  tr->service_name = ssh_xstrdup(service_name);
  tr->server_host_name = ssh_xstrdup(server_host_name);
  tr->key_check = key_check;
  tr->key_check_context = key_check_context;
}

/* Prepares the server side for key exchange. */

void ssh_tr_server_init_kex(SshTransportCommon tr,
                            SshHostKeysContext host_keys_ctx)
{
  tr->host_keys_ctx = host_keys_ctx;  /* No copying here */
}

/* Generate and set up a diffie-hellman-group, the secret and a exchange
   value.
   returns a SshCryptoStatus. */

SshCryptoStatus ssh_kexdh_make_group(SshTransportCommon tr,
                                     const char *group_name)
{
  int i;

  /* group1's p, lifted from draft-ietf-ipsec-oakley-02.txt
     "E.2. Well-Known Group 2:  a 1024 bit prime" */

  const char group1_p[] =
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE65381"
    "FFFFFFFFFFFFFFFF";

  /* group1's generator */
  const char group1_g[] = "2";

  /* we currently accept only this one group */

  if (strcmp(group_name, "diffie-hellman-group1") != 0)
    {
      ssh_debug("ssh_kexdh_make_group: unsupported type %s",
                group_name);
      return SSH_CRYPTO_UNKNOWN_GROUP_TYPE;
    }

  /* read the p and the g */

  ssh_mp_set_str(tr->dh_p, group1_p, 16);
  ssh_mp_set_str(tr->dh_g, group1_g, 16);

  /*
    Randomize our secret value (x or y) and public value (e or f).
    We'll use only 192 bits of entropy for faster exponentiation.

    For discussion, see:
      P. C. van Oorschot and M. J. Wiener, "On Diffie-Hellman Key Agreement
      with Short Exponents", proc. Eurocrypt 96
  */

  ssh_mp_set_ui(tr->dh_secret, 1);
  for (i = 0; i < 24; i++)
    {
      ssh_mp_mul_2exp(tr->dh_secret, tr->dh_secret, 8);
      ssh_mp_add_ui(tr->dh_secret, tr->dh_secret,
                 ssh_random_get_byte());
    }

  ssh_mp_powm(tr->server ? tr->dh_f : tr->dh_e,
           tr->dh_g, tr->dh_secret, tr->dh_p);

  return SSH_CRYPTO_OK;
}

/* Compute the shared secret and the exchange hash. Return TRUE on failure */

Boolean ssh_kexdh_compute_h(SshTransportCommon tr)
{
  SshMPIntC t;
  SshBuffer buf, pub_host_key_buf;

  /* check that the public value is within the range */

  if (ssh_mp_cmp_ui(tr->server ? tr->dh_e : tr->dh_f, 2) <= 0)
    return TRUE;

  ssh_mp_init(t);
  ssh_mp_sub_ui(t, tr->dh_p, 2);
  if (ssh_mp_cmp(tr->dh_e, t) >= 0)
    {
      ssh_mp_clear(t);
      return TRUE;
    }
  ssh_mp_clear(t);

  /* compute the shared secret */

  ssh_mp_powm(tr->dh_k, tr->server ? tr->dh_e : tr->dh_f,
           tr->dh_secret, tr->dh_p);

  /* ok, compute the exchange hash */

  buf = ssh_xbuffer_allocate();

  if (tr->server)
    {
      ssh_encode_buffer(buf,
                        SSH_FORMAT_UINT32_STR,
                          tr->remote_version, strlen(tr->remote_version),
                        SSH_FORMAT_UINT32_STR,
                          tr->own_version, strlen(tr->own_version),
                        SSH_FORMAT_END);
      pub_host_key_buf = ssh_host_key_get_pubkey_by_algname(tr->host_key_name,
                                                            tr->host_keys_ctx);
    }
  else
    {
      ssh_encode_buffer(buf,
                        SSH_FORMAT_UINT32_STR,
                          tr->own_version, strlen(tr->own_version),
                        SSH_FORMAT_UINT32_STR,
                          tr->remote_version, strlen(tr->remote_version),
                        SSH_FORMAT_END);
      pub_host_key_buf = tr->public_host_key_blob;
    }

  ssh_encode_buffer(buf,
                    SSH_FORMAT_UINT32_STR,
                      ssh_buffer_ptr(tr->client_kexinit_packet),
                      ssh_buffer_len(tr->client_kexinit_packet),
                    SSH_FORMAT_UINT32_STR,
                      ssh_buffer_ptr(tr->server_kexinit_packet),
                      ssh_buffer_len(tr->server_kexinit_packet),
                    SSH_FORMAT_UINT32_STR,
                      ssh_buffer_ptr(pub_host_key_buf),
                      ssh_buffer_len(pub_host_key_buf),
                    SSH_FORMAT_END);
  ssh_bufaux_put_mp_int_ssh2style(buf, tr->dh_e);
  ssh_bufaux_put_mp_int_ssh2style(buf, tr->dh_f);
  ssh_bufaux_put_mp_int_ssh2style(buf, tr->dh_k);

  ssh_hash_reset(tr->hash);
  ssh_hash_update(tr->hash, ssh_buffer_ptr(buf), ssh_buffer_len(buf));
  ssh_hash_final(tr->hash, tr->exchange_hash);
  tr->exchange_hash_len = ssh_hash_digest_length(tr->hash);

  /* our first key exchange ? */

  if (tr->doing_rekey == FALSE)
    {
      memcpy(tr->session_identifier, tr->exchange_hash, tr->exchange_hash_len);
      tr->session_identifier_len = tr->exchange_hash_len;
    }

  ssh_buffer_free(buf);

  return FALSE;
}


/* client makes the diffie-hellman kex1, which is a SSH_MSG_KEXDH_INIT */

SshBuffer ssh_kexdh_client_make_kex1(SshTransportCommon tr)
{
  SshBuffer packet;

  SSH_DEBUG(3, ("Making first key exchange packet."));

  /* generate group1 & the exchange value for the client */
  if (ssh_kexdh_make_group(tr, "diffie-hellman-group1") != SSH_CRYPTO_OK)
    return NULL;

  /* Allocate and construct a KEXDH_INIT packet. */

  packet = ssh_xbuffer_allocate();
  ssh_bufaux_put_char(packet, SSH_MSG_KEXDH_INIT);
  ssh_bufaux_put_mp_int_ssh2style(packet, tr->dh_e);

  return packet;
}

/* server recieves client's kex1, which is a SSH_MSG_KEXDH_INIT */

Boolean ssh_kexdh_server_input_kex1(SshTransportCommon tr, SshBuffer input)
{
  unsigned char code;

  /* decode the first byte to see that we got a SSH_MSG_KEXDH_INIT */

  code = ssh_bufaux_get_char(input);

  if (code != SSH_MSG_KEXDH_INIT)
    {
      SSH_TRACE(0, ("Expected SSH_MSG_KEXDH_INIT, got %d", (int) code));
      return FALSE;
    }

  /* server generates the group, secret, and the exchange value */

  if (ssh_kexdh_make_group(tr, "diffie-hellman-group1") != SSH_CRYPTO_OK)
    return FALSE;
  if (!ssh_bufaux_get_mp_int_ssh2style(input, tr->dh_e))
    return FALSE;

  return TRUE;
}

/* Internal struct for the make_kex2 callback, and the calling
   function. */
typedef struct SshServerMakeKex2CtxRec
{
  SshMakeKex2CompletionProc completion_proc;
  SshTransportCommon tr;
} *SshServerMakeKex2Ctx;

/* server creates a SSH_MSG_KEXDH_REPLY (kex2) */

void ssh_kexdh_server_make_kex2(SshTransportCommon tr,
                                SshMakeKex2CompletionProc completion)
{
  SshPrivateKey priv_key = NULL;
  SshServerMakeKex2Ctx cb_ctx;
  
  /* compute the exchange hash H */

  if (ssh_kexdh_compute_h(tr))
    {
      (*completion)(NULL, tr);
      return;
    }

  priv_key = ssh_host_key_get_privkey_by_algname(tr->host_key_name,
                                                 tr->host_keys_ctx);
  if (!priv_key)
    {
      SSH_DEBUG(2, ("Could not retrieve private host key"));
      (*completion)(NULL, tr);
      return;
    }

  if (!strcmp(tr->host_key_name, SSH_SSH_RSA) &&
      !ssh_compat_rsa_private_key_change_scheme
      (priv_key, !tr->ssh_old_rsa_hash_scheme_bug_compat))
    {
      (*completion)(NULL, tr);
      return;
    }

  cb_ctx = ssh_xcalloc(1, sizeof(struct SshServerMakeKex2CtxRec));
  cb_ctx->tr = tr;
  cb_ctx->completion_proc = completion;

  /* do the private key operation to sign H */
  ssh_private_key_sign_async(priv_key,
                             tr->exchange_hash, tr->exchange_hash_len,
#ifdef TEST_SIGN_TIMEOUT
                             trkex_sign_do_timeout_cb,
#else /* TEST_SIGN_TIMEOUT */
                             ssh_kexdh_server_make_kex2_cb,
#endif /* TEST_SIGN_TIMEOUT */
                             cb_ctx);
}

#ifdef TEST_SIGN_TIMEOUT
typedef struct TrKexSignTimeOutCtxRec {
  SshCryptoStatus status;
  unsigned char *signature_buffer;
  size_t signature_buffer_len;
  void *context;
} TrKexSignTimeOutCtxStruct, *TrKexSignTimeOutCtx;

void trkex_sign_do_timeout_cb(SshCryptoStatus status,
                              const unsigned char *signature_buffer,
                              size_t signature_buffer_len,
                              void *context)
{
  TrKexSignTimeOutCtx ctx = ssh_xcalloc(1, sizeof(*ctx));
  ctx->status = status;
  ctx->signature_buffer = ssh_xmemdup(signature_buffer, signature_buffer_len);
  ctx->signature_buffer_len = signature_buffer_len;
  ctx->context = context;
  ssh_register_timeout(2L, 0L, trkex_timeout_cb, ctx);
}

void trkex_timeout_cb(void *context)
{
  TrKexSignTimeOutCtx ctx = (TrKexSignTimeOutCtx) context;
  SSH_DEBUG(2, ("Timeout fired, starting real signature completion "
                "callback."));
  ssh_kexdh_server_make_kex2_cb(ctx->status,
                                ctx->signature_buffer,
                                ctx->signature_buffer_len,
                                ctx->context);
  ssh_xfree(ctx->signature_buffer);
  ssh_xfree(ctx);
}
#endif /* TEST_SIGN_TIMEOUT */
void ssh_kexdh_server_make_kex2_cb(SshCryptoStatus status,
                                   const unsigned char *signature_buffer,
                                   size_t signature_buffer_len,
                                   void *context)
{
  SshServerMakeKex2Ctx cb_ctx = (SshServerMakeKex2Ctx)context;
  SshBuffer packet;
  unsigned char *buffer;
  size_t buffer_len;
  SshBuffer public_host_key_blob = NULL;

  if (status != SSH_CRYPTO_OK)
    {
      SSH_DEBUG(2, ("Crypto operation failed when constructing KEX2 packet"));
      (*cb_ctx->completion_proc)(NULL, cb_ctx->tr);
      return;
    }

  public_host_key_blob =
    ssh_host_key_get_pubkey_by_algname(cb_ctx->tr->host_key_name,
                                       cb_ctx->tr->host_keys_ctx);
  if (!public_host_key_blob)
    {
      SSH_DEBUG(2, ("Could not retrieve public host key"));
      (*cb_ctx->completion_proc)(NULL, cb_ctx->tr);
      return;
    }

  if (!cb_ctx->tr->ssh_old_malformed_signatures_bug_compat)
    {
      if (!cb_ctx->tr->host_key_name)
        {
          (*cb_ctx->completion_proc)(NULL, cb_ctx->tr);
          return;
        }
      
      buffer_len =
        ssh_encode_array_alloc(&buffer,
                               SSH_FORMAT_UINT32_STR, cb_ctx->tr->host_key_name,
                               strlen(cb_ctx->tr->host_key_name),
                               SSH_FORMAT_UINT32_STR, signature_buffer,
                               signature_buffer_len,
                               SSH_FORMAT_END);
    }
  else
    {
      buffer = (unsigned char *)signature_buffer;
      buffer_len = signature_buffer_len;
    }
  
  /* construct the packet */
  packet = ssh_xbuffer_allocate();
  ssh_encode_buffer(packet,
                    SSH_FORMAT_CHAR, (unsigned int) SSH_MSG_KEXDH_REPLY,
                    SSH_FORMAT_UINT32_STR,
                      ssh_buffer_ptr(public_host_key_blob),
                      ssh_buffer_len(public_host_key_blob),
                    SSH_FORMAT_END);
  ssh_bufaux_put_mp_int_ssh2style(packet, cb_ctx->tr->dh_f);
  ssh_bufaux_put_uint32_string(packet, buffer, buffer_len);

  if (!cb_ctx->tr->ssh_old_malformed_signatures_bug_compat)
    {
      memset(buffer, 0, buffer_len);
      ssh_xfree(buffer);
    }

  /* we're ready to derive the keys */
  ssh_kex_derive_keys(cb_ctx->tr);

  (*cb_ctx->completion_proc)(packet, cb_ctx->tr);
}


/* A simple callback for the key check function */

/* Internal struct for the callback, and the calling function. */
typedef struct SshKex2KeyCheckCallbackContextRec
{
  SshBuffer input;
  size_t pubkey_len;
  unsigned char *pubkey;
  size_t sig_len;
  unsigned char *signature;
  SshKex2CompletionProc completion;

} *SshKex2KeyCheckCallbackContext;

/* This function is called by the registered key check function. (in
   ssh2 it is ssh_client_key_check in sshclient.c) */
/* XXX We should be able to pass error message from application level. */
void ssh_kex_keycheck_callback(Boolean result, void *ctx)
{
  SshTransportCommon tr;
  unsigned char *real_signature = NULL;
  size_t real_sig_len = 0L;
  SshKex2KeyCheckCallbackContext callback_context;

  tr = (SshTransportCommon) ctx;

  callback_context = tr->key_check_callback_context;

  tr->key_check_result = result;

  if (tr->key_check_result == FALSE)
    {
      if (callback_context->pubkey != NULL)
        ssh_xfree(callback_context->pubkey);
    }
  else
    {
      tr->public_host_key_blob = ssh_xbuffer_allocate();
      ssh_xbuffer_append(tr->public_host_key_blob, callback_context->pubkey,
                         callback_context->pubkey_len);

      if (!ssh_bufaux_get_mp_int_ssh2style(callback_context->input, tr->dh_f))
        {
          SSH_TRACE(2, ("Error parsing SSH_MSG_KEXDH_REPLY packet (couldn't "
                        "get mpint \"f\")"));

          ssh_xfree(callback_context->pubkey);
          tr->key_check_result = FALSE;
          goto error;
        }

      callback_context->signature =
        ssh_bufaux_get_uint32_string(callback_context->input,
                                 &(callback_context->sig_len));

      if (!tr->ssh_old_malformed_signatures_bug_compat)
        {
          char *recv_pubkeytype = NULL;
          size_t decoded_len;

          decoded_len =
            ssh_decode_array(callback_context->signature,
                             callback_context->sig_len,
                             SSH_FORMAT_UINT32_STR, &recv_pubkeytype, NULL,
                             SSH_FORMAT_UINT32_STR,
                             &real_signature, &real_sig_len,
                             SSH_FORMAT_END);

          if (decoded_len == 0 || decoded_len != callback_context->sig_len)
            {
              SSH_DEBUG(2, ("decoded_len: %ld, sig_len: %ld",
                            decoded_len, callback_context->sig_len));
              ssh_warning("Received malformed signature during server "
                          "keycheck.");
              memset(callback_context->signature, 0,
                     callback_context->sig_len);
              ssh_xfree(callback_context->signature);
              tr->key_check_result = FALSE;
              goto error;
            }

          memset(callback_context->signature, 0,
                 callback_context->sig_len);
          ssh_xfree(callback_context->signature);

          if (strcmp(recv_pubkeytype, tr->host_key_name) != 0)
            {
              ssh_warning("Received malformed signature during server "
                          "keycheck. (public key type doesn't match the "
                          "received hostkey.)");
              ssh_xfree(recv_pubkeytype);
              tr->key_check_result = FALSE;
              goto error;
            }
          ssh_xfree(recv_pubkeytype);
        }
      else
        {
          real_signature = callback_context->signature;
          real_sig_len = callback_context->sig_len;
          callback_context->signature = NULL;
        }

      ssh_xfree(callback_context->pubkey);

      /* ok, verify the signature */
      if (ssh_kexdh_compute_h(tr))
        {
          SSH_TRACE(2, ("Couldn't calculate exchange hash."));
          tr->key_check_result = FALSE;
          goto error;
        }

      if (!strncmp(tr->host_key_name, SSH_SSH_RSA, strlen(SSH_SSH_RSA)) &&
          !ssh_compat_rsa_public_key_change_scheme
          (tr->public_host_key, !tr->ssh_old_rsa_hash_scheme_bug_compat))
        {
          tr->key_check_result = FALSE;
          goto error;
        }

      if (ssh_public_key_verify_signature(tr->public_host_key,
                                          real_signature,
                                          real_sig_len,
                                          tr->exchange_hash,
                                          tr->exchange_hash_len) == FALSE)
        {
          SSH_TRACE(2, ("Signature didn't match."));
          tr->key_check_result = FALSE;
          goto error;
        }

      /* we're ready to derive the keys */
      ssh_kex_derive_keys(tr);
    }
 error:
  if (real_signature != NULL)
    {
      memset(real_signature, 0, real_sig_len);
      ssh_xfree(real_signature);
    }

  /* Call the supplied callback.*/
  (*callback_context->completion)(tr);

  if (callback_context->input)
    {
      ssh_buffer_free(callback_context->input);
      callback_context->input = NULL;
    }

  ssh_xfree(callback_context);
}

/* client parses the server's SSH_MSG_KEXDH_REPLY (kex2) */

void ssh_kexdh_client_input_kex2(SshTransportCommon tr, SshBuffer input,
                                 SshKex2CompletionProc finalize_callback)
{
  unsigned int code;
  unsigned char *pubkey;
  size_t pubkey_len;
  SshKex2KeyCheckCallbackContext callback_context;

  if ((code = ssh_bufaux_get_char(input)) != SSH_MSG_KEXDH_REPLY)
    {
      SSH_TRACE(0, ("Received illegal packet %d.", code));
      tr->key_check_result = FALSE;
      (*finalize_callback)(tr);
      return;
    }

  /* get the public key */

  if (ssh_decode_buffer(input, SSH_FORMAT_UINT32_STR, &pubkey, &pubkey_len,
                        SSH_FORMAT_END) == 0)
    {
      SSH_TRACE(0, ("Failed to parse the pubkey and certificates."));
      tr->key_check_result = FALSE;
      (*finalize_callback)(tr);
      return;
    }

  if (tr->public_host_key_blob != NULL)
    ssh_buffer_free(tr->public_host_key_blob);

  /* Here we decode the received public key and check that
     it matches with the negotiated public key type */
  tr->public_host_key =
    ssh_decode_pubkeyblob_general(pubkey, pubkey_len,
                                  (unsigned char*) tr->host_key_name);

  if (tr->public_host_key == NULL)
    {
      SSH_TRACE(0, ("Invalid host key."));
      tr->key_check_result = FALSE;
      (*finalize_callback)(tr);
      return;
    }

  /* Check the host key */
  SSH_ASSERT(tr->key_check != NULL_FNPTR);
  
  callback_context =
    ssh_xcalloc(1, sizeof(struct SshKex2KeyCheckCallbackContextRec));

  tr->key_check_callback_context = callback_context;

  callback_context->pubkey = pubkey;
  callback_context->pubkey_len = pubkey_len;
  callback_context->signature = NULL;
  callback_context->sig_len = 0;
  callback_context->completion = finalize_callback;
  callback_context->input = input;

  tr->received_state = RECEIVED_KEY_CHECK;

  (*tr->key_check)(tr->server_host_name, pubkey, pubkey_len,
                   (unsigned char*) tr->host_key_name,
                   ssh_kex_keycheck_callback, tr, tr->key_check_context);

  /* Rest is done in ssh_kex_keycheck_callback(). */
}

/* derive one key */

void ssh_kex_derive_key(SshTransportCommon tr,
                        unsigned char id,
                        unsigned char *key_ptr, size_t key_len)
{
  unsigned char buf[SSH_MAX_HASH_DIGEST_LENGTH];
  size_t len, hash_len, i;
  SshBufferStruct buffer;

  hash_len = ssh_hash_digest_length(tr->hash);

  /* Compute the first part of the key. */

  ssh_hash_reset(tr->hash);
  if (! tr->ssh_old_keygen_bug_compat)
    {
      ssh_buffer_init(&buffer);
      ssh_bufaux_put_mp_int_ssh2style(&buffer, tr->dh_k);
      ssh_hash_update(tr->hash,
                      ssh_buffer_ptr(&buffer),
                      ssh_buffer_len(&buffer));
    }
  ssh_hash_update(tr->hash, tr->exchange_hash, tr->exchange_hash_len);
  ssh_hash_update(tr->hash, &id, 1);
  ssh_hash_update(tr->hash,
                  tr->session_identifier,
                  tr->session_identifier_len);
  ssh_hash_final(tr->hash, buf);

  /* Expand the key. Note that this will not increase entropy in
     any real sense. */

  for (i = 0; i < key_len; /*NOTHING*/)
    {
      len = key_len - i;
      len = len > hash_len ? hash_len : len;
      memcpy(&key_ptr[i], buf, len);
      i += len;

      if (i < key_len)
        {
          ssh_hash_reset(tr->hash);
          if (! tr->ssh_old_keygen_bug_compat)
            {
              ssh_hash_update(tr->hash,
                              ssh_buffer_ptr(&buffer),
                              ssh_buffer_len(&buffer));
            }
          ssh_hash_update(tr->hash, tr->exchange_hash, tr->exchange_hash_len);
          ssh_hash_update(tr->hash, key_ptr, i);
          ssh_hash_final(tr->hash, buf);
        }
    }

  if (! tr->ssh_old_keygen_bug_compat)
    {
      memset(ssh_buffer_ptr(&buffer), 0, ssh_buffer_len(&buffer));
      ssh_buffer_uninit(&buffer);
    }
  memset(buf, 0, sizeof(buf));
}

/* Derive all keys */

Boolean ssh_kex_derive_keys(SshTransportCommon tr)
{
  ssh_kex_derive_key(tr, 'A', tr->c_to_s.iv, sizeof(tr->c_to_s.iv));
  ssh_kex_derive_key(tr, 'B', tr->s_to_c.iv, sizeof(tr->s_to_c.iv));
  ssh_kex_derive_key(tr, 'C', tr->c_to_s.encryption_key,
                     sizeof(tr->c_to_s.encryption_key));
  ssh_kex_derive_key(tr, 'D', tr->s_to_c.encryption_key,
                     sizeof(tr->s_to_c.encryption_key));
  ssh_kex_derive_key(tr, 'E', tr->c_to_s.integrity_key,
                     sizeof(tr->c_to_s.integrity_key));
  ssh_kex_derive_key(tr, 'F', tr->s_to_c.integrity_key,
                     sizeof(tr->s_to_c.integrity_key));

  return TRUE;
}

/* Returns NULL.  This is used as a kex1/kex2 packet generator when no such
   packet is sent for the kex type. */

SshBuffer ssh_kex1_return_no_packet(SshTransportCommon tr)
{
  return NULL;
}

void ssh_kex2_return_no_packet(SshTransportCommon tr,
                               SshMakeKex2CompletionProc completion)
{
  (*completion)(NULL, tr);
}

/* Definitions of supported key exchange algorithms. */

const struct SshKexTypeRec ssh_kex_algorithms[] =
{
  { "diffie-hellman-group1-sha1",
    /* crypto library identifier of the hash function */
    "sha1",
    /* need_encryption_capable_hostkey */
    FALSE,
    /* need_signature_capable_hostkey */
    TRUE,
    /* SshMakeKexProc client_make_kex1 */
    ssh_kexdh_client_make_kex1,
    /* SshMakeKexProc server_make_kex1 */
    ssh_kex1_return_no_packet,
    /* SshMakeKexProc client_make_kex2 */
    ssh_kex2_return_no_packet,
    /* SshMakeKexProc server_make_kex2 */
    ssh_kexdh_server_make_kex2,
    /* client_input_kex1 */
    NULL_FNPTR,
    /* server_input_kex1 */
    ssh_kexdh_server_input_kex1,
    /* client_input_kex2 */
    ssh_kexdh_client_input_kex2,
    /* server_input_kex2 */
    NULL_FNPTR },

#if 0
  { "double-encrypting-sha1", "sha1",
    TRUE, FALSE,
    ssh_kex_return_no_packet, ssh_kexde_server_make_hostkey,
    ssh_kexde_client_make_sessionkey, ssh_kex_return_no_packet,
    ssh_kexde_client_input_kex1, NULL,
    NULL, ssh_kexde_server_input_kex2 },
#endif

  { NULL }
};

/* Returns a list of supported key exchange algorithms.  The caller is must
   free the list with ssh_xfree. */

char *ssh_kex_get_supported()
{
  SshBufferStruct buffer;
  int i;
  char *cp;

  ssh_buffer_init(&buffer);
  for (i = 0; ssh_kex_algorithms[i].name != NULL; i++)
    {
      if (ssh_buffer_len(&buffer) > 0)
        ssh_xbuffer_append(&buffer, (unsigned char *) ",", 1);
      ssh_xbuffer_append(&buffer, (unsigned char *) ssh_kex_algorithms[i].name,
                         strlen(ssh_kex_algorithms[i].name));
    }
  ssh_xbuffer_append(&buffer, (unsigned char *) "\0", 1);
  cp = ssh_xstrdup(ssh_buffer_ptr(&buffer));
  ssh_buffer_uninit(&buffer);
  return cp;
}

/* Returns the SshKexType object for the kex method, or NULL if not found. */

SshKexType ssh_kex_lookup(const char *name)
{
  int i;
  for (i = 0; ssh_kex_algorithms[i].name != NULL; i++)
    if (strcmp(ssh_kex_algorithms[i].name, name) == 0)
      return &ssh_kex_algorithms[i];
  return NULL;
}

/* Return a SshHash object that matches the key exchange method, or
 * NULL on error. name is the name of the key exchange method. */

SshHash ssh_kex_allocate_hash(const char *name)
{
  int i;
  SshHash hash;

  for (i = 0; ssh_kex_algorithms[i].name != NULL; i++)
    {
      if (strcmp(ssh_kex_algorithms[i].name, name) == 0)
        {
          if (ssh_hash_allocate(ssh_kex_algorithms[i].hash_name, &hash)
              != SSH_CRYPTO_OK)
            return NULL;
          return hash;
        }
    }
  return NULL;
}
