/*

  sshhostkey.h

  Author: Tomi Salo <ttsalo@ssh.fi>

  Copyright (c) 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Fri Feb 23 21:07:27 2001.

  Functions and data structures for handling multiple host keys.

  */

#ifndef SSHHOSTKEY_H_INCLUDED
#define SSHHOSTKEY_H_INCLUDED

typedef struct SshHostKeysContextRec *SshHostKeysContext;

#include "sshincludes.h"
#include "sshcrypt.h"
#include "sshbuffer.h"
#include "sshadt_list.h"




typedef enum {
  /* The operation was successful and we have at least one hostkey for
     use immediately. */
  SSH_HOSTKEY_OK,

  /* Failure. We don't have anything even resembling a hostkey.
     Aborting recommended. */
  SSH_HOSTKEY_ERROR
} SshHostKeyStatus;

/* Callback type for the ssh_host_key_read_keys function.
   Success will be one of the values above. */
typedef void (*SshHostKeyReadCompletion)(SshHostKeyStatus success,
                                         void *context);

/* Keypair to be used as a host key.  The public part can be an
   certificate. */
typedef struct SshHostKeyRec {
  char *host_key_file;
  char *public_host_key_file;
  Boolean public_part_is_certificate;
  SshPrivateKey private_host_key;
  SshBuffer public_host_key; /* Public key (or certificate) in binary
                                form, ready to be sent over the
                                network. */
  char *public_key_algorithm; /* Public  key format  in draft-standard
                                 notation */
  Boolean valid; /* If this is FALSE, the structure does not contain a
                    valid key and should not be used. */




} *SshHostKey;

struct SshHostKeysContextRec {
  SshADTContainer keylist;
  char *private_key_file;
  char *public_key_file;
  char *certificate_file;
  SshHostKeyReadCompletion completion; /* Completion cb for key reading */
  void *completion_context; /* Completion ctx for key reading */
  SshLogFacility log_facility; /* Copied from SshConfig structure */
  Boolean valid_keys; /* Whether we have valid keys */














};

/* Initialize the list in the config structure to accept hostkeys */
SshHostKeysContext
ssh_host_key_list_init(void);

/* Destroy the list and it's contents */
void
ssh_host_key_list_free(SshHostKeysContext host_key_ctx);

/* Informs the host key manager about a private key file. When a
   private key and either a public key or a certificate have been
   added, a list element for a keypair is added. If the next private
   key is added without specifying the public part for the first
   one, the public part of the first one will be guesstimated and
   the resulting keypair inserted into the list.
   All filenames will be xstrdupped and may be freed immediately
   after the call. */
void
ssh_host_key_add_private_key(const char *private_key_file,
                             SshHostKeysContext ctx);

/* Informs the host key manager about a public key */
void
ssh_host_key_add_public_key(const char *public_host_key_file,
                            SshHostKeysContext ctx);























/* See if we can already add a keypair to the list */
void
ssh_host_key_try_add_keypair(SshHostKeysContext ctx);

/* Functions for accessing the stored keys */

/* Returns an ssh_xmallocated list of all supported host key
   algorithms. If `prefer_pk_algorithms' is TRUE, plain
   public key algorithms are listed first. If it is FALSE,
   they are listed last. */
char *
ssh_host_key_get_supported(Boolean prefer_pk_algorithms);

/* Returns an ssh_xmallocated list of comma-separated algorithm names
   (as defined in transport layer document) */
char *
ssh_host_key_get_algorithmlist(SshHostKeysContext ctx);

/* Returns a pointer to the private key of given algorithm type or
   NULL if a key of given type does not exist. */
SshPrivateKey
ssh_host_key_get_privkey_by_algname(const char *algorithm_name,
                                    SshHostKeysContext ctx);

/* Returns a buffer holding a public key blob of given algorithm type or
   NULL if a key of given type does not exist. */
SshBuffer
ssh_host_key_get_pubkey_by_algname(const char *algorithm_name,
                                   SshHostKeysContext ctx);

#endif /* SSHHOSTKEY_H_INCLUDED */
