/*

  sshhostkey.c

  Author: Tomi Salo <ttsalo@ssh.fi>

  Copyright (c) 2001 SSH Communications Security, Finland
  All rights reserved.

  Created Fri Feb 23 21:08:49 2001.

  Functions and data structures for handling multiple host keys.

  */

#include "sshhostkey.h"
#include "ssh2pubkeyencode.h"
#include "sshcipherlist.h"
#include "sshdsprintf.h"

#define SSH_DEBUG_MODULE "SshHostKey"

void
ssh_host_key_rec_init(void *obj, size_t size, void *context)
{
  SshHostKey hostkey = (SshHostKey)obj;
  memset(hostkey, 0, sizeof(*hostkey));



  hostkey->valid = FALSE;
}

void
ssh_host_key_rec_destroy(void *obj, void *context)
{
  SshHostKey hostkey = (SshHostKey)obj;
  if (hostkey->private_host_key)
    ssh_private_key_free(hostkey->private_host_key);
  if (hostkey->public_host_key)
    ssh_buffer_free(hostkey->public_host_key);
  ssh_xfree(hostkey->public_key_algorithm);
  ssh_xfree(hostkey->host_key_file);
  ssh_xfree(hostkey->public_host_key_file);



}

SshHostKeysContext
ssh_host_key_list_init(void)
{
  SshHostKeysContext ctx =
    ssh_xcalloc(1, sizeof(struct SshHostKeysContextRec));
  /* Create the list, let ADT handle memory allocation for
     objects */
  ctx->keylist = ssh_adt_create_generic(SSH_ADT_LIST,
                                        SSH_ADT_DESTROY,
                                        ssh_host_key_rec_destroy,
                                        SSH_ADT_INIT,
                                        ssh_host_key_rec_init,
                                        SSH_ADT_SIZE,
                                        sizeof(struct SshHostKeyRec),
                                        SSH_ADT_ARGS_END);














  ctx->valid_keys = FALSE;
  return ctx;
}

void
ssh_host_key_list_free(SshHostKeysContext ctx)
{







  ssh_xfree(ctx->public_key_file);
  ssh_xfree(ctx->private_key_file);



  ssh_adt_destroy(ctx->keylist);
  ssh_xfree(ctx);
}

void
ssh_host_key_try_add_keypair(SshHostKeysContext ctx)
{
  if (!ctx->private_key_file)
    return;
  if (ctx->public_key_file || ctx->certificate_file)
    {
      SshHostKey hostkey =
        ssh_adt_get(ctx->keylist, ssh_adt_alloc_to(ctx->keylist,
                                                   SSH_ADT_BEGINNING));

      hostkey->host_key_file = ctx->private_key_file;
      ctx->private_key_file = NULL;
      if (ctx->public_key_file)
        {
          hostkey->public_part_is_certificate = FALSE;
          hostkey->public_host_key_file = ctx->public_key_file;
          ctx->public_key_file = NULL;
        }
      else
        {
          hostkey->public_part_is_certificate = TRUE;
          hostkey->public_host_key_file = ctx->certificate_file;
          ctx->certificate_file = NULL;
        }
      SSH_DEBUG(4, ("Hostkey pair added, priv=%s pub=%s",
                    hostkey->host_key_file,
                    hostkey->public_host_key_file));
    }
}

/* Informs the host key manager about a private key file. */
void
ssh_host_key_add_private_key(const char *private_key_file,
                             SshHostKeysContext ctx)
{
  if (ctx->private_key_file)
    {
      /* If no public part has been specified yet, guess the name of the
         public part and construct a keypair from it and the previous
         private key file */
      if (ssh_dsprintf(&ctx->public_key_file, "%s.pub", ctx->private_key_file)
          > 0)
        ssh_host_key_try_add_keypair(ctx);
      else
        return;
    }
  ssh_xfree(ctx->private_key_file);
  ctx->private_key_file = ssh_xstrdup(private_key_file);
  ssh_host_key_try_add_keypair(ctx);
}

/* Informs the host key manager about a public key */
void
ssh_host_key_add_public_key(const char *public_host_key_file,
                            SshHostKeysContext ctx)
{
  ssh_xfree(ctx->public_key_file);
  ctx->public_key_file = ssh_xstrdup(public_host_key_file);
  ssh_host_key_try_add_keypair(ctx);
}









































/* Returns an ssh_xmallocated list of all supported host key
   algorithms. */
char *
ssh_host_key_get_supported(Boolean prefer_pk_algorithms)
{
  char *pk_algs = NULL;
  
  /* Get the list of plain public key algorithms first */
#ifdef SSHDIST_CRYPT_DSA
  ssh_cipher_list_append(&pk_algs, SSH_SSH_DSS);
#endif /* SSHDIST_CRYPT_DSA */
#ifdef SSHDIST_CRYPT_RSA
#ifdef WITH_RSA
  ssh_cipher_list_append(&pk_algs, SSH_SSH_RSA);  
#endif /* WITH_RSA */
#endif /* SSHDIST_CRYPT_RSA */



























  return pk_algs;
}


/* Returns an ssh_xmallocated list of comma-separated algorithm names
   (as defined in transport layer document) */
char *
ssh_host_key_get_algorithmlist(SshHostKeysContext ctx)
{
  SshBuffer alglist = ssh_xbuffer_allocate();
  SshADTHandle handle;
  char *tmp;
  Boolean hostkey_appended = FALSE;





















  
  handle = ssh_adt_enumerate_start(ctx->keylist);
  while (handle != SSH_ADT_INVALID)
    {
      SshHostKey hostkey;
      hostkey = ssh_adt_get(ctx->keylist, handle);
      
      if (hostkey->valid)
        {
          if (hostkey_appended)
            ssh_xbuffer_append(alglist, (unsigned char*) ",", strlen(","));
          ssh_xbuffer_append(alglist,
                             (unsigned char*) hostkey->public_key_algorithm,
                           strlen(hostkey->public_key_algorithm));
          hostkey_appended = TRUE;
        }
      
      handle = ssh_adt_enumerate_next(ctx->keylist, handle);
    }
  
  tmp = ssh_xcalloc(sizeof(char), (ssh_buffer_len(alglist) + 1));
  strncpy(tmp, (char*) ssh_buffer_ptr(alglist), ssh_buffer_len(alglist));
  ssh_buffer_free(alglist);
  return tmp;
}

/* Returns a pointer to the private key of given algorithm type or
   NULL if a key of given type does not exist. */
SshPrivateKey
ssh_host_key_get_privkey_by_algname(const char *algorithm_name,
                                    SshHostKeysContext ctx)
{
  SshADTHandle handle;















  
  handle = ssh_adt_enumerate_start(ctx->keylist);
  while (handle != SSH_ADT_INVALID)
    {
      SshHostKey hostkey;
      hostkey = ssh_adt_get(ctx->keylist, handle);
      
      if (hostkey->valid &&
          strcmp(algorithm_name, hostkey->public_key_algorithm) == 0)
        return hostkey->private_host_key;
      
      handle = ssh_adt_enumerate_next(ctx->keylist, handle);
    }

  return NULL;
}

/* Returns a buffer holding a public key blob of given algorithm type or
   NULL if a key of given type does not exist. */
SshBuffer
ssh_host_key_get_pubkey_by_algname(const char *algorithm_name,
                                   SshHostKeysContext ctx)
{
  SshADTHandle handle;















  
  handle = ssh_adt_enumerate_start(ctx->keylist);
  while (handle != SSH_ADT_INVALID)
    {
      SshHostKey hostkey;
      hostkey = ssh_adt_get(ctx->keylist, handle);
      
      if (hostkey->valid &&
          strcmp(algorithm_name, hostkey->public_key_algorithm) == 0)
        return hostkey->public_host_key;
      
      handle = ssh_adt_enumerate_next(ctx->keylist, handle);
    }
  
  return NULL;
}
