/*

auths-securid.c

  Authors: Graeme Ahokas <gahokas@ssh.com>
           Sami Lehtinen <sjl@ssh.com>
           
  Copyright (C) 2000, 2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  SecurID authentication, server-side. This calls the SecurID library
  functions to authenticate the user.
*/

/*
  Note on new API usage: We use SD_Check() instead of SD_ClientCheck(),
  because SD_ClientCheck() uses IPv4 addresses exclusively.

  XXX Mark this to the relevant documentation.
*/
#include "sshincludes.h"
#include "sshencode.h"
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "auths-common.h"
#include "sshfsm.h"

#ifdef SSH_SERVER_WITH_SECURID

/* Define this, if you wish to try compilation of this file, without the
   actual SecurID libraries. */
/*  #define TEST_WITHOUT_SECURID */
#ifndef TEST_WITHOUT_SECURID
#ifndef HAVE_OLD_RSA_ACE_API
/* With RSA ACE 5.0 API, only one header is needed. */
#ifdef HAVE_ACEXPORT_H
#include <acexport.h>
#endif /* HAVE_ACEXPORT_H */
#else /* HAVE_OLD_RSA_ACE_API */
#ifdef HAVE_SDI_ATHD_H
#include <sdi_athd.h>
#endif /* HAVE_SDI_ATHD_H */
#ifdef HAVE_SDI_DEFS_H
#include <sdi_defs.h>
#endif /* HAVE_SDI_DEFS_H */
#ifdef HAVE_SDACMVLS_H
#include <sdacmvls.h>
#endif /* HAVE_SDACMVLS_H */
#ifdef HAVE_SDCONF_H
#include <sdconf.h>
#endif /* HAVE_SDCONF_H */
#endif /* HAVE_OLD_RSA_ACE_API */
#else /* TEST_WITHOUT_SECURID */
#define HAVE_OLD_RSA_ACE_API 1
#include "auths-securid-dummy.h"
#endif /* TEST_WITHOUT_SECURID */

#define SSH_DEBUG_MODULE "Ssh2AuthSecurIDServer"

#define MIN_PIN_LEN 4
#define MAX_PIN_LEN 8

#ifdef HAVE_OLD_RSA_ACE_API
#ifdef HAVE_SDCONF_H
union config_record configure;
#endif /* HAVE_SDCONF_H */
#endif /* HAVE_OLD_RSA_ACE_API */

typedef struct SshServerSecurIDAuthRec
{
  SshFSM fsm;
  SshFSMThread main_thread;
  SshUser uc;

  SshBuffer client_packet;
  SshBuffer response_packet;
  Boolean pin_change_success;
  SshConfig config;

  /* handle for the ace server */
#ifndef HAVE_OLD_RSA_ACE_API
  SDI_HANDLE sd;
#else /* HAVE_OLD_RSA_ACE_API */
  struct SD_CLIENT *sd;
#endif /* HAVE_OLD_RSA_ACE_API */
  
  void **state_placeholder;
  SshAuthServerCompletionProc completion_proc;
  void *completion_context;
} *SshServerSecurIDAuth;

SSH_FSM_STEP(server_securid_start);
SSH_FSM_STEP(server_securid_auth_success);
SSH_FSM_STEP(server_securid_get_next_code);
SSH_FSM_STEP(server_securid_finish);
SSH_FSM_STEP(server_securid_process_next_code);
SSH_FSM_STEP(server_securid_reject);
SSH_FSM_STEP(server_securid_reject_and_disable);
SSH_FSM_STEP(server_securid_get_new_pin);
SSH_FSM_STEP(server_securid_process_new_pin);

/* Initializes connection with the SecurID server.
   Returns FALSE if no errors, TRUE otherwise */
Boolean ssh_server_securid_init(
#ifndef HAVE_OLD_RSA_ACE_API
                                SDI_HANDLE *sd
#else /* HAVE_OLD_RSA_ACE_API */
                                struct SD_CLIENT **sd
#endif /* HAVE_OLD_RSA_ACE_API */
                                )
{

#ifndef HAVE_OLD_RSA_ACE_API
  if (!AceInitialize())
    {
      SSH_DEBUG(2,("ACE initialization unsuccessful."));
      return TRUE;
    }
  
  if (SD_Init(sd) != ACM_OK)
    {
      SSH_DEBUG(2,("Error initializing connection with SecurID server."));
      return TRUE;
    }
#else /* HAVE_OLD_RSA_ACE_API */
  if (*sd == NULL)
    return TRUE;

  if (creadcfg())
    {
      SSH_DEBUG(2,("Error reading sdconf.rec"));
      return TRUE;
    }

  memset(*sd, 0, sizeof(**sd));

  if (sd_init(*sd))
    {
      SSH_DEBUG(2,("Error initializing connection with SecurID server."));
      return TRUE;
    }
#endif /* HAVE_OLD_RSA_ACE_API */  

  return FALSE;
}

/* Begins connection with the securid server and asks the client for the
   user's passphrase. */
SSH_FSM_STEP(server_securid_start)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;
  char *passcode;
  size_t length;
  int ret_code;

  if (ssh_server_securid_init(&gdata->sd))
    {
      ssh_buffer_free(gdata->client_packet);
      SSH_DEBUG(2,("Error connecting to SecurID server."));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }

  /* Connection with securid server established, so check passcode. */
  SSH_DEBUG(2,("Connection with SecurID server established."));

  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN,
                        NULL,
                        SSH_FORMAT_UINT32_STR,
                        &passcode, &length,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_free(gdata->client_packet);
      SSH_DEBUG(4,("Client packet decode failed.\r\n"));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }

#ifndef HAVE_OLD_RSA_ACE_API
  if (SD_Lock(gdata->sd, ssh_user_name(gdata->uc)) != ACM_OK)
    {
      SSH_DEBUG(2, ("Unable to lock server."));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }

  ret_code = SD_Check(gdata->sd, passcode, ssh_user_name(gdata->uc));
#else /* HAVE_OLD_RSA_ACE_API */
  /* XXX could check now for valid length of passcode */
  ret_code = sd_check(passcode, ssh_user_name(gdata->uc), gdata->sd);
#endif /* HAVE_OLD_RSA_ACE_API */
  ssh_xfree(passcode);
  ssh_buffer_free(gdata->client_packet);

  switch(ret_code)
    {
    case ACM_OK:

      SSH_DEBUG(2,("User AUTHENTICATED.\r\n"));
      SSH_FSM_SET_NEXT(server_securid_auth_success);
      return SSH_FSM_CONTINUE;

    case ACM_ACCESS_DENIED:

      gdata->config->securid_guesses--;
      if (gdata->config->securid_guesses <= 0)
        {
          SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
          return SSH_FSM_CONTINUE;
        }

      SSH_DEBUG(2,("Access denied from SecurID server"));
      SSH_FSM_SET_NEXT(server_securid_reject);
      return SSH_FSM_CONTINUE;

    case ACM_NEXT_CODE_REQUIRED:

      SSH_DEBUG(2,("Next code required"));
      SSH_FSM_SET_NEXT(server_securid_get_next_code);
      return SSH_FSM_CONTINUE;

    case ACM_NEW_PIN_REQUIRED:

      SSH_DEBUG(2,("New PIN required."));
      SSH_FSM_SET_NEXT(server_securid_get_new_pin);
      return SSH_FSM_CONTINUE;

    default:
      SSH_DEBUG(4,("Unknown return code from SecurID server sd_check"));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }
  SSH_NOTREACHED;
}

/* Successful authentication, send ACCEPTED msg and quit */
SSH_FSM_STEP(server_securid_auth_success)
{

  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_ACCEPTED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT(server_securid_finish);
  return SSH_FSM_CONTINUE;
}

/* Sends a CHALLENGE msg to client for next token */
SSH_FSM_STEP(server_securid_get_next_code)
{

  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;

  gdata->response_packet = ssh_xbuffer_allocate();

  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_CHAR,
                    (unsigned int)SSH_MSG_USERAUTH_SECURID_CHALLENGE,
                    SSH_FORMAT_BOOLEAN,
                    TRUE,
                    SSH_FORMAT_END);

  (*gdata->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK,
                            gdata->response_packet,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT(server_securid_process_next_code);
  return SSH_FSM_SUSPENDED;
}


/* Finishes the fsm */
SSH_FSM_STEP(server_securid_finish)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;

  /* close connection to securid server */
#ifndef HAVE_OLD_RSA_ACE_API
  SD_Close(gdata->sd);
#else /* HAVE_OLD_RSA_ACE_API */
  ssh_xfree(gdata->sd);
  sd_close();
#endif /* HAVE_OLD_RSA_ACE_API */
  ssh_fsm_destroy(gdata->fsm);
  gdata->fsm = NULL;

  *gdata->state_placeholder = NULL;

  return SSH_FSM_FINISH;
}

/* receives challenge response from client and tests*/
SSH_FSM_STEP(server_securid_process_next_code)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;
  char *passcode;
  size_t length;
  int ret_code;

  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN,
                        NULL,
                        SSH_FORMAT_UINT32_STR,
                        &passcode, &length,
                        SSH_FORMAT_END) == 0)
    {
      ssh_xfree(gdata->client_packet);
      SSH_DEBUG(4,("Client packet decode failed."));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }

  /* we're verifying the challenge from the server for the next token
     or making user authenticate with new pin */

  /* if we have changed the pin, we must call sd_check, NOT sd_next, even
     though we actually want the next token. unintuitive, i know... */
  if (gdata->pin_change_success)
    {
      gdata->pin_change_success = FALSE;
#ifndef HAVE_OLD_RSA_ACE_API
      ret_code = SD_Check(gdata->sd, passcode, ssh_user_name(gdata->uc));
#else /* HAVE_OLD_RSA_ACE_API */
      /* XXX could check now for valid length of passcode */
      ret_code = sd_check(passcode, ssh_user_name(gdata->uc), gdata->sd);
#endif /* HAVE_OLD_RSA_ACE_API */      
    }
  else
    {
#ifndef HAVE_OLD_RSA_ACE_API
      ret_code = SD_Next(gdata->sd, passcode);
#else /* HAVE_OLD_RSA_ACE_API */
      ret_code = sd_next(passcode, gdata->sd);
#endif /* HAVE_OLD_RSA_ACE_API */
    }
  
  ssh_xfree(passcode);
  ssh_buffer_free(gdata->client_packet);

  switch(ret_code)
    {
    case ACM_OK:

      SSH_DEBUG(2,("User AUTHENTICATED by challenge."));
      SSH_FSM_SET_NEXT(server_securid_auth_success);
      return SSH_FSM_CONTINUE;

    case ACM_ACCESS_DENIED:

      SSH_DEBUG(2,("User access denied from challenge."));
      SSH_FSM_SET_NEXT(server_securid_reject);
      return SSH_FSM_CONTINUE;

    case ACM_NEXT_CODE_REQUIRED:

      /* this probably isn't necessary, as we likely wouldn't get
         a next code req msg after a successful authentication with
         a new pin, but just to be safe */
      SSH_DEBUG(2,("Next code required"));
      SSH_FSM_SET_NEXT(server_securid_get_next_code);
      return SSH_FSM_CONTINUE;

    default:

      SSH_DEBUG(4,("Unknown return code from securid server sd_next"));
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }
}

/* Sends a reject authentication msg to client and stops fsm */
SSH_FSM_STEP(server_securid_reject)
{

  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;
  SSH_PRECOND(gdata);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_REJECTED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT(server_securid_finish);

  return SSH_FSM_CONTINUE;
}

/* sends reject and method disable msg and stops fsm */
SSH_FSM_STEP(server_securid_reject_and_disable)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;
  SSH_PRECOND(gdata);

  *gdata->state_placeholder = NULL;

  (*gdata->completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                            NULL,
                            gdata->completion_context);

  SSH_FSM_SET_NEXT(server_securid_finish);
  return SSH_FSM_CONTINUE;
}


/* sends the client a new pin request, complete with information
   on the limitations of the pin, and a server assigned pin if
   the user is not permitted to choose */
SSH_FSM_STEP(server_securid_get_new_pin)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;
  unsigned int user_selectable;
  Boolean alphanumeric;
  char *system_pin;
  SshUInt32 min_pin_len, max_pin_len;
#ifndef HAVE_OLD_RSA_ACE_API
  SD_PIN pin_params;
#endif /* HAVE_OLD_RSA_ACE_API */
  
  gdata->response_packet = ssh_xbuffer_allocate();

#ifndef HAVE_OLD_RSA_ACE_API
  if (AceGetPinParams(gdata->sd, &pin_params) != ACE_SUCCESS)
    {
      SSH_DEBUG(2,("Error getting PIN params."));
      ssh_buffer_free(gdata->response_packet);
      gdata->response_packet = NULL;
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }
  
  switch(pin_params.Selectable)
    {
    case CANNOT_CHOOSE_PIN:
      user_selectable = SSH_SECURID_CANNOT_CHOOSE_PIN;
      break;
    case MUST_CHOOSE_PIN:
      user_selectable = SSH_SECURID_MUST_CHOOSE_PIN;
      break;
    default:
      user_selectable = SSH_SECURID_USER_SELECTABLE_PIN;
      break;
    }

  if (pin_params.Alphanumeric == 0)
    alphanumeric = FALSE;
  else
    alphanumeric = TRUE;

  /* system PIN stuff. */
  SSH_ASSERT(pin_params.System);
  
  if (user_selectable == SSH_SECURID_MUST_CHOOSE_PIN)
    system_pin = ssh_xstrdup("");
  else
    system_pin = ssh_xstrdup(pin_params.System);

  min_pin_len = (SshUInt32)pin_params.Min;
  max_pin_len = (SshUInt32)pin_params.Max;

#else /* HAVE_OLD_RSA_ACE_API */
  /* assign the user_selectable field to inform
     the user about their limitations in pin selection*/
  if (gdata->sd->user_selectable == CANNOT_CHOOSE_PIN)
    user_selectable = SSH_SECURID_CANNOT_CHOOSE_PIN;
  else if (gdata->sd->user_selectable == MUST_CHOOSE_PIN)
    user_selectable = SSH_SECURID_MUST_CHOOSE_PIN;
  else
    user_selectable = SSH_SECURID_USER_SELECTABLE_PIN;

  if (gdata->sd->alphanumeric == 0)
    alphanumeric = FALSE;
  else
    alphanumeric = TRUE;

  if (user_selectable == SSH_SECURID_MUST_CHOOSE_PIN)
    system_pin = ssh_xstrdup("");
  else
    system_pin = ssh_xstrdup(gdata->sd->system_pin);

  max_pin_len = (SshUInt32)gdata->sd->max_pin_len;
  min_pin_len = (SshUInt32)gdata->sd->min_pin_len;
#endif /* HAVE_OLD_RSA_ACE_API */
  
  ssh_encode_buffer(gdata->response_packet,
                    SSH_FORMAT_CHAR,
                    SSH_MSG_USERAUTH_SECURID_NEW_PIN_REQD,
                    SSH_FORMAT_UINT32, min_pin_len,
                    SSH_FORMAT_UINT32, max_pin_len,
                    SSH_FORMAT_CHAR, (unsigned int)user_selectable,
                    SSH_FORMAT_BOOLEAN, alphanumeric,
                    SSH_FORMAT_UINT32_STR, system_pin, strlen(system_pin),
                    SSH_FORMAT_END);

  /* send packet to notify client of the message, and wait for reply */
  (*gdata->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK,
                            gdata->response_packet,
                            gdata->completion_context);

  ssh_xfree(system_pin);

  SSH_FSM_SET_NEXT(server_securid_process_new_pin);
  return SSH_FSM_SUSPENDED;
}


/* takes the pin from the client and informs the securid server */
SSH_FSM_STEP(server_securid_process_new_pin)
{
  SshServerSecurIDAuth gdata = (SshServerSecurIDAuth) fsm_context;

  int ret_code;
  char *new_pin;
  size_t new_pin_len;
#ifdef HAVE_OLD_RSA_ACE_API
  char cancel_new_pin;
#endif /* HAVE_OLD_RSA_ACE_API */

  /* decode the packet from the client */
  if (ssh_decode_buffer(gdata->client_packet,
                        SSH_FORMAT_BOOLEAN, NULL,
                        SSH_FORMAT_UINT32_STR, &new_pin, &new_pin_len,
                        SSH_FORMAT_END) == 0)
    {
      ssh_buffer_free(gdata->client_packet);
      SSH_FSM_SET_NEXT(server_securid_reject_and_disable);
      return SSH_FSM_CONTINUE;
    }

  ssh_buffer_free(gdata->client_packet);

  if (new_pin_len > MAX_PIN_LEN || new_pin_len < MIN_PIN_LEN)
    {
      SSH_DEBUG(2, ("Invalid PIN len %ld.", new_pin_len));
      goto pin_change_error;
    }

  /* XXX if user MUST_CHOOSE_PIN, verify that ``new_pin'' and
     ``system_pin'' are equivalent. */
#ifndef HAVE_OLD_RSA_ACE_API
  /* New API consideres PIN change canceled if ``new_pin'' is
     zero-length or NULL. */
  ret_code = SD_Pin(gdata->sd, new_pin);
#else /* HAVE_OLD_RSA_ACE_API */
  if (strlen(new_pin) == 0)
    cancel_new_pin = 1;
  else
    cancel_new_pin = 0;

  /* verify the new pin with the securid server */
  ret_code = sd_pin(new_pin, cancel_new_pin, gdata->sd);
#endif /* HAVE_OLD_RSA_ACE_API */
  
  ssh_xfree(new_pin);

  if (ret_code == ACM_NEW_PIN_ACCEPTED)
    {
      SSH_DEBUG(2,("New PIN operation successful"));
      gdata->pin_change_success = TRUE;
      /* new pin is ok so send a passphrase next token msg to client */
      SSH_FSM_SET_NEXT(server_securid_get_next_code);
      return SSH_FSM_CONTINUE;
    }

 pin_change_error:
  SSH_DEBUG(2,("New PIN operation failed."));
  /* new pin operation failed */
  SSH_FSM_SET_NEXT(server_securid_reject);
  return SSH_FSM_CONTINUE;
}

SshFSMStateDebugStruct server_securid_states[] =
{
  { "server_securid_start", "Starting securid authentication",
     server_securid_start },
  { "server_securid_finish", "Finished authentication",
     server_securid_finish },
  { "server_securid_reject", "Rejects authentication",
     server_securid_reject },
  { "server_securid_reject_and_disable", "Rejects and disables authentication",
     server_securid_reject_and_disable },
  { "server_securid_get_next_code", "Asks client for next passcode",
     server_securid_get_next_code },
  { "server_securid_process_next_code", "Checks next code with server",
     server_securid_process_next_code },
  { "server_securid_get_new_pin", "Asks or gives the user a new pin",
     server_securid_get_new_pin },
  { "server_securid_process_new_pin", "Register new pin with server",
     server_securid_process_new_pin },
  { "server_securid_auth_success", "Sends successful auth msg to client",
     server_securid_auth_success }
};

/* SecurID authentication, server side. */

void ssh_server_auth_securid(SshAuthServerOperation op,
                            const char *user,
                            SshUser uc,
                            SshBuffer packet,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            void **longtime_placeholder,
                            SshAuthServerCompletionProc completion_proc,
                            void *completion_context,
                            void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  SshServerSecurIDAuth state = *state_placeholder;

  SSH_TRACE(6, ("securid auth."));

  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:

      /* receives the start securid authentication request
         from the client. will send AUTH_SUCCESS or request
         new pin or challenge */

      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_SECURID))
        {
          (*completion_proc) (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                              packet,
                              completion_context);
          return;
        }


      if(ssh_user_uid(uc) == UID_ROOT &&
         (config->permit_root_login == SSH_ROOTLOGIN_FALSE ||
          config->permit_root_login == SSH_ROOTLOGIN_NOPWD))
        {
          /* XXX Add client addresses etc. */
          SSH_DEBUG(2, ("root logins are not permitted."));
          ssh_log_event(config->log_facility,
                        SSH_LOG_WARNING,
                        "root login denied for user %s.",
                        ssh_user_name(uc));

          (*completion_proc)
            (SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED, packet,
             completion_context);

          return;
        }

      /* start up the fsm */

      SSH_ASSERT(*state_placeholder == NULL);
      state = ssh_xcalloc(1, sizeof(*state));
      state->fsm = ssh_fsm_create(state);

      ssh_fsm_register_debug_names(state->fsm, server_securid_states,
                                   SSH_FSM_NUM_STATES(server_securid_states));
      state->uc = uc;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      state->client_packet = packet;
      state->response_packet = NULL;
#ifdef HAVE_OLD_RSA_ACE_API
      state->sd = ssh_xcalloc(1, (sizeof(*state->sd)));
#endif /* HAVE_OLD_RSA_ACE_API */
      state->pin_change_success = FALSE;
      state->config = config;
      *state_placeholder = state;

      state->main_thread = ssh_fsm_thread_create(state->fsm,
                                                 server_securid_start,
                                                 NULL_FNPTR, NULL_FNPTR, NULL);

      return;

    case SSH_AUTH_SERVER_OP_CONTINUE:

      /* pass the packet to the fsm for processing */
      SSH_ASSERT(state && state->main_thread);
      state->client_packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;

      ssh_fsm_continue(state->main_thread);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:

      SSH_ASSERT(state && state->main_thread);
      ssh_fsm_set_next(state->main_thread, server_securid_finish);
      ssh_fsm_continue(state->main_thread);
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet, completion_context);
      return;

    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      /* Clean up state. */
      if (state)
        if (state->main_thread)
          {
            ssh_fsm_set_next(state->main_thread, server_securid_finish);
            ssh_fsm_continue(state->main_thread);
          }

      *state_placeholder = NULL;
      (*completion_proc) (SSH_AUTH_SERVER_REJECTED, packet,
                          completion_context);
      return;

    default:
      ssh_fatal("ssh_server_auth_securid: unknown op %d", (int)op);
    }

  SSH_NOTREACHED;
}

#endif /* SSH_SERVER_WITH_SECURID */
