/*

  auths-kbd-interactive.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 2000-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Keyboard interactive authentication, server side. Done according to
  draft-ietf-secsh-auth-kbdinteract-02.txt.

*/

#include "sshincludes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#include "sshauth.h"
#include "sshmsgs.h"
#include "sshuser.h"
#include "sshserver.h"
#include "sshconfig.h"
#include "sshfsm.h"
#include "sshtimeouts.h"
#include "sshencode.h"
#include "sshmsgs.h"
#include "sshappcommon.h"
#include "auths-common.h"
#include "auths-kbd-int-submethods.h"

#define SSH_DEBUG_MODULE "Ssh2AuthKbdInteractiveServer"

typedef struct SshServerKbdIntAuthRec {
  SshUser uc;
  char *user;
  SshBuffer packet;
  SshAuthServerCompletionProc completion_proc;
  void *completion_context;
  void **state_placeholder;
  
  SshFSM fsm;
  SshFSMThread main_thread;

  char *instruction;
  size_t cur_num_reqs;
  char **cur_reqs;
  Boolean *cur_echo;

  size_t num_resps;
  char **resps;
  
  SshKbdIntSubMethodCB method_cb;
  void *method_ctx;

  Boolean fake_transaction;

  SshAuthServerResult result;
  
  SshServer server;
  SshConfig config;
  int submethod_index;
  SshAuthKbdIntSubMethods methods;
} SshServerKbdIntAuthStruct, *SshServerKbdIntAuth;

/* Forward declarations. */
SSH_FSM_STEP(ssh_server_kbd_int_next_submethod);
SSH_FSM_STEP(ssh_server_kbd_int_policy);
SSH_FSM_STEP(ssh_server_kbd_int_create_request);
SSH_FSM_STEP(ssh_server_kbd_int_process_response);
SSH_FSM_STEP(ssh_server_kbd_int_finish);

SSH_FSM_STEP(ssh_server_kbd_int_start)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  /* Do init. */
  state->methods = ssh_server_kbd_int_submethods_init(state->server,
                                                      state->user,
                                                      state->uc);

  SSH_FSM_SET_NEXT(ssh_server_kbd_int_policy);
  return SSH_FSM_CONTINUE;
}

void kbd_int_conv_cb(SshKbdIntResult code,
                     const char *instruction,
                     size_t num_reqs,
                     char **reqs,
                     Boolean *echo,
                     SshKbdIntSubMethodCB method_cb,
                     void *method_ctx,
                     void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) ssh_fsm_get_gdata(thread);
  int i;

  /* Free stored data. */
  ssh_xfree(state->instruction);
  state->instruction = NULL;
  
  if (state->cur_reqs)
    {
      for (i = 0; i < state->cur_num_reqs; i++)
        ssh_xfree(state->cur_reqs[i]);
      ssh_xfree(state->cur_reqs);
      state->cur_reqs = NULL;
    }

  if (state->cur_echo)
    {
      ssh_xfree(state->cur_echo);
      state->cur_echo = NULL;
    }
  
  if (state->resps != NULL)
    {
      SSH_DEBUG(4, ("Freeing response array."));
      for (i = 0; i < state->num_resps; i++)
        ssh_xfree(state->resps[i]);
      ssh_xfree(state->resps);
      state->resps = NULL;
    }

  state->method_cb = NULL_FNPTR;
  state->method_ctx = NULL;

  if (code != SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET)
    {
      SSH_ASSERT(instruction == NULL);
      SSH_ASSERT(num_reqs == 0);
      SSH_ASSERT(reqs == NULL);
      SSH_ASSERT(echo == NULL);
      SSH_ASSERT(method_cb == NULL_FNPTR);
      SSH_ASSERT(method_ctx == NULL);
      
      SSH_FSM_SET_NEXT(ssh_server_kbd_int_policy);
      /* Mark method result. */
      if (code == SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS &&
          !state->fake_transaction)
        {
          state->methods->submethods[state->submethod_index].status =
            SSH_KBDINT_SUBMETHOD_SUCCESS;
          SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);  
        }
      else
        {
          state->methods->submethods[state->submethod_index].status =
            SSH_KBDINT_SUBMETHOD_FAILED;

          SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
        }
      return;
    }

  SSH_ASSERT(instruction != NULL);
  state->instruction = ssh_xstrdup(instruction);
  state->cur_num_reqs = num_reqs;
  state->cur_reqs = ssh_xcalloc(num_reqs, sizeof(char *));
  for (i = 0; i < num_reqs; i++)
    {
      SSH_ASSERT(reqs[i] != NULL);
      SSH_DEBUG(2, ("Adding request '%s'.", reqs[i]));
      state->cur_reqs[i] = ssh_xstrdup(reqs[i]);
    }
  state->cur_echo = ssh_xmemdup(echo, num_reqs*(sizeof(Boolean)));
  state->method_cb = method_cb;
  state->method_ctx = method_ctx;
  
  SSH_FSM_SET_NEXT(ssh_server_kbd_int_create_request);
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);  
}

SSH_FSM_STEP(ssh_server_kbd_int_next_submethod)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  SSH_PRECOND(state != NULL);
  
  /* Choose next submethod. */
  SSH_FSM_ASYNC_CALL(ssh_server_kbd_int_submethod_init
                     (state->methods,
                      &state->methods->submethods[state->submethod_index],
                      kbd_int_conv_cb, thread));

}

SSH_FSM_STEP(ssh_server_kbd_int_create_request)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  const char *name;
  char *lang_tag;
  size_t ret;
  int i;
  
  SSH_PRECOND(state != NULL);

  SSH_FSM_SET_NEXT(ssh_server_kbd_int_process_response);
  
  /* Create actual INFO_REQUEST message. */
  ssh_buffer_clear(state->packet);

  name = ssh_server_kbd_int_submethod_get_name
    (&state->methods->submethods[state->submethod_index]);
  /* XXX lang tag*/
  lang_tag = "en";
  
  ret = ssh_encode_buffer(state->packet,
                          SSH_FORMAT_CHAR, SSH_MSG_USERAUTH_INFO_REQUEST,
                          SSH_FORMAT_UINT32_STR, name, strlen(name),
                          SSH_FORMAT_UINT32_STR, state->instruction,
                          strlen(state->instruction),
                          SSH_FORMAT_UINT32_STR, lang_tag, strlen(lang_tag),
                          SSH_FORMAT_UINT32, (SshUInt32) state->cur_num_reqs,
                          SSH_FORMAT_END);

  SSH_VERIFY(ret);
  for (i = 0; i < state->cur_num_reqs; i++)
    {
      if (strlen(state->cur_reqs[i]) == 0)
        {
          ssh_warning("auth-kbd-int: got an empty request from submethod "
                      "`%s', which is not allowed by the protocol. Failing "
                      "submethod.",
                      state->methods->submethods[state->submethod_index].name);
          SSH_FSM_SET_NEXT(ssh_server_kbd_int_policy);
          state->methods->submethods[state->submethod_index].status =
            SSH_KBDINT_SUBMETHOD_FAILED;
          ssh_buffer_clear(state->packet);
          return SSH_FSM_CONTINUE;
        }
      ret = ssh_encode_buffer(state->packet,
                              SSH_FORMAT_UINT32_STR, state->cur_reqs[i],
                              strlen(state->cur_reqs[i]),
                              SSH_FORMAT_BOOLEAN, state->cur_echo[i],
                              SSH_FORMAT_END);
      
      SSH_VERIFY(ret);
    }

  (*state->completion_proc)(SSH_AUTH_SERVER_CONTINUE_WITH_PACKET_BACK_SPECIAL,
                            state->packet, state->completion_context);
  /* wait for INFO_RESPONSE message */
  return SSH_FSM_SUSPENDED;
}

SSH_FSM_STEP(ssh_server_kbd_int_process_response)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  size_t ret;
  SshUInt32 num_resps, i;
  unsigned int packet_type;
  char *resp;

  SSH_FSM_SET_NEXT(ssh_server_kbd_int_create_request);
  
  ret = ssh_decode_buffer(state->packet,
                          SSH_FORMAT_CHAR, &packet_type,
                          SSH_FORMAT_UINT32, &num_resps,
                          SSH_FORMAT_END);

  if (ret == 0)
    {
      goto proto_error;
    }

  if (num_resps != state->cur_num_reqs)
    {
      ssh_log_event(state->config->log_facility, SSH_LOG_ERROR,
                    "auth-kbd-int: client sent us fewer responses than we "
                    "sent requests (sent: %ld, recv: %ld).",
                    state->cur_num_reqs, num_resps);
      goto proto_error;
    }

  SSH_ASSERT(state->resps == NULL);
  state->resps = ssh_xcalloc(num_resps, sizeof(char *));
  state->num_resps = num_resps;
  
  for (i = 0; i < num_resps; i++)
    {

      ret = ssh_decode_buffer(state->packet,
                              SSH_FORMAT_UINT32_STR, &resp, NULL,
                              SSH_FORMAT_END);
      if (ret == 0)
        goto proto_error;

      state->resps[i] = resp;
    }

  /* Pass replys to submethod. */
  SSH_FSM_ASYNC_CALL((*state->method_cb)(state->num_resps, state->resps,
                      FALSE, state->method_ctx));
  SSH_NOTREACHED;
  
 proto_error:
  ssh_warning("Protocol error in keyboard-interactive when parsing "
              "received response packet.");
  SSH_DEBUG_HEXDUMP(6, ("Rest of received packet:"),
                    ssh_buffer_ptr(state->packet),
                    ssh_buffer_len(state->packet));
  *state->state_placeholder = NULL;
  (*state->completion_proc)(SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED,
                            state->packet, state->completion_context);
  SSH_FSM_SET_NEXT(ssh_server_kbd_int_finish);
  return SSH_FSM_CONTINUE;
}

void failed_timeout_cb(void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  SshServerKbdIntAuth state;
  SSH_PRECOND(thread != NULL);
  state = (SshServerKbdIntAuth)ssh_fsm_get_gdata(thread);
  SSH_PRECOND(state != NULL);
  
  *state->state_placeholder = NULL;
  (*state->completion_proc)(state->result, state->packet,
                            state->completion_context);
  ssh_fsm_continue(thread);
}

SSH_FSM_STEP(ssh_server_kbd_int_policy)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  int i, num_optional = 0, optional_successes = 0, optional_undefined = 0;
  int num_successful = 0, not_used = 0;
  int first_undefined_optional = -1;
  
  /* decide on continuation. */
  /* Required methods */
  SSH_FSM_SET_NEXT(ssh_server_kbd_int_next_submethod);
  
  for (i = 0; state->methods->submethods[i].name; i++)
    {
      if (state->methods->submethods[i].status == SSH_KBDINT_SUBMETHOD_SUCCESS)
        num_successful++;

      if (state->methods->submethods[i].use == SSH_KBDINT_SUBMETHOD_NOTUSED)
        not_used++;
      
      if (state->methods->submethods[i].use == SSH_KBDINT_SUBMETHOD_REQUIRED)
        {
          if (state->methods->submethods[i].status ==
              SSH_KBDINT_SUBMETHOD_FAILED)
            {
              if (!state->fake_transaction)
                {
                  SSH_DEBUG(2, ("Some required submethods have failed."));
                  state->fake_transaction = TRUE;
                }
            }
          else if (state->methods->submethods[i].status ==
                   SSH_KBDINT_SUBMETHOD_UNDEFINED)
            {
              SSH_DEBUG(2, ("starting submethod '%s'.",
                            state->methods->submethods[i].name));
              state->submethod_index = i;
              return SSH_FSM_CONTINUE;
            }
        }
    }

  /* Just a sanity check, the configuration layer must check this (that
     no submethods are needed). */
  if (not_used == i)
    {
      ssh_warning("auth-kbd-int: no submethods configured as either "
                  "optional or required, failing.");
      goto auth_failed;
    }

  for (i = 0; state->methods->submethods[i].name; i++)
    if (state->methods->submethods[i].use == SSH_KBDINT_SUBMETHOD_OPTIONAL)
      {
        num_optional++;
        if (state->methods->submethods[i].status ==
            SSH_KBDINT_SUBMETHOD_SUCCESS)
          optional_successes++;
        if (state->methods->submethods[i].status ==
            SSH_KBDINT_SUBMETHOD_UNDEFINED)
          {
            if (first_undefined_optional == -1)
              first_undefined_optional = i;
            optional_undefined++;
          }
      }

  if (state->config->auth_kbd_int_optional_needed > num_optional)
    {
      ssh_warning("auth-kbd-int: not enough optional submethods to "
                  "satisfy NumOptional requirement (%d optional auth%s, "
                  "NumOptional %d). Failing.", num_optional,
                  num_optional == 1 ? "" : "s",
                  state->config->auth_kbd_int_optional_needed);
      goto auth_failed;
    }

  if (state->fake_transaction)
    {
      SSH_DEBUG(2, ("The authorization was faked, failing."));
      goto auth_failed;
    }
  
  if ((optional_successes >= state->config->auth_kbd_int_optional_needed ||
       num_optional == 0) &&
      num_successful > 0)
    {
      SSH_DEBUG(2, ("No more optional needed (%d success%s).",
                    optional_successes, optional_successes == 1 ? "" : "es"));
      /* If all required and required amount of optional submethods are
         successful, return SUCCESS. */  
      /* success, end auth */
      *state->state_placeholder = NULL;
      (*state->completion_proc)(SSH_AUTH_SERVER_ACCEPTED,
                                state->packet, state->completion_context);
      SSH_FSM_SET_NEXT(ssh_server_kbd_int_finish);
      return SSH_FSM_CONTINUE;
    }

  if (state->config->auth_kbd_int_optional_needed - optional_successes >
      optional_undefined)
    {
      SSH_DEBUG(2, ("Auth can't succeed, too many failed (%d).",
                    num_optional - optional_successes - optional_undefined));
      goto auth_failed;
    }

  /* Final check. */
  if (first_undefined_optional == -1)
    goto auth_failed;

  state->submethod_index = first_undefined_optional;
  return SSH_FSM_YIELD;

 auth_failed:
  SSH_DEBUG(2, ("auth-kbd-int: User '%s' failed to authenticate.",
                state->user));

  state->config->auth_kbd_int_retries--;

  if (state->config->auth_kbd_int_retries <= 0)
    state->result = SSH_AUTH_SERVER_REJECTED_AND_METHOD_DISABLED;
  else
    state->result = SSH_AUTH_SERVER_REJECTED;

  ssh_register_timeout(state->config->auth_interactive_failure_timeout,
                       0L, failed_timeout_cb, thread);
  SSH_FSM_SET_NEXT(ssh_server_kbd_int_finish);
  return SSH_FSM_SUSPENDED;
}

void destroy_real_cb(void *context)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) context;
  ssh_server_kbd_int_submethods_uninit(state->methods);
  memset(state, 'F', sizeof(*state));
  ssh_xfree(state);
}

SSH_FSM_STEP(ssh_server_kbd_int_finish)
{
  SshServerKbdIntAuth state = (SshServerKbdIntAuth) fsm_context;
  int i;

  ssh_cancel_timeouts(failed_timeout_cb, thread);
  
  /* Abort current, kill FSM etc. */
  if (state->method_cb)
    (*state->method_cb)(0, NULL, TRUE, state->method_ctx);
  
  ssh_xfree(state->instruction);
  
  if (state->cur_reqs)
    {
      for (i = 0; i < state->cur_num_reqs; i++)
        ssh_xfree(state->cur_reqs[i]);
      ssh_xfree(state->cur_reqs);
    }

  ssh_xfree(state->cur_echo);
  
  if (state->resps != NULL)
    {
      SSH_DEBUG(4, ("Freeing response array."));
      for (i = 0; i < state->num_resps; i++)
        ssh_xfree(state->resps[i]);
      ssh_xfree(state->resps);
    }

  ssh_xfree(state->user);
  ssh_fsm_destroy(state->fsm);

  ssh_register_timeout(0L, 0L, destroy_real_cb, state);
  return SSH_FSM_FINISH;
}

void ssh_server_auth_kbd_interact(SshAuthServerOperation op,
                                  const char *user,
                                  SshUser uc,
                                  SshBuffer packet,
                                  const unsigned char *session_id,
                                  size_t session_id_len,
                                  void **state_placeholder,
                                  void **longtime_placeholder,
                                  SshAuthServerCompletionProc completion_proc,
                                  void *completion_context,
                                  void *method_context)
{
  SshServer server = (SshServer)method_context;
  SshConfig config = server->config;
  SshServerKbdIntAuth state = *state_placeholder;

  SSH_TRACE(6, ("Keyboard interactive auth."));

  SSH_DEBUG(6, ("op = %d  user = %s", op, user));

  switch (op)
    {
    case SSH_AUTH_SERVER_OP_START:
      /* start FSM. Begin processing submethods. */
      /* This is the first operation for doing PAM authentication.
         We should not have any previous saved state when we come here. */
      SSH_ASSERT(*state_placeholder == NULL);

      state = ssh_xcalloc(1, sizeof(*state));

      if (ssh_server_auth_check(uc, user, config, server->common,
                                SSH_AUTH_KBD_INTERACTIVE))
        {
          /* go through the transaction as normal, but mark all
             results as failures. (this is what the draft says the
             server SHOULD do, even though this makes authentication
             problems a bitch to debug) */
          ssh_log_event(config->log_facility, SSH_LOG_WARNING,
                        "auth-kbd-int: User '%s' does not exist, faking "
                        "real transaction.", user);
          state->fake_transaction = TRUE;
        }

      if (state->fake_transaction == FALSE && uc &&
          ssh_user_uid(uc) == UID_ROOT &&
          config->permit_root_login == SSH_ROOTLOGIN_FALSE)
        {
          ssh_log_event(config->log_facility, SSH_LOG_WARNING,
                        "auth-kbd-int: root login denied for user '%s', "
                        "faking real transaction.", user);
          state->fake_transaction = TRUE;
        }
      
      state->uc = uc;
      state->user = ssh_xstrdup(user);
      state->packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      state->server = server;
      state->config = config;
      state->state_placeholder = state_placeholder;
      
      state->fsm = ssh_fsm_create(state);
      SSH_VERIFY(state->fsm != NULL);
      
      state->main_thread = ssh_fsm_thread_create(state->fsm,
                                                 ssh_server_kbd_int_start,
                                                 NULL_FNPTR, NULL_FNPTR,
                                                 NULL);
      SSH_VERIFY(state->main_thread != NULL);

      *state_placeholder = state;
      /* FSM will handle the rest. */
      return;
    case SSH_AUTH_SERVER_OP_CONTINUE:
      ssh_fatal("ssh_server_auth_kbd_interact: unexpected CONTINUE");
      return;
    case SSH_AUTH_SERVER_OP_CONTINUE_SPECIAL:
      SSH_ASSERT(state && state->main_thread);
      /* process client packet. */
      state->packet = packet;
      state->completion_proc = completion_proc;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      
      if (uc != state->uc)
        state->uc = uc;

      ssh_fsm_continue(state->main_thread);
      return;

    case SSH_AUTH_SERVER_OP_ABORT:
    case SSH_AUTH_SERVER_OP_UNDO_LONGTIME:
    case SSH_AUTH_SERVER_OP_CLEAR_LONGTIME:
      /* Clear state. */
      if (op == SSH_AUTH_SERVER_OP_ABORT)
        SSH_ASSERT(state && state->main_thread);

      if (state)
        {
          if (!state->server)
            state->server = NULL;

          ssh_cancel_timeouts(failed_timeout_cb, state->main_thread);
          ssh_fsm_set_next(state->main_thread, ssh_server_kbd_int_finish);
          ssh_fsm_continue(state->main_thread);
        }
      
      *state_placeholder = NULL;
      (*completion_proc)(SSH_AUTH_SERVER_REJECTED, packet,
                         completion_context);
      return;

    default:
      ssh_fatal("ssh_server_auth_kbd_interact: unknown op %d", (int)op);
    }

  SSH_NOTREACHED;
}

#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
