/*

auths-kbd-int-submethods.h

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Fri Feb  1 19:53:42 2002.

  Copyright (C) 2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#ifndef _AUTHS_KBD_INT_SUBMETHODS_H_
#define _AUTHS_KBD_INT_SUBMETHODS_H_

#include "sshconfig.h"
#include "sshserver.h"

/*
  Guidelines:

  The username, which is authenticated, might be invalid. Do not trust
  the user struct to be valid. (for some authentications, for example
  PAM w/LDAP, it is very possible for the user struct to be invalid at
  this stage, even with a valid user (the user's login data might not be
  in /etc/passwd)).

  The authentication should advance in the same way for denied and
  allowed logins. Of course, denied logins must always end with FAILED,
  but the prompts etc. offered to the client should be identical.

  Be careful! We are operating with root privileges here. Use
  sshuserfile if you need to access files.

  Check everything possible, trust no one.
*/

/*
   This function will be used by the main method to pass the responses from
   the client to the submethod.

   ``num_resp'' size of the ``resps'' array. (this is sanity checked by
                the main method, so that the ``num_reqs'' given to the
                conversation function will be identical to this. It is
                a protocol error for the client to send a different
                amount of responses than the server sent requests.)
   ``resps''    responses received from the client. This remains valid for
                one conversation round, e.g. it will be valid to the point
                where the submethod calls the ``conv'' function again. After
                that it is freed. The submethod MUST NOT modify the ``resps''
                array.
   ``cancel''   if TRUE, instructs the submethod to stop, kill any threads etc.
   ``context''  this will be set to the ``method_context'' given to the
                conversation function
*/
typedef void (*SshKbdIntSubMethodCB)(size_t num_resp,
                                     char **resps,
                                     Boolean cancel,
                                     void *context);

typedef enum {
  SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
  SSH_KBDINT_SUBMETHOD_RESULT_FAILED,
  SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS
} SshKbdIntResult;

/*
   This type of function is given to the (*init)() function. This should
   be called by the submethod when it has acquired the requests.
   
   ``code''        gives the result for the submethod, or NONE_YET if
                   authentication hasn't yet finished
   ``instruction'' instruction field for this request. MUST be
                   set to NULL if ``code'' is other than NONE_YET.
   ``num_reqs''    this specifies the size of the ``reqs'' and ``echo''
                   arrays. MUST only be >0 if ``code'' is NONE_YET.
   ``reqs''        array of prompts, or NULL if ``num_reqs'' is 0. These are
                   copied by the main method, so the caller may free the
                   array after this call.
   ``echo''        array of Boolean values, which specify whether the
                   specified answer by the user to a prompt (in the ``reqs''
                   array) should be echoed. These are copied by the main
                   method, so the caller may free the array after this call.
   ``method_cb''   used to pass the responses back to the submethod. MUST be
                   NULL_FNPTR if ``code'' is other than NONE_YET.
   ``method_ctx''  context to ``method_cb'' (this may be different than the
                   ``method_context'' initialized in (*init)()). MUST be NULL
                   if ``code'' is other than NONE_YET.
   ``context''     the context given when the submethod was initialized
*/
typedef void (*SshKbdIntSubMethodConv)(SshKbdIntResult code,
                                       const char *instruction,
                                       size_t num_reqs,
                                       char **reqs,
                                       Boolean *echo,
                                       SshKbdIntSubMethodCB method_cb,
                                       void *method_ctx,
                                       void *context);

typedef struct AuthKbdIntSubMethodRec
AuthKbdIntSubMethodStruct, *AuthKbdIntSubMethod;

typedef struct SshAuthKbdIntSubMethodsRec {
  SshServer server;
  SshUser uc;
  char *user;
  AuthKbdIntSubMethod submethods;
} SshAuthKbdIntSubMethodsStruct, *SshAuthKbdIntSubMethods;

struct AuthKbdIntSubMethodRec {
  char *name;        /* The short name, used in configuring this method. */
  char *long_name;   /* The name of the submethod [KBD-INT, section 3.2-3] */
  enum {SSH_KBDINT_SUBMETHOD_NOTUSED,
        SSH_KBDINT_SUBMETHOD_REQUIRED,
        SSH_KBDINT_SUBMETHOD_OPTIONAL } use;
  enum {SSH_KBDINT_SUBMETHOD_UNDEFINED,
        SSH_KBDINT_SUBMETHOD_FAILED,
        SSH_KBDINT_SUBMETHOD_SUCCESS } status;
  void (*method_init)(void **method_context,
               SshAuthKbdIntSubMethods methods,
               SshKbdIntSubMethodConv conv,
               void *conv_context);
  void (*method_free)(void *method_context);
  void *method_context;
};

/* The following functions are used in auths-kbd-interactive.c . */

/* Initialize the submethods using configuration settings.  */
SshAuthKbdIntSubMethods
ssh_server_kbd_int_submethods_init(SshServer server,
                                   const char *user_name,
                                   SshUser uc);

/* Unitialize (free) the submethods. (this calls the (*free)() function
   for every initialized submethod.)*/
void ssh_server_kbd_int_submethods_uninit(SshAuthKbdIntSubMethods methods);

/* Initialize the submethod ``method''. The ``conv'' and
   ``conv_context'' are given to the submethod's (*init)() function.  */
void ssh_server_kbd_int_submethod_init(SshAuthKbdIntSubMethods methods,
                                       AuthKbdIntSubMethod method,
                                       SshKbdIntSubMethodConv conv,
                                       void *conv_context);

/* Get the ``long_name'' field of the submethod (to send to the
   client). */
const char *ssh_server_kbd_int_submethod_get_name(AuthKbdIntSubMethod method);

#endif /* !_AUTHS_KBD_INT_SUBMETHODS_H_ */
