/*

auths-kbd-int-passwd.c

  Author: Sami J. Lehtinen <sjl@ssh.com>

  Created: Tue Mar 26 18:49:46 2002.

  Copyright (C) 2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

*/

#include "sshincludes.h"
#ifdef SSH_SERVER_WITH_KEYBOARD_INTERACTIVE
#include "auths-kbd-int-submethods.h"
#include "auths-kbd-int-passwd.h"
#include "sshfsm.h"
#include "sshdsprintf.h"

#define SSH_DEBUG_MODULE "Ssh2AuthKbdIntPasswd"

typedef struct PasswdStateRec {
  SshFSM fsm;
  SshFSMThread thread;

  SshAuthKbdIntSubMethods methods;
  
  SshKbdIntSubMethodConv conv;  
  void *conv_context;
  
  size_t num_reqs;
  char **reqs;
  Boolean *echo;
  
  size_t num_resps;
  char **resp;
} PasswdStateStruct, *PasswdState;

SSH_FSM_STEP(passwd_start);
SSH_FSM_STEP(passwd_process);
SSH_FSM_STEP(passwd_finish);

void submethod_passwd_method_cb(size_t num_resps,
                                char **resp,
                                Boolean cancel,
                                void *context)
{
  SshFSMThread thread = (SshFSMThread) context;
  PasswdState state = ssh_fsm_get_gdata(thread);

  if (cancel)
    {
      SSH_FSM_SET_NEXT(passwd_finish);
      SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
      return;
    }
  
  state->num_resps = num_resps;
  state->resp = resp;
  SSH_FSM_CONTINUE_AFTER_CALLBACK(thread);
}

void submethod_passwd_init(void **method_context,
                           SshAuthKbdIntSubMethods methods,
                           SshKbdIntSubMethodConv conv,
                           void *conv_context)
{
  PasswdState state = ssh_xcalloc(1, sizeof(*state));

  state->fsm = ssh_fsm_create(state);

  state->methods = methods;
  state->conv = conv;
  state->conv_context = conv_context;
  
  state->thread = ssh_fsm_thread_create(state->fsm, passwd_start, NULL_FNPTR,
                                        NULL_FNPTR, NULL);

  *method_context = state;
}

void submethod_passwd_free(void *method_context)
{
  PasswdState state = (PasswdState)method_context;
  int i;

  if (state->fsm)
    {
      ssh_fsm_kill_thread(state->thread);
      ssh_fsm_destroy(state->fsm);
    }

  for (i = 0; i < state->num_reqs; i++)
    ssh_xfree(state->reqs[i]);
  
  ssh_xfree(state->reqs);
  ssh_xfree(state->echo);
  memset(state, 'F', sizeof(*state));
  ssh_xfree(state);
}

SSH_FSM_STEP(passwd_start)
{
  PasswdState state = (PasswdState) fsm_context;

  SSH_FSM_SET_NEXT(passwd_process);

  state->reqs = ssh_xcalloc(1, sizeof(char *));
  state->echo = ssh_xcalloc(1, sizeof(Boolean));
  state->num_reqs = 1;

  ssh_xdsprintf(&state->reqs[0], "%s's password: ", state->methods->user);
  state->echo[0] = FALSE;
  SSH_POSTCOND(state->num_reqs);
  SSH_POSTCOND(state->reqs != NULL);
  SSH_POSTCOND(state->echo != NULL);
  
  SSH_FSM_ASYNC_CALL((*state->conv)(SSH_KBDINT_SUBMETHOD_RESULT_NONE_YET,
                                    "",
                                    state->num_reqs,
                                    state->reqs,
                                    state->echo,
                                    submethod_passwd_method_cb,
                                    thread,
                                    state->conv_context));
}

SSH_FSM_STEP(passwd_process)
{
  PasswdState state = (PasswdState) fsm_context;
  Boolean success = FALSE;
  SshUser uc = state->methods->uc;
  SshServer server = state->methods->server;
  SshConfig config = server->config;
  
  /* This MUST be enforced by the main method. */
  SSH_ASSERT(state->num_resps == state->num_reqs);

  if (uc == NULL)
    {
      SSH_DEBUG(3, ("User doesn't exist."));
      goto passwd_end;
    }

  if (ssh_user_uid(uc) == UID_ROOT &&
      config->permit_root_login == SSH_ROOTLOGIN_NOPWD)
    {
      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    "kbd-int: passwd: root login denied for user '%s'.",
                    ssh_user_name(uc));      
      goto passwd_end;
    }

  if (!config->permit_empty_passwords && strlen(state->resp[0]) == 0)
    {
      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    "kbd-int: passwd: login with empty password denied for "
                    "user '%s'.",
                    ssh_user_name(uc));
      goto passwd_end;
    }

  if (strlen(state->resp[0]) > 256)
    {
      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    "kbd-int: passwd: user '%s' tried to login with too "
                    "long password (256)", ssh_user_name(uc));
      goto passwd_end;
    }


  /* Try SECURE RPC passwords.  We do this first, as this might be
     needed to access disks. */
  if (ssh_user_validate_secure_rpc_password(uc, state->resp[0]))
    {
      ssh_log_event(config->log_facility,
                    SSH_LOG_NOTICE,
                    "kbd-int: passwd: user %s's secure rpc password accepted.",
                    ssh_user_name(uc));
      success = TRUE;
      goto passwd_end;
    }

  /* Try KERBEROS passwords.  This might also be needed to access
     disks. */
  if (ssh_user_validate_kerberos_password(uc, state->resp[0]))
    {
      ssh_log_event(config->log_facility,
                    SSH_LOG_NOTICE,
                    "kbd-int: passwd: kerberos password accepted for "
                    "user %s (%s).", ssh_user_name(uc),
                    ssh_user_kerberos_name(uc));
      success = TRUE;
      goto passwd_end;
    }


      /* Try a local password (either normal or shadow). */
  if (ssh_user_validate_local_password(uc, state->resp[0],
                                       server->common->remote_host))
    {















      ssh_log_event(config->log_facility,
                    SSH_LOG_NOTICE,
                    "kbd-int: passwd: user %s's local password accepted.",
                    ssh_user_name(uc));

      success = TRUE;
    }

  if (!success)
    {
      ssh_log_event(config->log_facility, SSH_LOG_WARNING,
                    "kbd-int: passwd: wrong password given for user '%s'.",
                    ssh_user_name(uc));
    if (uc)
      ssh_user_record_login_failure(uc, server->common->remote_host);
    }

  else
    {
      char *prompt = NULL;
      
      /* Check if the user's password needs to be changed. */
      if (ssh_user_password_must_be_changed(uc, &prompt))
        {
          SshCommon common = server->common;
          
          ssh_xfree(prompt);
          ssh_log_event(config->log_facility, SSH_LOG_INFORMATIONAL,
                        "kbd-int: passwd: user '%s' forced to change "
                        "password.", ssh_user_name(uc));
          /* XXX This is a problem. See auths-passwd.c . */
          common->forced_command = ssh_xstrdup(config->passwd_path);
        }
    }

 passwd_end:
  (*state->conv)(success ? SSH_KBDINT_SUBMETHOD_RESULT_SUCCESS :
                 SSH_KBDINT_SUBMETHOD_RESULT_FAILED,
                 NULL, 0, NULL, NULL, NULL_FNPTR, NULL,
                 state->conv_context);
  
  SSH_FSM_SET_NEXT(passwd_finish);
  return SSH_FSM_CONTINUE;
}

SSH_FSM_STEP(passwd_finish)
{
  PasswdState state = (PasswdState) fsm_context;
  SSH_DEBUG(2, ("Finishing..."));
  
  ssh_fsm_destroy(state->fsm);
  state->fsm = NULL;
  state->thread = NULL;
  return SSH_FSM_FINISH;
}

#endif /* SSH_SERVER_WITH_KEYBOARD_INTERACTIVE */
