/*

  auths-common.c

  Author: Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2002 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.
                  
  Functions for checking for user, host in lists. Conveniency functions
  for checking whether user is allowed to log in.

*/

#include "sshincludes.h"
#include "sshuser.h"
#include "auths-common.h"
#include "sshcommon.h"
#include "sshmatch.h"
#include "sshdsprintf.h"
#include "sshappcommon.h"
#include "sshinet.h"

#define SSH_DEBUG_MODULE "Ssh2AuthCommonServer"

/* Returns FALSE if user and host are allowed to connect and
   authenticate. */
Boolean ssh_server_auth_check(SshUser uc, const char *user, SshConfig config,
                              SshCommon common, char *auth_method_name)
{
  if (ssh_server_auth_check_user(uc, user, common))
    {
      /* failure */
      char *s = "%s authentication failed. Login to account %.100s not "
        "allowed or account non-existent.";
      
      ssh_log_event(config->log_facility,
                    SSH_LOG_WARNING,
                    s,
                    auth_method_name, user);
      SSH_TRACE(1, (s, auth_method_name, user));
      /* Reject the login if the user is not allowed to log in. */
      return TRUE;
    }
  
  /* Check whether logins from remote host are allowed */
  if (ssh_server_auth_check_host(common))
    {
      /* failure */
      char *s;
      s = "%s authentication failed. Connection from %.100s denied. "
        "Authentication as user %.100s was attempted.";
      
      /* logins from remote host are not allowed. */
      ssh_log_event(config->log_facility, SSH_LOG_WARNING,
                    s, auth_method_name,
                    common->remote_host, ssh_user_name(uc));
      SSH_TRACE(1, (s, auth_method_name, common->remote_host,   \
                    ssh_user_name(uc)));
      
      /* Reject the login if the user is not allowed to log in from
         specified host. */
      return TRUE;
    }
  return FALSE;
}

/* Match a given 'string' with the given 'pattern' using
   SshRegex. Return 'TRUE' if a match is found, 'FALSE' otherwise. */
Boolean ssh_match_string(const char *string, const char *pattern,
                         SshRegexSyntax rex_syntax, SshRegexContext rex_ctx)
{
  SshRegexMatcher regex_matcher;
  Boolean match = FALSE;
  
  SSH_PRECOND(string != NULL);
  SSH_PRECOND(pattern != NULL);
  
  regex_matcher = ssh_regex_create(rex_ctx, pattern, rex_syntax);
  if (!regex_matcher)
    {
      /* Failure to create matcher. */
      ssh_warning("Pattern \"%s\" is invalid.", pattern);

      return FALSE;
    }
  
  match = ssh_regex_match_cstr(regex_matcher, string);  
  SSH_TRACE(6, ("match: %s (pattern: '%s', string: '%s')",
                match ? (char *)ssh_regex_get_submatch(regex_matcher, 0)
                : "NULL",
                pattern, string));
  
  if (match)
    match = (strcmp(string,
                    (char *)ssh_regex_get_submatch (regex_matcher, 0))
             ? FALSE : TRUE);
  
  ssh_regex_free(regex_matcher);

  return match;  
}

/* Function to check whether given 'string' matches with a
   pattern in 'list'. Uses egrep-like-syntax. Returns TRUE if a match
   is found, and FALSE otherwise. list must not contain NULL holders. */
Boolean ssh_match_string_in_list(const char *string, SshADTContainer list,
                                 SshRegexContext rex_ctx)
{
  SshPatternHolder holder;
  SshADTHandle handle;
  
  for (handle = ssh_adt_enumerate_start(list);
       handle != SSH_ADT_INVALID;
       handle = ssh_adt_enumerate_next(list, handle))
    {
      holder = (SshPatternHolder) ssh_adt_get(list, handle);
      SSH_ASSERT(holder);
      if (ssh_match_string(string, holder->pattern, holder->regex_syntax,
                           rex_ctx))
        return TRUE;
    }
  
  return FALSE;
}

/* Helper function to check whether given 'string' or 'number' matches
   with a pattern in 'list'. Uses egrep-like-syntax. Returns FALSE if
   a match is found, and FALSE otherwise. list must not contain NULL
   holders. */
Boolean ssh_match_string_or_number_in_list(const char *string,
                                           unsigned long number,
                                           SshADTContainer list,
                                           SshRegexContext rex_ctx)
{
  char *num_str;
  Boolean ret;
  
  if (ssh_match_string_in_list(string, list, rex_ctx))
    return FALSE;
  
  ssh_dsprintf(&num_str, "%lu", number);

  ret = ssh_match_string_in_list(num_str, list, rex_ctx);
  
  ssh_xfree(num_str);
  
  return ret ? FALSE : TRUE;
}

/* Helper function to check whether given host name or ip-address
   matches a specified pattern. Returns TRUE if a match is found, and
   FALSE otherwise. */
static Boolean match_host_id_int(const char *host_name, const char *host_ip,
                                 const char *pattern,
                                 SshRegexSyntax rex_syntax,
                                 SshRegexContext rex_ctx)
{
  Boolean only_ip = FALSE;
  SSH_PRECOND(host_name != NULL);
  SSH_PRECOND(pattern != NULL);
#define HOST_IP_SPECIFIER "\\i"
#define IP_MASK_SPECIFIER "\\m"
  
  if (strlen(pattern) > strlen(HOST_IP_SPECIFIER) &&
      !strncmp(pattern, HOST_IP_SPECIFIER, strlen(HOST_IP_SPECIFIER)))
    {
      only_ip = TRUE;
      pattern += strlen(HOST_IP_SPECIFIER);
    }

  if (strlen(pattern) > strlen(IP_MASK_SPECIFIER) &&
      !strncmp(pattern, IP_MASK_SPECIFIER, strlen(IP_MASK_SPECIFIER)))
    {
      Boolean ret;
      pattern += strlen(IP_MASK_SPECIFIER);
      if (host_ip == NULL)
        {
          ssh_warning("ssh_match_host_id: address mask `%s' specified, "
                      "but no IP-address known for host (?).", pattern);
          return FALSE;
        }
      ret = ssh_inet_compare_netmask(pattern, host_ip);
      if (!ret)
        SSH_TRACE(0, ("Address mask `%s' didn't match ip `%s'",
                      pattern, host_ip));
      else
        SSH_DEBUG(2, ("Address mask `%s' matched ip `%s'",
                      pattern, host_ip));

      return ret;
    }
  
  if (host_ip && ssh_match_string(host_ip, pattern, rex_syntax, rex_ctx))
    return TRUE;
  else if (only_ip == FALSE)
    return ssh_match_string(host_name, pattern, rex_syntax, rex_ctx);
  else
    return FALSE;
}

/* Helper function to check whether given host name or ip-address
   matches a specified pattern in 'holder'. Uses regex-syntax specified
   in 'holder'. Returns TRUE if a match is found, and FALSE
   otherwise. */
Boolean ssh_match_host_id(const char *host_name, const char *host_ip,
                          SshPatternHolder holder, SshRegexContext rex_ctx)
{
  return match_host_id_int(host_name, host_ip, holder->pattern,
                           holder->regex_syntax, rex_ctx);
}

Boolean ssh_match_user_groups(const char *userp, const char *uid_str,
                              size_t num_groups,
                              char * const *groups,
                              char * const *gid_strs,
                              const char *remote_host,
                              const char *remote_ip,
                              SshPatternHolder holder,
                              SshRegexContext rex_ctx)
{
  char *user_pat, *host_pat, *group_pat, *user;
  Boolean match = FALSE;
  SshRegexSyntax rex_syntax;

  user = (char *)userp;















  
  SSH_ASSERT(holder);
  user_pat = ssh_xstrdup(holder->pattern);
  rex_syntax = holder->regex_syntax;
      
  /* separate user, group and host */
  if ((host_pat = strchr(user_pat, '@')) != NULL)
    {
      SSH_TRACE(6, ("'%s' is a user@host pattern.", user_pat));
      *host_pat = '\0';
      host_pat++;
    }

  if ((group_pat = strchr(user_pat, '%')) != NULL)
    {
      SSH_TRACE(6, ("'%s' is a user%group pattern.", user_pat));
      *group_pat = '\0';
      group_pat++;
    }      
  
  /* match user. */
  if (ssh_match_string(user, user_pat, rex_syntax, rex_ctx))
    {
      SSH_TRACE(5, ("'%s' matches '%s'.", user, user_pat));
    }
  else if (ssh_match_string(uid_str, user_pat, rex_syntax, rex_ctx))
    {
      SSH_TRACE(5, ("'%s' matches '%s'.",
                    uid_str, user_pat));
    }  
  else
    {
      SSH_TRACE(5, ("User '%s' (uid '%s') didn't match '%s'.",
                    user, uid_str, user_pat));
      match = FALSE;
      goto free_and_return;
    }

  /* match group (if available). */
  if (group_pat)
    {
      if (num_groups > 0)
        {
          int i;

          match = FALSE;
          SSH_ASSERT(groups != NULL);
          SSH_ASSERT(gid_strs != NULL);
          for (i = 0; i < num_groups; i++)
            {
              SSH_ASSERT(groups[i] != NULL);
              SSH_ASSERT(gid_strs[i] != NULL);
              if (ssh_match_string(groups[i], group_pat, rex_syntax,
                                   rex_ctx) ||
                  ssh_match_string(gid_strs[i], group_pat, rex_syntax,
                                   rex_ctx))
                {
                  SSH_TRACE(5, ("%s[%s] matches '%s'.",
                                groups[i], gid_strs[i], group_pat));
                  match = TRUE;
                  break;
                }
            }
          if (!match)
            {
              SSH_TRACE(5, ("No match with groups and '%s'.", group_pat));
              goto free_and_return;
            }
        }
      else
        {
          ssh_warning("ssh_match_user_groups: Pattern '%s' contained a "
                      "group pattern '%s', but groups was empty (i.e. the "
                      "group pattern will be ignored).", holder->pattern,
                      group_pat);
        }
    }
    
  match = TRUE;

  /* match host (if available). */
  if (host_pat)
    {
      if (!match_host_id_int(remote_host, remote_ip, host_pat, rex_syntax,
                             rex_ctx))
        {
          SSH_TRACE(5, ("'%s[%s]' didn't match with '%s'.",
                        remote_host, remote_ip, host_pat));
          match = FALSE;
        }
      else
        {
          SSH_TRACE(5, ("'%s[%s]' matches '%s'.",
                        remote_host, remote_ip, host_pat));
        }
    }

 free_and_return:  
  ssh_xfree(user_pat);



  return match;
}

Boolean ssh_match_user(const char *user, const char *uid_str,
                       const char *remote_host,
                       const char *remote_ip,
                       SshPatternHolder holder,
                       SshRegexContext rex_ctx)
{
  return ssh_match_user_groups(user, uid_str, 0, NULL, NULL, remote_host,
                               remote_ip, holder, rex_ctx);
}

Boolean ssh_match_user_in_list(const char *user, uid_t uid, 
                               const char *remote_host,
                               const char *remote_ip,
                               SshADTContainer list,
                               SshRegexContext rex_ctx)
{
  char *uid_str;
  SshADTHandle handle;
  Boolean match = FALSE;
  SshPatternHolder holder;
  ssh_dsprintf(&uid_str, "%lu", uid);

  for (handle = ssh_adt_enumerate_start(list);
       handle != SSH_ADT_INVALID && !match;
       handle = ssh_adt_enumerate_next(list, handle))
    {
      holder = (SshPatternHolder) ssh_adt_get(list, handle);
      SSH_ASSERT(holder);
      match = ssh_match_user(user, uid_str, remote_host, remote_ip,
                             holder, rex_ctx);
    }
  
  ssh_xfree(uid_str);
  return match;              
}

Boolean ssh_server_auth_check_user_generic(SshUser uc, const char *user,
                                           const char *allowed_msg,
                                           const char *denied_msg,
                                           const char *hostname,
                                           const char *host_ip,
                                           SshADTContainer allowed_users,
                                           SshADTContainer denied_users,
                                           SshADTContainer allowed_groups,
                                           SshADTContainer denied_groups)
{
  uid_t uid;
  SshRegexContext rex_ctx = NULL;

  SshGroup *groups;

  int i = 0;
  
  /* Do not allow login if the user was not found on the system. */
  if (uc == NULL)
    {
      SSH_TRACE(1, ("User %s does not exist.",
                    user));
      return TRUE;
    }
  
   /* Reject the login if the user is not allowed to log in.  These
      additional checks don't matter in tcp-forwarding etc. cause they
      would've already failed during attempted login. */
  if (!ssh_user_login_is_allowed(uc))
    {
      SSH_TRACE(1, ("User %s's login is not allowed due to system policy",
                    ssh_user_name(uc)));
      return TRUE;
    }

  /* List matching. */
  uid = ssh_user_uid(uc);
  rex_ctx = ssh_app_get_global_regex_context();
  
  if (denied_users)
    if (ssh_match_user_in_list(user, uid, hostname, host_ip,
                               denied_users,
                               rex_ctx))
      {
        SSH_TRACE(1, ("User %s %s because username matched "
                      "with deny list.", ssh_user_name(uc), denied_msg));
        return TRUE;
      }
  
  if (allowed_users)
    {
      if (ssh_match_user_in_list(user, uid, hostname, host_ip,
                                 allowed_users, rex_ctx))
        {
          SSH_TRACE(2, ("User %s %s because username matched "
                        "with allow list.", ssh_user_name(uc), allowed_msg));
          /* Fall through to group check. */
        }
      else
        {
          SSH_TRACE(1, ("User %s %s because allow list "
                        "exists, and username was not matched.",
                        ssh_user_name(uc), denied_msg));
          return TRUE;
        }
    }

  

  groups = ssh_user_get_groups(uc);

  if (denied_groups)
    for (i = 0; groups[i]; i++)
      if (!ssh_match_string_or_number_in_list(ssh_group_get_name(groups[i]),
                                              ssh_group_get_gid(groups[i]),
                                              denied_groups, rex_ctx))
        {
          SSH_TRACE(1, ("User %s %s because group deny list "
                        "exists, and user's group matched.",
                        ssh_user_name(uc), denied_msg));
          return TRUE;
        }
  
  if (allowed_groups)
    {
      for (i = 0; groups[i]; i++)
        if (!ssh_match_string_or_number_in_list(ssh_group_get_name(groups[i]),
                                                ssh_group_get_gid(groups[i]),
                                                allowed_groups, rex_ctx))
          {
            SSH_TRACE(2, ("User %s %s because user's group matched "
                          "with group allow list.",
                          ssh_user_name(uc), allowed_msg));
            return FALSE;
          }

      SSH_TRACE(1, ("User %s %s because group allow list "
                    "exists, and user's group didn't match.",
                    ssh_user_name(uc), denied_msg));
      return TRUE;
    }


  return FALSE;
}

/* returns FALSE on success. */
Boolean ssh_server_auth_check_user(SshUser uc, const char *user,
                                   SshCommon common)
{
  SSH_PRECOND(common != NULL);
  SSH_PRECOND(common->config != NULL);

  return ssh_server_auth_check_user_generic(uc, user,
                                            "is allowed to login",
                                            "is denied login",
                                            common->remote_host,
                                            common->remote_ip,
                                            common->config->allowed_users,
                                            common->config->denied_users,
                                            common->config->allowed_groups,
                                            common->config->denied_groups);
}


/* Checks whether given host name or ip-address is found in
   list. Returns TRUE if a match is found, and FALSE otherwise. */
Boolean ssh_match_host_in_list(const char *host_name, const char *host_ip,
                               SshADTContainer list)
{
  SshADTHandle handle;
  SshRegexContext rex_ctx = ssh_app_get_global_regex_context();

  for (handle = ssh_adt_enumerate_start(list);
       handle != SSH_ADT_INVALID;
       handle = ssh_adt_enumerate_next(list, handle))
    {
      SshPatternHolder pattern =
        (SshPatternHolder) ssh_adt_get(list, handle);
      
      if (match_host_id_int(host_name, host_ip, pattern->pattern,
                            pattern->regex_syntax, rex_ctx))
        {
          SSH_TRACE(4, ("'%s[%s] matched with '%s'.", host_name, host_ip,
                        pattern->pattern));
          return TRUE;
        }
      else
        SSH_TRACE(5, ("'%s[%s] didn't match with '%s'.", host_name, host_ip,
                      pattern->pattern));
    }
  
  return FALSE;
}

/* This is the function for checking a host{name,ip} against
   {Allow,Deny}Hosts parameters. Also checks remote host against
   statements in the AllowDenyHostsFile. Returns FALSE if connection
   from host is allowed, TRUE otherwise. */
Boolean ssh_server_auth_check_host(SshCommon common)
{
  return ssh_server_auth_check_host_generic
    (common->remote_host, common->remote_ip, common->config->denied_hosts,
     common->config->allowed_hosts, common->config->require_reverse_mapping);
}

Boolean ssh_server_auth_check_host_generic(const char *hostname,
                                           const char *host_ip,
                                           SshADTContainer denied,
                                           SshADTContainer allowed,
                                           Boolean require_reverse_mapping)
{
  /* XXX subnet masks "130.240.0.0/16" */
  /* XXX address ranges "130.240.20.15-130.240.21.76" */

  /* RequireReverseMapping */
  if (require_reverse_mapping)
    {
      if (strcmp(hostname, host_ip) == 0)
        {
          /* If remote host's ip-address couldn't be mapped to a
             hostname and RequireReverseMapping = 'yes', deny
             connection. */
          return TRUE;
        }
    }
  
  /* Check whether host is denied. Use ssh1-style policy, ie. if host
     is in DenyHosts, connection is denied even if the same host
     matches in AllowHosts. */
  if (denied)
    {
      if (ssh_match_host_in_list(hostname, host_ip, denied))
        return TRUE;
    }

  if (allowed)
    {
      if (ssh_match_host_in_list(hostname, host_ip, allowed))
        return FALSE;
      else
        return TRUE;
    }  
  
  return FALSE;
}
