/*
  
  authc-pubkey.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2001 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Public key authentication, client side.
  
*/

#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"

#include "readpass.h"

#include "sshuserfiles.h"
#include "sshmsgs.h"
#include "sshclient.h"
#include "sshdebug.h"
#include "sshagent.h"
#ifdef WITH_PGP
#include "sshpgp.h"
#include "ssh2pgp.h"
#endif /* WITH_PGP */






#include "sshkeyfile.h"
#include "ssh2compat.h"
#include "sshtimeouts.h"




#define SSH_DEBUG_MODULE "Ssh2AuthPubKeyClient"

/* Information about a candidate key to be used for authentication.
   Candidate keys may either be in private key files, or may be used
   through the authentication agent. */
typedef struct SshClientPubkeyAuthCandidateRec
{
  /* Type of this candidate.  The key may be available either from a
     key file, or from the authentication agent. */
  enum {
#ifdef WITH_PGP
    CANDIDATE_TYPE_PGPKEY,
#endif /* WITH_PGP */



    CANDIDATE_TYPE_NORMAL
  } type;

  enum { CANDIDATE_ORIGIN_AGENT,



         CANDIDATE_ORIGIN_FILE
  } origin;





  /* The public key blob for this candidate (may contain certificates).
     This field is allocated using ssh_xmalloc. */
  unsigned char *pubkeyblob;
  size_t pubkeyblob_len;

  /* the public key algorithm for this key (the type of the key) */
  unsigned char *pubkey_alg;
  
  /* Name of the file containing the private key, if of type
     CANDIDATE_KEYFILE or CANDIDATE_CERTIFICATE.  This string is
     allocated using ssh_xmalloc, or is NULL. */
  char *privkeyfile;

  /* This is used only with the external keys */
  SshPrivateKey privkey;

#ifdef WITH_PGP
  /* If type is CANDIDATE_PGPKEY, the following variables contain 
     secret keyblob in pgp packet format. */
  unsigned char *pgp_seckey;
  size_t pgp_seckey_len;
  char *pgp_keyname;
#endif /* WITH_PGP */
} *SshClientPubkeyAuthCandidate;
  
/* A context for public key authentication, this is stored in 
   *state_placeholder */
typedef struct SshClientPubkeyAuthContextRec
{
  /* Connection to the authentication agent.  This is NULL if we don't have
     an open connection to the agent. */
  SshAgent agent;
  
  /* If the state is SSH_AUTH_CLIENT_PUBKEY_TRYING_KEYS, this
     is the index of the last candidate that we have tried. */
  unsigned int last_tried_candidate;

  /* Number of candidate keys to try for authentication. */
  unsigned int num_candidates;

  /* Order of different public key authentication methods. */
  SshClientPubkeyType types[6];

  /* Info about the agent keys. */
  SshAgentKeyInfo agent_keys;

  /* Number of agent keys. */
  unsigned int num_agent_keys;

  /* Current index of the array above. */
  unsigned int current_type;

  /* An array of candidate keys to try for authentication.  Keys are stored
     in order of preference; in particular, keys available from the agent
     are stored in the beginning of the array. */
  SshClientPubkeyAuthCandidate candidates;

  /* Context information for the ongoing authentication request. */
  SshAuthClientCompletionProc completion;
  void *completion_context;
  void **state_placeholder;
  char *user; /* must free with ssh_xmalloc if non-NULL. */

  /* Context information for an asyncronous signature operation */
  unsigned char *signature_packet;
  SshPrivateKey signature_privkey;
  SshClientPubkeyAuthCandidate signature_candidate;




  
  /* The user may cancel the authentication. (by a "Cancel"-button in
     windows, and by "Ctrl-D" in unix. */
  Boolean auth_cancelled;

  /* This integer tracks the number of keys that the remote host has not
     accepted. If all keys were declined, a message will be
     displayed. */
  int declined_candidates;

  /* Whether or not we have sent a signature. */
  Boolean signature_sent;
  
  SshClient client;
} *SshClientPubkeyAuth;

/* Frees the authentication context and any memory referenced from it.  Also
   closes the agent connection if open. */

void ssh_client_auth_pubkey_free_ctx(SshClientPubkeyAuth state)
{
  int i;

  if (state == NULL)
    return;


  /* Close the agent connection if open. */
  if (state->agent)
    ssh_agent_close(state->agent);


  /* Free the candidates array. */
  for (i = 0; i < state->num_candidates; i++)
    {
      ssh_xfree(state->candidates[i].pubkeyblob);
      ssh_xfree(state->candidates[i].privkeyfile);      
      ssh_xfree(state->candidates[i].pubkey_alg);
#ifdef WITH_PGP
      ssh_xfree(state->candidates[i].pgp_seckey);
      ssh_xfree(state->candidates[i].pgp_keyname);
#endif /* WITH_PGP */
    }
  ssh_xfree(state->candidates);
  ssh_xfree(state->user);
            
  /* Free the state structure itself. */
  ssh_xfree (state);
}

/* Constructs a SSH_MSG_USERAUTH_REQUEST that basically asks if a 
   given private key can be used for login. Returns NULL on failure. */

SshBuffer ssh_client_auth_pubkey_try_key_packet(SshClientPubkeyAuthCandidate c,
                                                Boolean draft_incompatibility)
{
  SshBuffer b;

  /* Format the request packet. */
  b = ssh_xbuffer_allocate();

  if (!draft_incompatibility)
    {
      ssh_encode_buffer(b,
                        SSH_FORMAT_BOOLEAN, FALSE,
                        SSH_FORMAT_UINT32_STR, c->pubkey_alg,
                        strlen((char *)c->pubkey_alg),
                        SSH_FORMAT_UINT32_STR, c->pubkeyblob,
                        c->pubkeyblob_len,
                        SSH_FORMAT_END);
    }
  else
    {      
      /* Remote end has publickey draft incompatibility bug. */
      ssh_encode_buffer(b,
                        SSH_FORMAT_BOOLEAN, FALSE,
                        /* Against the draft. Here should be string
                           'publickey algorithm'*/
                        SSH_FORMAT_UINT32_STR, c->pubkeyblob,
                        c->pubkeyblob_len,
                        SSH_FORMAT_END);
    }
  
  return b;
}                                             

void ssh_client_auth_pubkey_try_this_candidate(SshClientPubkeyAuth state);
void ssh_client_auth_pubkey_add_candidates(void *context);

/* Completion procedure for signature opration.  `result' is the
   signature.  This will construct the packet to be sent to the
   server, and call the authentication method completion procedure
   (stored in state->completion).  The context is a pointer to the
   state object. */

void ssh_client_auth_pubkey_sign_complete(SshAgentError error,
                                          const unsigned char *result,
                                          size_t len,
                                          void *context)
{
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;
  SshClientPubkeyAuthCandidate c;
  SshBuffer b;
  unsigned char *signature_blob;
  size_t signature_blob_len;
  


















  if (error == SSH_AGENT_ERROR_FAILURE)
    {
      /* Skip to the next candidate if signing failed. */
      state->last_tried_candidate++;
      ssh_client_auth_pubkey_try_this_candidate(state);
      return;
    }
 
  /* Get pointer to the candidate being tried. */
  c = &state->candidates[state->last_tried_candidate];
  
  SSH_DEBUG_HEXDUMP(7, ("auth_pubkey_sign_complete: signature:"), result, len);

  /* Construct the body of the message to send to the server. */
  b = ssh_xbuffer_allocate();

  if (




      !*(state->client->common->compat_flags->
         malformed_signatures_draft_incompatibility)

      )
    {
      signature_blob_len =
        ssh_encode_array_alloc(&signature_blob,
                               SSH_FORMAT_UINT32_STR,
                               c->pubkey_alg, strlen(c->pubkey_alg),
                               SSH_FORMAT_UINT32_STR, result, len,
                               SSH_FORMAT_END);
    }
  else
    {
      signature_blob = (unsigned char *)result;
      signature_blob_len = len;
      result = NULL;
    }
  
  if (



      !(*state->client->common->compat_flags->publickey_draft_incompatibility)

      )
    {
      ssh_encode_buffer(b,
                        SSH_FORMAT_BOOLEAN, TRUE,
                        SSH_FORMAT_UINT32_STR, c->pubkey_alg,
                        strlen((char *)c->pubkey_alg),
                        SSH_FORMAT_UINT32_STR, c->pubkeyblob,
                        c->pubkeyblob_len,
                        SSH_FORMAT_UINT32_STR,
                        signature_blob, signature_blob_len,
                        SSH_FORMAT_END);
    }
  else
    {      
      /* Remote end has publickey draft incompatibility bug. */
      ssh_encode_buffer(b,
                        SSH_FORMAT_BOOLEAN, TRUE,
                        /* Against the draft. Here should be string
                           'publickey algorithm'*/
                        SSH_FORMAT_UINT32_STR,
                        c->pubkeyblob, c->pubkeyblob_len,
                        SSH_FORMAT_UINT32_STR,
                        signature_blob, signature_blob_len,
                        SSH_FORMAT_END);
    }

  if (




      !*(state->client->common->compat_flags->
         malformed_signatures_draft_incompatibility)

      )
    {
      memset(signature_blob, 'F', signature_blob_len);
      ssh_xfree(signature_blob);
    }

  /* Call the authentication method completion procedure. */
  (*state->completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE_MULTIPLE,
                       state->user, b, state->completion_context);

  state->signature_sent = TRUE;
  
  /* Free the buffer */
  ssh_buffer_free(b);
}

SshPrivateKey ssh_authc_pubkey_privkey_read(SshUser user,
                                            const char *fname,
                                            const char *passphrase,
                                            char **comment,
                                            SshClientPubkeyAuth state)
{




























  SshPrivateKey privkey = NULL;
  SshPrivKeyResult result;

  char buf[256];

  char *pass;

  if (passphrase)
    {
      if (ssh_privkey_read(user, fname, passphrase, comment,
                           &privkey) == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          ssh_warning("Could not read private key %s", fname);
          return NULL;
        }
      
      if (privkey != NULL)
        return privkey;
    }

  if (ssh_privkey_read(user, fname, "", comment,
                       &privkey) == SSH_PRIVKEY_KEY_UNREADABLE)
    {
      ssh_warning("Could not read private key %s", fname);
      return NULL;
    }
  
  if (privkey != NULL)
    return privkey;


  ssh_snprintf(buf, sizeof (buf),
               "Passphrase for key \"%s\"%s%s%s",
               fname,
               (*comment ? " with comment \"" : ":"),
               (*comment ? *comment : ""),
               (*comment ? "\":" : ""));

  if (!state->client->config->batch_mode)
    pass = ssh_read_passphrase(buf , FALSE);
  else
    pass = NULL;








  if (pass == NULL)
    state->auth_cancelled = TRUE;











  
  if (pass)
    {
      /* The passphrase may be empty at this stage, indicating that the
         user wants to skip this public key. */
      result = ssh_privkey_read(user, fname, pass, NULL, &privkey);
      if (result == SSH_PRIVKEY_KEY_UNREADABLE)
        {
          SSH_DEBUG(4, ("Private key `%s' is unreadable.", fname));
          privkey = NULL;
        }








      
      memset(pass, 'F', strlen(pass));
      ssh_xfree(pass);
      return privkey;
    }

  return NULL;

}

/* Callback for asynchronous signing in the
   ssh_client_auth_pubkey_send_signature function. */
void ssh_client_auth_pubkey_sign_cb(SshCryptoStatus status,
                                    const unsigned char *signature_buffer,
                                    size_t signature_buffer_len,
                                    void *context)
{
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;
  unsigned char *packet = state->signature_packet;
  SshPrivateKey privkey = state->signature_privkey;










  
  /* Check whether the operation was successful. */
  if (status != SSH_CRYPTO_OK)
    {
      ssh_debug("Private key operation failed: %s (%s)",
                state->signature_candidate->privkeyfile,
                ssh_crypto_status_message(status));
















































      ssh_xfree(packet);
      if (privkey)
        ssh_private_key_free(privkey);
      /* Tell the completion procedure that we failed. */
      ssh_client_auth_pubkey_sign_complete(SSH_AGENT_ERROR_FAILURE,
                                           NULL, 0, (void *)state);
      /* Note that state is no longer necessarily valid. */
      return;
    }










  
  /* Pass the result to the completion procedure. */
  ssh_client_auth_pubkey_sign_complete(SSH_AGENT_ERROR_OK,
                                       signature_buffer,
                                       signature_buffer_len,
                                       (void *)state);
  /* Free allocated data.  Note that state is no longer necessarily
     valid. */
  ssh_xfree(packet);
  if (privkey)
    ssh_private_key_free(privkey);
  return;
}

/* Constructs the data to be signed in a public key authentication request.
   Eventually calls state->completion when done. Returns FALSE if reading
   private key was successful and there are candidates left, TRUE if
   not. (note that even if other operations fail, this returns FALSE.)*/

Boolean ssh_client_auth_pubkey_send_signature(SshClientPubkeyAuth state,
                                              const char *user,
                                              unsigned char *session_id,
                                              size_t session_id_len)
{
  SshClientPubkeyAuthCandidate c;
  SshPrivateKey privkey = NULL;
  unsigned char *packet;
  char * key_comment = NULL;
  char *service;
  SshEncodingFormat format_session_id;
  
  size_t packet_len;

  ssh_debug("Constructing and sending signature in publickey "
            "authentication.");

  c = &state->candidates[state->last_tried_candidate];     

  /* Construct a throw-away SSH_MSG_USERAUTH_REQUEST message for signing. */

  if (




      !(*state->client->common->compat_flags->
        publickey_service_name_draft_incompatibility)

      )
    service = SSH_CONNECTION_SERVICE;
  else
    service = SSH_USERAUTH_SERVICE;

  if (




      !(*state->client->common->compat_flags->
        publickey_session_id_encoding_draft_incompatibility)

      )
    format_session_id = SSH_FORMAT_UINT32_STR;
  else
    format_session_id = SSH_FORMAT_DATA;
  
  if (



      !(*state->client->common->compat_flags->publickey_draft_incompatibility)

      )
    {
      packet_len =
        ssh_encode_array_alloc(&packet,
                               format_session_id, session_id, session_id_len,
                               SSH_FORMAT_CHAR,
                               (unsigned int) SSH_MSG_USERAUTH_REQUEST,
                               SSH_FORMAT_UINT32_STR, user, strlen(user),
                               SSH_FORMAT_UINT32_STR, service,
                               strlen(service),
                               SSH_FORMAT_UINT32_STR, SSH_AUTH_PUBKEY,
                               strlen(SSH_AUTH_PUBKEY),
                               SSH_FORMAT_BOOLEAN, TRUE,
                               SSH_FORMAT_UINT32_STR, c->pubkey_alg,
                               strlen(c->pubkey_alg),
                               SSH_FORMAT_UINT32_STR, c->pubkeyblob,
                               c->pubkeyblob_len,
                               SSH_FORMAT_END);
    }
  else
    {      
      /* Remote end has publickey draft incompatibility bug. */
      packet_len =
        ssh_encode_array_alloc(&packet,
                               format_session_id, session_id, session_id_len,
                               SSH_FORMAT_CHAR,
                               (unsigned int) SSH_MSG_USERAUTH_REQUEST,
                               SSH_FORMAT_UINT32_STR, user, strlen(user),
                               SSH_FORMAT_UINT32_STR, service,
                               strlen(service),
                               /* against the draft. Here should
                                  be 'string "publickey"'*/
                               SSH_FORMAT_BOOLEAN, TRUE,
                               /* against the draft. Here should
                                  be 'string public key algorith
                                  name'*/
                               SSH_FORMAT_UINT32_STR, c->pubkeyblob,
                               c->pubkeyblob_len,
                               SSH_FORMAT_END);  
    }

  /* Now sign the buffer.  How to do this depends on the type of the key. */
  switch (c->origin)
    {
    case CANDIDATE_ORIGIN_AGENT:

      /* Sanity check: the agent should be open. */
      SSH_ASSERT(state->agent != NULL);

      {
        SshAgentOp op = SSH_AGENT_HASH_AND_SIGN;
        
        /* RSA keys */
        if (!strncmp(c->pubkey_alg, "ssh-rsa", 7))
          {
            if (




                !*state->client->common->compat_flags->
                rsa_hash_scheme_draft_incompatibility

                )
              op = SSH_AGENT_HASH_AND_SIGN_PKCS1_SHA1;
            else
              op = SSH_AGENT_HASH_AND_SIGN_PKCS1_MD5;
          }
        
        /* Send the data to the agent for signing. */








        ssh_agent_op(state->agent, op,
                     c->pubkeyblob, c->pubkeyblob_len,
                     packet, packet_len,
                     ssh_client_auth_pubkey_sign_complete, (void *)state);
      }
      
      /* Free the data to be signed.  The agent will call _sign_complete
         once a response has been received.  Note that state is no longer
         necessarily valid when we get here. */
      ssh_xfree(packet);
      return FALSE;






























    case CANDIDATE_ORIGIN_FILE:
      switch (c->type)
        {
        case CANDIDATE_TYPE_NORMAL:
          SSH_TRACE(2, ("ssh_client_auth_pubkey_send_signature: reading %s",
                        c->privkeyfile));
          privkey = ssh_authc_pubkey_privkey_read(state->client->user_data,
                                                  c->privkeyfile,
                                                  NULL,
                                                  &key_comment,
                                                  state);
          ssh_xfree(key_comment);
          break;
      
#ifdef WITH_PGP
        case CANDIDATE_TYPE_PGPKEY:
          SSH_TRACE(2, 
                    ("ssh_client_auth_pubkey_send_signature: import pgpkey"));
          privkey = ssh2_pgp_privkey_import(c->pgp_seckey,
                                            c->pgp_seckey_len,
                                            NULL,
                                            c->pgp_keyname,
                                            state->client->config);
          break;
#endif /* WITH_PGP */


















        default:
          SSH_NOTREACHED;
        }













      if (privkey && !strncmp(c->pubkey_alg, "ssh-rsa", 7) &&
          !ssh_compat_rsa_private_key_change_scheme
          (privkey,




           !*state->client->common->compat_flags->
           rsa_hash_scheme_draft_incompatibility

           ))
        {
          SSH_TRACE(2, ("Changing private key scheme failed."));
          ssh_private_key_free(privkey);
          privkey = NULL;
        }          

      if (privkey == NULL)
        {
          /* The user probably gave the wrong passphrase. If this is the
             case, move to the next candidate. If we have tried all of
             them, notify the completion procedure.*/

          ssh_xfree(packet);

          if (state->auth_cancelled ||
              state->last_tried_candidate + 1 < state->num_candidates)
            {
              return TRUE;
            }

          ssh_client_auth_pubkey_sign_complete(SSH_AGENT_ERROR_FAILURE,
                                               NULL, 0, (void *)state);
          return FALSE;
        }

      SSH_DEBUG_HEXDUMP(5, ("auth_pubkey_send_signature: signing:"),
                        packet, packet_len);

      /* Use the private key to sign the data. */
      state->signature_packet = packet;
      state->signature_privkey = privkey;
      state->signature_candidate = c;
      ssh_private_key_sign_async(privkey, packet, packet_len, 
                                 ssh_client_auth_pubkey_sign_cb,
                                 (void *)state);
      return FALSE;
      
    default:
      ssh_fatal("ssh_client_auth_pubkey_send_signature: bad type %d",
                (int)c->type);

      /* NOTREACHED */
    }
  return FALSE; /* NOTREACHED */
}

/* Tries the authentication method indicated by state->last_tried_candidate.
   If there is no such candidate, fails authentication. */

void ssh_client_auth_pubkey_try_this_candidate(SshClientPubkeyAuth state)
{
  SshBuffer b;
  SshClientPubkeyAuthCandidate c;
  SshAuthClientResult command;



  do {
    if (state->auth_cancelled || state->num_candidates == 0 ||
        state->declined_candidates >= state->num_candidates ||
        state->last_tried_candidate >= state->num_candidates)
      {
        if (state->auth_cancelled)
          command = SSH_AUTH_CLIENT_CANCEL;
        else if (state->declined_candidates >= state->num_candidates ||
                 state->num_candidates == 0)
          command = SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD;
        else
          command = SSH_AUTH_CLIENT_FAIL;
         
        *state->state_placeholder = NULL;
        (*state->completion)(command, state->user, NULL,
                             state->completion_context);
        ssh_client_auth_pubkey_free_ctx(state);
        return;
      }

    /* Construct the first challenge packet */
    c = &state->candidates[state->last_tried_candidate];
    b = ssh_client_auth_pubkey_try_key_packet
      (c,




       *state->client->common->compat_flags->
       publickey_draft_incompatibility

       );
  } while (b == NULL);

  /* Probe whether the key is acceptable. */
  (*state->completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE_MULTIPLE,
                       state->user, b, state->completion_context);
  ssh_buffer_free(b);
}

/* Adds a key obtained from the agent to the list of candidates.
   This copies the certificates. */

void ssh_client_auth_pubkey_add_agent(SshClientPubkeyAuth state,
                                      const unsigned char *certs,
                                      size_t certs_len)
{
  SshClientPubkeyAuthCandidate c;
  char *pubkey_alg = NULL;




  if (certs_len == 0)
    return; /* Skip the URL keys. */








  if (pubkey_alg == NULL)
    pubkey_alg = ssh_pubkeyblob_type(certs, certs_len);
  
  if (pubkey_alg == NULL)
    return; /* Skip unknown alg types. */









  /* Extend the candidates array. */
  state->candidates =
    ssh_xrealloc(state->candidates,
                 (state->num_candidates + 1) * sizeof(state->candidates[0]));

  /* Get a pointer to the new candidate. */
  c = &state->candidates[state->num_candidates];

  /* Initialize the new candidate, copying memory for the certificate blob. */
  c->type = CANDIDATE_TYPE_NORMAL;



  c->origin = CANDIDATE_ORIGIN_AGENT;
  c->pubkeyblob = ssh_xmalloc(certs_len);
  memcpy(c->pubkeyblob, certs, certs_len);
  c->pubkeyblob_len = certs_len;
  c->pubkey_alg = pubkey_alg;
  c->privkeyfile = NULL;
#ifdef WITH_PGP
  c->pgp_keyname = NULL;
  c->pgp_seckey = NULL;
#endif
  /* Increase the number of candidates. */
  state->num_candidates++;
}

/* Adds a key file to the list of candidates.  This copies the file name. */

void ssh_client_auth_pubkey_add_keyfile(SshClientPubkeyAuth state,
                                        const char *privkeyfile)
{
  SshClientPubkeyAuthCandidate c;
  unsigned long magic;
  char buf[500];
  unsigned char *certs;
  size_t certs_len;
  unsigned char *pubkey_alg;

  /* Read the public key blob from the file. */
  ssh_snprintf(buf, sizeof(buf), "%s.pub", privkeyfile);
  magic = ssh2_key_blob_read(state->client->user_data, buf, TRUE, NULL,
                             &certs, &certs_len, state->client->config);
  if (magic != SSH_KEY_MAGIC_PUBLIC)
    {




      ssh_warning("Could not read public key file %s", buf);

      return;
    }

  pubkey_alg = (unsigned char *)ssh_pubkeyblob_type(certs, certs_len);
  if (pubkey_alg == NULL)
    {




      ssh_warning("Could not use public key file %s", buf);

      ssh_xfree(certs);
      return;
    }

  /* Extend the candidates array. */
  state->candidates =
    ssh_xrealloc(state->candidates,
                 (state->num_candidates + 1) * sizeof(state->candidates[0]));

  /* Get a pointer to the new candidate. */
  c = &state->candidates[state->num_candidates];

  /* Initialize the new candidate, copying memory for the certificate blob. */
  c->type = CANDIDATE_TYPE_NORMAL;
  c->origin = CANDIDATE_ORIGIN_FILE;
  c->pubkeyblob = certs;
  c->pubkeyblob_len = certs_len;
  c->pubkey_alg = pubkey_alg;
  c->privkeyfile = ssh_xstrdup(privkeyfile);
#ifdef WITH_PGP
  c->pgp_keyname = NULL;
  c->pgp_seckey = NULL;
#endif

  /* Increase the number of candidates. */
  state->num_candidates++;
}

#ifdef WITH_PGP
void ssh_client_auth_pubkey_add_pgpkey(SshClientPubkeyAuth state,
                                       const char *keyring,
                                       const char *name,
                                       const char *fingerprint,
                                       SshUInt32 id)
{
  SshClientPubkeyAuthCandidate c;
  unsigned char *blob;
  size_t blob_len;
  unsigned char *public_blob;
  size_t public_blob_len;
  Boolean found;
  SshPgpSecretKey secret_key;
  char *comment = NULL;
  unsigned char *pubkey_alg;

  if (name)
    found = ssh2_find_pgp_secret_key_with_name(state->client->user_data,
                                               keyring,
                                               name,
                                               &blob,
                                               &blob_len,
                                               &comment);
  else if (fingerprint)
    found = ssh2_find_pgp_secret_key_with_fingerprint(state->client->user_data,
                                                      keyring,
                                                      fingerprint,
                                                      &blob,
                                                      &blob_len,
                                                      &comment);
  else
    found = ssh2_find_pgp_secret_key_with_id(state->client->user_data,
                                             keyring,
                                             id,
                                             &blob,
                                             &blob_len,
                                             &comment);
  if (! found)
    {
      ssh_warning("Could not find pgp key");
      return;
    }
  if (ssh_pgp_secret_key_decode(blob, blob_len, &secret_key) == 0)
    {
      ssh_warning("Could not decode pgp key");
      memset(blob, 'F', blob_len);
      ssh_xfree(blob);
      ssh_xfree(comment);
      return;
    }
  if ((public_blob_len = ssh_encode_pubkeyblob(secret_key->public_key->key,
                                               &public_blob)) == 0)
    {
      ssh_warning("Could not encode pgp key to ssh2 keyblob");
      memset(blob, 'F', blob_len);
      ssh_xfree(blob);
      ssh_xfree(comment);
      return;
    }
  ssh_pgp_secret_key_free(secret_key);

  pubkey_alg =
    (unsigned char *)ssh_pubkeyblob_type(public_blob, public_blob_len);

  if (pubkey_alg == NULL)
    {
      ssh_warning("Could not get pk algorithm name from pgp key");
      ssh_xfree(public_blob);
      memset(blob, 'F', blob_len);
      ssh_xfree(blob);
      ssh_xfree(comment);
      return;
    }

  /* Extend the candidates array. */
  state->candidates =
    ssh_xrealloc(state->candidates,
                 (state->num_candidates + 1) * sizeof(state->candidates[0]));

  /* Get a pointer to the new candidate. */
  c = &state->candidates[state->num_candidates];

  /* Initialize the new candidate, copying memory for the certificate blob. */
  c->type = CANDIDATE_TYPE_PGPKEY;
  c->origin = CANDIDATE_ORIGIN_FILE;
  c->pubkeyblob = public_blob;
  c->pubkeyblob_len = public_blob_len;
  c->pubkey_alg = pubkey_alg;
  c->privkeyfile = NULL;
  c->pgp_keyname = comment;
  c->pgp_seckey = blob;
  c->pgp_seckey_len = blob_len;

  /* Increase the number of candidates. */
  state->num_candidates++;

  return;
}
#endif /* WITH_PGP */











































































void ssh_client_auth_pubkey_add_file_keys(SshClientPubkeyAuth state);












































































































































void ssh_client_auth_pubkey_add_agent_keys(void *context)
{
  int i;
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;

  for (i = 0; i < state->num_agent_keys; i++)
    ssh_client_auth_pubkey_add_agent(state, 
                                     state->agent_keys[i].certs, 
                                     state->agent_keys[i].certs_len);

  state->current_type++;
  ssh_client_auth_pubkey_add_candidates(state);
}

void ssh_client_auth_pubkey_add_candidates(void *context)
{
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;

  if (state->current_type > 5)
    {
      /* We have collected all potential keys from all sources. */

      /* Set the candidate to try.  Note that this may be equal to the number
         of candidates if we have none. */
      state->last_tried_candidate = 0;

      /* Try this candidate. */
      ssh_client_auth_pubkey_try_this_candidate(state);
      return;
    }
    
  switch (state->types[state->current_type])
    {
    case SSH_CLIENT_PUBKEY_FILE_KEYS:
      /* Add file/agent keys. */
      ssh_client_auth_pubkey_add_file_keys(state);
      break;




























    case SSH_CLIENT_PUBKEY_NONE:
      /* Start trying the candidates. */
      state->last_tried_candidate = 0;
      ssh_client_auth_pubkey_try_this_candidate(state);
      break;
    }
}

/* This completion function is called during the initialization of public
   key authentication when a list of keys supported by the agent has been
   obtained (the call is faked if we have no agent). */

void ssh_client_auth_pubkey_agent_list_complete(SshAgentError error,
                                                unsigned int num_keys,
                                                SshAgentKeyInfo keys,
                                                void *context)
{
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;
  unsigned int i;

  SSH_DEBUG(3, ("ssh_client_auth_pubkey_agent_list_complete err %d num %d",
                (int)error, (int)num_keys));

  /* Display a warning if we couldn't get the list. */
  if (error != SSH_AGENT_ERROR_OK)
    {
      ssh_warning("Obtaining a list of keys from the authentication agent "
                  "failed.");
      num_keys = 0;
    }
 

  /* Add all obtained keys as candidates for authentication. */
  for (i = 0; i < num_keys; i++)
    ssh_client_auth_pubkey_add_agent(state, keys[i].certs, keys[i].certs_len);








  i = 0;






  state->types[i++] = SSH_CLIENT_PUBKEY_FILE_KEYS;
  state->types[i] = SSH_CLIENT_PUBKEY_NONE;











  state->current_type = 0;  
  ssh_client_auth_pubkey_add_candidates(state);

}

































 
/* This functions adds the file keys to the public key authentication 
   candidates and start the authentication */
void ssh_client_auth_pubkey_add_file_keys(SshClientPubkeyAuth state)
{
  struct SshConfigPrivateKey **privkey;
  unsigned int i;

  /* This boolean is needed because we need to know if the operating system
     is windows in if statement. */
  Boolean windows;

  /* Construct a list of private key files that may be used to log in. */
  privkey = ssh_privkey_list(state->client->user_data, 
                             state->client->common->server_host_name,
                             state->client->common->config);
  windows = FALSE;









  /* If there are no suitable keys on our part, terminate early. */
  if (privkey != NULL)
    {
      for (i = 0; privkey[i]; i++)
        {
          if (!windows || 
              state->types[state->current_type] == SSH_CLIENT_PUBKEY_FILE_KEYS)
            {
              if (privkey[i]->type == SSH_PRIVKEY_PLAIN)
                {
                  SSH_DEBUG(2, ("adding keyfile \"%s\" to candidates", 
                                privkey[i]->keyfile));
                  ssh_client_auth_pubkey_add_keyfile(state, 
                                                     privkey[i]->keyfile);
                }
#ifdef WITH_PGP
              else if (privkey[i]->type == SSH_PRIVKEY_PGP)
                {
                  SSH_DEBUG(2, 
                            ("adding pgpkeyfile f=\"%s\" n=\"%s\" p=\"%s\" "
                             "i=0x%08lx to candidates", 
                             privkey[i]->pgp_keyring,
                             (privkey[i]->pgp_name ? 
                              privkey[i]->pgp_name : 
                              ""),
                             (privkey[i]->pgp_fingerprint ? 
                              privkey[i]->pgp_fingerprint : 
                              ""),
                             privkey[i]->pgp_id));
                  ssh_client_auth_pubkey_add_pgpkey
                    (state,
                     privkey[i]->pgp_keyring,
                     privkey[i]->pgp_name,
                     privkey[i]->pgp_fingerprint,
                     privkey[i]->pgp_id);
                }
#endif /* WITH_PGP */
            }













          
          ssh_xfree(privkey[i]->keyfile);
#ifdef WITH_PGP
          ssh_xfree(privkey[i]->pgp_keyring);
          ssh_xfree(privkey[i]->pgp_name);
          ssh_xfree(privkey[i]->pgp_fingerprint);
#endif /* WITH_PGP */
          /* XXX Free possible cert-specific fields here */
          ssh_xfree(privkey[i]);
        }
      ssh_xfree(privkey);
    }
  state->current_type++;
  ssh_client_auth_pubkey_add_candidates(state);
}

/* This completion function is called during the initialization of public
   key authentication when opening a connection to the agent is complete.
   If a connection was successful, `agent' will be the agent handle.
   Otherwise, `agent' will be NULL.  `context' points to the state object. */

void ssh_client_auth_pubkey_agent_open_complete(SshAgent agent, void *context)
{
  SshClientPubkeyAuth state = (SshClientPubkeyAuth)context;

  SSH_DEBUG(4, ("ssh_client_auth_pubkey_agent_open_complete agent=0x%lx",
                (unsigned long)agent));
  
  if (agent)
    {

      /* A connection to the agent was successfully opened.  Save the
         handle. */
      state->agent = agent;

      /* Request a list of keys supported by the agent. */
      ssh_agent_list(agent, ssh_client_auth_pubkey_agent_list_complete,
                     (void *)state);

    }
  else
    {
      /* No agent.  Fake a callback to the agent list completion, with no
         keys. */
      ssh_client_auth_pubkey_agent_list_complete(SSH_AGENT_ERROR_OK,
                                                 0, NULL,
                                                 (void *)state);
    }
}

/* Public key authentication, client-side.  This super-function handles
   everything related to the public key authentication method. */

void ssh_client_auth_pubkey(SshAuthClientOperation op,
                            const char *user,
                            unsigned int packet_type,
                            SshBuffer packet_in,
                            unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            SshAuthClientCompletionProc completion,
                            void *completion_context,
                            void *method_context)
{
  SshClientPubkeyAuth state;
  SshClient client;

  SSH_DEBUG(6, ("auth_pubkey op = %d  user = %s", op, user));

  client = (SshClient)method_context;
  state = *state_placeholder;

  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      /* For now, don't try to do anything in the non-interactive phase.
         However, we should probably later make this try those keys that
         don't need passphrases in this phase.  XXX */
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_START:
      /* This is the first operation for doing public key authentication.
         We should not have any previous saved state when we come here. */
      SSH_ASSERT(*state_placeholder == NULL);

      /* Initialize a context. */
      state = ssh_xcalloc(1, sizeof(*state));
      state->agent = NULL;
      state->last_tried_candidate = 0;
      state->num_candidates = 0;
      state->candidates = NULL;
      state->completion = completion;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      state->user = ssh_xstrdup(user);
      state->client = client;
      state->auth_cancelled = FALSE;
      state->declined_candidates = 0;

      /* Assign the state to the placeholder that survives across calls. */
      *state_placeholder = state;

      /* Try to open the authentication agent.  Rest of processing will be
         done in the callback. */









      ssh_agent_open(ssh_client_auth_pubkey_agent_open_complete,
                     (void *)state);



      break;
      
    case SSH_AUTH_CLIENT_OP_CONTINUE:
      /* Got a continuation packet from the server. */
      SSH_ASSERT(state != NULL);

      /* Save information in the state. */
      state->completion = completion;
      state->completion_context = completion_context;
      state->state_placeholder = state_placeholder;
      ssh_xfree(state->user);
      state->user = ssh_xstrdup(user);
      state->client = client;

      /* Process the received continuation packet. */
      switch (packet_type)
        {
        case SSH_MSG_USERAUTH_FAILURE:
          /* The server is not willing to accept the key.  Move to the next
             candidate, and try authenticating with it. */
          if (state->signature_sent)
            {
              Boolean partial_success;
              size_t bytes = 0L;
              
              /* Decode the packet to get the `partial success'
                 field. */
              bytes = ssh_decode_buffer(packet_in,
                                        /* We don't care about the
                                           continuations. */
                                        SSH_FORMAT_UINT32_STR, NULL, NULL,
                                        SSH_FORMAT_BOOLEAN, &partial_success,
                                        SSH_FORMAT_END);

              if (partial_success == TRUE )
                {
                  /* The auth layer has already sanity checked the
                     packet. (XXX we can in principle fail here because
                     of memory allocation errors) */
                  SSH_ASSERT(bytes);
                  /* If the public key authentication was a success, the
                     auth layer will handle calling other auth methods
                     (including this, if necessary). */
                  goto partial_success;
                }
              SSH_TRACE(0, ("Server rejected the signature."));
              state->signature_sent = FALSE;
            }
          
          state->declined_candidates++;
        try_again:
          state->last_tried_candidate++;
          ssh_client_auth_pubkey_try_this_candidate(state);
          break;

        case SSH_MSG_USERAUTH_SUCCESS:
          if (!state->signature_sent)
            ssh_warning("ssh_client_auth_pubkey: server gave success "
                        "without receiving a valid signature (?).");

        partial_success:
          SSH_TRACE(0, ("Public key authentication was successful."));
          ssh_client_auth_pubkey_free_ctx(state);
          *state_placeholder = NULL;
          break;
        case SSH_MSG_USERAUTH_PK_OK:            
          /* The server is willing to accept this key as authentication. */
          if (ssh_client_auth_pubkey_send_signature(state, user,
                                                    session_id,
                                                    session_id_len))
            {
              /* Reading private part of the key was failed. Move to the
                 next candidate.*/
              goto try_again;
            }
          
          break;

        default:
          /* Unexpected response. */
          ssh_fatal("ssh_client_auth_pubkey: unexpected response packet %d",
                    packet_type);
          /*NOTREACHED*/
        }
      break;

    case SSH_AUTH_CLIENT_OP_ABORT:
      /* Abort the authentication operation immediately. */
      ssh_client_auth_pubkey_free_ctx(state);
      *state_placeholder = NULL;
      break;

      /* something weird is going on.. */
    default:
      ssh_fatal("ssh_client_auth_pubkey: unknown op %d", (int)op);
    }
  return;
}
