/*

  authc-passwd.c

  Authors:
        Tatu Ylonen <ylo@ssh.com>
        Markku-Juhani Saarinen <mjos@ssh.com>
        Timo J. Rinne <tri@ssh.com>
        Sami Lehtinen <sjl@ssh.com>

  Copyright (C) 1997-2000 SSH Communications Security Corp, Helsinki, Finland
  All rights reserved.

  Password authentication, client side.
  
*/

#include "ssh2includes.h"
#include "sshencode.h"
#include "sshauth.h"

#include "readpass.h"

#include "sshclient.h"
#include "sshconfig.h"
#include "sshmsgs.h"
#include "sshencode.h"
#include "sshappcommon.h"

#define SSH_DEBUG_MODULE "Ssh2AuthPasswdClient"

#ifdef SSH_NEW_READPASS_ELOOP
typedef struct AuthPasswdCompletionCtxRec
{
  
} *AuthPasswdCompletionCtx;
#endif /* SSH_NEW_READPASS_ELOOP */

/* Password authentication, client-side. */

void ssh_client_auth_passwd(SshAuthClientOperation op,
                            const char *user,
                            unsigned int packet_type,
                            SshBuffer packet_in,
                            const unsigned char *session_id,
                            size_t session_id_len,
                            void **state_placeholder,
                            SshAuthClientCompletionProc completion,
                            void *completion_context,
                            void *method_context)
{
  char *password = NULL, *passagain = NULL, *prompt;
  SshBuffer b;
  char buf[100];
  SshClient client = (SshClient)method_context;
  SshConfig clientconf = client->config;
  SshCommon common = client->common;
#ifdef SSH_NEW_READPASS_ELOOP
  AuthPasswdCompletionCtx auth_completion_ctx;
#endif /* SSH_NEW_READPASS_ELOOP */
  char *old_password = *state_placeholder;
  size_t ret_val;
  Boolean match = FALSE;
  
  SSH_DEBUG(6, ("auth_password op = %d  user = %s", op, user));
  
  switch (op)
    {
    case SSH_AUTH_CLIENT_OP_START:
#ifdef SSH_NEW_READPASS_ELOOP
      auth_completion_ctx = ;
#endif /* SSH_NEW_READPASS_ELOOP */
      
      if (!clientconf->try_empty_password)
        {
          if (clientconf->password)
            {
              password = clientconf->password;
              clientconf->password = NULL;
            }
          else
            password = NULL;

          if (clientconf->password_prompt == NULL)
            ssh_snprintf(buf, sizeof(buf), "%s's password: ", user);
          else
            ssh_snprintf(buf, sizeof(buf), "%s", clientconf->password_prompt);

          if (!clientconf->batch_mode)
            {

              /* decrease and check number of password prompts remaining */
              clientconf->number_of_password_prompts--;
              if (clientconf->number_of_password_prompts < 0)
                {
                  (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                                user, NULL, completion_context);

                  break;
                }




              SSH_TRACE(2, ("Starting password query..."));          
#ifndef SSH_NEW_READPASS_ELOOP
              password = ssh_read_passphrase(buf, FALSE);
#else /* SSH_NEW_READPASS_ELOOP */
              ssh_readpass_eloop();
              return;
#endif /* SSH_NEW_READPASS_ELOOP */

            }
          else
            {
              SSH_TRACE(2, ("In Batchmode, so we're not asking the "
                            "user for password."));
              password = NULL;
              (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                            user, NULL, completion_context);
              break;
            }
      
          if (password == NULL)
            {
              (*completion)(SSH_AUTH_CLIENT_CANCEL, user, NULL,
                            completion_context);
              break;
            }
        }
      else
        {
          password = "";
          clientconf->try_empty_password = FALSE;
        }
      
      b = ssh_xbuffer_allocate();
      ret_val = ssh_encode_buffer(b,
                                  SSH_FORMAT_BOOLEAN, FALSE,
                                  SSH_FORMAT_UINT32_STR, password,
                                  strlen(password),
                                  SSH_FORMAT_END);
      SSH_VERIFY(ret_val > 0);
      (*completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                    user, b, completion_context);

      *state_placeholder = password;
      ssh_buffer_free(b);
      break;
      
    case SSH_AUTH_CLIENT_OP_START_NONINTERACTIVE:
      *state_placeholder = NULL;
      ssh_xfree(old_password);
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
      
    case SSH_AUTH_CLIENT_OP_CONTINUE:
      /* Changing of passwords (for servers which support this). */
      if (packet_type != SSH_MSG_USERAUTH_PASSWD_CHANGEREQ)
        {
          /* Wrong type of packet received. */
          SSH_DEBUG(2, ("Received packet type %d, expecting %d (%s).",
                        packet_type,
                        SSH_MSG_USERAUTH_PASSWD_CHANGEREQ,
                        "SSH_MSG_USERAUTH_PASSWD_CHANGEREQ"));
          ssh_warning("Protocol error in password authentication (expecting "
                      "change request, got something else.");
          *state_placeholder = NULL;
          ssh_xfree(old_password);
          (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                        user, NULL, completion_context);
          break;
        }

      if (*common->compat_flags->
          no_lang_tag_in_passwd_chreq_draft_incompatibility)
        {
          ret_val = ssh_decode_buffer(packet_in,
                                      SSH_FORMAT_UINT32_STR, &prompt, NULL,
                                      SSH_FORMAT_END);
        }
      else
        {
          ret_val = ssh_decode_buffer(packet_in,
                                      SSH_FORMAT_UINT32_STR, &prompt, NULL,
                                      SSH_FORMAT_UINT32_STR,
                                      /* XXX lang tag */ NULL, NULL,
                                      SSH_FORMAT_END);
        }

      if (ret_val == 0 || ssh_buffer_len(packet_in) != 0)
        {
          /* protocol error. */
          ssh_warning("Protocol error in password authentication (malformed "
                      "change request packet).");
          *state_placeholder = NULL;
          ssh_xfree(old_password);
          (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                        user, NULL, completion_context);
          break;
        }

      ssh_informational("%s\r\n", prompt);

      while (!match)
        {
          if (clientconf->batch_mode)
            {
              /* We should not come here, but checking anyways, if
                 somebody has made some pervertic changes (i.e. reading
                 the password from stdin or file, if in batchmode. */
              (*completion)(SSH_AUTH_CLIENT_FAIL_AND_DISABLE_METHOD,
                            user, NULL, completion_context);
              return;
            }
          ssh_xfree(password);
          password = ssh_read_passphrase("New password: ", FALSE);
          if (password == NULL)
            goto cancel;
          passagain = ssh_read_passphrase("Enter password again: ", FALSE);
          if (passagain == NULL)
            goto cancel;
          match = !strcmp(password, passagain);
          if (!match)
            ssh_informational("Passwords don't match, try again.\r\n");
          ssh_xfree(passagain);
        }
      
      b = ssh_xbuffer_allocate();

      ret_val = ssh_encode_buffer(b,
                                  SSH_FORMAT_BOOLEAN, TRUE,
                                  SSH_FORMAT_UINT32_STR,
                                  old_password, strlen(old_password),
                                  SSH_FORMAT_UINT32_STR,
                                  password, strlen(password),
                                  SSH_FORMAT_END);
      SSH_VERIFY(ret_val > 0);
      (*completion)(SSH_AUTH_CLIENT_SEND_AND_CONTINUE,
                    user, b, completion_context);
      ssh_buffer_free(b);
      
      break;
    cancel:
      /* Cancel the authentication (and password change). */
      ssh_xfree(old_password);
      ssh_xfree(password);
      ssh_xfree(passagain);
      *state_placeholder = NULL;
      (*completion)(SSH_AUTH_CLIENT_CANCEL,
                    user, NULL, completion_context);
      break;
    case SSH_AUTH_CLIENT_OP_ABORT:
      *state_placeholder = NULL;
      ssh_xfree(old_password);
      
      break;
      
    default:
      ssh_debug("ssh_client_auth_password: unknown op %d", (int)op);
      *state_placeholder = NULL;
      ssh_xfree(old_password);
      (*completion)(SSH_AUTH_CLIENT_FAIL, user, NULL, completion_context);
      break;
    }
}

