// pythoncom.cpp :
// $Id:$

/***
Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc
***/

#include "stdafx.h"
#include <comcat.h>
#include <objbase.h>
#include "PythonCOM.h"
#include "PythonCOMServer.h"
#include "PyFactory.h"

extern int PyCom_RegisterCoreSupport();
extern PyObject *g_obPyCom_MapNameToType;
extern PyObject *g_obPyCom_MapIIDToType;
extern PyObject *g_obPyCom_MapGatewayIIDToName;

static PyObject *g_obEmpty = NULL;
static PyObject *g_obMissing = NULL;

// Storage related functions.
extern PyObject *pythoncom_StgOpenStorage(PyObject *self, PyObject *args);
extern PyObject *pythoncom_StgIsStorageFile(PyObject *self, PyObject *args);
extern PyObject *pythoncom_StgCreateDocfile(PyObject *self, PyObject *args);
extern PyObject *pythoncom_StgCreateDocfileOnILockBytes(PyObject *self, PyObject *args);

// Typelib related functions
extern PyObject *pythoncom_loadtypelib(PyObject *self, PyObject *args);
extern PyObject *pythoncom_loadregtypelib(PyObject *self, PyObject *args);
/* Python exceptions */
extern PyObject* pycom_Error;     /* 'Python level' errors */

/* Debug/Test helpers */
extern LONG _PyCom_GetInterfaceCount(void);
extern LONG _PyCom_GetGatewayCount(void);

#pragma optimize ("y", off)
// This optimisation seems to screw things in release builds...

/* MODULE FUNCTIONS: pythoncom */
// @pymethod <o PyIUnknown>|pythoncom|CoCreateInstance|Create a new instance of an OLE automation server.
static PyObject *pythoncom_CoCreateInstance(PyObject *self, PyObject *args)
{
	PyObject *obCLSID;
	PyObject *obUnk;
	DWORD dwClsContext;
	PyObject *obiid;
	CLSID clsid;
	IUnknown *punk;
	CLSID iid;
	if (!PyArg_ParseTuple(args, "OOiO:CoCreateInstance",
		&obCLSID,	// @pyparm <o PyIID>|iid||The IID of the object to create
		&obUnk,	// @pyparm <o PyIUnknown>|unkOuter||The outer unknown, or None
		&dwClsContext,// @pyparm int|context||The create context for the object
		&obiid)) // @pyparm <o PyIID>|iid||The IID required from the object
		return NULL;
	if (!PyCom_CLSIDFromPyObject(obCLSID, &clsid))
		return NULL;
	if (!PyCom_IUnknownFromPyObject(obUnk, &punk))
		return NULL;
	if (!PyCom_CLSIDFromPyObject(obiid, &iid))
		return NULL;
	// Make the call.
	IUnknown *result = NULL;
	SCODE sc = CoCreateInstance(clsid, punk, dwClsContext, iid, (void **)&result);
	if (FAILED(sc))
		return OleSetOleError(sc);
	return PyCom_PyObjectFromIUnknown(result, iid);
}

// @pymethod int|pythoncom|CoRegisterClassObject|Registers an EXE class object with OLE so other applications can connect to it.
static PyObject *pythoncom_CoRegisterClassObject(PyObject *self, PyObject *args)
{
	DWORD reg;
	DWORD context;
	DWORD flags;
	PyObject *obIID;
	IID iid;

	if (!PyArg_ParseTuple(args, "Oii:CoRegisterClassObject",
		&obIID,	// @pyparm <o PyIID>|iid||The IID of the object to register
		&context, // @pyparm int|context||The create context for the server
		&flags))  // @pyparm int|flags||Create flags.
		return NULL;

	if (!PyCom_CLSIDFromPyObject(obIID, &iid))
		return NULL;

	// XXX - todo - hack - bug - etc!
	// LEAKS - should keep track (possibly in array indexed by the DWORD result?
	CPyFactory *pFact = new CPyFactory(iid);
	HRESULT hr = CoRegisterClassObject(iid, pFact, context, flags, &reg);
	if (FAILED(hr)) {
		delete pFact;
		return OleSetOleError(hr);
	}
	// @rdesc The result is a handle which should be revoked using <om pythoncom.CoRevokeClassObject>
	return PyInt_FromLong(reg);
}
// @pymethod |pythoncom|CoRevokeClassObject|Informs OLE that a class object, previously registered with the <om pythoncom.CoRegisterClassObject> method, is no longer available for use. 
static PyObject *pythoncom_CoRevokeClassObject(PyObject *self, PyObject *args)
{
	DWORD reg;

	if (!PyArg_ParseTuple(args, "i:CoRevokeClassObject",
		&reg))	// @pyparm int|reg||The value returned from <om pythoncom.CoRegisterClassObject>
		return NULL;

	HRESULT hr = CoRevokeClassObject(reg);
	if (FAILED(hr)) {
		return OleSetOleError(hr);
	}
	Py_INCREF(Py_None);
	return Py_None;
}


#pragma optimize ("", on)

// @pymethod int|pythoncom|_GetInterfaceCount|Retrieves the number of interface objects currently in existance
static PyObject *pythoncom_GetInterfaceCount(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":_GetInterfaceCount"))
		return NULL;
	return PyInt_FromLong(_PyCom_GetInterfaceCount());
}

// @pymethod int|pythoncom|_GetGatewayCount|Retrieves the number of interface objects currently in existance
static PyObject *pythoncom_GetGatewayCount(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":_GetGatewayCount"))
		return NULL;
	return PyInt_FromLong(_PyCom_GetGatewayCount());
}

// @pymethod <o PyIUnknown>|pythoncom|GetActiveObject|Retrieves an object representing a running object registered with OLE
static PyObject *pythoncom_GetActiveObject(PyObject *self, PyObject *args)
{
	PyObject *obCLSID;
	// @pyparm CLSID|cls||An identifier for the program.  Usually "program.item"
	if (!PyArg_ParseTuple(args, "O:GetActiveObject",
		&obCLSID))
		return NULL;
	CLSID clsid;
	if (!PyCom_CLSIDFromPyObject(obCLSID, &clsid))
		return NULL;
	// Make the call.
	IUnknown *result = NULL;
	SCODE sc = GetActiveObject(clsid, NULL, &result);
	if (FAILED(sc))
		return OleSetOleError(sc);
	return PyCom_PyObjectFromIUnknown(result, IID_IUnknown);
}

// @pymethod  <o PyIDispatch>|pythoncom|Connect|Connect to an already running OLE automation server.
static PyObject *pythoncom_connect(PyObject *self, PyObject *args)
{
	PyObject *obCLSID;
	// @pyparm CLSID|cls||An identifier for the program.  Usually "program.item"
	if (!PyArg_ParseTuple(args, "O:Connect",
		&obCLSID))
		return NULL;
	CLSID clsid;
	if (!PyCom_CLSIDFromPyObject(obCLSID, &clsid))
		return NULL;

	IUnknown *unk = NULL;
	HRESULT hr = GetActiveObject(clsid, NULL, &unk);
	if (FAILED(hr) || unk == NULL)
		return OleSetOleError(hr);
	IDispatch *disp = NULL;
	SCODE sc = unk->QueryInterface(IID_IDispatch, (void**)&disp);
	unk->Release();
	if (FAILED(sc) || disp == NULL)
		return OleSetOleError(sc);
	return PyCom_PyObjectFromIUnknown(disp, IID_IDispatch);
	// @comm This function is equivilent to <om pythoncom.GetActiveObject>(clsid).<om pythoncom.QueryInterace>(pythoncom.IID_IDispatch)
}

// @pymethod <o PyIDispatch>|pythoncom|new|Create a new instance of an OLE automation server.
static PyObject *pythoncom_new(PyObject *self, PyObject *args)
{
	PyErr_Clear();
	PyObject *progid;
	// @pyparm CLSID|cls||An identifier for the program.  Usually "program.item"
	if (!PyArg_ParseTuple(args, "O", &progid))
		return NULL;

	// @comm This is just a wrapper for the CoCreateInstance method.
	// Specifically, this call is identical to:
	// <nl>pythoncom.CoCreateInstance(cls, None, pythoncom.CLSCTX_SERVER, pythoncom.IID_IDispatch)
	PyObject *obIID = PyCom_PyIIDObjectFromIID(IID_IDispatch);
	PyObject *newArgs = Py_BuildValue("OOiO", progid, Py_None, CLSCTX_SERVER, obIID);
	PyTS_DECREF(obIID);
	PyObject *rc = pythoncom_CoCreateInstance(self, newArgs);
	PyTS_DECREF(newArgs);
	return rc;
}


// @pymethod <o PyIID>|pythoncom|CreateGuid|Creates a new, unique GUIID.
static PyObject *pythoncom_createguid(PyObject *self, PyObject *args)
{
	PyErr_Clear();
	if (PyTuple_Size(args) != 0)
		return OleSetTypeError("function requires no arguments");
	GUID guid;
	CoCreateGuid(&guid);

	return PyCom_PyIIDObjectFromIID(guid);
}

// @pymethod string|pythoncom|ProgIDFromCLSID|Converts a CLSID string to a progID.
static PyObject *pythoncom_progidfromclsid(PyObject *self, PyObject *args)
{
	PyObject *obCLSID;
	// @pyparm IID|clsid||A CLSID in string or native format
	if (!PyArg_ParseTuple(args, "O", &obCLSID))
		return NULL;
	
	CLSID clsid;
	if (!PyCom_CLSIDFromPyObject(obCLSID, &clsid))
		return NULL;
	LPOLESTR progid = NULL;
	HRESULT sc = ProgIDFromCLSID(clsid, &progid);
	if (FAILED(sc))
		return OleSetOleError(sc);

	PyObject *ob = MakeOLECHARToObj(progid);
	CoTaskMemFree(progid);
	return ob;
}

// @pymethod string|pythoncom|GetScodeString|Returns the string for an OLE scode.
static PyObject *pythoncom_GetScodeString(PyObject *self, PyObject *args)
{
	SCODE scode;
	char buf[512];
	// @pyparm int|scode||The OLE error code for the scode string requested.
	if (!PyArg_ParseTuple(args, "i", &scode))
		return NULL;
	GetScodeString(scode, buf, sizeof(buf));
	return Py_BuildValue("s", buf);
}

// @pymethod string|pythoncom|GetScodeRangeString|Returns the scode range string, given an OLE scode.
static PyObject *pythoncom_GetScodeRangeString(PyObject *self, PyObject *args)
{
	SCODE scode;
	// @pyparm int|scode||An OLE error code to return the scode range string for.
	if (!PyArg_ParseTuple(args, "i", &scode))
		return NULL;
	return Py_BuildValue("z", GetScodeRangeString(scode) );
}

// @pymethod string|pythoncom|GetSeverityString|Returns the severity string, given an OLE scode.
static PyObject *pythoncom_GetSeverityString(PyObject *self, PyObject *args)
{
	SCODE scode;
	// @pyparm int|scode||The OLE error code for the severity string requested.
	if (!PyArg_ParseTuple(args, "i", &scode))
		return NULL;
	return Py_BuildValue("z", GetSeverityString(scode) );
}

// @pymethod string|pythoncom|GetFacilityString|Returns the facility string, given an OLE scode.
static PyObject *pythoncom_GetFacilityString(PyObject *self, PyObject *args)
{
	SCODE scode;
	// @pyparm int|scode||The OLE error code for the facility string requested.
	if (!PyArg_ParseTuple(args, "i", &scode))
		return NULL;
	return Py_BuildValue("z", GetFacilityString(scode) );
}

static PyObject *pythoncom_RecoverFromID(PyObject *self, PyObject *args)
{
	int p = 0;

	if ( !PyArg_ParseTuple(args, "i", &p) )
		return NULL;

	if ( !p )
	{
		Py_INCREF(Py_None);
		return Py_None;
	}

	return (PyObject *)p;
}

// @pymethod <o PyIDispatch>|pythoncom|UnwrapObject|Unwraps a Python instance in a gateway object.
static PyObject *pythoncom_UnwrapObject(PyObject *self, PyObject *args)
{
	PyObject *ob;
	// @pyparm object|ob||The object to unwrap.
	if ( !PyArg_ParseTuple(args, "O", &ob ) )
		return NULL;
	if ( !PyIBase::is_object(ob, &PyIUnknown::type) ) {
		PyErr_SetString(PyExc_ValueError, "argument is not a COM object");
		return NULL;
	}

	IInternalUnwrapPythonObject *pUnwrapper;
	if (S_OK!=((PyIUnknown *)ob)->m_obj->QueryInterface(IID_IInternalUnwrapPythonObject, (void **)&pUnwrapper)) {
		PyErr_SetString(PyExc_ValueError, "argument is not a Python gateway");
		return NULL;
	}
	PyObject *retval;
	HRESULT hr = pUnwrapper->Unwrap(&retval);
	pUnwrapper->Release();
	if (S_OK!=hr)
		return OleSetOleError(hr);
	return retval;
}

// @pymethod <o PyIDispatch>|pythoncom|WrapObject|Wraps a Python instance in a gateway object.
static PyObject *pythoncom_WrapObject(PyObject *self, PyObject *args)
{
	PyObject *ob;
	PyObject *obIID = NULL;
	IID iid = IID_IDispatch;
	PyObject *obIIDInterface = NULL;
	IID iidInterface = IID_IDispatch;

	// @pyparm object|ob||The object to wrap.
	// @pyparm <o PyIID>|gatewayIID|IID_IDispatch|The IID of the gateway object to create (ie, the interface of the server object wrapped by the return value)
	// @pyparm <o PyIID>|interfaceIID|IID_IDispatch|The IID of the interface object to create (ie, the interface of the returned object)
	if ( !PyArg_ParseTuple(args, "O|OO", &ob, &obIID, &obIIDInterface) )
		return NULL;

	// @rdesc Note that there are 2 objects created by this call - a gateway (server) object, suitable for
	// use by other external COM clients/hosts, as well as a Python interface (client) object, which
	// maps to the new gateway.
	// <nl>There are some unusual cases where the 2 interfaces are not identical.
	// If you need to do this, you should know exactly what you are doing, and why!
	if (obIID && obIID != Py_None) {
		if (!PyCom_CLSIDFromPyObject(obIID, &iid))
			return NULL;
	}
	if (obIIDInterface && obIIDInterface != Py_None) {
		if (!PyCom_CLSIDFromPyObject(obIIDInterface, &iidInterface))
			return NULL;
	}
	// Make a gateway of the specific IID we ask for.
	// The gateway must exist (ie, we _must_ support PyGIXXX
	IUnknown *pDispatch;
	HRESULT hr = PyCom_MakeRegisteredGatewayObject(iid, ob, (void **)&pDispatch);
	if ( FAILED(hr) )
		return OleSetOleError(hr);

	/* pass the pDispatch reference into this thing */
	/* ### this guy should always AddRef() ... */
	PyObject *result = PyCom_PyObjectFromIUnknown(pDispatch, iidInterface, FALSE);
	if ( !result )
	{
		pDispatch->Release();
		return NULL;
	}

	return result;
}

// @pymethod <o PyIID>|pythoncom|MakeIID|Creates a new IID object.
static PyObject *pythoncom_MakeIID(PyObject *self, PyObject *args)
{
	return PyWinMethod_NewIID(self, args);
}

// @pymethod <o PyTime>|pythoncom|MakeTime|Creates a new time object.
static PyObject *pythoncom_MakeTime(PyObject *self, PyObject *args)
{
	return PyWinMethod_NewTime(self, args);
}

// @pymethod <o PyIMoniker>,int,<o PyIBindCtx>|pythoncom|MkParseDisplayName|Parses a moniker display name into a moniker object. The inverse of IMoniker::GetDisplayName.
static PyObject *pythoncom_MkParseDisplayName(PyObject *self, PyObject *args)
{
	const char *displayName;
	PyObject *obBindCtx = NULL;

	// @pyparm string|displayName||The display name to parse
	// @pyparm <o PyIBindCtx>|bindCtx|<o PyIBindCtx>|The bind context object to use.
	// @comm If a binding context is not provided, then one will be created.
	// Any binding context created or passed in will be returned to the
	// caller.
	if ( !PyArg_ParseTuple(args, "s|O:MkParseDisplayName", &displayName, &obBindCtx) )
		return NULL;

	HRESULT hr;
	IBindCtx *pBC;
	if ( obBindCtx == NULL )
	{
		hr = CreateBindCtx(0, &pBC);
		if ( FAILED(hr) )
			return OleSetOleError(hr);

		/* pass the pBC ref into obBindCtx */
		obBindCtx = PyCom_PyObjectFromIUnknown(pBC, IID_IBindCtx, FALSE);
	}
	else
	{
		if ( !PyCom_InterfaceFromPyObject(obBindCtx, IID_IBindCtx, (LPVOID*)&pBC, FALSE) )
			return NULL;

		/* we want our own ref to obBindCtx, but not pBC */
		Py_INCREF(obBindCtx);
		pBC->Release();
	}
	/* at this point: we own a ref to obBindCtx, but not pBC */

	USES_CONVERSION;
	ULONG chEaten;
	IMoniker *pmk;
	hr = MkParseDisplayName(pBC, A2W(displayName), &chEaten, &pmk);
	if ( FAILED(hr) )
	{
		Py_DECREF(obBindCtx);
		return OleSetOleError(hr);
	}

	/* pass ownership of the moniker into the result */
	PyObject *obMoniker = PyCom_PyObjectFromIUnknown(pmk, IID_IMoniker, FALSE);

	/* build the result */
	PyObject *result = Py_BuildValue("OiO", obMoniker, chEaten, obBindCtx);

	/* done with these obs */
	Py_XDECREF(obMoniker);
	Py_DECREF(obBindCtx);

	return result;
}

// @pymethod <o PyIMoniker>|pythoncom|CreatePointerMoniker|Creates a new <o PyIMoniker> object.
static PyObject *pythoncom_CreatePointerMoniker(PyObject *self, PyObject *args)
{
	PyObject *obUnk;
	// @pyparm <o PyIUnknown>|IUnknown||The interface for the moniker.
	if ( !PyArg_ParseTuple(args, "O:CreatePointerMoniker", &obUnk) )
		return NULL;

	IUnknown *punk;
	if ( !PyCom_InterfaceFromPyObject(obUnk, IID_IUnknown, (LPVOID*)&punk, FALSE) )
		return NULL;

	IMoniker *pmk;
	HRESULT hr = CreatePointerMoniker(punk, &pmk);
	punk->Release();

	if ( FAILED(hr) )
		return OleSetOleError(hr);

	return PyCom_PyObjectFromIUnknown(pmk, IID_IMoniker, FALSE);
}

// @pymethod <o PyIMoniker>|pythoncom|CreateFileMoniker|Creates a new <o PyIMoniker> object.
static PyObject *pythoncom_CreateFileMoniker(PyObject *self, PyObject *args)
{
	PyObject *obName;
	// @pyparm string/<o PyUnicode>|filename||The name of the file.
	if ( !PyArg_ParseTuple(args, "O:CreateFileMoniker", &obName) )
		return NULL;

	BSTR bstrName;
	if (!PyWinObject_AsBstr(obName, &bstrName))
		return NULL;

	IMoniker *pmk;
	HRESULT hr = CreateFileMoniker(bstrName, &pmk);
	PyWinObject_FreeBstr(bstrName);

	if ( FAILED(hr) )
		return OleSetOleError(hr);

	return PyCom_PyObjectFromIUnknown(pmk, IID_IMoniker, FALSE);
}

// @pymethod <o PyIID>|pythoncom|GetClassFile|Supplies the CLSID associated with the given filename.
static PyObject *pythoncom_GetClassFile(PyObject *self, PyObject *args)
{
	CLSID clsid;
	PyObject *obFileName;
	BSTR fname;
	// @pyparm string|fileName||The filename for which you are requesting the associated CLSID.
	if (!PyArg_ParseTuple(args, "O", &obFileName))
		return NULL;

	if (!PyCom_BstrFromPyObject(obFileName, &fname, FALSE))
		return NULL;

	HRESULT hr = GetClassFile(fname, &clsid);
	SysFreeString(fname);
	if (FAILED(hr))
		return OleSetOleError(hr);
	return PyCom_PyIIDObjectFromIID(clsid);
}

static PyObject *pythoncom_CoInitialize(PyObject *self, PyObject *args)
{
	HRESULT hr = PyCom_CoInitialize(NULL);
	if (FAILED(hr))
		OleSetOleError(hr);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *pythoncom_CoUninitialize(PyObject *self, PyObject *args)
{
	PyCom_CoUninitialize();
	Py_INCREF(Py_None);
	return Py_None;
}

// @pymethod |pythoncom|CoFreeUnusedLibraries|Unloads any DLLs that are no longer in use and that, when loaded, were specified to be freed automatically.
static PyObject *pythoncom_CoFreeUnusedLibraries(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":CoFreeUnusedLibraries"))
		return NULL;
	CoFreeUnusedLibraries();
	Py_INCREF(Py_None);
	return Py_None;
}

// @pymethod <o PyIRunningObjectTable>|pythoncom|GetRunningObjectTable|Creates a new <o PyIRunningObjectTable> object.
static PyObject *pythoncom_GetRunningObjectTable(PyObject *self, PyObject *args)
{
	DWORD reserved = 0;
	if ( !PyArg_ParseTuple(args, "|l:GetRunningObjectTable", &reserved) )
		return NULL;
	IRunningObjectTable *pROT = NULL;
	HRESULT hr = GetRunningObjectTable(reserved,&pROT);
	if (S_OK!=hr)
		return OleSetOleError(hr);
	return PyCom_PyObjectFromIUnknown(pROT, IID_IRunningObjectTable, FALSE);
}

// @pymethod <o PyIBindCtx>|pythoncom|CreateBindCtx|Creates a new <o PyIBindCtx> object.
static PyObject *pythoncom_CreateBindCtx(PyObject *self, PyObject *args)
{
	DWORD reserved = 0;
	if ( !PyArg_ParseTuple(args, "|l:CreateBindCtx", &reserved) )
		return NULL;
	IBindCtx *pBC = NULL;
	HRESULT hr = CreateBindCtx(reserved,&pBC);
	if (S_OK!=hr)
		return OleSetOleError(hr);
	return PyCom_PyObjectFromIUnknown(pBC, IID_IBindCtx, FALSE);
}

// @pymethod int|pythoncom|RegisterActiveObject|Register an object as the active object for its class
static PyObject *pythoncom_RegisterActiveObject(PyObject *self, PyObject *args)
{    
	DWORD    dwflags=0, dwkey=0;
     HRESULT  hr;
     CLSID    clsid;
     PyObject *obclsid;
     PyObject *obunk;
     IUnknown *punk;

	// @pyparm <o PyIUnknown>|obUnknown||The object to register.
	// @pyparm <o PyIID>|clsid||The CLSID for the object
	// @pyparm int|flags||Flags to use.
    if (!PyArg_ParseTuple(args, "OOi:RegisterActiveObject",
                                &obunk,
                                &obclsid,
                                &dwflags)) return NULL;

    if (!PyCom_IUnknownFromPyObject(obunk, &punk)) return NULL;
    if (!PyCom_CLSIDFromPyObject(obclsid, &clsid)) return NULL;

    hr = RegisterActiveObject(punk, clsid, dwflags, &dwkey);
    if (S_OK!=hr) return OleSetOleError(hr);
    return PyInt_FromLong(dwkey);
	// @rdesc The result is a handle which should be pass to <om pythoncom.RevokeActiveObject>
}

// @pymethod |pythoncom|RevokeActiveObject|Ends an objects status as active.
static PyObject *pythoncom_RevokeActiveObject(PyObject *self, PyObject *args)
{
	DWORD    dw_x=0;
    HRESULT  hr;
     
	// @pyparm int|handle||A handle obtained from <om pythoncom.RegisterActiveObject>
    if(!PyArg_ParseTuple(args,"l:RevokeActiveObject", &dw_x))
       return NULL;
    hr = RevokeActiveObject(dw_x, NULL);
    if (S_OK!=hr) return OleSetOleError(hr);
    Py_INCREF(Py_None);
    return Py_None;
}

// @pymethod |pythoncom|Unicode|Converts a string into a <o PyUnicode> object.
static PyObject *pythoncom_Unicode(PyObject *self, PyObject *args)
{
	const char *s;

	// @pyparm string|s||The string to convert into a Unicode object
    if ( !PyArg_ParseTuple(args, "s:Unicode", &s))
       return NULL;
	return PyUnicodeObject_FromString(s);
}


/* List of module functions */
// @module pythoncom|A module, encapsulating the OLE automation API
static struct PyMethodDef pythoncom_methods[]=
{
	{ "_GetInterfaceCount",  pythoncom_GetInterfaceCount, 1},    // @pymeth _GetInterfaceCount|Retrieves the number of interface objects currently in existance
	{ "_GetGatewayCount",    pythoncom_GetGatewayCount, 1},    // @pymeth _GetInterfaceCount|Retrieves the number of gateway objects currently in existance
	{ "CoCreateInstance",    pythoncom_CoCreateInstance, 1 },    // @pymeth CoCreateInstance|Create a new instance of an OLE automation server.
	{ "CoFreeUnusedLibraries", pythoncom_CoFreeUnusedLibraries, 1}, // @pymeth CoFreeUnusedLibraries|Unloads any DLLs that are no longer in use and that, when loaded, were specified to be freed automatically.
	{ "CoInitialize",        pythoncom_CoInitialize, 1 },
	{ "CoUninitialize",      pythoncom_CoUninitialize, 1 },
	{ "CoRegisterClassObject",pythoncom_CoRegisterClassObject, 1 },// @pymeth CoRegisterClassObject|Registers an EXE class object with OLE so other applications can connect to it.
	{ "CoRevokeClassObject",pythoncom_CoRevokeClassObject, 1 },// @pymeth CoRevokeClassObject|Informs OLE that a class object, previously registered with the <om pythoncom.CoRegisterClassObject> method, is no longer available for use. 
	{ "Connect",             pythoncom_connect, 1 },			 // @pymeth Connect|Connects to a running instance of an OLE automation server.
	{ "connect",             pythoncom_connect, 1 },
	{ "CreateGuid",          pythoncom_createguid, 1 },          // @pymeth CreateGuid|Creates a new, unique GUIID.
	{ "CreateBindCtx",       pythoncom_CreateBindCtx, 1 },       // @pymeth CreateBindCtx|Obtains a <o PyIBindCtx> object.
	{ "CreateFileMoniker",   pythoncom_CreateFileMoniker, 1 }, // @pymeth CreateFileMoniker|Creates a file moniker given a file name.
	{ "CreatePointerMoniker", pythoncom_CreatePointerMoniker, 1 }, // @pymeth CreatePointerMoniker|Creates a pointer moniker based on a pointer to an object.
	{ "GetActiveObject",     pythoncom_GetActiveObject, 1 },     // @pymeth GetActiveObject|Retrieves an object representing a running object registered with OLE
	{ "GetClassFile",        pythoncom_GetClassFile, 1 },        // @pymeth GetClassFile|Supplies the CLSID associated with the given filename.
	{ "GetFacilityString",   pythoncom_GetFacilityString, 1 },   // @pymeth GetFacilityString|Returns the facility string, given an OLE scode.
	{ "GetRunningObjectTable", pythoncom_GetRunningObjectTable, 1 }, // @pymeth GetRunningObjectTable|Obtains a <o PyIRunningObjectTable> object.
	{ "GetScodeString",      pythoncom_GetScodeString, 1 },      // @pymeth GetScodeString|Returns the string for an OLE scode.
	{ "GetScodeRangeString", pythoncom_GetScodeRangeString, 1 }, // @pymeth GetScodeRangeString|Returns the scode range string, given an OLE scode.
	{ "GetSeverityString",   pythoncom_GetSeverityString, 1 },   // @pymeth GetSeverityString|Returns the severity string, given an OLE scode.
	{ "LoadRegTypeLib",      pythoncom_loadregtypelib, 1 },		 // @pymeth LoadRegTypeLib|Loads a registered type library by CLSID
	{ "LoadTypeLib",         pythoncom_loadtypelib, 1 },		 // @pymeth LoadTypeLib|Loads a type library by name
	{ "MakeIID",             pythoncom_MakeIID, 1 },             // @pymeth MakeIID|Makes an IID object from a string.
	{ "MakeTime",			pythoncom_MakeTime, 1 },			// @pymeth MakeTime|Makes a time object from the argument.  Argument can be an integer/float or a tuple (as returned by time module functions).
	{ "MkParseDisplayName",	pythoncom_MkParseDisplayName, 1 },	// @pymeth MkParseDisplayName|Parses a moniker display name into a moniker object. The inverse of IMoniker::GetDisplayName.
	{ "new",                 pythoncom_new, 1 },
	{ "New",                 pythoncom_new, 1 },                 // @pymeth New|Create a new instance of an OLE automation server.
	{ "ProgIDFromCLSID",     pythoncom_progidfromclsid, 1 },     // @pymeth ProgIDFromCLSID|Converts a CLSID string to a progID.
	{ "RecoverFromID",       pythoncom_RecoverFromID, 1 },
	{ "RegisterActiveObject",pythoncom_RegisterActiveObject, 1 }, // @pymeth RegisterActiveObject|Register an object as the active object for its class
	{ "RevokeActiveObject",  pythoncom_RevokeActiveObject, 1 },   // @pymeth RevokeActiveObject|Ends an objects status as active.
	{ "StgCreateDocfile",      pythoncom_StgCreateDocfile, 1 },       // @pymeth StgCreateDocfile|Creates a new compound file storage object using the OLE-provided compound file implementation for the <o PyIStorage> interface.
	{ "StgCreateDocfileOnILockBytes",      pythoncom_StgCreateDocfileOnILockBytes, 1 }, // @pymeth StgCreateDocfileOnILockBytes|Creates a new compound file storage object using the OLE-provided compound file implementation for the <o PyIStorage> interface.
	{ "StgIsStorageFile",    pythoncom_StgIsStorageFile, 1 },       // @pymeth StgIsStorageFile|Indicates whether a particular disk file contains a storage object.
	{ "StgOpenStorage",      pythoncom_StgOpenStorage, 1 },       // @pymeth StgOpenStorage|Opens an existing root storage object in the file system.
	{ "WrapObject",          pythoncom_WrapObject, 1 }, // @pymeth WrapObject|Wraps an object in a gateway.
	{ "UnwrapObject",        pythoncom_UnwrapObject, 1 },
	{ "Unicode",			pythoncom_Unicode, 1 },
	{ NULL, NULL }
};

int AddConstant(PyObject *dict, const char *key, long value)
{
	PyObject *oval = PyInt_FromLong(value);
	if (!oval)
	{
		return 1;
	}
	int rc = PyDict_SetItemString(dict, (char*)key, oval);
	PyTS_DECREF(oval);
	return rc;
}

#define ADD_CONSTANT(tok) AddConstant(dict, #tok, tok)

static char *modName = "pythoncom";

// Get the dictionary for the Pythoncom module.  Note - NO REFERENCE
// is added to the result.
// NOTE - this assumes module initialised, else disaster!
PyObject *PyCom_GetCoreModuleDict()
{
	PyObject *mod = PyImport_AddModule(modName);
	if (!mod) return NULL;
	return mod ? PyModule_GetDict(mod) : NULL;
}

void PyCom_InitCoreModule()
{
	static PyObject *oModule = NULL;
	if (oModule) return;

	// Create the module and add the functions
	oModule = Py_InitModule(modName, pythoncom_methods);

	PyObject *dict = PyModule_GetDict(oModule);
	PyDict_SetItemString(dict, "TypeNames", g_obPyCom_MapNameToType);
	PyDict_SetItemString(dict, "TypeIIDs", g_obPyCom_MapIIDToType);
	PyDict_SetItemString(dict, "ServerInterfaces", g_obPyCom_MapGatewayIIDToName);

	g_obEmpty = new PyOleEmpty;
	PyDict_SetItemString(dict, "Empty", g_obEmpty);

	g_obMissing = new PyOleMissing;
	PyDict_SetItemString(dict, "Missing", g_obMissing);

	// Add some symbolic constants to the module   
	pycom_Error = PyString_FromString("pythoncom.error");
	if (pycom_Error == NULL || PyDict_SetItemString(dict, "error", pycom_Error) != 0)
	{
		PyErr_SetString(PyExc_MemoryError, "can't define error");
		return;
	}
	if (PyDict_SetItemString(dict, "ole_error", PyWinExc_COMError) != 0)
	{
		PyErr_SetString(PyExc_MemoryError, "can't define ole_error");
		return;
	}
	// Add the same constant, but with a "new name"
	if (PyDict_SetItemString(dict, "com_error", PyWinExc_COMError) != 0)
	{
		PyErr_SetString(PyExc_MemoryError, "can't define com_error");
		return;
	}
	// Add a few types.
	PyDict_SetItemString(dict, "PyTimeType", (PyObject *)&PyTimeType);
	PyDict_SetItemString(dict, "PyIIDType", (PyObject *)&PyIIDType);
	PyDict_SetItemString(dict, "PyUnicodeType", (PyObject *)&PyUnicodeType);

	// Symbolic constants.
	ADD_CONSTANT(ACTIVEOBJECT_STRONG);
	ADD_CONSTANT(ACTIVEOBJECT_WEAK);

	ADD_CONSTANT(CLSCTX_ALL);
	ADD_CONSTANT(CLSCTX_INPROC);
	ADD_CONSTANT(CLSCTX_SERVER);

	ADD_CONSTANT(CLSCTX_INPROC_SERVER);
	ADD_CONSTANT(CLSCTX_INPROC_HANDLER);
	ADD_CONSTANT(CLSCTX_LOCAL_SERVER);
	ADD_CONSTANT(CLSCTX_REMOTE_SERVER);

	// DISPATCH
	ADD_CONSTANT(DISPATCH_PROPERTYGET);
	ADD_CONSTANT(DISPATCH_PROPERTYPUT);
	ADD_CONSTANT(DISPATCH_PROPERTYPUTREF);
	ADD_CONSTANT(DISPATCH_METHOD);
	ADD_CONSTANT(DISPID_CONSTRUCTOR);
	ADD_CONSTANT(DISPID_DESTRUCTOR);
	ADD_CONSTANT(DISPID_COLLECT);
	

	// DISPID
	ADD_CONSTANT(DISPID_VALUE);
	ADD_CONSTANT(DISPID_UNKNOWN);
	ADD_CONSTANT(DISPID_PROPERTYPUT);
	ADD_CONSTANT(DISPID_NEWENUM);
	ADD_CONSTANT(DISPID_EVALUATE);

	// FUNCFLAGS
	ADD_CONSTANT(FUNCFLAG_FRESTRICTED);
	ADD_CONSTANT(FUNCFLAG_FSOURCE);
	ADD_CONSTANT(FUNCFLAG_FBINDABLE);
	ADD_CONSTANT(FUNCFLAG_FREQUESTEDIT);
	ADD_CONSTANT(FUNCFLAG_FDISPLAYBIND);
	ADD_CONSTANT(FUNCFLAG_FDEFAULTBIND);
	ADD_CONSTANT(FUNCFLAG_FHIDDEN);
	ADD_CONSTANT(FUNCFLAG_FUSESGETLASTERROR);

	// FUNCKIND
	ADD_CONSTANT(FUNC_VIRTUAL);
	ADD_CONSTANT(FUNC_PUREVIRTUAL);
	ADD_CONSTANT(FUNC_NONVIRTUAL);
	ADD_CONSTANT(FUNC_STATIC);
	ADD_CONSTANT(FUNC_DISPATCH);

	// IMPLTYPEFLAGS
	ADD_CONSTANT(IMPLTYPEFLAG_FDEFAULT);
	ADD_CONSTANT(IMPLTYPEFLAG_FSOURCE);
	ADD_CONSTANT(IMPLTYPEFLAG_FRESTRICTED);

	// IDLFLAGS
	ADD_CONSTANT(IDLFLAG_NONE);
	ADD_CONSTANT(IDLFLAG_FIN);
	ADD_CONSTANT(IDLFLAG_FOUT);
	ADD_CONSTANT(IDLFLAG_FLCID);
	ADD_CONSTANT(IDLFLAG_FRETVAL);

	// Moniker types.
	ADD_CONSTANT(MKSYS_NONE);
	ADD_CONSTANT(MKSYS_GENERICCOMPOSITE);
	ADD_CONSTANT(MKSYS_FILEMONIKER);
	ADD_CONSTANT(MKSYS_ANTIMONIKER);
	ADD_CONSTANT(MKSYS_ITEMMONIKER);
	ADD_CONSTANT(MKSYS_POINTERMONIKER);
	ADD_CONSTANT(MKSYS_CLASSMONIKER);

	// PARAMFLAGS
	ADD_CONSTANT(PARAMFLAG_NONE);
	ADD_CONSTANT(PARAMFLAG_FIN);
	ADD_CONSTANT(PARAMFLAG_FOUT);
	ADD_CONSTANT(PARAMFLAG_FLCID);
	ADD_CONSTANT(PARAMFLAG_FRETVAL);
	ADD_CONSTANT(PARAMFLAG_FOPT);
	ADD_CONSTANT(PARAMFLAG_FHASDEFAULT);

	// INVOKEKIND
	ADD_CONSTANT(INVOKE_FUNC);
	ADD_CONSTANT(INVOKE_PROPERTYGET);
	ADD_CONSTANT(INVOKE_PROPERTYPUT);
	ADD_CONSTANT(INVOKE_PROPERTYPUTREF);

	ADD_CONSTANT(REGCLS_SINGLEUSE);
	ADD_CONSTANT(REGCLS_MULTIPLEUSE);
	ADD_CONSTANT(REGCLS_MULTI_SEPARATE);
	ADD_CONSTANT(REGCLS_SUSPENDED);

	// TYPEFLAGS
	ADD_CONSTANT(TYPEFLAG_FAPPOBJECT);
	ADD_CONSTANT(TYPEFLAG_FCANCREATE);
	ADD_CONSTANT(TYPEFLAG_FLICENSED);
	ADD_CONSTANT(TYPEFLAG_FPREDECLID);
	ADD_CONSTANT(TYPEFLAG_FHIDDEN);
	ADD_CONSTANT(TYPEFLAG_FCONTROL);
	ADD_CONSTANT(TYPEFLAG_FDUAL);
	ADD_CONSTANT(TYPEFLAG_FNONEXTENSIBLE);
	ADD_CONSTANT(TYPEFLAG_FOLEAUTOMATION);
	ADD_CONSTANT(TYPEFLAG_FRESTRICTED);
	ADD_CONSTANT(TYPEFLAG_FAGGREGATABLE);
	ADD_CONSTANT(TYPEFLAG_FREPLACEABLE);
	ADD_CONSTANT(TYPEFLAG_FDISPATCHABLE);
	ADD_CONSTANT(TYPEFLAG_FREVERSEBIND);

	// TYPEKIND
	ADD_CONSTANT(TKIND_ENUM);
	ADD_CONSTANT(TKIND_RECORD);
	ADD_CONSTANT(TKIND_MODULE);
	ADD_CONSTANT(TKIND_INTERFACE);
	ADD_CONSTANT(TKIND_DISPATCH);
	ADD_CONSTANT(TKIND_COCLASS);
	ADD_CONSTANT(TKIND_ALIAS);
	ADD_CONSTANT(TKIND_UNION);

	// VARFLAGS
	ADD_CONSTANT(VARFLAG_FREADONLY);

	// VARKIND
	ADD_CONSTANT(VAR_PERINSTANCE);
	ADD_CONSTANT(VAR_STATIC);
	ADD_CONSTANT(VAR_CONST);
	ADD_CONSTANT(VAR_DISPATCH);

	// VARTYPE
	ADD_CONSTANT(VT_EMPTY);
	ADD_CONSTANT(VT_NULL);
	ADD_CONSTANT(VT_I2);
	ADD_CONSTANT(VT_I4);
	ADD_CONSTANT(VT_R4);
	ADD_CONSTANT(VT_R8);
	ADD_CONSTANT(VT_CY);
	ADD_CONSTANT(VT_DATE);
	ADD_CONSTANT(VT_BSTR);
	ADD_CONSTANT(VT_DISPATCH);
	ADD_CONSTANT(VT_ERROR);
	ADD_CONSTANT(VT_BOOL);
	ADD_CONSTANT(VT_VARIANT);
	ADD_CONSTANT(VT_UNKNOWN);
	ADD_CONSTANT(VT_DECIMAL);
	ADD_CONSTANT(VT_I1);
	ADD_CONSTANT(VT_UI1);
	ADD_CONSTANT(VT_UI2);
	ADD_CONSTANT(VT_UI4);
	ADD_CONSTANT(VT_I8);
	ADD_CONSTANT(VT_UI8);
	ADD_CONSTANT(VT_INT);
	ADD_CONSTANT(VT_UINT);
	ADD_CONSTANT(VT_VOID);
	ADD_CONSTANT(VT_HRESULT);
	ADD_CONSTANT(VT_PTR);
	ADD_CONSTANT(VT_SAFEARRAY);
	ADD_CONSTANT(VT_CARRAY);
	ADD_CONSTANT(VT_USERDEFINED);
	ADD_CONSTANT(VT_LPSTR);
	ADD_CONSTANT(VT_LPWSTR);
	ADD_CONSTANT(VT_FILETIME);
	ADD_CONSTANT(VT_BLOB);
	ADD_CONSTANT(VT_STREAM);
	ADD_CONSTANT(VT_STORAGE);
	ADD_CONSTANT(VT_STREAMED_OBJECT);
	ADD_CONSTANT(VT_STORED_OBJECT);
	ADD_CONSTANT(VT_BLOB_OBJECT);
	ADD_CONSTANT(VT_CF);
	ADD_CONSTANT(VT_CLSID);
	ADD_CONSTANT(VT_VECTOR);
	ADD_CONSTANT(VT_ARRAY);
	ADD_CONSTANT(VT_BYREF);

	ADD_CONSTANT(VT_RESERVED);
	ADD_CONSTANT(VT_ILLEGAL);
	ADD_CONSTANT(VT_ILLEGALMASKED);
	ADD_CONSTANT(VT_TYPEMASK);

}
/* Module initialisation */
extern "C" __declspec(dllexport) void initpythoncom()
{
	// The DLL Load inited the module.
	// All we do here is init COM itself.  Done here
	// so other clients get a chance to beat us to it!
	HRESULT hr = PyCom_CoInitialize(NULL);
	if (FAILED(hr))
		PyErr_SetString(PyExc_RuntimeError, "The COM libraries could not be initialized");
}
